package Intapi::AutobudgetOptimizedBy;

=head1 NAME

    Intapi::AutobudgetOptimizedBy

=head1 DESCRIPTION

    IntAPI методы для перфоманс баннеров

=cut

use Direct::Modern;

use Settings;
use LogTools;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Log;
use Yandex::Validate qw/is_valid_id/;

use Readonly;

# Коды ошибок
Readonly my $UNKNOWN_SHARD     => 1;
Readonly my $UNKNOWN_FILTER    => 2;
Readonly my $BAD_ID            => 3;
Readonly my $BAD_OPTIMIZEDBY   => 4;
Readonly my $UNKNOWN_ORDERID   => 5;


=head2 new

=cut
sub new
{
    bless {};
}

=head2 setOrder

    Установить CPC/CPA оптимизацию на заказ

    Тестировать:
    perl -MDDP -MJSON::RPC::Simple::Client -e '
        my $records = [
            {"OrderID" => 123456,"OptimizedBy" => "CPA"},
            {"OrderID" => 654321,"OptimizedBy" => "CPC"},
            {"OrderID" => 666,"OptimizedBy" => "CPC"},
            {"OrderID" => 667,"OptimizedBy" => "CPC"},
        ];
        p $records;
        my $c = JSON::RPC::Simple::Client->new("http://8145.beta1.direct.yandex.ru/jsonrpc/AutobudgetOptimizedBy");
        my $results = $c->setOrder($records);
        p $results;'

=cut

sub setOrder
{
    my ($self, $params, $procedure, @extra_args) = @_;

    my $log = Yandex::Log->new(
        log_file_name => 'setOrder.log',
        date_suf      => "%Y%m%d",
        msg_prefix    => "[$$]",
        lock          => 1,
        use_syslog    => 0,
    );

    $log->out({request => $params});

    my $errors = [];

    my $validation_result = _validate_setOrder_data($params);
    # если нашли ошибки во входных параметрах
    push @$errors, @{$validation_result->{errors}} if @{$validation_result->{errors}};

    my @log_prices;
    foreach_shard OrderID => ( $validation_result->{valid_orders} ), with_undef_shard => 1, sub {
        my ($shard, $orders_chunk) = @_;

        if ($shard) {
            my @orderids = map { $_->{OrderID} } @$orders_chunk;
            my $order_prices = get_hashes_hash_sql(PPC(shard => $shard), [
                    "SELECT c.OrderID
                        , c.cid
                        , cast(s.strategy_data->>'\$.filter_avg_bid' as decimal(8,2)) as filter_avg_bid
                        , cast(s.strategy_data->>'\$.filter_avg_cpa' as decimal(8,2)) as filter_avg_cpa
                        , IFNULL(c.currency, 'YND_FIXED') AS currency
                    FROM campaigns c
                    LEFT JOIN strategies s ON c.strategy_id = s.strategy_id",
                    WHERE => {
                        'c.OrderID' => \@orderids,
                        'c.type' => 'performance',
                    },
                ]);
            my %update_data;
            foreach my $order (@$orders_chunk) {
                my $price_info = $order_prices->{$order->{OrderID}};
                if (defined $price_info->{cid}) {
                    $update_data{$price_info->{cid}} = $order->{OptimizedBy};
                    push @log_prices, {
                            cid       => $price_info->{cid},
                            id        => $price_info->{cid},
                            type      => "perf_order_set_$order->{OptimizedBy}",
                            price     => $price_info->{filter_avg_bid},
                            price_ctx => $price_info->{filter_avg_cpa},
                            currency  => $price_info->{currency},
                    };
                } else {
                    push @$errors, {OrderID => $order->{OrderID}, code => $UNKNOWN_ORDERID, msg => "unknown OrderID"};
                }
            }
            my %update_fields;
            $update_fields{"now_optimizing_by__dont_quote"} = sql_case(cid => \%update_data, default__dont_quote => 'now_optimizing_by');
            do_update_table(PPC(shard => $shard), 'campaigns_performance', \%update_fields, where => {cid => [ keys %update_data ]});
        } else {
            # если не смогли определить шард для заказов
            push @$errors, map { {OrderID => $_->{OrderID}, code => $UNKNOWN_SHARD, msg => "unknown shard"} } @$orders_chunk;
        }
    };

    $log->out({response => $errors});
    # save logs
    if (@log_prices) {
        LogTools::log_price(\@log_prices);
    }

    return $errors;
}

=head2 setFilter

    Установить CPC/CPA оптимизацию на фильтр

    Тестировать:
    perl -MDDP -MJSON::RPC::Simple::Client -e '
        my $records = [
            {"OrderID" => 20063,"GroupExportID" => 1168,"PhraseID" => 157186,"OptimizedBy" => "CPA"},
            {"OrderID" => 20063,"GroupExportID" => 350,"PhraseID" => 157186,"OptimizedBy" => "CPC"},
            {"OrderID" => 20159,"GroupExportID" => 587,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
            {"OrderID" => 20159,"GroupExportID" => 667,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
            {"OrderID" => 666,"GroupExportID" => 666,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
            {"OrderID" => 667,"GroupExportID" => 667,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
        ];
        p $records;
        my $c = JSON::RPC::Simple::Client->new("http://8145.beta1.direct.yandex.ru/jsonrpc/AutobudgetOptimizedBy");
        my $results = $c->setFilter($records);
        p $results;'

=cut

sub setFilter
{
    my ($self, $params, $procedure, @extra_args) = @_;

    my $log = Yandex::Log->new(
        log_file_name => 'setFilter.log',
        date_suf      => "%Y%m%d",
        msg_prefix    => "[$$]",
        lock          => 1,
        use_syslog    => 0,
    );

    $log->out({request => $params});

    my $errors = [];

    my $validation_result = _validate_setFilter_data($params);
    # если нашли ошибки во входных параметрах
    push @$errors, @{$validation_result->{errors}} if @{$validation_result->{errors}};

    my @log_prices;
    foreach_shard OrderID => ( $validation_result->{valid_orders} ), chunk_size => 100, with_undef_shard => 1, sub {
        my ($shard, $orders_chunk) = @_;

        if ($shard) {
            my $orders_ids = [ map { {'c.OrderID'=> $_->{OrderID}, 'p.pid' => $_->{GroupExportID}, 'bp.perf_filter_id' => $_->{PhraseID}} } @$orders_chunk ];
            my $where = join( 'OR', map { '(' . sql_condition( $_, 'operation' => 'AND' ) . ')' } @{ $orders_ids } );
            my $existing_filters = get_hashes_hash_sql(PPC(shard => $shard),
                                                  "SELECT bp.perf_filter_id AS PhraseID, p.pid AS GroupExportID, c.OrderID
                                                        , c.cid
                                                        , IFNULL(c.currency, 'YND_FIXED') AS currency
                                                        , bp.price_cpc
                                                        , bp.price_cpa
                                                   FROM campaigns c
                                                   INNER JOIN phrases p ON c.cid=p.cid
                                                   INNER JOIN bids_performance bp ON p.pid=bp.pid
                                                   WHERE $where");
            my %update_data;
            foreach my $order (@$orders_chunk) {
                my $cur_filter = $existing_filters->{$order->{PhraseID}};
                if (defined $cur_filter
                    && $cur_filter->{GroupExportID} == $order->{GroupExportID}
                    && $cur_filter->{OrderID} == $order->{OrderID})
                {
                    $update_data{$order->{PhraseID}} = $order->{OptimizedBy};
                    push @log_prices, {
                            cid       => $cur_filter->{cid},
                            pid       => $cur_filter->{GroupExportID},
                            id        => $cur_filter->{PhraseID},
                            type      => "perf_filter_set_$order->{OptimizedBy}",
                            price     => $cur_filter->{price_cpc},
                            price_ctx => $cur_filter->{price_cpa},
                            currency  => $cur_filter->{currency},
                    };
                } else {
                    push @$errors, {OrderID => $order->{OrderID}, GroupExportID => $order->{GroupExportID}, PhraseID => $order->{PhraseID}, code => $UNKNOWN_FILTER, msg => "unknown OrderID, GroupExportID or PhraseID"};
                }
            }
            my %update_fields;
            $update_fields{"now_optimizing_by__dont_quote"} = sql_case(perf_filter_id => \%update_data, default__dont_quote => 'now_optimizing_by');
            do_update_table(PPC(shard => $shard), 'bids_performance', \%update_fields, where => {perf_filter_id => [ keys %update_data ]});
        } else {
            # если не смогли определить шард для заказов
            push @$errors, map { {OrderID => $_->{OrderID}, GroupExportID => $_->{GroupExportID}, PhraseID => $_->{PhraseID}, code => $UNKNOWN_SHARD, msg => "unknown shard"} } @$orders_chunk;
        }
    };

    $log->out({response => $errors});
    # save logs
    if (@log_prices) {
        LogTools::log_price(\@log_prices);
    }

    return $errors;
}

=head2 _is_valid_OptimizedBy

    Проверка типа оптимизации. 
    Допустимо только два варианта: CPC и CPA

=cut

sub _is_valid_OptimizedBy
{
    my $OptimizedBy = shift;
    return ( $OptimizedBy =~ /^CP[C|A]$/ );
}

=head2 _validate_setOrder_data

    Валидируем входные данные для метода setOrder
    Принимает на вход массив хешей вида:
    $input => [
        {"OrderID" => 123456,"OptimizedBy" => "CPA"},
        {"OrderID" => 654321,"OptimizedBy" => "CPC"},
        {"OrderID" => "lol","OptimizedBy" => "CPC"},
        {"OrderID" => 667,"OptimizedBy" => "lol"},
    ],

    Отвечает хешем содержащим валидные заказы в поле valid_orders и заказы, которые не прошли валидацию в поле errors вида:
    $result = {
        valid_orders => [
            {"OrderID" => 123456,"OptimizedBy" => "CPA"},
            {"OrderID" => 654321,"OptimizedBy" => "CPC"},
        ],
        errors => [
            {OrderID => 667, code => $BAD_OPTIMIZEDBY, msg => "OptimizedBy must be CPC or CPA"},
            {OrderID => "lol", code => $BAD_ORDERID, msg => "OrderID must be greater than zero"},
        ],
    }

=cut

sub _validate_setOrder_data
{
    my $orders = shift;

    my @valid_orders;
    my @errors;

    foreach my $order (@$orders) {
        if ( is_valid_id($order->{OrderID}) ) {
            if ( _is_valid_OptimizedBy($order->{OptimizedBy}) ) {
                push @valid_orders, $order;
            } else {
                push @errors, {OrderID => $order->{OrderID}, code => $BAD_OPTIMIZEDBY, msg => "OptimizedBy must be CPC or CPA"};
            }
        } else {
            push @errors, {OrderID => $order->{OrderID}, code => $BAD_ID, msg => "OrderID must be greater than zero"};
        }
    }

    return { valid_orders => \@valid_orders, errors => \@errors };
}

=head2 _validate_setFilter_data

    Валидируем входные данные для метода setFilter
    Принимает на вход массив хешей вида:
    $input = [
            {"OrderID" => 42356,"GroupExportID" => 88287,"PhraseID" => 456789,"OptimizedBy" => "CPA"},
            {"OrderID" => 20063,"GroupExportID" => 350,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
            {"OrderID" => 667,"GroupExportID" => 667,"PhraseID" => 987654,"OptimizedBy" => "cp"},
            {"OrderID" => 0,"GroupExportID" => 667,"PhraseID" => 987654,"OptimizedBy" => "CPC"},
        ];
    Отвечает хешем содержащим валидные заказы в поле valid_orders и заказы, которые не прошли валидацию в поле errors вида:
    $result = {
        valid_orders => [
            {OrderID => 42356, GroupExportID => 88287, PhraseID => 456789, OptimizedBy => "CPA"},
            {OrderID => 20063, GroupExportID => 350, PhraseID => 987654, OptimizedBy => "CPC"},
        ],
        errors => [
            {OrderID => 667, GroupExportID => 667, PhraseID => 987654, code => $BAD_OPTIMIZEDBY, msg => "OptimizedBy must be CPC or CPA"},
            {OrderID => 0, GroupExportID => 667, PhraseID => 987654, code => $BAD_ID, msg => "OrderID, GroupExportID, PhraseID must be greater than zero"},
        ],
    }

=cut

sub _validate_setFilter_data
{
    my $orders = shift;

    my @valid_orders;
    my @errors;

    foreach my $order (@$orders) {
        if ( is_valid_id($order->{OrderID}) && is_valid_id($order->{GroupExportID}) && is_valid_id($order->{PhraseID}) ) {
            if ( _is_valid_OptimizedBy($order->{OptimizedBy}) ) {
                push @valid_orders, $order;
            } else {
                push @errors, {OrderID => $order->{OrderID}, GroupExportID => $order->{GroupExportID}, PhraseID => $order->{PhraseID}, code => $BAD_OPTIMIZEDBY, msg => "OptimizedBy must be CPC or CPA"};
            }
        } else {
            push @errors, {OrderID => $order->{OrderID}, GroupExportID => $order->{GroupExportID}, PhraseID => $order->{PhraseID}, code => $BAD_ID, msg => "OrderID, GroupExportID, PhraseID must be greater than zero"};
        }
    }

    return { valid_orders => \@valid_orders, errors => \@errors };
}


1;
