package Intapi::AutobudgetPrices;

=head1 NAME

    Intapi::AutobudgetPrices

=head1 DESCRIPTION

    JSON-RPC интерфейс для приёма ставок от автобюджета

=cut


use strict;
use warnings;
use utf8;

our $WRITE_INCOMING_DATA_TO_LOG //= 0;

use List::Util qw(max);
use List::MoreUtils qw(uniq);

use Yandex::Log::Messages;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::Validate qw(is_valid_id);
use Yandex::Memcached::Lock ();

use Settings;
use AdGroupTools;
use Tools ();
use LogTools;
use Currencies qw(currency_price_rounding get_currency_constant get_frontpage_min_price);
use Currency::Rate qw(convert_currency);
use PhrasePrice ();
use CampAutoPrice::Common ();


=head2 new

=cut

sub new
{
    bless {};
}

=head2 set

    Приём выставленных автобюджетом ставок и сохранение их в bids[_retargeting] для показа клиентам
    Ставки могут приходить не все (с пропусками).
    Возвращает ссылку на массив с неприменёнными записями и причинами этого (см. $result).

    формат по ретаргетингу - https://jira.yandex-team.ru/browse/YABS-36961

    $params => [
        {
            GroupExportID => 85458114,  # phrases.pid
            PhraseID => 4,
            ContextType => 2,
            price => 0.33,
            context_price => 0.33,
            currency => 0, # 0 => в price[_context] фишки, 1 => price[_context] в валюте кампании
        },
        ...
    ];

    $result => [
        {
            GroupExportID => 85458114,
            PhraseID => 4,
            code => 1|6|7|8|9,
            msg => "autobudget disabled (retargetings)",
        },
        ...
    ];

    тестирование:
    perl -MDDP -MJSON::RPC::Simple::Client -e '
        my $records = [{GroupExportID => 85458114, PhraseID => 4, ContextType => 2, context_price => 0.33, currency => 0}];
        p $records;
        my $c = JSON::RPC::Simple::Client->new("http://9704.beta2.direct.yandex.ru/jsonrpc/AutobudgetPrices");
        my $results = $c->set($records);
        p $results;
    '

=cut

sub set
{
    my ($self, $params, $procedure, @extra_args) = @_;

    my $log = Yandex::Log::Messages->new();
    $log->msg_prefix("[$$]");
    $log->out('START');
    $log->out('Got ' . scalar(@$params) . ' records');

    Tools::_save_vars(0);

    # Лочимся
    my $mcl = Yandex::Memcached::Lock->new(
        servers => $Settings::MEMCACHED_SERVERS,
        max_locks_num => 13,
        entry => "locks/$Settings::SPECIFY_PRICES_LOCK_NAME",
        );
    if (!$mcl->get_lock()) {
        $log->die("Can't get lock '$Settings::SPECIFY_PRICES_LOCK_NAME' in memcached");
    }

    if ($WRITE_INCOMING_DATA_TO_LOG) {
        # Пишем данные только после взятия лока, чтобы не засорять лог дубликатами
        LogTools::log_autobudget_prices($params);
    }

    my @errors;
    eval {
        # Перегруппируем данные для более удобной работы
        my %bids;
        my %retargetings;

        for my $ph ( @$params ) {
            # переименовываем, у нас и в БК разные названия
            if (exists $ph->{context_price}) {
                $ph->{price_context} = $ph->{context_price};
            }
            if ( !is_valid_id($ph->{GroupExportID})
                 || !is_valid_id($ph->{PhraseID})
                 || ! $ph->{ContextType} || $ph->{ContextType} !~ /^(?:1|2)$/
                 || ($ph->{price} && $ph->{price} !~ /^[\.\d]+$/)
                 || ($ph->{price_context} && $ph->{price_context} !~ /^[\.\d]+$/)
            ) {
                $log->out({'bad record' => $ph});
                push @errors, {GroupExportID => $ph->{GroupExportID}, PhraseID => $ph->{PhraseID}, code => 1, msg => "bad format"};
                next;
            }

            if ($ph->{ContextType} == 1) {
                $bids{$ph->{GroupExportID}}->{$ph->{PhraseID}} = hash_cut $ph, qw/price price_context currency/;
            } elsif ($ph->{ContextType} == 2) {
                $retargetings{$ph->{GroupExportID}}->{$ph->{PhraseID}} = hash_cut $ph, qw/price_context currency/;
            }
        }
        $log->out('Total ' . scalar(keys %bids) . ' groups');

        my @log_prices;
        for my $chunk (sharded_chunks pid => [keys %bids], with_undef_shard => 1) {
            my $shard = $chunk->{shard};
            my @to_update;
            for my $pid (@{$chunk->{pid}}) {
                my $phrases = $bids{$pid};
                # Получаем данные группы, её кампании и баннеров
                my $data;
                if ($shard) {
                    $data = get_one_line_sql(PPC(shard => $shard), "
                                                   SELECT c.cid, c.autobudget, c.archived, b.bid
                                                        , IFNULL(cl.work_currency, 'YND_FIXED') AS client_currency
                                                        , IFNULL(c.currency, 'YND_FIXED') AS currency
                                                        , (NOW() > ccq.start_convert_at - INTERVAL ? MINUTE AND ccq.state != 'DONE') AS going_to_convert_soon
                                                        , s.enable_cpc_hold = 'Yes' AS enable_cpc_hold
                                                        , p.adgroup_type
                                                      FROM phrases p
                                                           join banners b on b.pid = p.pid
                                                           join campaigns c on c.cid = p.cid
                                                      left join strategies s ON c.strategy_id = s.strategy_id
                                                      left join users u ON c.uid = u.uid
                                                      left join clients cl ON u.ClientID = cl.ClientID
                                                      left join currency_convert_queue ccq ON u.ClientID = ccq.ClientID
                                                     WHERE p.pid = ?
                                                     LIMIT 1", $Settings::STOP_OPERATION_MINUTES_BEFORE_CONVERT, $pid );
                }
                # Проверяем свойства баннера
                if ( $data && $data->{cid} ) {
                    # Если кампания в архиве и ее валюта не равна валюте клиента, значит баннер принадлежит
                    # старой версии конвертированной кампании
                    if ($data->{archived} eq 'Yes' && ($data->{currency} ne $data->{client_currency})) {
                        $log->out("skipping pid=$pid because it's an archived old version of converted campaign");
                        push @errors, map {
                                    {GroupExportID => $pid, PhraseID => $_, code => 9, msg => "archived old version of converted campaign"}
                                } keys %$phrases;
                        next;
                    } elsif ( !$data->{autobudget} || $data->{autobudget} ne 'Yes' ) {
                        if ($data->{enable_cpc_hold}) {
                            $log->out("skipping pid=$pid without error because it's not an autobudget company, but CPC hold is enabled");
                        } else {
                            $log->out("bad record (autobudget disabled), pid=$pid");
                            push @errors, map {
                                        {GroupExportID => $pid, PhraseID => $_, code => 1, msg => "autobudget disabled"}
                                    } keys %$phrases;
                        }
                        next;
                    } elsif ($data->{going_to_convert_soon}) {
                        $log->out("skipping pid=$pid because it's going to convert to real money soon");
                        push @errors, map {
                                    {GroupExportID => $pid, PhraseID => $_, code => 8, msg => "currency converting soon"}
                                } keys %$phrases;
                        next;
                    }
                } else {
                    $log->out("bad record (no such banner), pid=$pid");
                    push @errors, map {
                                {GroupExportID => $pid, PhraseID => $_, code => 1, msg => "no such adgroup"}
                            } keys %$phrases;
                    next;
                }
                # Получаем список старых цен фраз
                my $phrases_old = get_hashes_hash_sql( PPC(shard => $shard), "
                                                   SELECT PhraseID, price, price_context, id
                                                     FROM bids 
                                                    WHERE pid = $pid 
                                                      and PhraseID in (".join(",", keys %$phrases).")" );
                # Сравниваем новые цены со старыми
                PHRASE: while( my ($PhraseID, $prices_hash) = each %$phrases ) {
                    # если кампания в у.е., а ставка пришла в валюте, возвращаем ошибку, т.к. не знаем в какой валюте эта ставка
                    if ($prices_hash->{currency} && $data->{currency} eq 'YND_FIXED') {
                        push @errors, {GroupExportID => $pid, PhraseID => $PhraseID, code => 6, msg => 'cannot save currency price into conventional units campaign'};
                        next;
                    }
                    # если валюта присланной ставки не совпадает с валютой кампании, то конвертируем ставку из у.е. в валюту кампании
                    # в дальнейшем ставка накроется "правильной" валютной ставкой, а сконвертированная (хоть и приблизительно) ставка лучше неактуальной
                    # ну и логгируем такое на всякий случай
                    my $got_currency = ($prices_hash->{currency}) ? $data->{currency} : 'YND_FIXED';
                    if ($got_currency ne $data->{currency}) {
                        $log->out(sprintf("got record with unmatched currency (%s,%s,%s,%s,%s,%s)",
                            $pid,
                            $PhraseID,
                            $data->{currency},
                            $prices_hash->{currency}//'',
                            $prices_hash->{price}//'',
                            $prices_hash->{price_context}//''));
                        for my $field (qw/price price_context/) {
                            next unless defined $prices_hash->{$field};
                            my $new_price = convert_currency($prices_hash->{$field}, $got_currency, $data->{currency});
                            $new_price = currency_price_rounding($new_price, $data->{currency}, up => 1);
                            $prices_hash->{$field} = $new_price;
                        }
                    }
                    # на всякий случай валидируем полученную цену с учётом валюты
                    for my $price_field (qw/price price_context/) {
                        my $price_error = PhrasePrice::validate_phrase_price($prices_hash->{$price_field}, $data->{currency}, dont_support_comma => 1, dont_validate_min_price => 1);
                        if ($price_error) {
                            push @errors, {GroupExportID => $pid, PhraseID => $PhraseID, code => 7, msg => "bad $price_field: $price_error"};
                            next PHRASE;
                        }
                    }

                    if ( !defined $phrases_old->{$PhraseID} ) {
                        $log->out("bad record (no such phrase) ($pid,$PhraseID)");
                        # Если у фразы в результате редактирования изменился PhraseID или фраза удалилась и эти
                        # изменения отклонили на модерации, то у нас в БД пары (pid,PhraseID) уже нет, 
                        # а фраза показывается (в составе предыдущей версии объявления). Соответственно, автобюджет изменяет
                        # по ней ставки и присылает их нам. Поэтому ошибкой в таком случае не отвечаем, но в лог
                        # на всякий случай записываем.
                    } elsif (
                        $prices_hash->{price} && abs( $phrases_old->{$PhraseID}->{price} - $prices_hash->{price} ) > $Currencies::EPSILON
                        || $prices_hash->{price_context} && abs( $phrases_old->{$PhraseID}->{price_context} - $prices_hash->{price_context} ) > $Currencies::EPSILON
                    ) {
                        # Автобюджет может присылать цены ниже минимальной, обрабатываем их так, чтобы в базу записывалась минимальная цена, а в лог изменений цен реальная.
                        my $constant_name = AdGroupTools::min_price_constant_name_by_type($data->{adgroup_type});
                        my $min_price = get_currency_constant($data->{currency}, $constant_name);

                        # TODO: научиться не ходить в базу, когда обновление не нужно после поправки на минимальную ставку
                        push @to_update, hash_merge {
                            id => $phrases_old->{$PhraseID}->{id},
                            price => max($min_price, $prices_hash->{price} || $phrases_old->{$PhraseID}->{price}),
                            price_context => max($min_price, $prices_hash->{price_context} || $phrases_old->{$PhraseID}->{price_context}),
                        };
                        push @log_prices, {
                            cid => $data->{cid},
                            bid => $data->{bid},
                            pid => $pid,
                            id => $phrases_old->{$PhraseID}->{id},
                            type => 'insert3',
                            price => $prices_hash->{price},
                            price_ctx => $prices_hash->{price_context},
                            auto_broker => 'Yes',
                            currency    => $data->{currency},
                        };
                    }
                }
            }

            # обновляем цены на фразах
            if (@to_update) {
                @to_update = sort {$a->{id} <=> $b->{id}} @to_update;
                while (my @chunk = splice(@to_update, 0, 1000)) {
                    my @ids = map {$_->{id}} @chunk;
                    my $price_context_case = {map {$_->{id} => $_->{price_context}} @chunk};
                    my $price_case = {map {$_->{id} => $_->{price}} @chunk};
                    
                    do_update_table(PPC(shard => $shard),
                        bids => {
                            price_context__dont_quote => sql_case(id => $price_context_case, default__dont_quote => 'price_context'),
                            price__dont_quote => sql_case(id => $price_case, default__dont_quote => 'price'),
                        },
                        where => { id => \@ids }
                    );
                }
            }
        } # / sharded_chunks

        # условия ретаргетинга
        for my $chunk (sharded_chunks pid => [keys %retargetings], with_undef_shard => 1) {
            my $shard = $chunk->{shard};
            for my $pid (@{$chunk->{pid}}) {
                my $retargetings = $retargetings{$pid};
                my $data;
                if ($shard) {
                    $data = get_one_line_sql(PPC(shard => $shard), "
                                                   SELECT c.cid, b.bid, c.autobudget, c.archived
                                                        , IFNULL(cl.work_currency, 'YND_FIXED') AS client_currency
                                                        , IFNULL(c.currency, 'YND_FIXED') AS currency
                                                        , (NOW() > ccq.start_convert_at - INTERVAL ? MINUTE AND ccq.state != 'DONE') AS going_to_convert_soon
                                                        , p.adgroup_type
                                                        , p.geo
                                                        , fc.allowed_frontpage_types
                                                     FROM phrases p
                                                     JOIN banners b on b.pid = p.pid
                                                     JOIN campaigns c on c.cid = p.cid
                                                LEFT JOIN campaigns_cpm_yndx_frontpage fc on p.cid = fc.cid
                                                LEFT JOIN users u ON c.uid = u.uid
                                                LEFT JOIN clients cl ON u.ClientID = cl.ClientID
                                                LEFT JOIN currency_convert_queue ccq ON u.ClientID = ccq.ClientID
                                                    WHERE p.pid = ?
                                                    LIMIT 1", $Settings::STOP_OPERATION_MINUTES_BEFORE_CONVERT, $pid);
                }
                # Проверяем свойства баннера
                if ($data && $data->{cid}) {
                    if ($data->{archived} eq 'Yes' && ($data->{currency} ne $data->{client_currency})) {
                        $log->out("skipping pid=$pid because it's an archived old version of converted campaign");
                        push @errors, map {
                                    {GroupExportID => $pid, PhraseID => $_, code => 9, msg => "archived old version of converted campaign"}
                                } keys %$retargetings;
                        next;
                    } elsif (! $data->{autobudget} || $data->{autobudget} ne 'Yes') {
                        $log->out("bad record (autobudget disabled), pid=$pid (retargetings)\n");
                        push @errors, map {
                                    {GroupExportID => $pid, PhraseID => $_, code => 1, msg => "autobudget disabled (retargetings)"}
                                } keys %$retargetings;
                        next;
                    } elsif ($data->{going_to_convert_soon}) {
                        $log->out("skipping pid=$pid because it's going to convert to real money soon");
                        push @errors, map {
                                    {GroupExportID => $pid, PhraseID => $_, code => 8, msg => "currency converting soon (retargetings)"}
                                } keys %$retargetings;
                        next;
                    }
                } else {
                    $log->out("bad record (no such banner), pid=$pid (retargetings)\n");
                    push @errors, map {
                                {GroupExportID => $pid, PhraseID => $_, code => 1, msg => "no such adgroup (retargetings)"}
                            } keys %$retargetings;
                    next;
                }

                # Получаем список старых цен условий ретаргетинга
                my $retargetings_old = get_hashes_hash_sql(PPC(shard => $shard), [
                    "SELECT ret_cond_id, ret_id, price_context
                     FROM bids_retargeting",
                     WHERE => {
                         pid => $pid
                         , ret_cond_id => [keys %$retargetings]
                     }
                ]);

                my @to_update_retargetings;
                # Сравниваем новые цены со старыми
                RETARGETING:
                while (my ($ret_cond_id, $prices_hash) = each %$retargetings) {
                    # если валюта присланной ставки не совпадает с валютой кампании, то конвертируем ставку из у.е. в валюту кампании
                    # в дальнейшем ставка накроется "правильной" валютной ставкой, а сконвертированная (хоть и приблизительно) ставка лучше неактуальной
                    # ну и логгируем такое на всякий случай
                    my $got_currency = ($prices_hash->{currency}) ? $data->{currency} : 'YND_FIXED';
                    if ($got_currency ne $data->{currency}) {
                        $log->out(sprintf("got retargeting record with unmatched currency (%s,%s,%s,%s,%s,%s)",
                            $pid,
                            $ret_cond_id,
                            $data->{currency},
                            $prices_hash->{currency}//'',
                            $prices_hash->{price}//'',
                            $prices_hash->{price_context}//''));
                        for my $field (qw/price_context/) {
                            $prices_hash->{$field} = convert_currency($prices_hash->{$field}, $got_currency, $data->{currency}) if defined $prices_hash->{$field};
                        }
                    }

                    if (! defined $retargetings_old->{$ret_cond_id}) {
                        # условие удалили
                        $log->out("bad record (no such retargetings) (pid: $pid, ret_cond_id: $ret_cond_id)\n");
                    } elsif ($prices_hash->{price_context} && abs( $retargetings_old->{$ret_cond_id}->{price_context} - $prices_hash->{price_context} ) >= $Currencies::EPSILON) {
                        # валидируем полученную ставку с учётом валюты
                        my $price_error = PhrasePrice::validate_phrase_price($prices_hash->{price_context}, $data->{currency}, dont_support_comma => 1, dont_validate_min_price => 1);
                        if ($price_error) {
                            push @errors, {GroupExportID => $pid, ret_cond_id => $ret_cond_id, code => 7, msg => "bad price_context: $price_error"};
                            next RETARGETING;
                        }

                        my $min_price;
                        if ($data->{adgroup_type} eq 'cpm_yndx_frontpage' ) {
                            my @allowed_frontpage_types = split ',' => $data->{allowed_frontpage_types};
                            $min_price = get_frontpage_min_price($data->{currency}, $data->{geo}, \@allowed_frontpage_types);
                        } else {
                            my $constant_name = AdGroupTools::min_price_constant_name_by_type($data->{adgroup_type});
                            $min_price = get_currency_constant($data->{currency}, $constant_name);
                        }
                        # TODO: научиться не ходить в базу, когда обновление не нужно после поправки на минимальную ставку
                        push @to_update_retargetings, hash_merge {
                            ret_id => $retargetings_old->{$ret_cond_id}->{ret_id},
                            price_context =>  max($min_price, $prices_hash->{price_context} || $retargetings_old->{$ret_cond_id}->{price_context}),
                        };
                        push @log_prices, {
                            cid => $data->{cid},
                            bid => $data->{bid},
                            pid => $pid,
                            id => $retargetings_old->{$ret_cond_id}->{ret_id},
                            type => 'ret_update_autobudget',
                            price => 0,
                            price_ctx => $prices_hash->{price_context},
                            auto_broker => 'Yes',
                            currency    => $data->{currency},
                        };
                    }
                }

                if (@to_update_retargetings) {
                    @to_update_retargetings = sort {$a->{ret_id} <=> $b->{ret_id}} @to_update_retargetings;

                    while (my @chunk = splice(@to_update_retargetings, 0, 1000)) {
                        do_update_table(
                            PPC(shard => $shard),
                            'bids_retargeting',
                            {price_context__dont_quote => sql_case('ret_id', {map {$_->{ret_id} => $_->{price_context}} @chunk}, default__dont_quote => 'price_context')},
                            where => {ret_id => [map {$_->{ret_id}} @chunk]}
                        );
                    }
                }
            }
        } # / sharded_chunks

        # save logs
        if (@log_prices) {
            LogTools::log_price(\@log_prices);
            CampAutoPrice::Common::clear_auto_price_queue([uniq map {$_->{cid}} @log_prices]);
        }

    };

    $mcl->unlock();
    if ( $@ ) {
        $log->die("error: $@");
    } else {
        $log->out('FINISH');
    }

    return \@errors;
}


1;
