package Intapi::BannerLand;

=head1 NAME

Intapi::BannerLand

=cut

use Direct::Modern;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log;
use Yandex::Validate qw/is_valid_id/;
use Yandex::HashUtils qw/hash_kmap hash_kgrep/;

use PrimitivesIds qw/get_orderid2cid/;

use constant MAX_ITEMS_TO_UPDATE => 500;

=head2 new
=cut

sub new {
    return bless {};
}

=head2 setStatusForAdGroups

Установка statusBlGenerated в таблицах adgroups_[dynamic/performance].
Принимает структуру, вида: {cid1 => {gid1 => "empty", gid2 => "non-empty"}, ...}

Пример вызова:
    perl -Mmy_inc=for,protected -MIntapi::BannerLand -lE 'Intapi::BannerLand->setStatusForAdGroups({123 => {456 => "non-empty"}})'

=cut

sub setStatusForAdGroups {
    my ($self, $params, $procedure, @extra_args) = @_;

    state $supported_adgroup2table = {
        dynamic     => 'adgroups_dynamic',
        performance => 'adgroups_performance',
    };

    my $log = Yandex::Log->new(
        log_file_name => 'BannerLand.log',
        date_suf      => "%Y%m%d",
        msg_prefix    => "[$$][setStatusForAdGroups]",
        lock          => 1,
    );
    $log->out({input => $params});

    my ($response, @error_items);

    eval {
        my (@adgroup_ids, %items_to_update, @errors);

        # Проверим переданную структуру
        for my $cid (keys %$params) {
            if (!is_valid_id($cid)) {
                push @errors, "Found invalid CampaignId: $cid";
                push @error_items, {cid => $cid, adgroup_id => $_} for keys %{$params->{$cid}};
                delete $params->{$cid};
                next;
            }

            for my $adgroup_id (keys %{$params->{$cid}}) {
                if (!is_valid_id($adgroup_id)) {
                    push @errors, "Found invalid AdGroupId: $adgroup_id (CampaignId: $cid)";
                    push @error_items, {cid => $cid, adgroup_id => $adgroup_id};
                    delete $params->{$cid}->{$adgroup_id};
                    next;
                }

                push @adgroup_ids, $adgroup_id;
            }
        }

        my $adgroup_by_id = get_hashes_hash_sql(PPC(pid => \@adgroup_ids), [
            "SELECT g.pid, g.cid, g.adgroup_type FROM phrases g",
            WHERE => {pid => SHARD_IDS, adgroup_type => [keys %$supported_adgroup2table]},
        ]);

        for my $cid (keys %$params) {
            for my $adgroup_id (keys %{$params->{$cid}}) {
                my $adgroup = $adgroup_by_id->{$adgroup_id};
                my $status = $params->{$cid}->{$adgroup_id};

                if (!defined $adgroup) {
                    push @errors, sprintf("Cannot find AdGroup $adgroup_id (of type: %s)", join('/', sort keys %$supported_adgroup2table));
                    push @error_items, {cid => $cid, adgroup_id => $adgroup_id};
                    next;
                }
                if ($adgroup->{cid} != $cid) {
                    push @errors, "AdGroup $adgroup_id inside CampaignId $cid but actual CampaignId is ".($adgroup->{cid});
                    push @error_items, {cid => $cid, adgroup_id => $adgroup_id};
                    next;
                }
                if ($status !~ /^(?:non\-)?empty$/) {
                    push @errors, "AdGroup $adgroup_id has unknown status: $status";
                    push @error_items, {cid => $cid, adgroup_id => $adgroup_id};
                    next;
                }

                # Разделим статусы для записи в БД по типу группы
                push @{$items_to_update{ $adgroup->{adgroup_type} }}, {
                    cid => $cid,
                    adgroup_id => $adgroup_id,
                    statusBlGenerated => $status eq 'empty' ? 'No' : 'Yes'
                };
            }
        }

        $log->out({errors => \@errors}) if @errors;

        while (my ($adgroup_type, $items) = each %items_to_update) {
            # with_undef_shard не нужен, т.к. выше уже отфильтровали входные данные
            for my $chunk (sharded_chunks(pid => $items, by => sub { $_->{adgroup_id} }, chunk_size => MAX_ITEMS_TO_UPDATE)) {
                my ($shard, $shard_items) = ($chunk->{shard}, $chunk->{pid});

                my %values;
                $values{$_->{adgroup_id}}->{statusBlGenerated} = $_->{statusBlGenerated} for @$shard_items;

                eval {
                    do_mass_update_sql(PPC(shard => $shard), $supported_adgroup2table->{$adgroup_type}, 'pid', \%values);
                    1;
                } or do {
                    $log->out({"Cannot update items in shard $shard: $@" => $shard_items});
                    push @error_items, @$shard_items;
                    next;
                };
            }
        }

        1;
    } or do {
        $log->out({"An error occured" => $@});
        $response = {success => 0};
        # При "success: 0" error_items не нужен, т.к. следует перепослать всю очередь
        @error_items = ();
    };

    $response //= {success => 1};
    if (@error_items) {
        push @{$response->{error_items}->{ $_->{cid} }}, $_->{adgroup_id} for @error_items;
    }

    return $response;
}

=head2 setStatusForAdGroupsWithOrderID

    Аналог setStatusForAdGroups, но вместо cid - OrderID

=cut
sub setStatusForAdGroupsWithOrderID {
    my ($self, $params, $procedure, @extra_args) = @_;
    my $orderid2cid = get_orderid2cid(OrderID => [ keys %$params ]);
    
    my $modified_params = hash_kmap {$orderid2cid->{$_}} hash_kgrep {$orderid2cid->{$_}} $params;
    
    my $ret = setStatusForAdGroups($self, $modified_params, $procedure, @extra_args);
    
    if ($ret->{error_items}) {
        my $cid2orderid = +{ reverse % $orderid2cid };
        $ret->{error_items} = hash_kmap {$cid2orderid->{$_}} $ret->{error_items};
    }
    
    for my $orderid (grep {!$orderid2cid->{$_}} keys %$params) {
        $ret->{error_items}->{$orderid} = [ keys %{$params->{$orderid}} ];
    }
    
    return $ret;
}

1;
