package Intapi::BsFront;

=head1 NAME

    Intapi::BsFront

=head1 DESCRIPTION

=cut

use Direct::Modern;

use List::MoreUtils qw/any uniq/;
use Yandex::ListUtils qw/chunks xuniq/;
use JSON;
use Try::Tiny qw/try catch/;

use Direct::Creatives qw/sync_creatives/;
use Direct::Feeds;
use JavaIntapi::SaveTurbolandings;

use PrimitivesIds qw/get_clientid get_uid/;
use RBACDirect qw/rbac_get_subclients_uids rbac_get_agencies_of_client rbac_get_subclients_rights rbac_manager_can_create_camp/;
use RBACElementary qw/rbac_get_chief_rep_of_client rbac_get_chief_rep_of_client_rep rbac_get_chief_rep_of_agency_rep rbac_who_is/;
use Yandex::Validate qw/is_valid_id/;
use User qw//;
use Client qw/get_client_data/;

=head2 new

=cut 

sub new {
    bless {};
}

=head2 auth

получение прав пользователя

Параметры:
    operator_uid - passport uid человера, работающего в интерфейсе
    client_login - login клиента, для которого создаёт баннеры оператор, может отсутствовать, если operator создаёт баннер для себя
    creative_id - id редактируемого баннера. Необязательный параметр, нужен только при редактировании
    creatives - [ { id => $id1 }, { id => $id2 } ] -- массив, где id -- creative_id
        нельзя указать одновременно creative_id и creatives

Результат:
    availableActions - массив из прав, которыми обладает operator по отношению к client/creative_id
        допустимые права: creative_create, creative_edit, creative_delete, creative_get
    availableTemplates - список идентификаторов доступных operator-у шаблонов

=cut

sub auth {
    my ($self, $params) = @_;
    
    my $error = validate_auth_params($params);
    die {message => $error} if $error;

    my @creative_ids;
    if ($params->{creative_id}) {
        push @creative_ids, $params->{creative_id};
    }
    if ($params->{creatives}) {
        if (@creative_ids) {
            die { message => 'params creative_id/creatives are ambiguous' };
        }
        @creative_ids = map { $_->{id} } @{$params->{creatives}};
    }
    
    my $client_uid = $params->{client_login} ? get_uid(login => $params->{client_login}) : $params->{operator_uid}; 
    return get_available_operations($client_uid, $params->{operator_uid}, \@creative_ids);
}

=head2 change_notify

колбэк, вызываемый при изменении баннера

Параметры:
    при создании креатива: operator_uid, client_login, creatives (список креативов) - аналогично auth
    при изменении креатива(ов) только creatives (может происходить при модерации креатива)
    
Пример запроса:
    curl -i -X POST "http://9700.beta2.direct.yandex.ru/jsonrpc/BsFront" -d '{"method":"change_notify","params":{"operator_uid":194540966,"client_login":"ya-imperiatechno2012","creatives":[{"id":1109589},{"id":1109597}]}}'

Пример ответа:
    {
      "jsonrpc": "2.0",
      "id": null,
      "result": [
        {"id":234235, "result":1}
        {"id":234234, "result":0, "errorMessage": "creative not found"}
        {"id":234236, "result":1, "warningMessage": "creative was skipped (actual version status - in editing)"}
        {"id":234234, "result":0, "errorMessage": "creative is wrong"}
        ....
      ]
    }
    id - INT, номер креатива
    result - INT 0/1, баннер сохранён/изменен в базу директа
    errorMessage - человеко понятная ошибка по обработке конкретного креатива

=cut

sub change_notify {
    my ($self, $params) = @_;
    
    die {message => 'missing required param creatives(must be array)'} if !exists $params->{creatives} || ref $params->{creatives} ne 'ARRAY'; 
    if (any { !is_valid_id($_->{id}) } @{$params->{creatives}}) {
        die {message => 'incorrect param creatives(must be array of positive int)'}
    }

    my $creatives = [xuniq { $_->{creative_id} } map { {creative_id => $_->{id}} } @{$params->{creatives}}];
    Direct::Creatives::fill_in_clients($creatives);
    # update/create from BS interafce
    if ($params->{operator_uid}) {
        
        my $error = validate_auth_params($params);
        die {message => $error} if $error;
        
        my $client_uid = $params->{client_login} ? get_uid(login => $params->{client_login}) : $params->{operator_uid};
        my $client_id = get_clientid(uid => $client_uid);

        my $perf_creatives_count = Client::get_count_perf_creatives($client_id);
        my $new_creatives = [grep { !$_->{client_id} } @$creatives];
        my $count_limit = $Settings::DEFAULT_PERFORMANCE_CREATIVE_COUNT_LIMIT;
        die {message => "not allowed more $count_limit performance creatives"}
            if ($perf_creatives_count + scalar @$new_creatives) > $count_limit;

        my $operations = get_available_operations(
                            $client_uid, $params->{operator_uid},
                            [map { $_->{creative_id} } grep { $_->{client_id} } @$creatives]);
        
        die {message => "operator not allowed create/edit creatives"} unless any {/(create|edit|delete)$/} @{$operations->{availableActions}};

        $creatives = [grep { $_->{client_id} == $client_id } map {
            $_->{client_id} = $client_id if $_->{client_id} == 0;
            $_;
        } @$creatives];
    }
    
    $creatives = [grep { $_->{client_id} > 0 } @$creatives]; 
    my $synced_creatives = sync_creatives($creatives);
    my %is_synced_creative = map { $_ => undef } @$synced_creatives;

    my @result;

    return [map {+{
        id => $_->{id},
        exists $is_synced_creative{$_->{id}}
            ? (result => 1)
            : (result => 0, errorMessage => 'creative not found') # текст на данный момент совпадает, но можно разделить. 
                                                                  # Сюда попадают, например, креативы с одним СlientID, при указанном в params - другим.

    }} @{$params->{creatives}}] ;
}

=head2 turbolanding_change_notify

callback, вызываемый при создании/изменении турболендинга

Параметры:
    "operator_uid": id,
    "client_id":    id,
    "turbolanding": {
        "id": id,
        "name": string,
        "url": string,
        "counters": [{"id": id, "goals":[id1, id2, ...]}, ...]
    }

    
Пример запроса:
    curl -i -X POST "http://9206.beta2.direct.yandex.ru/jsonrpc/BsFront"\
    -d '{"method":"turbolanding_change_notify","params":{"operator_uid":194540966,"client_id":2369864,"turbolanding":{"id":222,"url":"https://yandex.ru/turbo","name":"test","counters":[{"id":555,"goals":[11,22,33]}]}}}'
   
Пример ответа:
    {
      "result": 0,
      "errorMessage":"access denied"
    }

=cut

sub turbolanding_change_notify {
    my ($self, $params) = @_;
    
    my ($operator_uid, $client_id, $turbolanding) = @$params{qw/operator_uid client_id turbolanding/};
    return JavaIntapi::SaveTurbolandings
        ->new(client_id => $client_id, operator_uid => $operator_uid, turbolandings => [$turbolanding])
        ->call();
}

=head2 validate_auth_params

Проверка валидности параметров метода auth

=cut

sub validate_auth_params {
    my ($params) = @_;
    
    return 'incorrect param operator_uid(must be positive int)' unless is_valid_id($params->{operator_uid});
    return 'incorrect param creative_id(must be positive int)' if exists $params->{creative_id} && !is_valid_id($params->{creative_id});
    
    if (exists $params->{creatives}
        && (ref $params->{creatives} ne 'ARRAY' || any { !is_valid_id($_->{id}) } @{$params->{creatives}})) {
        return 'incorrect param creatives(must be array of positive int)'
    }
    
    if (exists $params->{client_login}) {
        return sprintf q[client login '%s' not found], $params->{client_login} unless get_uid(login => $params->{client_login});
    }
    
    return sprintf "operator uid %s not found", $params->{operator_uid} unless User::is_user_exists($params->{operator_uid});
    
    return undef;
}

=head2 get_available_operations($client_uid, $operator_uid, $creative_ids)

Операции доступные для выполнения $operator_uid над креативами $creative_ids клиента $client_uid

Результат:
    {
        availableActions => [''],
        availableTemplates => [123, 456]
    }

=cut

sub get_available_operations {
    my ($client_uid, $operator_uid, $creative_ids) = @_;

    # Если запрашивается доступ фрилансера к данным клиента - он имеет те же права, что клиент
    if (RBACDirect::rbac_is_related_freelancer(undef, $operator_uid, $client_uid)){
        $operator_uid = $client_uid;
    }

    my $rbac = _get_rbac($operator_uid);
    $client_uid = rbac_get_chief_rep_of_client_rep($client_uid);

    my $creatives_found = 1;
    my $existing_templates = {};
    if (@$creative_ids) {
        $existing_templates = Direct::Creatives::get_existing_template_ids(uid => $client_uid, creative_ids => $creative_ids);
        $creatives_found = $existing_templates->{total} < @$creative_ids ? 0 : 1;
    }

    my $empty_response = {availableActions => [], availableTemplates => []};
    return $empty_response  if !$creatives_found;

    my $operator_has_access = _does_operator_have_access($operator_uid, $client_uid, rbac => $rbac);
    return $empty_response  if !$operator_has_access;

    my @availableActions = qw/creative_create creative_edit creative_delete creative_get/;
    my $role = rbac_who_is($rbac, $operator_uid);
    if ($role =~ /^(super|superreader|support|limited_support|(super)?media|(super)?placer)$/) {
        @availableActions = ('creative_get');
        push @availableActions, 'creative_create', 'creative_edit' if $role =~ /^(support|super)$/i;
        push @availableActions, 'creative_delete' if $role eq 'super';
    } elsif ($role eq 'client') {
        my $agency_uids = rbac_get_agencies_of_client($rbac, $client_uid);
        # there isn't client's freedom
        if (@$agency_uids) {
            my $is_super_subclient
                = get_client_data(get_clientid(uid => $client_uid), ['allow_create_scamp_by_subclient'])->{allow_create_scamp_by_subclient};
            for my $agency_uid ($is_super_subclient ? () : @$agency_uids) {
                my $agency_client_id = get_clientid(uid => $agency_uid); 
                my $rights = rbac_get_subclients_rights($rbac, $agency_uid, $client_uid)->{$agency_client_id};
                if ($rights->{isSuperSubClient}) {
                    $is_super_subclient = 1;
                    last;
                }
            }
            @availableActions = ('creative_get') unless $is_super_subclient;
        }
    }

    return {availableActions => [], availableTemplates => []} unless($operator_has_access && $creatives_found);

    my $client_id = get_clientid(uid => $client_uid);
    my @template_ids = uniq @{$existing_templates->{ids}}, @{Direct::Creatives::get_available_template_ids_by_role($role, $client_id)};

    return {
        availableActions => \@availableActions, 
        availableTemplates => \@template_ids,
    };    
}


sub _get_rbac {
    my ($operator_uid) = @_;
    return eval { RBAC2::Extended->get_singleton($operator_uid) } || die {message => "can't init RBAC"};
}


sub _does_operator_have_access {
    my ($operator_uid, $client_uid, %O) = @_;

    my $rbac = $O{rbac} || _get_rbac($operator_uid);
    my $role = rbac_who_is($rbac, $operator_uid);

    $client_uid = rbac_get_chief_rep_of_client_rep($client_uid);

    my $operator_has_access;
    if ($role eq 'agency') {
        my $agency_chief = rbac_get_chief_rep_of_agency_rep($operator_uid);
        my $agency_uids = rbac_get_agencies_of_client($rbac, $client_uid);
        $operator_has_access = any { $_ == $agency_chief } @$agency_uids;
    } elsif ($role eq 'manager') {
        $operator_has_access = rbac_manager_can_create_camp($rbac, $operator_uid, $client_uid);
    } elsif ($role =~ /^(super|superreader|support|limited_support|(super)?media|(super)?placer)$/) {
        $operator_has_access = rbac_who_is($rbac, $client_uid) eq 'client';
    } elsif ($role eq 'client') {
        $operator_has_access = rbac_get_chief_rep_of_client_rep($operator_uid) == $client_uid;
    }

    return $operator_has_access;
}


=head2 get_client_feeds

Ручка для получения списка фидов клиента с примерами offers из фида.

Параметры
    operator_uid - текущий паспортный uid пользователя
    client_login - для какого клиента выполняется запрос

Результат
    [{
      feed_id: 123,
      business_type: '', # токен типа бизнеса
      name: '', # имя фида для человека 
      offers: [
        {
          filter_name: '', # название условие (задает Боря, директ прозрачно транслирует) по которому BL отфильтровал "examples"
          examples: [{
            # здесь атрибуты оффера которые сможет распарсить BL (директ передает прозрачно)
          }]
        }
      ]
    }]

=cut

sub get_client_feeds {
    my ($self, $params) = @_;

    my $error = validate_auth_params($params);
    die {message => $error} if $error;

    my $operator_uid = $params->{operator_uid};

    my $client_uid = $params->{client_login}
        ? get_uid(login => $params->{client_login})
        : $operator_uid;
    
    # Если запрашивается доступ фрилансера к данным клиента - он имеет те же права, что клиент
    if (RBACDirect::rbac_is_related_freelancer(undef, $operator_uid, $client_uid)){
        $operator_uid = $client_uid;
    }

    return []  if !_does_operator_have_access($operator_uid, $client_uid);

    my $client_id = get_clientid(uid => $client_uid);

    my $feeds = Direct::Feeds->get_by($client_id)->items;
    my @result = map {{
            feed_id => $_->id,
            business_type => $_->business_type,
            name => $_->name,
            offers => from_json($_->offer_examples || '{}'),
        }} @$feeds;

    return \@result;
}


=head2 spawn_creatives

Массово создает директе креативы указанные во входных параметрах креативы.
На вход принимает id шаблона и список пар родительский:дочерний креатив.
Дочерние креативы пытаемся добавить в ту же группу, которой принадлежат родительские, если не можем - оставляем без привязки.

Параметры:
    creatives_pairs: пары креативов родительский:дочерний, разделенные ';':
        parent0:child0; parent1:child1; ... parentN:childN
    
Пример запроса:
    curl -i -X POST "http://9700.beta2.direct.yandex.ru/jsonrpc/BsFront" -d '{"method":"spawn_creatives","params":{"creatives_pairs":"123456:87654321; 2345678:98765432; 234236:654321"}}'

Пример ответа:
    {
      "jsonrpc": "2.0",
      "id": null,
      "result": [
        {"parent": 123456, "child":87654321, "result":1}
        {"child":98765432; "result":0, "errorMessage": "parent creative not found"}
        {"child":654321, "result":1, "warningMessage": "adgroup banners limit exceeded"}
        ....
      ]
    }
    id - INT, номер креатива
    result - INT 0/1, дочерний креатив создан в базе директа
    errorMessage - человекопонятная ошибка по обработке конкретного креатива

=cut

sub spawn_creatives {
    my ($self, $params) = @_;

    die {message => 'missing required param "creatives_pairs"'} unless exists $params->{creatives_pairs};
        
    my @creatives_pairs = split /\s*;\s*/, $params->{creatives_pairs};
    my $chunk_size = $params->{chunk_size} // scalar @creatives_pairs;

    my @result;
    foreach my $chunked_pairs ( chunks(\@creatives_pairs, $chunk_size) ){
        my %child2parent;
        my %errors;
        foreach my $pair (@$chunked_pairs){
            my $error = '';
            my ($parent, $child) = split /\s*:\s*/, $pair;
            do { $error .= 'Invalid id "'.$_.'"- must be positive int. ' unless is_valid_id($_) } foreach ($parent, $child);
            if ( $error ){
                $errors{$child} = $error;
                next;
            }
            $child2parent{$child} = $parent;
        }
       
        my $info = {};
        my $general_error;
        eval{
            $info = Direct::Creatives::create_child_creatives(\%child2parent);
            1;
        } || do { $general_error = $@ };
        
        foreach my $child_id (keys %child2parent){
            my $state = { child => $child_id, result => $info->{created}->{$child_id} ? 1 : 0};
            $state->{parent} = $child2parent{$child_id} if $child2parent{$child_id};

            if (defined $general_error) {
                $state->{fatal} = $general_error;
                push @result, $state;
                next;
            }

            my $error = $errors{$child_id} // $info->{errors}->{$child_id};
            $state->{errorMessage} = $error if $error;
            $state->{warningMessage} = $info->{warnings}->{$child_id} if $info->{warnings}->{$child_id};
            
            push @result, $state;
        }
    }
    return \@result;
}

1;
