package Intapi::CLegacy::News;

# curl -d '{"method": "get_news"}' https://8998.beta1.direct.yandex.ru/jsonrpc/CLegacyNews
# curl -d '{"method": "get_news", "params":{"lang":"en"}}' https://8998.beta1.direct.yandex.ru/jsonrpc/CLegacyNews

use strict;
use warnings;
use utf8;

sub new { bless {} }

use Carp;
use JSON;

use Yandex::DBTools;
use Settings;

use List::MoreUtils qw/ none /;
use Yandex::HashUtils qw/ hash_cut /;
use Yandex::TimeCommon;


# время жизни кеша в секундах
our %CACHE;
our $CACHE_TIMEOUT = 60;



=head2 get_news

title: получение списка последних новостей
description: получить песледние новости для указанного региона и языка

param:
  properties:
    region:
      type: string
      description: идентификатор региона
      default: ru
    lang:
      type: string
      description: идентификатор языка
      default: ru
    limit:
      type: integer
      description: количество новостей
      default: 3

result:
  type: array
  items:
    type: object
    properties:
      date:
        type: string
      content:
        type: array
        items:
          type: object
          properties:
            text:
              type: string
            link:
              type: string
              format: uri
          required: [ text ]
    required: [ date, context ]


=cut

sub get_news 
{
    my ($self, $param) = @_;

    # TODO: checks
    my $lang = $param->{lang} || 'ru';
    my $region = _get_feed_region($param->{region});

    die {code => 'InvalidFeed', error => "Unknown feed <$region-$lang>"}
        if !_is_valid_feed($region, $lang);

    my $key = "front_$region$lang";
    my $result = _get_cached_value( $key => sub {
            my $news_items = get_one_column_sql(PPCDICT, [
                    'SELECT data FROM adv_news_items',
                    WHERE => { region => $region, lang => $lang },
                    'ORDER BY ext_news_id DESC',
                    LIMIT => $Settings::NUM_OF_NEWS_ON_FRONT,
                ]);
            return [ map {from_json $_} @$news_items ];
        });
    return $result;
}


sub get_news_by_id  {
    my ($self, $param) = @_;

    my $lang = $param->{lang} || 'ru';
    my $region = _get_feed_region($param->{region});
    my $ext_news_id = $param->{ext_news_id};

#    return { result => 'fail' }  if !_is_valid_feed($region, $lang) || !$ext_news_id;
    die {code => 'InvalidFeed', error => "Unknown feed <$region-$lang>"}
        if !_is_valid_feed($region, $lang);

    my $key = "item_$region$lang$ext_news_id";
    my $result = _get_cached_value( $key => sub {
            my $news_json = get_one_field_sql(PPCDICT, [
                    'SELECT data FROM adv_news_items',
                    WHERE => { region => $region, lang => $lang, ext_news_id => $param->{ext_news_id} },
                    'ORDER BY ext_news_id',
                    LIMIT => 1,
                ]);
            return if !$news_json;
            return hash_cut from_json($news_json), qw/ date content /;
        });

    die {code => 'NewsItemNotFound', error => "No item <$ext_news_id> in feed <$region-$lang>"}
        if !$result;

    my $item = {
            region => $region,
            lang => $lang,
            ext_news_id => $ext_news_id,
            data => $result,
    };
    return $item;
}


=head2 get_news_regions_langs

title: Возвращает полный список новостей с группами языков, начиная с определенного ext_id
description: по указанному uid сгерерировать токен (строку), которая будет валидна в течении некоторого времени

param:
  properties:
    min_ext_news_id:
      type: string
      description: минимальный ext_news_id, с которого надо отдать новости.
    min_date:
      type: integer
      description: минимальная дата (в формате unixtime), с которой надо отдать новости

result: 
  properties:
    items:
      type: array
      description: массив из новостей
      items:
        type: object
        description: метаинформация о новости
        properties:
          ext_news_id:
            type: string
          region:
            type: string
          lang:
            type: string
        required: [ext_news_id, region, lang]
  reqired: [items]

=cut
sub get_news_regions_langs  {
    my ($self, $param) = @_;

    my $min_ext_news_id = delete $param->{min_ext_news_id};
    my $min_date = delete $param->{min_date};

    die {code => 'InvalidArgs', error => "Either <min_ext_news_id> or <min_date> param required"}
        if !$min_ext_news_id && !$min_date;

    # преобразуем в формат mysql и округляем до даты, чтобы лучше кешировалось
    my $min_date_mysql = $min_date ? mysql_round_day(unix2mysql($min_date)) : undef;

    my $key = join q{_}, map {$_ // q{}} ("nrl", $min_ext_news_id, $min_date_mysql);
    my $result = _get_cached_value($key => sub {
            return get_all_sql(PPCDICT, [
                    'SELECT ext_news_id, lang, region
                       FROM adv_news_items',
                     WHERE => {
                          ($min_ext_news_id ? (ext_news_id__ge => $min_ext_news_id) : ()),
                          ($min_date_mysql ? (news_date__ge => $min_date_mysql) : ()),
                      },
                    'ORDER BY ext_news_id ASC',
                ]);
        });
    return $result;
}


sub _get_feed_region {
    my ($region) = @_;
    $region ||= 'ru';
    $region = $Settings::NEWS_REGION_ALIAS{$region} || $region;
    return $region;
}


sub _is_valid_feed {
    my ($region, $lang) = @_;

    my $feed_info = $Settings::NEWS_LANG_REGIONS{$lang};
    return 0  if !$feed_info;
    return 0  if none { $region eq $_ } @{$feed_info->{region}};

    return 1;
}


sub _get_cached_value {
    my ($key, $renew_sub) = @_;

    if ( $CACHE{$key} && time() >= $CACHE{$key}->{time} ) {
        delete $CACHE{$key};
    }

    $CACHE{$key} //= { time => time()+$CACHE_TIMEOUT, value => $renew_sub->() };
    return $CACHE{$key}->{value};
}


1;
