package Intapi::CampaignInfo;

# $Id$

=head1 NAME


=head1 DESCRIPTION


=cut

use Direct::Modern;

use List::MoreUtils qw/uniq any/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate;

use Settings;
use CampaignQuery;
use Primitives qw//;

=head2 new
=cut

sub new
{
    bless {};
}

=head2 getCampaignManager

По cid-у отдаёт менеджера кампании (или агентства)

На входе:

    { cids => [ $cid1, $cid2, ... ] }

На выходе:

    [
        {
            cid => $cid1,
            manager => { uid => , email => , login => , fio => }
        },
        {
            cid => $cid2,
            ag_manager => { uid => , email => , login => , fio => }
        },
        ...
    ]

=cut

sub getCampaignManager
{
    my ($self, $params, $procedure, @extra_args) = @_;

    my $cids = $params->{cids};
    my $servicing_info = Primitives::get_servicing_info_by_cids($cids);

    my @agency_ids = uniq grep {$_} map {$_->{AgencyID}} values %$servicing_info;
    my $agency_manager = get_hash_sql( PPC(ClientID => \@agency_ids), [
        'SELECT ClientID, primary_manager_uid FROM clients',
        WHERE => { ClientID => SHARD_IDS },
    ]);

    my @manager_uids = uniq grep {$_} (
        ( map {$_->{ManagerUID}} values %$servicing_info ),
        ( values %$agency_manager ),
    );
    my $manager_info = get_hashes_hash_sql( PPC(uid => \@manager_uids), [
        'SELECT uid, login, fio, email FROM users',
        WHERE => { uid => SHARD_IDS },
    ]);

    my @result;
    for my $cid (@$cids) {
        my $servicing = $servicing_info->{$cid};
        my $subresult = { cid => $cid };
        if ( my $uid = $servicing->{ManagerUID} ) {
            $subresult->{manager} = $manager_info->{$uid};
        }
        if ( my $uid = $servicing->{AgencyID} && $agency_manager->{$servicing->{AgencyID}} ) {
            $subresult->{ag_manager} = $manager_info->{$uid};
        }
        next if !$subresult->{manager} && !$subresult->{ag_manager};
        push @result, $subresult;
    }

    return \@result;
}


=head2 getCampaignInfo

    По списку номеров кампаний или баннеров (не более 1000) получить информацию 
    по кампаниям.
    
    На входе:
    {'args': {'cids':[123,234,346], 'fields':['uid','ClientID']}}
    или в случае поиска по баннерам:
    {'args': {'bids':[123,234,346], 'fields':['uid','ClientID']}}
    
    На выходе (порядок не гарантирован):
    {'result': [{'cid': 123, 'error':'not_found'}, {'cid':234, 'uid':12343, 'ClientID': 23454}, ...]
    или, соответственно:
    {'result': [{'bids':[346], 'error':'not_found'}, {'cid':234, 'bids':[123,234], 'uid':12343, 'ClientID': 23454}, ...]

    curl -s 'http://8402.beta1.direct.yandex.ru/jsonrpc/CampaignInfo' -d '{"method":"getCampaignInfo","params":{"args": {"cids":[88700, 768],"fields":["uid", "login", "ClientID"]}}}' | json_xs
    curl -s 'http://8402.beta1.direct.yandex.ru/jsonrpc/CampaignInfo' -d '{"method":"getCampaignInfo","params":{"args": {"bids":[88700, 768],"fields":["uid", "login", "ClientID"]}}}' | json_xs

=cut
{
my %AVAILABLE_FIELDS = map {$_ => 1} qw/cid OrderID ClientID/;
sub getCampaignInfo {
    my ($self, $params) = @_;
    # проверка аргументов
    if (!defined $params->{args} || ref($params->{args}) ne 'HASH') {
        return {error => {code => "INCORRECT_ARGS", description => "No args"}};
    }
    my $cids = delete $params->{args}->{cids};
    my $bids = delete $params->{args}->{bids};
    if (defined $bids && defined $cids) {
        return {error => {code => "INCORRECT_ARGS", description => "Both bids and cids cannot be used in one request or omitted"}};
    }
    if (defined $cids) {
        if (ref($cids) ne 'ARRAY' || any {!is_valid_id($_)} @$cids) {
            return {error => {code => "INCORRECT_ARGS", description => "Argument cids is incorrect or absent"}};
        } elsif (@$cids > 1000) {
            return {error => {code => "INCORRECT_ARGS", description => "Argument cids contains more than 1000 elements"}};
        }
    } elsif (defined $bids) {
        if (ref($bids) ne 'ARRAY' || any {!is_valid_id($_)} @$bids) {
            return {error => {code => "INCORRECT_ARGS", description => "Argument bids is incorrect or absent"}};
        } elsif (@$bids > 1000) {
            return {error => {code => "INCORRECT_ARGS", description => "Argument bids contains more than 1000 elements"}};
        }
    } else {
        return {error => {code => "INCORRECT_ARGS", description => "Both bids and cids cannot be omitted"}};
    }

    my $fields = delete $params->{args}->{fields};
    if (!defined $fields || ref($fields) ne 'ARRAY') {
        return {error => {code => "INCORRECT_ARGS", description => "Argument fields is incorrect or absent"}};
    } elsif (my @incorrect_fields = grep {!defined $_ || !exists $AVAILABLE_FIELDS{$_}} @$fields) {
        return {error => {code => "INCORRECT_ARGS", description => "Argument fields is incorrect: unsupported fields: ".join(', ', @incorrect_fields)}};            
    }

    # Если поиск осуществляется по номерам баннеров, получаем их кампании
    my %cid2bids;
    my @unfound_bids;
    if (defined $bids) {
        my $banners = get_all_sql(PPC(bid => $bids), [
            "SELECT b.bid, p.cid FROM banners b JOIN phrases p ON p.pid = b.pid", where => {"b.bid" => SHARD_IDS}
        ]);
        for my $banner (@$banners) {
            push @{ $cid2bids{ $banner->{cid} } }, $banner->{bid};
        }
        $cids = [ keys %cid2bids ];
        
        # Сохраняем список баннеров, которые мы не нашли
        my @found_bids = map {$_->{bid}} @$banners;
        my %all_bids = map {$_ => 1} @$bids;
        delete @all_bids{@found_bids};
        @unfound_bids = keys %all_bids;
    }

    # получаем данные
    push @$fields, 'cid' if !grep {$_ eq 'cid'} @$fields;
    my %camps_info = map {$_->{cid} => $_}
        @{CampaignQuery->get_campaign_data_multi({cid => $cids}, $fields)};

    # формируем ответ
    my @result;
    for my $cid (uniq @$cids) {
        if (my $camp = $camps_info{$cid}) {
            if (exists $camp->{OrderID} && !$camp->{OrderID}) {
                $camp->{OrderID} = undef;
            }
            if (defined $bids) {
                $camp->{bids} = $cid2bids{$cid};
            }
            push @result, $camp;
        } else {
            push @result, {cid => $cid, error => 'not_found'};
        }
    }
    
    if (@unfound_bids) {
        push @result, {bids => \@unfound_bids, error => 'not_found'}
    }

    return {result => \@result};
}
}

1;
