package Intapi::CampaignStatistics;

=head1 NAME

    Intapi::CampaignStatistics

=head1 DESCRIPTION

=cut

use Direct::Modern;
use Settings;
use Yandex::I18n;

use Stat::ReportMaster;
use Stat::Tools;

use Yandex::Validate qw/is_valid_id is_valid_date/;

=head2 handler

    Получение статистики по кампании

    Параметры:
        uid - uid клиента
        UID - uid оператора
        report_options - настройки отчета, может содержать следующие поля:
            cid - id кампании, по которой нужно построить отчет
            group_by - список полей, по которым группируем строки отчета
            date_from - начало периода, за который строим отчет
            date_to - начало периода, за который строим отчет
            group_by_date - режим группировки строк отчета по дате
            columns - список полей, которые нужно вернуть в отчете

=cut

sub handler {
    my ($r, $multiform) = @_;
    my $request = JSON::from_json($r->content, {utf8 => 1});

    Yandex::I18n::init_i18n($request->{locale});

    my $error = validate($request);
    if ($error) {
        return {text => $error, code => 400};
    }

    my $report_options = $request->{report_options};

    Stat::ReportMaster::add_report_options_defaults($report_options, no_paging => 1, no_default_columns => 1);

    my $stat_data = {};
    my $too_much_statistics_error;
    my $available_cids;

    my $UID = $request->{UID};
    my $uid = $request->{uid};
    ($stat_data, $too_much_statistics_error, $available_cids) = eval {
        Stat::ReportMaster::get_stat_report($UID, $uid, $report_options, dont_check_camp_type_by_role => 1);
    };
    if ($@) {
        die $@;
    }
    if ($too_much_statistics_error) {
        Stat::Tools::log_report_master_heavy_request({error => $too_much_statistics_error, format => 'default'}, $report_options);
        die "Too much statistics error";
    }

    return {text => JSON::to_json($stat_data), code => 200};
}

=head2 validate

    Валидирует запрос.
    Возвращает ошибку в строковом представлении, если валидация успешна - пустую строку.

=cut

sub validate {
    my ($request) = @_;

    my $error = '';

    if (!is_valid_id($request->{uid})) {
        $error .= "uid is not valid id; ";
    }
    if (!is_valid_id($request->{UID})) {
        $error .= "UID is not valid id; ";
    }
    $error .= validate_report_options($request->{report_options});

    return $error;
}

=head2 validate_report_options

    Валидирует настройки отчета.
    Возвращает ошибку в строковом представлении, если валидация успешна - пустую строку.

=cut

sub validate_report_options {
    my ($report_options) = @_;

    if (!defined $report_options) {
        return "report_options field is not set; ";
    }
    if (ref $report_options ne 'HASH') {
        return "report_options field has wrong format; ";
    }

    my $error = '';

    if (!is_valid_id($report_options->{cid})) {
        $error .= "cid is not valid id; ";
    }
    if (!is_valid_date($report_options->{date_from})) {
        $error .= "date_from is not valid date; ";
    }
    if (!is_valid_date($report_options->{date_to})) {
        $error .= "date_to is not valid date; ";
    }

    return $error;
}

1;

