package Intapi::CampaignsSumsForBS;

=head1 NAME

    jsonrpc/CampaignsSumsForBS

=head1 DESCRIPTION

    Экспорт сумм зачисленного на кампаниях для аудита БК
    https://jira.yandex-team.ru/browse/DIRECT-32677

=cut

use Direct::Modern;

use Yandex::Log;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::ListUtils qw/xflatten/;
use Yandex::DBTools qw/get_all_sql/;
use Yandex::DBShards qw/foreach_shard/;
use Yandex::Validate qw/is_valid_id/;

use BS::Export ();
use Settings;
use Campaign::Types;


sub new
{
    return bless {};
}

=head2 get

    Принимает список номеров кампаний и отдаёт зачисленные деньги

    wget -q -O - 'http://9702.beta2.direct.yandex.ru/jsonrpc/CampaignsSumsForBS?method=get&params={"cid":[263,8346816,666,9999999999]}' | json_xs

=cut

sub get
{
    my ($self, $params, @other) = @_;

    my $log = Yandex::Log->new(
        log_file_name => 'CampaignsSumsForBS.log',
        date_suf      => "%Y%m%d",
        msg_prefix    => "[$$]",
    );
    $log->out($params);

    if (ref $params eq 'HASH') {
        my $cids = $params->{cid};
        $log->die("No cids given") unless $cids;
        $cids = [$cids] unless ref($cids);
        my %uniq_cids;
        for my $cid (xflatten($cids)) {
            if (is_valid_id($cid)) {
                $uniq_cids{$cid} = undef;
            } else {
                $log->die('Invalid cid: ' . $cid // 'undef');
            }
        }

        my @result;
        foreach_shard cid => [keys(%uniq_cids)], chunk_size => 20_000, sub {
            my ($shard, $chunk) = @_;
            my $data_chunk = get_all_sql(PPC(shard => $shard), [
                    'SELECT c.cid, c.ClientID, c.statusBsSynced, c.type',
                    $BS::Export::SQL_SELECT_FIELDS_FOR_EXTRACT_SUM,
                    'FROM campaigns c',
                    'LEFT JOIN campaigns_multicurrency_sums cms ON cms.cid = c.cid',
                    'LEFT JOIN users u ON u.uid = c.uid',
                    'LEFT JOIN clients_options clo ON clo.ClientID = u.ClientID',
                    'LEFT JOIN currency_convert_queue ccq ON ccq.ClientID = u.ClientID',
                    'LEFT JOIN client_currency_changes ccc ON ccc.ClientID = u.ClientID',
                    'LEFT JOIN wallet_campaigns wwc ON wwc.wallet_cid = IF(c.type = "wallet", c.cid, c.wallet_cid)',
                    WHERE => {
                        'c.cid__int' => $chunk,
                        'c.statusEmpty' => 'No',
                    },
                    'ORDER BY c.cid',
            ]);
            for my $db_row (@$data_chunk) {
                my $result_row = hash_cut($db_row, qw/cid type/);
                $result_row->{StatusActivating} = $db_row->{statusBsSynced} eq 'Yes' ? 'Yes' : 'Pending';
                $result_row->{CurrencyISOCode} = BS::Export::currency2bsisocode($db_row->{currency} || 'YND_FIXED');

                if (Campaign::Types::is_wallet_camp(type => $db_row->{c_type})
                    && $db_row->{auto_overdraft_lim} != 0)
                {
                    my $sum_debts = WalletUtils::get_sum_debts_for_wallets([ $db_row->{cid} ]);
                    $db_row->{wallet_sum_debt} = $sum_debts->{$db_row->{cid}} // 0;
                }

                my $tmp_row = {};
                my $client_info = {
                    clientID            => $db_row->{ClientID},
                    debt                => $db_row->{client_debt},
                    overdraft_lim       => $db_row->{overdraft_lim},
                    auto_overdraft_lim  => $db_row->{auto_overdraft_lim},
                    statusBalanceBanned => $db_row->{statusBalanceBanned}
                };
                BS::Export::extract_sum($tmp_row, $db_row, $db_row->{type}, $client_info);
                
                $result_row->{sum} = $result_row->{CurrencyISOCode} == -1 ? $tmp_row->{SUM} : $tmp_row->{SUMCur};

                push @result, $result_row;
            }
        };
        %uniq_cids = ();

        return \@result;
    } else {
        $log->die("Need hash");
    }
}

1;
