package Intapi::ClientData;

=head1 NAME

    jsonrpc/ClientData

=head1 DESCRIPTION

    Отдаёт данные по клиенту для Метрики

=cut

use Direct::Modern;

use Settings;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate qw(is_valid_id);
use Yandex::HashUtils qw(hash_merge hash_cut);
use Yandex::ListUtils;
use Yandex::Log;

use RBAC2::Extended;
use RBACElementary;

use constant DB_CHUNK_SIZE => 10_000;

=head2 new

=cut

sub new
{
    return bless {};
}

=head2 get_by_ClientID

    Отдаёт по списку ClientID хеш с данными по клиенту

    Принимает именованные параметры:
     - mark_chief_reps, по которому в ответе будет помечен главный представитель агентства/клиента.
     - fields - поля, которые выводить, допустимы login, FIO, по-умолчанию ['FIO']
    

    Практически брат-близнец ручки GetUidsByClientID

    $ curl -s 'http://9705.beta2.direct.yandex.ru/jsonrpc/ClientData' -d '{"method":"get_by_ClientID","params":{"client_ids":[15155],"mark_chief_reps":true,"fields":["login", "FIO"]}}' | json_xs
    {
       "jsonrpc" : "2.0",
       "id" : null,
       "result" : {
          "15155" : {
             "name" : null,
             "uids" : {
                "6138950" : {
                   "chief" : 1,
                   "FIO" : "Serj Берсенева"
                },
                "224490274" : {
                   "FIO" : "Анька Бакулина"
                }
             }
          }
       }
    }

=cut

sub get_by_ClientID
{
    my ($self, $params, $procedure, @extra_args) = @_;

    my $log = Yandex::Log->new(
        log_file_name => 'ClientData.log',
        date_suf      => "%Y%m%d",
        msg_prefix    => "[$$]",
        lock          => 1,
    );
    $log->out({get_by_ClientID => $params});

    my $fields = $params->{fields} || ['FIO'];
    unless ($fields && ref($fields) eq 'ARRAY' && !(grep {!/^(login|FIO)$/} @$fields)) {
        $log->die('Incorrect fields: ', $fields);
    }

    my $client_ids = $params->{client_ids};
    unless ($client_ids && ref($client_ids) eq 'ARRAY' && @$client_ids) {
        $log->die('Incorrect ClientIDs given', $client_ids);
    }
    for my $client_id (@$client_ids) {
        unless ( is_valid_id($client_id) ) {
            $log->die("Invalid ClientID: $client_id");
        }
    }

    my $result = {};
    foreach_shard ClientID => $client_ids, chunk_size => DB_CHUNK_SIZE, sub {
        my ( $shard, $chunk ) = @_;
        my $rows = get_all_sql(
            PPC( shard => $shard ),
            [
                "SELECT u.ClientID, u.uid, u.FIO, u.login, cl.name
                 FROM users u
                 LEFT JOIN clients cl ON u.ClientID = cl.ClientID",
                 WHERE => { 'u.ClientID' => $chunk }
            ]
        );

        foreach my $row (@$rows) {
            my $clres = $result->{ $row->{ClientID} } ||= {};
            $clres->{name} = $row->{name};
            $clres->{uids}->{ $row->{uid} } = hash_cut $row, @$fields;
        }
    };

    if ($params->{mark_chief_reps}) {
        my $rbac = eval { RBAC2::Extended->get_singleton(1) }
            or $log->die("Can't connect to RBAC: $@");

        my @client_ids = keys %$result;
        for my $portion (chunks \@client_ids, 1_000) {
            my $chiefs = hash_merge
                rbac_get_chief_reps_of_clients($portion),
                rbac_get_chief_reps_of_agencies($portion);
            for my $client_id (@$portion) {
                my $chief_uid = $chiefs->{$client_id};
                unless ($chief_uid) {
                    $log->warn("no chief for client <$client_id> found in rbac");
                    next;
                }
                $result->{$client_id}->{uids}->{$chief_uid}->{chief} = 1;
            }
        }
    }

    return $result;
}

1;
