package Intapi::Clients;

=pod

=encoding utf8

    $Id:$

=head1 NAME

    Intapi::Clients

=head1 SYNOPSIS

    curl -X PUT -d'{ "login": "yndx-geoctx-someuser", "name": "Ivan", "surname": "Ivanov", "country": 225, "currency": "RUB" }' https://10924.beta3.direct.yandex.ru/clients
    curl -X POST -d'{ "login": "yndx-geoctx-someuser", "name": "Ivan", "surname": "Ivanov", "country": 225, "currency": "RUB" }' https://10924.beta3.direct.yandex.ru/clients/create
    curl -X POST -d'{ "agency_login": "ra-trinet", "login": "yndx-int-someuser", "name": "Ivan", "surname": "Ivanov", "currency": "RUB" }' https://intapi.direct.yandex.ru/clients/create

=head1 DESCRIPTION

    Используется в регрессии в тестовых степах
    Сервис внутреннего API для создания прямых, сервисируемых и агентских
    клиентов. Создавать в директе можно только новых пользователей,
    пользователи уже существующие в пасспорте созданы не будут

=cut

use Direct::Modern;

use JSON;
use Router::Simple;
use Try::Tiny;

use Yandex::Balance;

use Client;
use Direct::Validation::Client;
use HashingTools;
use Intapi;
use Primitives;
use PrimitivesIds;
use RBACElementary;
use RBACDirect;
use Settings;
use TextTools;
use User;
use Yandex::Trace;
use Yandex::TVM2;

use constant ERROR_METHOD_NOT_FOUND => "MethodNotFound";
use constant ERROR_BAD_REQUEST => "BadRequest";
use constant ERROR_INTERNAL_FAILURE => "InternalServerError";
use constant ERROR_CAMPAIGN_CREATION_FAILURE => "CantCreateCampaign";

my $ROUTER = Router::Simple->new();

$ROUTER->connect(qr!/?! => { sub => \&clients_create }, { method => 'PUT' });
$ROUTER->connect('/create' => { sub => \&clients_create }, { method => 'POST'});

our $LOGGER = Intapi::request_logger('clients');

=head2 handler($r, $params)

    $r - Plack::Request
    $parameters - hashref multiform parameters

=cut

sub handler {
    my ($r, $parameters) = @_;

    my $json = JSON->new->utf8;

    my $path = $r->path_info;

    my $method_definition = $ROUTER->match($r->env)
        or return _error_log_and_return(404, ERROR_METHOD_NOT_FOUND, "method $path not found");

    my $method = delete $method_definition->{sub} or die "broken method definition";

    my ($content, $content_parsing_error);
    if ($r->content) {
        try {
            $content = $json->decode($r->content);
        } catch {
            $content_parsing_error = _error_log_and_return(400, ERROR_BAD_REQUEST, "Request body ins't valid JSON");
        };

        return $content_parsing_error if $content_parsing_error;
    }

    if ($content && ref $content ne 'HASH') {
        return _error_log_and_return(400, ERROR_BAD_REQUEST, "Request body must be a JSON object");
    }

    my ($response, $error, $error_code);
    try {
        $response = $method->({
            %{$content//{}},
            %$parameters,
            %$method_definition,
        });
    } catch {
        $error_code = 500;
        $error = _error(ERROR_INTERNAL_FAILURE, "Unexpected error, please contact Yandex.Direct development team");
        _log_out(status => $error_code, error => $error, exception => $_);
    };

    return $error_code
        ? { code => $error_code, json => $error }
        : ( exists $response->{code} ? $response : { json => $response } );
}

=head2 clients_create

    Parameters:
        Login - will be created in passport, only new users are ok
        name - first name
        surname - surname
        currency - currency name (RUB, CHF, EUR, KZT, TRY, UAH, USD, BYN)
        country - geo region id, (see geo_regions.pm or Direct.API5 dictionaries.get), will be ignored if creating client under agency

    Response:
        client_id - ClientID in billing system
        user_id - passport uid
        login - Login (normalized so could differ from request)

    With PUT, on beta
    curl -X PUT -d'{ "login": "yndx-int-someuser", "name": "Ivan", "surname": "Ivanov", "country": 225, "currency": "RUB" }' https://10924.beta3.direct.yandex.ru/clients

    With POST (different URI), on beta
    curl -X POST -d'{ "login": "yndx-int-someuser", "name": "Ivan", "surname": "Ivanov", "country": 225, "currency": "RUB" }' https://10924.beta3.direct.yandex.ru/clients/create

    For agency, in production,
    curl -X POST -d'{ "agency_login": "ra-trinet", "login": "yndx-int-someuser", "name": "Ivan", "surname": "Ivanov", "currency": "RUB" }' https://intapi.direct.yandex.ru/clients/create

    For manager, in production,
    curl -X POST -d'{ "manager_login": "yndx-mngr-full", "login": "yndx-int-someuser", "name": "Ivan", "surname": "Ivanov", "currency": "RUB" }' https://intapi.direct.yandex.ru/clients/create

=cut

sub clients_create {
    my $client = shift;

    if (ref $client ne 'HASH') {
        return _ebr("Parameters should be json object");
    }

    return _ebr("name can't be empty") unless $client->{name};
    return _ebr("surname can't be empty") unless $client->{surname};
    return _ebr("currency can't be empty") unless $client->{currency};
    return _ebr("agency_login and manager_login are mutually exclusive parameters")
        if $client->{agency_login} && $client->{manager_login};

    return _ebr("Login can't start with 'yndx' or 'yandex-team' prefixes, it's reserved for manually created internal users: DIRECT-64544") if $client->{login} =~ /^(yndx|yandex-team)/;

    my $login = normalize_login($client->{login}) or return _ebr("login can't be empty");
    if (my $uid = get_uid_by_login($login)) {
        return _ebr(qq!login $client->{login} already exists in passport with uid: $uid!);
    }

    my ($operator_uid, $agency_uid, $manager_uid, $agency_client_id);
    if (my $agency_login = $client->{agency_login}) {
        $agency_uid = get_uid_by_login($agency_login) or return _ebr("Agency $agency_login not found");
        return _ebr("$agency_login ain't agency") unless rbac_who_is(_rbac(), $agency_uid) eq 'agency';
        $operator_uid = $agency_uid;
        $agency_client_id = get_clientid(uid => $agency_uid);
        # у агентских клиентов не спрашиваем страну
        delete $client->{country} if $client->{country};
    } elsif (my $manager_login = $client->{manager_login}) {
        $manager_uid = get_uid_by_login($manager_login) or return _ebr("Manager $manager_login not found");
        return _ebr("$manager_login ain't manager") unless rbac_who_is(_rbac(), $manager_uid) eq 'manager';
        $operator_uid = $manager_uid;
    } else {
        $operator_uid = 1;
        return _ebr("country can't be empty") unless $client->{country};
    }

    my $vr = Direct::Validation::Client::check_country_and_currency({
            uid => $operator_uid,
            initial_country => $client->{country},
            initial_currency => $client->{currency}
        }, $agency_client_id, is_direct => 1);

    return _ebr("Country and currency validation failed: " . $vr->get_first_error_description()) unless $vr->is_valid;

    my ($passport_error, $created_uid, $password) = _create_user_in_passport($login, $client->{name}, $client->{surname});
    if ($passport_error) {
        return _error_log_and_return(500, ERROR_INTERNAL_FAILURE, $passport_error);
    }

    my $email = "$login\@yandex.ru";
    my $created_client_id = _create_client_in_balance({
        agency      => $agency_uid ? 'yes' : 'no',
        client_uid  => $created_uid,
        email       => $email,
        name        => qq~$client->{name} $client->{surname}~,
        currency    => $client->{currency},
        country_region_id => $client->{country},
    }, $operator_uid);


    create_update_user( $created_uid, {
        _login      => $login,
        ClientID    => $created_client_id,
        role        => 'client',
        email       => $email,
        UID         => $operator_uid,
        not_create_client_id_association => 1,
        hide_market_rating => 0,
        initial_currency => $client->{currency},
        initial_country => $client->{country},
    });
    API::ClientOptions::add($created_client_id, {'api_enabled' => 'No'});

    if ( $agency_uid ) {
        if ( my $err = rbac_create_agency_subclient(_rbac(), $agency_uid, $created_uid, 'commit') ) {
            _log_out(rbac_error => $err, agency_uid => $agency_uid);
            return _error_log_and_return(500, ERROR_INTERNAL_FAILURE, "Failed to create angecy client in RBAC");
        }
    } elsif ($manager_uid) {
        if ( my $err = rbac_create_manager_client(_rbac(), $manager_uid, $created_uid, 'commit') ) {
            _log_out(rbac_error => $err, manager_uid => $manager_uid);
            return _error_log_and_return(500, ERROR_INTERNAL_FAILURE, "Failed to create manager client in RBAC");
        }
    } else {
        if ( my $err = rbac_create_client(_rbac(), $created_uid, 'commit') ) {
            _log_out(rbac_error => $err);
            return _error_log_and_return(500, ERROR_INTERNAL_FAILURE, "Failed to client in RBAC");
        }
    }

    my %manager_or_agency_uid = $manager_uid ? ( manager_uid => $manager_uid )
        : $agency_uid ? ( agency_uid => $agency_uid ) : ();

    return {
        login => $login,
        user_id => $created_uid,
        client_id => $created_client_id,
        password => $password,
        %manager_or_agency_uid
    };
}

sub _create_user_in_passport { # returns ($error, $uid, $password)
    my ($login, $name, $surname) = @_;

    my $password = HashingTools::get_random_string(length=> 8, alphabets=>'wWd');

    my $ticket = eval { Yandex::TVM2::get_ticket($Settings::PASSPORT_TVM2_ID) } or die "Cannot get ticket for $Settings::PASSPORT_TVM2_ID: $@";
    my $result = Yandex::Passport::create_passport_login($login, $name, $surname, $password,
        ip => $ENV{'X-Real-IP'} || $ENV{REMOTE_ADDR}, tvm_ticket => $ticket);

    if ($result->{'status'} ne 'ok') {
        _log_out(passport_failure => $result);
        return "Creation in passport failed";
    }

    return (undef, $result->{uid}, $password);
}

sub _create_client_in_balance { # returns ($error, $new_clientid)
    my ($client_data, $operator_uid) = @_;

    if (create_client_id($client_data, $operator_uid)) {
        return "Can't create client in balance"
    }

    my $created_client_id = delete $client_data->{new_clientID};
    if (!create_client_id_association($client_data->{client_uid}, $created_client_id, $operator_uid)) {
        return "Can't create client association in balance";
    }

    return (undef, $created_client_id);
}

sub _error_log_and_return {
    my ($status, $error, $message) = @_;

    my $error_data = _error($error, $message);

    $LOGGER->out({ reqid => _trace_id(), status => $status, response => $error_data });

    return { code => $status, json => $error_data };
};

sub _trace_id { Yandex::Trace::trace_id() }

sub _error {
    my ($error, $message) = @_;
    my $error_data = {error => $error};
    $error_data->{message} = $message if $message;
    return $error_data;
}

sub _log_out {
    my %log_data = @_;
    $LOGGER->out({ reqid => Yandex::Trace::trace_id(), %log_data });
}

sub _ebr { # _error_log_and_return for ERROR_BAD_REQUEST shortcut
    my $detail = shift;
    return _error_log_and_return(400, ERROR_BAD_REQUEST, $detail);
}

sub _rbac {
    state $rbac = RBAC2::Extended->get_singleton(1);
    return $rbac;
}


1;

__END__
