package Intapi::DostupUserManagement;

# $Id$

=head1 NAME

    intapi/idm

=head1 DESCRIPTION

 DostupUserManagement - API для создания внутренних ролей

 http://wiki.yandex-team.ru/Upravljator/API
 https://jira.yandex-team.ru/browse/DIRECT-8857

=cut

use Direct::Modern;

use base qw/DoCmd::Base/;

use Settings;
use Rbac;
use RBACElementary;
use RBACDirect;
use RBAC2::Extended;
use User;
use Client;
use EnvTools;
use HttpTools qw/multiform2directform/;
use HashingTools qw/md5_base64ya hmac_sha256_hex/;
use Yandex::DBTools;
use Yandex::Log::Messages;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::Trace;

use Carp;
use JSON;


our $SLUG = 'direct'; # метка проекта
our $MAX_PASSPORT_LOGINS_ON_DOMAIN = is_beta() ? 30 : 10; # кол-во паспортных логинов на один доменный

our $TOKEN_HMAC_KEY = 'e687265f7911ab809623b05cf73fd44dbde786ca7d7e0480153625f3cd6cc15b';
our %VALID_TOKEN_HMACS = (
    '5493255188a3b2e6aaf60ce6580c68c0e4d2faf34d4b60a78461979b9d34044a' => 1, # to delete
    'a24e3f58be51df0d1999367cd8d7f9812a5264e9d97d836f43f7ef29321db46d' => 1,
    );

# роли которыми можно управлять через управлятор
our %IS_VALID_ROLE = (
    teamleader => 1
    , superteamleader => 1
    , internal_ad_admin => 1
    , internal_ad_manager => 1
    , internal_ad_superreader => 1
    , map {$_ => 1}
      grep {rbac_is_internal_user(undef, role => $_)}
      keys %RBACDirect::IS_BASE_ROLE
);


# вообще говоря, http-коды не очень нужны: они только пишутся в лог, а наружу всегда отдается 200
my $OK = 200;
my $SERVER_ERROR = 500;

#-----------------------------------------------------------
sub handler
{
    my ($r, $multiform) = @_;

    _deprecated();
    Yandex::I18n::init_i18n('ru'); # тексты ошибок выдаем на русском

    my $form = multiform2directform($multiform);
    my $log = Yandex::Log::Messages->new();
    my $ctx = {FORM => $form
               , r => $r
               , path_info => $r->path_info # имя метода забираем из path урла (/idm/method-name?param=value)
               , log => $log
              };

    my ($status, $data_result) = dostup_user_management($ctx);

    my $resp_text = '';
    if (ref($data_result)) {
        $resp_text = to_json($data_result, {pretty => is_beta() ? 1 : 0});
    } 

    my $log_data = {map {$_ => $form->{$_}} grep {$_ ne "token"} keys %$form};
    $ctx->{log}->out({method => $ctx->{method}, request => $log_data, status => $status, response => $data_result, IP => $r->address, reqid => Yandex::Trace::current_span_id()});

    return {text => $resp_text, content_type => "application/json; charset=utf-8"};
}

#-----------------------------------------------------------
sub dostup_user_management
{
    my $ctx = shift;

    my ($status, $data_result) = check_auth($ctx);

    if ($status == $OK) {
        eval {
            $ctx->{rbac} = get_rbac();
            ($status, $data_result) = $DoCmd::Base::cmds{ "dostup_" . $ctx->{method} }->($ctx);
            $data_result->{code} = 0 if ref($data_result) eq 'HASH' && ! defined $data_result->{code}; # код возврата заполняем только если результат в виде хеша
        };

        if ($@) {

            my $error_hash = {};

            if (ref($@) eq 'HASH') {
                $error_hash = hash_cut($@, qw/code warning error fatal/);
            } else {
                $error_hash = {code => 1, error => $@};
            }

            ($status, $data_result) = ($SERVER_ERROR, $error_hash);
        }
    }

    return ($status, $data_result);
}

#-----------------------------------------------------------

=head2 check_auth

  проверяем что запрос можно выполнить:

   - доступ только по https
   - в параметрах должен приходить валидный токен

   генерация токена, завязанного на ip:
   perl -Iprotected -ME -MIntapi::DostupUserManagement -e 'p Intapi::DostupUserManagement::get_auth_token($ARGV[0])' 127.0.0.1

=cut

sub check_auth
{
    my $ctx = shift;

    return ($SERVER_ERROR, {code => 1, fatal => 'внутренняя ошибка сервера: обращение возможно только по https'}) unless $ctx->{r}->scheme eq 'https';

    if (is_production()) {
        return ($SERVER_ERROR, {code => 1, fatal => 'внутренняя ошибка сервера: токен не найден'}) unless $ctx->{FORM}->{token};
        return ($SERVER_ERROR, {code => 1, fatal => 'внутренняя ошибка сервера: токен неверен'}) unless is_valid_token($ctx->{FORM}->{token}, $ctx->{r}->address);
    }

    return ($SERVER_ERROR, {code => 1, fatal => 'внутренняя ошибка сервера: не определен метод'}) unless defined $ctx->{path_info};
    my $method = $ctx->{path_info};
    # кажется эта две замена устарела, сейчас в path_info уже только метод
    $method =~ s|^/idm/||;
    $method =~ s|/||g;
    $method =~ s|-|_|g;
    return ($SERVER_ERROR, {code => 1, fatal => 'внутренняя ошибка сервера: метод не найден'}) unless $DoCmd::Base::cmds{"dostup_$method"};
    $ctx->{method} = $method;

    return ($OK, {});
}

=head2 is_valid_token

    Токены бывают завязанные на конкретный ip (они дыряво проверяются налету) и просто случайные (проверяются через hmac-sha1)

=cut
sub is_valid_token
{
    my ($token, $ip) = @_;

    my $hmac = hmac_sha256_hex($token, $TOKEN_HMAC_KEY);
    return $VALID_TOKEN_HMACS{$hmac} || $token eq get_auth_token($ip);
}

#-----------------------------------------------------------
sub get_auth_token
{
    my $ip = shift;

    return md5_base64ya($ip . $Settings::SECRET_PHRASE . "dostup-api");
}

#-----------------------------------------------------------
sub get_rbac
{
    my $rbac = eval { RBAC2::Extended->get_singleton(1) }
        or die {code => 1, error => "внутренняя ошибка сервера: can't initialise RBAC: $@"};

    return $rbac;
}

# -----------------------------------------------------------------------------

=head2 get_and_check_domain_login

  по пришедшему в форме доменному логину (login) находим все паспортные логины заведенные в директе

=cut

sub get_and_check_domain_login
{
    my $domain_login = shift() or die {code => 1, fatal => "domain login required"};

    my $logins = get_all_sql(PPC(shard => 'all'),
                             "select u.uid, login, domain_login, statusBlocked
                              from users u
                                join internal_users iu using(uid)
                              where domain_login = ?
                             ", $domain_login
                            );

    return $logins;
}

#-----------------------------------------------------------

=head2 get_role_with_teamleaders

  тоже самое что и rbac_who_is(), только дополнительно определяем роли (супер)тимлидеров

=cut

sub get_role_with_teamleaders($$)
{
    my ($rbac, $uid) = @_;

    my $roles_hash = rbac_who_is_detailed($rbac, $uid);

    my $role = $roles_hash->{role};
    $role = 'teamleader' if $roles_hash->{is_teamleader};
    $role = 'superteamleader' if $roles_hash->{is_superteamleader};

    return $role;
}

#-----------------------------------------------------------

=head1 API methods

=cut

sub _deprecated
{
    state $log //= Yandex::Log::Messages->new();
    $log->warn(Carp::longmess("deprecated (see stacktrace)"));
}

#-----------------------------------------------------------

=head2 add_role_in_db_balance_and_rbac

    Создает пользователя с указанной ролью в БД, Балансе (если требуется) и в RBAC'е

    Параметры именованные 
        rbac
        role
        uid
        user_data

    Не делает никаких проверок, поэтому использовать с осторожностью и аккуратностью. 
    По возможности лучше вызывать более высокоуровневые функции с должными проверками.

=cut 

sub add_role_in_db_balance_and_rbac
{
    my (%O) = @_;
    my ($rbac, $role, $uid, $user_data) = @O{qw/ 
         rbac   role   uid   user_data/};

    # для всех ролей нужен ClientID (оператора в баланс передаем c UID = 1 если не передали извне отдельно)
    $user_data = hash_merge $user_data, {UID => 1} unless $user_data->{UID};
    $user_data->{rep_type} = 'chief';
    my %role_info = (role => undef, subrole => undef);
    if (grep {$_ eq $role} qw/super superreader support limited_support placer media manager internal_ad_admin internal_ad_manager internal_ad_superreader/) {
        $role_info{role} = $role;
    } elsif ($role eq 'teamleader' || $role eq 'superteamleader') {
        $role_info{role} = 'manager';
        $role_info{subrole} = $role;
    } else {
        die {code => 1, fatal => "роль $role нельзя создать"};
    }

    # записываем юзера в БД
    $user_data->{ClientID} = Client::create_client_in_balance(
        $user_data->{UID} || 1, $uid, 
        initial_country => $geo_regions::RUS,
        (map {$_ => $user_data->{$_}} qw/fio phone email/)
        );
    create_update_client({client_data => {ClientID => $user_data->{ClientID}, role => $Rbac::ROLE_EMPTY, chief_uid => $uid}});
    create_update_user($uid, $user_data);

    # заводим роль в rbac-е
    my $rbac_error;

    if ($role eq 'super') {
        $rbac_error = rbac_create_superuser($rbac, $uid);
    } elsif ($role eq 'superreader') {
        $rbac_error = rbac_create_superreader($rbac, $uid);
    } elsif ($role eq 'support') {
        $rbac_error = rbac_create_support($rbac, $uid);
    } elsif ($role eq 'limited_support') {
        $rbac_error = rbac_create_support($rbac, $uid);
    } elsif ($role eq 'placer') {
        $rbac_error = rbac_create_placer($rbac, $uid);
    } elsif ($role eq 'media') {
        $rbac_error = rbac_create_mediaplanner($rbac, $uid);
    } elsif ($role eq 'manager') {
        $rbac_error = rbac_create_manager($rbac, $uid);
    } elsif ($role eq 'teamleader') {
        $rbac_error = rbac_create_teamleader($rbac, $uid);
    } elsif ($role eq 'superteamleader') {
        $rbac_error = rbac_create_superteamleader($rbac, $uid);
    } elsif ($role eq 'internal_ad_admin') {
        $rbac_error = rbac_create_internal_ad_admin($rbac, $uid);
    } elsif ($role eq 'internal_ad_manager') {
        $rbac_error = rbac_create_internal_ad_manager($rbac, $uid);
    } elsif ($role eq 'internal_ad_superreader') {
        $rbac_error = rbac_create_internal_ad_superreader($rbac, $uid);
    } else {
        die {code => 1, fatal => "роль $role нельзя создать"};
    }
    if (!$rbac_error) {
        Client::update_role($user_data->{ClientID}, $role_info{role}, $role_info{subrole});
    }

    return $rbac_error
}

#-----------------------------------------------------------

=head2 get-user-roles

    получить роли по логину
    на входе доменный логин

    https://8771.beta.direct.yandex.ru/idm/get-user-roles?token=sB5GiyiU9k_YQJVmPIBrVA&login=msa

=cut

sub cmd_get_user_roles :Cmd(dostup_get_user_roles)
{
    my $ctx = shift;

    _deprecated();
    my $logins = get_and_check_domain_login($ctx->{FORM}->{login});
    my @roles;

    for my $row (@$logins) {
        my $role = get_role_with_teamleaders($ctx->{rbac}, $row->{uid});

        # логин заблокирован, считаем, что роли у него нет
        next if defined $row->{statusBlocked}
             && $row->{statusBlocked} eq 'Yes';

        next unless $IS_VALID_ROLE{$role};

        push @roles, [{$SLUG => $role}, {'passport-login' => lc($row->{login})}];
        # для тимлидеров возвращаем и их базовую роль менеджера (DIRECT-17723)
        push @roles, [{$SLUG => 'manager'}, {'passport-login' => lc($row->{login})}] if $role =~ /^(?:superteamleader|teamleader)$/;
    }

    return ($OK, {roles => \@roles});
}

1;
