package Intapi::ExportCmdLog;

=head1 NAME

    Intapi::ExportCmdLog

=head1 DESCRIPTION

    Отдаём данные из ppclog_cmd в БК, для антифрода
    Принимает два cgi-параметра:
    from - дата-время в формате \d{14} - с какого времени начинать отдавать данные
    limit - сколько строк отдать, по-умолчанию - 1e6
    Выдаёт tsv:
    uid ip yandexuid log_ts

=cut

use strict;
use warnings;

use POSIX qw/strftime/;

use Yandex::DBTools;
use Yandex::TimeCommon;
use Yandex::Log::Messages;
use Yandex::Memcached::Lock;

use Settings;

use Tools qw/get_clickhouse_handler/;

use utf8;

my $LOCK_NAME = 'EXPORT_CMD_LOG';
# на сколько секунд отступает от текущего времени (чтобы логи успели доехать со всех серверов)
my $TIME_GAP = 900;
my $DEFAULT_LIMIT = 1_000_000;

sub handler {
    my ($r, $multiform) = @_;

    my $log = Yandex::Log::Messages->new();

    my %param = %$multiform; 
    if ( $param{from} && $param{from} !~ /^(\d{8}|\d{14})$/ ){
        $log->die("incorrect from parameter");
    }
    if ( $param{limit} && $param{limit} !~ /^\d+$/ ){
        $log->die("incorrect limit parameter");
    }

    # Лочимся
    my $mcl = Yandex::Memcached::Lock->new(
        servers => $Settings::MEMCACHED_SERVERS,
        max_locks_num => 2,
        entry => "locks/$LOCK_NAME",
        );
    if (!$mcl->get_lock()) {
        $log->die("Can't get lock '$LOCK_NAME' in memcached");
    }

    # На входе один параметр - from, начиная с какого времени интересны логи
    my $today = strftime("%Y%m%d", localtime);
    my $from = $param{from} || $today;
    my $limit = $param{limit} || $DEFAULT_LIMIT;

    $log->out("getCmdLog.pl started from=$from");

    return sub {
        my $respond = shift;
        my $writer = $respond->([200, ['Content-Type', 'text/tab-separated-values']]);

        my $clh = get_clickhouse_handler('cloud');

        for my $date (get_distinct_dates(substr($from, 0, 8), $today)) {
            my $sql = "SELECT uid, ip, yandexuid, toInt64(log_time) as ts
                       FROM ppclog_cmd
                       WHERE "
                .sql_condition({
                    log_date => mysql_round_day($date, delim => "-"),
                    log_time__ge => unix2human(mysql2unix($from)),
                    service__in => ['direct.perl.web', 'direct.java.webapi'],
                    uid__gt__int => 0,
                    log_time__lt__dont_quote => "now() - $TIME_GAP",
                })
                ." ORDER BY log_date, log_time 
                   LIMIT $limit 
                   FORMAT JSON";
            my $cnt = 0;
            for my $row ( @{ $clh->query($sql)->json->{data} } ) {
                $writer->write(join("\t", @{$row}{qw/uid ip yandexuid ts/})."\n");
                $cnt++;
            };
            $log->out("getCmdLog.pl finished, $cnt for date=$date lines printed");
            if ($cnt >= $limit) {
                last;
            } else {
                $limit -= $cnt;
            }
        }

        $writer->close();
    };
}

1;
