package Intapi::ExportRightsUidToCid;

=head1 NAME

    Intapi::ExportRightsUidToCid

=head1 DESCRIPTION

    Отдаёт для Метрики пары uid => cid, где uid может видеть кампанию cid.
    Сейчас это:
        * представители агентства на агентские кампании клиента
          [с учётом ограниченных представителей]
        * менеджеры агентств на агентские кампании клиента
          [с учётом тимлидеров и групп менеджеров]
        * менеджеры на менеджерские кампании клиента
          [с учётом тимлидеров и групп менеджеров]
    По всей базе работает час-два.

    Для отладки умеет принимать дополнительный параметр:
        cid — выбрать не по всей базе, а только по указанным кампаниям

    curl -s 'http://8700.beta1.direct.yandex.ru/ExportRightsUidToCid'
    curl -s 'http://8700.beta1.direct.yandex.ru/ExportRightsUidToCid?cid=263&cid=456'

=cut

use Direct::Modern;

use Settings;
use Campaign::Types qw(get_camp_kind_types);
use ShardingTools qw(ppc_shards);
use RBACDirect qw(
    rbac_get_all_managers_of_agencies_clientids
    rbac_mass_get_agencies_of_clients
    rbac_mass_get_teamleaders_of_managers
    rbac_mass_get_managers_of_teamleaders
);
use List::MoreUtils qw(uniq);

use Carp;

use Yandex::DBTools;
use Yandex::ListUtils qw(nsort chunks xflatten);
use Yandex::Memcached::Lock;

my $CIDS_CHUNK_LIMIT = 500_000;

# на всякий случай лок долгий, на полчаса; 
# в норме сами разлочим после окончания запроса,
# а если что-то сломается и не будет заканчиваться нормально --
# перестраховываемся и не разрешаем много запросов.
our $LOCK_TIMEOUT      ||= 1800;
our $PARALLEL_LEVEL    ||= 3;
 

=head2 handler

=cut

sub handler {
    my ($r, $multiform) = @_;

    my $params = $multiform->as_hashref_mixed;

    my %additional_cond;
    if (exists $params->{cid}) {
        $additional_cond{'c.cid'} = delete $params->{cid};
    }
    if (exists $params->{min_cid}) {
        $additional_cond{'c.cid__ge'} = delete $params->{min_cid};
    }
    if (exists $params->{max_cid}) {
        $additional_cond{'c.cid__le'} = delete $params->{max_cid};
    }

    if (%$params) {
        my $unknown_param_names = join ', ', keys %$params;
        croak "unknown params given: $unknown_param_names";
    }

    my $mcl = Yandex::Memcached::Lock->new(
        servers => $Settings::MEMCACHED_SERVERS,
        entry => "locks/ExportRightsUidToCid",
        expire => $LOCK_TIMEOUT, 
        max_locks_num => $PARALLEL_LEVEL,
    );

    # Если все слоты лока заняты -- возвращаем ошибку; 
    # если серверы Memcached недоступны -- то же самое, разрешать бесконтрольные запросы не хотим 
    unless ( $mcl && $mcl->get_lock() ) {
        return {text => "Too many requests", code => 429};
    }

    return sub {
        my ($respond) = @_;

        my @FIELDS = qw(uid cid);
        my $writer = $respond->([200, ['Content-Type' => 'text/tab-separated-values']]);
        $writer->write('#' . join("\t", @FIELDS) . "\n");

        for my $shard (ppc_shards()) {
            my $min_cid = 0;
            my $cids_fetched;

            do {
                my $campaigns = get_all_sql(PPC(shard => $shard), ['
                    SELECT c.cid, c.uid, c.AgencyID, c.ManagerUID
                    FROM campaigns c
                    JOIN clients cl ON (c.ClientID = cl.ClientID)
                 ', WHERE => {
                        _OR => {'c.AgencyID__gt' => 0, 'c.ManagerUID__gt' => 0},
                        'c.OrderID__gt' => 0,
                        'c.statusEmpty' => 'No',
                        'c.type' => get_camp_kind_types('web_edit_base'),
                        'c.cid__gt' => $min_cid || 0,
                        'c.AgencyID__dont_quote' => 'IF (c.AgencyID > 0, cl.agency_client_id, c.AgencyID)',
                        %additional_cond,
                    }, '
                    ORDER BY c.cid
                    LIMIT ?
                '], $CIDS_CHUNK_LIMIT);
                $cids_fetched = scalar(@$campaigns);

                my (@agency_ids, @manager_uids, @client_uids);
                for my $camp (@$campaigns) {
                    my $agid = $camp->{AgencyID};
                    my $manuid = $camp->{ManagerUID};
                    my $cid = $camp->{cid};
                    push @agency_ids, $agid if $agid;
                    push @manager_uids, $manuid if $manuid;
                    push @client_uids, $camp->{uid};
                }
                $min_cid = $cids_fetched ? $campaigns->[-1]->{cid} : 0 ;

                my $agid2manager_uids = @agency_ids ? rbac_get_all_managers_of_agencies_clientids(undef, \@agency_ids) : {};
                undef @agency_ids;

                my $client_uid2agency_uids = @client_uids ? rbac_mass_get_agencies_of_clients(undef, \@client_uids) : {};
                undef @client_uids;

                my @possible_manager_uids = (@manager_uids, uniq xflatten values %$agid2manager_uids);
                my $manager_uid2teamleader_uid = @possible_manager_uids ? rbac_mass_get_teamleaders_of_managers(undef, \@possible_manager_uids) : {};
                undef @manager_uids;
                undef @possible_manager_uids;

                my @teamleader_uids = uniq values %$manager_uid2teamleader_uid;
                my $teamleader_uid2manager_uids = @teamleader_uids ? rbac_mass_get_managers_of_teamleaders(undef, \@teamleader_uids, with_idm_groups => 1) : {};
                undef @teamleader_uids;

                my %uid2possible_cids;
                for my $camp (@$campaigns) {
                    my $cid = $camp->{cid};
                    my (@possible_uids, @manager_uids);
                    if ($camp->{ManagerUID}) {
                        push @manager_uids, $camp->{ManagerUID};
                    } elsif ($camp->{AgencyID}) {
                        push @possible_uids, @{ $client_uid2agency_uids->{ $camp->{uid} } // [] };
                        push @manager_uids, @{ $agid2manager_uids->{ $camp->{AgencyID} } // [] };
                    }
                    if (@manager_uids) {
                        push @possible_uids, @manager_uids;
                        my @teamleader_uids = grep { $_ } map { $manager_uid2teamleader_uid->{$_} } @manager_uids;
                        push @possible_uids, @teamleader_uids;
                        my @group_manager_uids = grep { $_ } xflatten map { $teamleader_uid2manager_uids->{$_} } @teamleader_uids;
                        push @possible_uids, @group_manager_uids;
                    }
                    for my $uid (uniq @possible_uids) {
                        push @{ $uid2possible_cids{$uid} }, $cid;
                    }
                }
                undef $campaigns;

                for my $uid (nsort keys %uid2possible_cids) {
                    for my $cid (nsort @{ $uid2possible_cids{$uid} }) {
                        $writer->write("$uid\t$cid\n");
                    }
                }
            } while ($cids_fetched == $CIDS_CHUNK_LIMIT);
        }

        $writer->write("#End\n");
        $writer->close();
        $mcl->unlock();
    };

}

1;
