package Intapi::GetRelatedUsersByClientID;

# $Id: GetRelatedUsersByClientID.pm 149221 2017-09-15 18:00:12Z adubinkin $

=head1 NAME

    Intapi::GetRelatedUsersByClientID

=head1 DESCRIPTION

    Получает список ClientID, возвращает список кортежей [{"login":".....", "uid": 0000, "is_agency_uid":1|0}, ]
    для всех представителей каждого клиента, включая представителей агенств

=cut

use strict;
use warnings;

use Readonly;

use Settings;

use TextTools qw/get_num_array_by_str/;
use Yandex::Log;
use Yandex::Validate;
use Client;

use utf8;

Readonly my $DB_CHUNK_SIZE => 10_000;

=head2 handler

    Входные параметры
      client_ids - id клиентов, полученные Метрикой от Биллинга, разделенные
                   запятой
      
    Возвращаем хэш, сериализованный в JSON:
      {
          ClientID => [{ login => , uid =>, is_agency_uid }, ...],
          ...
      }

=cut

sub handler {
    my ($r, $multiform) = @_;

    my $log = Yandex::Log->new(
        date_suf => "%Y%m%d",
        log_file_name => 'GetRelatedUsersByClientID.log',
        msg_prefix    => "[$$]",
    );
    $log->out('start');

    my %param = %$multiform;
    my $client_ids_str = $param{client_ids};
    
    if (!$client_ids_str) {
        $log->out('finish: incorrect params');
        return export_result(500, { result => "client_id required" });
    }
    
    my $client_ids = get_num_array_by_str($client_ids_str);

    my @invalid_clients_ids = grep {!is_valid_id($_)} @$client_ids;
    return export_result(500, { result => 'Invalid ClientIDs: '.join(', ', @invalid_clients_ids) }) if @invalid_clients_ids;

    my $result = Client::get_related_users_by_client_id($client_ids);
    
    $log->out('finish');
    
    return export_result(200, $result);
}

=head2 export_result(code, ret) 

    Получает код и структуру данных ответа.
    Возвращает сформированную на их основе стандартную структуру ответа intapi handler'а.

=cut

sub export_result($$) 
{
    my ($code, $ret) = @_;
    return {json => $ret, code => $code, content_type => "text/x-javascript"}; 
}


1;
