package Intapi::RequestBlockNumberDetailed;

# $Id$

=head1 NAME

Intapi::RequestBlockNumberDetailed

=head1 DESCRIPTION

Получить количество объявлений и фраз, в разрезе по состояниям: активные, остановленные, отклоненные, черновики. 

=cut

use strict;
use warnings;
use utf8;

use Settings;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_merge/;
use JSON;
use Encode;
use MTools;


=head2 handler

На вход:
string CampaignCode - ID кампании
На выход:
ResultCode, Status, ActiveBlock, ActiveRequest, InactiveBlock, InactiveRequest, RejectedBlock, RejectedRequest, DraftBlock, DraftRequest

test:
wget -q -O - --post-data='{"CampaignCode":263}' beta.direct.yandex.ru:8501/RequestBlockNumberDetailed
wget -q -O - beta.direct.yandex.ru:8501/RequestBlockNumberDetailed?cid=263

=cut

use constant PARAM_ERROR => {
    ResultCode => 0,
    status     => 'error', 
    error      => 'Parameter CampaignCode required',
};


my $SQL_MODERATE = "(
        b.statusModerate in ('Ready', 'Sent', 'Sending')
        or ph.statusModerate in ('Ready', 'Sent', 'Sending')
        or (
            ( b.phoneflag in ('Ready', 'Sent', 'Sending') or b.statusSitelinksModerate in ('Ready', 'Sent', 'Sending') )
            and b.statusModerate != 'No'
            and ph.statusModerate != 'No'
        )
    ) and b.statusArch = 'No'";
my $SQL_INACTIVE = "b.statusShow='No' AND b.statusArch='No'";
my $SQL_ACTIVE = $MTools::IS_ACTIVE_CLAUSE;
my $SQL_REJECTED = "(
        b.statusModerate='No'
        OR ph.statusModerate='No'
        OR b.phoneflag='No'
        OR b.statusSitelinksModerate='No'
    )
    AND b.statusArch = 'No'";
my $SQL_ARCHIVED = "b.statusArch='Yes'";
my $SQL_DRAFT = "b.statusModerate = 'New' and ph.statusModerate = 'New' and b.statusArch = 'No'";


sub handler {
    my ($r, $multiform) = @_;
    my $in = _parse_input($r, $multiform);
    if ($in->{error}) {
        $in->{ResultCode} = 0;
        return {json => $in};
    }
    my $cid = $in->{CampaignCode} || '';
    unless ($cid and $cid =~ /^\d+$/) {
        return {json => PARAM_ERROR};
    }

    my $does_exists = get_one_field_sql(PPC(cid => $cid), "select count(*) from campaigns where cid = ?", $cid);
    unless ($does_exists) {
        return { json => { 
            status      => 'error',
            error       => "campaign $cid does not exists",
            ResultCode  => 0,
        }};
    }
    
    my $result = {
        status => 'ok',
        ResultCode => 1,
    };

    my $blocks = get_one_line_sql(PPC(cid => $cid), "
        SELECT 
           SUM(1) as BlockNumber,
           SUM(IF($SQL_MODERATE,1,0)) as `ModerateBlock`,
           SUM(IF($SQL_INACTIVE,1,0)) as `InactiveBlock`,
           SUM(IF($SQL_ACTIVE,1,0)) as `ActiveBlock`,
           SUM(IF($SQL_REJECTED,1,0)) as `RejectedBlock`,
           SUM(IF($SQL_ARCHIVED,1,0)) as `ArchiveBlock`,
           SUM(IF($SQL_DRAFT,1,0)) as `DraftBlock`
      FROM banners b
      JOIN phrases ph USING(pid)
     WHERE ph.cid = ?",
           $cid);
    hash_merge $result, $blocks;
    
    my $requests = get_one_line_sql(PPC(cid => $cid), "
        SELECT 
           SUM(1) as RequestNumber,
           SUM(Moderate) as `ModerateRequest`,
           SUM(Inactive) as `InactiveRequest`,
           SUM(Active) as `ActiveRequest`,
           SUM(Rejected) as `RejectedRequest`,
           SUM(Archive) as `ArchiveRequest`,
           SUM(Draft) as `DraftRequest`
        FROM (
            SELECT
                SUM(IF($SQL_MODERATE,1,0)) > 0 as `Moderate`,
                SUM(IF(not ($SQL_INACTIVE),1,0)) = 0 as `Inactive`,
                SUM(IF($SQL_ACTIVE,1,0)) > 0 as `Active`,
                SUM(IF(not ($SQL_REJECTED),1,0)) = 0 as `Rejected`,
                SUM(IF(not ($SQL_ARCHIVED),1,0)) = 0 as `Archive`,
                SUM(IF(not ($SQL_DRAFT),1,0)) = 0 as `Draft`
            FROM banners b
            JOIN phrases ph ON ph.pid = b.pid
            JOIN bids bi on ph.pid = bi.pid
            WHERE ph.cid = ?
            GROUP BY bi.id, bi.pid
        ) kw",
        $cid);
    hash_merge $result, $requests;

    my $groups = get_one_line_sql(PPC(cid => $cid), "
        SELECT 
           SUM(1) as GroupNumber,
           SUM(Moderate) as `ModerateGroup`,
           SUM(Inactive) as `InactiveGroup`,
           SUM(Active) as `ActiveGroup`,
           SUM(Rejected) as `RejectedGroup`,
           SUM(Archive) as `ArchiveGroup`,
           SUM(Draft) as `DraftGroup`
        FROM (
            SELECT
                SUM(IF($SQL_MODERATE,1,0)) > 0 as `Moderate`,
                SUM(IF(not ($SQL_INACTIVE),1,0)) = 0 as `Inactive`,
                SUM(IF($SQL_ACTIVE,1,0)) > 0 as `Active`,
                SUM(IF(not ($SQL_REJECTED),1,0)) = 0 as `Rejected`,
                SUM(IF(not ($SQL_ARCHIVED),1,0)) = 0 as `Archive`,
                SUM(IF(not ($SQL_DRAFT),1,0)) = 0 as `Draft`
            FROM banners b
            JOIN phrases ph ON ph.pid = b.pid
            WHERE ph.cid = ?
            GROUP BY b.pid
        ) kw",
        $cid);
    hash_merge $result, $groups;

    return { json => $result };
}

sub _parse_input
{
    my ($r, $multiform) = @_;
    my %in = %$multiform;
    $in{CampaignCode} = $in{cid} if defined $in{cid};
    if ($in{CampaignCode}) {
        return { CampaignCode => $in{CampaignCode} };
    }

    my $data = decode 'utf8', $r->content;
    unless ($data) {
        return PARAM_ERROR;
    }
    eval {
        $data = from_json $data;
    };
    if ($@) {
        return { status => 'error', error => $@ };
    }
    return $data;
}

1;

