package Intapi::ResyncCampaigns;

=pod
=head1 NAME

    Intapi::ResyncCampaigns - ручка для переотправки кампаний через ленивую очередь по инициативе БК.

=head1 DESCRIPTION

    Ручка позволяет переотправить переданный в неё список кампаний вместе с баннерами и условиями в БК через ленивую очередь.
    Используется БК в случае проблем на их стороне.

=cut

use strict;
use warnings;
use utf8;

use BS::ResyncQueue;

use Yandex::ListUtils qw/ chunks /;

# Делим полученный массив на части, чтобы не выйти за лимиты по памяти
use constant RESYNC_CHUNK_SIZE => 500;

=head2 new

=cut
sub new
{
    bless {};
}

=head2 add

    Принимает ссылку на массив номеров кампаний (cid) и ставит их на переотправку вместе с баннерами и условиями в БК через ленивую очередь.

    Запросы:
        GET:
            curl -s 'http://9140.beta2.direct.yandex.ru/jsonrpc/ResyncCampaigns?method=add&params=\[123,456\]'
            curl -s 'http://intapi.direct.yandex.ru/jsonrpc/ResyncCampaigns?method=add&params=\[123,456\]'
        POST:
            curl -d '{"method":"add","params":[123,456]}' http://9140.beta2.direct.yandex.ru/jsonrpc/ResyncCampaigns
            curl -d '{"method":"add","params":[123,456]}' http://intapi.direct.yandex.ru/jsonrpc/ResyncCampaigns
    Ответы:
        В случае успеха:
            {"jsonrpc":"2.0","id":null,"result":{"success":1}}
        Если передали неправильные параметры или метод:
            {"jsonrpc":"2.0","error":{"message":"Error message","code":-32603},"id":null}

=cut

sub add 
{
    my ($self, $cids) = @_;

    die "Incorrect params" unless (ref($cids) eq 'ARRAY');

    my $log = Yandex::Log->new(
        log_file_name => 'bs_resync_campaigns',
        date_suf      => '%Y%m%d',
        lock          => 1,
    );

    $log->out('start');
    $log->out({ resync_campaigns => $cids});

    for my $chunk (chunks($cids, RESYNC_CHUNK_SIZE)) {
        $log->out({resync_chunk => $chunk});
        my $result = bs_resync_whole_camps($chunk, priority => BS::ResyncQueue::PRIORITY_INTAPI_RESYNC_CAMPAIGNS);
        $log->out({result => $result});
    }
    
    $log->out('finish');

    return { success => 1 };
}

1;
