package Intapi::TestScriptRun;

# $Id$

=head1 NAME

Intapi::TestScriptRun

=head1 DESCRIPTION

Запуск Директовых скриптов через Intapi. Разрешены для запуска только
скрипты, занесённые в белый список.

Список пожеланий на будущее:

* запуск скриптов в фоне (nohup + проверка файла с выводом)

* настройка таймаута (сейчас 5 минут)

* блокировка, чтобы одновременно выполнялось не больше одного процесса
каждого типа (некоторые скрипты уже делают это сами)

* упрощение белого списка

=head1 SYNOPSIS

 wget -qO - 'http://8880.beta1.direct.yandex.ru/secret-jsonrpc/TestScriptRun?method=run_script&params={"script":"ppcCampAutoPrice.pl","cmdline":"--cid 5194568"}'

 # делается в Intapi.pm
 my $handler = Intapi::TestScriptRun->new;
 print $handler->run_script( {
     script  => 'ppcCampAutoPrice.pl',
     cmdline => '--cid 5194568',
 } );

=cut

use strict;
use warnings;

use File::Slurp;
use File::Temp qw/tempfile/;
use Readonly;

use Settings;

use utf8;

Readonly my $TIMEOUT      => 300;                 # 5 минут
Readonly my $VALID_SECRET => 'b3uPkvVcVQGdGKpu';

Readonly my @SCRIPT_WHITELIST => qw(
    apiReportsBuilder.pl
    balanceGetClientNDSDiscountSchedule.pl
    balanceGetClientBrands.pl
    bsCheckUrlAvailability.pl
    bsClientData.pl
    bsExportMaster.pl
    bsFullLBExportMaster.pl
    conv/update_perf_counters.pl
    getNews.pl
    getOptimizeStat.pl
    maintenance/fill_currency_rates.pl
    moderateSendWarn.pl
    moderateSyncDiags.pl
    ppcArchiveOldBanners.pl
    ppcArchiveOldCampaigns.pl
    ppcAutobudgetForecast.pl
    ppcBMReports.pl
    ppcCampAutoPrice.pl
    ppcCampCopyReportsDelete.pl
    ppcCampCopyReportsMonitor.pl
    ppcCampCopyReportsQueue.pl
    ppcCampGetGoals.pl
    ppcCampQueue.pl
    ppcCheckCompletedSurveys.pl
    ppcClearOldReports.pl
    ppcClearOldUnpaidCampaigns.pl
    ppcClientPotentialReports.pl
    ppcCopySubclientsForceCurrency.pl
    ppcDeleteAgencyUnproveClients.pl
    ppcDeleteUnlinkVCards.pl
    ppcFeedsRejectedOffers.pl
    ppcFeedToBannerLand.pl
    ppcFetchAgencyCountryCurrency.pl
    ppcFetchClientMulticurrencyTeaserData.pl
    ppcFirstAidSend.pl
    ppcForceCurrencyConvert.pl
    ppcForceCurrencyConvertNotify.pl
    ppcManageExperiments.pl
    ppcMassChangeOwner.pl
    ppcMoneyOutReminder.pl
    ppcMonitorUpdateStoreContent.pl
    ppcPdfReports.pl
    ppcPrepareClientDomains.pl
    ppcPrepareCurrencyTeaserClients.pl
    ppcProcessAutoPayments.pl
    ppcProcessImageQueue.pl
    ppcSetAutoResources.pl
    ppcProcessResyncMediaQueue.pl
    ppcProcessUrlAvailabilityPctChange.pl
    ppcPushNotificationsQueue.pl
    ppcRedirectCheck.pl
    ppcResendDomainsBS.pl
    ppcRetargetingCheckGoals.pl
    ppcSearchQueryStatus.pl
    ppcSendAutooptimizations.pl
    ppcSendBalanceInfoChanges.pl
    ppcSendMailMaster.pl
    ppcSendMailNew.pl
    ppcSendOrderWarnings.pl
    ppcSendPausedByDayBudgetMails.pl
    ppcSendWarnAboutCPMLimit.pl
    ppcSendWarnPlace.pl
    ppcServicingCampEvents.pl
    ppcSetAutoResources.pl
    ppcStatRollbackNotify.pl
    ppcSyncModifedCreatives.pl
    ppcXLSImageUploadFailureNotification.pl
    ppcXLSReports.pl
    ppcProcessModLicensesQueue.pl
    ppcSendSearchBannersResults.pl
    ppcSendSMS.pl
    ppcResyncRelevanceMatchCampaigns.pl
);

Readonly my %SCRIPT_IN_WHITELIST => map { $_ => 1 } @SCRIPT_WHITELIST;

Readonly my %SCRIPT_SUPPORTS_ONCE => map { $_ => 1 } qw(
    bsClientData.pl
    bsExportMaster.pl
    bsFullLBExportMaster.pl
    ppcCampAutoPrice.pl
    ppcCampQueue.pl
    ppcCurrencyConvertMaster.pl
    ppcFeedToBannerLand.pl
    ppcFirstAidSend.pl
    ppcPdfReports.pl
    ppcProcessImageQueue.pl
    ppcProcessUrlAvailabilityPctChange.pl
    ppcXLSReports.pl
);

=head1 SUBROUTINES/METHODS

=head2 new

Конструктор.

=cut

sub new {
    my ($class) = @_;
    return bless {}, $class;
}

=head2 run_script

Обработчик: вызывает скрипт и возвращает его вывод; если скрипт не находится
в белом списке, порождает исключение.

=cut

sub run_script {
    my ( $self, $params, $procedure, @extra_args ) = @_;

    my $script  = $params->{script};
    my $cmdline = $params->{cmdline};
    my $secret  = $params->{secret};

    if ( !$SCRIPT_IN_WHITELIST{$script} ) {
        die "The specified script $script is not in the whitelist; contact a developer to have it added";
    }

    if ( !$secret || $secret ne $VALID_SECRET ) {
        die "Invalid secret";
    }

    unless ( $cmdline =~ /^[\w\.\-=:,\/" ]*$/ ) {
        die 'Invalid command line';
    }

    my @cmd_parts;
    my $scripts_dir = ($script =~ /\.py$/) ? "python/scripts" : "protected";
    my $script_path = "$Settings::ROOT/$scripts_dir/$script";
    if ($params->{log_tee}) {
        push @cmd_parts, 'LOG_TEE=1';
    }
    push @cmd_parts, ( "timeout ${TIMEOUT}s", $script_path, $cmdline );

    if ( $SCRIPT_SUPPORTS_ONCE{$script} ) {
        push @cmd_parts, '--once';
    }

    my $cmd = join ' ', @cmd_parts;

    $ENV{TMPDIR} = "/tmp/temp-ttl/ttl_2d"; # если каталог существует, он будет использоваться как временная директория
    my (undef, $out_file) = tempfile("TestScriptRun-$$-out-XXXXXXXXXX", UNLINK => 0, OPEN => 0, TMPDIR => 1);
    my (undef, $err_file) = tempfile("TestScriptRun-$$-err-XXXXXXXXXX", UNLINK => 0, OPEN => 0, TMPDIR => 1);

    system( "$cmd > $out_file 2> $err_file" );
    
    my $child_status = $? >> 8;

    my $stdout = File::Slurp::read_file($out_file, binmode => ':utf8' );
    my $stderr = File::Slurp::read_file($err_file, binmode => ':utf8' );

    unlink $out_file;
    unlink $err_file;

    # timeout выходит с таким кодом, если отведённое время вышло
    if ( $child_status == 124 ) {
        $stderr .= "\n\nTIMED OUT";
    }

    return {stdout => $stdout, stderr => $stderr, exit_code => $child_status};
}

=head2 get_whitelist

Возвращает список скриптов, разрешенных к запуску

=cut

sub get_whitelist {
    return \@SCRIPT_WHITELIST;
}

1;
