package InternalReports;

# $Id$

=head1 NAME
    
    InternalReports

    Модуль для составления несложных внутренних Директ-отчетов

=head1 DESCRIPTION

=head2 Простое 

    Чтобы добавить новый типовой отчет: внести запись в %PREDEFINED_REPORT 
      * придумать уникальный report_id,
      * report_type - сейчас доступны 'predefined_sql' и 'predefined_sub' (формально есть еще monitor_values, но это один отчет с параметрами)
      * title -- человекопонятный заголовок отчета 
      * description (необязательно) -- краткое описание отчета, будет выведено на странице
      * method (необязательно) — GET/POST; по-умолчанию GET
      * multipart (необязательно) — нужен ли multipart/form-data
      * submit_title (необязательно) — текст на кнопке отправки; по-умолчанию "Показать!"

    Для отчета типа 'predefined_sql' заполнить поля: 
        DB
        sql 
        make_bind_values_arr (необязательно)
        field_titles (необязательно) -- человекопонятные названия для колонок 

    Для отчета типа 'predefined_sub' заполнить поля: 
        code -- ссылка на функцию, вычисляющую данные.  
        field_titles (необязательно) -- человекопонятные названия для колонок 
        field_order (необязательно) -- ссылка на массив идентификаторов колонок. 
            Указанные колонки будут выведены в указанном порядке слева направо, все остальные -- правее, в порядке "как придется"
        default_param_values (необязательно) -- хеш умолчальных значений параметров. Могут быть скаляры или функции.
  
    Для любого отчета можно:
      details_required - список полей, по которым надо показывать не просто значение, а красивый детальный блок 
        сейчас можно: cid, OrderID, uid, login, agency_login, ClientID, url, pre, BannerID, bids_id, PhraseID
      expected_params - какие параметры принимает отчет. 
        Ссылка на массив из идентификаторов и массивов идентификаторов, пример: [ 'logins', 'cids', ['date_from', 'date_to'] ].
        Соответствующие поля для ввода будут выведены на странице: одиночные идентификаторы -- каждый на отдельной строке, объединенные в массив -- все на одной строчке.
        Все введенные значения будут среди именованных параметров функции, считающей отчет.
        Пример использования -- отчет 'bind_html5_creative_to_image_ad_banner'
      group_id - название группы отчетов на индексной странице, например, group_by => 'API'
      pivot - хэш с информацией для разворота таблиц
        fields - поля, по которым можно разворачивать
        measures - поля, которые можно выбирать для отображения

    Для любого отчета можно: 
      на странице internal_reports_index добавить ссылку на новый отчет

    Для любого отчета: можно добавить в параметрах формы xls=<что-нибудь истинное в понимании Perl'а> 

=head2 Посложнее

    Чтобы окрасивить изображение кампании или какого-то другого поля: 
      по необходимости добавить в add_detailed_info_to_report извлечение нужных данных
      в шаблоне internal_reports.html исправить/дополнить блок detailed

    По необходимости улучшать параметры для отчетов: 
      добавить календарик для дат (=надо знать тип парметра)
      придумать что-нибудь со значениями по умолчанию

    По необходимости добавить параметры со страницы для отчетов типа 'predefined_sql'

=head2 Права на отчеты

    По умолчанию отчеты доступны для суперпользователя и суперридера. 
    
    Кроме того, в хеше с описанием отчета можно указать rbac_check, значение -- ссылка на хеш, аналогично атрибуту :Rbac. 
        Отличие -- barewords не разрешены, т.е. названия ролей должны быть в кавычках, а на функции должны быть обычные ссылки. 

    rbac_check может ослабить умолчальную проверку вплоть до прав собственно на cmd_internalReports (rbac_cmd_internal_networks_only)

=head2 Кстати 
    
    еще полезно посмотреть на cmd_getReport

=head1 FUNCTIONS

=cut

use Direct::Modern;
use POSIX qw/strftime ceil/;

use Encode;
use File::Slurp qw/read_file write_file/;
use File::Temp qw/tempfile/;
use JSON;
use List::Util qw/maxstr max/;
use List::MoreUtils qw/any all part uniq none minmax/;
use URI;

use Yandex::Balance qw(balance_get_all_equal_clients);
use Yandex::TimeCommon;
use Yandex::ReportsXLS;
use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::Validate;
use Yandex::DateTime;
use Yandex::MirrorsTools::Hostings qw/strip_domain/;
use Yandex::I18n;
use Yandex::Blackbox;
use Yandex::ScalarUtils;
use Yandex::Staff3 qw( get_staff_info );
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::DBTools;
use Yandex::Memcached::Lock;
use Yandex::DBQueue;
use Yandex::Log::Messages;
use Yandex::Retry;
use Yandex::HTTP qw/http_fetch/;
use Yandex::Compress qw/deflate/;

use Settings;
use RBACElementary;
use RBACDirect;
use RBAC2::DirectChecks;
use Rbac;
use Primitives;
use PrimitivesIds;
use geo_regions;
use GeoTools;
use Yandex::CSV;
use URLDomain;
use AggregatorDomains qw//;
use Currencies;
use Common;
use RedirectCheckQueue;
use Currency::Rate;
use Yandex::Balance qw/balance_get_currency_rate/;
use TTTools;
use Tools qw/get_clickhouse_handler/;
use EventLog;
use YAML::Syck;
use EnvTools;
use TextTools;
use HashingTools;
use ShardingTools;
use Stat::OrderStatDay;
use PhraseText;
use ADVQ6;
use Campaign;
use Direct::ResponseHelper;
use TestUsers;
use BillingAggregateTools;
use BS::Export ();
use BS::ExportMaster ();
use BS::ResyncQueue;
use Property;
use Client::ConvertToRealMoney;
use Client::CurrencyTeaserData ();
use Client;
use LogTools;
use Moderate::ResyncQueue;
use Campaign::Types;
use Direct::Model::Creative::Manager;
use Wallet ();
use Direct::BillingAggregates;
use Direct::Clients;
use Direct::Feeds;
use Direct::Wallets;
use Direct::YaAgency;
use DoCmdAdGroup::Helper;
use Models::Banner qw//;
use XLSParse;

use API::ReportCommon;
use API::Settings;

use DateTime;

use base qw/Exporter/;
our @EXPORT = qw/
    extract_params_from_form
    get_data_for_internal_report
/;

use utf8;

=head2 Глобальные атрибуты для параметров формы отчёта

    Сделано для быстрого создания отчётов.
    
    Сюда стоит писать определение полей ввода, если они используются более чем в одном отчёте
    Или если они настолько хороши, что просто обязаны быть в общем пространстве имён.

    поле parent используется для указания родительского набора атрибутов.
    циклические зависимости обраваются при первом повторении

    в будущем можно добавить множественное наследование ;)

=cut

our %PARAM_TYPES = (
    'text'  => {
        field_size => 50,
        default_value   => '',
        type    => 'text',
    },

    'textarea'  => {
        default_value   => '',
        type    => 'textarea',
    },

    'date'  => {
        # Для даты хочется иметь шаблон с красивым календариком во всплывающем div-блоке
        default_value   => sub {
                my $self = shift;
                my $date = Yandex::DateTime->now();
                eval {
                    $date->add_duration(duration(days => $self->{default_date_shift})) if $self->{default_date_shift};
                };
                $date->date()
                #unix2human(time, "%Y-%m-%d"); 
            },
        title   => 'Дата',
        type    => 'date',
    },


    'region'  => {   
        type => 'region',
        default_value => '0',
        POSTPROCESS => sub {
            my $self = shift;
            $self->{default_value_text} = GeoTools::get_geo_names($self->{default_value});
            $self->{value_text} = GeoTools::get_geo_names($self->{value});
        },
    }, 

    'checkbox'  => {
        type    => 'checkbox',
    },

    'select'  => {
        type    => 'select',
    },

    'date_from' => {
        parent  => 'date',
        title   => 'Начало периода',
    },

    'date_to'   => {
        parent  => 'date',
        title   => 'Конец периода',
    },

    'cids'  => {
        parent  => 'text',
        title   => 'Номера кампаний',
    },

    'logins' => {
        parent  => 'text',
        title   => 'Логины',
        value_max_len => 1024
    },
    'href' => {
        title => 'Href',
        field_size => 30,
        value_max_len => 1024,
    },

    'shard' => {
        parent => 'select',
        name => 'shard',
        title => 'Шард PPC',
        options => [
            (map { ({ name => "ppc:$_", value => $_ }) } 1 .. $Settings::SHARDS_NUM),
            { name => 'Все шарды', value => 'all' },
        ],
    },

    hidden => {
        type => 'hidden',
    },

    file => {
        type => 'file',
    },
);

=head2 %INT_GROUP_NAMES

    Названия групп в списке отчетов - соответствуют group_id из параметров отчета

=cut

our %INT_GROUP_NAMES = (
    'API' => "Отчеты по API",
    'Stat' => "Статистика по интерфейсу",
    'SpecList' => "Специальные списки",
    'Moderate' => 'Модерация',
    'DEBUG' => "Данные для тестирования",
    experiment => 'Эксперимент',
    multicurrency => 'Мультивалютность',
    BSExport => 'Экспорт в БК',
    BLExport => 'Взаимодействие c BannerLand',
    MobileContent => 'Мобильный контент',
    VideoAudioAds => 'Видеореклама',
    yaAgency => 'Сервис "Настройка Яндекс Директа"',
    SUPPORT => 'Поддержка'
);

my $REPORT_NEW_VERSION_PROPERTY_PREFIX = 'INTERNAL_REPORT_NEW_VERSION_';
my $REPORT_NEW_VERSION_PATH_PREFIX = '/internal_tools/#';

=head2 %PREDEFINED_REPORT

    Предопределенные отчеты 
    Хеш: ( report_id => { report_type => 'predefined_sql|...', ...описание...} );


    Описание отчета по предопределенному sql-запросу: 
    {
        report_type => 'predefined_sql',
        DB   => ..., # база, в которой надо выполнить запрос
        sql  => ..., # текст запроса
    }

    TODO 
      * параметры для запроса (даты и т.п.)
        можно в описание запроса добавить ссылку на (анонимную) функцию, которая из %OPT собирает массив значений, которые надо передать в get_all_sql
      * настраиваемый порядок полей

    TODO 
    отчет по данным, которые собирает отдельная функция
    

=cut

our %PREDEFINED_REPORT = (

    manage_skip_locked_wallets => {
        report_type => 'predefined_sub',
        code => \&bs_manage_skip_locked_wallets,
        title => 'Ограничение отправки (разными потоками) в БК изменений по кампаниям под общим счетом',
        description => 'Эта настройка влияет на отправку в БК кампаний с изменениями (где несинхронна запись из campaigns, т.е. camps_num = 1)
    под одним общим счетом (включая саму кампанию-кошелёк) для случаев отправки <i>разными</i> потоками экспорта.
При ограничении - изменения по кампаниям в пределах одного общего счета эффективно отправляются только одним потоком.

<b>0 - Выключено</b> - без ограничений, кампании могут отправляться разными потоками параллельно;
<b>1 - Включено</b> - каждый поток экспорта при выборке кампаний для блокировки (и последующей отправки)
    исключает из выборки уже залоченные другими потоками кампании с общим счетом.',
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        group_id => 'BSExport',
        expected_params => [
            {
                name => 'enable',
                parent => 'select',
                title => 'Выключатель',
                default_value => 1,
                options => [
                    { name => '0 - Выключено', value => 0 },
                    { name => '1 - Включено', value => 1 },
                ],
            },
        ],
        fields => [
            { id => 'enabled', title => 'Текущее значение' },
        ],
    },

    resend_mobile_content_to_bs => {
        report_type => 'predefined_sub',
        code => \&resend_mobile_content_to_bs,
        title => 'Переотправка данных о мобильном контенте в БК',
        description => 'Сбрасывает statusBsSynced у записей (в таблице mobile_content, в выбранном шарде) c перечисленными mobile_content_id',
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        group_id => 'BSExport',
        expected_params => [
            {
                parent => 'shard',
            }, {
                name    => 'mobile_content_ids',
                title   => "Идентификаторы мобильного контента (mobile_content_id)",
                parent  => 'textarea',
            },
        ],
        fields => [
            { id => 'result', title => 'результат выполнения' },
            { id => 'shard', title => 'шард' },
            { id => 'db_res', title => 'обновлено записей' },
        ],
    },

    # Ежедневные замеры 
    monitor_values => {
        report_type => 'monitor_values',
        title       => 'Ежедневные замеры',
        chart       => 1,
    },

    # последние 100 нотификаций в баланс, отправленных через спец очередь
    some_notifications_to_balance => {
        report_type => 'predefined_sql',
        title => 'Специальные нотификации в Баланс',  
        DB   => PPC(shard => 'all'), 
        sql  => "select cid_or_uid, obj_type, operator_uid, add_time, send_status, priority from balance_info_queue order by id desc limit 100",
       
        fields => [
            'cid_or_uid',
            { id => 'obj_type', text_sort  => 1},
            'operator_uid', 
            { id => 'add_time', text_sort =>1},
            { id => 'send_status', text_sort =>1},
            'priority',
        ],
        details_required => [
            'cid',
        ],
    },

    # отладка нового грубого прогноза
    new_rough_forecast => {
        report_type => 'predefined_sub',
        title => '"Новый" "грубый" прогноз',
        description => "Новый грубый прогноз по кампании.\n currency - валюта из (YND_FIXED, RUB, USD, EUR, KZT, CHF, TRY, BYN)",
        rbac_check => {Role => [qw/super manager/], AllowDevelopers => 1},
        expected_params => [ 'cid', 'currency', 'date' ],
        code => \&new_rough_forecast,
    },

    bind_html5_creative_to_image_ad_banner => {
        title => 'Привязка html5 креатива к объявлению в текстовой кампании',
        description => 'В рамках эксперимента с html5 игрой для hezzl, нужно показывать html5 креатив с оплатой по кликам',
        code => \&bind_html5_creative_to_image_ad_banner,

        # остальные значения предполагаются одинаковыми для всех подобных отчетов
        group_id => 'API',
        report_type => 'predefined_sub',
        rbac_check => { Role => [ 'super', 'superreader' ], AllowDevelopers => 1 },

        fields => [
            {id => 'current_state', title => 'Текущее состояние' },
            {id => 'new_creative', title => 'Привязываем креатив' }
        ],
        expected_params => [
            {name => 'bid', parent => 'text', title => 'Id banner-а в Директе'},
            {name => 'creative_id', parent => 'text', title => 'Id html5 креатива для привязки'},
            {name => 'do_it', title => 'Внести изменения!', parent => 'checkbox' },
        ],
    },

    # список периодов, когда на указанной кампании были деньги
    when_money_on_camp_was => {
        report_type => 'predefined_sql',
        title => 'Когда на кампании были деньги', 
        description => "Когда на кампании были деньги для указанной кампании.", 

        DB => PPC(shard => 'all'),
        sql => 'select cid, interval_start, interval_end from when_money_on_camp_was where cid = ? order by interval_end desc limit 1000',
        make_bind_values_arr => sub {return [$_[0]->{cid}]},

        fields => [
            { id => 'cid', title => 'Кампания', },
            { id => 'interval_start', title => 'Начало денежного периода' },
            { id => 'interval_end', title => 'Окончание денежного периода' },
        ],
        expected_params => [ 
            [{name => 'cid', title => 'Кампания',}, ],
        ],

        details_required => [ 
           'cid'
        ],

        rbac_check => { Role => ["support", "limited_support", "super", "superreader", "manager"] },

    },

    api_forecast_is_get_data_from_bs => {
        report_type => 'predefined_sub',
        code => \&api_forecast_is_get_data_from_bs,
        title => "Откуда брать данные для прогноза бюджета в API",
        description => "Если значение 1 берем из БК, если 0 из advq", 
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        expected_params => [
            {
                name => 'is_get_data_from_bs',
                title => 'Брать ли данные из БК',
            }
        ],
        fields => [
            { id => 'result', title => 'результат выполнения' },
            { id => 'data_source', title => 'источник данных' },
        ],
        group_id => 'API',
    },

    api_reports_online_offline_mode_adjuster => {
        report_type => 'predefined_sub',
        code => \&api_reports_online_offline_mode_adjuster,
        title => "Настройка online-offline режима построения отчетов в API5.Report",
        description => "Параметры которыми можно регулировать режим построения отчетов (онлайн или оффлайн, через очередь), см. api/v5/API/Service/Reports/ProcessingModeChooser.pm",
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        expected_params => [
            {
                name => 'days_ok_for_online',
                title => 'Количество дней в диапазоне, при которых отчет можно строить онлайн (по умолчанию: 7)',
            },
            {
                name => 'campaigns_count_ok_for_online',
                title => 'Количество кампаний для которого отчет можно строить онлайн (по умолчанию: 10)',
            }
        ],
        fields => [
            { id => 'result', title => 'результат выполнения' },
            { id => 'data_source', title => 'источник данных' },
        ],
        group_id => 'API',
    },

    api_use_camp_aggregated_lastchange => {
        report_type => 'predefined_sub',
        code => \&api_use_camp_aggregated_lastchange,
        title => "Использовать в Changes данные из таблицы camp_aggregated_lastchange",
        description => "Если значение 1 - используем данные из camp_aggregated_lastchange, если 0 - делаем запросы в таблицы подобъектов",
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        expected_params => [
            {
                name => 'use_camp_aggregated_lastchange',
                title => 'Использовать ли в changes данные из camp_aggregated_lastchange',
            }
        ],
        fields => [
            { id => 'result', title => 'результат выполнения' },
            { id => 'data_source', title => 'источник данных' },
        ],
        group_id => 'API',
    },

    profile_logs_new => {
        report_type => 'predefined_sub',
        title => 'Статистика по сырым trace.log-ам',
        description => "% cat trace.log.20181203 | direct-log tr | less 
2018-12-02 21:00:11.331329   direct.intapi/DisplayCanvas.authenticate  ela:0.016,cpu:0.010/0.000,mem:+1.771m  4061218140252008845
    db:read/ppc:12      0.009   calls:11
    db:read/ppcdict     0.003   calls:5",
        code => \&profile_logs_http_new,

        expected_params => [ 
            [
                {name => 'time_period', title => 'За последние', parent => 'select',
                 options => [
                     {name => '', value => ''},
                     {name => '30 мин.', value => '30M'},
                     {name => '1 час', value => '1H'},
                     {name => '2 часа', value => '2H'},
                     {name => '4 часа', value => '4H'},
                     {name => '8 часов', value => '8H'},
                     ]},
                {name => 'stat_time_agg', title => 'группировка времени', parent => 'select',
                 default_value => 'hour', options => [map {{name => $_, value => $_}} qw/hour 30min 10min 5min 1min/]},
            ],
            [
                {name => 'date_from', parent => 'date', title => 'с (дата)'},
                {name => 'time_from', title => 'c (чч:мм)'},
                {name => 'date_to', parent => 'date', title => 'по (дата)'},
                {name => 'time_to', title => 'по (чч:мм)'},
            ],
            [
             map {{name => "filter_$_", title => "$_ LIKE", field_size => 20, value_max_len => 200}}
             qw/cmd_type cmd tags stat_time func func_param host_name/
            ],
            [
             map {
                  {   
                    name    => "group_by_$_",
                    title   => "Группировка $_",
                    options => [map {{name => $_, value => $_}} '', qw/stat_time cmd_type cmd tags func func_param host_name/],
                    parent  => 'select',
                  }
             } 1..7
            ],
        ],

        pivot => {
            fields => [qw/stat_time cmd_type cmd tags func func_param host_name/],
            measures => [qw/ela cnt ela_avg cpu_user cpu_system mem func_ela func_cnt func_obj_num func_ela_avg func_obj_ela_avg/],
            },
        chart => 1,
    },

    send_to_lazy_moderation_queue => {
        report_type => 'predefined_sub',
        title => 'Ленивая перепосылка в модерацию',
        description => 'Добавляет объекты в очереди ленивой переотправки на модерацию.
Находит и добавляет группы/баннеры/картинки/сайтлинки/логотипы/кнопки/визитки/мобильный контент/графические объявления.
Умеет работать со следующими типами кампаний: ' . join(',', @{Campaign::Types::get_camp_kind_types('lazy_moderate_resync')}),
        rbac_check => {Role => ['super', 'superreader', 'support', 'placer']},
        code => \&send_to_lazy_moderation_queue,
        fields  => [
            {
                id => 'direct_cids_count',
                title => 'Поставлено в очередь кампаний',
            },
            {
                id => 'result',
                title => 'Результат',
            },
        ],
        expected_params => [
            {
                name => 'cids',
                title => "Номера кампаний (cid'ы)",
                parent => 'textarea',
            },
            {
                name => 'priority',
                title => 'Приоритет (от -128 до 127)',
                parent => 'input',
                default_value => 0,
            },
            {
                name => 'remoderate',
                title => 'Запрет автомодерации',
                parent => 'checkbox',
            },
        ],
        group_id => 'Moderate',
    },

    advq_chrono_factor => {
        report_type => 'predefined_sub',
        title => 'Сезонность фраз', 
        description => "Поправочный коэффициент для прогноза показов/кликов с учетом сезонности.", 
        rbac_check => {Role => ['super', 'superreader', 'placer']},
        code => \&advq_chrono_factor_report,

        fields => [
            { id => 'phrase', title => 'Фраза', },
            { id => 'mon1', title => 'Январь'}, 
            { id => 'mon2', title => 'Февраль'}, 
            { id => 'mon3', title => 'Март'}, 
            { id => 'mon4', title => 'Апрель'}, 
            { id => 'mon5', title => 'Май'}, 
            { id => 'mon6', title => 'Июнь'}, 
            { id => 'mon7', title => 'Июль'}, 
            { id => 'mon8', title => 'Август'}, 
            { id => 'mon9', title => 'Сентябрь'}, 
            { id => 'mon10', title => 'Октябрь'}, 
            { id => 'mon11', title => 'Ноябрь'}, 
            { id => 'mon12', title => 'Декабрь'}, 
            { id => 'q1', title => 'Q1'}, 
            { id => 'q2', title => 'Q2'}, 
            { id => 'q3', title => 'Q3'}, 
            { id => 'q4', title => 'Q4'}, 
            { id => 'year', title => 'Год'}, 
        ],
        expected_params => 
        [
            {   
                name    => 'phrases',
                title   => 'Фразы',
                parent  => 'textarea',
            },
        ],
    },

    advq_hist => {
        report_type => 'predefined_sub',
        title => 'Статистика advq по неделям или по месяцам', 
        description => "Для TechSales", 
        rbac_check => { Code => [ \&RBAC2::DirectChecks::rbac_cmd_internal_user_only] },        

        code => \&advq_hist_report,
        
        expected_params => [
            {
                name => 'type', parent => 'select', title => 'Тип статистики',
                options => [
                  {name => 'По месяцам', value => 'month'},
                  {name => 'По неделям', value => 'week'},
                  ]
            },
            {
                name => 'advq_lang', parent => 'select', title => 'Язык',
                options => [
                  {name => 'Русский/Английский', value => ''},
                  {name => 'Турецкий', value => 'tr'},
                ],
            },
            {
                name => 'devices', parent => 'select', title => 'Площадка',
                options => [
                  {name => 'Все', value => 'all'},
                  {name => 'Десктопы', value => 'desktop'},
                  {name => 'Мобильные', value => 'mobile'},
                  {name => 'Только телефоны', value => 'phone'},
                  {name => 'Только планшеты', value => 'tablet'}
                ],
            },
            {   
                name    => 'phrases',
                title   => 'Фразы',
                parent  => 'textarea',
                cols    => 80,
                rows    => 10,
            },
            {   
                name    => 'region',
                title   => 'Гео-таргетинг',
                parent  => 'region',
            },
        ],
    },

    analyze_text_lang => {
        report_type => 'predefined_sub',
        title => 'Определение языка текста', 
        rbac_check => {Role => ['super', 'superreader', 'placer']},
        code => \&analyze_text_lang_report,
        expected_params => [
            {name => 'text', parent => 'textarea', cols => 80, rows => 3},
        ],
        fields  => [
            { id => 'text',      title => 'Текст', text_sort => 1},    
            { id => 'lang',      title => 'Язык'},    
            { id => 'queryrec',  title => 'Вероятности из Queryrec'},    
        ],

    },

    # Очередь отчетов
    api_queue_stat => {
        report_type => 'predefined_sub',
        code => \&api_queue_stat,
        rbac_check => { Role => ["support", "super", "superreader"] },
        title => 'Очередь отчетов',
        expected_params => [
            {
                name    => "type",
                title   => "Тип отчетов",
                options => [map {{name => $_, value => $_}} '', qw/report forecast wordstat/],
                parent  => 'select',
            }
        ],
        group_id => 'API',
    },

    # Очередь простукивания на редирект
    redirect_check_queue => {
        report_type => 'predefined_sub',
        code => \&redirect_check_queue,
        rbac_check => { Role => ["support", "super", "superreader", "limited_support", 'manager'] },
        title => 'Очередь простукивания на редирект',
        description => "Отрицательные 'максимальные ожидания' -- это запланированные на будущее проверки, все нормально, не баг.",
        field_titles => {domain => 'домен', cnt => 'объявлений', cnt_cids => 'кампаний', max_age => 'макс.время'},
        fields  => [
                { id => 'domain',  title => 'Домен', text_sort => 1},
                { id => 'cnt',  title => 'кол-во объявлений'},
                { id => 'cnt_cids',  title => 'кол-во кампаний'},
                { id => 'max_age',  title => 'максимальное ожидание (сек.)'},
            ],
        expected_params => [
            {title => 'Перепроверить ВСЕ объявления в кампаниях', type => 'cids', name => 'cids', field_size => 30, value_max_len => 1024},
            {title => 'Перепроверить список объявлений', type => 'bids', name => 'bids', field_size => 50, value_max_len => 1024}
        ],
        group_id => 'DEBUG',
    },
    
    # просмотр forecast_ctr
    forecast_ctr_debug => {
        report_type => 'predefined_sub',
        title => 'Просмотр forecast_ctr', 
        description => "Статистика ctr по фразам", 
        rbac_check => {Role => ['super', 'superreader']},

        code => \&forecast_ctr_report,
        
        fields  => [
                { id => 'phrase',  title => 'Фраза', text_sort => 1},    
                { id => 'norm_phrase', text_sort => 1},    
                { id => 'hash', text_sort => 1},    
                { id => 'ctr'},    
                { id => 'p_ctr'},    
            ],
        expected_params => 
        [
            [
            {   
                name    => 'phrases',
                title   => 'Фразы',
                parent  => 'textarea',
            },
            {   
                name    => 'hashes',
                title   => 'Phrase hashes',
                parent  => 'textarea',
            },
            ],
        ],
    },  

    # здесь была выгрузка списков email'ов для сейлз-маркетинга

    generate_promo_codes => {
        report_type => 'predefined_sub',
        title => 'Генерация списка промо-кодов', 
        description => "Для сейлз-маркетинга", 
        rbac_check => { Code => [ \&RBAC2::DirectChecks::rbac_cmd_internal_user_only] },        

        code => \&generate_promo_codes,
        
        fields  => [
             qw/code
                start_dt
                end_dt
                middle_dt
                payment
                bonus1
                bonus2
                event
                /
        ],
        expected_params => [ 
            [
             {name =>'cnt', default_value => 500, title => 'Количество'},
             {name =>'prefix', title => 'Префикс'},
            ],
            [
             {name =>'start_dt', parent => 'date', title => 'Дата начала действия'},
             {name =>'end_dt', parent => 'date', title => 'Дата окончания действия'},
            ],
            [
             {name =>'payment', default_value => 10, title => 'Минимальная сумма платежа'},
             {name =>'bonus1', title => 'Размер бонуса'},
            ],
            [
             {name =>'middle_dt', parent => 'date', title => 'Дата смены номинала'},
             {name =>'bonus2', title => 'Размер бонуса после смены номинала'},
            ],
            {name =>'event', title => 'Событие'},
            ],
        group_id => 'SpecList',
    },

    # Статистика, используемая для подсчета юнитов юнитов (по домену или по баннеру)
    api_domain_stat => {
        report_type => 'predefined_sub',
        title => 'Статистика для подсчета юнитов', 
        description => "Статистика, используемая для подсчета юнитов (собирается по домену или по баннеру)", 

        code => \&get_api_domain_stat,

        fields  => [
                { id => 'filter_domain' },
                { id => 'stat_date'},
                { id => 'accepted_items'},    
                { id => 'declined_items'},
                { id => 'declined_phrases'},
                { id => 'bad_ctr_phrases'},    
                { id => 'bad_reasons'},    
                { id => 'clicks_approx'},    
                { id => 'good_ctr_phrases'},
                { id => 'shows_approx'},
                { id => 'sum_approx'},
            ],

        expected_params => [ ['domain', 'bid'] ],
        group_id => 'API',
    },
 
    # Пользователи api за последние n дней
    actual_api_users => {
        report_type => 'predefined_sub',
        title => 'Актуальные пользователи за N дней', 
        description => "Уникальные пользователи API за последние n дней", 

        code => \&get_actual_api_users,

        fields  => [
                { id => 'login', text_sort => 1 },
                { id => 'role',text_sort => 1},
                { id => 'email',text_sort => 1},
                { id => 'count',text_sort => 0},
                { id => 'last_usage',text_sort => 1},
            ],

        expected_params => [
            {   
                name    => 'days',
                title   => 'количество дней',
                default_value => '90',
                parent  => 'text',
            },
            { title => "По приложениям", name => "application_id", type => "select", 
              options => [{name => 'Все', value => 0}, map { {name => $API::Settings::SPECIAL_API_APPS->{$_}->{name}, value => $_} } keys %$API::Settings::SPECIAL_API_APPS] },
            { title => "Application ID", name => "application_id_manual", value_max_len => 1024 },
            {name => "count_subclients", title => "Считать не агенства, а их субклиентов", type => "checkbox"}, 
        ],
        group_id => 'API',
    },

    # Статистика пользователей API интерфейса
    daily_api_users_stat => {
        report_type => 'predefined_sub',
        title => 'Статистика пользователей API интерфейса', 
        description => "Количество актуальных пользователей API по дням", 

        code => \&get_daily_api_users_stat,

        fields  => [
                { id => 'date', text_sort => 1 },
                { id => 'users'},
            ],

        expected_params => [ ['date_from', 'date_to'], 
            {   
                name    => 'actuality_period',
                title   => 'Период сбора статистики',
                default_value => '30',
                parent  => 'text',
            },
            { title => "По приложениям", name => "application_id", type => "select", 
              options => [{name => 'Все', value => 0}, map { +{name => $API::Settings::SPECIAL_API_APPS->{$_}->{name}, value => $_} } keys %$API::Settings::SPECIAL_API_APPS] },
            { title => "Application ID", name => "application_id_manual", value_max_len => 1024 },
            {name => "count_subclients", title => "Считать не агенства, а их субклиентов", type => "checkbox"}, 
           ],
        group_id => 'API',
        chart => 1,
    },

    # отслеживание редиректов с урлов
    check_redirect_domain => {
        report_type => 'predefined_sub',
        title => 'Редиректы с доменов', 
        description => "Редиректы с доменов", 
        rbac_check => { Role => ['super', 'support', 'superreader', 'limited_support', 'manager', 'placer'] },

        code => \&check_href_redirect,

        fields  => [
            { id => 'href',  title => 'Url', text_sort => 1}
            , { id => 'code',  title => 'HTTP-код ответа'}
            , { id => 'method',  title => 'Способ редиректа', text_sort => 1}
            , { id => 'req',  title => 'Заголовок запроса', text_sort => 1, details_key => 'pre',}
            , { id => 'resp',  title => 'Заголовок ответа', text_sort => 1, details_key => 'pre',}
            , { id => 'content',  title => 'Тело ответа', text_sort => 1, details_key => 'pre',}
        ],

        expected_params => [ 'href' ],
        group_id => 'DEBUG',
    },

    # кампании потенциальные мошенники
    bad_redir_campaigns => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ["support", "super", "superreader", "placer"] },
        title => 'Кампании - потенциальные мошенники',  
        code => \&bad_redir_campaigns,
        fields => [
            'cid',
            { id => 'domains', text_sort  => 1},
            { id => 'bids',
              text_sort  => 1,            
              data_cb => sub {
                  return "/registered/main.pl?cmd=searchBanners&text_search=".$_[0]->{bids}."&what=num";
              },
              details_key => 'url',
            }
        ],
        details_required => [
            'cid',
        ],
    },

    # курсы валют
    currency_rates => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ['support', 'super', 'superreader'] },
        title => 'Курсы валют',
        group_id => 'multicurrency',
        code => \&currency_rates,
        expected_params => [
            [
                {
                    name => 'currency', title => 'Валюта', type => 'select',
                    options => [map {{name => $_ . ' (' . $Currencies::_CURRENCY_DESCRIPTION{$_}->{name} . ')', value => $_}} keys %Currencies::_CURRENCY_DESCRIPTION],
                },
                'date_from', 'date_to',
            ],
        ],
        fields => [
            {id => 'date', title => 'Дата'},
            {id => 'our_rate', title => 'Курс в рублях из нашей БД', text_sort => 0},
            {id => 'balance_rate', title => 'Курс в рублях из Баланса', text_sort => 0},
        ],
    },

    # просмотр лога событий
    search_eventlog => {
        report_type => 'predefined_sub',
        rbac_check => { Perm => 'ShowLogs' },
        title => 'Лог событий',
        code => \&search_eventlog,
        expected_params => [
            {name => 'logins', title => 'логины', type => 'text', field_size => 50, value_max_len => 1024},
            {name => 'clientids', title => 'ClientID', type => 'text', field_size => 50, value_max_len => 1024},
            'date_from',
            'date_to',
            {
                name => 'event_type', title => 'Вид события', type => 'select',
                options => [{name => 'все', value => 'any'}, map {{name => $EventLog::EVENTS{$_}->{name}, value => $_}} keys %EventLog::EVENTS],
            },
            'cids',
            {name => 'bids', title => 'Номера объявлений', type => 'text', field_size => 50},
            {name => 'bids_ids', title => 'Идентификаторы фраз', type => 'text', field_size => 50},
            {name => 'only_last_event', title => 'Только последнее события каждого типа по объекту', type => 'checkbox'},
        ],
        fields => [
            {id => 'ClientID', title => 'Клиент'},
            {id => 'eventtime', title => 'Время события', text_sort => 0},
            {id => 'type', title => 'Вид события'},
            {id => 'cid', title => 'Кампания'},
            {id => 'bid', title => 'Объявление'},
            {id => 'bids_id', title => 'Фраза'},
            {id => 'params', title => 'Параметры события', details_key => 'pre'},
        ],
        details_required => ['ClientID', 'cid', 'bid', 'bids_id'],
    },

    # поиск пользователей по IP
    search_spam_users => {
        report_type => 'predefined_sub',
        code => \&search_spam_users,
        rbac_check => { Role => ["support", "super", "superreader", "placer"] },
        title => 'Поиск логинов по IP',
        fields  => [
                { id => 'login', type => 'login', title => 'Логин', text_sort => 1},
                { id => 'blocked',  title => 'Заблокирован', text_sort => 1},
                { id => 'domain',  title => 'Домен', text_sort => 1},
                { id => 'cnt_declined',  title => 'Отклонено', text_sort => 1},
                { id => 'cnt_accepted',  title => 'Принято', text_sort => 1},
                { id => 'ip',  title => 'IP', details_key => 'ip'},
            ],
        expected_params => [
            [
                {title => 'Логин', type => 'login', name => 'login', field_size => 30, value_max_len => 1024},
                {title => 'Исходный IP', type => 'ip', name => 'ip', field_size => 50, value_max_len => 1024},
                {name => 'logapi', title => 'Лог API', type => 'checkbox'}
            ],[
                {name => 'date_from', parent => 'date', title => 'с'},
                {name => 'date_to', parent => 'date', title => 'по'},
            ]
        ],
        details_required => [
            'login'
        ],
    },

    # получить по oauth токену владельца и описание приложения
    auth_api_token => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ['super', 'support', 'superreader'] },
        title => 'Авторизация токена API',
        method => 'POST',
        code => \&auth_api_token,
        expected_params => [
            {name => 'oauth_token', title => 'oauth токен', type => 'text', field_size => 50, value_max_len => 100},
        ],
        fields => [
            {id => 'uid', title => 'UID'},
            {id => 'login', title => 'Логин'},
            {id => 'application_id', title => 'application_id'},
            {id => 'application_name', title => 'Название приложения'},
            {id => 'application_page', title => 'Страница'},
            {id => 'scope', title => 'Права на доступ'},
            
            {id => 'token_status', title => 'Статус токена'},
            {id => 'status', title => 'Результат'},
        ],
        details_required => ['oauth_token'],
        group_id => 'API',
    },

    # просмотр списка страна-валюта-фирма, используемого при создании нового клиента
    country_currency_firm => {
        report_type => 'predefined_sql',
        rbac_check => { Role => ['support', 'super', 'superreader'] },
        title => 'Список страна-валюта-фирма с разбивкой по признаку агентства',
        group_id => 'multicurrency',
        DB => PPCDICT, 
        sql => q/
            SELECT gr.name AS country
                 , GROUP_CONCAT(DISTINCT cc.currency SEPARATOR ', ') AS currencies
                 , GROUP_CONCAT(DISTINCT cc.firm_id SEPARATOR ', ') AS firm
                 , cc.is_agency
            FROM country_currencies cc
            LEFT JOIN geo_regions gr ON cc.region_id = gr.region_id
            GROUP BY cc.is_agency, cc.region_id
        /,
        fields => [
            {id => 'is_agency', title => 'Признак агентства'},
            {id => 'country', title => 'Страна'},
            {id => 'currencies', title => 'Валюты'},
            {id => 'firm', title => 'Фирма'},
        ],
        details_required => ['firm'],
    },

    # просмотр списка страна-валюта-фирма по конкретному клиенту
    client_country_currency => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ['support', 'super', 'superreader'] },
        title => 'Список страна-валюта по клиенту',
        group_id => 'multicurrency',
        code => \&client_country_currency,
        expected_params => [{name => 'client_login', title => 'Логин', type => 'text',}],
        fields => [
            {id => 'country', title => 'Страна'},
            {id => 'currencies', title => 'Валюты'},
            {id => 'last_update', title => 'Время последнего обновления'},
        ],
    },

    # просмотр графика скидок клиента
    client_discount_schedule => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ['support', 'super', 'superreader'] },
        title => 'График скидок клиента',
        code => \&client_discount_schedule,
        expected_params => [{name => 'client_login', title => 'Логин', type => 'text',}],
        fields => [
            {id => 'date_from', title => 'С', text_sort => 0},
            {id => 'date_to', title => 'По', text_sort => 0},
            {id => 'discount', title => 'Скидка, %', text_sort => 0},
        ],
    },

    # минимальные ставки на поиске на домены
    domains_min_price => {
        report_type => 'predefined_sql',
        title => 'Минимальные ставки на поиске на домены', 
        DB => PPCDICT,
        sql => 'SELECT filter_domain, min_price / 1e6 AS min_price FROM cpm_limit_domains',
        fields => [
            { id => 'filter_domain', title => 'Домен главного зеркала', text_sort => 1 },
            { id => 'min_price', title => 'Минимальная ставка на поиске, у.е.' },
        ],
    },

    banner_images_search => {
        report_type => 'predefined_sub',
        title => 'Поиск объявлений с картинкой',
        description => 'Поиск баннера с картинкой по ее ID',
        code => \&banner_images_search,

        expected_params => [  
            { name => 'image_id', title => '№ баннеров с картинкой, через запятую' }, 
        ],
        
        fields => [
            { id => 'bid', title => 'Номер баннера' },
            { id => 'banner', title => 'Ссылка на баннер' },
        ],
    },

    optimizing_campaign_requests_by_media => {
        report_type => 'predefined_sql',
        title => 'Список заявок на оптимизацию по медиапланеру',  
        DB => PPC(shard => 'all'),
        sql => 'SELECT ocr.cid, ocr.status, ocr.create_time, ocr.ready_time, ocr.accept_time FROM optimizing_campaign_requests ocr WHERE ocr.MediaUID = ? AND ocr.create_time BETWEEN ? AND (? + INTERVAL 1 DAY)',
        make_bind_values_arr => sub {
            my $row = $_[0];
            my $login = $row->{login};
            my $uid;
            if ($login) {
                $uid = get_uid_by_login2($login);
                if (!$uid) {
                    error("no uid found for login $login");
                }
            } else {
                return [-1, 0, 0];
            }
            return [$uid, $row->{date_from}, $row->{date_to}];
        },

        fields => [
            { id => 'cid', title => '№ кампании' },
            { id => 'status', title => 'Статус заявки'},    
            { id => 'create_time', title => 'Время создания заявки'},    
            { id => 'ready_time', title => 'Окончание оптимизации'},    
            { id => 'accept_time', title => 'Принятие заявки пользователем'},
        ],
        details_required => [ 'cid' ],
        expected_params => [
            {name => 'login', parent => 'logins', title => 'Логин'},
            {name => 'date_from', parent => 'date', title => 'cозданные с'},
            {name => 'date_to', parent => 'date', title => 'созданные по'},
        ],
        group_id => 'DEBUG',
    },

    test_data_generator_campaign => {
        report_type      => 'predefined_sub',
        title            => 'Кампании для тестирования',
        code             => \&test_data_generator_campaign_report,
        details_required => [ qw( cid ) ],
        group_id         => 'DEBUG',
        index_list       => 1,

        description =>
            'Количество групп учитывается с погрешностью в 20% в обе стороны',

        fields => [
            { id => 'cid',           title => 'Кампания' },
            { id => 'phrases_count', title => 'Количество групп' }
        ],

        expected_params => [
            [
                {
                    name          => 'type',
                    parent        => 'select',
                    title         => 'Тип кампании',
                    default_value => 'text',

                    options => [
                        { name => 'Текстово-графические объявления', value => 'text' },
                        { name => 'Динамические объявления',         value => 'dynamic' },
                        { name => 'Реклама мобильных приложений',    value => 'mobile_content' },
                        { name => 'Смарт-баннеры',                   value => 'performance' },
                        { name => 'Баннер на поиске',                value => 'mcbanner' },
                        { name => 'Медийная кампания',               value => 'cpm_banner' },
                        { name => 'Медийная кампания со сделками',   value => 'cpm_deals' },
                        { name => 'Общий счёт',                      value => 'wallet' },
                    ],
                },
            ],
            [
                {
                    name          => 'phrases_count',
                    parent        => 'text',
                    title         => 'Количество групп',
                    default_value => -1,
                    field_size    => 10,
                },
            ],
            [
                {
                    name   => 'metrika_goals',
                    parent => 'checkbox',
                    title  => 'Есть цели в метрике',
                },
            ],
            [
                {
                    name          => 'status',
                    parent        => 'select',
                    title         => 'Состояние',
                    default_value => '',

                    options => [
                        { name => 'Любое',        value => '' },
                        { name => 'Активна',      value => 'active' },
                        { name => 'Неактивна',    value => 'inactive' },
                        { name => 'На модерации', value => 'moderate' },
                    ],
                },
            ],
            [
                {
                    name   => 'nonzero_balance',
                    parent => 'checkbox',
                    title  => 'Ненулевой остаток денег',
                },
            ],
        ]
    },

    test_data_generator_banner => {
        report_type      => 'predefined_sub',
        title            => 'Объявления для тестирования',
        code             => \&test_data_generator_banner_report,
        details_required => [ qw( bid ) ],
        group_id         => 'DEBUG',
        index_list       => 1,

        descrption => '',

        fields => [
            { id => 'bid', title => 'Объявление' },
            { id => 'pid', title => 'Группа' }
        ],

        expected_params => [
            [
                {
                    name          => 'status',
                    parent        => 'select',
                    title         => 'Состояние',
                    default_value => '',

                    options => [
                        { name => 'Любое',         value => '' },
                        { name => 'Активное',      value => 'active' },
                        { name => 'Остановленное', value => 'stopped' },
                        { name => 'Отклонённое на модерации',
                            value => 'moderate_rejected' },
                        { name => 'Архивное',      value => 'archived' },
                    ],
                },
            ],
            [
                {
                    name   => 'with_vcard',
                    parent => 'checkbox',
                    title  => 'С визиткой',
                },
            ],
            [
                {
                    name   => 'with_href',
                    parent => 'checkbox',
                    title  => 'Со ссылкой на сайт',
                },
            ],
            [
                {
                    name   => 'with_sitelinks',
                    parent => 'checkbox',
                    title  => 'С быстрыми ссылками',
                },
            ],
            [
                {
                    name   => 'with_image',
                    parent => 'checkbox',
                    title  => 'С картинкой',
                },
            ],
            [
                {
                    name   => 'with_retargeting',
                    parent => 'checkbox',
                    title  => 'С условиями подбора аудитории',
                },
            ],
            [
                {
                    name   => 'lowctr_disabled',
                    parent => 'checkbox',
                    title  => 'Отключёно за низкий CTR',
                },
            ],
        ]
    },

    testusers => {
        report_type      => 'predefined_sub',
        title            => 'Тестовые суперпользователи',
        code             => \&testusers_report,
        details_required => [ qw( uid ) ],
        group_id         => 'DEBUG',
        index_list       => 1,

        description => 'Пользователей можно добавлять и блокировать с помощью формы:',

        fields => [
            { id => 'uid',          title => 'Пользователь' },
            { id => 'domain_login', title => 'Доменный логин', text_sort => 1 },
            { id => 'role',         title => 'Роль',           text_sort => 1 },
        ],

        expected_params => [
            [
                {
                    name       => 'login',
                    parent     => 'text',
                    title      => 'Логин',
                    field_size => 30,
                },
            ],
            [
                {
                    name   => 'set_inactive',
                    parent => 'checkbox',
                    title  => 'Прекратить выдавать роль',
                },
            ],
            [
                {
                    name       => 'domain_login',
                    parent     => 'text',
                    title      => 'Доменный логин',
                    field_size => 30,
                },
            ],
            [
                {
                    name          => 'role',
                    parent        => 'select',
                    title         => 'Роль',
                    default_value => 'super',
                    options       => [
                        { name => 'manager',     value => 'manager' },
                        { name => 'media',       value => 'media' },
                        { name => 'placer',      value => 'placer' },
                        { name => 'super',       value => 'super' },
                        { name => 'superreader', value => 'superreader' },
                        { name => 'support',     value => 'support' },
                        { name => 'limited_support',     value => 'limited_support' },
                        { name => 'internal_ad_admin',   value => 'internal_ad_admin' },
                        { name => 'internal_ad_manager', value => 'internal_ad_manager' },
                        { name => 'internal_ad_superreader', value => 'internal_ad_superreader' },
                    ],
                },
            ],
        ],
    },

    currency_convert_queue => {
        report_type => 'predefined_sub',
        title => 'Очередь конвертации',
        description => "Показывает все добавленные заявки на конвертацию в реальную валюту",
        group_id => 'multicurrency',
        code => \&currency_convert_queue_report,

        fields => [
            { id => 'start_convert_at', title => 'Когда начнётся' },
            { id => 'login', title => 'Логин главного представителя' },
            { id => 'create_time', title => 'Создана' },
            { id => 'convert_type', title => 'Тип конвертации' },
            { id => 'state', title => 'Текущее состояние' },
            { id => 'new_currency', title => 'Валюта' },
            { id => 'country', title => 'Страна' },
            { id => 'convert_started_at', title => 'Началась' },
            { id => 'convert_finished_at', title => 'Закончилась' },
            { id => 'in_state_since', title => 'Находится в этом состоянии с' },
            { id => 'finish_forecast_time', title => 'Прогноз завершения' },
            { id => 'balance_convert_finished', title => 'Конвертация в Балансе завершена' },
        ],
        details_required => [ qw( login country ) ],
        expected_params => [
            {name => 'include_done', parent => 'checkbox', title => 'включая сконвертировавшихся'},
            {name => 'with_finish_forecast', parent => 'checkbox', title => 'рассчитать прогноз окончания конвертации'},
        ],
    },

    force_currency_convert_queue => {
        report_type => 'predefined_sub',
        title => 'Очередь принудительной конвертации',
        description => "Показывает все добавленные заявки на принудительную конвертацию в реальную валюту",
        group_id => 'multicurrency',
        code => \&force_currency_convert_queue_report,

        fields => [
            { id => 'convert_date', title => 'Запланированная дата конвертации' },
            { id => 'login', title => 'Логин главного представителя' },
            { id => 'accepted_at', title => 'Когда принял оферту' },
            { id => 'currency', title => 'Валюта' },
            { id => 'country', title => 'Страна' },
        ],
        details_required => [ qw( login country ) ],
    },

    currency_convert_teaser_trigger => {
        report_type => 'predefined_sub',
        title => 'Включение/выключение тизера конвертации',
        description => "Позволяет суперам включать/выключать тизер конвертации в валюты",
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        code => \&currency_convert_teaser_trigger,
        expected_params => [
            {name => 'teaser_disabled', parent => 'checkbox', title => 'отключить тизер'},
            {name => 'set_teaser_disabled', parent => 'hidden', default_value => 1},
        ],
    },

    currency_convert_modify_percent => {
        report_type => 'predefined_sub',
        title => 'Процент клиентов для включения тизера конвертации без копирования',
        description => "Позволяет суперам изменять процент клиентов, которым можно включать тизер конвертации без копирования",
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        code => \&currency_convert_modify_percent,
        expected_params => [
            {name => 'percent', title => 'Процент клиентов'},
        ],
    },

    api5_java_proxy_service_operations => {
        report_type => 'predefined_sub',
        title => 'Изменение операций api5, которые проксируются в java',
        description => "Позволяет DevOps-ам изменять операции api5, которые проксируются в java.",
        group_id => 'API',
        rbac_check => {
            # дополнительно в коде метода есть проверка на сохранение в проде - только для is_devops
            Role => ['super', 'superreader', 'support'],
            },
        code => \&api5_java_proxy_service_operations,
        expected_params => [
            {name => 'set_property', parent => 'hidden', default_value => 1},
            map {
                +{name => "method_".$_, title => $_, parent => 'checkbox' }
            } sort keys %Settings::API5_JAVA_ALLOW_PROXYING_OPERATION
        ],
    },

    priority_clients_api5_units_threshold => {
        report_type => 'predefined_sub',
        title => 'Порог дневного лимита баллов API5 для приоритетных клиентов',
        description => 'Позволяет изменить количество баллов, после которого клиент считается приоритетным',
        submit_title => 'Изменить',
        method => 'POST',
        group_id => 'API',
        rbac_check => {Role => ['super']},
        code => \&priority_clients_api5_units_threshold,
        expected_params => [
            {name => 'threshold', title => 'Баллы'},
        ],
    },

    teaser_data_fetch_parallel_level => {
        report_type => 'predefined_sub',
        title => 'Количество потоков для обновления данных для мультивалютного тизера',
        description => 'Позволяет изменять количество потоков для обновления данных для мультивалютного тизера. Это значение на каждый шард, т.е. общее количество потоков будет умноженным на количество шардов.',
        group_id => 'multicurrency',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&teaser_data_fetch_parallel_level,
        expected_params => [
            {name => 'parallel_level', title => 'Количество потоков'},
        ],
    },

    force_convert_daily_client_limit => {
        report_type => 'predefined_sub',
        title => 'Количество клиентов на принудительную конвертацию в сутки [письмо]',
        description => 'Позволяет суперам изменять количество клиентов, которым будем назначать дату конвертации в каждом шарде на принудительную конвертацию каждые сутки',
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        code => \&force_convert_daily_client_limit,
        expected_params => [
            { name => 'limit', title => 'Количество клиентов' },
        ],
    },

    force_convert_daily_queue_client_limit => {
        report_type => 'predefined_sub',
        title => 'Количество клиентов на принудительную конвертацию в сутки [конвертация]',
        description => 'Позволяет суперам изменять количество клиентов, которых будем ставить в каждом шарде на принудительную конвертацию каждые сутки',
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        code => \&force_convert_daily_queue_client_limit,
        expected_params => [
            { name => 'limit', title => 'Количество клиентов' },
        ],
    },

    mass_enable_agency_clients_currency_convert_teaser => {
        report_type => 'predefined_sub',
        title => 'Массовое включение тизера конвертации для субклиентов агентства',
        description => "Позволяет суперам включать тизер конвертации для всех субклиентов агентства",
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        code => \&mass_enable_agency_clients_currency_convert_teaser,
        expected_params => [
            { name => 'logins', title => 'Логины агентств', parent => 'textarea', },
        ],
        fields => [
            { id => 'ClientID', title => 'Клиент' },
            { id => 'teaser_enabled', title => 'Включили ли тизер' },
            { id => 'reasons', title => 'Почему нет' },
        ],
        details_required => [ qw( ClientID ) ],
    },

    mass_enable_freedom_for_clients => {
        report_type => 'predefined_sub',
        title => 'Включение/выключение клиентам свободы',
        description => 'ВНИМАНИЕ! Использовать только для клиентов, зависших на конвертации в валюту копированием. После завершения конвертации отключить!
Позволяет включить/выключить клиентам по списку галочку "свободы" (поле allow_create_scamp_by_subclient таблицы clients)',
        group_id => 'multicurrency',
        rbac_check => {Code => [ \&RBAC2::DirectChecks::rbac_cmd_all_super_specific_users]},
        code => \&mass_enable_freedom_for_clients,
        expected_params => [
            { name => 'ClientIDs', title => 'ClientIDs', parent => 'textarea', },
            { name => "just_do_it", title => "Да, я прочитал(a) предупреждение и принимаю ответственность за возможные поломки", type => "checkbox" },
            { name => "enable_freedom", title => "Включить свободу", type => "checkbox" },
        ],
        fields => [
            { id => 'ClientID', title => 'ClientID' },
            { id => 'allow_freedom', title => 'Свобода включена' },
        ],
        submit_title => iget_noop('Применить!'),
    },

    prohibit_payments_for_login => {
        report_type => 'predefined_sub',
        title => 'Запрет/разрешение оплаты на всех кампаниях на логине',
        description => '',

        rbac_check => {Role => ['super', 'support']},

        code => \&prohibit_payments_for_login,
        fields  => [
            {
                id => 'login',
                title => 'Логин пользователя',
            },
            {
                id => 'result',
                title => 'Результат операции',
            },
        ],
        expected_params => [
            {
                name    => 'logins',
                title   => "Список логинов",
                parent  => 'textarea',
                cols => 32, rows => 10,
            },
            {
                name    => 'reenable',
                title   => 'Разрешить оплату',
                type    => 'checkbox',
            },
        ],
    },

    reset_moderation_docs_limit => {
        report_type      => 'predefined_sub',
        title            => 'Сброс лимита загрузки файлов для модерации',
        code             => \&reset_moderation_docs_limit,
        group_id         => 'DEBUG',
        rbac_check       => {Role => ['super']},

        expected_params => [
            [
                {
                    name          => 'login',
                    parent        => 'logins',
                    title         => 'Логин',
                },
            ],
        ]
    },

    geotools_reg_by_ip => {
        report_type => 'predefined_sub',
        group_id => 'DEBUG',
        title => 'GeoTools - получение региона по ip-адресу',
        description => 'получение региона по ip-адресу (для тестирования новых версий geobase)',

        code => \&geotools_reg_by_ip,

        expected_params => [
            { name    => 'ips', title   => "ip-адреса",
              parent  => 'textarea', cols => 80, rows => 10,
            },
            ],
    },

    agency_subclient_clientids => {
        report_type => 'predefined_sub',
        title => 'ClientID субклиентов агентства',
        index_list => 1,
        description => '',

        code => \&agency_subclient_clientids_report,
        rbac_check => { Role => [ qw( super suppereader ) ] },

        fields => [
            { id => 'uid', title => 'Клиент', },
            { id => 'clientid', title => 'ClientID в базе', },
            { id => 'clientid_from_rbac', title => 'ClientID в RBAC', },
        ],
        details_required => ['uid'],
        expected_params => [
            {
                name          => 'agency_login',
                parent        => 'logins',
                title         => 'Логин агентства',
            },
            {
                name          => 'only_mismatching_clientids',
                parent        => 'checkbox',
                title         => 'Только клиенты с несовпадающими ClientID',
                default_value => 1,
            },
        ],

        group_id => 'DEBUG',
    },

    force_unbind_agency => {
        report_type => 'predefined_sub',
        title => 'Отвязать клиента от агентства',
        description => "ТОЛЬКО ДЛЯ БЕТ И ТС:\n" .
            "Все кампании клиента будут удалены (даже если через интерфейс их удалить нельзя),\n" .
            "а потом клиент будет отвязан от агентства",

        # процедура в Code должна возвращать значение 0 "разрешено" или 1 "запрещено"
        # этот отчёт доступен только на бетах и ТС
        rbac_check => { Role => [ qw( super superreader ) ], Code => [ sub { is_production() } ] },

        code => \&force_unbind_agency_report,

        fields => [],
        expected_params => [
            { name => 'client_login', parent => 'logins', title => 'Логин клиента' },
            { name => 'agency_login', parent => 'logins', title => 'Логин агентства' },
        ],

        group_id => 'DEBUG',
    },

    manage_bs_resync_constants => {
        report_type => 'predefined_sub',
        code => \&manage_bs_resync_constants,
        group_id => 'BSExport',
        title => 'Управление параметрами ленивой переотправки в БК',
        description => "Отображает и задает параметры обработки ленивой очереди отправки в БК.

Приоритетная переотправка - если размер или возраст очереди больше пороговых значений,
то будут обрабатываться только записи с приоритетом больше $BS::ExportMaster::RESYNC_PRIORITY_BORDER.

Умолчания (используются экспорт-мастером для нулевых или неопределенных значений):
Размер чанка при обычной переотправке: $BS::ExportMaster::DEFAULT_RESYNC_CHUNK_SIZE
Размер чанка при приоритетной переотправке: $BS::ExportMaster::DEFAULT_RESYNC_PRIORITY_CHUNK_SIZE
Пороговое значение интегрального размера очереди: $BS::ExportMaster::DEFAULT_RESYNC_ISIZE_BORDER
Пороговое значение возраста std-очереди: $BS::ExportMaster::DEFAULT_RESYNC_AGE_BORDER
Пороговое значение интегрального размера heavy-очереди: $BS::ExportMaster::DEFAULT_RESYNC_HEAVY_ISIZE_BORDER

Для сброса значений к умолчаниям - нужно сохранить 0.
",
        rbac_check => { Role => ['super'], AllowDevelopers => 1 },
        expected_params => [
            {
                name => 'RESYNC_CHUNK_SIZE',
                title => 'Размер чанка обычной переотправки',
            }, {
                name => 'RESYNC_PRIORITY_CHUNK_SIZE',
                title => 'Размер чанка приоритетной переотправки',
            }, {
                name => 'RESYNC_ISIZE_BORDER',
                title => 'Пороговое значение интегрального размера очереди',
            }, {
                name => 'RESYNC_AGE_BORDER',
                title => 'Пороговое значение возраста очереди (в минутах)',
            }, {
                name => 'RESYNC_HEAVY_ISIZE_BORDER',
                title => 'Пороговое значение интегрального размера heavy-очереди',
            },
        ],
        fields => [
            { id => 'RESYNC_CHUNK_SIZE' },
            { id => 'RESYNC_PRIORITY_CHUNK_SIZE' },
            { id => 'RESYNC_ISIZE_BORDER' },
            { id => 'RESYNC_AGE_BORDER' },
            { id => 'RESYNC_HEAVY_ISIZE_BORDER' },
        ],
    },

    view_clients_stat => {
        report_type => 'predefined_sub',
        title => 'Просмотр бюджетов клиентов (clients_stat)',
        description => 'Просмотр бюджетов собранных данных о бюджетах клиентов как самостоятельно, так и в составе бренда',
        rbac_check => {Role => ['super', 'superreader']},
        code => \&view_clients_stat,
        fields => [
            {
                id => 'type',
                title => 'type',
            },
            {
                id => 'total_sum_rub',
                title => 'суммарные траты клиента за всё время в рублях с НДС',
            },
            {
                id => 'active_28days_sum_rub',
                title => 'траты клиента в рублях с НДС, за последние 28 дней активности',
            },
            {
                id => 'daily_spent_rub_estimate',
                title => 'приблизительный расход клиента за день в рублях с НДС, оценка сверху по данным за последний месяц',
            },
        ],
        expected_params => [
            { name => 'login', parent => 'logins', title => 'Логин' },
        ],
    },

  manage_moderate_url => {
        report_type => 'predefined_sub',
        rbac_check => { Role => ['super'], AllowDevelopers => 1 },
        description => sprintf('Флаг использования прокси модерации для jsonrpc запросов. (ppc_properties  name = "%s")
Затрагивает все запросы в модерацию, кроме быстрой модерации (soap) и конфигураций DevTest, Dev7.
Значения:
 1 - используем url прокси модерации, из Moderate::Settings::MODERATE_JSON_PROXY_URL
 0 (или не задано) - используется старый url модерации из Moderate::Settings::MODERATE_JSON_URL', $Moderate::Settings::MODERATE_JSON_RPC_URL_PROPERTY),
        title => 'Использовать прокси модерации',
        code => \&manage_moderation_url,
        expected_params => [
            { name => 'flag' },
        ],
        fields => [
            {id => 'result', title => 'результат выполнения' },
            {id => 'current_value', title => 'Текущее значение'},
        ],
        group_id => 'Moderate'
    },

    # Пока таких пользователей немного, получаем их полный список
    locked_common_wallet_queue => {
        report_type => 'predefined_sql',
        title => 'Пользователи с неотключаемым общим счётом',
        description => "Пользователи, для которых была заблокирована возможность отключения общего счёта",
        DB   => PPC(shard => 'all'),
        sql  => "SELECT ClientID, ClientID as ci, state, lock_set_time as date
                   FROM wallet_campaigns_lock w
                  ORDER BY date DESC",
        fields => [
            { id => 'ci', title => 'ClientID' },
            { id => 'ClientID', title => 'Клиент' },
            { id => 'date', text_sort => 1, title => 'Дата блокировки отключения счёта'},
            { id => 'state', text_sort => 1},
        ],
        details_required => ['ClientID'],
        group_id => 'experiment',
    },

    mass_set_users_hidden => {
        report_type => 'predefined_sub',
        title => 'Массовая пометка пользователей тестовыми',
        description => 'Интерфейс для массовой пометки пользователей тестовыми (users.hidden), чтобы не попадали под удалялки и чистилки',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&mass_set_users_hidden,
        expected_params => [
            {name => 'logins', parent => 'textarea', title => 'Логины'},
        ],
        group_id => 'DEBUG',
    },

    api_deprecated_method_error_show_properties => {
        report_type => 'predefined_sub',
        title => 'Параметры выдачи клиентам API ошибки для устаревших методов',
        description => "Позволяет суперам изменять параметры выдачи ошибки для устаревших методов API",
        group_id => 'API',
        rbac_check => {Role => ['super']},
        code => \&api_deprecated_method_error_show_properties,
        expected_params => [
            {name => 'rounds_count', title => 'Количество интервалов выдачи в сутки'},
            {name => 'interval', title => 'Длительность интервала в минутах'},
        ],
    },

    mass_archive_clients_for_agency => {
        report_type => 'predefined_sub',
        title => "Массовая архивация клиентов агентства",
        description => "Позволяет заархивировать клиентов агентства по списку ClientID или логинов",
        rbac_check => {Role => ['super', 'placer']},
        expected_params => [
            {name => 'agency_login',        title => 'Логин агентства'},
            {name => 'agency_clients_list', title => 'Список ClientID или логинов клиентов агентства',
                parent => "textarea",
                cols => 32, rows => 20,
            },
            {name => 'clients_list_type',   title => 'В списке указаны',
                parent => 'select',
                default_value => 'logins',
                options => [
                    {name => 'Логины',      value => 'logins'},
                    {name => 'ClientID',    value => 'clientid'},
                ],
            },
        ],
        fields => [
            { id => 'clientid', title => 'ClientID' },
            { id => 'login',    title => 'Логин' },
            { id => 'result',   title => 'Результат'},
        ],
        code => \&mass_archive_clients_for_agency,
    },

    enable_context_shows_for_logins => {
        report_type => 'predefined_sub',
        title => 'Включение показов в сети для списка логинов',
        description => 'Включение показов в сети для всех кампаний по списку логинов в стандартном режиме:
Удерживать расход: 100%
Максимальная цена клика: 100%
Удерживать среднюю цену клика, ниже цены поиска - да
Не учитывать предпочтения пользователей - нет',
        code => \&enable_context_shows_for_logins,
        rbac_check => { Role => [ 'super' ] },
        group_id => 'experiment',
        expected_params => [
            {
                name    => 'logins',
                title   => 'Список логинов',
                parent  => 'textarea',
                cols => 32,
                rows => 20,
            },
        ],
        fields => [
            { id => 'login', title => 'Логин' },
            { id => 'cid', title => 'Кампания'},
            { id => 'result', title => 'Результат'},
        ],
        submit_title => iget_noop('Включить!'),
    },

    bs_sync_creatives => {
        report_type => 'predefined_sub',
        code => \&bs_sync_creatives,
        title => 'Принудительная синхронизация креативов с BS',
        description => 'Создает задания на синхронизацию указанных или всех креативов с BS. Опционально вместо полной может быть произведена только синхронизация скриншотов.',
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        group_id => 'BSExport',
        expected_params => [
            {
                name => 'sync_mode',
                type => 'select',
                title => 'Тип синхронизации',
                default_value => 0,
                options => [
                    { name => 'Полная', value => 0 },
                    { name => 'Только скриншоты', value => 1 },
                ],
            },
            {
                name    => 'creative_ids',
                title   => "Идентификаторы креативов, через запятую",
                parent  => 'textarea',
            },


            
        ],
        fields => [
            { id => 'result', title => 'результат выполнения' },
            { id => 'shard', title => 'шард' },
            { id => 'db_res', title => 'заданий на сихронизацию' },
        ],
    },

    bs_disable_smart_tgo => {
        report_type => 'predefined_sub',
        code => \&bs_disable_smart_tgo,
        title => 'Принудительная остановка смарт-тго баннеров',
        description => 'Остановка смарт-тго баннеров по списку OrderID',
        rbac_check => { Role => [ 'super' ], AllowDevelopers => 1 },
        group_id => 'BSExport',
        expected_params => [
            {
                name    => 'order_ids',
                title   => "БК-идентификаторы кампаний, через запятую",
                parent  => 'textarea',
            },
            {
                name    => 'need_stop_banners',
                title   => "Остановить баннеры",
                type    => 'checkbox',
            },
        ],
        fields => [
            { id => 'OrderID', title => 'OrderID' },
            { id => 'cid', title => 'cid' },
            { id => 'bid', title => 'bid' },
            { id => 'creative_id', title => 'creative_id' },
        ],
    },

    add_agency_clients_to_currency_convert_queue => {
        report_type => 'predefined_sub',
        title => "Конвертация по списку логинов для субклиентов агентства",
        description => "Позволяет суперам ставить на конвертацию субклиентов агентства по списку логинов.
ВНИМАНИЕ! Клиенты на конвертацию текущей ночью должны быть добавлены до 20:00 по московскому времени.
Клиенты, добавленные позже будут сконвертированы в следующую ночь.
Если конвертация уже запланированна, то её дату нельзя изменить в этом отчете.
        ",
        group_id => 'multicurrency',
        rbac_check => {Role => ['super']},
        expected_params => [
            {name => 'agency_login',        title => 'Логин агентства'},
            {name => 'convert_date',        title => 'Дата конвертации YYYY-MM-DD (опционально, по умолчанию ближайшая возможная)'},
            {name => 'agency_clients_list', title => 'Список логинов клиентов агентства',
                parent => "textarea",
                cols => 32, rows => 20,
            },
        ],
        fields => [
            { id => 'clientid',     title => 'ClientID' },
            { id => 'login',        title => 'Логин' },
            { id => 'convert_date', title => 'Дата конвертации' },
            { id => 'result',       title => 'Результат'},
        ],
        code => \&add_agency_clients_to_currency_convert_queue,
    },

    mass_unblock_users => {
        report_type => 'predefined_sub',
        title => "Массовая блокировка/разблокировка пользователей",
        description => "Позволяет блокировать/разблокировать пользователей по списку логинов. В Модерацию отправляется сообщение, что пользователя нужно добавить или убрать из черного списка.
ВНИМАНИЕ! Разблокировка пользователей в фишках невозможна.",
        rbac_check => {Role => ['super']},
        expected_params => [
            {name => 'logins_list', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 20,
            },
            {name => 'do_block', title => 'Заблокировать',
                parent => 'checkbox',
            },
        ],
        fields => [
            { id => 'login',    title => 'Логин' },
            { id => 'result',   title => 'Результат'},
        ],
        code => \&mass_unblock_users,
    },

    mass_drop_cant_unblock_client_flag => {
        report_type => 'predefined_sub',
        title => "Массовое снятие флага cant_unblock",
        description => "Позволяет снять флаг cant_unblock по списку логинов, после этого пользователя можно разблокировать.",
        rbac_check => {Code => [ \&RBAC2::DirectChecks::rbac_cmd_specific_super_or_support_or_sustain ]},
        expected_params => [
            {name => 'logins_list', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 20,
            },
        ],
        fields => [
            { id => 'login',    title => 'Логин' },
            { id => 'result',   title => 'Результат'},
        ],
        code => \&mass_drop_cant_unblock_client_flag,
    },

    remove_clients_from_currency_convert_queue => {
        report_type => 'predefined_sub',
        title => "Отмена конвертации по списку логинов",
        description => "Позволяет суперам отменить конвертацию клиентов по списку логинов.
ВНИМАНИЕ! Можно отменить конвертацию только в статусе NEW. После отмены необходимо завести тикет в очереди SUPBS на откат валютной статистики и сброс валютности в БК",
        group_id => 'multicurrency',
        rbac_check => {Code => [ \&RBAC2::DirectChecks::rbac_cmd_specific_super]},
        expected_params => [
            {name => 'logins', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 20,
            },
        ],
        code => \&remove_clients_from_currency_convert_queue,
    },

    mass_set_users_not_convert_to_currency => {
        report_type => 'predefined_sub',
        title => 'Массовая пометка пользователей неконвертируемыми в валюту',
        description => 'Интерфейс для массовой пометки пользователей неконвертируемыми в валюту (clients_options.client_flags.not_convert_to_currency)',
        group_id => 'multicurrency',
        rbac_check => {Role => ['super', 'support'], AllowDevelopers => 1},
        expected_params => [
            {name => 'flag_disabled', parent => 'checkbox', title => 'снять флаг неконвертируемости'},
            {name => 'logins', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 20,
            },
        ],
        code => \&mass_set_users_not_convert_to_currency,
    },

    not_convert_to_currency_user_list => {
        report_type => 'predefined_sub',
        title => 'Спиcок пользователей отмеченных неконвертируемыми в валюту',
        description => 'Пользователи, которые не будут сконвертированы в валюту (установлен флаг clients_options.client_flags.not_convert_to_currency)',
        group_id => 'multicurrency',
        rbac_check => {Role => ['super','superreader'], AllowDevelopers => 1},
        fields => [
            { id => 'login',        title => 'Логин' },
            { id => 'ClientID',     title => 'ClientID' },
        ],
        code => \&not_convert_to_currency_user_list,
    },

     mass_drop_users_nds => {
        report_type => 'predefined_sub',
        title => 'Сбросить НДС пользователей',
        description => 'Интерфейс для массового сброса НДС пользователям.',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        expected_params => [
            {name => 'logins', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 5,
            },
        ],
        fields => [
            { id => 'login', title => 'Логин' },
            { id => 'result', title => 'Результат' },
        ],
        code => \&mass_drop_users_nds,
    },

    drop_user_goals => {
        report_type => 'predefined_sub',
        title => 'Удаление целей пользователя',
        description => 'Удалить можно только цели без достижений!',
        rbac_check => {Role => ['super']},
        expected_params => [
            {name => 'login', title => 'Логин пользователя'},
            {name => 'goal_deletion_mode',
                parent => 'select',
                title => 'Что надо сделать:',
                default_value => 'ask',
                options => [
                    { name => '', value => 'ask' },
                    { name => 'удалить все цели кроме тех что в списке goal_id', value => 'except_list' },
                    { name => 'удалить цели из списка goal_id', value => 'within_list' },
                ],
            },
            {name => 'goal_ids_list', title => 'список goal_id',
                parent => "textarea",
                cols => 32, rows => 5,
            },
        ],
        fields => [
            { id => 'cid', title => 'Id кампании(cid)' },
            { id => 'goal_id', title => 'Id цели(goal_id)' },
        ],
        code => \&drop_user_goals,
        submit_title => iget_noop('Удалить!'),
    },

    currency_convert_forecast => {
        report_type => 'predefined_sub',
        title => "Прогноз принудительной конвертации",
        description => "Позволяет оценить время завершения конвертации для всех клиентов с проставленным тизером, принявших оферту.",
        group_id => 'multicurrency',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&currency_convert_forecast,
    },

    upload_banner_resources => {
        report_type => 'predefined_sub',
        title => 'Загрузка нового видео/аудио в видеоплатформу',
        description => 'Принимает медийный файл для загрузки в видеоплатформу и дополнительные параметры. 
Результат загрузки можно посмотреть в отчете <a href="/registered/main.pl?cmd=internalReports&report_id=media_resources_report">"Просмотр доступных видео/аудио ресурсов"</a>',
        rbac_check => { Role => ['super'] },

        expected_params => [
            {
                name => 'input_file',
                title => 'Выбрать файл',
                parent => 'file',
            },
            {name => 'name', title => 'Название'},
            {name => 'type', title => 'Тип файла', parent => 'select',
                 default_value => 'video', options => [map {{name => $_, value => $_}} qw/video/]},
            {name => 'colors', title => 'Цвет подложки'},
            {name => 'categories', title => 'Категории'},
        ],
        fields => [
            { id => 'file',       title => 'Файл'},
            { id => 'type',       title => 'Тип'},
            { id => 'name',       title => 'Название'},
            { id => 'colors',     title => 'Цвет подложки'},
            { id => 'categories', title => 'Категории'},
            { id => 'result',     title => 'Результат'},
        ],
        code => \&upload_banner_resources,
        group_id => 'VideoAudioAds',
        method => 'POST',
        multipart => 1,
        submit_title => iget_noop('Загрузить!'),
    },

    media_resources_report => {
        report_type => 'predefined_sub',
        title => 'Просмотр доступных видео/аудио ресурсов',
        description => 'Выводит список успешно загруженных в видеоплатформу и сконвертировавшихся видео/аудио ресурсов.
Добавить новый ресурс можно в отчете <a href="/registered/main.pl?cmd=internalReports&report_id=upload_banner_resources">"Загрузка нового видео/аудио в видеоплатформу"</a>',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},

        fields => [
            { id => 'media_resource_id',   title => 'media_resource_id (CreativeID для БК)'},
            { id => 'resource_type',       title => 'Тип'},
            { id => 'name',                title => 'Название'},
            { id => 'yacontextCategories', title => 'Категории'},
            { id => 'colors',              title => 'Цвет подложки'},
            { id => 'preview_url',         title => 'URL превью'},
            { id => 'resources_url',       title => 'URL ресурсов'},
        ],
        code => \&media_resources_report,
        group_id => 'VideoAudioAds',
    },

    update_bs_ad_duration_report => {
        report_type => 'predefined_sub',
        title => 'Изменение длительности видеоподложек',
        description => 'В БК ролик будет обрезаться до этой длительности',
        rbac_check => {Role => ['super']},
        submit_title => 'Обновить',
        code => \&update_bs_ad_duration_report,
        expected_params => [
            { name => 'ad_duration_limit', parent => 'text', title => 'Длительность видеоподложек, сек' },
            { name => 'set_ad_duration_limit', parent => 'hidden', default_value => 1}
        ],
        fields => [
            { id => 'current_ad_duration_limit', title => 'текущее значение, сек'}
        ],
        group_id => 'VideoAudioAds'
    },

    feed_to_banner_land_settings => {
        report_type => 'predefined_sub',
        title => 'Настройки скрипта загрузки фидов в BannerLand',
        description => 'Позволяет суперам изменять параметры скрипта загрузки фидов ppcFeedToBannerland.pl',
        group_id => 'BLExport',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&feed_to_banner_land_settings,
        expected_params => [
            { name => 'select_chunk_size', title => 'Количество записей выбираемых из базы за один раз' },
            { name => 'bl_chunk_size', title => 'Количество одновременных запросов в BL' },
            { name => 'max_errors_count', title => 'Количество попыток загрузить фид в BL' },
            { name => 'recheck_interval_error_feeds', title => 'Через сколько дней переобходим фиды в статусе Error' },
            { name => 'bl_max_sleep_time_seconds', title => 'Максимальное время сна между итерациями в секундах' },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    delete_moderation_cmd_queue_entry => {
        report_type => 'predefined_sub',
        title => 'Удаление команд копирования из очереди модерации',
        description => 'Удаление команд копирования (по id + сid) из очереди команд в модерацию(moderation_cmd_queue)',
        group_id => 'Moderate',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&delete_moderation_cmd_queue_entry,
        expected_params => [
            { name => 'queue_id',  title => 'id команды в очереди' },
            { name => 'cid', title => 'id кампании (cid)' },
            { name => 'delete_moderation_cmd_queue_entry', parent => 'hidden', default_value => 1},
            { name => 'delete_from_all_shards', parent => 'checkbox', title => 'пробовать удалить из всех шардов'}
        ],
        submit_title => 'Удалить'
    },
   # Подтверждение заявок на настройку директа
    ya_agency_orders => {
        report_type => 'predefined_sub',
        title => 'Список клиентов',
        description => 'Просмотр и подтверждение заявок на настройку Яндекс Директа',
        group_id => 'yaAgency',
        rbac_check => {Role => [qw/super manager/], AllowDevelopers => 1},
        code => \&ya_agency_orders,
        fields => [
            { id => 'client_id', title => 'ClientID'},
            { id => 'ClientID',  title => 'Клиент'},
            { id => 'product_type', title => 'Тариф'},
            { id => 'created', title => 'Создана' },
        ],
        expected_params => [
            { name => 'ClientID',  title => 'id клиента' },
            { name => 'product_type', title => 'Тариф', parent => 'select',
              options => [{name => 'не менять', value => 0}, map {{name => $Direct::YaAgency::PRODUCTS->{$_}, value => $_}} @$Direct::YaAgency::ALLOWED_PRODUCT_TYPES],
            },
        ],

        submit_title => 'Подтвердить заявку',

        details_required => [
            'ClientID',
        ],
    },
    # Закрытие заявок на настройку директа 
    ya_agency_clients => {
        report_type => 'predefined_sub',
        title => 'Рассервисирование клиентов',
        description => 'Просмотр клиентов, оплативших заказ настройки Яндекс Директа. Закрытие заявок обслуженных клиентов.',
        group_id => 'yaAgency',
        rbac_check => {Role => [qw/super manager/], AllowDevelopers => 1},
        code => \&ya_agency_clients,
        fields => [
            { id => 'client_id', title => 'ClientID'},
            { id => 'ClientID',  title => 'Клиент'},
            { id => 'product_type', title => 'Тариф'},
            { id => 'created', title => 'Создана' },
            { id => 'changed', title => 'Оплачена' },
        ],
        expected_params => [
            { name => 'desiredStatus',  title => 'Отображать клиентов со статусом заявки', parent => 'select', 
                options => [{name => 'Ожидает рассервисирования', value => 'Paid'}, {name => 'Завершена', value => 'Completed'}, {name => 'Переотправлена', value => 'Resurrected'}] 
            },
            { name => 'ClientID',  title => 'id клиента, для которого нужно закрыть заявку' },
            { name => 'force_zero', title => "отправлять нулевую открутку", type => 'checkbox'},
        ],

        submit_title => 'Выполнить',

        details_required => [
            'ClientID',
        ],
    },

    bs_resync_relevance_match_campaigns => {
        report_type => 'predefined_sub',
        title => 'Переотправка в БК кампаний с relevance_match',
        description => 'Переотправка в БК кампаний заафекченных клиентов при включении/выключении фичи context_relevance_match_allowed',
        group_id => 'BSExport',
        rbac_check => {Role => [qw/super superreader manager/], AllowDevelopers => 1},
        code => \&bs_resync_relevance_match_campaigns,
        fields => [],
        expected_params => [
            {
                name    => 'resync_all_clients',
                title   => "Синхронизировать кампании всех клиентов, а не только указанных в списке",
                type    => 'checkbox',
            },
            {
                name    => 'client_ids',
                title   => "Идентификаторы клиентов, через запятую",
                parent  => 'textarea',
            },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    balance_resync_clients_of_limited_agency_reps => {
        report_type => 'predefined_sub',
        title => 'Переотправка в Баланс клиентов ограниченных представителей',
        description => 'Отдавать в новый метод биллинга клиентов ограниченных представителей',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&balance_resync_clients_of_limited_agency_reps,
        fields => [],
        expected_params => [
            {
                name    => 'resync_all_clients',
                title   => "Синхронизировать все агентства, а не только указанные в списке",
                type    => 'checkbox',
            },
            {
                name    => 'client_ids',
                title   => "ClientID агентств, через запятую",
                parent  => 'textarea',
            },
            {
                name    => 'force_reset',
                title   => "Отправить пустой список клиентов для представителей агентств с выключенной фичей",
                type    => 'checkbox',
            },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    update_aggregator_domains => {
        report_type => 'predefined_sub',
        title => 'Обновление доменов для кластеризации',
        description => 'Обновляет данные для расклейки доменов-агрегаторов',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&update_aggregator_domains,
        fields => [],
        expected_params => [
            {
                name    => 'resync_all_clients',
                title   => "Обновить для всех клиентов, а не только для указанных в списке",
                type    => 'checkbox',
            },
            {
                name    => 'client_ids',
                title   => "Идентификаторы клиентов, через запятую",
                parent  => 'textarea',
            },
            { 
                name    => 'domain', 
                title   => "Домен", 
                type    => 'select', 
                options => [{name => '--', value => ''}, map { {name => $_, value => $_} } @AggregatorDomains::ALLOWED_DOMAINS],
            },
            {   name => 'update_aggregator_domains', parent => 'hidden', default_value => 1 },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    wallet_daily_budget_stop_warning_settings => {
        report_type => 'predefined_sub',
        title => 'Настройка предупреждений об остановке общего счета по дневному бюджету',
        description => 'Позволяет суперам выставить время, до которого остановки по дневному бюджету считаются слишком ранними и показывается предупреждение.',
        rbac_check => {Role => ['super']},
        code => \&wallet_daily_budget_stop_warning_settings,
        expected_params => [
            { name => 'warning_time', title => 'время в формате HH:MM:SS, например 15:00:00' },
        ],
        fields => [
            { id => 'warning_time', title => 'Текущее значение' },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    belarus_bank_change_warning_trigger => {
        report_type => 'predefined_sub',
        title => 'Включение/выключение предупреждения об изменении банковских реквизитов для Белорусов',
        description => "Включение/выключение предупреждения об изменении банковских реквизитов для Белорусов",
        rbac_check => {Role => ['super']},
        code => \&belarus_bank_change_warning_trigger,
        expected_params => [
            {name => 'show_belarus_bank_change_warning', parent => 'checkbox', title => 'включить предупреждение'},
            {name => 'change_option', parent => 'hidden', default_value => 1},
        ],
        submit_title => 'Поменять настройку!',
    },

    clients_features => {
        report_type => 'predefined_sub',
        title => 'Спец-фичи для клиентов', 
        description => "Особые тизеры, функциональность по приглашениям и т.п.",

        code => \&clients_features,

        fields  => [
                {id => 'num', title => '№ п/п'},
                {id => 'raw_ClientID', title => 'ClientID'},
                {id => 'ClientID', title => 'Клиент'},
            ],

        expected_params => [
            { title => "Фича", name => "feature_id", type => "select", 
              options => [
                  {name => '--', value => ''},
                  {name => 'Безакцептный перевод в BYN', value => $Client::SHOW_BELARUS_OLD_RUB_WARNING_PROPERTY_NAME},
                  {name => 'Скрыть тизер про изменение реквизитов счетов Белорусов', 
                   value => $Client::HIDE_BELARUS_BANK_CHANGE_WARNING_CLIENT_IDS_PROPERTY_NAME},
                  {name => 'Признак фрилансера', value => $Client::IS_FREELANCER_PROPERTY_NAME},
              ],
              },
            { title => "Добавить ClientID", name => "add_ClientIDs", parent => 'textarea', },
            { title => "Удалить ClientID", name => "remove_ClientIDs", parent => 'textarea', },
        ],
        details_required => [ qw( ClientID ) ],
        submit_title => 'Go!',
    },

    clients_agencies_managers_report => {
        report_type => 'predefined_sub',
        title => 'Выгрузка агентств и менеджеров по списку клиентов',
        description => "Выгрузка агентств и менеджеров по списку клиентов",
        rbac_check => {Role => [qw/super/]},
        code => \&clients_agencies_managers_report,

        fields  => [
                { id => 'ClientID', title => 'ClientID' },
                { id => 'uid', title => 'uid' },
                { id => 'login', title => 'login', text_sort => 1 },
                { id => 'role', title => 'role', text_sort => 1 },
                { id => 'agency_login', title => 'agency_login', text_sort => 1 },
                { id => 'manager_login', title => 'manager_login', text_sort => 1 },
            ],

        expected_params => [
            {
                name => 'items',
                title => 'Список логинов/ClientID/uid',
                type => 'textarea',
                cols => 40,
                rows => 10,
            },
            {
                title => "Что указано в списке", name => "item_type", type => "select", 
                options => [
                    {name => 'Логины', value => 'login'},
                    {name => 'ClientID', value => 'ClientID'},
                    {name => 'uid', value => 'uid'},
                ],
            },
            {
                title => "Выгрузить в xls",
                name => 'to_xls',
                type => 'checkbox',
            },
        ],
        method => 'POST',
    },

    upload_banner_experiments => {
        report_type => 'predefined_sub',
        title => 'Загрузка данных для экспериментов на СЕРПе и РСЯ',
        description => 'Принимает файл в формате xls или xlsx для отправки в БК данных для экспериментов на СЕРПе и РСЯ.
Если название поля начинается с href_, то при отправке в БК будет произведено раскрытие макросов аналогично ссылкам в баннерах.
Формат файла:
<table>
  <tr><td>cid</td><td>bid</td><td>param1</td><td>param2</td><td>...</td></tr>
  <tr><td>123</td><td>456</td><td>aaa</td><td>bbb</td><td>...</td></tr>
  <tr><td>...</td><td>...</td><td>...</td><td>...</td><td>...</td></tr>
</table>
Прочитать подробнее о формате файла с данными можно в <a href="https://wiki.yandex-team.ru/direct/TechnicalDesign/Zalivka-dannyx-dlja-jeksperimentov-na-SERPe-i-RSJa-cherez-Direkt/#praviladljafajjlasdannymidljaotchjota" target="_blank">Вики</a>.',
        rbac_check => {Code => [ sub {
            my ($rbac, $vars) = @_;
            Client::ClientFeatures::can_upload_banner_experiments(client_id => $vars->{rights}->{ClientID}, rights => $vars->{rights}) ? 0 : 1
        }]},
        expected_params => [
            {
                name => 'input_file',
                title => 'Выбрать файл',
                parent => 'file',
            },
            {
                name => "ignore_errors",
                title => "Игнорировать ошибки парсинга (сохранять все валидные эксперименты)",
                type => "checkbox",
            },
            {
                name => "delete_experiment",
                title => "Удалить эксперимент (требуется файл с идентификаторами кампаний и баннеров)",
                type => "checkbox",
            },
            {
                name => "delete_all_experiments",
                title => "Удалить ВСЕ эксперименты (файл не требуется)",
                type => "checkbox",
            },
        ],
        fields => [
            { id => 'bid',      title => 'Баннер'},
            { id => 'json',     title => 'Эксперимент'},
        ],
        code => \&upload_banner_experiments,
        group_id => 'experiment',
        method => 'POST',
        multipart => 1,
        submit_title => iget_noop('Применить!'),
    },

    # работа с потерянными счетчиками турболендингов
    lost_turbolanding_metrica_counters => {
        report_type => 'predefined_sub',
        code => \&lost_turbolanding_metrica_counters,
        title => iget_noop('Актуализация данных в camp_turbolanding_metrica_counters'),
        submit_title => iget_noop('Выполнить актуализацию'),
        description => "Поиск и исправление кампаний с потерянными счетчиками", 

        fields => [
            { id => 'cid', title => 'Кампания', },
            { id => 'b_count', title => 'Баннеров с потерянными счетчиками' },
        ],
        expected_params => [ 
            [
                {name => 'cids', parent => 'textarea', title => 'Пересчитать счетчики кампаний', },
                {name => 'repair_all', parent => 'checkbox', title => 'Пересчитать все сломанные кампании'}
                
            ],
        ],

        rbac_check => { Role => ["support", "super", "superreader", "manager"] },

    },

    preprod_limtest => {
        report_type => 'predefined_sub',
        title => 'Переключение limtest с/на preprod БК',
        description => 'Позволяет изменять выбор урла для воркеров dev и devprice (preprod или prod)',
        group_id => 'BSExport',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&preprod_limtest,
        expected_params => [
            {
                name => 'use_preprod_limtest_1',
                parent => "checkbox" ,
                title => 'Отправлять в препрод БК на limtest1',
            },
            {
                name => 'use_preprod_limtest_2',
                parent => "checkbox",
                title => 'Отправлять в препрод БК на limtest2'
            },
            {
                name => 'change_option',
                parent => 'hidden',
                default_value => 1
            },
        ],
        submit_title => 'Применить',
        method => 'POST',
    },

    resync_moderate_settings => {
        report_type => 'predefined_sub',
        title => 'Настройки ленивой перепосылки в Модерацию',
        description => 'Позволяет суперам и разработчикам изменять параметры ленивой перепосылки в Модерацию.
Значения по умолчанию:
CRIT_LIMIT        => 250;
NOCRIT_LIMIT      => 500;
MAX_TOTAL_SIZE    => 10000;
MAX_AGE_MINUTES   => 15;
',
        group_id => 'Moderate',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&resync_moderate_settings,
        expected_params => [
            { name => 'crit_limit', title => 'CRIT_LIMIT' },
            { name => 'nocrit_limit', title => 'NOCRIT_LIMIT' },
            { name => 'max_total_size', title => 'MAX_TOTAL_SIZE' },
            { name => 'max_age_minutes', title => 'MAX_AGE_MINUTES' },
        ],
        submit_title => iget_noop('Сохранить!'),
    },

    isize_props => {
        report_type => 'predefined_sub',
        title => 'Изменение коэффициентов для вычисления интегрального размера очереди',
        description => "Позволяет изменять коэффициенты для вычисления интегрального размера очереди.
        <b>isize = camps_num * \$camps_num_coef ($BS::ExportMaster::DEFAULT_ISIZE_CAMPS_NUM_COEF по ум.)
              + banners_num * \$banners_num_coef ($BS::ExportMaster::DEFAULT_ISIZE_BANNERS_NUM_COEF по ум.)
              + contexts_num * \$contexts_num_coef ($BS::ExportMaster::DEFAULT_ISIZE_CONTEXTS_NUM_COEF по ум.)
              + bids_num * \$bids_num_coef ($BS::ExportMaster::DEFAULT_ISIZE_BIDS_NUM_COEF по ум.)
              + prices_num * \$prices_num_coef ($BS::ExportMaster::DEFAULT_ISIZE_PRICES_NUM_COEF по ум.)</b>",
        group_id => 'BSExport',
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&isize_props,
        expected_params => [
            { name => $BS::ExportMaster::ISIZE_CAMPS_NUM_COEF_PROP_NAME, title => "camps_num_coef" },
            { name => $BS::ExportMaster::ISIZE_BANNERS_NUM_COEF_PROP_NAME, title => "banners_num_coef" },
            { name => $BS::ExportMaster::ISIZE_CONTEXTS_NUM_COEF_PROP_NAME, title => "contexts_num_coef" },
            { name => $BS::ExportMaster::ISIZE_BIDS_NUM_COEF_PROP_NAME, title => "bids_num_coef" },
            { name => $BS::ExportMaster::ISIZE_PRICES_NUM_COEF_PROP_NAME, title => "prices_num_coef" },
            {
                name => 'change_option',
                parent => 'hidden',
                default_value => 1
            },
        ],
        submit_title => iget_noop('Сохранить!')
    },

    create_billing_aggregates => {
        report_type => 'predefined_sub',
        title => 'Создание Биллинговых Агрегатов по списку клиентов',
        description => 'Принимается список логинов или ClientID, по одному на строку. Можно указать не больше 100 клиентов.',
        rbac_check => {Role => ['super']},
        code => \&create_billing_aggregates,
        expected_params => [
            {name => 'clients', parent => 'textarea', title => 'Список логинов/ClientID', cols => 40, rows => 10 },
            {name => 'product_type', type => "select", title => 'Тип продукта',
                options => [
                    (map { {name => $_, value => $_} } _get_all_billing_aggregate_products()),
                ],
            }
        ],
        fields => [
            { id => 'ClientID' },
            { id => 'login' },
            { id => 'result' },
        ],
        submit_title => iget_noop('Создать!'),
    },

    create_billing_aggregates_script_mgr => {
        report_type => 'predefined_sub',
        title => 'Управление скриптом для создания Биллинговых Агрегатов',
        description => 'Можно указать список ClientID/логинов, или диапазон ClientID.'
            .'<p>Скрипт будет проверять каждого клиента - не нужно ли ему создать недостающий биллинговый агрегат.'
            .'<p>Нужно, если:'
            .'<br>* Клиент попал под фичу "автосоздание биллинговых агрегатов" (auto_create_billing_aggregates)'
            .'<br>* У клиента есть ОС'
            .'<br>* У клиента есть кампании, для которых нужны биллинговые агрегаты, которых у клиента еще нет'
            .'<br>* Для недостающих биллинговых агрегатов включено автосоздание ($Settings::BILLING_AGGREGATES_AUTO_CREATE_TYPES)'
            ,
        rbac_check => {Role => ['super'], AllowDevelopers => 1},
        code => \&create_billing_aggregates_script_mgr,
        expected_params => [
            {name => 'clients', parent => 'textarea', title => 'Список логинов/ClientID', cols => 40, rows => 10 },
            [
                {name => 'client_id_from', title => 'ClientID от'}, {name => 'client_id_to', title => 'ClientID до, не включительно'},
                {name => 'has_camp_type', type => "select", title => 'У которых есть кампании типа',
                    options => [
                        (map { {name => $_, value => $_} } '---', sort keys %Campaign::Types::TYPES),
                    ],
                },
            ]
        ],
        fields => [
            { id => 'shard' },
            { id => 'task' },
        ],
        submit_title => iget_noop('Создать!'),
    },

    archive_campaigns_by_type => {
        report_type => 'predefined_sub',
        code => \&archive_campaigns_by_type,
        rbac_check => {Role => [qw/super/], AllowDevelopers => 1},
        title => 'Архивация кампаний по типам',
        fields => [
            { id => 'uid', text_sort => 1 },
            { id => 'cid', text_sort => 1 },
            { id => 'result', title => 'Результат' },
            { id => 'error', title => 'Ошибка' },
        ],
        details_required => [ 'uid', 'cid' ],
        expected_params => [
            {title => 'Тип кампании', name => 'campaign_type', type => 'select', options => [
                {name => 'Геопродуктовая', value => 'cpm_geoproduct'},
                {name => 'Сделки', value => 'cpm_deals'},
                {name => 'Продвижение контента: видео', value => 'content_promotion_video'},
                {name => 'Продвижение контента: коллекции', value => 'content_promotion_collection'},
                {name => 'Медийная: только аудио группы', value => 'cpm_audio'},
                {name => 'Медийная: только индор группы', value => 'cpm_indoor'},
                {name => 'Медийная: только аутдор группы', value => 'cpm_outdoor'},
            ]},
            {title => 'Список ClientId/cid', type => 'textarea', name => 'ids', cols => 40, rows => 10},
            {title => 'Что указано в списке', name => 'item_type', type => 'select', options => [
                {name => 'ClientID', value => 'ClientID'},
                {name => 'cid', value => 'cid'},
            ]},
            {title => 'Архивировать всё, кроме указанного', name => 'all_except', type => 'checkbox'},
            {title => 'Архивировать неостановленные кампании', name => 'archive_non_stopped', type => 'checkbox'},
        ],
        submit_title => iget_noop('Архивировать'),
    },

    # Установка кампании флага has_turbo_smarts
    modify_has_turbo_smarts => {
        report_type => 'predefined_sub',
        code => \&modify_has_turbo_smarts,
        rbac_check => { Role => ["super", "superreader"] },
        title => 'Установка кампании флага has_turbo_smarts',
        description => "Флаг можно устанавливать смарт-кампаниям клиентов, имеющих фичу has_turbo_smarts",
        field_titles => {cid => 'cid', 'title' => 'Название кампании', 'has_turbo_smarts' => 'has_turbo_smarts'},
        fields  => [
                { id => 'cid', text_sort => 1},
                { id => 'title',  title => 'название кампании'},
                { id => 'has_turbo_smarts'},
            ],
        expected_params => [
            {title => 'Кампания', type => 'cids', name => 'cids', field_size => 30, value_max_len => 1024},
            {title => 'Значение флага (0|1)', type => 'text', name => 'has_turbo_smarts', field_size => 5, value_max_len => 5}
        ],
        group_id => 'DEBUG',
        submit_title => iget_noop('Установить/показать'),
    },

    # Управление режимом обогащения данными шаблонов, обрабатываемых ShowDnaPb
    enrich_data_for_show_dna_pb => {
        report_type => 'predefined_sub',
        code => \&enrich_data_for_show_dna_pb,
        rbac_check => { Role => ["super", "superreader"] },
        title => 'Управление режимом обогащения данными шаблонов в ShowDnaPb',
        expected_params => [
            {
                name    => "mode",
                title   => "Режим обогащения",
                options => [
			{name => 'обогащение выключено', value => 'none'},
			{name => 'включено, при наличии параметра _enrich_data в url', value => 'by_flag'},
			{name => 'включено', value => 'enabled'}
			
		],
                parent  => 'select',
            }
        ],
        group_id => 'DEBUG',
	submit_title => 'Сохранить',
    },
    # Включение/выключение возможности добавлять автовидео смарт-тго кампаниям
    change_is_autovideo_allowed => {
        report_type => 'predefined_sub',
        title => 'Кампании со включенным флагом is_auto_video_allowed',
        description => 'Установка/снятие флага is_auto_video_allowed для кампаний',
        group_id => 'DEBUG',
        rbac_check => {Role => [qw/super superreader/], AllowDevelopers => 1},
        code => \&change_is_autovideo_allowed,
        fields => [
            { id => 'ClientID', title => 'ClientID'},
            { id => 'cid',  title => 'cid'},
            { id => 'type', title => 'Тип кампании'},
            { id => 'name', title => 'Имя кампании' },
        ],
        expected_params => [
            { name => 'cid',  title => 'Идентификаторы кампаний (cid)', field_size => 30, value_max_len => 1024 },
            { name => 'action_type', title => 'Тип действия', parent => 'select',
              options => [{name => 'не менять', value => 0}, {name => 'установить флаг', value => 1}, {name => 'снять флаг', value => -1} ],
            },
        ],

        submit_title => 'Выполнить',
    },
    # Включение/выключение блокировки пользователя по карме
    change_passport_karma_lock_state => {
        report_type => 'predefined_sub',
        title => 'Блокировка пользователя по карме паспорта',
        description => 'Установка/снятие блокировки по карме для заданного uid',
        group_id => 'SUPPORT',
        rbac_check => {Role => [qw/super limited_support/], AllowDevelopers => 1},
        code => \&change_passport_karma_lock_state,
        fields => [
            { id => 'login', title => 'Логин'},
            { id => 'uid',  title => 'uid'},
            { id => 'karma_lock', title => 'Блокировка по карме'},
        ],
        expected_params => [
            { name => 'uid',  title => 'uid клиента', field_size => 12, value_max_len => 30 },
            { name => 'action_type', title => 'Блокировка', parent => 'select',
              options => [{name => 'не менять', value => 0}, {name => 'заблокировать', value => 1}, {name => 'разблокировать', value => -1} ],
            },
        ],

        submit_title => 'Выполнить',
    }
    
);

sub _get_all_billing_aggregate_products {
    my $all_camp_types = xminus(
        Campaign::Types::get_camp_kind_types('all'),
        Campaign::Types::get_camp_kind_types('without_billing_aggregates')
    );

    my @prod_types;
    for my $camp_type (@$all_camp_types) {
        push @prod_types,
            Campaign::Types::default_product_type_by_camp_type($camp_type),
            @{Direct::BillingAggregates::get_special_product_types_by_camp_type($camp_type)};
    }

    return uniq @prod_types;
}

=head2 make_array_of_arrays

    Преобразует массив хешей/массивов_хеший в массив массивов хешей,
    оборачивая отдельные хеши в массивы из одного елемента
=cut

sub make_array_of_arrays($){
    my $src = shift;

    ref $src eq 'ARRAY' or die "USAGE: make_array_of_arrays([...])";

    my $dst=[];
    for my $row (@{$src}) {
        push @$dst, (ref $row eq 'ARRAY') ? $row : [$row];
    }
    return $dst;
}

=head2 get_rbac_check_for_report

    Отдает хеш с проверками для доступа к отчету $report_id.

    Если в отчете не задан rbac_check -- возвращает умолчальную проверку.

=cut 

sub get_rbac_check_for_report
{
    my ($report_id) = @_;

    return $PREDEFINED_REPORT{$report_id}->{rbac_check} || { Role => ['super', 'superreader'] };
}

=head2 get_data_for_internal_report 

    Знает, какой report_type можно получить из какой функции

    Результат -- структура "отчет": 
            {
            data => [ ... ],           # Значения -- массив хешей
            fields => [],  # Описания полей -- массив хешей
        }

    Описание поля: {
        id    => ..., # строка -- ключ в хеше со значениями
        title => ..., # заголовок для столбца в таблице
        c     => ..., # DirectContext 
        r     => ..., # Apache::Request
    }


=cut

sub get_data_for_internal_report
{
    my %OPT = @_;

    my $report = {};

    my $report_descr = $PREDEFINED_REPORT{$OPT{report_id}};
    my $report_type = $report_descr->{report_type};

    $report_descr->{expected_params} = make_array_of_arrays($report_descr->{expected_params}) if exists $report_descr->{expected_params}; 

    # Здесь должно быть последовательное разбирательство с разными типами отчетов: 
    if ($report_type eq 'monitor_values'){
        # из monitor_values
        $report = get_report_from_monitor_values(%OPT);
    } else {
        make_param_hashes($report_descr);
        calc_default_values($report_descr);
        if($report_type eq 'predefined_sql') {
            # просто sql-запрос
            $report = get_report_from_predefined_sql( report_descr => $report_descr , %OPT);
        } elsif($report_type eq 'predefined_sub') {
            # специальная функция 
            $report = get_report_from_predefined_sub( report_descr => $report_descr , %OPT);
        } else {
            die "get_data_for_internal_report: unknown report_type '$report_type'";
        }
        postprocess_params(report => $report, report_descr => $report_descr, %OPT);
    }

    # если требуется -- добавляем подробные данные для красивого отображения полей вроде кампании или пользователя
    add_detailed_info_to_report(report => $report, report_descr => $report_descr );

    # если требуется -- преобразуем таблицу(pivot), вынося некоторые строки в столбцы
    pivot_report_data(report => $report, report_descr => $report_descr, %OPT);

    # добавляем некоторые данные для рисования графиков
    if ($OPT{show_chart}) {
        add_chart_data(report => $report, report_descr => $report_descr, %OPT);
    }

    # TODO 
    # запрос от msa: делать еще и json-представление отчета -- тогда можно воспользоваться каким-нибудь легким движком для графиков

    $report->{report_id} = $OPT{report_id};
    $report->{title} = $report_descr->{title} || '';
    $report->{index_list} = $report_descr->{index_list} || '';

    # если надо -- переводим что получилось в эксель-таблицу
    if ($OPT{xls} || $report->{to_xls}) {
        $report->{ready_xls} = to_xls($report);
    }

    if ($OPT{csv}){
        $report->{ready_csv} = to_csv($report);
    }
    
    return $report;
}


=head2 add_detailed_info_to_report(report => $report, report_descr => $report_descr );

    если требуется -- добавляет к каждой строке отчета подробную информацию по определенным объектам (кампания, пользователь)

=cut

my %PLACE_FOR_DETAILS = (
    cid       => 'camp_detailed',
    OrderID   => 'camp_detailed',
    uid       => 'user_detailed',
    login     => 'user_detailed',
    agency_login     => 'agency_detailed',
    ClientID  => 'user_detailed',
    bid       => 'banner_detailed',
    BannerID  => 'banner_detailed',
    bids_id   => 'phrase_detailed',
    PhraseID  => 'phrase_detailed',
    geo       => 'geo_detailed',
    firm      => 'firm_detailed',
    country   => 'country_detailed',
);

=head2 add_detailed_info_to_report

=cut
sub add_detailed_info_to_report
{
    my %OPT = @_;

    my $RD = $OPT{report_descr};
    my $report = $OPT{report};
    # если в отчете не описаны никакие поля со сложными данными -- ничего не делаем, выходим
    return unless $RD->{details_required};

    my %D = map { $_ => 1 } @{$RD->{details_required}};
    for my $f (@{$report->{fields}}){
        next unless $D{$f->{id}};
        $f->{details_key} = $PLACE_FOR_DETAILS{$f->{id}} or die "add_detailed_info_to_report: don't know how to get detailed info for field '$f->{id}'";
    }

    # перебираем строчку за строчкой, к каждой добавляем что требуется
    # если вдруг захочется над кампаниями делать camps_add_rbac_actions -- то было бы лучше обрабатывать не по строчке, а все разом... 
#    for my $str (@{$report->{data}}){
    my $new_data = [];
    while(my @chunk = splice @{$report->{data}}, 0, 1000) {
    
        for my $f (@{$report->{fields}}){
            # если определена функция, меняющая данные - вызываем её
            if ($f->{data_cb}) {
                for my $str (@chunk){
                    $str->{$f->{id}} = $f->{data_cb}->($str, $f->{id});
                }
            }
            next unless exists $f->{details_key};
            my $type = $f->{details_key};

            if ( $type eq 'camp_detailed' ){
                # подробные сведения о кампании
                my @ids = grep {$_ && /^\d+$/} map {$_->{$f->{id}}} @chunk;
                if (@ids) {
                    my $where = { 'c.' . $f->{id} => \@ids };
                    # multicurrency: не передаём в get_user_camps_by_sql ключи client_nds/client_discount, т.к. суммы на кампании не используются
                    my $shard_cond = $f->{id} =~ /^(uid|cid|OrderID)$/ ? {$f->{id} => \@ids} : {shard => 'all'};
                    my $chunk_data = Common::get_user_camps_by_sql($where, {shard => $shard_cond})->{campaigns};
                    my $chunk_data_hash = {map {$_->{$f->{id}} => $_} @$chunk_data};
                
                    for my $str (@chunk){
                        $str->{$type} = $chunk_data_hash->{$str->{$f->{id}}};
                    }
                }
            } elsif ( $type eq 'agency_detailed' ){
                # подоробные сведения о пользователе
                my $key_field = $f->{id};
                $key_field =~ s/^agency_//;

                my $key_list = [map {$_->{$f->{id}}} @chunk];
                my $chunk_data = get_all_sql( PPC($key_field => $key_list), ["select u.FIO fio, u.uid, u.login from users u ", 
                                                                 where => {$key_field => SHARD_IDS}] )||[];
                
                my $chunk_data_hash = {map {$_->{$key_field} => $_} @$chunk_data};
                
                for my $str (@chunk){
                    $str->{$type} = $chunk_data_hash->{$str->{$f->{id}}};
                }
            } elsif ( $type eq 'user_detailed' ){
                # подоробные сведения о пользователе
                my $key_list = [map {$_->{$f->{id}}} @chunk];
                my $chunk_data = get_all_sql( PPC($f->{id} => $key_list), ["select u.FIO fio, u.uid, u.login, u.ClientID from users u ", 
                                                                 where => {$f->{id} => SHARD_IDS}] )||[];

                my $chunk_data_hash = {map {$_->{$f->{id}} => $_} @$chunk_data};
                
                for my $str (@chunk){
                    $str->{$type} = $chunk_data_hash->{$str->{$f->{id}}};
                }
            } elsif ( $type eq 'banner_detailed' ) {
                my $key_list = [map {$_->{$f->{id}}} @chunk];
                # TODO adgroup: разобраться что может прийти в $f->{id}
                # желательно заменить на вызов Models::AdGroup::get_pure_creatives
                my $chunk_data = get_all_sql( PPC($f->{id} => $key_list), [q/
                    SELECT b.cid, b.bid, b.title, b.BannerID, u.login AS ulogin
                    FROM banners b
                    INNER JOIN campaigns c ON b.cid = c.cid
                    INNER JOIN users u ON c.uid = u.uid
                /, where => {$f->{id} => SHARD_IDS}] )||[];

                my $chunk_data_hash = {map {$_->{$f->{id}} => $_} @$chunk_data};

                for my $str (@chunk){
                    $str->{$type} = $chunk_data_hash->{$str->{$f->{id}}} if defined $str->{$f->{id}};
                }
            } elsif ( $type eq 'phrase_detailed' ) {            #Кроме ID должен быть обязательно ClientID
                my $key_list = [map {$_->{$f->{id}}} @chunk];
                my $key_field_name = $f->{id} eq 'bids_id'? 'bi.id' : $f->{id};
                my $chunk_data = get_all_sql( PPC(ClientID => [map {$_->{ClientID}} @chunk]), [q/
                    SELECT bi.id AS bids_id, b.bid, b.cid, bi.phrase, bi.PhraseID
                    FROM bids bi
                    INNER JOIN banners b ON bi.pid = b.pid
                /, where => {$key_field_name => $key_list}] )||[];

                my $chunk_data_hash = {map {$_->{$f->{id}} => $_} @$chunk_data};

                for my $str (@chunk){
                    $str->{$type} = $chunk_data_hash->{$str->{$f->{id}}} if defined $str->{$f->{id}};
                }
            } elsif ( $type eq 'geo_detailed' ) {
                for my $str (@chunk){
                    $str->{$type} = {geo_names => get_geo_names($str->{$f->{id}})} if defined $str->{$f->{id}};
                }
            } elsif ( $type eq 'firm_detailed' ) {
                for my $str (@chunk){
                    $str->{$type} = {firm_name => $Currencies::FIRM_NAME{ $str->{ $f->{id} } } } if defined $str->{$f->{id}};
                }
            } elsif ( $type eq 'country_detailed' ) {
                my $lang = Yandex::I18n::current_lang();
                for my $str (@chunk){
                    my $region_id = $str->{ $f->{id} };
                    if (defined $region_id) {
                        my $region = GeoTools::get_country_names_by_id($region_id);
                        $str->{$type} = {country_name => $region->{"name_$lang"}, region_id => $region_id};
                    }
                }
            }
            # что-нибудь еще?
        }

        push @$new_data, @chunk;
    }
    $report->{data} = $new_data;

    return;
}

=head2 pivot_report_data {

# если требуется -- преобразуем таблицу(pivot), вынося некоторые строки в столбцы

=cut
sub pivot_report_data {
    my %OPT = @_;

    # поля, которые вынесутся в столбцы
    my %fields_to_pivot = map {$_ => 1} split /\s*,\s*/, ($OPT{pivot_fields}||'');
    return if !keys %fields_to_pivot || !$OPT{pivot_measure} || !@{$OPT{report}{data}};

    my %measure_fields = map {$_ => 1} @{$OPT{report_descr}->{pivot}->{measures}};
    
    # поля, определяющие строку
    my @row_keys = grep {!exists $fields_to_pivot{$_} && !exists $measure_fields{$_}} map {$_->{id}} @{$OPT{report}->{fields}};
    # поля, определяющие столбец
    my @column_keys = grep {exists $fields_to_pivot{$_}} map {$_->{id}} @{$OPT{report}->{fields}};

    # хэш{строка}{столбец} = значение
    my %data;
    # соотвествие сериализованных полей строки и хэша с этими полями
    my %row_info;
    # всевозможные ключи столбцов
    my %column_keys;
    for my $row (@{$OPT{report}{data}}) {
        my $row_key = join '/', map {$row->{$_} // ''} @row_keys;
        my $column_key = join '/', map {$row->{$_}} @column_keys;
        $column_keys{$column_key} = 1;
        $row_info{$row_key} ||= hash_cut $row, @row_keys;
        push @{$data{$row_key}{$column_key}}, $row->{$OPT{pivot_measure}};
    }

    # пересобираем из хэша %data массив
    my @data;
    for my $row_key (sort keys %data) {
        my %row = %{$row_info{$row_key}};
        for my $column_key (keys %{$data{$row_key}}) {
            $row{$column_key} = join ', ', @{$data{$row_key}{$column_key}};
        }
        push @data, \%row;
    }
    
    # подменяем данные в отчёте
    $OPT{report}->{data} = \@data;
    $OPT{report}->{fields} = [
        (map {{id => $_, title => $_, text_sort=>1}} @row_keys),
        (map {{id => $_, title => $_}} sort keys %column_keys),
        ];
}

=head2 add_chart_data

    добавляем данные для построения графиков
    на входе именованные параметры:
    - report
    - chart_series_limit
    в результате в report добавляется ссылка на хэш chart_data с ключами
    - categories - массив из меток оси x
    - series - рисуемые линии, массив из хешей, в каждом хэше есть
        - name - название линии
        - data - массив значений

=cut
sub add_chart_data {
    my %OPT = @_;
    # Получаем список строковых и числовых полей
    my @str_fields_ids = map {$_->{id}} grep {$_->{text_sort}} @{$OPT{report}{fields}};
    my @num_fields = grep {!$_->{text_sort}} @{$OPT{report}{fields}};
    my @num_fields_ids = map {$_->{id}} @num_fields;
    
    # пустой chart
    my $chart = $OPT{report}{chart_data} = {
        categories => [],
        series => [],
    };
    
    # перегруппировываем данные по сериям, попутно считаем для каждой серии максимальное значение
    my %SERIES;
    my %SERIES_MAXVAL;
    for my $row (@{$OPT{report}{data}}) {
        push @{$chart->{categories}}, join('/', map {$row->{$_} // ''} @str_fields_ids);
        for my $id (@num_fields_ids) {
            my $val = is_valid_float($row->{$id}) ? $row->{$id} : undef;
            push @{$SERIES{$id}}, $val;
            if (! defined $SERIES_MAXVAL{$id} || defined $val && $SERIES_MAXVAL{$id} < $val) {
                $SERIES_MAXVAL{$id} = $val;
            }
        }
    }

    # отбираем нужное количество максимальный серий, сохраняя порядок
    my $limit = $OPT{chart_series_limit} && is_valid_int($OPT{chart_series_limit}) ? $OPT{chart_series_limit} : 10;
    if (@num_fields > $limit) {
        my %choosed = map {$_->{id} => 1} (xsort {-($SERIES_MAXVAL{$_->{id}}||0)} @num_fields)[0..$limit-1];
        @num_fields = grep {$choosed{$_->{id}}} @num_fields;
    }

    # делаем из серий массив, дописываем name
    for my $num_field (@num_fields) {
        push @{$chart->{series}}, {
            name => $num_field->{title},
            data => $SERIES{$num_field->{id}},
        };
    }
}

=head2 get_report_from_monitor_values(target_id => '34,56,73' )

    Для страницы "внутренние отчеты", 
    извлекает данные из таблицы monitor_values_day 

    Параметры, именованные: 
        target_id -- число или несколько чисел через запятую (ссылка на массив чисел???) -- target_id, по которым надо выбрать замеры

      TODO
        date_start
        date_end

    Результат:  
        хеш "отчет"    

    Примеры 
        get_report_from_monitor_values(target_id => '69, 75');

=cut

sub get_report_from_monitor_values
{
    my %OPT = @_;

    my $target_ids = [split /[^\d]+/, $OPT{target_id}];

    my $res;

    # если не указаны даты -- берем последние 45 дней
    $res->{date_from} = $OPT{date_from} || unix2human(time - 3600 * 24 * 45, "%Y-%m-%d");
    $res->{date_to}   = $OPT{date_to}   || unix2human(time, "%Y-%m-%d");

    # Собираем данные:
    # извлекаем все, что нужно, из таблицы,
    my $agg = $OPT{agg} =~ /^(sec|hour|day|5min)$/ ? $OPT{agg} : 'day';
    if ($agg eq '5min') {
        $agg = 'sec';
    }
    my $time_sql = $agg eq 'day' ? 'date(measure_time)' : 'measure_time';
    my $all_data = get_all_sql(MONITOR, ["select $time_sql as measure_time, target_id, value from monitor_values_$agg", 
                                         where => {target_id => $target_ids}, 
                                             and => "measure_time >= ?", and => "measure_time < ? + interval 1 day"], $res->{date_from}, $res->{date_to});

    # перегруппируем как надо,
    my %data_by_date;
    for my $h (@$all_data) {
        if ($OPT{agg} && $OPT{agg} eq '5min') {
            my $d = datetime($h->{measure_time});
            $d->truncate(to => 'minute');
            $d->subtract(minutes => ($d->minute % 5));
            my $measure_time = $d->strftime('%F %T');

            $data_by_date{$measure_time}->{$h->{target_id}} += $h->{value};
        } else {
            $data_by_date{$h->{measure_time}}->{$h->{target_id}} = $h->{value};
        }
    }
    $data_by_date{$_}->{measure_time} = $_ for keys %data_by_date;
    $res->{data} = [sort {$a->{measure_time} cmp $b->{measure_time}} values %data_by_date];

    # Составляем описания полей:
    # первое поле -- время замера
    $res->{fields} = [
        { 
            title => 'время замера',
            id => "measure_time",
            text_sort => 1,
        }
    ];
    # извлекаем описания из БД
    my $all_targets = get_all_sql(MONITOR, ["select target_id as id, name, description, units from monitor_targets", 
                                            where => {target_id => $target_ids}, "order by target_id"]);
    # и определяем заголовок для столбца
    $_->{title} = $_->{description} || $_->{name} for @$all_targets;
    push @{$res->{fields}}, @$all_targets;

    # округляем данные: штуки -- до целых, проценты -- до двух знаков
    for my $d (@{$res->{data}}){
        for my $target (@$all_targets){
            if ($target->{units} eq 'num'){
                $d->{$target->{id}} = sprintf( "%d", $d->{$target->{id}});
            } elsif ($target->{units} eq 'proc') {
                $d->{$target->{id}} = sprintf( "%0.2f", $d->{$target->{id}});
            }
        }
    }

    return $res;
}


=head2 get_report_from_predefined_sql

    Для страницы "внутренние отчеты", 
    выполняет sql-запрос из числа заранее написанных 

    Параметры, именованные: 
        report_descr


    Результат:  
        хеш "отчет"    

=cut

sub get_report_from_predefined_sql
{
    my %OPT = @_;

    # RD stands for Report Description
    my $RD = $OPT{report_descr};
    
    # выполняем запрос
    my $params = $RD->{make_bind_values_arr} ? $RD->{make_bind_values_arr}->(\%OPT) : [];
    my $db = ref $RD->{DB} eq 'CODE' ? $RD->{DB}->(\%OPT) : $RD->{DB};
    my $sth = exec_sql($db, $RD->{sql}, @$params);
    
    # составляем описание полей

    # Список полей из ответа БД
    my $cnt=0;
    my $sql_fields = {map {$_ => $cnt++} @{$sth->{NAME}} };
    for my $field (@{$RD->{fields}}) {
        if (ref $field eq 'HASH') {
            delete $sql_fields->{$field->{id}};
        } else {
            delete $sql_fields->{$field};
        }
    }
    # Добавляем недостающие поля в конец
    push @{$RD->{fields}}, sort { $sql_fields->{$a} <=> $sql_fields->{$b} }  keys %$sql_fields;

    # Преобразуем поля, заданные скалярами в хеши
    for my $field (@{$RD->{fields}}) {
        if (ref $field ne 'HASH') {
            $field = {
                id  => $field,
                title => $field,
                text_sort => 0,
            };
        }
    }

    # получаем данные
    my $data = $sth->fetchall_arrayref({}) || [];

    # все готово! 
    my $report = {
        data   => $data, 
        fields => $RD->{fields},
    };
    return $report;
}


=head2 details_for_xls

    По данному описанию текущего поля возвращает список заголовков/значений для вывода в эксель 

    Параметры именованные
        title  -- флаг: 1 -- надо вернуть заголовок, 0 -- данные 
        field  -- хеш с описанием текущего поля 
        values -- хеш, соответствующий текущей строке данных (для заголовка не требуется)

=cut
sub details_for_xls
{
    my %OPT = @_;

    my    ($title, $field, $values) = 
    @OPT{qw/title   field   values/};

    if ( !exists $field->{details_key} ){
        return $title ? $field->{title} : $values->{$field->{id}};
    }
    
    my $type = $field->{details_key};
    if ( $type eq 'camp_detailed' ){ 
        return ( 'Кампания', 'Номер') if $title;
        return ( qq/'$values->{camp_detailed}->{name}'/, $values->{camp_detailed}->{cid});
    } elsif ( $type eq 'user_detailed' ){ 
        return 'Логин' if $title;
        return $values->{user_detailed}->{login} ;
    } else {
        return  $title ? $field->{title} : $values->{$field->{id}};
    }
}



=head2 to_xls

    Превращает хеш "отчет" в эксель-файл    
    Результат -- скаляр, пригодный к выводу в файл

=cut

sub to_xls
{
    my ($report) = @_;

    my $xls_report = Yandex::ReportsXLS->new();

    my $title_row = [map { details_for_xls(title => 1, field => $_) } @{$report->{fields}}];

    my @data;
    for my $h (@{$report->{data}}){
        push @data, [ map { details_for_xls(values => $h, field => $_ ) } @{$report->{fields}} ];
    }

    my $xls_data = [ $title_row, @data ];

    if ($report->{message_to_xls} && $report->{status_message}) {
        my @messages = split(/<br>/, $report->{status_message});
        push @$xls_data, [undef], map { [ $_ ] } @messages;
    }

    return $xls_report->array2excel2scalar($xls_data, {freeze_panes => [1,0]});
}

=head2 to_csv

    Вывод в формат CSV

=cut

sub to_csv
{
    my ($report) = @_;


    my $title_row = [map { details_for_xls(title => 1, field => $_) } @{$report->{fields}}];

    my @data;
    for my $h (@{$report->{data}}){
        push @data, [ map { details_for_xls(values => $h, field => $_ ) } @{$report->{fields}} ];
    }

    my $csv_data = data2csv([$title_row, @data], {bom_header => 1, sep_char => ';'});

    return $csv_data;
}

=head2 get_report_from_predefined_sub

=cut
sub get_report_from_predefined_sub
{
    my %OPT = @_;   

    # RD stands for Report Description
    my $RD = $OPT{report_descr};

    # данные
    my $report = $RD->{code}->(%OPT);

    # описания полей отчета (если функция сама об этом не позаботилась)
    # Преобразуем поля, заданные скалярами в хеши
    if (! exists $report->{fields} ){ 
        for my $field (@{$RD->{fields}}) {
            if (ref $field ne 'HASH') {
                $field = {
                    id  => $field,
                    title => $field,
                    text_ver => 0,
                };
            }
        }
        $report->{fields} = $RD->{fields};
    }

    return $report; 
}


=head2 extract_params_from_form

=cut

sub extract_params_from_form($) { 
    my ($form) = @_;
    
    my $opts={};
    for my $f ( grep { /^ir_param_/ } keys %$form ){
        (my $p = $f) =~ s/^ir_param_//;
        $opts->{$p} = $form->{$f}; 
    }
    return $opts;    
}

=head2 get_param_type_by_name

    автоопределение типа, это удобно

=cut

sub get_param_type_by_name($) {
    my $name = shift;

    if ($name =~ m/^(option|checkbox)/) {
        return 'checkbox';
    } elsif ($name =~ m/^select$/) {
        return 'select';
    } elsif ($name =~ m/^date/) {
        return 'date';
    } else {
        return 'text';
    }
}   

=head2 get_global_param_by_name

=cut
sub get_global_param_by_name($) {
    my $name = my $_name = shift;

    my $parent_stack=[];
    my $parent_dict={};

    while($#$parent_stack < 255 ) {   # max_parent_stack_length
        if (exists $PARAM_TYPES{$name}) { 
            last if exists $parent_dict->{$name};
            push @$parent_stack, $name;
            $parent_dict->{$name}=1;    
            $name = $PARAM_TYPES{$name}->{parent};
            last unless defined $name;
        } else {
            #warn "[$name] dosn't exist in \$PARAM_TYPES\n";
            last;
        }
    }

    my $param = hash_merge {}, map { $PARAM_TYPES{$_} } reverse @$parent_stack;
    return $param;
}   

=head2 make_param_hashes

    Преобразовать все параметры, определённые как скаляры в хеши
    
=cut

sub make_param_hashes($) {
    
    my ($RD)  = @_;

    for my $param_str  ( @{$RD->{expected_params}} ) {
        for my $param  ( @$param_str ) {
            next if !defined $param;

            if (ref $param eq 'HASH') {
                $param = hash_merge {}, get_global_param_by_name($param->{parent}), $param if exists $param->{parent};

            } else {

                my $new_param = {
                    name    => $param,
                    default_value => '',
                    title   => $param,
                    type    => get_param_type_by_name($param),
                };

                if (exists $PARAM_TYPES{$param} ) {
                    hash_merge $new_param,
                        get_global_param_by_name($param),     # Берём атрибуты из общего словаря
                        {name => $param};                   # запрещаем менять имя параметра
                }

                # Заменяем имя параметра на хеш
                $param = $new_param;
            }
        }
    }
}

=head2 calc_default_values

=cut
sub calc_default_values($) {
    my ($RD)  = @_;

    for my $param_str  ( @{$RD->{expected_params}} ) {
        for my $param  ( @$param_str ) {
            for my $key ( grep {m/^default_value/} keys %$param) {
                $param->{$key} = $param->{$key}->($param) if defined $param->{$key} and ref $param->{$key} eq 'CODE';
            }
        }
    }

}

=head2 postprocess_params

# постобработка параметров, записываем значения из формы в expected_params

=cut
sub postprocess_params {
    my %OPT = @_;
    my $report = $OPT{report};

    # список параметров (для отображения на странице), если функция сама не позаботилась
    my $expected_params_list = [map {@$_} @{$OPT{report_descr}->{expected_params}}];
    
    my $skip_setting_form_values = exists $report->{params};

    # use Data::Dumper; warn "params: ", Dumper $expected_params_list;
    for my $param (@$expected_params_list) {
        next if !defined $param->{name};    # skip labels and other non informational fields
        my $value = exists $OPT{$param->{name}} ? $OPT{$param->{name}} : '';
            
        $report->{params}{$param->{name}} = $OPT{$param->{name}} if !$skip_setting_form_values;
        $param->{value} = $report->{params}{$param->{name}} if exists $report->{params}{$param->{name}};

        if ( $param->{POSTPROCESS} &&  ref $param->{POSTPROCESS} eq 'CODE') {
            $param->{POSTPROCESS}->($param);
        }
    }
}

=head2 new_rough_forecast

    Новый грубый прогноз расходов кампании

=cut
sub new_rough_forecast
{
    my %OPT = @_;

    return unless $OPT{cid};
    my $camp_info = get_camp_info($OPT{cid}, undef, short => 1);
    return { status_message => "Кампания не найдена" } unless $camp_info && %$camp_info;

    my $date = mysql2unix($OPT{date}) || time;
    my $camp_forecast = Stat::OrderStatDay::get_camp_bsstat_forecast([$OPT{cid}], $OPT{currency})->{$OPT{cid}};

    my $camp_min_rest = Campaign::get_camp_min_rest($OPT{cid}, camp_info => $camp_info);

    my $readable_date = human_date($date);
    return {
        status_message => [
            "Прогноз расхода на $readable_date: $camp_forecast.",
            "Минимальный остаток, необходимый для переноса денег с кампании (на $Settings::TRANSFER_DELAY_AFTER_STOP секунд, на сегодняшний день): $camp_min_rest."
        ],
    };
}

=head2 bind_html5_creative_to_image_ad_banner

    Привязываем html5 креатив к image_ad объявлению в текстовой группе и
    кампании, нужно для эксперимента с html5 игрой, с оплатой по кликово (для
    hezzl)

=cut

sub bind_html5_creative_to_image_ad_banner {
    my %OPT = @_;
    my ($bid, $creative_id, $doit) = @OPT{qw/bid creative_id do_it/};

    my $logger = Yandex::Log::Messages->new();

    return unless defined $bid || defined $creative_id;

    return error("Необходимо задать id объявления") unless defined $bid;
    return error("Необходимо задать id креатива") unless defined $creative_id;

    my $shard = PPC(bid => $bid) or return error("Баннер не найден");

    my $banners = get_all_sql($shard, ["select
                c.clientid banner_clientid, c.type campaign_type,
                b.banner_type,
                bp.banner_creative_id, bp.creative_id, bp.statusModerate,
                b.statusModerate bannerStatusModerate,
                p_cr.yabs_data, p_cr.creative_type
            FROM banners b
                LEFT JOIN banners_performance bp USING(bid)
                LEFT JOIN campaigns c ON b.cid = c.cid
                LEFT JOIN perf_creatives p_cr ON p_cr.creative_id = bp.creative_id
            ", where => {bid => $bid}]);

    return error("Объявление не найдено") unless @$banners;
    return error("Неконсистентное состояние, более двух записей в banners_performance на объявление", $banners) if @$banners > 1;

    my $banner = $banners->[0];
    return error("Тип объявлениея должен быть image_ad") unless $banner->{banner_type} eq 'image_ad';
    return error("У объявления нет записи в banners_performance") unless $banner->{banner_type};
    return error("Тип кампании должен быть text") unless $banner->{campaign_type} eq 'text';
    return error("Объявление должно быть промодерировано; с привязанным  html5-креативом модерация может работать некорректно") unless $banner->{bannerStatusModerate} eq 'Yes';
    $banner->{yabs_data} = from_json($banner->{yabs_data}) if $banner->{yabs_data};

    my $creative = get_one_line_sql($shard, ["select creative_id, creative_type, clientid creative_clientid, yabs_data from perf_creatives", where => {creative_id => $creative_id }]);
    return error("Креатив не найден в perf_creatives для данного шарда") unless $creative;
    return error("Креатив не должен быть типа html5_creative") unless $creative->{creative_type} eq 'html5_creative';
    $creative->{yabs_data} = from_json($creative->{yabs_data}) if $creative->{yabs_data};

    return error("Объявление и креатив принадлежат разным клиентам") unless $banner->{banner_clientid} eq $creative->{creative_clientid};

    if($doit) {
        do_update_table($shard, 'banners_performance', { creative_id => $creative_id, statusModerate => "Yes"},
            where => {
                banner_creative_id => $banner->{banner_creative_id},
                creative_id => $banner->{creative_id},
            });
        $logger->bulk_out(bind_html5_creative_to_image_ad_banner => [{
            current_state => { banner => $banner },
            new_creative =>  $creative
        }]);
    };

    my $json = JSON->new->indent(2);
    return { data => [{
        current_state => $json->encode({ banner => $banner }),
        new_creative =>  $json->encode($creative)
    }]};
}

=head2 profile_logs_http_new

    Получение сырых  trace-логов из http-ручки

=cut

sub profile_logs_http_new {
    my %OPT = @_;

    return {data => []} if !exists $OPT{date_from} && !exists $OPT{date_to} && !$OPT{time_period};

    my %args = (
        time_agg => $OPT{stat_time_agg},
    );

    if ($args{time_agg}) {
        $args{time_agg} =
            $args{time_agg} eq 'hour' ? 60 * 60 :
            $args{time_agg} eq '30min' ? 30 * 60 :
            $args{time_agg} eq '10min' ? 10 * 60 :
            $args{time_agg} eq '5min' ? 5 * 60 :
            $args{time_agg} eq '1min' ? 60 :
            $args{time_agg} =~ /^(\d+)$/ ? $1 :
            croak "Unsupported time_agg format: $args{time_agg}";
    }

    $args{group_by} = [
        map {$OPT{$_}}
        xsort {/^group_by_(\d+)$/ ? int($1) : 0}
        grep {/^group_by_(\d+)$/ && defined $OPT{$_} && $OPT{$_} ne ''}
        keys %OPT
    ];

    $args{filters} = {};
    while(my ($key, $val) = each %OPT) {
        if ($key =~ /^filter_(\w+)$/ && defined $val) {
            my $filter = $1;
            my @vals = grep {$_ ne ''} split /\s+/, $val;
            $args{filters}{$filter} = \@vals if @vals;
	}
    } 
    
    if ($OPT{time_period}) {
        $args{time_period} =
            $OPT{time_period} =~ /^(\d+)M$/ ? $1 * 60 :
            $OPT{time_period} =~ /^(\d+)H$/ ? $1 * 60 * 60 :
            croak "Unsupported time_period format: $OPT{time_period}";
    } else {
        my $date_from = strftime("%Y-%m-%d", localtime mysql2unix($OPT{date_from}));
        my $date_to = strftime("%Y-%m-%d", localtime mysql2unix($OPT{date_to}));
        $args{time_from} = "$date_from ".($OPT{time_from} && is_valid_time($OPT{time_from}) ? "$OPT{time_from}:00" : "00:00:00");
        $args{time_to} = "$date_to ".($OPT{time_to} && is_valid_time($OPT{time_to}) ? "$OPT{time_to}:59" : "23:59:59");
    }

    my $body = encode_json(\%args);
    my $res;
    my %opts = (
        timeout => 420,
        num_attempts => 2,
        ipv6_prefer => 1,
        headers => {
            'Content-Type' => 'application/json',
        },
    );
    eval {
        my $content = http_fetch('POST', $Settings::DIRECT_JAVA_INTAPI_URL . 'trace_logs/profile_stats', $body, %opts);
        my $response = decode_json($content);
        if ($response->{result}) {
            $res = $response->{result};
        } else {
            die "got malformed response: $content\n";
        }
    };

    if (!$res) {
        return {data => [], status_message => "$@"};
    }

    return $res;
}


=head2 bad_redir_campaigns

    Подозрительные кампании, принятые на модерации (с редиректом и известным domain)

=cut

sub bad_redir_campaigns
{
    my %OPT = @_;

    my @bad_domains = qw/
            mamba.ru
            odnoklassniki.ru
            kontakte.ru
            vk.com
            skype.com
            /;

    my $banners = get_all_sql(PPC(shard => 'all'), "
                SELECT p.cid, b.domain, b.bid, b.href
                  FROM banners b
                       JOIN phrases p on p.pid = b.pid
                       JOIN campaigns c ON c.cid = p.cid
                       JOIN bids bi ON bi.pid = p.pid
                       LEFT JOIN bs_auction_stat auct on auct.pid = p.pid and auct.PhraseID = bi.PhraseID
                 WHERE c.archived = 'No'
                   AND b.statusModerate = 'Yes'
                   AND bi.statusModerate = 'Yes'
                   AND bi.showsForecast > 500000
                   AND ifnull(auct.rank,1) > 0
                   AND (".
                     join(" OR ", map {"b.reverse_domain like reverse(".sql_quote("%$_").")"} @bad_domains)
                            .")
                 GROUP BY b.bid
                 ");
    my %camps;
    for my $banner (@$banners) {
        next if strip_domain($banner->{href}) eq strip_domain($banner->{domain})
            || strip_domain($banner->{href}) =~ /\.begun\.ru$/;
        $camps{$banner->{cid}}->{domains}->{$banner->{domain}} = 1;
        push @{$camps{$banner->{cid}}->{banners}}, $banner;
    }

    my @camps;
    for my $cid (sort {$b <=> $a} keys %camps) {
        my $camp = $camps{$cid};
        push @camps, {
            cid => $cid,
            domains => join(",", sort keys %{$camp->{domains}}),
            bids => join(",", map {$_->{bid}} @{$camp->{banners}}),
        };
    }
    
    my $report = {
        data   => \@camps, 
    };

    return $report;
}

=head2 advq_chrono_factor_report

=cut
sub advq_chrono_factor_report 
{
    my %O = @_;

    $O{phrases} ||= '';
 
    my $phrases = [ grep {$_} split /\s*,\s*/, $O{phrases} ];

    my %CT; # stands for chrono table 

    for my $m ( 1 .. 12 ){
        $CT{"mon$m"} = ADVQ6::advq_get_time_coef(geo => '0', phrases => $phrases, period => {month => $m});
    }

    for my $q ( 1 .. 4 ){
        $CT{"q$q"} = ADVQ6::advq_get_time_coef(geo => '0', phrases => $phrases, period => {quarter => $q});
    }

    $CT{"year"} = ADVQ6::advq_get_time_coef(geo => '0', phrases => $phrases, period => {year => 1});

    my $report;
    for my $ph (@$phrases){
        my $rec = { phrase => $ph };
        for my $period ( keys %CT ){
            $rec->{$period} = sprintf( "%0.2f", $CT{$period}->{$ph}->{coef});
        }
        push @{$report->{data}}, $rec;
    }

    return $report;
}

=head2 advq_hist_report

=cut
sub advq_hist_report 
{
    my %O = @_;

    $O{phrases} ||= '';
 
    my $phrases = [ grep {$_} split /\s*[,\n]\s*/, $O{phrases} ];

    my $method = $O{type} eq 'week' ? 'weekly_hist' : 'monthly_hist';
    my $geo = GeoTools::modify_translocal_region_before_save($O{region}, {host => $O{c}->{site_host}});

    my $res_arr = advq_get_stat($phrases, $geo,
                    timeout => 180, function => $method,
                    ($O{advq_lang} && $O{advq_lang} eq 'tr' ? (lang => 'tr') : ()), 
                    ($O{devices} && $O{devices} =~ /^[a-z]+$/i
                     ? (devices  => $O{devices} eq 'mobile' ? [qw/phone tablet/] : [$O{devices}])
                     : ()));

    my %all_dates;
    my @data;

    my %advq_success;
    for my $res (@$res_arr) {
        if ($res->{tainted} !~ m/^(false|0)$/i) {
            print STDERR "Tainted response for phrase '$res->{req}'\n";
            next;
        }
        my $row = {phrase => $res->{req}};
        $advq_success{$row->{phrase}} = 1;   # помечаем фразы, для которых ADVQ вернул результат
        for my $h (@{$res->{hist}}) {
            my $date = $O{type} eq 'week'
                ? $h->{monday} 
                : sprintf("%04d-%02d", $h->{year}, $h->{month});
            $row->{$date} = $h->{total_count};
            $all_dates{$date} = undef;
        }
        push @data, $row;
    }

    for my $failed_phrase (grep { !$advq_success{$_} } @$phrases) {
        my $row = {phrase => $failed_phrase};
        for my $date (keys %all_dates) {
            $row->{$date} = '—';
        }
        push @data, $row;
    }

    return {data => \@data, fields => [map {{id => $_, title => $_}} 'phrase', sort keys %all_dates]};
}

=head2 forecast_ctr_report

    Просмотр данных из forecast_ctr

=cut
sub forecast_ctr_report
{
    my %OPT = @_;

    $OPT{phrases} ||= '';
 
    my $phrases = [ 
    (map { { phrase => $_} } grep {/\S/} split /\s*,\s*/, $OPT{phrases}), 
    (map { {   hash => $_} } grep {/\S/} split /\s*,\s*/, $OPT{hashes} ),
    ];

    for my $p (@$phrases){
        if ($p->{phrase}){
            $p->{norm_phrase} = get_phrase_props($p->{phrase})->{norm_phrase};
            $p->{hash} = PhraseText::get_norm_hash_for_ctr_data($p->{norm_phrase});
        } elsif ($p->{hash}){
            # пытаемся по хешу восстановить фразу
            $p->{phrase} = get_one_field_sql(PPCDICT, "select phrase from suggest_phrases where phrase_hash = ?", $p->{hash});
        }

        my $ctr_data = PhraseText::get_ctr_data($p->{hash});

        hash_merge $p, $ctr_data;
    }

    my $report = {
        data   => $phrases, 
    };

    return $report;
}


=head2 generate_promo_codes

    Генерация промо-кодов для sales-маркетинга

=cut
sub generate_promo_codes
{
    my %OPT = @_;
    my $c = $OPT{c};

    my $ALPHABET = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';

    my $report = { data => [] };
    if (!$OPT{cnt}) {
    } elsif ($OPT{cnt} !~ /^\d+$/) {
        $report->{status_message} = "Неверно указано поле 'Количество'";
    } elsif ($OPT{cnt} !~ /^\d+$/) {
        $report->{status_message} = "Неверно указано поле 'Количество'";
    } elsif ($OPT{prefix} && $OPT{prefix} =~ /([^$ALPHABET])/) {
        $report->{status_message} = "В поле 'Префикс' использован недопустимый символ '$1', допустимы: '$ALPHABET'";
    } else {
        srand();
        for my $i (1 .. $OPT{cnt}) {
            my $code = get_random_string(
                length => 16, prefix => uc($OPT{prefix}), symbols => $ALPHABET
                );
            $code =~ s/(.{4})(?=.)/$1-/g;
            push @{$report->{data}}, {
                code => $code,
                start_dt => human_date(mysql2unix($OPT{start_dt})),
                end_dt => human_date(mysql2unix($OPT{end_dt})),
                middle_dt => human_date(mysql2unix($OPT{middle_dt})),
                payment => $OPT{payment},
                bonus1 => $OPT{bonus1},
                bonus2 => $OPT{bonus2},
                event => $OPT{event}
                };
        };
        $report->{status_message} = "При выгрузке в cvs/xls коды сгенерируются заново";
    }
    return $report;
}

=head2 get_api_domain_stat

  Собирает статистику по домену или по баннеру, которая используется при подсчете юнитов

=cut

sub get_api_domain_stat {

    my %OPT = @_;

    my $data;

    my $base_sql = "select * from api_domain_stat where filter_domain = ? order by stat_date desc";

    if ($OPT{bid}) {

       $data = get_all_sql(PPCDICT, $base_sql, get_domain_for_stat_by_bid($OPT{bid}));

    } elsif ($OPT{domain}) {

       $data = get_all_sql(PPCDICT, $base_sql, $OPT{domain});
    }

    my $report = { data => $data };

    return $report;

}

=head2 get_actual_api_users

  Возвращает список пользоватей, использовавших api за последние days дней

=cut

sub get_actual_api_users {
    my %OPT = @_;
    my $c = $OPT{c};

    $OPT{days} or $OPT{days} = 90;

    my $max_day = $OPT{date} ? mysql2unix($OPT{date}) : time();

    my $where_clause = {};

    if ($OPT{application_id}) {
        $where_clause->{application_id} = $OPT{application_id};
    } elsif ($OPT{ex_application_id}) {
        $where_clause->{application_id__not_in} = $OPT{ex_application_id};
    }
    if ($OPT{application_id_manual}) {
        $where_clause->{application_id} = $OPT{application_id_manual};
    }

    my $users = {};

    my $clh = get_clickhouse_handler('cloud');
    my $result = [];

    $where_clause->{log_date__ge} = strftime("%Y-%m-%d", localtime($max_day - $OPT{days}*60*60*24));;
    my $sql;
    if ($OPT{count_subclients}) {
        $sql = "select distinct uid, max(log_date) as date, count() count from ppclog_api ARRAY JOIN cluid AS uid where " . sql_condition($where_clause) . " and uid > 0 GROUP BY uid ORDER BY count FORMAT JSON";
    } else {
        $sql = "select distinct uid, max(log_date) as date, count() count from ppclog_api where " . sql_condition($where_clause) . " and uid > 0 GROUP BY uid ORDER BY count FORMAT JSON";
    }

    eval {
        $result = $clh->query($sql)->json->{data};
    };
    if($@) {
        warn "clickhouse data retrieval failed: $@";
    }

    foreach my $user (@{$result}) {
        $user->{date} =~ s/\-//g;
        $users->{$user->{uid}} = { uid => $user->{uid}, last_usage => $user->{date}, count => $user->{count} };
    }

    my $roles = rbac_multi_who_is($c->rbac, [keys %{$users}]);
    my $users_data = get_all_sql(PPC(uid=>[keys %{$users}]), ["select uid, login, email from users", where=>{uid=>SHARD_IDS}]);

    foreach (@{$users_data}) {
        $_->{role} = $roles->{$_->{uid}};
        hash_merge $users->{$_->{uid}}, $_;
    }

    return {data => [ sort { $a->{login} cmp $b->{login} } map { $_->{login} = '[unknown] uid:' . $_->{uid} if ! $_->{login}; $_ } values %{$users} ]};
}

=head2 get_daily_api_users_stat

  Сбор статистики по пользователям api за преиод времени.
  Пользователями на определенную дату считаем тех, 
  кто обращался к api в пределах $ACTUAL_PERIOD от текущей даты

=cut

sub get_daily_api_users_stat {
    my %OPT = @_;

    my $ACTUAL_PERIOD = $OPT{actuality_period} || 30;

    my $where_clause;
    if ($OPT{application_id_manual}) {
        $where_clause->{application_id} = $OPT{application_id_manual};
    } elsif ($OPT{application_id}) {
        $where_clause->{application_id} = $OPT{application_id}
    }

    my @stat;
    if ($OPT{date_from} && $OPT{date_to} && $OPT{date_from} =~ m/^\d{4}\-\d{2}\-\d{2}$/ && $OPT{date_to} =~ m/^\d{4}\-\d{2}\-\d{2}$/){

        my $clh = get_clickhouse_handler('cloud');

        my @dates = get_distinct_dates($OPT{date_from}, $OPT{date_to});

        # на каждый день актуальными считаются пользователи пришедшие в этот день, а также пользователи
        # приходившие $ACTUAL_PERIOD дней до этого
        foreach my $yyyymmdd (@dates) {
            my $date = join('-', ($yyyymmdd =~ /(\d\d\d\d)(\d\d)(\d\d)/));
            my $field_name = $OPT{count_subclients} ? 'cluid[1]' : 'uid';
            my $sql = "SELECT toDate('$date') date, uniq($field_name) users FROM ppclog_api WHERE " . ($where_clause ? sql_condition($where_clause) . " AND " : "" ) . "log_date >= (toDate('$date') - $ACTUAL_PERIOD) AND log_date <= toDate('$date') FORMAT JSON";
            eval {
                unshift @stat, $clh->query($sql)->json->{data}->[0];
            };
            print STDERR "InternalReports::get_daily_api_users_stat : $@ for query '$sql`" if $@;
        }
    }

    return { data => \@stat };
}

=head2 send_to_lazy_moderation_queue

    Отправляет все дочерние объекты указанных кампаний в ленивые очереди
    модерации: mod_resync_queue

=cut

sub send_to_lazy_moderation_queue {
    my %params = @_;

    my @cids = grep {is_valid_id($_)} split qr/[^0-9]+/, $params{cids};
    if (!@cids) {
        return {data => [{result => 'Введите корректные cids'}]};
    }

    my $remoderate = $params{remoderate}? 1 : 0;
    my $priority = $params{priority};
    if (!Moderate::ResyncQueue::is_valid_priority($priority)
    ) {
        return {data => [{result => 'Неправильное значение приоритета'}]};
    }

    my $direct_cids = get_one_column_sql(PPC(cid => \@cids), [
        'SELECT cid FROM campaigns',
        WHERE => {
            cid => SHARD_IDS,
            type => get_camp_kind_types('lazy_moderate_resync'),
            statusEmpty => "No",
        },
    ]);

    Moderate::ResyncQueue::mod_resync(Moderate::ResyncQueue::get_all_objects_by(
        cid => $direct_cids, priority => $priority, remoderate => $remoderate));

    return {data => [{
        direct_cids_count => scalar(@{$direct_cids // []}),
        result => 'Ok',
    }]};
}

=head2 resend_mobile_content_to_bs

=cut
sub resend_mobile_content_to_bs {
    my %params = @_;
    my ($shard, $db_res, @ids, $result);

    if (!$params{shard} && !$params{mobile_content_ids}) {
        $result = 'ничего не введено';
    }

    $shard = $params{shard};
    if (!is_valid_int($params{shard}, 1, $Settings::SHARDS_NUM)
        && $shard ne 'all'
    ) {
        $result //= 'неправильный номер шарда';
    }

    @ids = grep { is_valid_id($_) } split(qr/[^0-9]+/, $params{mobile_content_ids} // '');
    if (!@ids) {
        $result //= 'нет ни одного правильного mobile_content_id';
    } elsif (!$result) {
        $db_res = do_update_table(PPC(shard => $shard), 'mobile_content',
                                    {statusBsSynced => 'No'},
                                    where => {
                                        mobile_content_id__int => \@ids,
                                        statusBsSynced__ne => 'No',
                                    },
                                 );
        $result //= 'OK';
    }

    return {
        data => [{
            shard => ($shard // '—'),
            db_res => ($db_res // 0),
            result => $result,
        }],
    }

}

=head2 check_href_redirect

=cut
sub check_href_redirect
{
    my %OPT = @_;

    my @data;

    if ($OPT{href}) {
        my $result_check = get_url_domain($OPT{href}, {check_for_http_status => 1});
        my ($res, $domain_msg) = @{$result_check} {qw/res msg/};
        for my $i (@{$result_check->{redirect_chain} || []}) {
            my $str = {
                href => $i->{url}
                , title => $i->{href}
                , method => $i->{label}
                , req => ($i->{request} ? $i->{request}->headers->as_string : '')
                , resp => ($i->{response} ? $i->{response}->headers->as_string : '')
                , code => ($i->{response} ? $i->{response}->code : '')
                , content => ($i->{response} ? $i->{response}->decoded_content : '')
            };

            # подсвечиваем ошибки
            if ($i->{code} && $i->{code} >= 400) {
                $str->{style}{color} = 'red';
                $str->{style}{bold} = 1;
            }

            push @data, $str;
        }

        my $domain = $res ? $domain_msg : '-';
        my $str = {
            href => $domain
            , req => ($res ? "" : "Ошибка: $domain_msg")
            , label => 'total result'
            , style => {
                bold => 1,
            }
        };

        if (! $res) {
            $str->{style}{color} = '#ff0000';
        }

        push @data, $str;
    }

    return {data => \@data};
}

=head2 analyze_text_lang_report

    просмотр результата queryrec

=cut
sub analyze_text_lang_report
{
    my %O = @_;

    return undef unless $O{text};

    {
        my $t = Yandex::Trace::new_profile("queryrec:init");
        require Yandex::Queryrec;
        require Lang::Guess;
    }

    my $prob = Yandex::Queryrec::queryrec($O{text});
    my $report = {
        data => [{
            text => $O{text},
            lang => Lang::Guess::analyze_text_lang($O{text}),
            queryrec => join(", ", map {"$_:$prob->{$_}"} sort {$prob->{$b} <=> $prob->{$a}} keys %$prob),
            }],
    };

    return $report;
}

=head2 redirect_check_queue

=cut
sub redirect_check_queue
{
    my %O = @_;

    my @bids_data;
    if ($O{cids}) {
        my @cids = grep {/^\d+$/} split /\s*,\s*/, $O{cids};
        my @chunks = sharded_chunks(cid => \@cids, 1_000);
        foreach my $chunk (@chunks) {
            my $cids = $chunk->{cid};
            my $shard = $chunk->{shard};

            my $push_bids_data = get_all_sql(PPC(shard => $shard), ["select bid, href from banners", where => { cid => $cids }, " and href is not null"]) || [];
            push(@bids_data, @$push_bids_data) if @$push_bids_data;
        }
    }

    if ($O{bids}) {
        my @bids = grep {/^\d+$/} split /\s*,\s*/, $O{bids};
        my @chunks = sharded_chunks(bid => \@bids, 1_000);
        foreach my $chunk (@chunks) {
            my $bids = $chunk->{bid};
            my $shard = $chunk->{shard};

            my $push_bids_data = get_all_sql(PPC(shard => $shard), ["select bid, href from banners", where => { bid => $bids }, " and href is not null"]) || [];
            push(@bids_data, @$push_bids_data) if @$push_bids_data;
        }
    }

    if (@bids_data) {
        # сбрасываем кэш для ссылок из перепростукиваемых баннеров
        RedirectCheckQueue::clear_dict( [uniq map { $_->{href} } @bids_data] );
        # добавляем баннеры в очередь
        RedirectCheckQueue::push_banners( [map { {bid => $_->{bid}} } @bids_data] );
    }

    my $data = overshard group => 'domain', sum => 'cnt, cnt_cids', order => '-max_age:num', get_all_sql(PPC(shard => 'all'),
        qq!select domain, count(*) cnt, count(distinct cid) cnt_cids, max(unix_timestamp(NOW()) - unix_timestamp(logtime)) max_age
             from redirect_check_queue rcq
                  left join banners b on rcq.object_id=bid
         group by domain!);

    my $report = {
        data => $data,
    };

    return $report;
}

=head2 search_spam_users

    По login - находим какие еще пользователи заходили с этих же ip за указанный период.
    Либо список логинов по указанному ip за указанный период.

=cut

sub search_spam_users
{
    my %O = @_;

    my ($data, @status_messages);
    
    my ($start, $end) = ( $O{date_from} || today(), $O{date_to} || today() );
    my @dates = get_distinct_dates($start, $end);

    my $flag = ( scalar @dates < 30 || $O{show_all} );

    my $logtable = $O{logapi} ? "ppclog_api" : "ppclog_cmd";
    
    if ($flag && ($O{login} || $O{ip})) {

        my (@list_ips, $list_uids);
        
        if ($O{login}) {

            $list_uids = [ uniq map { get_uid_by_login($_) } split /,\s+/, $O{login} ];

            # ищем всех пользователей с плохой спам-кармой
            my $where = {
                uid__in__int => $list_uids,
                log_date__ge => $start,
                log_date__le => $end,
            };

            my $data = _search_spam_users_select_from_clickhouse($logtable, $where);

            push @list_ips, map {$_->{ip}} @$data;
        } else {
            push @list_ips, split /,\s+/, $O{ip};
        }

        @list_ips = uniq @list_ips if scalar @list_ips;

        my $new_spam_uids;

        if (@list_ips) {

            my $where = {
                ip__in => \@list_ips,
                log_date__ge => $start,
                log_date__le => $end,
            };

            $where->{uid__not_in__int} = $list_uids if $list_uids && @$list_uids;

            # есть запрос '/alive', который делается не от существующего пользователя - в логи для него пишется uid = 0
            push @{ $where->{uid__not_in__int} //= [] }, 0;

            my $data;
            eval {
                $data = _search_spam_users_select_from_clickhouse($logtable, $where);
            };

            warn $@ if $@;
            push @$new_spam_uids, @$data if $data;
        }

        # ищем домен пользователя и статистику по модерации кампаний
        my $res_by_uid = get_hashes_hash_sql(PPC(uid => [ map { $_->{uid} } @$new_spam_uids ]), [
            "SELECT c.uid, u.login, u.statusBlocked as blocked, 
                   group_concat(distinct ifnull(b.domain, '') separator ', ') as domain, 
                   sum(if(b.statusModerate='No',1,0)) as cnt_declined,
                   sum(if(b.statusModerate='Yes',1 ,0)) as cnt_accepted 
            FROM campaigns c 
                   JOIN banners b on c.cid = b.cid
                   JOIN users u on u.uid = c.uid",
            WHERE => { 'c.uid' => SHARD_IDS }, 
            "GROUP BY c.uid"
        ]);

        foreach my $item (@$new_spam_uids) {
            my $res = $res_by_uid->{ $item->{uid} };
            next unless $res;
            push @$data, hash_merge {ip => $item->{ip}}, $res;
        }
    } elsif (! $flag) {
        push @status_messages, "Нельзя выбирать период более 30 дней";
    }
    
    my $report = {
        data => $data,
        status_message => \@status_messages
    };

    return $report;
}

=head2 _search_spam_users_select_from_clickhouse

=cut

sub _search_spam_users_select_from_clickhouse {
    my ($table, $where) = @_;

    my $clh = get_clickhouse_handler('cloud');

    my %fixed_where = %$where;

    if ( $table eq 'ppclog_cmd' ) {
        $fixed_where{service__in} = ['direct.perl.web', 'direct.java.webapi'];
    }

    my $query = $clh->format(["SELECT uid, ip FROM $table WHERE", \%fixed_where, "GROUP BY uid, ip"]);

    $clh->query_format('JSON');

    my $res = $clh->query($query);

    return $res->json->{data};
}

=head2 api_queue_stat

=cut
sub api_queue_stat
{
    my %O = @_;

    my $data;
    my $type = $O{type} || 'report';
    my $table_name = $API::ReportCommon::TABLE_BY_REPORT_TYPE{$type};

    my $report = {
        fields => [
            { id => 'loop_index', title => '#'},
            { id => 'id', title => 'ID отчета'},
            { id => 'uid', title => 'uid'},
            { id => 'options', title => 'Параметры'},
            { id => 'status', title => 'Статус'},
            { id => 'timecreate', title => 'Время&nbsp;создания'},
            { id => 'timeprocess', title => 'Время&nbsp;обработки'},
            { id => 'rank', title => 'Кол-во попыток'},
            { id => 'proc_id', title => 'id процесса'},
        ],
    };

    if ($table_name) {
        my $sort = $O{sort} ?
                        ( grep { $_->{id} eq $O{sort} } @{$report->{fields}} ?
                            " ORDER BY ".$O{sort}." ".( $O{reverse} ? "DESC" : "ASC") : '' ) : '';

        if ($type eq 'report') {
            $report->{data} = get_all_sql(PPCDICT, "select id, uid, options, status, timecreate, timeprocess, rank, proc_id
                                                      from $table_name
                                                     where status in ('Ready', 'Process')
                                                     $sort");
        } else {
            $report->{data} = get_all_sql(PPCDICT, "select id, uid, options, status, timecreate, timeprocess
                                                      from $table_name
                                                     where status in ('Process')
                                                     $sort");
        }

        my $cnt = 0;
        $_->{loop_index} = $cnt++ for @{ $report->{data} };
    }


    return $report;
}

=head2 currency_rates

=cut
sub currency_rates {
    my %O = @_;

    my @data;
    if ($O{currency}) {
        for my $date (get_distinct_dates($O{date_from}, $O{date_to})) {
            my $our_rate = eval { get_currency_rate_in_roubles($O{currency}, $date); } || '<не найден>';
            my $balance_rate = eval { balance_get_currency_rate($O{currency}, $date); } || '<не найден>';
            push @data, {date => TTTools::format_date($date), our_rate => $our_rate, balance_rate => $balance_rate};
        }
    }
    return { data => \@data };
}

=head2 search_eventlog

=cut
sub search_eventlog {
    my (%O) = @_;

    my (%clientid_cond, $clientids, @data);

    if ($O{clientids}) {
        my @orig_clientids = grep { is_valid_int($_) } split /\s*,\s*/, $O{clientids};
        $clientid_cond{'u.ClientID'} = \@orig_clientids if @orig_clientids;
    }

    if ($O{logins}) {
        my @orig_logins = split /\s*,\s*/, $O{logins};
        $clientid_cond{'u.login'} = \@orig_logins if @orig_logins;
    }

    $clientid_cond{'c.cid'} = [split /\s*,\s*/, $O{cids}] if $O{cids};
    $clientid_cond{'b.bid'} = [split /\s*,\s*/, $O{bids}] if $O{bids};
    $clientid_cond{'bi.id'} = [split /\s*,\s*/, $O{bids_ids}] if $O{bids_ids};
    if (%clientid_cond) {
        # если есть логи, то будет и пользователь; в чёрный ящик не ходим
        $clientids = get_one_column_sql(PPC(choose_shard_param (\%clientid_cond, [qw/cid bid id/], allow_shard_all => 1)), [q/
            SELECT DISTINCT u.ClientID
            FROM users u
            LEFT JOIN campaigns c ON u.uid = c.uid
            LEFT JOIN phrases p ON p.cid = c.cid
            LEFT JOIN banners b ON b.pid = p.pid
            LEFT JOIN bids bi ON bi.pid = p.pid
        /, where => \%clientid_cond]) || [];
    }

    if ($clientids && @$clientids) {
        my %conds;
        $conds{'cid'} = $clientid_cond{'c.cid'} if $clientid_cond{'c.cid'};
        $conds{'bid'} = $clientid_cond{'b.bid'} if $clientid_cond{'b.bid'};
        $conds{'bids_id'} = $clientid_cond{'bi.id'} if $clientid_cond{'bi.id'};
        $conds{'type'} = $EventLog::EVENTS{$O{event_type}}->{type} if $O{event_type} && $O{event_type} ne 'any';
        my $events = EventLog::get_events(ClientID => $clientids, date_from => $O{date_from}, date_to => tomorrow($O{date_to}), only_last_event => $O{only_last_event} ? 1 : 0, conditions => \%conds);
        if ($events && @$events) {
            local $YAML::Syck::Headless = 1;
            local $YAML::Syck::SortKeys = 1;
            local $YAML::Syck::ImplicitTyping = 1;
            for my $event(@$events) {
                my $row = hash_cut $event, qw/ClientID eventtime cid bid bids_id/;
                $row->{type} = iget($EventLog::EVENTS{$event->{slug}}->{name});
                $row->{params} = YAML::Syck::Dump $event->{params} if $event->{params};
                $row->{$_} ||= undef for qw/cid bid bids_id/; # не показываем 0, если для события нет кампании/баннера/фразы
                push @data, $row;
            }
        }
    }
    return {data => \@data};
}

=head2 auth_api_token

=cut
sub auth_api_token
{
    my (%O) = @_;

    if ($O{oauth_token}) {
        my $result = Yandex::Blackbox::bb_oauth_token($O{oauth_token}, $ENV{REMOTE_ADDR} || '127.0.0.1', $Settings::API_SERVER_PATH, [$BB_LOGIN]);

        my $data = {
            uid => $result->{uid}
            , login => $result->{dbfield}{$BB_LOGIN}
            , application_id => $result->{client_id}
            , application_name => $result->{OAuth}{client_name}{content}
            , application_page => $result->{OAuth}{client_homepage}{content}
            , scope => $result->{'OAuth'}{scope}{content}
            , status => $result->{error}
            , token_status => $result->{status}
        };

        return {data => [$data]};
    }

    return {};
}

=head2 client_country_currency

=cut

sub client_country_currency {
    my (%O) = @_;

    if ($O{client_login}) {
        my $client_id = get_clientid(login => $O{client_login});
        if (!$client_id) {
            return error("Клиент $O{client_login} не существует или у него нет ClientID");
        }
        my $data_raw = get_all_sql(PPC(ClientID => $client_id), '
                SELECT gr.name AS country
                 , GROUP_CONCAT(DISTINCT cfcc.currency SEPARATOR ", ") AS currencies
                 , cfcc.last_update
                FROM client_firm_country_currency cfcc
                LEFT JOIN geo_regions gr ON cfcc.country_region_id = gr.region_id
                WHERE cfcc.ClientID = ?
                GROUP BY cfcc.country_region_id
            ', $client_id);
        return {data => $data_raw};
    } else {
        return {data => []};
    }
}

=head2 client_discount_schedule

=cut
sub client_discount_schedule {
    my (%O) = @_;

    my @data;
    if ($O{client_login}) {
        my $client_id = get_clientid(login => $O{client_login});
        if (!$client_id) {
            return error("Клиент $O{client_login} не существует или у него нет ClientID");
        }
        my $data_raw = get_all_sql(PPC(ClientID => $client_id), 'SELECT date_from, date_to, discount FROM client_discounts WHERE ClientID = ? ORDER BY date_from', $client_id);
        for my $row(@$data_raw) {
            push @data, {
                date_from => TTTools::format_date($row->{date_from}),
                date_to => TTTools::format_date($row->{date_to}),
                discount => sprintf('%.2f', $row->{discount}),
            };
        }
    }
    return {data => \@data};
}

=head2 banner_images_search

Поиск объявлений с картинкой

=cut

sub banner_images_search
{
    my (%opt) = @_;
    my $image_ids = get_num_array_by_str($opt{image_id});
    return {} unless @$image_ids;
    my $hashes = get_one_column_sql(PPC(shard => 'all'), ["select image_hash from banner_images bim", where => {
        image_id => $image_ids,
    }]) || [];
    my $banners = overshard order => 'bid:num', get_all_sql(PPC(shard => 'all'), ["select c.cid, b.bid, u.login, b.title, b.body
            from campaigns c 
            join users u using(uid)
            join banners b using(cid)
            join banner_images bim using(bid)",
        where => {
            image_hash => $hashes,
        },
    ]);
    my $links = [ map {
        { 
        bid => $_->{bid},
        banner => sprintf(q{<a href="%s#%d">%s</a> %s}, TTTools::get_url('/registered/main.pl', undef, 'showCamp', { 
            cid => $_->{cid}, 
            ulogin => $_->{login}, 
            search_by => 'num', 
            search_banner => $_->{bid},
            tab => 'all'}), $_->{bid}, $_->{title}, $_->{body})
        }
    } @$banners ];


    return { data => $links };
}

my %IS_TEST_HOSTNAME = map { $_ => 1 } (
    'test-direct.yandex.ru',
    'test2-direct.yandex.ru',
    'loadtest-direct.yandex.ru',
);

my $TEST_SECRET_RPC_PORT = 9000;

=head2 _secret_jsonrpc_client

=cut
sub _secret_jsonrpc_client {
    my ( $siteroot, $method ) = @_;

    my $uri = URI->new($siteroot);
    $uri->path("/secret-jsonrpc/$method");

    my $hostname = $uri->host;

    if ( $IS_TEST_HOSTNAME{$hostname} ) {
        $uri->port($TEST_SECRET_RPC_PORT);
    }

    require JSON::RPC::Simple::Client;
    return JSON::RPC::Simple::Client->new( $uri->as_string, { timeout => 600 } );
}

=head2 test_data_generator_campaign_report

=cut
sub test_data_generator_campaign_report {
    my (%OPT) = @_;

    my $ctx = $OPT{'c'};
    my $client = _secret_jsonrpc_client(
        $ctx->site_root, 'TestDataGenerator' );

    my $args = {
        type            => $OPT{'type'},
        phrases_count   => $OPT{'phrases_count'},
        metrika_goals   => $OPT{'metrika_goals'},
        status          => $OPT{'status'},
        nonzero_balance => $OPT{'nonzero_balance'},
    };

    my $result = $client->get_campaigns($args); 

    return { data => $result || [] };
}

=head2 test_data_generator_banner_report

=cut
sub test_data_generator_banner_report {
    my (%OPT) = @_;

    my $ctx = $OPT{'c'};
    my $client = _secret_jsonrpc_client(
        $ctx->site_root, 'TestDataGenerator' );

    my $args = {
        status           => $OPT{'status'},
        with_vcard       => $OPT{'with_vcard'},
        with_href        => $OPT{'with_href'},
        with_sitelinks   => $OPT{'with_sitelinks'},
        with_image       => $OPT{'with_image'},
        with_retargeting => $OPT{'with_retargeting'},
        lowctr_disabled  => $OPT{'lowctr_disabled'},
    };

    my $result = $client->get_banners($args);

    return { data => $result || [] };
}

=head2 _testusers_report_data

=cut
sub _testusers_report_data {
    my $rows = TestUsers::get_all();

    my @result;
    foreach my $row ( @{$rows} ) {
        my $row_out = {
            uid          => $row->{uid},
            domain_login => $row->{domain_login},
            role         => $row->{role},
        };

        push @result, $row_out;
    }

    return \@result;
}

=head2 _testusers_report_error

=cut
sub _testusers_report_error {
    my ($msg) = @_;
    return {
        data           => _testusers_report_data(),
        status_message => $msg,
    };
}

=head2 testusers_report

=cut
sub testusers_report {
    my (%OPT) = @_;

    my $ctx = $OPT{c};
    my $UID = $ctx->UID;

    my $role = $ctx->{login_rights}->{role} || '';
    unless ( $role eq 'super' || $role eq 'superreader' ) {
        die 'Access denied: either a super or a superreader is required';
    }

    if ( my $login = $OPT{login} ) {
        my $uid = get_uid_by_login2($login);

        unless ($uid) {
            return _testusers_report_error("Пользователя $login не существует");
        }

        if ( $OPT{set_inactive} ) {
            my $rows = TestUsers::get_all();
            my ($matching_row) = grep { $_->{'uid'} eq $uid } @$rows;

            unless ($matching_row) {
                return _testusers_report_error("У пользователя $login прав уже нет");
            }

            TestUsers::remove( uid => $uid, UID => $UID );
        } else {
            my $domain_login = $OPT{domain_login};
            my $role         = $OPT{role};

            my $domain_profile = eval { get_staff_info($domain_login) };
            unless ($domain_profile->{login}) {
                return _testusers_report_error("Пользователь $domain_login не найден на Стаффе");
            }

            TestUsers::create_or_replace(
                uid          => $uid,
                domain_login => $domain_login,
                role         => $role,
                UID          => $UID,
            );
        }
    }

    return { data => _testusers_report_data() };
}

=head2 currency_convert_queue_report

    Просмотр очереди клиентов на конвертацию в реальную валюту

=cut

sub currency_convert_queue_report {
    my (%O) = @_;

    my $requests = overshard order => '-create_time', get_all_sql(PPC(shard => 'all'), [q(
        SELECT ClientID
             , create_time
             , convert_type
             , state
             , new_currency
             , country_region_id AS country
             , start_convert_at
             , convert_started_at
             , convert_finished_at
             , in_state_since
             , balance_convert_finished
        FROM currency_convert_queue
        ), ( !$O{include_done} ? (WHERE => {'state__ne' => 'DONE'}) : () ), q(
        ORDER BY create_time DESC
    )]);
    if ($requests && @$requests) {
        my @client_ids = map { $_->{ClientID} } @$requests;
        my $rbac = $O{c}->{rbac};
        my $clientid2chief_uid = {};
        for my $clids_chunk (chunks(\@client_ids, 5_000)) {
            # разбиваем на чанки, чтобы не падать по памяти при большом количестве клиентов
            hash_merge($clientid2chief_uid, RBACElementary::rbac_get_chief_reps_of_clients($clids_chunk));
        }
        my @uids = values %$clientid2chief_uid;
        my $uid2chief_login = get_uid2login(uid => \@uids);

        for my $request (@$requests) {
            my $uid = $clientid2chief_uid->{ $request->{ClientID} };
            $request->{login} = $uid2chief_login->{ $uid };
            if ($O{with_finish_forecast} && $request->{state} ne 'DONE') {
                my $convert_started_at = mysql2unix ( check_mysql_date($request->{convert_started_at}) ? $request->{convert_started_at} : $request->{start_convert_at} );
                my $duration_forecast = Client::ConvertToRealMoney::get_client_convert_duration_forecast($uid, $request->{convert_type});
                $request->{finish_forecast_time} = unix2human($convert_started_at + ($duration_forecast // 0));
            }
        }
    }

    return { data => $requests };
}

=head2 force_currency_convert_queue_report

    Просмотр очереди клиентов на принудительную конвертацию в реальную валюту

=cut

sub force_currency_convert_queue_report {
    my (%O) = @_;

    my $requests = overshard order => 'convert_date', get_all_sql(PPC(shard => 'all'), [q(
        SELECT fcc.ClientID
             , fcc.accepted_at
             , fcc.convert_date
             , fcc.currency
             , fcc.country_region_id AS country
        FROM force_currency_convert fcc
        LEFT JOIN currency_convert_queue q ON fcc.ClientID = q.ClientID
        WHERE
                q.ClientID IS NULL
            AND fcc.convert_date IS NOT NULL
        ORDER BY convert_date
    )]);
    if ($requests && @$requests) {
        my @client_ids = map { $_->{ClientID} } @$requests;
        my $rbac = $O{c}->{rbac};
        my $clientid2chief_uid = rbac_get_chief_reps_of_clients(\@client_ids);
        my @uids = values %$clientid2chief_uid;
        my $uid2chief_login = get_uid2login(uid => \@uids);

        for my $request (@$requests) {
            my $uid = $clientid2chief_uid->{ $request->{ClientID} };
            $request->{login} = $uid2chief_login->{ $uid };
        }
    }

    return { data => $requests };
}


=head2 bs_manage_skip_locked_wallets

    Управляем пропуском отправки в БК кампаний под ОС, заблокированных другими потоками

=cut

sub bs_manage_skip_locked_wallets {
    my (%O) = @_;

    my $prop = Property->new(BS::Export::SKIP_LOCKED_WALLETS_PROP_NAME);

    # set new value only if it is not empty!
    if ( defined $O{enable} && length $O{enable} ) {
        $prop->set(0 + !!$O{enable});
    }

    return { data => [ { enabled => $prop->get() // 'не задано' } ] };
}

=head2 currency_convert_teaser_trigger

    Включение/выключение тизера конвертации в валюты

=cut

sub currency_convert_teaser_trigger {
    my %O = @_;

    my $prop = Property->new($Client::CURRENCY_CONVERT_TEASER_DISABLED_PROPERTY_NAME);

    my $teaser_disabled;
    if ($O{set_teaser_disabled}) {
        $teaser_disabled = ($O{teaser_disabled}) ? 1 : 0;
        if ($teaser_disabled) {
            $prop->set(1);
        } else {
            $prop->delete();
        }
    } else {
        $teaser_disabled = $prop->get();
    }

    return {
        params => {
            teaser_disabled => $teaser_disabled,
            # для снятого чекбокса ничего не передаётся, используем отдельное скрытое поле, чтобы различать первое открытие страницы и последующие отправки
            set_teaser_disabled => 1,
        },
    };
}

=head2 currency_convert_modify_percent

    Изменение процента клиентов для включения тизера конвертации без копирования

=cut

sub currency_convert_modify_percent {
    my %O = @_;

    my $prop = Property->new($Client::CurrencyTeaserData::MODIFY_TEASER_SHOW_PERCENT_PROPERTY_NAME);

    if (defined $O{percent} && is_valid_int($O{percent}, 0)) {
        $prop->set($O{percent});
    }

    return {
        params => {
            percent => $prop->get() // $Client::CurrencyTeaserData::CURRENCY_CONVERT_TEASER_DEFAULT_PERCENT,
        },
    };
}

=head2 api5_java_proxy_service_operations

    Изменение операций api5, которые проксируются в java

=cut

sub api5_java_proxy_service_operations {
    my %O = @_;

    my $prop = Property->new($Settings::API5_JAVA_PROXY_SERVICE_OPERATIONS_PROP);

    my $message = '';
    if ($O{set_property}) {
        if (is_production() && !$O{c}->login_rights->{is_devops}) {
            $message = "Ошибка: В продакшене настройки могут менять только DevOps-ы";
        } else {
            my @operations = grep { $Settings::API5_JAVA_ALLOW_PROXYING_OPERATION{$_} } grep {$_} map {s/^method_//r} keys %O;
            my $normalized_value = join( ' ', sort @operations );
            $prop->set($normalized_value);
            $message = "Сохранено";
        }
    }

    my %checked = map {$_ => 1} split / /, $prop->get() // '';
    return {
        status_message => $message,
        params => {
            map {"method_".$_ => $checked{$_} ? 1 : 0} keys %Settings::API5_JAVA_ALLOW_PROXYING_OPERATION,
        },
    };
}

=head2 priority_clients_api5_units_threshold

    Изменение количества баллов API5, после которого клиент считается приоритетным

=cut

sub priority_clients_api5_units_threshold {
    my %O = @_;

    my $prop = Property->new($Settings::PRIORITY_CLIENTS_API5_UNITS_THRESHOLD_PROPERTY_NAME);

    if ($O{r}->method() eq 'POST' && defined $O{threshold} && is_valid_int($O{threshold}, 0)) {
        $prop->set($O{threshold});
    }

    return {
        params => {
            threshold => $prop->get() // $Settings::PRIORITY_CLIENTS_API5_UNITS_DEFAULT_THRESHOLD,
        },
    };
}

=head2 force_convert_daily_client_limit

    Изменение количества клиентов на принудительную конвертацию в сутки,
    которым назначаем дату и пишем письмо
    Фактически, лимит клиентов для ppcForceCurrencyConvertNotify.pl

=cut

sub force_convert_daily_client_limit {
    my %O = @_;

    my $limit_cnt;
    if (defined $O{limit} && is_valid_int($O{limit}, 0)) {
        Client::ConvertToRealMoney::set_force_convert_daily_client_limit($O{limit});
        $limit_cnt = $O{limit};
    } else {
        $limit_cnt = Client::ConvertToRealMoney::get_force_convert_daily_client_limit();
    }

    return {
        params => {
            limit => $limit_cnt,
        },
    };
}

=head2 force_convert_daily_queue_client_limit

    Изменение количества клиентов на принудительную конвертацию в сутки,
    который ставим в очередь конвертации
    Фактически, лимит клиентов для ppcForceCurrencyConvert.pl

=cut

sub force_convert_daily_queue_client_limit {
    my %O = @_;

    my $limit_cnt;
    if (defined $O{limit} && is_valid_int($O{limit}, 0)) {
        Client::ConvertToRealMoney::set_force_convert_daily_queue_client_limit($O{limit});
        $limit_cnt = $O{limit};
    } else {
        $limit_cnt = Client::ConvertToRealMoney::get_force_convert_daily_queue_client_limit();
    }

    return {
        params => {
            limit => $limit_cnt,
        },
    };
}

=head2 feed_to_banner_land_settings

    Изменение настроек скрипта ppcFeedToBannerland.pl

=cut

sub feed_to_banner_land_settings {
    my %O = @_;

    if (is_valid_int($O{bl_chunk_size}, 1)) {
        Direct::Feeds::set_bl_chunk_size($O{bl_chunk_size});
    }
    if (is_valid_int($O{select_chunk_size}, 1)) {
        Direct::Feeds::set_bl_select_chunk_size($O{select_chunk_size});
    }
    if (is_valid_int($O{max_errors_count}, 1)) {
        Direct::Feeds::set_bl_max_errors_count($O{max_errors_count});
    }
    if (is_valid_int($O{recheck_interval_error_feeds}, 1)) {
        Direct::Feeds::set_bl_recheck_interval_error($O{recheck_interval_error_feeds});
    }
    if (is_valid_int($O{bl_max_sleep_time_seconds}, 1)) {
        Direct::Feeds::set_max_sleep_time_seconds($O{bl_max_sleep_time_seconds});
    }

    return {
        params => {
            bl_chunk_size => Direct::Feeds::get_bl_chunk_size(1),
            select_chunk_size => Direct::Feeds::get_bl_select_chunk_size(1),
            max_errors_count => Direct::Feeds::get_bl_max_errors_count(1),
            recheck_interval_error_feeds => Direct::Feeds::get_bl_recheck_interval_error(1),
            bl_max_sleep_time_seconds => Direct::Feeds::get_max_sleep_time_seconds(1),
        },
    };
}

=head2 resync_moderate_settings

    Изменение настроек ленивой переотправки в Модерацию

=cut

sub resync_moderate_settings {
    my %O = @_;
    if (is_valid_int($O{crit_limit}, 0)) {
        Moderate::ResyncQueue::set_crit_limit($O{crit_limit});
    }
    if (is_valid_int($O{nocrit_limit}, 0)) {
        Moderate::ResyncQueue::set_nocrit_limit($O{nocrit_limit});
    }
    if (is_valid_int($O{max_total_size}, 0)) {
        Moderate::ResyncQueue::set_max_total_size($O{max_total_size});
    }
    if (is_valid_int($O{max_age_minutes}, 0)) {
        Moderate::ResyncQueue::set_max_age_minutes($O{max_age_minutes});
    }

    return {
        params => {
            crit_limit => Moderate::ResyncQueue::get_crit_limit(1),
            nocrit_limit => Moderate::ResyncQueue::get_nocrit_limit(1),
            max_total_size => Moderate::ResyncQueue::get_max_total_size(1),
            max_age_minutes => Moderate::ResyncQueue::get_max_age_minutes(1),
        },
    };
}

=head2 mass_enable_agency_clients_currency_convert_teaser

    Массовое включение/выключение тизера конвертации для субклиентов агентства

=cut

sub mass_enable_agency_clients_currency_convert_teaser {
    my %O = @_;

    my $c = $O{c};
    my $rbac = $c->rbac;

    my @data;
    if ($O{logins}) {
        my @logins = map { split /\s*,\s*/ } split /\s*\n+\s*/, $O{logins};
        my $login2uid = get_login2uid(login => \@logins);
        if (my @bad_logins = grep { !$login2uid->{$_} } @logins) {
            return error('Логины не найдены: ' . join(', ', @bad_logins));
        }

        my @uids = values %$login2uid;
        my $uid2rep_type = rbac_get_agencies_rep_types(\@uids);
        if (my @bad_logins = grep { my $uid = $login2uid->{$_}; !$uid2rep_type->{$uid} } @logins) {
            return error('Логины не являются представителями агентств: ' . join(', ', @bad_logins));
        }


        my $balance_aliases = balance_get_all_equal_clients();
        my %alias_clientids = map { $_ => 1 } map { ($_->{CLIENT_ID}, $_->{CLASS_ID}) } @$balance_aliases;
        undef $balance_aliases;

        my %known_bad_clientids = map { $_ => 1 } @Client::ConvertToRealMoney::KNOWN_BAD_CLIENTIDS;

        my @skip_clientids = uniq(keys %known_bad_clientids, keys %alias_clientids);

        my $subclient_clientids = rbac_get_subclients_clientids($rbac, \@uids);
        my $good_clientids = xminus $subclient_clientids, \@skip_clientids;

        # откидываем валютных субклиентов, свободных и тех, у кого уже есть тизер
        my $clids_to_enable_teaser = get_one_column_sql(PPC(ClientID => $good_clientids), ['
            SELECT u.ClientID, u.login
            FROM users u
            LEFT JOIN clients cl ON u.ClientID = cl.ClientID
            LEFT JOIN clients_to_force_multicurrency_teaser fmt ON u.ClientID = fmt.ClientID
            LEFT JOIN campaigns c ON u.uid = c.uid
            WHERE
                    IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
                AND fmt.ClientID IS NULL
                AND', {
                    'u.ClientID' => SHARD_IDS,
                    _OR => [
                        'c.type__is_null' => 1,
                        _AND => [
                            'c.type__not_in'  => Campaign::Types::get_camp_kind_types('non_currency_convert'),
                            'c.statusEmpty' => 'No',
                        ],
                    ]
                }, '
                GROUP BY u.ClientID
                HAVING COUNT(DISTINCT IFNULL(c.AgencyID,0)) <= 1
        ']);

        Client::ConvertToRealMoney::mass_enable_currency_convert_teaser($clids_to_enable_teaser, modify_convert_allowed => 1);

        my %was_teaser_enabled = map { $_ => undef } @$clids_to_enable_teaser;
        for my $client_id (@$subclient_clientids) {
            my $teaser_enabled = (exists $was_teaser_enabled{$client_id}) ? 1 : 0;

            my @reasons;
            if (exists $known_bad_clientids{$client_id}) {
                push @reasons, 'входит в чёрный список';
            }
            if (exists $alias_clientids{$client_id}) {
                push @reasons, 'эквивалентность (алиас) в Балансе';
            }
            if (!$teaser_enabled && !@reasons) { # отсёкся SQLем выше
                push @reasons, 'уже валютный, свободный или есть тизер';
            }

            push @data, {ClientID => $client_id, teaser_enabled => $teaser_enabled, reasons => join(', ', @reasons)};
        }
        @data = xsort { $_->{teaser_enabled} } @data;
    }

    return {
        params => {
            logins => $O{logins},
        },
        data => \@data,
    };
}

=head2 teaser_data_fetch_parallel_level

    Изменение процента клиентов для включения тизера конвертации без копирования

=cut

sub teaser_data_fetch_parallel_level {
    my %O = @_;

    my $prop = Property->new($Client::CurrencyTeaserData::PARALLEL_LEVEL_PROPERTY_NAME);

    if (defined $O{parallel_level} && is_valid_int($O{parallel_level}, 1)) {
        $prop->set($O{parallel_level});
    }

    return {
        params => {
            parallel_level => $prop->get() || $Client::CurrencyTeaserData::DEFAULT_PARALLEL_LEVEL,
        },
    };
}

=head2 prohibit_payments_for_login

Массовый запрет (или разрешение) оплаты - на всех кампаниях на логине.

=cut
sub prohibit_payments_for_login {
    my %O = @_;

    my $logger;
    my @data;
    if ($O{logins}) {
        my @logins = uniq grep { $_ } split /[\s,]+/, $O{logins};
        my $login2uid = get_login2uid(login => \@logins);
        for my $login (@logins) {
            my $uid = $login2uid->{$login};

            my $status_message;
            if ($uid) {
                my $client_chief_uid = rbac_get_chief_rep_of_client_rep($uid);

                my $statusNoPay_orig = $O{reenable} ? 'Yes' : 'No';
                my $statusNoPay_new = $O{reenable} ? 'No' : 'Yes';

                my $cids_to_change = get_one_column_sql(PPC(uid => $client_chief_uid), "select cid from campaigns where uid = ? and statusNoPay = '$statusNoPay_orig'", $client_chief_uid);
                my $cid_count = int(do_sql(PPC(uid => $client_chief_uid), ["update campaigns set statusNoPay = '$statusNoPay_new'",
                                                                            where => {cid => $cids_to_change}]));
                if ($cid_count > 0) {
                    if (!defined $logger) {
                        $logger = LogTools::make_logger('mass_nopay_change');
                    }
                    $logger->({
                        uid => $client_chief_uid,
                        cids => $cids_to_change,
                        cid_count => $cid_count,
                        login => $O{login},
                        statusNoPay => $statusNoPay_new,
                    });
                }
                $status_message = "Затронуто $cid_count кампаний";
            } else {
                $status_message = 'Пользователь не найден';
            }
            push @data, {login => $login, result => $status_message};
        }
    }
    return { data => \@data };
}

# --------------------------------------------------------------------

=head2 reset_moderation_docs_limit

    Сброс лимита загрузки файлов для модерации

=cut

sub reset_moderation_docs_limit {
    my %O = @_;

    my $c = $O{c};
    my $status_message = '';

    if ($O{login}) {
        my $client_id = get_clientid(login => $O{login});
        my $mcl = Yandex::Memcached::Lock->new(
            servers => $Settings::MEMCACHED_SERVERS,
            entry => "moderation_docs_per_day_$client_id",
            max_locks_num => $Settings::MAX_ATTACHED_DOCUMENTS_DAY_COUNT,
            expire => 24 * 3600,
            no_auto_lock => 1,
            no_auto_unlock => 1,
        );

        $status_message = $mcl->unlock_by_entry() ? "Reset ok" : "User limit not found";
    }

    return {
        status_message => $status_message,
    };
}

=head2 manage_bs_resync_constants

    управление параметрами ленивой переотправки в БК

=cut

sub manage_bs_resync_constants {
    my %O = @_;

    my $resync_chunk_size = Property->new($BS::ExportMaster::RESYNC_CHUNK_SIZE_PROP_NAME);
    my $resync_priority_chunk_size = Property->new($BS::ExportMaster::RESYNC_PRIORITY_CHUNK_SIZE_PROP_NAME);
    my $resync_isize_border = Property->new($BS::ExportMaster::RESYNC_ISIZE_BORDER_PROP_NAME);
    my $resync_age_border = Property->new($BS::ExportMaster::RESYNC_AGE_BORDER_PROP_NAME);
    my $resync_heavy_isize_border = Property->new($BS::ExportMaster::RESYNC_HEAVY_ISIZE_BORDER_PROP_NAME);

    if (is_valid_int($O{RESYNC_CHUNK_SIZE}, 0)) {
        $resync_chunk_size->set($O{RESYNC_CHUNK_SIZE});
    }
    if (is_valid_int($O{RESYNC_PRIORITY_CHUNK_SIZE}, 0)) {
        $resync_priority_chunk_size->set($O{RESYNC_PRIORITY_CHUNK_SIZE});
    }
    if (is_valid_int($O{RESYNC_ISIZE_BORDER}, 0)) {
        $resync_isize_border->set($O{RESYNC_ISIZE_BORDER});
    }
    if (is_valid_int($O{RESYNC_AGE_BORDER}, 0)) {
        $resync_age_border->set($O{RESYNC_AGE_BORDER});
    }
    if (is_valid_int($O{RESYNC_HEAVY_ISIZE_BORDER}, 0)) {
        $resync_heavy_isize_border->set($O{RESYNC_HEAVY_ISIZE_BORDER});
    }

    return { data => [ {
        RESYNC_CHUNK_SIZE => $resync_chunk_size->get() || 'не задано',
        RESYNC_PRIORITY_CHUNK_SIZE => $resync_priority_chunk_size->get() || 'не задано',
        RESYNC_ISIZE_BORDER => $resync_isize_border->get() || 'не задано',
        RESYNC_AGE_BORDER => $resync_age_border->get() || 'не задано',
        RESYNC_HEAVY_ISIZE_BORDER => $resync_heavy_isize_border->get() || 'не задано',
    } ] };
}

=head2 geotools_reg_by_ip

    определение региона по ip

=cut
sub geotools_reg_by_ip {
    my %O = @_;
    return {} unless $O{ips};

    my @data;
    my $lookup = GeoTools::get_geobase_lookup();
    my $region = geobase5::region->new;
    for my $ip (grep {$_} split /[\s,]+/, $O{ips}) {
        my $region_id = GeoTools::get_geo_from_ip($ip);
        my $region_name;
        if (defined $region_id) {
            $lookup->region_by_id($region_id, $region);
            $region_name = Encode::decode_utf8($region->name);
        }
        push @data, {
            ip => $ip,
            region_id => $region_id // '',
            region_name => $region_name // 'не определён',
        };
    }

    return {
        data => \@data,
        fields => [
            {id => "ip", text_sort => 1},
            {id => "region_id"},
            {id => "region_name", text_sort => 1},
            ],
    };
}


=head2 agency_subclient_clientids_report

Субклиенты агентства с ClientID из обычной базы и из RBAC.

=cut
sub agency_subclient_clientids_report {
    my %O = @_;

    my $rbac = $O{c}->rbac;

    my $data = [];
    my $status_message;

    if ( my $agency_login = $O{agency_login} ) {
        if ( my $agency_uid = get_uid_by_login($agency_login) ) {
            my $subclient_uids = rbac_get_subclients_uids( $rbac, $agency_uid );
            my $clientid_map = get_key2clientid( uid => $subclient_uids );
            my $clientid_map_from_rbac = rbac_get_client_clientids_by_uids($subclient_uids);
            for my $client_uid (@$subclient_uids) {
                my $client_clientid = $clientid_map->{$client_uid};
                my $client_clientid_from_rbac = $clientid_map_from_rbac->{$client_uid};
                push @$data, {
                    uid                => $client_uid,
                    clientid           => $client_clientid,
                    clientid_from_rbac => $client_clientid_from_rbac,
                };
            }

            if ( $O{only_mismatching_clientids} ) {
                @$data = grep { $_->{clientid} != $_->{clientid_from_rbac} } @$data;
            }
        } else {
            $status_message = 'Агентство не найдено';
        }
    }

    return {
        data => $data,
        ($status_message ? (status_message => $status_message) : ()),
    };
}

=head2 force_unbind_agency_report

Принудительно отвязать клиента от агентства. Нужно на разработческих/тестовых серверах.
Базы Баланса и наши ТС/devtest переливаются в разное время, так что может оказаться,
что у одного и того же клиента в Балансе и в RBAC разные ClientID. Если такое происходит,
некоторые страницы падают при просмотре от агентства (не могут получить валюту из Баланса,
пример: перенос средств, transferStepZero).

Чтобы найти проблемных клиентов агентства, есть внутренний отчёт agency_subclient_clientids.

=cut
sub force_unbind_agency_report {
    my %O = @_;
    my $rbac = $O{c}->rbac;

    my $status_message;

    my $err = sub { return { data => [], status_message => 'Ошибка: ' . shift } };

    return $err->('Этот отчёт не работает в production') if is_production();
    die if is_production();

    if ( ( my $agency_login = $O{agency_login} ) && ( my $client_login = $O{client_login} ) ) {
        $status_message = '';

        my $agency_uid = get_uid_by_login($agency_login);
        return $err->("Нет такого пользователя: $agency_login") unless $agency_uid;
        return $err->("$agency_login — не агентство") unless rbac_who_is( $rbac, $agency_uid ) eq 'agency';

        my $client_uid = get_uid_by_login($client_login);
        return $err->("Нет такого пользователя: $agency_login") unless $client_uid;
        return $err->("$agency_login — не клиент") unless rbac_who_is( $rbac, $client_uid ) eq 'client';

        # TODO проверить, что агентство имеет отношение к клиенту, а то потом rbac_unbind_agency упадёт
        # с ошибкой 10001 NO_SUCH_RELATION

        my $cids = get_one_column_sql( PPC( uid => $client_uid ),
            [ 'select cid from campaigns', where => { uid => $client_uid, AgencyUID => $agency_uid } ] );

        unless (@$cids) {
            $status_message .= "У этого клиента не было агентских кампаний этого агентства\n";
        }

        my ( @cids_removed_ok, @cids_removed_forcefully );
        for my $cid (@$cids) {
            # первая попытка удалить кампнанию: удаляем с помощью del_camp
            my $del_camp_ok = del_camp( $rbac, $cid, $client_uid, force => 1 );

            if ($del_camp_ok) {
                push @cids_removed_ok, $cid;
            } else {
                # вторая попытка удалить кампанию: чистим таблицы
                # код здесь написан очень приблизительно, в del_camp_data удаления намного больше;
                # но пока этот код работает только на бетах/ТС и правит рассинхронизацию базы Баланса и наших тестовых,
                # это нормально
                do_sql( PPC( uid => $client_uid ), [ 'delete from campaigns', where => { cid => $cid } ] );
                delete_shard( cid => $cid );

                push @cids_removed_forcefully, $cid;
            }
        }

        Rbac::clear_cache();

        if (@cids_removed_ok) {
            $status_message .= 'Кампании, удалённые успешно: ' . join( ', ', @cids_removed_ok ) . "\n";
        }

        if (@cids_removed_forcefully) {
            $status_message .= 'Кампании, удалённые принудительно: ' . join( ', ', @cids_removed_forcefully ) . "\n";
        }

        $status_message .= "Клиент от агентства отвязан успешно.\n";

        $status_message = '<pre>' . string2html($status_message) . '</pre>';
    }

    return {
        data => [],
        ($status_message ? (status_message => $status_message) : ()),
    };
}

=head2 api_forecast_is_get_data_from_bs

    Изменение источника данных для прогноза бюджета в API

=cut
sub api_forecast_is_get_data_from_bs {
    my %O = @_;

    my $prop = Property->new('forecast_is_get_data_from_bs');
    my $is_get_data_from_bs = $prop->get();

    my $data_source;
    unless (is_valid_int($O{is_get_data_from_bs}, 0, 1)) {
        $data_source = $is_get_data_from_bs ? 'БК' : 'ADVQ';
        return {data => [{result => 'Значение должно быть 0 или 1', data_source => $data_source}]}
    }
    $prop->set($O{is_get_data_from_bs});
    $data_source = $prop->get() ? 'БК' : 'ADVQ';
    return {data => [{result => 'OK', data_source => $data_source}]}
}

=head2 api_reports_online_offline_mode_adjuster

    Отчет для настройки параметров на основании которых принимается решение о
    режиме построения отчета (онлайн или оффлайн), см.
    api/v5/API/Service/Reports/ProcessingModeChooser.pm

=cut
sub api_reports_online_offline_mode_adjuster {
    my %O = @_;

    my $online_mode_days_range_property = Property->new($API::Settings::REQUIRE_OFFLINE_MODE_FOR_DATE_RANGE_TYPE_GREATER_THAN_PROPERTY);
    my $online_mode_campaigns_limit_property = Property->new($API::Settings::REQUIRE_OFFLINE_MODE_FOR_LIST_OF_CAMPAIGNS_LONGER_THAN_PROPERTY);

    my $current_params = "Отчет в режиме онлайн можно строить количества дней, не превышающее: "
        . ($O{days_ok_for_online}//"не задано")
        . " и кампаний, не превышающее " . ($O{campaigns_count_ok_for_online}//"не задано");

    my @errors;
    push @errors, 'Количество дней должно быть в диапазоне от 1 до 1200'
        if $O{days_ok_for_online} && !is_valid_int($O{days_ok_for_online}, 1, 1200);
    push @errors, 'Количество кампаний должно быть в диапазоне от 1 до 10000'
        if $O{campaigns_count_ok_for_online} && !is_valid_int($O{campaigns_count_ok_for_online}, 1, 10000);

    return {data => [
        map { {result => "Ошибка", data_source => $_} } @errors
    ]} if @errors;

    if (exists $O{days_ok_for_online} || exists $O{campaigns_count_ok_for_online}) {
        if ($O{days_ok_for_online}) {
            $online_mode_days_range_property->set($O{days_ok_for_online});
        } else {
            $online_mode_days_range_property->delete();
        }
        if ($O{campaigns_count_ok_for_online}) {
            $online_mode_campaigns_limit_property->set($O{campaigns_count_ok_for_online});
        } else {
            $online_mode_campaigns_limit_property->delete();
        }
    }

    return {data => [
        {result => 'Количество дней', data_source =>  $online_mode_days_range_property->get() // "не задано" },
        {result => 'Количество кампаний', data_source =>  $online_mode_campaigns_limit_property->get() // "не задано" }
    ]}
}

=head2 api_use_camp_aggregated_lastchange

    Использовать ли данные из camp_aggregated_lastchange в Changes

=cut
sub api_use_camp_aggregated_lastchange {
    my %O = @_;

    my $prop = Property->new('use_camp_aggregated_lastchange');

    unless (is_valid_int($O{use_camp_aggregated_lastchange}, 0, 1)) {
        return {data => [{result => 'Значение должно быть 0 или 1', data_source => $prop->get()}]}
    }
    $prop->set($O{use_camp_aggregated_lastchange});
    return {data => [{result => 'OK', data_source => $prop->get()}]}
}

=head2 view_clients_stat

    Просмотр бюджетов клиентов (таблицы clients_stat)

=cut

sub view_clients_stat {
    my (%params) = @_;

    my $login = $params{login};
    my $rows;
    if ($login) {
        my $client_id = get_clientid(login => $login);
        $rows = get_all_sql(PPC(ClientID => $client_id), ['
            SELECT type, total_sum_rub, active_28days_sum_rub, daily_spent_rub_estimate
            FROM clients_stat
         ', WHERE => {
                ClientID => SHARD_IDS,
            },
        ]);
    }
    return { data => $rows };
}

=head2 manage_moderation_url
 
    Установка флага использования нового url'a модерации. https://st.yandex-team.ru/DIRECT-6702
    Свойство определяет какой url модерации из настроек следует использовать для jsonrpc запросов.
 
=cut
sub manage_moderation_url {
    my (%O) = @_;
 
    my $prop = Property->new($Moderate::Settings::MODERATE_JSON_RPC_URL_PROPERTY);
 
    unless ( is_valid_int($O{flag}, 0, 1)) {
        return {data => [{result => 'Значение должно быть 0 или 1', current_value => $prop->get()}]};
    }
    $prop->set($O{flag});
    return { data => [ {result => 'Ok', current_value => $prop->get()}]};
}
 
=head2 mass_set_users_hidden

    Массовая пометка пользователей тестовыми

=cut

sub mass_set_users_hidden {
    my %O = @_;

    my @status;
    if ($O{logins}) {
        my @logins = uniq map { TextTools::normalize_login($_) } split /(?:\s*,\s*|\s+)/, $O{logins};
        if (@logins) {
            foreach_shard login => \@logins, chunk_size => 500, with_undef_shard => 1, sub {
                my ($shard, $logins_chunk) = @_;

                my $logins_chunk_str = join(', ', @$logins_chunk);
                if ($shard) {
                    do_update_table(PPC(shard => $shard), 'users', {hidden => 'Yes'}, where => {login => $logins_chunk, hidden => 'No'});
                    push @status, "OK: $logins_chunk_str";
                } else {
                    push @status, "ОШИБКА: не удалось определить шард для логинов $logins_chunk_str";
                }
            };
        }
    }

    return {
        params => {
            logins => $O{logins},
        },
        status_message => \@status,
    };
}

=head2 api_deprecated_method_error_show_properties

    Изменение процента клиентов для включения тизера конвертации без копирования

=cut

sub api_deprecated_method_error_show_properties {
    my %O = @_;

    my $rounds_count_property = Property->new($API::Settings::DEPRECATED_ERROR_SHOW_ROUNDS_COUNT_PROPERTY);
    my $interval_property = Property->new($API::Settings::DEPRECATED_ERROR_SHOW_INTERVAL_PROPERTY);

    if (is_valid_int($O{rounds_count}, 0, 24)) {
        $rounds_count_property->set($O{rounds_count});
        if (is_valid_int($O{interval}, 0, 60 * $O{rounds_count})) {
            $interval_property->set($O{interval});
        }
    }

    return {
        params => {
            rounds_count => $rounds_count_property->get() // $API::Settings::DEPRECATED_ERROR_SHOW_ROUNDS_COUNT_DEFAULT,
            interval => $interval_property->get() // 0,
        },
    };
}

=head2 mass_archive_clients_for_agency

    По списку логинов или ClientID архивирует клиентов указанного агентства.
    Клиенты, у которых есть кампании с положительным остатком, не архивируются, и об этом пишется сообщение.

=cut
sub mass_archive_clients_for_agency {
    my %O = @_;
    my $rbac = $O{c}->rbac;

    my $err = sub { return { data => [], status_message => 'Ошибка: ' . shift } };

    unless ($O{agency_login} && $O{agency_clients_list}) {
        return {data => []};
    }

    # проверка входных данных на валидность

    my $agency_login = $O{agency_login};
    my $agency_uid = get_uid_by_login($agency_login);
    unless ($agency_uid) {
        return $err->("Нет такого пользователя: $agency_login");
    }
    unless (rbac_who_is( $rbac, $agency_uid ) eq 'agency') {
        return $err->("$agency_login — не агентство");
    }
    my $agency_client_id = get_clientid(uid => $agency_uid);

    my @list = grep { $_ } map { s/(^\s+|\s+$)//gr } (split /[\n,]+/, $O{agency_clients_list});

    my $client_client_ids;
    my $client_id_2_login;
    if ($O{clients_list_type} eq 'clientid') {
        my @bad_ids = grep { !Yandex::Validate::is_valid_id($_) } @list;
        if (@bad_ids) {
            return $err->("Указаны плохие ClientID: ".join(',', @bad_ids));
        }

        $client_client_ids = \@list;
    } else {
        my $login_2_clientid = get_login2clientid(login => \@list);

        my @bad_logins = grep { !$login_2_clientid->{$_} } @list;
        if (@bad_logins) {
            return $err->("Клиенты не найдены по логинам: ".join(',', @bad_logins));
        }

        $client_client_ids = [map { $login_2_clientid->{$_} } @list];
        $client_id_2_login = {map { $login_2_clientid->{$_} => $_ } @list};
    }

    my %results;
    foreach_shard ClientID => $client_client_ids, with_undef_shard => 1, chunk_size => 1_000, sub {
        my ($shard, $clid_chunk) = @_;

        if (!$shard) {
            $results{$_} = 'Не удалось определить шард или шард нулевой' for @$clid_chunk;
        } else {
            my $client_details = get_hashes_hash_sql(PPC(shard => $shard), ["
                SELECT acr.client_client_id, acr.client_archived
                FROM agency_client_relations acr
                    LEFT JOIN users u ON (u.ClientID = acr.client_client_id)
                    LEFT JOIN clients cl ON (cl.ClientID = u.ClientID)",
                   # Сложное условие указано в JOIN а не в WHERE для того, чтобы запрос выводил всех клиентов агентства,
                   # даже если условия по кампании не выполняются.
                   # Отсутствие клиента в результате запроса трактуется как отсутствие клиента у агентства, и он не архивируется.
                   "LEFT JOIN campaigns c ON (c.uid = u.uid AND c.AgencyID = ? AND c.archived = 'No')",
                WHERE => {
                    'acr.client_client_id__int' => $clid_chunk,
                    'acr.agency_client_id__int' => $agency_client_id
                }, "
                GROUP BY acr.client_client_id"], $agency_client_id);

            my @clid2update;
            my $sums = mass_client_total_sums(
                ClientIDs => $clid_chunk,
                type => get_camp_kind_types('web_edit_base', 'media', 'wallet'),
            );

            for my $client_id (@$clid_chunk) {
                my $details = $client_details->{$client_id};
                my $result;

                if (!$details) {
                    $result = 'Не является клиентом агентства';
                } elsif ($details->{client_archived} eq 'Yes') {
                    $result = 'Уже заархивирован';
                } elsif ($sums->{$client_id} && ($sums->{$client_id}->{total} > get_currency_constant($sums->{$client_id}->{currency}, 'MAX_CLIENT_ARCHIVE'))) {
                    $result = sprintf('Остаток средств превышает допустимый для архивации (%s %s)',
                        get_currency_constant($sums->{$client_id}->{currency}, 'MAX_CLIENT_ARCHIVE'),
                        get_currency_constant($sums->{$client_id}->{currency}, 'name'));
                } else {
                    push @clid2update, $client_id;
                    $result = 'Отправлен в архив';
                }

                $results{$client_id} = $result;
            }

            if (@clid2update) {
                 do_update_table(PPC(shard => $shard), 'agency_client_relations', {client_archived => 'Yes'},
                        where => {client_client_id => \@clid2update, agency_client_id => $agency_client_id});
            }
        }
    };

    my @data;
    for my $client_id (keys %results) {
        my $login = $client_id_2_login->{$client_id} // '';
        push @data, {clientid => $client_id, login => $login, result => $results{$client_id}};
    }

    return {data => [xsort { $_->{result} } @data]};
}


=head2 bs_disable_smart_tgo

Остановка смарт-тго баннеров по списку OrderID

=cut

sub bs_disable_smart_tgo {
    my %params = @_;
    my ($order_ids_str, $need_stop_banners) = @params{qw/order_ids need_stop_banners/};
    my $order_ids = get_num_array_by_str($order_ids_str);

    return _error_result('не выбраны кампании') if (!@$order_ids);

    my $banners = get_all_sql(PPC(OrderID => $order_ids), [
            'SELECT OrderID, cid, bid, creative_id
            FROM campaigns c
            JOIN banners b USING(cid)
            JOIN banners_performance bp USING(cid, pid, bid)
            JOIN perf_creatives cr USING(creative_id)',
            WHERE => {
                OrderID => SHARD_IDS,
                'b.statusShow' => 'Yes',
                layout_id => 44, # smart-tgo layout
            },
            'ORDER BY OrderID, bid'
        ]);

    if ($need_stop_banners) {
        Models::Banner::stop_banners($banners);
    }

    return {data => $banners};
}

=head2 bs_sync_creatives
    Отправляет все или указанные креативы на синхронизацию
=cut
sub bs_sync_creatives {
    my %params = @_;
    my ($sync_mode, $creative_ids, $all_creatives) = @params{qw/sync_mode creative_ids all_creatives/};

    my @result;
    my $MAX_ROWS_PER_INSERT = 500;
    my $CHUNK_SIZE = 400;

    return _error_result( 'не выбраны креативы для синхронизации' ) if (! $all_creatives && ! $creative_ids);

    my $full_sync = $sync_mode ? 0 : 1;
    
    my (%WHERE, @shards);
    
    if ($all_creatives) {
         @shards = ShardingTools::ppc_shards;
    }
    else {
        my $arr_creative_ids = get_num_array_by_str($creative_ids);
        return _error_result( 'не задано ни одного корректного идентификатора креатива' ) unless @$arr_creative_ids;
        $WHERE{creative_id} = \@$arr_creative_ids;
        @shards = uniq values %{Yandex::DBShards::get_shard_multi(creative_id => $arr_creative_ids)};
    }
   
    @shards = grep {defined $_ && $_ gt ''} @shards;
    return _error_result( 'не задано ни одного существующего креатива' ) unless @shards;
    
    foreach my $shard (nsort @shards){
        my $filtered_creatives;
          my $tasks_count = 0;
        while (1) {
            $filtered_creatives = get_one_column_sql(PPC(shard => $shard),
                [qq/SELECT creative_id FROM perf_creatives/, (WHERE => {%WHERE, creative_type => ['performance', 'bannerstorage']}), 'ORDER BY' => 'creative_id', LIMIT => $CHUNK_SIZE]
            );
            last unless @$filtered_creatives;
            $tasks_count += Direct::Model::Creative::Manager->create_creative_sync_tasks($shard => $filtered_creatives, $full_sync, $MAX_ROWS_PER_INSERT);
            $WHERE{creative_id__gt} = $filtered_creatives->[-1];
        }
        delete $WHERE{creative_id__gt};
        push @result, {shard => $shard, db_res => $tasks_count, result => 'OK'}
    }

    return {
        data => \@result
    }
}

sub _error_result {
    my ($result) = @_;
    
    return {
        data => [{
            result => $result,
        }],
    }
}

=head2 add_agency_clients_to_currency_convert_queue

    Cтавит в очередь на конвертацию субклиентов указанного агенства по списку логинов.
    Для попадания в очередь у клиентов должен быть проставлен тизер мультивалютности.

=cut

sub add_agency_clients_to_currency_convert_queue {
    my %O = @_;
    my $rbac = $O{c}->rbac;

    my $err = sub { return { data => [], status_message => 'Ошибка: ' . shift } };

    unless ($O{agency_login} && $O{agency_clients_list}) {
        return {data => []};
    }

    # проверка входных данных на валидность

    my $agency_login = $O{agency_login};
    my $convert_date = $O{convert_date};
    my $agency_uid = get_uid_by_login($agency_login);
    unless ($agency_uid) {
        return $err->("Нет такого пользователя: $agency_login");
    }
    unless (rbac_who_is( $rbac, $agency_uid ) eq 'agency') {
        return $err->("$agency_login — не агентство");
    }
    my $agency_id = get_clientid(uid => $agency_uid);

    my $convert_start_ts = Client::ConvertToRealMoney::get_closest_modify_convert_start_ts();
    if ($convert_date) {
        return $err->("Неправильный формат даты. Нужно указать дату в формате YYYY-MM-DD") unless is_valid_date($convert_date);
        if (mysql2unix($convert_date) < $convert_start_ts) {
            # дата конвертации не должна быть раньше ближайшей возможной
            my $closest_modify_convert_start = human_date($convert_start_ts);
            return $err->("Дата конвертации не должна быть раньше ближайшей возможной: $closest_modify_convert_start");
        } else {
            $convert_start_ts = mysql2unix($convert_date);
        }
    }

    my @list = grep { $_ } map { s/(^\s+|\s+$)//gr } (split /[\s,]+/, $O{agency_clients_list});
    my $login_2_clientid = get_login2clientid(login => \@list);
    my @bad_logins = grep { !$login_2_clientid->{$_} } @list;
    if (@bad_logins) {
        return $err->("Клиенты не найдены по логинам: ".join(',', @bad_logins));
    }
    if (scalar @list > $Client::ConvertToRealMoney::MASS_CONVERT_BUTTON_CLIENTS_LIMIT) {
        return $err->("Количество клиентов на конвертацию не должно превышать $Client::ConvertToRealMoney::MASS_CONVERT_BUTTON_CLIENTS_LIMIT");
    }

    my %result;

    my $client_id_2_login = {map { $login_2_clientid->{$_} => $_ } @list};
    my $clids = [ keys $client_id_2_login ];

    # оставляем только тех, кто обслуживается в укзанном агентсве
    my $agency_clids = [];
    my $relation = get_agency_client_relations($agency_id, $clids);
    foreach my $client_id (@$clids) {
        if (defined $relation->{$client_id} && $relation->{$client_id}->{agency_bind}) {
            push @$agency_clids, $client_id;
        } else {
            $result{$client_id}->{msg} = "ОШИБКА: клиент не обслуживается в агентстве $agency_login";
        }
    }

    # оставляем только потенциально годных к конвертации клиентов: в YND_FIXED и c тизером
    my $clid2currencies = mass_get_client_currencies($agency_clids);
    my ($good_clids, $bad_clids) = part { $clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' ? 0 : 1 } @$agency_clids;
    foreach my $client_id (@$bad_clids) {
        $result{$client_id}->{msg} = 'ОШИБКА: клиент уже в валюте';
    }

    my $clients = get_all_sql(PPC(ClientID => $good_clids), ['
        SELECT ctfmt.ClientID
             , cfcc.country_region_id
             , cfcc.currency
        FROM clients_to_force_multicurrency_teaser ctfmt
        LEFT JOIN currency_convert_queue q ON ctfmt.ClientID = q.ClientID
        INNER JOIN client_firm_country_currency cfcc ON ctfmt.ClientID = cfcc.ClientID
     ', WHERE => {
            'ctfmt.ClientID' => SHARD_IDS,
            'q.ClientID__is_null' => 1,
        }, '
        GROUP BY ctfmt.ClientID
        HAVING count(cfcc.ClientID) = 1
    ']);

    my $not_selected_clids = xminus($good_clids, [ map { $_->{ClientID} } @$clients ]);
    foreach my $client_id (@$not_selected_clids) {
        my $start_convert_at = get_one_field_sql(PPC(ClientID => $client_id), 'select start_convert_at from currency_convert_queue where ClientID = ?', $client_id);
        if ($start_convert_at) {
            $result{$client_id}->{msg} = 'ОК';
            $result{$client_id}->{convert_date} = unix2human(mysql2unix($start_convert_at) - 60);
        } else {
            $result{$client_id}->{msg} = 'ОШИБКА: у клиента нет тизера мультивалютности';
        }
    }

    if (@$clients) {
        my $client_ids = [ map { $_->{ClientID} } @$clients ];
        my $clid2nds = mass_get_client_NDS($client_ids, fetch_for_ynd_fixed_too => 1);
        my $clid2first_agency = Primitives::mass_get_client_first_agency($client_ids);
        my $clients_data = mass_get_clients_data($client_ids, [qw/allow_create_scamp_by_subclient/]);
        my $clid2chief_uid = {};
        for my $client_ids_chunk (chunks($client_ids, 5_000)) {
            # разбиваем на чанки, чтобы не падать по памяти при большом количестве клиентов
            hash_merge($clid2chief_uid, RBACElementary::rbac_get_chief_reps_of_clients($client_ids_chunk));
        }

        # по факту, уже продублировали почти все проверки из Client::can_convert_to_real_money в SQL запросе выше, поэтому её не используем
        my @convert_requests;
        my $clients_count = 0;
        for my $client (@$clients) {
            my $client_id = $client->{ClientID};
            my $login = $client_id_2_login->{$client_id};

            my $client_nds = $clid2nds->{$client_id};
            unless (defined $client_nds) {
                $result{$client_id}->{msg} = 'ОШИБКА: не определен НДС';
                next;
            }

            my $is_free_client = $clients_data->{$client_id}->{allow_create_scamp_by_subclient};
            if ($is_free_client) {
                $result{$client_id}->{msg} = 'ОШИБКА: несколько типов обслуживания ("свобода")';
                next;
            }

            my $currency = $client->{currency};
            my $country = $client->{country_region_id};

            $clients_count++;

            # единственность стран и валют и то, что они такие -- проверили в исходном SQL-запросе
            my @country_currencies = ({region_id => $country, currency => $currency});
            my $convert_type = Client::ConvertToRealMoney::get_convert_type($client_id, $currency, $client_nds, \@country_currencies);
            my $to_convert = {
                ClientID => $client_id,
                uid => undef, # SMS никому не пишем
                convert_type => $convert_type,
                new_currency => $currency,
                country_region_id => $country,
                email => undef, # отдельных писем не пишем, пишем только общее письмо по окончанию конвертации всех поставленных клиентов
                start_convert_ts => $convert_start_ts,
            };
            $result{$client_id}->{msg} = 'OK';
            $result{$client_id}->{convert_date} = unix2human($convert_start_ts - 60);
            push @convert_requests, $to_convert;
        }

        if (@convert_requests) {
            Client::ConvertToRealMoney::mass_queue_currency_convert(\@convert_requests, ignore => 1, AgencyID => $agency_id);
        }
    }

    my @data;
    for my $client_id (keys %result) {
        my $login = $client_id_2_login->{$client_id} // '';
        push @data, {clientid => $client_id, login => $login, convert_date => $result{$client_id}->{convert_date}, result => $result{$client_id}->{msg}};
    }

    return {data => [xsort { $_->{result} } @data], no_reports_links => 1};
}

=head2 mass_unblock_users

    По списку логинов блокирует/разблокирует пользователей.
    Клиентам в фишках при блокировке дополнительно проставляется флаг cant_unblock - разблокировка невозможна.

        expected_params => [
            {name => 'logins_list', title => 'Список логинов',
                parent => "textarea",
                cols => 32, rows => 20,
            },
        ],
        fields => [
            { id => 'login',    title => 'Логин' },
            { id => 'result',   title => 'Результат'},
        ],

=cut
sub mass_unblock_users {
    my %O = @_;

    my $err = sub { return { data => [], status_message => 'Ошибка: ' . shift } };

    unless ($O{logins_list}) {
        return {data => []};
    }

    my $do_block = $O{do_block};

    # проверка входных данных на валидность

    my @list = grep { $_ } map { s/(^\s+|\s+$)//gr } (split /[\n,]+/, $O{logins_list});

    my $login_2_uid = get_login2uid(login => \@list);

    my @bad_logins = grep { !$login_2_uid->{$_} } @list;
    if (@bad_logins) {
        return $err->("Пользователи не найдены по логинам: ".join(',', @bad_logins));
    }

    my $uid_2_login = {map { $login_2_uid->{$_} => $_ } @list};

    my %results;
    foreach_shard uid => [values %$login_2_uid], with_undef_shard => 1, chunk_size => 1_000, sub {
        my ($shard, $uid_chunk) = @_;

        if (!$shard) {
            $results{$_} = 'Не удалось определить шард или шард нулевой' for @$uid_chunk;
            return;
        }

        my $user_details = get_hashes_hash_sql(PPC(shard => $shard), ["
            SELECT u.uid, u.statusBlocked, u.ClientID, MAX(c.cid) as cid, IFNULL(cl.work_currency, 'YND_FIXED') as currency, FIND_IN_SET('cant_unblock', clo.client_flags) > 0 as cant_unblock
            FROM users u
                 LEFT JOIN campaigns c ON c.uid = u.uid
                 LEFT JOIN clients cl ON cl.ClientID = u.ClientID
                 LEFT JOIN clients_options clo ON clo.ClientID = u.ClientID",
            WHERE => {
                'u.uid__int' => $uid_chunk,
            },
            "GROUP BY u.uid"]);

        my @uid2unblock;
        my @uid2block;
        my @clientid2block;
        for my $uid (@$uid_chunk) {
            my $details = $user_details->{$uid};

            my $result;

            if (!$details) {
                $result = 'Не найден';
                next;
            }
            if (!$details->{cid}) {
                $result = 'Нельзя '.($do_block ? 'заблокировать' : 'разблокировать').' - нет ни одной кампании';
            }
            if ($do_block) {
                if ($details->{statusBlocked} eq 'No') {
                    push @uid2block, $uid;
                    # клиентам в фишках нужно установить флаг cant_unblock
                    if ($details->{currency} eq 'YND_FIXED') {
                        push @clientid2block, $details->{ClientID};
                    }
                    $result = 'Заблокирован';
                } else {
                    $result = 'Уже заблокирован';
                }
            } else {
                if ($details->{statusBlocked} eq 'No') {
                    $result = 'Не заблокирован';
                } elsif ($details->{cant_unblock}) {
                    $result = 'Нельзя разблокировать, установлен флаг cant_unblock';
                } else {
                    push @uid2unblock, $uid;
                    $result = 'Разблокирован';
                }
            }

            $results{$uid} = $result;
        }

        if (@clientid2block) {
            do_update_table(PPC(shard => $shard), 'clients_options', {client_flags__smod => {cant_unblock => 1}},
                    where => {ClientID => [ uniq @clientid2block ]});
        }

        if (@uid2unblock) {
            do_update_table(PPC(shard => $shard), 'users', {statusBlocked => 'No'},
                    where => {uid => \@uid2unblock});

        }

        if (@uid2block) {
            do_update_table(PPC(shard => $shard), 'users', {statusBlocked => 'Yes'},
                    where => {uid => \@uid2block});
        }
    };

    my @data;
    for my $uid (keys %results) {
        my $login = $uid_2_login->{$uid} // '';
        push @data, {login => $login, result => $results{$uid}};
    }

    return {data => [xsort { $_->{result} } @data]};
}

=head2 mass_drop_cant_unblock_client_flag

    По списку логинов снимает флаг сant_unblock.

=cut
sub mass_drop_cant_unblock_client_flag {
    my %O = @_;

    my $err = sub { return { data => [], status_message => 'Ошибка: ' . shift } };

    unless ($O{logins_list}) {
        return {data => []};
    }

    # проверка входных данных на валидность

    my @list = grep { $_ } map { s/(^\s+|\s+$)//gr } (split /[\n,]+/, $O{logins_list});

    my $login_2_uid = get_login2uid(login => \@list);

    my @bad_logins = grep { !$login_2_uid->{$_} } @list;
    if (@bad_logins) {
        return $err->("Пользователи не найдены по логинам: ".join(',', @bad_logins));
    }

    my $uid_2_login = {map { $login_2_uid->{$_} => $_ } @list};

    my %results;
    foreach_shard uid => [values %$login_2_uid], with_undef_shard => 1, chunk_size => 1_000, sub {
        my ($shard, $uid_chunk) = @_;

        if (!$shard) {
            $results{$_} = 'Не удалось определить шард или шард нулевой' for @$uid_chunk;
            return;
        }

        my $user_details = get_hashes_hash_sql(PPC(shard => $shard), ["
            SELECT u.uid, u.ClientID, FIND_IN_SET('cant_unblock', clo.client_flags) > 0 as cant_unblock
            FROM users u
            LEFT JOIN clients_options clo ON clo.ClientID = u.ClientID",
            WHERE => {
                'u.uid__int' => $uid_chunk,
            },
            "GROUP BY u.uid"]);

        my @clientid2unblock;
        for my $uid (@$uid_chunk) {
            my $details = $user_details->{$uid};

            my $result;

            if (!$details) {
                $result = 'Не найден';
                next;
            }

            if ($details->{cant_unblock}) {
                push @clientid2unblock, $details->{ClientID};
                $result = 'Флаг cant_unblock снят';
            } else {
                $result = 'Флаг cant_unblock отсутствует';
            }
            $results{$uid} = $result;
        }

        if (@clientid2unblock) {
            do_update_table(PPC(shard => $shard), 'clients_options', {client_flags__smod => {cant_unblock => 0}},
                    where => {ClientID => [ uniq @clientid2unblock ]});
        }
    };

    my @data;
    for my $uid (keys %results) {
        my $login = $uid_2_login->{$uid} // '';
        push @data, {login => $login, result => $results{$uid}};
    }

    return {data => [xsort { $_->{result} } @data]};
}

=head2 remove_clients_from_currency_convert_queue

    Массовая отмена конвертации пользователей в валюту
    Удаляет задание из очереди конвертации, если оно в статусе NEW

=cut

sub remove_clients_from_currency_convert_queue {
    my %O = @_;

    my @status;
    if ($O{logins}) {
        my @logins = uniq map { TextTools::normalize_login($_) } split /(?:\s*,\s*|\s+)/, $O{logins};
        if (@logins) {
            foreach_shard login => \@logins, chunk_size => 500, with_undef_shard => 1, sub {
                my ($shard, $logins_chunk) = @_;

                if ($shard) {
                    my $login2clientid = get_login2clientid(login => $logins_chunk);
                    my $clientid2login = { reverse %$login2clientid };
                    my $client_ids = [ keys $clientid2login ];
                    foreach my $client_id (@$client_ids) {
                        my $deleted_rows_cnt = do_sql(PPC(shard => $shard), [
                                                       "DELETE ccq, ccc
                                                        FROM currency_convert_queue ccq
                                                            LEFT JOIN client_currency_changes ccc ON ccc.ClientID = ccq.ClientID",
                                                        WHERE => {'ccq.ClientID' => $client_id, 'ccq.state' => 'NEW'}
                                                    ]);

                        if ($deleted_rows_cnt eq '0E0') {
                            push @status, "ОШИБКА: $clientid2login->{$client_id} (ClientID = $client_id) логин отсутсвует в очереди на конвертацию или статус конверации не NEW";
                        } else {
                            push @status, "OK: $clientid2login->{$client_id} (ClientID = $client_id)";
                        }
                    }
                } else {
                    my $logins_chunk_str = join(', ', @$logins_chunk);
                    push @status, "ОШИБКА: не удалось определить шард для логинов $logins_chunk_str";
                }
            };
        }
    }

    return {
        params => {
            logins => $O{logins},
        },
        status_message => \@status,
    };
}

=head2 mass_set_users_not_convert_to_currency

    Массовая пометка пользователей неконвертируемыми в валюту по списку логинов.
    Позволяет устанавливать и снимать флаг not_convert_to_currency в clients_options.client_flags.
    При установке флага снимает тизер мультивалютности.

=cut

sub mass_set_users_not_convert_to_currency {
    my %O = @_;

    my @status;
    if ($O{logins}) {
        my @logins = uniq map { TextTools::normalize_login($_) } split /(?:\s*,\s*|\s+)/, $O{logins};
        if (@logins) {
            foreach_shard login => \@logins, chunk_size => 500, with_undef_shard => 1, sub {
                my ($shard, $logins_chunk) = @_;

                if ($shard) {
                    my $login2clientid = get_login2clientid(login => $logins_chunk);
                    my $clientid2login = { reverse %$login2clientid };
                    my $clids = [ keys $clientid2login ];

                    # Проверяем, что клиент в фишках
                    my $clid2currencies = mass_get_client_currencies($clids);
                    my ($good_currency_client_ids, $bad_currency_client_ids) = part { $clid2currencies->{$_}->{work_currency} eq 'YND_FIXED' ? 0 : 1 } @$clids;
                    if (defined $bad_currency_client_ids && @$bad_currency_client_ids) {
                        my $bad_currency_logins_str = join(', ', map { $clientid2login->{$_} } @$bad_currency_client_ids);
                        push @status, "ОШИБКА: клиенты уже в валюте - $bad_currency_logins_str" if $bad_currency_logins_str;
                    }

                    # Проверяем, что клиент не стоит в очереди на конвертацию
                    my $clids_in_queue =get_hash_sql(PPC(ClientID => $good_currency_client_ids), ["select ClientID, 1 from currency_convert_queue", where => {ClientID => SHARD_IDS}]);
                    my $client_ids = [ map { $clids_in_queue->{$_} ? () : $_ } @$good_currency_client_ids ];
                    if (keys %$clids_in_queue) {
                        my $logins_in_queue_str = join(', ', map { $clientid2login->{$_} } keys %$clids_in_queue);
                        push @status, "ОШИБКА: клиенты стоят в очереди на конвертацию - $logins_in_queue_str";
                    }

                    foreach my $client_id (@$client_ids) {
                        create_update_client({client_data => {ClientID => $client_id, not_convert_to_currency => !$O{flag_disabled}}});
                        push @status, "OK: $clientid2login->{$client_id} (ClientID = $client_id)";
                    }
                    # снимаем тизер мультивалютности
                    do_delete_from_table(PPC(shard => $shard), 'clients_to_force_multicurrency_teaser', where => {ClientID => $client_ids}) if !$O{flag_disabled};
                } else {
                    my $logins_chunk_str = join(', ', @$logins_chunk);
                    push @status, "ОШИБКА: не удалось определить шард для логинов $logins_chunk_str";
                }
            };
        }
    }

    return {
        params => {
            logins => $O{logins},
            flag_disabled => $O{flag_disabled},
        },
        status_message => \@status,
    };
}

=head2 not_convert_to_currency_user_list

    Просмотр списка пользователей, помеченных флагом неконвертируемости в валюту (clients_options.client_flags)

=cut

sub not_convert_to_currency_user_list {
    my %O = @_;

    my $data = get_all_sql(PPC(shard => 'all'),
        "SELECT u.login, co.ClientID
         FROM clients_options co
         LEFT JOIN users u USING(clientid)
         WHERE FIND_IN_SET('not_convert_to_currency', co.client_flags)>0");

    return {data => [xsort { $_->{login} } @$data]};
}

=head2 mass_drop_users_nds

    Массовый сброс НДС у пользователей и постановка их в очередь на переполучение НДС в Балансе

=cut

sub mass_drop_users_nds {
    my %O = @_;
    return {data => []} if !$O{logins};

    my @logins = uniq grep { $_ }
            map { TextTools::normalize_login(TextTools::smartstrip2($_)) } split /[\s,]+/, $O{logins};
    return {data => []} if !@logins;
    my $logger = Yandex::Log::Messages->new();

    my @data;
    foreach_shard login => \@logins, chunk_size => 100, with_undef_shard => 1, sub {
        my ($shard, $logins_chunk) = @_;
        my $result = 'Логин не найден или некорректен';

        if ($shard) {
            my $client_ids = get_clientids(login => $logins_chunk);

            # Удаляем текущую информацию об НДС
            do_delete_from_table(PPC(shard => $shard), 'client_nds', where => { ClientID => $client_ids });
            # Ставим в очередь на получение новой информации
            do_mass_insert_sql(PPC(shard => $shard),
                "INSERT IGNORE INTO clients_to_fetch_nds (ClientID) VALUES %s", [ map {[ $_ ]} @$client_ids ]);
            $result = 'НДС сброшен';
            $logger->bulk_out(drop_nds_for_clientid => [{client_ids => $client_ids}]);
        }

        for my $login (@$logins_chunk) {
            push @data, { login => $login, result => $result };
        }
    };

    return { data => \@data };
}

=head2 currency_convert_forecast

    Прогноз принудительной конвертации

=cut

sub currency_convert_forecast {

    # считаем количество клиентов, которых могли бы поставить на принудительную конвертацию
    # немного модифицированным запросом из ppcForceCurrencyConvertNotify.pl
    my $clients_can_convert_cnt = overshard_sum(PPC(shard => 'all'), "SELECT count(*) FROM (
            SELECT fcc.ClientID
            FROM force_currency_convert fcc
            INNER JOIN users u ON u.ClientID = fcc.ClientID
            LEFT JOIN clients cl ON fcc.ClientID = cl.ClientID
            INNER JOIN clients_to_force_multicurrency_teaser t ON fcc.ClientID = t.ClientID
            INNER JOIN client_firm_country_currency cfcc ON cfcc.ClientID = fcc.ClientID
            LEFT JOIN currency_convert_queue q ON fcc.ClientID = q.ClientID
            INNER JOIN campaigns c ON u.uid = c.uid
            WHERE
                q.ClientID IS NULL
                AND IFNULL(cl.work_currency, 'YND_FIXED') = 'YND_FIXED'
            GROUP BY fcc.ClientID
            HAVING
                    COUNT(DISTINCT cfcc.country_region_id) = 1
                AND COUNT(DISTINCT cfcc.currency) = 1
        ) qq");
    # количество клиентов на принудительную конвертацию в сутки, который ставим в очередь конвертации
    my $limit_cnt = Client::ConvertToRealMoney::get_force_convert_daily_queue_client_limit();
    my $days_to_complete_currency_convert = int($clients_can_convert_cnt / $limit_cnt);
    my $date = Yandex::DateTime->now()->add(days => $days_to_complete_currency_convert)->strftime("%Y-%m-%d");

    return {
        status_message => [
            "Всего клиентов, пригодных принудительной к конвертации: $clients_can_convert_cnt",
            "Скорость конвертации: $limit_cnt клиентов в сутки",
            "Примерная дата завершения конвертации: $date"
        ],
    };
}

=head2 enable_context_shows_for_logins

    Включение показов в сети в стандартном режиме для списка клиентов.
    Сеть будет включена для типов кампаний text, mobile_content, performance

=cut

sub enable_context_shows_for_logins {
    my %O = @_;

    my $c = $O{c};
    my $data;

    if ($O{logins}) {
        my @logins = split /(?:\s*,\s*|\s+)/, $O{logins};
        if (scalar @logins > 1000) {
            return {data => [], status_message => "Превышен лимит в 1000 логинов. Уменьшите количество логонов и повторите запрос."};
        }

        my $login2uid = get_login2uid(login => \@logins);
        my $bad_logins = xminus(\@logins, [keys %$login2uid]);
        foreach my $bad_login (@$bad_logins) {
            push @$data, {login => $bad_login, result => 'Логин не найден'};
        }

        foreach my $login (keys %$login2uid) {
            my $uid = $login2uid->{$login};
            my $cids = get_one_column_sql(PPC(uid => $uid),
                ['SELECT cid from campaigns',
                  WHERE => {
                    uid => SHARD_IDS,
                    statusEmpty => 'No',
                    platform => 'search',
                    type => [qw/text mobile_content performance/]}
                ]);
            if (!@$cids) {
                push @$data, {login => $login, result => 'Сеть уже включена на всех кампаниях логина'};
                next;
            }
            foreach my $cid (@$cids) {
                my $camp = get_camp_info($cid);

                $camp->{strategy} = Campaign::campaign_strategy($camp);
                # включаем сеть в стандартном режиме
                $camp->{ContextLimit} = 0;
                $camp->{ContextPriceCoef} = 100;
                $camp->{opts}->{enable_cpc_hold} = 1;

                # изменяем стратегию
                my $strategy = $camp->{strategy};
                $strategy->{is_net_stop} = '';
                $strategy->{net}->{name} = 'default';

                my $err = Campaign::validate_camp_strategy($camp, $strategy);
                if ($err) {
                    push @$data, {login => $login, cid => $cid, result => $err}
                } else {
                    my $old_camp_strategy = $camp->{strategy};

                    Campaign::camp_set_strategy($camp, $strategy, {
                        uid => $uid,
                        i_know_strategy_min_price => 1,
                        send_notifications => 1
                    });

                    if($old_camp_strategy->{is_autobudget} || $camp->{strategy}->{is_autobudget}){
                        AutobudgetAlerts::update_on_strategy_change({
                            cid => $cid,
                            old_strategy => $old_camp_strategy,
                            new_strategy => $camp->{strategy}
                        });
                    }

                    $camp->{fio} = $camp->{FIO}; # save_camp ожидает получить ФИО маленькими буквами, а get_camp_info возвращает большими (sic!)
                    # save_camp также ожидает часть полей в логическом виде, хотя из get_camp_info они приходят в виде Yes/No
                    $camp->{$_} = ($camp->{$_} && $camp->{$_} eq 'Yes') ? 1 : undef for qw/sendWarn sendAccNews statusMetricaControl statusContextStop/;
                    # ещё часть полей приходят в виде логического значения, а вот сохраняются в зависимости от defined этого значения
                    $camp->{$_} = ($camp->{$_}) ? 1 : undef for qw/offlineStatNotice fairAuction broad_match_flag/;
                    # а еще, get_camp_info возвращает minus_words, а save_camp ожидает campaign_minus_words
                    $camp->{campaign_minus_words} = delete $camp->{minus_words};

                    save_camp($c, $camp, $uid);
                    push @$data, {login => $login, cid => $cid, result => 'ОК'};
                }
            }
        }
    }
    return {data => $data};
}

=head2 drop_user_goals

    Удаление целей пользователя

=cut
sub drop_user_goals {
    my %O = @_;
    my $goal_deletion_mode = $O{goal_deletion_mode};
    if (!$O{login} && !$O{goal_ids_list}) {
        return {};
    }

    my $goal_id_condition_key;
    if ($goal_deletion_mode eq 'ask') {
        return { status_message => ["Не выбран режим работы"] };
    } elsif($goal_deletion_mode eq 'except_list') {
        $goal_id_condition_key = 'goal_id__not_in';
    } elsif ($goal_deletion_mode eq 'within_list') {
        $goal_id_condition_key = 'goal_id__in';
    } else {
        return { status_message => ["Неизвестный режим $goal_deletion_mode"] };
    }

    my $login = TextTools::normalize_login(TextTools::smartstrip2($O{login}));
    my $client_uid = get_uid(login => $login);
    unless ($client_uid) {
        return { status_message => ["Логин $login не найден"] };
    }

    my @goal_ids_list = uniq grep { is_valid_id($_) } split /[^\d]+/, $O{goal_ids_list} || '';
    unless (@goal_ids_list) {
        return { status_message => ["В списке нет корректных id"] };
    }

    my $cids = get_cids(uid => $client_uid);

    my $total_deleted_qty = 0;
    my @deleted_key_pairs;
    foreach_shard cid => $cids, chunk_size => 100, sub {
        my ($shard, $cids_chunk) = @_;
        my $goals_to_delete = get_all_sql(PPC(shard => $shard), ["select cid, goal_id from camp_metrika_goals",
                                                                   where => { cid => $cids_chunk,
                                                                              $goal_id_condition_key => \@goal_ids_list,
                                                                              goals_count => 0,
                                                                              context_goals_count => 0 }]);
        my %goals_to_delete_by_cid = ();
        for my $row (@$goals_to_delete) {
            push @{$goals_to_delete_by_cid{$row->{cid}}}, $row;
        }
        for my $cid (keys %goals_to_delete_by_cid) {
            relaxed sub {
                my $goal_ids_for_query = [ map { $_->{goal_id} } @{$goals_to_delete_by_cid{$cid}} ];
                my $deleted_qty = do_delete_from_table(PPC(cid => $cid), 'camp_metrika_goals',
                                                                         where => { cid => $cid,
                                                                                    goal_id => $goal_ids_for_query,
                                                                                    goals_count => 0,
                                                                                    context_goals_count => 0 });

                my @deleted_keys = map { { cid => $cid, goal_id => $_ } } @$goal_ids_for_query;
                push @deleted_key_pairs, @deleted_keys;
                $total_deleted_qty += $deleted_qty;
            };
        }
    };
    return {
        status_message => ["Количество удаленных записей из таблицы camp_metrika_goals: $total_deleted_qty"],
        data => \@deleted_key_pairs
    };
}

=head2 upload_banner_resources

    Загрузка медийных файлов в видеоплатформу

=cut

sub upload_banner_resources {
    my %O = @_;

    unless ($O{input_file}) {
        return {data => []};
    }

    my ($file, $type, $name, $colors, $categories) = @O{qw/input_file type name colors categories/};
    my $file_name = $file . '';

    my $data = [
        {
            file => $file_name,
            type => $type,
            name => $name,
            colors => $colors,
            categories => $categories,
        },
    ];
    if (!$type || none { $type eq $_ } qw/video audio/ ) {
        $data->[0]->{result} = 'Ошибка: не указан тип ресурса (video/audio)';
        return {data => $data};
    }
    if (!$name || $name eq '' ) {
        $data->[0]->{result} = 'Ошибка: не указано название ресурса';
        return {data => $data};
    }
    if (!$colors || $colors eq '' ) {
        $data->[0]->{result} = 'Ошибка: не указан цвет подложки';
        return {data => $data};
    }

    # Добавление информации о видеоролике
    my $params = {
        uid => $Settings::VIDEO_API_UID,
        format => 'json',
    };

    my $url = Yandex::HTTP::make_url($Settings::VIDEO_API_URL. "/users/$Settings::VIDEO_API_UID/films", $params);
    my $response = Yandex::HTTP::http_parallel_request(
        POST => {
            0 => {
                url => $url,
                body => {title => $name},
            }
        },
        timeout => 30,
        soft_timeout => 0.5,
        num_attempts => 2,
    )->{0};

    # Если видео успешно загрузилось, то ждем пока оно сконвертируется и возвращаем результат
    if (!$response->{is_success}) {
        $data->[0]->{result} = 'Ошибка: не удалось добавить информацию о файле в видеоплатформу';
    } else {
        my $content = decode_json($response->{content}) || {};

        my $api_url = $content->{apiUrl};
        my $targetUrl = $content->{targetUrl};

        if ($targetUrl) {
            my $ua = LWP::UserAgent->new;
            # Объект запроса
            my $request = HTTP::Request->new('POST', $targetUrl);

            # Формируем случайный разделитель, так как если мы его не укажем принудительно, то при $ua->request($request) он у нас не войдет в основной заголовок
            my $boundary = 'X';
            my @symbols = ('a'..'z', 'A'..'Z');
            for (0..14) {$boundary .= $symbols[int(rand(@symbols))];}

            # Формируем заголовок:
            $request->header('Content-Type' => 'multipart/form-data; boundary='.$boundary);
            my $header = HTTP::Headers->new;
            $header->header('Content-Disposition' => "form-data; name=$type; filename=$file_name"); # Хотя filename можно вычислить и из имени файла
            $header->header('Content-Type' => 'application/octet-stream'); # Или соответсвующий типу файла
            my $file_content = HTTP::Message->new($header);
            $file_content->add_content($_) while <$file>;
            $request->add_part($file_content);

            my $response = $ua->request($request);
            if ($response->is_success) {
                my $job = Yandex::DBQueue->new(PPCDICT, 'convert_media_resources')->insert_job({
                                           job_id => get_new_id('job_id'),
                                           uid => $O{c}->UID,
                                           ClientID => 0,
                                           args => {
                                               file => $file_name,
                                               type => $type,
                                               name => $name,
                                               colors => $colors,
                                               categories => $categories,
                                               api_url => $api_url,
                                           },
                                       });
                $data->[0]->{result} = 'OK: задание на конвертацию файла добавлено';
            } else {
                $data->[0]->{result} = 'Ошибка: не удалось загрузить файл в видеоплатформу';
            }
        }
    }
    return {data => $data};
}

=head2 media_resources_report

    Показать информацию про успешно загруженные видео/аудио ресурсы

=cut

sub media_resources_report {

    my $data = get_all_sql(PPCDICT, "SELECT media_resource_id, resource_type, name, yacontextCategories, colors, preview_url, resources_url FROM media_resources where resource_type != 'undefined'");

    foreach my $rec (@$data) {
        # оставляем из исходного json-a только ссылки для лучшей человекочитаемости
        $rec->{preview_url} = join (' ', map { $_->{url}} @{decode_json($rec->{preview_url})});
        $rec->{resources_url} = join (' ', map { $_->{url}} @{decode_json($rec->{resources_url})});
    }

    return {data => $data};
}

=head2 update_bs_ad_duration_report

    Изменение длительности видеоподложек. 
    
=cut

sub update_bs_ad_duration_report {
    my %O = @_;
    my $db_property = Property->new($Settings::BS_VIDEO_AD_DURATION_LIMIT_PROP);
    my @status;
    my $new_ad_duration_limit;

    if ($O{'set_ad_duration_limit'}) {
        my $provided_ad_duration_limit =  $O{ad_duration_limit};
        if (is_valid_int($provided_ad_duration_limit, 1)) {
            $new_ad_duration_limit = $provided_ad_duration_limit;
            $db_property->set($new_ad_duration_limit);
            push @status, "OK: длительность видеоподложек изменена на $new_ad_duration_limit";
        } else {
            push @status, "ОШИБКА: длительность должна быть целым числом больше нуля (было введено значение '$provided_ad_duration_limit')";
            $new_ad_duration_limit = $db_property->get();
        }
    } else {
        $new_ad_duration_limit = $db_property->get();
    }
    return { 
               data => [{
                   current_ad_duration_limit => $new_ad_duration_limit
               }],
               status_message => \@status
           };
}

=head2 delete_moderation_cmd_queue_entry

Удаление команд копирования (по id + cid) из очереди модерации moderation_cmd_queue

=cut

sub delete_moderation_cmd_queue_entry {
    my %O = @_;
    my @status;
    my $result;
    if ($O{'delete_moderation_cmd_queue_entry'}) {
        my $qid;
        if (is_valid_id($O{'queue_id'})) {
            $qid = $O{'queue_id'};
        }
        push @status, "ОШИБКА: некорректный id" unless $qid;
        my $cid;
        if (is_valid_id($O{'cid'})) {
            $cid = $O{'cid'};
        }
        push @status, "ОШИБКА: некорректный cid" unless $cid;
        if ($qid && $cid) {
            my $deleted_count;
            if ($O{'delete_from_all_shards'}) {
                $deleted_count = do_delete_from_table(PPC(shard => 'all'), 'moderation_cmd_queue', where => { cid => $cid, id => $qid, cmd => 'copyCampModerateInfo' });
            } else {
                $deleted_count = do_delete_from_table(PPC(cid => $cid), 'moderation_cmd_queue', where => { cid => $cid, id => $qid, cmd => 'copyCampModerateInfo' });
            }
            push @status, "число удаленных строк:".($deleted_count == '0E0' ? 0 : $deleted_count);
        }
    }
    return {
        status_message => \@status
    };
}

=head2 ya_agency_orders

    Просмотр и подтверждение заявок на настройку Яндекс Директа
    
=cut

sub ya_agency_orders {
    my %IN = @_;
    
    my @status;
    my $client_id = $IN{ClientID};
    if ($client_id) {
        push @status, 'Ошибка: неверный ClientID' unless is_valid_id($client_id);
        unless (@status) {
            my $order_data = {
                ClientID => $client_id,
            };
            $order_data->{product_type} = $IN{product_type} if $IN{product_type};
            my $res = Direct::YaAgency::send_order(%$order_data);
            push @status, 'Ошибка: для заданного ClientID нет неподтвержденной заявки' unless $res->{ok};
            push @status, 'Выставлен счет: '.$res->{url} if $res->{url};
        }
    }
    
    push @status, '','Неподтвержденные заявки на обслуживание';
    my $orders = get_all_sql(PPC(shard => 'all'), ['SELECT ClientID as client_id, ClientID, product_type, created FROM yandex_agency_orders',
        WHERE => {yaOrderStatus => 'New'}, ORDER => 'BY created']);
    
    $_->{product_type} = ($Direct::YaAgency::PRODUCTS->{$_->{product_type}} // 'устаревший') foreach @$orders;
    
    return {
        data =>  $orders,
        status_message => \@status,
    }
}

=head2 ya_agency_clients

    Просмотр оплаченных и закрытие отработанных заявок на настройку Яндекс Директа
    
=cut

sub ya_agency_clients {
    my %IN = @_;
    
    my @status;
    my $client_id = $IN{ClientID};
    my $desired_status = $IN{desiredStatus} // 'Paid';
    my $force_zero =  $IN{force_zero};

    if ($client_id) {
        push @status, 'Ошибка: неверный ClientID' unless is_valid_id($client_id);
        push @status, 'Ошибка: нулевые открутки можно отправлять только для заявок со статусом "Завершена"!'
            if ($desired_status ne 'Completed' && $force_zero);
        push @status, 'Ошибка: закрытие допускается только для заявок со статусом "Ожидает рассервисирования"!'
            if (!$force_zero && $desired_status ne 'Paid');

        unless (@status) {
            my $ok = Direct::YaAgency::finalise_order(ClientID => $client_id, force_zero => $force_zero);
            push @status, 'Ошибка: не удалось закрыть заявку клиента '.$client_id unless $ok;
        }
    }
    my $order_type = $desired_status eq 'Paid' ? 'Оплаченные' : 'Завершенные';
    $order_type = 'Переотправленные' if $desired_status eq 'Resurrected';
    push @status, '',sprintf('%s заявки на обслуживание', $order_type);

    my $orders = get_all_sql(PPC(shard => 'all'), ['SELECT
                        ClientID as client_id, ClientID, product_type, created, LastChange as changed
                FROM yandex_agency_orders',
        WHERE => {yaOrderStatus => $desired_status}]);
    @$orders = sort {$b->{changed} cmp $a->{changed}} @$orders;
    
    $_->{product_type} = ($Direct::YaAgency::PRODUCTS->{$_->{product_type}} // 'устаревший') foreach @$orders;
    
    return {
        data =>  $orders,
        status_message => \@status,
    }
}

=head2 bs_resync_relevance_match_campaigns

    Добавляет в очередь задание на переотправку в БК кампаний с relevance_match
    
=cut

sub bs_resync_relevance_match_campaigns {
    my %params = @_;

    my ($client_ids_str, $resync_all_clients) = @params{qw/client_ids resync_all_clients/};
    my $operator_uid = $params{c}->UID;

    my @status;
    if ($client_ids_str) {
        my $client_ids = get_num_array_by_str($client_ids_str);
        return { status_message => ['не задано ни одного корректного идентификатора клиента'] } unless @$client_ids;

        my $shard_by_client_id = get_shard_multi(ClientID => $client_ids);
        my ($found, $not_found) = part { $shard_by_client_id->{$_} ? 0 : 1 } keys %$shard_by_client_id;

        if ($found && @$found) {
            my $job = Yandex::DBQueue->new(PPCDICT, 'bs_resync_relevance_match')->insert_job({
                job_id => get_new_id('job_id'),
                uid => $operator_uid,
                ClientID => 0,
                args => {client_ids => $found},
            });
            push @status, "добавлено задание на переотправку в БК кампаний клиентов: " . join(", ", @$found); 
        }
        push @status, "клиенты не найдены: " . join(", ", @$not_found) if ($not_found && @$not_found);

    } elsif ($resync_all_clients) {
        my $job = Yandex::DBQueue->new(PPCDICT, 'bs_resync_relevance_match')->insert_job({
            job_id => get_new_id('job_id'),
            uid => $operator_uid,
            ClientID => 0,
            args => {resync_all_clients => 1},
        });
        push @status, "добавлено задание на переотправку в БК кампаний всех клиентов";

    } else {
        return { status_message => ['не выбраны клиенты'] };
    }

    return { status_message => \@status };
}

=head2 balance_resync_clients_of_limited_agency_reps

    Добавляет в очередь задание на переотправку в Баланс клиентов ограниченных представителей
    
=cut

sub balance_resync_clients_of_limited_agency_reps {
    my %params = @_;

    my ($client_ids_str, $resync_all_clients) = @params{qw/client_ids resync_all_clients/};
    my $force_reset = $params{force_reset} ? 1 : 0;
    my $operator_uid = $params{c}->UID;

    my @status;
    if ($client_ids_str) {
        my $client_ids = get_num_array_by_str($client_ids_str);
        return { status_message => ['не задано ни одного корректного идентификатора клиента'] } unless @$client_ids;

        my $shard_by_client_id = get_shard_multi(ClientID => $client_ids);
        my ($found, $not_found) = part { $shard_by_client_id->{$_} ? 0 : 1 } keys %$shard_by_client_id;

        if ($found && @$found) {
            my $job = Yandex::DBQueue->new(PPCDICT, 'balance_resync_limited_rep')->insert_job({
                job_id => get_new_id('job_id'),
                uid => $operator_uid,
                ClientID => 0,
                args => {client_ids => $found, force_reset => $force_reset},
            });
            push @status, "добавлено задание на переотправку для агентств: " . join(", ", @$found); 
        }
        push @status, "агентства не найдены: " . join(", ", @$not_found) if ($not_found && @$not_found);

    } elsif ($resync_all_clients) {
        my $job = Yandex::DBQueue->new(PPCDICT, 'balance_resync_limited_rep')->insert_job({
            job_id => get_new_id('job_id'),
            uid => $operator_uid,
            ClientID => 0,
            args => {resync_all_clients => 1, force_reset => $force_reset},
        });
        push @status, "добавлено задание на переотправку для всех агентств";

    } else {
        return { status_message => ['не выбраны агентства'] };
    }

    return { status_message => \@status };
}

=head2 update_aggregator_domains

    Добавляет в очередь задание на обновление aggregator_domains
    
=cut

sub update_aggregator_domains {
    my %params = @_;

    unless ($params{'update_aggregator_domains'}) {
        return { status_message => [] };
    }

    my ($resync_all_clients, $domain) = @params{qw/resync_all_clients domain/};
    my $client_ids = get_num_array_by_str($params{client_ids});
    my $operator_uid = $params{c}->UID;
    my (%args, @errors);

    if ($domain && any { $domain eq $_ } @AggregatorDomains::ALLOWED_DOMAINS) {
        $args{domain} = $domain;
    } else {
        push @errors, 'не выбран домен';
    }

    if (@$client_ids) {
        my $shard_by_client_id = get_shard_multi(ClientID => $client_ids);
        my @not_found = grep { !$shard_by_client_id->{$_} } @$client_ids;
 
        if (@not_found) {
            push @errors, "клиенты не найдены: " . join(", ", @not_found); 
        } else {
            $args{client_ids} = [uniq @$client_ids];
        }
    } elsif ($resync_all_clients) {
        $args{resync_all_clients} = 1;
    } else {
        push @errors, 'не выбраны клиенты';
    }

    my @status_message;
    if (@errors) {
        push @status_message, @errors;
    } else {
        my $job = Yandex::DBQueue->new(PPCDICT, 'update_aggregator_domains')->insert_job({
            job_id => get_new_id('job_id'),
            uid => $operator_uid,
            ClientID => 0,
            args => \%args,
        });
        push @status_message, sprintf("в очередь добавлено задание для %s с доменом %s", 
            ($args{resync_all_clients} ? 'всех клиентов' : 'клиентов: ' . join(", ", @{ $args{client_ids} })), $args{domain}); 
    }
    return { status_message => \@status_message };
}

=head2 wallet_daily_budget_stop_warning_settings

    Настройка предупреждений об остановке общего счета по дневному бюджету

=cut

sub wallet_daily_budget_stop_warning_settings {
    my %FORM = @_;

    my $warning_time_prop = Property->new($Wallet::DAILY_BUDGET_STOP_WARNING_TIME_PROP);
    my @messages;

    if ($FORM{warning_time}) {
        if ($FORM{warning_time} =~ /^\d\d:\d\d:\d\d$/) {
            $warning_time_prop->set($FORM{warning_time});
            push @messages, "значение изменено";
        } else {
            push @messages, "неправильный формат";
        }
    }

    return { data => [{warning_time => $warning_time_prop->get() // $Wallet::DAILY_BUDGET_STOP_WARNING_TIME_DEFAULT}],
            status_message => \@messages
        };
}

=head2 belarus_bank_change_warning_trigger

    Включение/выключение предупреждения об изменении банковских реквизитов для белорусов

=cut

sub belarus_bank_change_warning_trigger {
    my %O = @_;

    my $prop = Property->new($Client::SHOW_BELARUS_BANK_CHANGE_WARNING_PROPERTY_NAME);

    my $warning_enabled;
    if ($O{change_option}) {
        $warning_enabled = ($O{show_belarus_bank_change_warning}) ? 1 : 0;
        if ($warning_enabled) {
            $prop->set(1);
        } else {
            $prop->delete();
        }
    } else {
        $warning_enabled = $prop->get();
    }

    return {
        params => {
            show_belarus_bank_change_warning => $warning_enabled,
            # для снятого чекбокса ничего не передаётся, используем отдельное скрытое поле, чтобы различать первое открытие страницы и последующие отправки
            change_option => 1,
        },
    };
}

=head2 clients_features

    Включение/выключение фич на ClientID

=cut

sub clients_features
{
    my %params = @_;

    my $feature = $params{feature_id};

    if (!$feature){
        return {data => []};
    }

    # валидация $feature должна быть максимально строгой, т.к. ниже ее как есть подставляем в sql запросы
    die "bad feature '$feature'" unless $feature =~ /^[a-z0-9_]+\z/i;

    my @add_ClientIDs = split /[^0-9]+/, $params{add_ClientIDs} || '';
    my @remove_ClientIDs = split /[^0-9]+/, $params{remove_ClientIDs} || '';

    my $redirect;
    if ( @add_ClientIDs ){
        for my $shard (ppc_shards()) {
            my $filtered_ClientIDs = get_one_column_sql(PPC(shard=>$shard), [ "select ClientID from clients", where => {ClientID => \@add_ClientIDs} ]);
            my $to_insert = [ map { [$_, $feature, 1] } @$filtered_ClientIDs ];
            do_mass_insert_sql( 
                PPC(shard => $shard), 
                "INSERT INTO clients_custom_options (ClientID, keyname, value) VALUES %s ON DUPLICATE KEY UPDATE value=VALUES(value)", $to_insert
            );
        }
        $redirect = 1;
    }

    if ( @remove_ClientIDs ){
        for my $shard (ppc_shards()) {
            my $filtered_ClientIDs = get_one_column_sql(PPC(shard=>$shard), [ "select ClientID from clients", where => {ClientID => \@remove_ClientIDs} ]);
            do_delete_from_table(
                PPC(shard => $shard), 
                'clients_custom_options', 
                where => {ClientID => $filtered_ClientIDs, keyname => $feature} 
            );
        }
        $redirect = 1;
    }

    # Если данные менялись -- редиректим на страницу без доп. параметров для просмотра
    if ($redirect){
        return { redirect => "main.pl?cmd=internalReports&report_id=clients_features&ir_param_feature_id=$feature" };
    }

    # TODO аккуратное условне value <> 0
    my $total_count = overshard_sum(PPC(shard => 'all'), ["select count(*) from clients_custom_options", where => {keyname => $feature, value => 1}]);
    my $clients = get_all_sql(PPC(shard=>'all'), ["select ClientID as raw_ClientID, ClientID from clients_custom_options", where => {keyname => $feature, value => 1}, limit => 100]);
    my $i = 0;
    for my $cl (@$clients){
        $i++;
        $cl->{num} = "$i / $total_count";
    }
    return {
        data => $clients,
        params => { feature_id => $feature },
    }
}

=head2 get_mass_report_check_for_new_version

    Отдает хеш с относительным путем на новый отчет, если для отчета в базе есть проперти с именем нового:
    {old_report_id => new_version_path}
=cut

sub get_mass_report_check_for_new_version {
    my $data = get_all_sql(PPCDICT, [
            'SELECT name, value FROM ppc_properties',
            WHERE => { name__like => $REPORT_NEW_VERSION_PROPERTY_PREFIX . "%" }
        ]);

    my $prefixLength = length $REPORT_NEW_VERSION_PROPERTY_PREFIX;
    my %result = map {substr($_->{name}, $prefixLength) => $REPORT_NEW_VERSION_PATH_PREFIX.$_->{value}} @$data;
    return \%result;
}

=head2 get_report_check_for_new_version

    Отдает относительный путь на новый отчет, если для переданного отчета в базе есть проперти с именем нового
    иначе возвращает undef
=cut

sub get_report_check_for_new_version
{
    my ($report_id) = @_;
    my $prop = Property->new($REPORT_NEW_VERSION_PROPERTY_PREFIX.$report_id);
    my $prop_val = $prop->get();
    if ( ($prop_val // '') ne '') {
        return $REPORT_NEW_VERSION_PATH_PREFIX.$prop_val;
    }

    return undef;
}

=head2 clients_agencies_managers_report

    Выгрузка агентств и менеджеров по списку клиентов

=cut

sub clients_agencies_managers_report {
    my (%params) = @_;

    unless ($params{items}) {
        return {};
    }

    my $item_type = $params{item_type};
    unless ($item_type =~ /^(:?ClientID|uid|login)$/) {
        return {status_message => "неподдерживаемый тип $item_type"};
    }

    my $rbac = $params{c}->rbac;

    my $all_items = [ split(/[\s,]+/, $params{items} =~ s/^\s+//r) ];
    my $is_id = $item_type =~ /^(:?ClientID|uid)$/;
    my ($items, $filtered_items) = part { ($is_id && !is_valid_id($_)) ? 1 : 0 } @$all_items;

    my $item2client;
    my $item2user;
    for my $chunk_items (chunks($items, 500)) {
        my $chunk_item2client_id;
        my $chunk_item2user;

        if ($item_type =~ /^(:?uid|login)$/) {
            my $chunk_item2uid;
            if ($item_type eq 'login') {
                $chunk_item2uid = get_login2uid(login => $chunk_items);
            } else {
                $chunk_item2uid = { map { $_ => $_ } @$chunk_items };
            }
            my @chunk_uids = uniq values %$chunk_item2uid;
            my $users_by_id = Direct::Users->get_by(id => \@chunk_uids)->items_by('id');

            $chunk_items = [
                grep { $chunk_item2uid->{$_} && $users_by_id->{$chunk_item2uid->{$_}} }
                    @$chunk_items
            ];

            $chunk_item2user = {
                map { $_ => $users_by_id->{$chunk_item2uid->{$_}} } @$chunk_items
            };
            $chunk_item2client_id = {
                map { $_ => $chunk_item2user->{$_}->client_id() } @$chunk_items
            };
        } elsif ($item_type eq 'ClientID') {
            $chunk_item2client_id = { map { $_ => $_ } @$chunk_items };
        }

        my @client_ids = uniq values %$chunk_item2client_id;
        my $clients_col = Direct::Clients->get_by(id => \@client_ids);
        my $clients_by_id = $clients_col->items_by('id');
        my @clients = @{$clients_col->items()};
        $clients_col->enrich_with_agencies();

        my @agencies = map { @{$_->get_agencies()} } grep { $_->role() eq $Rbac::ROLE_CLIENT } @clients;
        my $clients_and_agencies_col = Direct::Clients->new([@clients, @agencies]);
        $clients_and_agencies_col->enrich_with_managers();

        my @managers = map { @{$_->managers()} }
            grep { $_->role() eq $Rbac::ROLE_CLIENT || $_->role() eq $Rbac::ROLE_AGENCY }
                @{$clients_and_agencies_col->items()};
        Direct::Clients->new([@clients, @agencies, @managers])->enrich_with_chief_reps();

        $chunk_items = [
            grep { $chunk_item2client_id->{$_} && $clients_by_id->{$chunk_item2client_id->{$_}} }
                @$chunk_items
        ];

        my $chunk_item2client = {
            map { $_ => $clients_by_id->{$chunk_item2client_id->{$_}} } @$chunk_items
        };
        if ($item_type eq 'ClientID') {
            $chunk_item2user ={ map { $_ => $chunk_item2client->{$_}->chief_rep() } @$chunk_items };
        }

        $item2client->{$_} = $chunk_item2client->{$_} for keys %$chunk_item2client;
        $item2user->{$_} = $chunk_item2user->{$_} for keys %$chunk_item2user;
    }

    my @data;
    my @missing_items;
    for my $item (@$items) {
        my $user = $item2user->{$item};
        unless ($user) {
            push @missing_items, $item;
            next;
        }
        my $client = $item2client->{$item};
        my @agencies;
        my @managers;
        if ($client->role() eq $Rbac::ROLE_CLIENT) {
            @agencies = @{$client->get_agencies()};
        }
        if ($client->role() eq $Rbac::ROLE_CLIENT || $client->role() eq $Rbac::ROLE_AGENCY) {
            @managers = (@{$client->managers()}, map { @{$_->managers()} } @agencies);
        }
        my @agency_logins = uniq sort map { $_->chief_rep()->login() } @agencies;
        my @manager_logins = grep { $_ ne 'yndx-geomanager' } uniq sort map { $_->chief_rep()->login() } @managers;
        my $login = $item_type eq 'login' ? $item : $user->login();

        push @data, {
            ClientID => $client->id(),
            uid => $user->id(),
            login => $login,
            role => $client->role(),
            agency_login => join(', ', @agency_logins),
            manager_login => join(', ', @manager_logins),
        };
    }

    my @messages;
    if ($filtered_items && @$filtered_items) {
        push @messages, 'Следующие строки были проигнорированы, так как их формат не соответствует типу, выбранному для списка: '.join(', ', @$filtered_items);
    }
    if (@missing_items) {
        push @messages, 'Следующие строки были проигнорированы, так как по ним не нашлось клиентов: '.join(', ', @missing_items);
    }

    return {
        data => \@data,
        status_message => join("<br>", @messages),
        to_xls => $params{to_xls},
        no_reports_links => 1,
        message_to_xls => 1,
    };
}


=head2 upload_banner_experiments

    Загрузка xls, xlsx файлов для экспериментов на СЕРПе и РСЯ

=cut

sub upload_banner_experiments {
    my %O = @_;

    # максимальная длина json-строки в, которую можем записать в базу
    my $EXPERIMENT_JSON_LIMIT = 65_535;

    my @messages;
    my $banners_for_update = {};

    unless ($O{input_file} || $O{delete_all_experiments}) {
        return {data => []};
    } elsif($O{delete_all_experiments}){
        my $bids_with_experiment = get_one_column_sql(PPC(shard => 'all'),
            "select ba.bid from additions_item_experiments aie
             join banners_additions ba on ba.additions_item_id = aie.additions_item_id");
        for my $bid (@$bids_with_experiment) {
            $banners_for_update->{$bid} = undef;
        }
    } else {
        my ($fh) = @O{qw/input_file/};
        my $file_name = "$fh";
        my ($file_type) = $file_name =~ /\.([^.]+)$/;
        my $data = [ {file => $file_name} ];

        my $content;
        my $xlsx_filename;
        read($fh, $content, (stat($fh))[7]);
        if ($file_type eq 'xlsx') {
            (undef, $xlsx_filename) = tempfile(undef, UNLINK => 1, OPEN => 0);
            write_file($xlsx_filename, {binmode => ':raw'}, $content);
        }

        my @worksheets;
        my $buff = $content;
        Encode::_utf8_off($buff);
        my $compressed_data = deflate($buff);
        if (length($compressed_data) > $Settings::MAX_COMPRESSED_XLS_SIZE) {
            push @messages, 'Ошибка: размер файла слишком велик';
        } elsif (any {$file_type eq $_} qw/xls xlsx/) {
            @worksheets = XLSParse::read_excel($file_type => $xlsx_filename || $content);
        } else {
            push @messages, 'Ошибка: неподдерживаемый формат файла';
        }
        unlink $xlsx_filename or warn "XLSX temporary file can't remove: $!" if $xlsx_filename;

        if (!@worksheets) {
            push @messages, 'Ошибка: не удалось загрузить файл или в нем нет листов';
        } else {
            my $xls_data = {};
            # Ключи в json должны быть упорядочены, т. к. будем считать хеш
            my $json = JSON->new->canonical([1]);
            my $sheet_number = 0;
            foreach my $worksheet (@worksheets) {
                if (ref $worksheet eq 'ARRAY' && @$worksheet) {
                    my $keys = shift @$worksheet;
                    $sheet_number++;
                    unless (ref $keys eq 'ARRAY' && @$keys) {
                        push @messages, "Ошибка: первая строка на странице $sheet_number не содержит названия колонок";
                        next;
                    }
                    unless (shift @$keys eq 'cid') {
                        push @messages, "Ошибка: первый столбец на странице $sheet_number должен называться cid и содержать номера кампаний";
                        next;
                    }
                    unless (shift @$keys eq 'bid') {
                        push @messages, "Ошибка: второй столбец на странице $sheet_number должен называться bid и содержать номера баннеров";
                        next;
                    }
                    if (any { $_ eq '' } @$keys) {
                        push @messages, "Ошибка: все столбцы на странице $sheet_number должны иметь названия";
                        next;
                    }
                    unless (scalar @$keys == scalar(uniq @$keys)) {
                        push @messages, "Ошибка: столбцы должны иметь уникальные названия на странице $sheet_number";
                        next;
                    }
                    
                    my @bad_href_keys;
                    foreach my $key (@$keys) {
                        if ($key =~ /^href_(.*)$/) {
                            my $experiment = $1;
                            unless ($experiment =~ /^\d+_.*$/) {
                                push @bad_href_keys, $key;
                            }
                        }
                    }
                    if (@bad_href_keys) {
                        push @messages, 'Ошибка: href-поля должны соответствовать маске "href_{ExperimentN}_{some_extra_useless_in_bs_info}", где {ExperimentN} - положительное число. Поля с ошибками: ' . join(", ", @bad_href_keys);
                        next;
                    }

                    my $row_cnt = 1;
                    foreach my $row (@$worksheet) {
                        $row_cnt++;
                        my $cid = shift @$row;
                        my $bid = shift @$row;
                        if (is_valid_id($cid) && is_valid_id($bid)) {
                            if ($O{delete_experiment}) {
                                $xls_data->{$cid}->{$bid} = undef;
                            } else {
                                my %bid_exp;
                                @bid_exp{@$keys} = @$row;
                                my $json_string = $json->encode({map { defined $bid_exp{$_} && $bid_exp{$_} ne '' ? ($_ => $bid_exp{$_}) : () } keys %bid_exp});
                                if (length $json_string <= $EXPERIMENT_JSON_LIMIT) {
                                    if (!exists $xls_data->{$cid}->{$bid}) {
                                        $xls_data->{$cid}->{$bid} = $json_string
                                    } else {
                                        push @messages, "Ошибка: cid=$cid bid=$bid не должны иметь дубликатов в пределах файла (страница $sheet_number, строка $row_cnt)";
                                    }
                                } else {
                                    delete $xls_data->{$cid}->{$bid};
                                    push @messages, "Ошибка: cid=$cid bid=$bid слишком много экспериментальных данных (страница $sheet_number, строка $row_cnt)";
                                }
                            }
                        } elsif (any { $_ ne '' } ($cid, $bid, @$row)) {
                            # показываем ошибку, только если в строке есть данные
                            push @messages, "Ошибка: cid=$cid bid=$bid невалидный номер кампании или баннера (страница $sheet_number, строка $row_cnt)";
                        }
                    }
                }
            }
            foreach my $cid (keys %$xls_data) {
                my $bids = { map { $_ => 1 } @{get_bids(cid => $cid)} };
                my $xls_bids = [keys %{$xls_data->{$cid}}];
                my ($exists_bids, $unexists_bids) = part { $bids->{$_} ? 0 : 1 } @$xls_bids;

                foreach my $bid (@$unexists_bids) {
                    push @messages, "Ошибка: cid=$cid bid=$bid баннер не найден в указанной кампании";
                }

                my $exists_banners = {};
                foreach my $bid (@$exists_bids) {
                    $exists_banners->{$bid} = $xls_data->{$cid}->{$bid};
                }
                hash_merge($banners_for_update, $exists_banners);
            }
        }
    }

    my $is_saved = 0;
    if (%$banners_for_update && (!@messages || $O{ignore_errors})) {
        Direct::BannersAdditions::save_banners_experiments($banners_for_update);
        $is_saved = 1;
        if ($O{delete_all_experiments}) {
            push @messages, 'OK: все эксперименты удалены';
        } elsif ($O{delete_experiment}) {
            push @messages, 'OK: эксперимент удален';
        } else {
            push @messages, 'OK: эксперимент добавлен';
        }
    } elsif (!@messages) {
        push @messages, 'Предупреждение: не найдено ни одного баннера';
    }

    return {
        data => $is_saved ? [map { {bid => $_, json => $banners_for_update->{$_}} } keys %$banners_for_update ] : [],
        status_message => join("<br>", @messages)
    }
}

=head2 mass_enable_freedom_for_clients

    Включение/выключение галочки свободы по списку клиентов

=cut

sub mass_enable_freedom_for_clients {
    my %O = @_;

    my $allow_create_scamp_by_subclient = $O{enable_freedom} ? 'Yes' : 'No';
    my $clients_data = {};

    unless ($O{ClientIDs}) {
        return {};
    } elsif (!$O{just_do_it}) {
        return {
            status_message => "Нужно внимательно прочитать и понять описание отчета"
        };
    } else {
        my $client_ids = [ split /[^0-9]+/, $O{ClientIDs} || '' ];

        for my $chunk (chunks $client_ids, 1000) {
            my $clients_data_chunk = mass_get_clients_data($chunk, [qw/ClientID role/]);
            for my $client_id (keys %$clients_data_chunk) {
                $clients_data_chunk->{$client_id}->{allow_create_scamp_by_subclient} = $allow_create_scamp_by_subclient;
                create_update_client({client_data => $clients_data_chunk->{$client_id}});
            }
            hash_merge $clients_data, $clients_data_chunk;
        }
    }

    return {
        data => %$clients_data ? [ map { {ClientID => $_, allow_freedom => $allow_create_scamp_by_subclient} } keys %$clients_data ] : [],
    }
}

=head2 lost_turbolanding_metrica_counters

    Просмотр кампаний с отсутствующими с camp_turbolanding_metrika_counters счетчиками турболендингов.
    Актуализация данных для заданных/всех найденных кампаний.

=cut

sub lost_turbolanding_metrica_counters {
    my %IN = @_;
    
    my @status;
    my $cids = get_num_array_by_str ($IN{cids});
    my $repair_all = $IN{repair_all};
    if (@$cids || $repair_all) {
        if ($repair_all) {
            $cids = [keys %{_get_broken_camps() // {}}];
        }

        if(@$cids){
            my $result = _repair_broken_camps($cids);
            if (@{$result->{errors}}){
                 push @status, iget('Ошибка: %s', $_) foreach @{$result->{errors}};
            }
            push @status, iget('Добавлено записей: %s.', $result->{info}->{inserted} // 0);
            push @status, iget('Удалено записей: %s.', $result->{info}->{deleted} // 0);
        }
        else{
            push @status, 'Не задано кампаний для восстановления!'
        }
    }
    
    push @status, '','Потерянные счетчики турболендингов';
   
    my $broken_camps = _get_broken_camps();
    return {
        data =>  [map { {cid => $_, b_count => $broken_camps->{$_}} } keys %$broken_camps],
        status_message => \@status,
    }
}

sub _get_broken_camps {
    my $main_query = 
            'FROM
                turbolandings t
                JOIN campaigns c ON (c.ClientID = t.ClientID)
                JOIN banners b ON (c.cid = b.cid)
                %s
                LEFT JOIN camp_turbolanding_metrika_counters cc ON (c.cid = cc.cid && b.bid = cc.bid)
             WHERE 
                cc.metrika_counter is null ';
    # Обычно турболендинг сайтлинка прописывается тем баннерам, у которых есть турболендинг и для основной ссылки
    # OR в условии сильно утяжеляет запрос, поэтому выберем в хеш все кампании с потерянными счетчиками у которых турболендинг указан для баннера
    # и добавим туда кампании с потерянными счетчиками турболендингов сайтлинков, если они все-таки не попали в первую выборку
    my $lost_camps = get_hash_sql(PPC(shard => 'all'),
            'SELECT c.cid, count(DISTINCT b.bid) as b_count '
            .sprintf($main_query, 'LEFT JOIN banner_turbolandings bt ON (bt.bid = b.bid)').
            'AND bt.tl_id is not null
            GROUP BY c.cid ORDER BY c.cid
            '
    );
    my $lost_stl = get_all_sql(PPC(shard => 'all'),
            'SELECT c.cid, count(DISTINCT b.bid) as b_count '
            .sprintf($main_query,
                'LEFT JOIN sitelinks_set_to_link s2l ON (b.sitelinks_set_id = s2l.sitelinks_set_id)
                 LEFT JOIN sitelinks_links sl ON (sl.sl_id = s2l.sl_id)'
            ).
            'AND sl.tl_id is not null
            GROUP BY c.cid ORDER BY c.cid');

    foreach my $row (@$lost_stl) {
        #Число баннеров с потерянными счетчиками на кампанию - оценочное значение,
        #поэтому игнорируем маловероятную комбинацию "счетчики сайтлинков и баннеров потеряны для несовпадающих групп банеров"
        next if $lost_camps->{$row->{cid}};
        $lost_camps->{$row->{cid}} = $row->{b_count}
    }

    return $lost_camps;
}

sub _repair_broken_camps {
    my ($cids) = @_;

    my (%info, @errors);
    my $camp_turbolandings = get_all_sql(PPC(cid => $cids),
        ['SELECT DISTINCT b.cid, b.bid, bt.tl_id as btl, sl.tl_id as stl
            FROM
                banners b 
                LEFT JOIN banner_turbolandings bt ON (bt.bid = b.bid)
                LEFT JOIN sitelinks_set_to_link s2l ON (b.sitelinks_set_id = s2l.sitelinks_set_id)
                LEFT JOIN sitelinks_links sl ON (sl.sl_id = s2l.sl_id)
            ',
            WHERE => {'b.cid' => SHARD_IDS, _OR => {'bt.tl_id__is_not_null' => 1, 'sl.tl_id__is_not_null' => 1}}
        ]
    );

    my (%counters_by_tl_id, $tl_by_cid_and_bid);
    foreach my $row (@$camp_turbolandings) {
        foreach my $field (qw/btl stl/) {
            next unless defined $row->{$field};
            next if $tl_by_cid_and_bid->{$row->{cid}}->{$row->{bid}}->{$row->{$field}};
            $counters_by_tl_id{$row->{$field}} //= [];
            $tl_by_cid_and_bid->{$row->{cid}}->{$row->{bid}}->{$row->{$field}} = $counters_by_tl_id{$row->{$field}};
        }
    }

    undef $camp_turbolandings;

    my $turbolandings_info = get_hash_sql(PPC(shard => 'all'), 
            ['SELECT tl_id, metrika_counters_json FROM turbolandings',
                WHERE => {tl_id => [keys %counters_by_tl_id]}
            ]);
    while (my ($id, $counters_json) = each %$turbolandings_info ){
        my $counters;
        eval {$counters = from_json($counters_json)};
        push @errors, iget('Неверная структра json для счетчиков турболендинга %s : %s', $id, $counters_json)
            unless $counters && @$counters;
        push @{$counters_by_tl_id{$id}}, $_->{id} foreach @$counters;
    }
    undef $turbolandings_info;

    my $existing_records = get_all_sql(PPC(cid => $cids),
        ['SELECT cid, bid, metrika_counter FROM camp_turbolanding_metrika_counters',
        WHERE => {cid => SHARD_IDS},
        ]
    );

    #Строим индекс существующих счетчиков;
    my $existing_counters;

    foreach my $row (@$existing_records) {
        $existing_counters->{$row->{cid}}->{$row->{bid}}->{$row->{metrika_counter}} = 1;
    }
    undef $existing_records;
    $info{inserted} = 0;

    foreach my $cid (keys %$tl_by_cid_and_bid){
        foreach my $bid (keys %{$tl_by_cid_and_bid->{$cid}}) {
            foreach my $counter (map {@$_} values  %{$tl_by_cid_and_bid->{$cid}->{$bid}}){
                unless ($existing_counters->{$cid}->{$bid}->{$counter}){
                    $info{inserted}++;
                    do_insert_into_table(PPC(cid => $cid), camp_turbolanding_metrika_counters => {cid => $cid, bid => $bid, metrika_counter => $counter} );
                    $existing_counters->{$cid}->{$bid}->{$counter} = 1;
                }
                $existing_counters->{$cid}->{$bid}->{$counter}++;
            }
        }
    }

    #Если после обработки tl_by_cid_and_bid в existing_records что-то осталось - это записи не привязанных счетчиков, удалим их

    $info{deleted} = 0;
    if (keys %$existing_counters){
        foreach my $cid (keys %$existing_counters){
            foreach my $bid (keys %{$existing_counters->{$cid}}){
                foreach my $counter (keys %{$existing_counters->{$cid}->{$bid}}){
                    #Если значение в $existing_counter > 1, значит запись проходилась при обработке tl_by_cid_and_bid, т.е. связка живая
                    next if $existing_counters->{$cid}->{$bid}->{$counter} > 1;
                    #Если прохода не было - такой связки нет, запись удаляем
                    $info{deleted}++;
                    do_delete_from_table(PPC(cid => $cid), 'camp_turbolanding_metrika_counters', where => {cid => $cid, bid => $bid, metrika_counter => $counter});
                }
            }
        }
    }

    return {info => \%info, errors => \@errors};
}

=head2 preprod_limtest

    Переключение limtest с/на preprod БК

=cut

sub preprod_limtest {
    my %O = @_;

    my $res = {params => {}};

    my $param_prefix = "use_preprod_limtest_";
    my @limtest_nums = (1, 2);

    if ($O{change_option}) {

        foreach my $i (@limtest_nums) {
            my $param_name = $param_prefix.$i;
            my $param_value = $O{$param_name} // 0;

            my $prop = Property->new(BS::Export::PREPROD_LIMTEST_PROP_PREFIX.$i);

            if (($prop->get() // 0) != $param_value) {
                $prop->set($param_value);
            }

            $res->{params}{$param_name} = $param_value;
        }
    } else {
        # просто загрузка странички
        foreach my $i (@limtest_nums) {
            my $prop = Property->new(BS::Export::PREPROD_LIMTEST_PROP_PREFIX.$i);
            $res->{params}{$param_prefix.$i} = $prop->get() // 0;
        }
    }

    return $res;
}

=head2 isize_props

    Изменение коэффициентов для вычисления интегрального размера очереди

=cut

sub isize_props {
    my %O = @_;

    my $res = {params => {}};

    foreach my $prop_name ((
        $BS::ExportMaster::ISIZE_CAMPS_NUM_COEF_PROP_NAME,
        $BS::ExportMaster::ISIZE_BANNERS_NUM_COEF_PROP_NAME,
        $BS::ExportMaster::ISIZE_CONTEXTS_NUM_COEF_PROP_NAME,
        $BS::ExportMaster::ISIZE_BIDS_NUM_COEF_PROP_NAME,
        $BS::ExportMaster::ISIZE_PRICES_NUM_COEF_PROP_NAME)) {

        my $prop = Property->new($prop_name);

        if ($O{change_option}
            && defined($O{$prop_name})
            && is_valid_float($O{$prop_name})
            && $O{$prop_name} >= 0) {

            $prop->set($O{$prop_name} + 0);
        }
        $res->{params}{$prop_name} = $prop->get();
    }

    return $res;
}

=head2 create_billing_aggregates

    Создание биллинговых агрегатов клиентам по списку.

=cut
sub create_billing_aggregates {
    my %O = @_;

    my $res = {params => {}};

    my $target_product_type = $O{product_type};
    my $operator_uid = $O{c}->UID;

    my $rows = eval { _text_to_login_and_client_id($O{clients}, 100); };
    if ($@) {
        my $err = $@;
        return {
            status_message => ref $err eq 'SCALAR' ? $$err : $err,
        }
    }


    my @get_wallet_cid = grep { !$_->{wallet_cid} && !$_->{result} } @$rows;
    if (@get_wallet_cid) {
        my @client_ids = map { $_->{client_id} } @get_wallet_cid;
        my $client_info = get_hashes_hash_sql(PPC(ClientID => \@client_ids), ["
            SELECT c.ClientID
                , COUNT(DISTINCT IF(c.AgencyID > 0, c.AgencyID, NULL)) as agencies
                , COUNT(IF(c.AgencyID = 0, 1, NULL)) as non_agency_camps
            FROM campaigns c",
            WHERE => [
                'c.ClientID' => SHARD_IDS,
                'c.statusEmpty' => 'No',
            ], "
            GROUP BY c.ClientID
        "]);

        for my $row (@get_wallet_cid) {
            my $ci = $client_info->{$row->{client_id}};
            if ($ci->{agencies} > 0 && $ci->{non_agency_camps} > 0
                || $ci->{agencies} > 1
            ) {
                $row->{result} = "у клиента несколько видов сервисирования";
            }
        }

        @get_wallet_cid = grep { !$_->{result} } @get_wallet_cid;
        @client_ids = map { $_->{client_id} } @get_wallet_cid;
        $client_info = get_hashes_hash_sql(PPC(ClientID => \@client_ids), ["
            SELECT c.ClientID
                , COUNT(DISTINCT c.wallet_cid) as wallets_count
                , MIN(c.wallet_cid) as wallet_cid
            FROM campaigns c",
            WHERE => [
                'c.ClientID' => SHARD_IDS,
                'c.type' => get_camp_kind_types("web_edit_base"),
                'c.statusEmpty' => 'No',
            ], "
            GROUP BY c.ClientID
        "]);

        for my $row (@get_wallet_cid) {
            my $ci = $client_info->{$row->{client_id}};
            if ($ci->{wallets_count} != 1) {
                $row->{result} = "не удалось определить ОС";
            } else {
                $row->{wallet_cid} = $ci->{wallet_cid};
            }
        }
    }

    my @check_ba = grep { !$_->{result} } @$rows;
    if (@check_ba) {
        my @client_ids = map { $_->{client_id} } @check_ba;

        my $existing_aggs = Direct::BillingAggregates->get_by(client_id => \@client_ids)->items_by_wallet_and_product();
        for my $row (@check_ba) {
            if ($existing_aggs->{$row->{wallet_cid}} && $existing_aggs->{$row->{wallet_cid}}{$target_product_type}) {
                $row->{result} = "биллинговый агрегат для $target_product_type уже существует";
            }
        }
    }

    my @create_ba = grep { !$_->{result} } @$rows;
    if (@create_ba) {
        for my $row (@create_ba) {
            my $wallet = Direct::Wallets->get_by(campaign_id => $row->{wallet_cid}, no_additional => 1)->items->[0];
            my $ba = Direct::BillingAggregates->make_new_aggregates_for_client($row->{client_id}, [$target_product_type], $wallet);
            my $camp_types_to_resync = Direct::BillingAggregates::get_relevant_camp_types_by_product_types([$target_product_type]);
            my $success = eval {
                $ba->create($operator_uid);
                BillingAggregateTools::_bs_resync_camps_with_types($row->{wallet_cid}, $camp_types_to_resync);
                return 1;
            };
            my $err = $@;
            if (!$success) {
                LogTools::log_messages('create_billing_aggregates', "Error creating billing aggregate: $err");
                $row->{result} = "Ошибка: ".substr($err, 0, 1_000);
            } else {
                $row->{result} = "OK";
            }
        }
    }

    return {
        data =>  [map {
            {
                ClientID => $_->{client_id} // '?',
                login => $_->{login} // '?',
                result => $_->{result} // '?',
            }
        } @$rows],
        status_message => '',
    }
}

sub _text_to_login_and_client_id {
    my ($text, $max_rows) = @_;

    my @rows;
    my @lines = split(/\n/, $text // '');
    for my $line (@lines) {
        chomp($line);
        $line =~ s/(^\s+|\s+$)//g;
        next unless $line;
        my @parts = split(/\s+/, $line);
        my $row = {};
        if ($parts[0] =~ /^\d+$/) {
            $row->{client_id} = $parts[0];
        } else {
            $row->{login} = $parts[0];
        }
        if ($parts[1] && $parts[1] =~ /^\d+$/) {
            $row->{wallet_cid} = $parts[1];
        }
        push @rows, $row;
    }
    if (@rows > $max_rows) {
        die \'слишком много клиентов';
    }

    my @get_logins = grep { !$_->{login} } @rows;
    if (@get_logins) {
        my @client_ids = map { $_->{client_id} } @get_logins;
        my $client2chief = Rbac::get_chiefs_multi(ClientID => \@client_ids, role => 'client');
        my $uid2login = get_uid2login(uid => [values %$client2chief]);

        for my $row (@get_logins) {
            my $chief = $client2chief->{$row->{client_id}};
            if (!$chief) {
                $row->{result} = 'клиент не найден';
                next;
            }
            my $login = $uid2login->{$chief};
            if (!$login) {
                $row->{result} = 'клиент не найден';
                next;
            }
            $row->{login} = $login;
        }
    }

    my @get_client_ids = grep { !$_->{client_id} } @rows;
    if (@get_client_ids) {
        my @logins = map { $_->{login} } @get_client_ids;
        my $login2client_id = get_login2clientid(login => \@logins);

        for my $row (@get_client_ids) {
            my $client_id = $login2client_id->{$row->{login}};
            if (!$client_id) {
                $row->{result} = 'клиент не найден';
            }
            $row->{client_id} = $client_id;
        }
    }

    return \@rows;
}

=head2 create_billing_aggregates_script_mgr

    Управление скриптом для создания Биллинговых Агрегатов

=cut
sub create_billing_aggregates_script_mgr {
    my (%O) = @_;

    if ($O{clients}) {
        my $rows = eval { _text_to_login_and_client_id($O{clients}, 1000); };
        if ($@) {
            my $err = $@;
            return {
                status_message => (ref $err eq 'SCALAR' ? $$err : $err),
            };
        }
        # игнорим строки с несуществующими клиентами
        my @client_ids = map { $_->{client_id } } grep { !$_->{result} } @$rows;

        my %shard2client_ids;
        foreach_shard ClientID => \@client_ids, sub {
            my ($shard, $client_id_chunk) = @_;

            $shard2client_ids{$shard} = $client_id_chunk;
        };

        for my $shard (ppc_shards()) {
            my $task_prop = Property->new("PPC_CREATE_BILLING_AGGREGATES_SHARD_".$shard);
            $task_prop->set(to_json({client_id_list => $shard2client_ids{$shard} // [], finished => 0}, {canonical => 1}));
        }

        return {
            redirect => "main.pl?cmd=internalReports&report_id=create_billing_aggregates_script_mgr",
        };
    } elsif (defined $O{client_id_from} && defined $O{client_id_to}) {
        if (!is_valid_id($O{client_id_from}) || !is_valid_id($O{client_id_to})) {
            return {
                status_message => 'поля ClientID от и до должны быть положительными целыми числами',
            };
        }
        my $task = {
            client_id_from => $O{client_id_from},
            client_id_to => $O{client_id_to},
        };
        if ($O{has_camp_type} ne '---') {
            $task->{with_camp_types} = [$O{has_camp_type}];
        }

        for my $shard (ppc_shards()) {
            my $task_prop = Property->new("PPC_CREATE_BILLING_AGGREGATES_SHARD_".$shard);
            $task_prop->set(to_json($task, {canonical => 1}));
        }

        return {
            redirect => "main.pl?cmd=internalReports&report_id=create_billing_aggregates_script_mgr",
        };
    }

    my @data;
    for my $shard (ppc_shards()) {
        my $task_prop = Property->new("PPC_CREATE_BILLING_AGGREGATES_SHARD_".$shard);
        push @data, {shard => $shard, task => $task_prop->get()};
    }
    return {
        data => \@data,
    };
}


=head2 archive_campaigns_by_type

    Архивация кампаний геопродуктовых/сделок/продвижения контента/медийных без фич

=cut

use constant REQUIRED_FEATURE_BY_CAMPAIGN_TYPE => { cpm_geoproduct => 'cpm_geoproduct_enabled',
    cpm_deals => 'cpm_deals',
    content_promotion_video => 'content_promotion_video',
    content_promotion_collection => 'content_promotion_collection',
    cpm_audio => 'cpm_audio_groups_edit_for_dna',
    cpm_indoor => 'cpm_indoor_groups_edit_for_dna',
    cpm_outdoor => 'cpm_outdoor_groups_edit_for_dna' };

sub archive_campaigns_by_type {
    my %O = @_;
    my @data;

    my $campaign_type = $O{campaign_type};
    my $item_type = $O{item_type};
    my @ids = uniq @{get_num_array_by_str($O{ids})};
    my $all_except = $O{all_except};
    my $archive_non_stopped = $O{archive_non_stopped};

    unless ($item_type && $campaign_type) {
        return {};
    }

    unless ($campaign_type =~ /^(:?cpm_geoproduct|cpm_deals|content_promotion_video|content_promotion_collection|cpm_audio|cpm_indoor|cpm_outdoor)$/) {
        return {status_message => "неподдерживаемый тип $campaign_type"};
    }

    unless ($item_type =~ /^(:?cid|ClientID)$/) {
        return {status_message => "неподдерживаемый тип $item_type"};
    }

    my $cids_to_archive;
    if ($all_except) {
        if ($campaign_type eq 'cpm_geoproduct') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct p.cid
                from phrases p
                    join campaigns c using (cid)
                where p.adgroup_type in ('cpm_geoproduct', 'cpm_geo_pin')
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'cpm_banner')");
        } elsif ($campaign_type eq 'cpm_deals') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct cid
                from campaigns
                where type = 'cpm_deals'
                    and archived = 'No'");
        } elsif ($campaign_type eq 'content_promotion_video') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct c.cid
                from phrases p
                    left join adgroups_content_promotion acp using (pid)
                    join campaigns c using (cid)
                where p.adgroup_type = 'content_promotion'
                    and acp.content_promotion_type = 'video'
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'content_promotion')");
        } elsif ($campaign_type eq 'content_promotion_collection') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct c.cid
                from phrases p
                    left join adgroups_content_promotion acp using (pid)
                    join campaigns c using (cid)
                where p.adgroup_type = 'content_promotion'
                    and acp.content_promotion_type = 'collection'
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'content_promotion')");
        } elsif ($campaign_type eq 'cpm_audio') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct p.cid
                from phrases p
                    join campaigns c using (cid)
                where p.adgroup_type = 'cpm_audio'
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'cpm_banner')
                    and not exists (
                        select p2.pid
                        from phrases p2 
                        where p.cid = p2.cid
                            and p2.adgroup_type <> 'cpm_audio'
                    )");
        } elsif ($campaign_type eq 'cpm_indoor') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct p.cid
                from phrases p
                    join campaigns c using (cid)
                where p.adgroup_type = 'cpm_indoor'
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'cpm_banner')
                    and not exists (
                        select p2.pid
                        from phrases p2 
                        where p.cid = p2.cid
                            and p2.adgroup_type <> 'cpm_indoor'
                    )");
        } elsif ($campaign_type eq 'cpm_outdoor') {
            $cids_to_archive = get_one_column_sql(PPC(shard => 'all'),
                "select distinct p.cid
                from phrases p
                    join campaigns c using (cid)
                where p.adgroup_type = 'cpm_outdoor'
                    and c.archived = 'No'
                    and p.cid in (select c.cid from campaigns c where type = 'cpm_banner')
                    and not exists (
                        select p2.pid
                        from phrases p2 
                        where p.cid = p2.cid
                            and p2.adgroup_type <> 'cpm_outdoor'
                    )");
        }
    } else {
        if ($item_type eq 'ClientID') {
            my $cids = get_cids(ClientID => \@ids);
            $cids_to_archive = get_cids_to_archive($campaign_type, $cids);
        } elsif ($item_type eq 'cid') {
            $cids_to_archive = get_cids_to_archive($campaign_type, \@ids);
        }
    }

    my $client_ids_by_cid = get_cid2clientid(cid => $cids_to_archive);

    if ($all_except) {
        if ($item_type eq 'ClientID') {
            my %except_client_ids = map { $_ => 1 } @ids;
            $cids_to_archive = [ grep {
                exists($client_ids_by_cid->{$_}) && !exists($except_client_ids{$client_ids_by_cid->{$_}})
            } @$cids_to_archive ];
        } elsif ($item_type eq 'cid') {
            my %except_cids = map { $_ => 1 } @ids;
            $cids_to_archive = [ grep { !exists($except_cids{$_}) } @$cids_to_archive ];
        }
    }

    my $feature_name = REQUIRED_FEATURE_BY_CAMPAIGN_TYPE->{$campaign_type};
    my $has_feature = {};
    my @filtered_client_ids = @{$client_ids_by_cid}{@$cids_to_archive};
    foreach my $client_ids_chunk (chunks(\@filtered_client_ids, 10)) {
        hash_merge($has_feature, Client::ClientFeatures::_is_feature_allowed_for_client_ids($client_ids_chunk, $feature_name));
    }

    my @cids_without_feature = grep { !$has_feature->{$client_ids_by_cid->{$_}} } @$cids_to_archive;

    if (@cids_without_feature) {
        my $uids = get_cid2uid(cid => \@cids_without_feature);

        for my $cid(@cids_without_feature) {
            my $uid = $uids->{$cid};

            my ($result, $error) = Common::_arc_camp($uid, $cid, archive_non_stopped => $archive_non_stopped, archived_is_error => 1);
            push @data, {
                uid => $uid,
                cid => $cid,
                result => $result ? 'Success' : 'Failed',
                error => $error
            };
        }
    }

    return { data => \@data };
}

=head2 get_cids_to_archive($campaign_type, $cids)
    Получить из списка id кампаний только соответствующие переданному типу
    Параметры:
        $campaign_type -- тип кампании
        $cids -- список id кампаний
    Результат:
        массив id-ников кампаний переданного типа
=cut

sub get_cids_to_archive {
    my ($campaign_type, $cids) = @_;
    my $cids_to_archive;
    if ($campaign_type eq 'cpm_geoproduct') {
        $cids_to_archive = Campaign::get_geoproduct_campaigns($cids);
    } elsif ($campaign_type eq 'cpm_deals') {
        my $campaign_type_by_cid = Campaign::Types::get_camp_type_multi(cid => $cids);
        $cids_to_archive = grep { $campaign_type_by_cid->{$_} eq 'cpm_deals' } @$cids;
    } elsif ($campaign_type eq 'content_promotion_video'
            || $campaign_type eq 'content_promotion_collection') {
        my $campaign_type_by_cid = Campaign::Types::get_camp_type_multi(cid => $cids);
        my @content_promotion_cids = grep { $campaign_type_by_cid->{$_} eq 'content_promotion' } @$cids;
        my $content_promotion_types = CampaignTools::mass_get_content_promotion_content_type(\@content_promotion_cids);
        my $content_promotion_type_value = $campaign_type eq 'content_promotion_video' ? 'video' : 'collection';
        $cids_to_archive = grep { $content_promotion_types->{$_} eq $content_promotion_type_value } @content_promotion_cids;
    } elsif ($campaign_type eq 'cpm_audio'
            || $campaign_type eq 'cpm_indoor'
            || $campaign_type eq 'cpm_outdoor') {
        $cids_to_archive = Campaign::get_campaign_ids_with_only_one_group_type($campaign_type, $cids);
    }
    return $cids_to_archive;
}


=head2 modify_has_turbo_smarts

    Включение/выключение турбированности смарт-кампаний

=cut

sub modify_has_turbo_smarts {
    my %O = @_;


    my $cids = get_num_array_by_str($O{cids});
    my $has_turbo_smarts = $O{has_turbo_smarts} // '';
    my $is_turbo_smarts_changed = $has_turbo_smarts =~/[01]/ ? 1 : 0;

    my $res = {params => {
        has_turbo_smarts => $has_turbo_smarts,
        cids => $O{cids} 
    }};

    $has_turbo_smarts = $has_turbo_smarts ? 1 : 0;
    if (@$cids) {
        my $campaigns = get_all_sql(PPC(cid => $cids),
            ['
                SELECT cid, ClientID, type, name as title, FIND_IN_SET(opts, "has_turbo_smarts") as has_turbo_smarts
                FROM campaigns
                ', where => {cid => SHARD_IDS}]
            );
        my (@data, @wrong_campaigns, %wrong_clients);

        foreach my $camp (@$campaigns) {
            if ($camp->{type} ne 'performance'){
                push @wrong_campaigns, $camp->{cid};
            }
            elsif(!Client::ClientFeatures::has_turbo_smarts($camp->{ClientID})){
                $wrong_clients{$camp->{ClientID}} //= 1;
            }
            else{
                if ($is_turbo_smarts_changed){
                    $camp->{has_turbo_smarts} =  $has_turbo_smarts;
                }
                push @data, $camp;
            }
        }
        
        $res->{data} = \@data;

        if ($is_turbo_smarts_changed && @data){
            do_update_table(PPC(cid => [map {$_->{cid}} @data]),
                    campaigns => {opts__smod => {has_turbo_smarts => $has_turbo_smarts} },
                    where => {cid => SHARD_IDS});
        }

        $res->{status_message} = '<br>Неверный тип кампании, cid:'.join(', ', @wrong_campaigns)
            if (@wrong_campaigns);
        $res->{status_message} .= '<br>Для клиента не включена фича turbo_smarts, ClientId:'.join(', ', keys %wrong_clients)
            if  %wrong_clients;
        $res->{status_message} = '<span style="color:red">'.$res->{status_message}.'</span>'
            if $res->{status_message};
    }

    return $res;
}

=head2 enrich_data_for_show_dna_pb

    Управление режимом обогащения данных шаблонов в shwDnaPb

=cut

sub enrich_data_for_show_dna_pb {
    my %O = @_;

    my $prop = Property->new('ENRICH_DATA_AT_SHOW_DNA_PB');

    my $mode = $O{mode};
    if ($mode) {
	if ($mode eq 'none'){
	    $prop->delete();
	}
	elsif ($mode eq 'by_flag') {
	    $prop->set('by_flag');
	}
	elsif ($mode eq 'enabled') {
	    $prop->set(1);
	}
    }
    my $current_mode = $prop->get();
    $current_mode = 'none' if !$current_mode;
    $current_mode = 'enabled' if $current_mode eq '1';

    return {
        params => {
            mode => $current_mode,
        },
    }
}

=head2 change_is_autovideo_allowed

    Просмотр и управление флагом is_autovideo_allowed на кампаниях
    
=cut

sub change_is_autovideo_allowed {
    my %IN = @_;
    
    my @status;
    my $cids = get_num_array_by_str($IN{cid});
    my $mode = $IN{action_type};
    if (@$cids && defined $mode && $mode != 0) {
        foreach my $cid (@$cids) {
            push @status, 'Ошибка: неверный cid' unless is_valid_id($cid);
        }
        unless (@status) {
            my $camp_type_by_cid = get_hash_sql(PPC(cid => $cids),
            ['SELECT cid,type FROM campaigns', WHERE => {cid => $cids}]);
            foreach my $cid (keys %$camp_type_by_cid) {
                push @status, sprintf('Ошибка: неверный тип кампании %s - %s ', $cid, $camp_type_by_cid->{$cid})
                    unless $camp_type_by_cid->{$cid} eq 'performance';
            }
        }
        unless (@status) {
             my $ok = do_update_table(PPC(cid => $cids), 'campaigns', {opts__smod => {is_auto_video_allowed => $mode > 0 ? 1 : 0}},
                             where => {cid => $cids});
             push @status, $ok ?
                 'Флаг is_auto_video_allowed успешно '.($mode > 0 ? 'установлен' : 'сброшен').' для '.($ok > 1 ? "$ok кампаний." : "кампании $cids->[0]")
               : 'Не удалось изменить значение флага';
        }
    }
    
    push @status, '','Кампании с установленным флагом is_auto_video_allowed';
    my $campaigns = get_all_sql(PPC(shard => 'all'),
        'SELECT cid, ClientID, type, name FROM campaigns WHERE FIND_IN_SET("is_auto_video_allowed", opts) > 0 ORDER BY cid',
    );
    
    return {
        data =>  $campaigns,
        status_message => \@status,
    }
}

=head2 change_passport_karma_lock_state

    Установка/сброс блокировки пользователя по карме паспорта
    
=cut

sub change_passport_karma_lock_state {
    my %IN = @_;
   
    my @status;
    my $uid = $IN{uid};
    my $mode = $IN{action_type};
    my $operator_uid = $IN{c}->UID();
    
    my $login;
    if (defined $uid) {
        push @status, 'Ошибка: неверный uid' if !is_valid_id($uid);

        push @status, "Ошибка: uid:$uid недоступен текущему оператору" if !rbac_is_owner(undef, $operator_uid, $uid);
    
        $login = get_login(uid => $uid) if !@status;
        push @status, 'Ошибка: пользователь не найден' unless $login;
    }
    
    my $data;
    if (defined $mode && !@status && $uid) {
        if ($mode != 0) {
            my $is_bad_passport_karma = User::is_bad_passport_karma($uid);
            if ($is_bad_passport_karma && $mode == -1) {
                #Разблокируем
                User::create_update_user($uid, {passport_karma => -1});
                do_delete_from_table(PPCDICT, 'bad_karma_clients', where => { uid => $uid });
                push @status, "Пользователь $login разблокирован";
            } elsif (!$is_bad_passport_karma && $mode == 1) {
                #Блокируем, +21 - чтобы поставить то же значение, что в Staff::save_edited_user (101)
                User::create_update_user($uid, {passport_karma => $Settings::KARMA_API_AUTOBLOCK + 21});
                push @status, "Пользователь $login заблокирован";
            }
        }
        my $is_bad_passport_karma = User::is_bad_passport_karma($uid);

        $data = {login => $login, uid => $uid, karma_lock => $is_bad_passport_karma ? "заблокирован" : "нет"};
    }
   
    return {
        data => [$data],
        status_message => \@status,
    }
}

1;
