use warnings;
use strict;

=head NAME

IpTools -- простые функции для работы с ip-адресами и сетями. 

=head1 DESCRIPTION

Специфика Директа должна ограничиваться только знанием о внутренних сетях.

Зависеть может от: 
  * Settings (можно избавиться, втащив путь к файлу внутр. сетей сюда)
  * внешних модулей для работы с адресами и сетями

=cut

package IpTools;

use base qw/Exporter/;

use List::Util qw/any/;
use Socket qw//;
use Net::Patricia;
use Net::IPv6Addr;

use Settings;

our @EXPORT = qw(
    is_ip_in_list
    is_internal_ip
    get_first_non_private_ip
    my_aton
    is_ip_in_networks
    validate_networks
);

our @EXPORT_OK = qw(
    get_hostname_fcrdns
);

our $IPRE = qr/(?:\d{1,3}\.){3}\d{1,3}/;

=head2 is_ip_in_list

** Функция скопирована в direct-utils/Plack/Middleware/ACL.pm с точностью до замены print STDERR на запись в лог. **
** Если нужно тут что-то поправить, посмотрите, нужно ли это сделать в ACL.pm **

Параметры: 
1. IP-адрес (строка)
2. Файл с IP-адресами и масками сетей (полный путь)

Проверяет, записан ли данный адрес в файле (явно или через маску сети)

Для каждого файла создаем 2 объекта Net::Patricia, все эти объекты и времена модификации файлов
храним в хеше:
%patricia_cache = (
   'имя файла' => {
                   mtime => <время изменения файла>,
                   checkers => [<проверяющая функция>, ...],
                  },
   .....
);

Если с последнего обращения время модификации файла изменилось -- перечитываем файл.

ipv6-ready: да; обрабатывает сети и адреса
hbf-ready: да

=cut

{
    
    my %patricia_cache;

sub is_ip_in_list
{
    my ($ip_addr, $ip_file_name) = @_;
    
    #проверяем, что для файла $ip_file_name уже есть запись. Если ее нет -- делаем запись с mtime=0
    my $check = $patricia_cache{$ip_file_name} //= {mtime => 0};

    my $cur_mtime = (stat($ip_file_name))[9];

    if ($cur_mtime > $check->{mtime} 
        || !defined $check->{checkers}
    ) {
        # разбираем файл, записываем в Patricia'вские объекты: отдельно ipv4, отдельно ipv6 (Patricia работает отдельно с теми и другими)
        # history: в качестве альтернатив предлагались: NetAddr::IP[::Lite] (lena-san) и Net::CIDR[::Lite] (pankovpv) 
        my $patricia = new Net::Patricia;
        my $patricia_v6 = new Net::Patricia AF_INET6;
        my @checkers = (
            sub {scalar $patricia->match_string(shift)}, 
            sub {scalar $patricia_v6->match_string(shift)},
            );
        open my $fh, "<", $ip_file_name or die "Error: $!";
        while (my $network = <$fh>) {
            $network =~ s/#.*$//; # remove comments
            $network =~ s/\s+//g; # remove spaces
            next if length($network) == 0;
            if ($network =~ /^ \d{1,3} (\.\d{1,3}){3} ( \/ \d+ )? $/x) {
                # похоже на ipv4
                eval {
                    $patricia->add_string($network, 1);
                };
                if ($@) {
                    print STDERR "ERROR in parse $ip_file_name: \"$network\"\n$@\n";
                }
            } elsif ( $network =~ /^ [0-9a-f:]+ (\/[0-9]+)? $/x  ) {
                # похоже на ipv6
                eval {
                    $patricia_v6->add_string($network, 1);
                };
                if ($@){
                    print STDERR "ERROR in parse $ip_file_name: (ipv6) \"$network\"\n$@\n";
                }
            } elsif ( my ($proj, $net) = $network =~ /^ ([0-9a-f]+) @ ([0-9a-f:]+ (\/[0-9]+)?) $/x  ) {
                # похоже на host-based-firewall
                eval {
                    my $proj_id = hex $proj;
                    if ($proj_id == 0 || $proj_id >= 2 ** 32) {
                        die "Incorrect proj: $proj";
                    }
                    my $p = new Net::Patricia AF_INET6;
                    $p->add_string($net, 1);
                    push @checkers, sub { my $ip = shift; return $p->match_string($ip) && _hbf_proj_id($ip) == $proj_id; };
                };
                if ($@){
                    print STDERR "ERROR in parse $ip_file_name: (ipv6) \"$network\"\n$@\n";
                }
            } else {
                # ни на что не похоже
                print STDERR "ERROR in parse $ip_file_name: \"$network\"\n";
            }
        }
        close $fh;
        $check->{checkers} = \@checkers;
        $check->{mtime} = $cur_mtime;
    }

    return scalar any {eval {$_->($ip_addr)}} @{$check->{checkers}};
}
}

# описание формата ip https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/
sub _hbf_proj_id {
    my $ip = shift;
    my @parts = eval { Net::IPv6Addr->new($ip)->to_array(); };
    return undef unless @parts;
    return hex("$parts[4]$parts[5]");
}

=head2 is_internal_ip

    Определяет входит ли переданный IP-адрес в список внутренних сетей. Адрес передаётся в строковом представлении.

    $is_internal_ip = is_internal_ip('95.108.173.92'); # $is_internal_ip => 1|0
                      is_internal_ip('2a02:6b8:0:1495::ae'); 

    ipv6-ready: да, как и is_ip_in_list 

=cut

sub is_internal_ip
{
    my ($ip_addr) = @_;

    return is_ip_in_list($ip_addr, $Settings::INTERNAL_NETWORKS) ? 1 : 0;
}


=head2 get_first_non_private_ip

  $ip = get_first_non_private_ip('127.0.0.1', '213.43.123.11'); # '213.43.123.11'
  $ip = get_first_non_private_ip(split(/\s*,\s*/, $ENV{HTTP_X_FORWARDED_FOR}), $ENV{REMOTE_ADDR});

=cut

sub get_first_non_private_ip
{
    for my $ip (@_) {
        return $ip unless is_ip_in_list($ip,$Settings::PRIVATE_NETWORKS);
    }

    return $_[-1]; # else return last ip
}


=head2 my_aton

    вызывается из bsClientData.pl, чтобы заблокированные IP-адреса отправлять в БК в числовом виде

=cut

sub my_aton
{
    my $ip_addr = shift;
    return unpack("N", Socket::inet_aton($ip_addr));
}

# округлить ip-число до начала сети
# ipv6-ready: нет 
sub network_start_num 
{
    my ($num_ip, $mask) = @_;
    return $num_ip & (~(2 ** (32 - $mask) - 1));
}

# проверить, принадлежит ли ip перечисленным сетям
# ipv6-ready: нет 
sub is_ip_in_networks 
{
    my ($ip, $networks) = @_;
    my $num_ip = my_aton($ip);
    for my $net (split /\s*,\s*/, $networks) {
        $net =~ s/\s//g;
        if ($net =~ /^$IPRE$/) {
            return 1 if my_aton($net) == $num_ip;
        } elsif ($net =~ /^($IPRE)-($IPRE)$/) {
            return 1 if my_aton($1) <= $num_ip && $num_ip <= my_aton($2);
        } elsif ($net =~ /^($IPRE)\/(\d{1,2})$/) {
            my ($netip, $netmask) = ($1, $2);
            my $netstart = network_start_num(my_aton($netip), $netmask);
            return 1 if $num_ip >= $netstart && $num_ip < $netstart + 2 ** (32-$netmask);
        } else {
            print STDERR "Incorrect network '$net'\n";
        }
    }
    return 0;
}

# проверить валидность описания сетей
# ipv6-ready: нет 
sub validate_networks 
{
    my ($networks) = @_;
    return 1 if !defined $networks || $networks =~ /^\s*$/;
    return !grep {!/^$IPRE$/ && !/^($IPRE)-($IPRE)$/ && !/^($IPRE)\/(\d{1,2})$/}
           map {s/\s//g; $_} 
           split /\s*,\s*/, $networks;
}


=head2 my $hostname = get_hostname_fcrdns($ip)

    разрезолвить ip в доменное имя и проверить ещё одним резолвом, используя механизм
    Forward-confirmed reverse DNS

    если ip не разрезолвился в домен - возвращается сам ip
    если проверка прошла - возвращается домен
    если forward-проверка не прошла - вернуть undef
    при ошибке резолвинга - die

=cut
sub get_hostname_fcrdns($) {
    my ($ip) = @_;

    my ($err, @addrs) = Socket::getaddrinfo($ip, 0);
    die $err if $err;
    my $addr_bin = $addrs[0]->{addr};

    ($err, my $hostname) = Socket::getnameinfo($addr_bin);
    die $err if $err;

    ($err, my @addrs2) = Socket::getaddrinfo($hostname, 0);
    die $err if $err;

    return $hostname if grep {$addr_bin eq $_->{addr}} @addrs2;

    return undef;
}


=head2 check_acl($ip, [$acl_rule, ...])

    Проверка, что ip подходит хотя бы под одно правило из списка acl
    Правила могут быть:
    - networks:имя_файла - файл со списками сетей
    - регулярное выражение на хост
    - строка - хост
    - строка - ip

    Возвращает 0/1

    ipv6-ready: да, как и is_ip_in_list

=cut
sub check_acl {
    my ($ip, $acl) = @_;

    for my $rule (@$acl) {
        if ($rule =~ /^networks:(.*)/) {
            return 1 if is_ip_in_list($ip, "$Settings::NETWORKS_DIR/$1");
        } elsif (ref($rule) eq 'Regexp') {
            next;
        } else {
            return 1 if $ip eq $rule;
        }
    }

    # получаем хост только в случае необходимости
    my $hostname = get_hostname_fcrdns($ip);
    return 0 unless defined $hostname;
    for my $rule (@$acl) {
        if ($rule =~ /^networks:(.*)/) {
            next;
        } elsif (ref($rule) eq 'Regexp') {
            return 1 if $hostname =~ /^$rule$/;
        } else {
            return 1 if $hostname eq $rule;
        }
    }

    return 0;
}


1;
