#!/usr/bin/perl

package JSSchema::Validator;

use Direct::Modern;
use Mouse;
use Mouse::Util::TypeConstraints;

use Carp qw/carp/;
use Cache::SizeAwareMemoryCache;
use Try::Tiny qw/try catch/;
use Path::Tiny qw/path/;

use Yandex::Clone qw/yclone/;
use Yandex::Trace;

use JSON;
use JSV::Validator;
use JSSchema::Result;

use Settings;


enum 'Type' => [qw/input output/];

=head2 type
    Тип данных, которые будем валидировать - входные или выходные.
    Влияет на то, в каком из наборов будет искаться схема по имени cmd
    Может задаваться в конструкторе как через type => ... так и через for => ...
=cut

has type => (
    is => 'ro',
    isa => 'Type',
    required => 1,
);

has _schema_root => (
    is  => 'ro',
    isa => 'Path::Tiny',
    lazy_build => 1,
);

has _schema_file_suffix => (
    is => 'ro',
    isa => 'Str',
    default => '.schema.json'
);

has _jsv_environment => (
    is => 'ro',
    isa => 'Str',
    default => 'draft4'
);

has _validator => (
    is => 'ro',
    isa => 'JSV::Validator',
    lazy_build => 1,
);

has _schemas => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1,
);

has _config => (
    is => 'ro',
    isa => 'HashRef',
    lazy_build => 1,
);


our $JSSCHEMA_BASE_URL //= 'https://direct.yandex.com/';
our $JSSCHEMA_ROOT     //= $Settings::ROOT.'/cmd-schema';
our $JSSCHEMA_INDEX    //= 'index.json';

#Настройки Cache::SizeAwareMemoryCache
our $JSSCHEMA_CACHE_OPTIONS //= {
    namespace => 'JSSchema',
    max_size => 33_554_432, #32Mb 
    default_expires_in => 3600,  
};
my $_CACHED = Cache::SizeAwareMemoryCache->new($JSSCHEMA_CACHE_OPTIONS);


around BUILDARGS => sub {
    my ($orig, $class, %args) = @_;
    
    my $type = delete $args{for};
    
    $args{type} //= $type // 'input';
    
    return $class->$orig(%args)
};

=head2 init
    Загружает схемы и подсхемы с диска, регистрирует подсхемы и помещает их в кеш.
    Имеет смысл вызывать в preloader'е, чтобы первая валидация не проседала по времени
=cut

sub init {
    my ($self) = @_;
    return $self->_schemas;
}

=head2 validate
    Выполняет валидацию данных из data, по схеме, соответствующей переданной cmd
    Возвращает JSSchema::Result с результатами валидации.
=cut

sub validate {
    my ($self, $cmd, $data) = @_;
   
    my $profile = Yandex::Trace::new_profile('jss_validator:validation', tags => 'JSS');
    my $schema = $self->_schemas->{$cmd} // $self->_throw(sprintf 'JSON-schema for %s doesn`t exist', $cmd);

    return  JSSchema::Result->new($self->_validator->validate($schema => $data));
}


=head2 bite_by_schema
    Вырезает из датасета data все данные, не покрытые схемой, сответствующей переданной cmd
    Возвращает датасет с оставшимися после вырезания данными.
=cut

sub bite_by_schema {
    my ($self, $cmd, $data) = @_;
    
    my $profile = Yandex::Trace::new_profile('jss_validator:bite_by_schema', tags => 'JSS');
    my $schema = $self->_schemas->{$cmd} // $self->_throw(sprintf 'JSON-schema for %s doesn`t exist', $cmd);
    
    return $self->_validator->bite_data($schema, yclone $data);
}

#Билдеры

sub _build__schema_root {
    my ($self) = @_;

    return path($JSSCHEMA_ROOT);
}

sub _build__schemas {
    my $self = shift;

    my $schemas_key = 'schemas:'.$self->type;
    my $schemas = $_CACHED->get($schemas_key);
    unless ($schemas){
        $schemas = $self->_read_json_schemas();
        $_CACHED->set($schemas_key => $schemas);
    }

    my $refs = $_CACHED->get('refs');
    unless ($refs) {
        $refs = $self->_read_refs;
        $_CACHED->set(refs => $refs);
    }

    $self->_validator->register_schema($JSSCHEMA_BASE_URL.($refs->{$_}->{id} // $_) => $refs->{$_}) foreach keys %$refs;

    return $schemas;
}

sub _build__validator {
    my $self = shift;
    
    JSV::Validator->load_environments( $self->_jsv_environment );
    return JSV::Validator->new(
        environment => $self->_jsv_environment,
        enable_history => 1,
    );
}

sub _build__config {
    my $self = shift;
    
    my $cfg = decode_json($self->_schema_root->child($JSSCHEMA_INDEX)->slurp);
    $self->_throw('Wrong '.$JSSCHEMA_INDEX) unless $cfg && keys %$cfg;
    
    return $cfg;
}

#Приватные методы

sub _read_json_schemas {
    my $self = shift;
    
    my $schemas = {};
    my $schemas_dir = $self->_config->{$self->type.'-schemas'}->{dir} // $self->_throw($self->type.' dir not defined');
    $self->_read_json_schema_subdirs($self->_schema_root->child($_) => $schemas) foreach @$schemas_dir;

    # preprocess cmd's schemas
    while (my ($name, $cmd_schema) = each %$schemas) { 
        $self->_throw( sprintf "schema %s id already set (%s) ", $name, $cmd_schema->{id} ) if exists $cmd_schema->{id};
        $cmd_schema->{id} //= $JSSCHEMA_BASE_URL;
    }
    
    return $schemas;    
}

sub _read_refs {
    my $self = shift;
    
    my $refs = {};
    my $refs_dir = $self->_config->{subschemas}->{dir} // $self->_throw('Subschemas dir not defined');;
    $self->_read_json_schema_subdirs($self->_schema_root->child($_) => $refs) foreach @$refs_dir;

    return $refs;
}

sub _read_json_schema_subdirs {
    my ($self, $dir, $schemas) = @_;

    foreach my $entry ($dir->children) {
        if ($entry->is_dir) {
            $self->_read_json_schema_subdirs($entry, $schemas);
        } elsif ($entry->is_file ) {
            $self->_load_schema_from_file($entry, $schemas);
        } else {
            $self->_throw( 'unknown entry: '.$entry );
        }
    }
}

sub _read_schema_file {
    my ($self, $file) = @_;

    my $schema;
    my $content = $file->slurp;

    try {
        $schema = decode_json($content);
    }
    catch {
        my $error = shift;
        $self->_throw($error) unless $error =~ /offset ([0-9]+)/;
        my $offset = int $1;
        my $slot = substr $content, 0, $offset;
        my $line = 0;
        $line++ while $slot =~ /\n/g;
        $line++ if $line;
        confess sprintf '%s %s line %d', $error, $file, $line; 
    };

    return $schema;
}

sub _load_schema_from_file {
    my ($self, $file, $schemas) = @_;
    
    my $suffix = $self->_schema_file_suffix;
    my ($schema_name) = $file->stringify =~ qr{([^/]+)$suffix$};
    $self->_throw( 'unknown file: '.$file ) unless $schema_name;
    
    $schemas->{$schema_name} = $self->_read_schema_file($file);

    return;
}

sub _throw {
    my $self = shift;
    carp @_;
    return die $self;
}


1;

