package JavaIntapi::GeosearchSearchObject;

use JSON;
use Settings;
use Yandex::HTTP;
use Yandex::I18n;
use Yandex::Trace;
use Yandex::TVM2;
use Yandex::HashUtils qw/hash_copy/;

use Direct::Modern;

our $INTAPI_REQUEST_TIMEOUT = 15;

our @EXPORT = qw/
    _search_address
    _search_metro
    /;


sub _search_address {
    my ($self, $text, $O) = @_;

    my $profile = Yandex::Trace::new_profile('java_geosearch:search_address');
    my $lang = $O->{lang} // Yandex::I18n::current_lang();

    my $url = Yandex::HTTP::make_url(
        $Settings::GEOSEARCH_JAVA_INTAPI_URL . '/address', { text => $text, lang => uc $lang }
    );
    my $response = _do_request($url);

    my ($content, $error) = _prepare_response($response);
    my $geo_data;
    if (!$error && @$content) {
        $geo_data = _convert_address($content);
    }
    return $geo_data, $error;
}

sub _search_metro {
    my ($self, $x, $y) = @_;

    my $profile = Yandex::Trace::new_profile('java_geosearch:search_metro');
    my $lang = Yandex::I18n::current_lang();

    my $url = Yandex::HTTP::make_url(
        $Settings::GEOSEARCH_JAVA_INTAPI_URL . '/metro', { text => "$x,$y", lang => uc $lang }
    );
    my $response = _do_request($url);

    my ($content, $error) = _prepare_response($response);
    my $result;
    if (!$error && @$content) {
        my $geo_data = _convert_address([$content->[0]]);
        $result = $geo_data->{points}->[0];
    }

    return $result, $error;
}

sub _do_request {
    my ($url) = @_;

    state $log //= _prepare_log();

    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = join(',', map {$_ // 0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0);
    my $ticket = eval{Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi})} 
        or $log->die("Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@");
    
    $log->out(['GET', $url, 'X-Yandex-Trace:', $trace_header]);

    my $response = Yandex::HTTP::http_parallel_request(GET => { 1 => {
                url => $url
            }},
        timeout => $INTAPI_REQUEST_TIMEOUT,
        log => $log,
        headers => {
            'Accept' => 'application/json',
            'X-Detected-Locale' => Yandex::I18n::current_lang(),
            'X-Ya-Service-Ticket' => $ticket,
            'X-Yandex-Trace' => $trace_header
        },
    )->{1};

    return $response;
}


sub _prepare_response {
    my ($response) = @_;

    state $log //= _prepare_log();

    state $json //= JSON->new->allow_unknown(1)->allow_blessed(1)->convert_blessed(1);

    my ($result, $error);
    if ($response->{is_success}) {
        utf8::decode($response->{content});
        $result = $json->decode($response->{content});
    } else {
        $error = {
            server_status => $response->{headers}->{Status},
            server_error => 'Server returned an error'
        };
        $log->out(['Error', $error]);
    }

    return $result, $error;
}


sub _prepare_log {
    return Yandex::Log->new(
        no_log => 1,
        use_syslog => 1,
        syslog_prefix => 'INTAPI-JAVA',
        log_file_name => "GeosearchSearchObject.log",
    );
}

=head3 _convert_address(result)

    Подготавливает более удобный для работы массив из XML дерева

    Формат возвращаемых данных:
      {
        found => int,
        points => [
            {
                kind =>
                precision =>
                text =>
                address => {
                    country => string
                    city => string
                    administrative_area => string
                    street => string
                    house =>
                    build =>
                },
                point => {
                    x, y => int
                },
                bound => {
                    x1, y1, x2, y2 => int
                }
                auto_precision => 'exact'
                auto_point => '37.727409,55.799947'
                auto_bounds => '37.723304,55.797635,37.731515,55.802260'
            }, .....
        ]
      }        

=cut

sub _convert_address {
    my ($content) = @_;

    my %data;
    $data{found} = scalar @$content;

    foreach my $item (@$content) {
        my $address = {};
        $address->{geo_id} = $item->{geoId};

        hash_copy $address, $item, qw/text kind country city street house/;

        $address->{precision} = lc $item->{precision};
        $address->{auto_precision} = $address->{precision};
        $address->{'sub-city'} = undef;
        $address->{'administrative-area'} = $item->{administrativeArea};
        $address->{'sub-administrative-area'} = $item->{administrativeArea};
        $address->{'sub-street'} = undef; 

        ($address->{bound}{x1}, $address->{bound}{y1}) = ($item->{x1}, $item->{y1});
        ($address->{bound}{x2}, $address->{bound}{y2}) = ($item->{x2}, $item->{y2});
        $address->{auto_bounds} = join(",", $address->{bound}{x1}, $address->{bound}{y1}, $address->{bound}{x2}, $address->{bound}{y2});
    
        ($address->{point}{'x'}, $address->{point}{'y'}) = ($item->{'x'}, $item->{'y'});
        $address->{auto_point} = join(",", $address->{point}{'x'}, $address->{point}{'y'});
        push @{$data{points}}, $address;
    }
    
    return \%data;
}

1;

