package JavaIntapi::ValidateVcards;

use Direct::Modern;
use Mouse;

extends 'JavaIntapiClientBase';

use Yandex::I18n qw/iget/;

use Direct::ValidationResult qw//;
use Direct::Validation::Errors;


has '+profile_name' => (default => 'intapi_java:validate_vcards');
has '+warn_if_request_failed' => (default => 0);
has 'operator_uid' => (is => 'ro', isa => 'Id');
has 'client_id'    => (is => 'rw', isa => 'Id');

sub _method { 'vcard/validate_vcards' }

sub _get_url_params {
    my ($self) = @_;
    return {operator_uid => $self->operator_uid, client_id => $self->client_id};
}

sub _prepare {
    my ($self) = @_;
    return JavaIntapi::ValidateVcards->_convert_vcards_to_java_model($self->items);
}

sub _postprocess {
    my ($self, $result, $error) = @_;

    my $validation_errors;
    if ($result) {
        $validation_errors = $result->{validation_result}->{errors} // []; 

        my @fields = _get_java_vcards_field_names();
        my %pathes = map { $_->[0] => $_->[1] } @fields;
        my %names = map { $_->[0] => '"' . $_->[2] . '"' } @fields;

        foreach my $validation_error (@$validation_errors) {
            if ($validation_error->{text} && $validation_error->{path} 
                    && $validation_error->{path} =~ /(\[\d+\]\.)(.+)$/ && $names{$2}) {
                my ($index, $field) = ($1, $2);
                $validation_error->{text} =~ s/\Q$validation_error->{path}\E/$names{ $field }/g;
                $validation_error->{path} = $index . $pathes{ $field };
            } else {
                $error = "field not found for validation error with path=$validation_error->{path} 
                    and text=$validation_error->{text}";
                last;
            }
        }
    } else {
        $error = "error in call_intapi_java('vcard/validate_vcards'): " . ($error // '');
    }

    my $vr_main = Direct::ValidationResult->new();
    if ($error) {
        warn $error;
        $vr_main->add_generic(error_InvalidField(iget('Сервис временно недоступен')));
    } else {
        $vr_main = $self->convert_error_list_to_validation_result($validation_errors);
    }
    return $vr_main;
}

sub _convert_vcards_to_java_model {
    my ($class, $vcards) = @_;

    my %fields_j2p = map { $_->[0] => $_->[1] } _get_java_vcards_field_names();
    my $result = [];

    foreach my $vcard_values (@$vcards) {
        my %java_vcard = map { $_ => $vcard_values->{ $fields_j2p{$_} } }
            qw/campaignId companyName contactPerson email ogrn workTime extraMessage country city street house build apart metroId/;

        if ($vcard_values->{phone}) {
            @{ $java_vcard{phone} }{qw/countryCode cityCode phoneNumber extension/} = split('#', $vcard_values->{phone});
        }
 
        if ($vcard_values->{im_client} || $vcard_values->{im_login}) {
            $java_vcard{instantMessenger} = {
                type => $vcard_values->{im_client},
                login => $vcard_values->{im_login},
            };
        }
        if ($vcard_values->{'manual_point'}) {
            @{ $java_vcard{'manualPoint'} }{qw/x y/} = $vcard_values->{'manual_point'} =~ /^(-?\d+\.\d+),(-?\d+\.\d+)$/;
        }
        if ($vcard_values->{'manual_bounds'}) {
            @{ $java_vcard{'manualPoint'} }{qw/x1 y1 x2 y2/} = $vcard_values->{'manual_bounds'} =~ /^(-?\d+\.\d+),(-?\d+\.\d+),(-?\d+\.\d+),(-?\d+\.\d+)$/;
        }
        push @$result, \%java_vcard;
    }
    return $result;
}

sub _get_java_vcards_field_names {(
    ['campaignId',    'campaign_id',    iget('Идентификатор кампании')], 
    ['companyName',   'name',           iget('Название компании/ФИО')],
    ['contactPerson', 'contact_person', iget('Контактное лицо')],
    ['email',         'contact_email',  iget('E-mail')],
    ['ogrn',          'ogrn',           iget('ОГРН/ОГРНИП')],
    ['workTime',      'work_time',      iget('Время работы')],
    ['extraMessage',  'extra_message',  iget('Подробнее о товаре/услуге')],

    ['country',       'country',        iget('Страна')],
    ['city',          'city',           iget('Город')],
    ['street',        'street',         iget('Улица')],
    ['house',         'house',          iget('Дом')],
    ['build',         'building',       iget('Корпус')],
    ['apart',         'apartment',      iget('Офис')],
    ['metroId',       'metro',          iget('Метро')],
    
    ['phone',               'phone',                iget('Телефон')],
    ['phone.countryCode',   'phone.country_code',   iget('Телефон/Код страны')],
    ['phone.cityCode',      'phone.city_code',      iget('Телефон/Код города')],
    ['phone.phoneNumber',   'phone.phone_short',    iget('Телефон')],
    ['phone.extension',     'phone.phone_ext',      iget('Телефон/Добавочный')],

    ['instantMessenger.type',  'im_client', iget('Интернет-пейджер')],
    ['instantMessenger.login', 'im_login',  iget('Интернет-пейджер')],

    ['manualPoint.x', 'manual_point.x',  iget('Положение на карте')],
    ['manualPoint.y', 'manual_point.y',  iget('Положение на карте')],
    ['manualPoint.x1', 'manual_bounds.x1',  iget('Положение на карте')],
    ['manualPoint.y1', 'manual_bounds.y1',  iget('Положение на карте')],
    ['manualPoint.x2', 'manual_bounds.x2',  iget('Положение на карте')],
    ['manualPoint.y2', 'manual_bounds.y2',  iget('Положение на карте')],
)}


__PACKAGE__->meta->make_immutable;

1;
