package JavaIntapiClientBase;

use Direct::Modern;
use Mouse;

use Settings;
use JSON qw/encode_json decode_json/;
use Yandex::HTTP qw//;
use Yandex::Trace qw//;
use Yandex::I18n qw/iget/;
use Yandex::DBTools;
use Yandex::TVM2;
use LogTools qw/log_messages/;

use Direct::ValidationResult qw//;
use Direct::Validation::Errors;


has 'items'  => (is => 'ro', isa => 'ArrayRef[HashRef]');
# Имя тега профайла, в который будет записываться время ответа ручки
# Также служит префиксом для логов в messages (если они пишутся)
has 'profile_name' => (is => 'ro', isa => 'Str');
has 'warn_if_request_failed' => (is => 'ro', isa => 'Bool', default => 1);
# Метод HTTP, с которым функция _apply() будет ходить в ручку. Сейчас поддерживаются GET и POST
has 'http_method' => (is => 'ro', isa => 'Str', default => 'POST');
# Заголовок Accept
has 'accept_header' => (is => 'ro', isa => 'Str', default => '*/*');
# Писать ли логи. По умолчанию false
has 'log_enabled' => (is => 'ro', isa => 'Bool', default => 0);
# таймаут на запрос
has 'request_timeout' => (is => 'ro', isa => 'Int', default => 15);

=head2 call

    Вызывает java intapi
    Возвращает результат постпроцессинга ответа от сервера

=cut

sub call {
    my ($self) = @_;

    my ($result, $error);
    if ($self->http_method eq 'GET') {
        ($result, $error) = $self->_apply(undef);
    } else {
        my $request_body = $self->_prepare();
        ($result, $error) = $self->_apply($request_body);
    }

    return $self->_postprocess($result, $error);
}

sub _encode_log {
    my ($self, $data) = @_;
    state $encoder = JSON->new->allow_nonref->canonical;
    return $encoder->encode($data);
}

=head2 _apply($request_body)

    Выполняет запрос к Java IntAPI указанным методом (GET, POST).
    Возвращает результат ответа сервера в виде кортежа из двух объектов: $result и $error.
    Если ответ сервера пришёл с кодом 2хх, то в $result будет записано десериализованное содержимое, а в $error - undef.
    Если же ответ сервера пришёл с другим кодом, то в $result будет undef, а в $error - объект со статусом http-ответа.

    Помимо обычных Accept и Content-Type добавляются ещё заголовки trace_id и локали.
    Локаль необходимо передавать для того, чтобы возвращаемые тексты ошибок были в том же языке, в котором работает пользователь.

    $request_body -- optional

=cut

sub _apply {
    my ($self, $request_body) = @_;

    my $url = Yandex::HTTP::make_url(
        $Settings::DIRECT_JAVA_INTAPI_URL . $self->_method, $self->_get_url_params()
    );
    my $trace_id = Yandex::Trace::current_span_id();
    my $trace_header = defined $trace_id ? join(',', map {$_//0} $trace_id, Yandex::Trace::generate_traceid(), $trace_id, 0) : undef;

    my $trace_comment_vars = $Yandex::DBTools::TRACE_COMMENT_VARS->();
    my $sanitize = sub {return substr shift=~s/[^0-9a-zA-Z_\.\-]/_/gr, 0, 64; };
    my $trace_comment_vars_header = join(',',
        map {$sanitize->($_).'='.$sanitize->($trace_comment_vars->{$_})}
        sort(keys %$trace_comment_vars)
    );

    my $profile = $self->profile_name ? Yandex::Trace::new_profile($self->profile_name) : undef;
    my $lang = _get_lang_for_request();

    if ($self->log_enabled) {
        $self->_log(sprintf "call $url: %s, trace: %s" => $self->_encode_log($request_body), $trace_header);
    }

    my $ticket = eval{Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi})} or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
    my $http_method = $self->http_method;
    my $response = Yandex::HTTP::http_parallel_request($http_method => { 1 => {
            url => $url,
            $http_method eq 'GET' ? () : (body => encode_json($request_body))
        }},
        timeout => $self->request_timeout // 15,
        headers => { 
            'Content-Type' => 'application/json', 
            'Accept' => $self->accept_header,
            'X-Detected-Locale' => $lang,  # Сейчас Java IntAPI определяет локаль из этого заголовка, игнорируя Accept-Language
            'Accept-Language' => $lang,  # Передаём на случай, если в будущем Java IntAPI начнёт определять локаль из Accept-Language
            'X-Yandex-Trace' => $trace_header,
            'X-Yandex-Trace-Comment-Vars' => $trace_comment_vars_header,
            'X-Ya-Service-Ticket' => $ticket,
        },
    )->{1};
    undef $profile;

    my ($result, $error);
    if ($response->{is_success} || $self->_need_parse_failed_response($response)) {
        if ($response->{content}) {
            $result = decode_json($response->{content});
        }
    } else {
        $error = $response->{content} || ('Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')');
        utf8::decode($error);
        warn "java intapi request failed $url: $error" if $self->warn_if_request_failed; 
    }

    if ($self->log_enabled) {
        $self->_log(sprintf "result: %s; error: %s" => $self->_encode_log($result), $self->_encode_log($error));
    }

    return ($result, $error);
}

# Метод должен вернуть название ручки
sub _method         { croak 'must be overridden' }

# Метод должен вернуть хеш с url-параметрами
sub _get_url_params { croak 'must be overridden' }

# Флаг, нужно ли парсить ответ со статусом отличным от 200.
sub _need_parse_failed_response { return 0; }

# Метод должен вернуть подготовленный body для запроса либо undef, если body в запросе не нужен
sub _prepare        { croak 'must be overridden' }

# Метод конвертирует ($result, $error) в некоторый произвольный результат
sub _postprocess    { croak 'must be overridden' }

=head2

    Логирует сообщение в messages
    Сообщения можно будет посмотреть в logviewer, указав нужный префикс

=cut

sub _log {
    my ($self, $msg) = @_;
    my $prefix = $self->profile_name;
    log_messages($prefix, $msg);
}

=head2 convert_error_list_to_validation_result($errors)

    Складывает тексты ошибок валидации в Direct::ValidationResult
    $errors = [{path => '[0].phone', text => 'В поле Телефон указано некорректное значение'}, {...}, ...]
        массив хешей с полями 
            path - по нему будет строиться Direct::ValidationResult
            text - текст ошибки 

=cut

sub convert_error_list_to_validation_result {
    my ($self, $errors) = @_;
    my $errors_to_process = $self->_convert_paths($errors);
    return $self->_convert_errors_to_vr($errors_to_process);
}

sub _convert_paths {
    my ($self, $errors) = @_;
    my @result;

    foreach my $error (@$errors) {
        next unless $error->{text} && defined($error->{path});
        my $path = [map { $_ =~ /\[(\d+)\]/ ? {name => $1, is_index => 1} : {name => $_} } 
                     $error->{path} =~ /(\[\d+\]|\w+)/g];
        push @result, { text => $error->{text}, path => $path };
    }
    return \@result;
}

sub _convert_errors_to_vr {
    my ($self, $errors) = @_;
    return Direct::ValidationResult->new() unless $errors && @$errors;

    my (@generic_errors, %errors_by_field, @errors_by_index);
    foreach my $error (@$errors) {
        if ($error->{path} && @{ $error->{path} }) {
             my $first_node = shift @{ $error->{path} };
             if ($first_node->{is_index}) {
                 push @{ $errors_by_index[ $first_node->{name} ] }, $error;
             } else {
                 push @{ $errors_by_field{ $first_node->{name} } }, $error;
             }
        } else {
            push @generic_errors, $error;
        }
    }

    my $main_vr = Direct::ValidationResult->new();
    if (@generic_errors) {
        $main_vr->add_generic([map {error_InvalidField(iget($_->{text}))} @generic_errors]);
    }
    foreach my $field_name (keys %errors_by_field) {
        my $field_vr = $self->_convert_errors_to_vr($errors_by_field{ $field_name });
        $main_vr->add($field_name => $field_vr);
    }
    foreach my $error_by_index (@errors_by_index) {
        my $field_vr = $self->_convert_errors_to_vr($error_by_index);
        $main_vr->next($field_vr);
    }
    return $main_vr;
}

sub _get_lang_for_request {
    my %LANG_ALIAS = (ua => 'uk');
    my $lang = Yandex::I18n::current_lang();
    return $LANG_ALIAS{$lang} || $lang;
}

__PACKAGE__->meta->make_immutable;

1;
