package KnownDomains;

=head1 DESCRIPTION

Модуль для поддержки списков известных доменов

=cut

use Direct::Modern;

use Cache::SizeAwareMemoryCache;
use Yandex::DBTools;
use Yandex::URL qw/get_host get_protocol strip_subdomain strip_protocol/;

use Settings;

use Yandex::I18n;
use Direct::Validation::Errors;

our @DOMAIN_TYPES = qw/
    short
    counter
    mobile_app_counter
    mobile_app_impression_counter
/;
our %IS_KNOWN_DOMAIN_TYPE = map {($_ => 1)} @DOMAIN_TYPES;
our %AVAILABLE_OPTS = (
    https_only => 1,
    allow_wildcard => 1
);

our %TRACKING_HREF_ERRORS = (
    0 => sub { undef },
    1 => sub { error_BadUsage_UnsupportedDomain(iget('Данный домен трекинговой системы не поддерживается')) },
    2 => sub { error_BadUsage_UnsupportedProtocol(iget('Данный домен трекинговой системы поддерживает только протокол https')) },
);

my $CACHE = Cache::SizeAwareMemoryCache->new({namespace => 'KnownDomains', default_expires_in => 60});


=head2 add_domain

Добавить домен

=cut

sub add_domain {
    my ($type, $domain, $opts) = @_;
    croak "Unknown type: $type"  if !$IS_KNOWN_DOMAIN_TYPE{$type};
    # need check domain?

    $CACHE->remove($type);
    do_insert_into_table(PPCDICT, 'trusted_redirects', {
            domain => $domain,
            redirect_type => $type,
            opts => join(',', grep { $AVAILABLE_OPTS{$_} && $opts->{$_} } keys %$opts)
        },
        ignore => 1
    );

    return;
}


=head2 remove_domain

Удалить домен

=cut

sub remove_domain {
    my ($type, $domain) = @_;
    croak "Unknown type: $type"  if !$IS_KNOWN_DOMAIN_TYPE{$type};

    $CACHE->remove($type);
    do_delete_from_table(PPCDICT, 'trusted_redirects',
        where => {
            domain => $domain,
            redirect_type => $type,
        },
    );

    return;
}


=head2 get_domains_by_type

Список доменов запрошенного типа

Опции:
    no_cache - сбросить кеш
    from_list - вернуть только те домены, что есть в переданом списке ()
    get_as - тип возвращаемого значения (см. _cached_get)

=cut

sub get_domains_by_type {
    my ($type, %opt) = @_;
    croak "Unknown type: $type"  if !$IS_KNOWN_DOMAIN_TYPE{$type};
    croak "Arg get_as must be as_list if passed from_list arg" if $opt{from_list} && ($opt{get_as} // 'as_list') ne 'as_list';

    $CACHE->remove($type)  if $opt{no_cache};
    my $domains = _cached_get($type, $opt{get_as} ? $opt{get_as}: ());

    if ($opt{from_list}) {
        my %is_from_list = map {($_ => 1)} @{$opt{from_list}};
        $domains = [ grep {$is_from_list{$_}} @$domains ];
    }

    return $domains;
}


=head2 is_known_href

    if (is_known_href($type => $href)) { ... }

Проверка, попадает ли домен url'а в список известных
Возвращает 0 если все хорошо, если нет, то код ошибки:
 1 - домен не поддерживается
 2 - домен поддерживает только https ссылки

=cut

sub is_known_href {
    my ($type, $href, %opt) = @_;
    croak "Unknown type: $type"  if !$IS_KNOWN_DOMAIN_TYPE{$type};

    $CACHE->remove($type)  if $opt{no_cache};
    my $domains2opts = _cached_get($type, 'as_hash_with_opts');

    my $protocol = get_protocol($href);
    my $domain = get_host($href);
    my $super_domain = strip_subdomain($domain);

    if ($domains2opts->{$domain}) {
        if ($domains2opts->{$domain}->{https_only} && $protocol ne 'https://') {
            return 2;
        }
        return 0;
    }
    if ($domains2opts->{$super_domain} && $domains2opts->{$super_domain}->{allow_wildcard}) {
        if ($domains2opts->{$super_domain}->{https_only} && $protocol ne 'https://') {
            return 2;
        }
        return 0;
    }
    return 1;
}


sub _cached_get {
    my ($type, $key) = @_;

    my $data = $CACHE->get($type);

    if (!$data) {
        my $domains2opts = get_hash_sql(PPCDICT, [
            'SELECT domain, opts FROM trusted_redirects',
            WHERE => {
                redirect_type => $type,
            },
            "ORDER BY domain",
        ]);
        my $domains = [ keys %$domains2opts ];
        my %domain_hash = map {($_ => 1)} @$domains;
        my %domain_hash_with_opts = map { my $domain = $_; $domain => { map {$_ => scalar($domains2opts->{$domain} =~ /\b$_\b/);} keys %AVAILABLE_OPTS }; } keys %$domains2opts;
        $data = {
            as_list           => $domains,
            as_hash           => \%domain_hash,
            as_hash_with_opts => \%domain_hash_with_opts,
        };
        $CACHE->set($type => $data);
    }

    $key ||= 'as_list';
    croak "Unknown key $key"  if !exists $data->{$key};
    return $data->{$key};
}

1;
