package LWPRedirect;
## no critic (TestingAndDebugging::RequireUseWarnings)

=pod
    $Id$
    Модуль для отслеживания редиректов
=cut

use strict;

use URI;
use LWP::UserAgent;
use LWP::Protocol::http;
use HTTP::Response;
use Encode qw/encode is_utf8/;
use Time::Local;
use Time::HiRes qw/time/;

use Yandex::Trace;
use Yandex::HashUtils;
use Yandex::IDN;

our $MAX_REDIRECTS_NUM ||= 5;
our $TIMEOUT ||= 7;
my $PREDEFINED_REDIRECT_CODE = 399;

=head2

    По урлу получить цепочку редиректов

    Позиционнае параметры:
    - url - урл, который нужно протестировать
    Именованные параметры:
    - ua - объект LWP::UserAgent
    - max_redirects_num - максимальое количество редиректов
    - timeout - суммарный таймаут
    - predefined_redirect - ссылка на функция - получения редиректа без фактического запроса (если это возможно)

    Результат:
    - ссылка на хэш, по ключу chain - массив из ссылок на хэши с запросами:
        url - урл запроса
        redirect - урл редиректа, если он был
        label - информачия о том, какого типа редирект: http / predefined / meta / ..
        request - объект HTTP::Request
        response - объект HTTP::Response

=cut
sub get_redirect_chain {
    my ($url, %O) = @_;
    my $profile = Yandex::Trace::new_profile('lwp_redirect:get_redirect_chain');
    my $ua = $O{ua} || LWP::UserAgent->new();
    $ua->protocols_allowed(['http', 'https']);
    $ua->ssl_opts(verify_hostname => 0);
    my $redirects_num = $O{max_redirect_num} || $MAX_REDIRECTS_NUM;
    my $timeout = $O{timeout} || $TIMEOUT;
    
    # timegm не умеет работать с датами, старше unix-конца света (7-Feb-2038)
    # и сыпет варнингами. помогаем ему в этом
    no strict 'refs';
    no warnings 'redefine';
    my $orig_timegm = *{"Time::Local::timegm"}{CODE};
    local *{"Time::Local::timegm"} = sub {
        $orig_timegm->(@_[0..4], $_[5] > 2038 ? 2037 : $_[5]);
    };
    # у некоторых сайтов очень много заголовков
    local @LWP::Protocol::http::EXTRA_SOCK_OPTS = (MaxHeaderLines => 1000);

    my @chain;
    my $start_time = time();
    while(1) {
        $url = Yandex::IDN::idn_to_ascii($url);
        $url =~ s!([a-z]+://[^/]+)\/\.+(/|\?|$|#)!$1$2!i;
        utf8::encode($url) if Encode::is_utf8($url);

        if ( !$O{is_known_redirect} || $O{is_known_redirect}->($url) ) {
            my $req_start = time();
            my $req_timeout = $timeout - ($req_start - $start_time);
            $ua->timeout($req_timeout < 0.5 ? 0.5 : $req_timeout);

            my $req = HTTP::Request->new(GET => $url);
            my $resp;
            if ($req_timeout <= 0.1) {
                $resp = HTTP::Response->new(503, 'METAREDIRECT read timeout');
            } elsif (@chain >= $redirects_num) {
                $resp = HTTP::Response->new(503, 'METAREDIRECT limit exceeded');
            } elsif ($O{predefined_redirect} && (my $redir = $O{predefined_redirect}->($url))) {
                $resp = HTTP::Response->new($PREDEFINED_REDIRECT_CODE, 'PREDEFINED REDIRECT', HTTP::Headers->new(Location => $redir, 'Content-Base' => $url));
            } else {
                $resp = $ua->simple_request($req, sub {
                        my ($data, $r) = @_;
                        $r->content($r->content.$data);
                        if (time - $req_start > $req_timeout) {
                            $r->code(503);
                            $r->message("METAREDIRECT slow read timeout");
                            die 'Slow read timeout';
                        }
                    });
            }
    
            push @chain, my $item = {
                url => $url,
                request => $req,
                response => $resp,
            };
    
            if ($resp->code == $PREDEFINED_REDIRECT_CODE) {
                hash_merge $item, {
                    redirect => $resp->headers->header('Location'),
                    label => 'predefined',
                };
            } elsif ($resp->is_redirect) {
                hash_merge $item, {
                    redirect => $resp->headers->header('Location'),
                    label => 'http',
                };
            } elsif ($resp->is_error) {
                hash_merge $item, {
                    label => $resp->code == 404 ? 'Not found' : 'Error',
                };
            } else {
                if (my $meta = _parse_meta_redirect($resp)) {
                    hash_merge $item, {
                        redirect => $meta->{url},
                        label => 'meta',
                    };
                }
            }
            if ($item->{redirect}) {
                # обработка неабсолютных редиректов
                local $URI::ABS_ALLOW_RELATIVE_SCHEME = 1;
                my $base = $resp->base;
                $url = $item->{redirect} = URI->new($item->{redirect}, $base)->abs($base)->as_string;
            } else {
                foreach my $item (@chain) {
                    foreach (qw/url redirect/) {
                        next unless $item->{$_};
                        utf8::decode($item->{$_}) if !Encode::is_utf8($item->{$_}); 
                    }
                }
                last;
            }
        } else {
            push @chain, {
                url => $url,
                request => undef,
                response => undef,
            };
            last;
        }
    }

    return {chain => \@chain};
}

# по HTTP::Response попробовать найти meta-редирект
sub _parse_meta_redirect 
{
    my ($resp) = @_;
    my $content = $resp->decoded_content || '';

    # на больших объёмах регулярка может очень сильно тормозить
    $content = substr $content, 0, 100_000;
    $content =~ s/<\!(-){2,4}>//g;
    $content =~ s/<\!--.*?-->//gs;
    #$content =~ s/<\!(?:--(?:[^-]*(?:-[^-]+)*)--\s*)*>//g; #   Realy true SGML comments but IE and FF goes in other way

    $content = substr $content, 0, 10_000;
    while ( $content=~/<meta((?:\s*
                                    [^\s=<>\'\"]+
                                    \s*=\s*
                                    (?: \'[^\']*\'
                                    | \"[^\"]*\"
                                    | [^\s>=]+
                                    )
                                )*)\s*\/?>/xgi
    ) {
        my $atrs = $1;
        my %opt=();
        while ( $atrs =~ /  ([^\s=<>\'\"]+)
                                \s*=\s*
                                (?:\'([^\']*)\'
                                | \"([^\"]*)\"
                                | ([^\s>=]+)
                                )/xgi
        ) {
            $opt{lc($1)} = $^N;
        }
        
        if ( ($opt{'http-equiv'} and $opt{'http-equiv'} =~ /refresh/i
              or $opt{'name'} and $opt{'name'} =~ /refresh/i )
              and $opt{content}
        ) {
            
            if ( $opt{content} =~ /\s*0*(\d+)\s*\;(?:\s*url\s*=)?\s*["']?(\w+\:\/\/\S+)["']?\s*/i and $1 <= 60 ) {
                return {time=>$1, url=>$2};
            }
        }
    }
    return undef;
}

1;
