package Lang::Unglue;
## no critic (TestingAndDebugging::RequireUseWarnings)

# $Id$
# Автор: Елена Большакова

use strict;
use vars qw(@ISA @EXPORT @EXPORT_OK);
require Exporter;

use List::MoreUtils qw/uniq/;

use Yandex::MyGoodWords;
use Settings;
use List::Util qw/reduce min max/;
use MinusWords;
use Models::Phrase qw/validate_phrases/;
use Direct::PhraseTools;

use utf8;

@ISA         = qw(Exporter);
@EXPORT_OK = @EXPORT = qw( 
        unglue_phrases
        unglue_phrases_lite
        glue_minus_words
    );

#Расклеивание фраз внутри объявлений
#Если в объявлении есть фразы "A" и "A B" -- делаем "A -B" и "A B"
#Если есть пересекающиеся фразы "D E" и "E F" -- не трогаем.
#Метод: сравниваем каждую фразу с каждой, если одна фраза -- подфраза другой, 
#то добавляем к первой недостающие минус-слова.
#
#Основная функция -- unglue_phrases.


# Проверка $p1 \in $p2 : is_subpharse($p1, $p2);
#
# С опцией $exact_flag функция учитывает флаг точного соответствия у слов во фрвзах и совпадение словоформ если флаг выставлен
# Для Директа и unglue_phrases следует использовать функцию без опции
# $p1, $p2 - фразы после get_phrase_extended_info()

sub is_subphrase($$;$)
{
    my ($p1, $p2, $exact_flag) = @_;
    my $res = 1;

    for my $w (keys %{$p1->{plus}}) {
        $res = 0
            if 
                !$p2->{plus}{$w} or $exact_flag and
                (
                    $p1->{plus}{$w}{strict} > $p2->{plus}{$w}{strict} or  # учитываем требование точного соотвествия фраз
                    $p1->{plus}{$w}{strict} == 1 and $p2->{plus}{$w}{strict} == 1 and $p1->{plus}{$w}{word} ne $p2->{plus}{$w}{word} # точные формы совпадают
                );
    }

    return $res;
}

#"Вычитаем" одну фразу из другой. 
#Если в $p1 -- слово "A", а в $p2 -- слова "A B", 
#то после subtract_phrase($p1, $p2) в $p1 будет "A -B".
#Представление фразы "A !B -C -!D": 
#$p = { plus => {
#            "A" => {old=>1, strict => 0}, 
#            "B" => {old=>1, strict => 1}
#        },
#        minus => {
#            "C" => {old=>1, strict => 0}, #минус-слово, которое изначально было в фразе
#            "D" => {old=>0, strict => 1},  #минус-слово, добавленное в результате расклейки
#        }
#      }
#$p1 = $p1 - $p2 : subtract_phrase($p1, $p2)
sub subtract_phrase
{
    my ($p1, $p2) = @_;
   
    for my $w (keys %{$p2->{plus}}) {
        if (!$p1->{plus}->{$w}) {
            $p1->{minus}->{$w} ||= {   old    => 0, 
                                       strict => $p2->{plus}->{$w}->{strict}, 
                                       word  => $p2->{plus}->{$w}->{word}
                                   };
        }
    }
}

=head2 unglue_phrases

    "Расклеиваем" фразы. На вход получаем "блок" 
    $block = [ { ... Phrases =>[{... phrase=>..., md5=>...}, {}, ...] },
            { ... Phrases =>[{}, {} ] } ]
    Т.о. из мульти-редактирования можно сделать $changed=unglue_phrases($vars->{banners});
    из редактирования одного баннера -- $changed=unglue_phrases([$vars]);
    Возвращаемое значение: 0, если фразы не были изменены и 1, если были.
    Для каждой фразы добавляется поле phrase_unglued_suffix, которое содержит "раскливающий суффикс". 

    При мультиредактировании расклеивание производится по фразам из всех редактируемых баннеров.

    Второй (необязательный) параметр $max_length: ограничение на суммарную длину ключевых слов. 
    Если после добавления очередного минус-слова длина ключевых слов стала бы больше $max_length -- минус-слово не добавляем.
    Если $max_length > 0 -- используем это значение, 
    если $max_length <= 0 или отсутствует -- суммарную длину не ограничиваем

=cut 

sub unglue_phrases
{
    my ($block, $max_length) = @_;
    my $profile = Yandex::Trace::new_profile('unglue:unglue_phrases');

    $max_length = 0 if !$max_length || $max_length < 0;

    # разбираем фразы из объявлений в удобные хеши
    my %PHRASES = ();
    for my $banner ( @$block ) {
        for my $row_phrase (@{$banner->{Phrases} || $banner->{phrases}}){
            my $ph = get_phrase_extended_info($row_phrase);

            # Ввиду частого повторения ошибки, когда разминусование молча
            # не работает из-за отсутствия md5, появился этот warning
            warn "Unglue wouldn't work while phrase don't have md5"
                unless $row_phrase->{md5};

            $PHRASES{($banner->{bid}||$banner->{pid}||"")."_".$row_phrase->{md5}} = $ph;
        }
    }


    #%PHRASES -- хеш ("$bid_$md5" => $phrase), @phrases -- массив фраз
    my @phrases = values %PHRASES;
    #Основной цикл -- находим, какие минус-слова надо добавить
    for my $p1 (@phrases) {
        for my $p2 (@phrases) {
            #Для того, чтобы не вычесть лишнего, сравниваем однословные фразы только с двухсловными, двухсловные с трехсловными и т.д. 
            #Пример: 'A, A B C' не должно стать 'A -B -C, A B C', т.к. потеряются показы по фразе 'A B'
            next if $p1->{numword} != $p2->{numword} - 1; 
            #Если фраза -- в кавычках, то ей минус-слов не надо
            next if $p1->{quoted};
            #сравниваем и добавляем минус-слова, если надо
            subtract_phrase($p1, $p2) if is_subphrase($p1, $p2);
        }
    }

    #приклеиваем минус-слова к каждой фразе
    my $changed = 0;
    for my $ph (@phrases) {
        $ph->{unglued_suffix}='';
        for my $w (sort keys %{$ph->{minus}}) {
            # пропускаем неподходящие минус-слова: 
            #  * слова с пробелами, дефисами и точками не вычитаем (но цифры, вида 12.34(.56) - учитываем)
            next if $w =~ /[\ \-\.]/ && $w !~ /^\d+(?:\.\d+)+$/;
            #  * если слово было раньше -- не вычитаем (чтобы не получилось "львы -автомобили -автомобиль") 
            next if $ph->{minus}->{$w}->{old};
            #  * против случаев экзотической нормализации
            #    например: 
            #    "керамогранит" нормализуется как "керамогранять", "керамогранита" -- как "керамогранит" 
            #    В результате без проверки из 
            #     'поставщик керамогранит, керамогранит поставщик керамогранита' получалось 
            #     'поставщик керамогранит -керамогранит, керамогранит поставщик керамогранита'
            #    Возможно, стала излишней из-за проверки с validate_key_phrase
            next if exists $ph->{all}->{$w};
            #  * против совсем экзотических случаев с несколькими нормальными формами
            #    например: 
            #    в 'песни тексты песен, текст песни' ко второй фразе приклеивается '-песня', а потом это дает проблемы при сохранении 
	    #  И просто проверяем, что фраза получилась допустимой, потому что вдальнейшем именно Models::Phrase::validate_phrases будет проверять на допустимость.
	    next if @{ Models::Phrase::validate_phrases([{phrase => "$ph->{phrase} $ph->{unglued_suffix} -$ph->{minus}->{$w}->{word}"}]) } > 0 ;

            # все проверки пройдены, добавляем минус-слово
            $ph->{unglued_suffix}.= " -".$ph->{minus}->{$w}->{word};
        }
    }

    #записываем расклеенные фразы обратно в блок
    for my $banner ( @$block ) {
        my $bid = $banner->{bid}||$banner->{pid}||"";
        # При переходе на группы отрываем массив Phrases вообще, поэтому, на момент перехода смотрим оба массива.
        my $phrases = $banner->{Phrases} || $banner->{phrases};
        #запоминаем длину ключевых слов до расклеивания
        my $len = length join ',', map {$_->{phrase}} @$phrases ;    
        next if $max_length && $len > $max_length;
        for my $row_phrase (@$phrases){
            my $key = $bid."_".$row_phrase->{md5};
            my $suffix = $PHRASES{$key}->{unglued_suffix} || "";
            my $suf_len = length $suffix;
            $row_phrase->{phrase_unglued_suffix} = '';
            #если после добавления очередного минус-слова длина отдельной фразы или всех фраз вместе станет слишком большой -- не добавляем. 
            next if $max_length && $len + $suf_len > $max_length; 
            $row_phrase->{phrase_unglued_suffix} = $suffix;
            $row_phrase->{numword} = $PHRASES{$key}->{numword};
            $len += $suf_len;
            $changed=1 if $row_phrase->{phrase_unglued_suffix} ne "";
        }
    }

    return $changed;
}


=head2 unglue_phrases_lite

    Параметры позиционные
        $phrases -- ссылка на массив строк (фраз)

    Параметры именованные 
        max_len -- максимальная допустимая суммарная длина фраз после расклеивания
                   0 -- ограничение по умолчанию, отрицательное значение -- нет ограничения

    Результат 
    {
        phrases => { 
            $phrase_orig_1 => {
                phrase_orig => ...,
                phrase_unglued => ...,
                unglued_suffix => ...,
            },
        }, 
        unglued => 1,    # флаг: были ли добавлены минус-слова
    }

=cut 

# TODO когда-нибудь хорошо бы unglue_phrases_lite (функция с простым массивом на входе) сделать основной функцией, 
# а добавление расклеивающх минус-слов in place (как в unglue_phrases) реализовать уже через нее

sub unglue_phrases_lite
{
    my ($phrases, %O) = @_;

    $O{max_len} = -1 unless exists $O{max_len};

    my $i = 1;
    my $block = [
        {
            bid => 1,
            Phrases => [
                map { {phrase=>$_, md5 => $i++} } @$phrases
                ]
        }
        ];
    unglue_phrases($block, $O{max_len});

    my $res = { 
        phrases =>{}, 
        unglued => 0, 
    };
    for my $p (@{ $block->[0]->{Phrases} }) {
        $res->{phrases}->{$p->{phrase}} = {
            unglued_suffix => $p->{phrase_unglued_suffix},
            phrase_unglued => $p->{phrase}.$p->{phrase_unglued_suffix},
            phrase_orig => $p->{phrase},
        };
        $res->{unglued} = 1 if $p->{phrase_unglued_suffix};
    }

    return $res; 
}

=head2 glue_minus_words($phrases, %opt)

    Функция расклеивает миус-слова. Если есть фразы A -B -C и A -D, то на выходе получаем фразу A -B -C -D
    В качестве параметра принимает массив фраз, где каждая фраза - это хеш вида: {phrase => 'текст фразы', is_new=>0|1}.
    phrase - собственно текст фразы
    is_new - флаг означающий является ли фраза новой.
    Также хеш может содержать другие поля.

    На выходе возвращается такой же массив объектов, только со склеенными минус-словами.
    Одновременно удаляются дубликаты.

    Опции %opt
        store_collapsing_key - если задано, то в каждый элемент входящего
            массива $phrases записывается ключ
            по которому схлопываются фразы, из двух одинаковых по этому ключу
            фраз остается одна с объедененными минус словами

=cut

sub glue_minus_words {
    my ($phrases, %opts) = @_;
    my $store_collapsing_key = $opts{store_collapsing_key} ? 1 : 0;

    my %word_hash = ();
    my $i = 0;
    for my $phrase (@{$phrases}) {
        # Разбираем строку на ключевые слова и минус-слова.
        my ($key_words, $minus_words) = split /(?:^|\s)\-/, $phrase->{phrase}, 2;
        $minus_words = MinusWords::polish_minus_words($minus_words, add_minus => 1);
        $key_words = Direct::PhraseTools::polish_phrase_text($key_words, $minus_words);
        # Нормализуем ключевые слова.
        my $norm_key_words = Yandex::MyGoodWords::norm_words($key_words);
        $phrase->{collapsing_key} = $norm_key_words if $store_collapsing_key;
        # Склеиваем минус-слова
        if (!defined($word_hash{$norm_key_words})) {
            $word_hash{$norm_key_words} = {key_words => $key_words, minus_words => [split /\s+/, $minus_words], order => $i, phrase_object=>$phrase };
        } else {
            push @{$word_hash{$norm_key_words}->{minus_words}}, split /\s+/, $minus_words;
            # Если фраза не новая, то обновляем phrase_object, т.е. приклеиваем минус слова к старым, а не новым фразам.
            if (!$phrase->{is_new} &&  $word_hash{$norm_key_words}->{phrase_object} && $word_hash{$norm_key_words}->{phrase_object}->{is_new}) {
                $word_hash{$norm_key_words}->{phrase_object} = $phrase;
            }
        }
        $i++;
    }

    my @result = ();
    for my $phrase (sort  {$a->{order} <=> $b->{order}} values (%word_hash)) {
        my $phrase_object = $phrase->{phrase_object};
        $phrase_object->{phrase} = $phrase->{key_words};
        $phrase_object->{phrase} .= join " ", "", uniq @{$phrase->{minus_words}} if ($phrase->{minus_words});
        push @result, $phrase_object;
    }
    return @result;
}

=head2 get_phrase_extended_info

    подготавливает фразу к разнообразным сравнениям, формируя хеш с дополнительной информацией
    на вход - фраза в формате
        {phrase => текст фразы}
    на выходе - фраза в формате
        {phrase => текст фразы,
          ...}

=cut

sub get_phrase_extended_info {
    my $row_phrase = shift;
    my $ph = {};
    my $str = $row_phrase->{phrase};

    # заменаем минус слова через дефис на пробелы
    $str =~ s/(\s\-[^\-\s]+)\-(?=[^\s])/$1 \-/g;
    
    # заменяем дефисы в словах на пробелы
    $str =~ s/(?<=\S)\-(?=\S)/ /g;
    
    $str =~ s/\\*//g;
    $str =~ s/\-\s+/-/g;
    $str =~ s/^\s+//g;
    $str =~ s/\s+$//g;
    $str =~ s/\+\s+/+/g;
    $str =~ s/\]\[/] [/g;
    
    $ph->{quoted} = 1 if $str =~ s/\"//g;
    $ph->{phrase} = $str;
    for my $w ( split(' ', $str) ) {
        $w =~ s/\[|\]//g;
        my $pm = $w =~ s/^-// ? "minus" : "plus"; 
        my $st = $w =~ s/^!//;
        my $all = $w =~ s/^\+//;
        my $norm = $w =~ /^\d+(?:\.\d+)*$/ ? $w : (Yandex::MyGoodWords::norm_words($w) || "");
        $st ||= $norm eq "" && $all;
        $norm = $w if $norm eq "" && $st; #предлоги с восклицательным знаком (строгое соответствие: !для !по и т.п.)
        $ph->{$pm}->{$norm} = {old => 1, strict => $st}; 
        $ph->{$pm}->{$norm}->{word} = $st ? "!$w": $norm;

        # И дополнительно запоминаем все слова -- в ненормализованной форме, как они были во фразе. 
        $ph->{all}->{$w} = undef;
    }

    #В плюс- или минус- слова могло попасть пустое слово (нормальная форма для предлогов и т.п.). Его удаляем.
    delete $ph->{minus}->{''};
    delete $ph->{plus}->{''};

    $ph->{numword} = scalar( keys %{$ph->{plus}} );

    return $ph;
}

1;
