#######################################################################
#
#  Direct.Yandex.ru
#
#   Named lock DB-based realisation
#
#  $Id$
#

package LockObject;

use warnings;
use strict;

use List::MoreUtils qw/all/;

use Yandex::DBTools;
use Yandex::HashUtils;

use Settings;

#
#   You can use this module either as Named locks provider or as anonymous one
#   id (i.e. name) defined for any lock object in DB
#   
#   to load or delete named lock try this code:
#       new LockObject({d => $id, object_type=>'user', object_id=>$uid})->load();
#   
#   to load unnamed one just skip the id:
#       new LockObject({object_type=>'user', object_id=>$uid})->load();
#
#   to make a new lock:
#       new LockObject({object_type=>'user', object_id=>$uid})->save() and warn "Yep, I've jsut got the lock!" or die "can't lock the object";
#
#   params values:
#
#   object_id : $cid|$uid|$bid
#   object_type : m/user|campaign|banner/
#   id: bigint unsigned auto_increment not null
#   half_md5_hash: bigint_unsigned not null - additational key for named locks
#   duration : lock_time duration in seconds 300 by default


my $allowed_types = {map {$_=>1} qw/campaign user banner/};
#   Implementing RoseDB like Interface
#   
#   Default constructor
sub new($$){
    my $class = shift;
    my $data = shift;

    die "mandatory params missed : object_type object_id" unless all {exists $data->{$_}} qw/object_type object_id/;
    die "object_type must be equal to one of allowed types: ".join(", ", keys %$allowed_types) unless $allowed_types->{$data->{object_type}};

    bless { %$data }, $class;
}

#
#   save object
#   returns:
#   id - on success
#   undef - else
#
sub save($) {
    my $self=shift;
    die "mandatory params missed : object_type object_id" unless all {exists $self->{$_}} qw/object_type object_id/;
    my $lock_interval = ( $self->{duration} ||= 5*60);

    my $params = hash_cut $self, qw/object_type object_id/;
    # наверное, условие не нужно и стоит добавить индекс по lock_time
    do_delete_from_table(PPCDICT, "lock_object", where => {%$params, lock_time__le__dont_quote => 'NOW()'});
    $self->{id} = do_insert_into_table(PPCDICT, "lock_object",
                                       {
                                           %$params,
                                           half_md5_hash => $self->{half_md5_hash}||0,
                                           lock_time__dont_quote => "NOW() + interval $lock_interval second"
                                       },
                                       ignore => 1,
        );

    if(!$self->{id}) {
        return undef;
    }
    return $self;
}

#
#   locad object with specified params
#   
#   returns:
#   object - if there is a lock with specified params that is valid till a moment in future
#   undef - no valid object located
sub load($) {
    my $self = shift;
    my $params = hash_cut $self, qw/id object_type object_id half_md5_hash/;
    die "mandatory params missed : object_type object_id" unless all {exists $params->{$_}} qw/object_type object_id/;
    
    my $res = get_one_line_sql(PPCDICT, ["
        SELECT id, object_type, object_id, lock_time, half_md5_hash,
                IF(lock_time > NOW(), 1,0) as is_locked,
                unix_timestamp(lock_time) - unix_timestamp(NOW()) as duration
           from lock_object
            ", WHERE => {
                %$params,
                lock_time__gt__dont_quote => 'NOW()'
            }
            ]);

    if ($res and keys %$res) {
        return hash_merge $self, $res;
    } else {
        return undef;
    }
}

#
#
#   remove lock with specified params
#   returns the count of removed locks (1 or 0)
#
#   my $lock  = new LockObject({object_type=>"user", object_id=>123})->load() or die "The user has been unlocked before the transaction finished";
#
#   do something with user ...
#
#   $lock->delete(); # don't die here - the lock timeout coul'd have gone off before delete() call
#
sub delete($){
    my $self = shift;
    my $params = hash_cut $self, qw/id object_type object_id half_md5_hash/;

    die "mandatory params missed : object_type object_id" unless all {exists $params->{$_}} qw/object_type object_id/;
    
    return do_delete_from_table(PPCDICT, "lock_object", where => $params);
}

1;

