package LogTools;
use Direct::Modern;

use JSON;
use Carp;
use POSIX qw/strftime/;

use Yandex::HashUtils;
use Yandex::Log;
use Yandex::Log::Messages;
use Yandex::Trace;
use Yandex::ListUtils qw/chunks/;

use Settings;
use EnvTools;
use DirectContext;
use HierarchicalMultipliers::Base;
use Tools qw/log_cmd/;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    log_metrika_query
    log_hierarchical_multiplier
    log_auctions_limit
    log_api_tech_limit_update
    log_bs_resync_market_ratings
    log_messages
    log_stacktrace
    log_zora_exception
    log_gozora_exception
/;

=head2 %LogTools::context

Неявный контекст для операций логгирования. Можно сделать так:

    sub cmd_XXXXXXX ... {
       ...
       local $LogTools::context{uid} = $uid;
       local $LogTools::context{reqid} = $c->reqid;
    }

И в рамках выполнения этого контроллера всегда иметь эти данные в записях в логи.

=cut
our %context;

=head2 get_context

    получить ссылку на копию текущего контекста

=cut
sub get_context {
    return {%context};
}

=head2 make_logger

Возвращает функцию, которая пишет логи в указанный файл. К данным,
туда переданным, добавляет содержимое %LogTools::context.

    my $logger = make_logger('log_file_name');
    $logger->({some => 1, data => 2, ...});

=cut
sub make_logger($;@) {
    my ($filename, %opts) = @_;

    my $log_syslog = Yandex::Log->new(
        use_syslog => 1,
        # пока пишем и в файл тоже
        # no_log => 1,
        log_file_name => $filename . ".log",
        date_suf => "%Y%m%d",
        %opts,
    );
    $log_syslog->msg_prefix("[$$]");

    return sub {
        my ($data) = @_;
        my (undef, undef, undef, $subroutine) = caller(2);
        my $log_data = {
            caller => $subroutine,
            %context
        };
        hash_merge $log_data, $data;
        eval {
            $log_syslog->out($log_data);
        };
        if ($@) {
            print STDERR "failed to syslog: $@\n";
        }
    };
}

=head2 log_metrika_query

Запись в лог об обращении в метрику. К данным, туда переданным, добавляет содержимое %LogTools::context.

    my $some_arbitary_data = {
        action => 'metrika_http_result',
        http_requests => { 1 => { url => "http://ya.ru", ... }, ... }
    };
    log_metrika_query($some_data)

=cut
sub log_metrika_query($) {
    state $logger;
    $logger //= make_logger('metrika_calls', syslog_prefix => 'external_service');
    &$logger;
}

=head2 log_hierarchical_multiplier

Изменения коэффициентов цены (мобильные, соцдем, ретаргетинг, ...). Ожидается, что запись относится к одной
кампании или группе, а данные имеют 'cid' и (возможно) 'pid' как ключи верхнего уровня.

=cut
sub log_hierarchical_multiplier($) {
    my $mult_data = shift;
    state $logger;
    $logger //= messages_logger('hierarchical_multiplier');
    $logger->out($mult_data);
    my $data = { %context, %$mult_data, cmd => '_save_hierarchical_multipliers' };
    for my $mult_type (@{HierarchicalMultipliers::Base::known_types()}) {
        $data->{$mult_type} = JSON->new->pretty(1)->encode($data->{$mult_type}) if exists $data->{$mult_type};
    }
    Tools::log_cmd($data);
}

=head2 log_bs_resync_market_ratings

Компании, которые будут переотправлены через ленивую очередь при изменении настройки "Рейтинг магазина на Маркете"

=cut
sub log_bs_resync_market_ratings($) {
    my $data = shift;

    state $logger;
    $logger //= messages_logger('bs_resync_market_ratings');

    $data->{action} = 'Add campaigns to bs_resync';
    $logger->out( $data );
}

=head2 log_auctions_limit

Запись в лог информации о ограничении запросов в апи по количеству торгов

=cut

sub log_auctions_limit($) {
    state $logger;
    $logger //= make_logger('auctions_limit');
    &$logger;
}

=head2 log_api_tech_limit_update

    Запись в лог информации о ограничении запросов в апи по количеству торгов

=cut

sub log_api_tech_limit_update($) {
    state $logger;
    $logger //= make_logger('api_tech_limit_update');
    &$logger;
}

=head2 log_redirects

Временное логирование редиректов (чтобы проанализировать и закрыть уязвимость open redirect)

=cut
sub log_redirects($) {
    state $logger;
    $logger //= make_logger('redirects');
    &$logger;
}

=head2 log_messages

Логгирование в messages.log с определённым префиксом
Префикс принудительно берется в квадратные скобки (если их нет),
чтобы при парсинге попадать в отдельную колонку

=cut
sub log_messages($$) {
    my ($prefix, $msg) = @_;
    state $logger;
    $logger //= Yandex::Log::Messages->new();
    if ($prefix && $prefix !~ m/^\[/) {
        $prefix = "[$prefix]";
    }
    my $g = $logger->msg_prefix_guard($prefix);
    $msg =~ s/\n/|/g if !ref $msg;
    $logger->out($msg);
}

=head2 messages_logger

    создвать logger, пишущий в messages.log с определённым префиксом
    
=cut 
sub messages_logger($) {
    my ($prefix) = @_;
    my $logger = Yandex::Log::Messages->new();
    if ($prefix && $prefix !~ m/^\[/) {
        $logger->msg_prefix("[$prefix]");
    }
    return $logger;
}

=head2 log_stacktrace

Логирование со стректрейсом, обычно для каких-то временных разбирательств

=cut
sub log_stacktrace($$) {
    my ($prefix, $msg) = @_;
    log_messages($prefix, Carp::longmess($msg));
}

=head2 log_zora_exception

Логирует в messages подробности об исключении ZoraException, выкинутом из LWP::UserAgent::Zora

=cut

sub log_zora_exception($) {
    my ($e) = @_;

    my %details;
    if (ref $e eq 'ZoraException') {
        eval {
            $details{code} = $e->response->code;
            $details{message} = $e->response->message;
            $details{uri} = $e->response->request->uri->as_string;
            $details{"X-Yandex-Http-Code"} = $e->response->header("X-Yandex-Http-Code");
            $details{"X-Yandex-Status"} = $e->response->header("X-Yandex-Status");
        };
        if ($@) {
            $details{error} = "Failed to extract exception details: " . $@;
        }
    } else {
        $details{error} = $e;
    }

    log_messages(ZoraException => \%details);
}

=head2 log_gozora_exception

Логирует в messages подробности об исключении GoZoraException, выкинутом из GoZoraClient

=cut

sub log_gozora_exception($) {
    my ($e) = @_;

    my %details;
    if (ref $e eq 'GoZoraException') {
        eval {
            $details{code} = $e->response->code;
            $details{message} = $e->response->message;
            $details{uri} = $e->response->request->uri->as_string;
            $details{"X-Yandex-Gozora-Error-Code"} = $e->response->header("X-Yandex-Gozora-Error-Code");
            $details{"X-Yandex-Gozora-Error-Description"} = $e->response->header("X-Yandex-Gozora-Error-Description");
        };
        if ($@) {
            $details{error} = "Failed to extract exception details: " . $@;
        }
    } else {
        $details{error} = $e;
    }

    log_messages(GoZoraException => \%details);
}

=head2 log_price

    Логирование изменения ставок - пишем в файл

    log_price([{
        cid =>
        pid =>
        bid =>
        id =>
        type => 'insert1'|'insert3'|'update'|... [см. схему таблицы logprice_YYYYMMDD]
        price =>
        price_ctx => [ОСТОРОЖНО, не price_context]
        currency    =>
    }, {...}, ...]);

=cut
sub log_price
{
    my $input = ref $_[0] eq 'HASH' ? [$_[0]] : $_[0];

    # нормализуем данные
    my @data;
    for my $row (@$input) {
        my $rec = {};
        foreach my $c (qw/cid pid bid id price price_ctx/) {
            $rec->{$c} = $row->{$c} // 0;
        }
        foreach my $c (qw/type currency/) {
            $rec->{$c} = $row->{$c} // '';
            $rec->{$c} =~ s/['"\n\r]//g;
        }
        push @data, $rec;
    }

    # пишем в файл для LogBroker-a
    LogTools::_lograw_price(\@data);
}

=head2 log_campaign_balance(\@records)

    пишем данные о изменении начисленных средств на кампаниях (NofifyOrder)

=cut
sub log_campaign_balance {
    my ($recs) = @_;
    _lograw('campaign_balance', [map {+{data => $_}} @$recs]);
}

=head2 log_mail(\@records)

    пишем данные о отправленных письмах в лог

=cut
sub log_mail {
    my ($email, $template_name, $subj, $content, $client_id) = @_;
    _lograw('mails', [+{data => {email => $email, template_name => $template_name, subject => $subj, content => $content, client_id => $client_id}}]);
    if (EnvTools::is_beta()) {
        # дублируем лог в один общий файл для всех бет, чтобы загружать его push-client'ом
        # полагаемся на то, что в _lograw объекты Yandex::Log создаются с lock => 1
        local $Yandex::Log::LOG_ROOT; $Yandex::Log::LOG_ROOT = '/var/log/yandex';
        local $Yandex::Log::UMASK;    $Yandex::Log::UMASK = 0;
        local $EnvTools::hostname = sprintf "%s:beta.%s.%s", $EnvTools::hostname, $Settings::BETA_PORT // '', $Settings::CONFIGURATION; # чтобы можно было выбирать логи с этой беты
        # beta_mails вместо mails, потому что _lograw привязывает к значению первого аргумента объект Yandex::Log с умолчальным LOG_ROOT
        # на момент внесения этих правок парсер логов писем никак не обрабатывает log_type
        _lograw('beta_mails', [+{data => {email => $email, template_name => $template_name, subject => $subj, content => $content, client_id => $client_id}}]);
    }
}


=head2 _lograw(log_type, [rec1, ...], log_file_name)

    Хэлпер для записи в лог, удобный для push-client/logbroker/...

=cut

sub _lograw
{
    my ($log_type, $recs, $log_file_name) = @_;

    $log_file_name //= "$log_type.log";
    state %loggers;
    my $logger = $loggers{$log_type} ||= Yandex::Log->new(
        use_syslog => 0,
        log_file_name => $log_file_name,
        no_date => 1,
        date_suf => '%Y%m%d',
        lock => 1,
        auto_rotate => 1,
        );

    my $trace = Yandex::Trace::current_trace();
    my $c = $DirectContext::current_context;

    my $log_rec = {
        log_time => strftime("%Y-%m-%d %H:%M:%S", localtime), 
        log_type => $log_type,
        log_hostname => $EnvTools::hostname,

        reqid => ($trace ? $trace->trace_id : 0) // 0,
        service => ($trace ? $trace->service : ''),
        method => ($trace ? $trace->method : ''),

        uid => ($c ? $c->UID : 0) // 0,
    };

    for my $rec (@$recs) {
        $logger->out({%$log_rec, %$rec});
    }
}


=head2 _lograw_price

    пишем данные изменения ставок в лог-файл,
    вызывается из log_price

=cut

sub _lograw_price {
    my ($prices) = @_;
    
    for my $chunk (chunks $prices, 1000) {
        _lograw('ppclog_price', [{
            ip => $ENV{REMOTE_ADDR},
            data => $chunk}
                ]);
    }
}

=head2 log_autobudget_prices

Пишем данные, пришедшие в ручку AutobudgetPrices.set

=cut

sub log_autobudget_prices {
    my ($data) = @_;

    for my $chunk (chunks $data, 500) {
        _lograw('autobudget_prices', [map {+{data => $_}} @$chunk], 'common_data.log');
    }
}

=head2 log_xls_user_units

    Пишем юниты, потраченные на заливку xls

=cut

sub log_xls_user_units {
    my ($data) = @_;

    for my $chunk (chunks $data, 500) {
        _lograw('xls_users_units', [map {+{data => $_}} @$chunk], 'common_data.log');
    }
}

=head2 log_user_units

    Записать рассчитанные баллы на заливку данных через XLS/API

=cut

sub log_user_units {
    my ($data) = @_;

    for my $chunk (chunks $data, 500) {
        _lograw('user_units', [map {+{data => $_}} @$chunk], 'common_data.log');
    }
}

=head2 log_user_ratings

    Пишем рейтинги для расчёта баллов

=cut

sub log_user_ratings {
    my ($data) = @_;

    for my $chunk (chunks $data, 500) {
        _lograw('ratings', [map {+{data => $_}} @$chunk], 'common_data.log');
    }
}

1;
