#######################################################################
#
#  Direct.Yandex.ru
#
#  MailNotification
#  Notification on mail
#
#  $Id$
#
#######################################################################

=head1 NAME

MailNotification - Notification on mail

=head1 DESCRIPTION

Notification on mail

=cut

package MailNotification;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

require Exporter;

our $VERSION = '0.01';
our @ISA = qw(Exporter);
our @EXPORT = qw(
    mail_notification
    mass_mail_notification
);

use warnings;
use strict;

use JSON;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log;

use Settings;
use PrimitivesIds;
use RBACElementary;
use RBACDirect;

my $UID;
my $host;

#======================================================================
# for execute from main.pl - save global $UID and host
sub save_UID_host
{
    $UID = shift;
    $host = shift || '';
}

#======================================================================

=head2 mail_notification

mail notification

  mail_notification($event_object, $event_type, $object_id, $old_text, $uid);

  $event_object : 'camp' || 'banner' || 'phrase'
  $event_type   : 'c_start', 'c_finish', 'c_status',
                  'b_create', 'b_text', 'b_status', 'b_word', 'b_category', 'b_geo',
                  'ph_price', 'ph_change', 'c_day_budget',
                  'b_retargeting', 'ret_delete'
  $object_id    : $cid, $bid or $pid
  $old_text     : text for mail (status/state of object before edit)
  $uid          : uid of user - owner of object

=cut

sub mail_notification
{
    my ($event_object, $event_type, $object_id, $old_text, $new_text, $uid) = @_;

    my $rec = {
        object     => $event_object,
        event_type => $event_type,
        object_id  => $object_id,
        old_text   => $old_text,
        new_text   => $new_text,
        uid        => $uid,
    };

    mass_mail_notification( [ $rec ] );
}


=head2 mass_mail_notification

    Добавление событый в очередь на отправку email-оповещений

    Параметры позиционные:
    $events -- ссылка на массив событий.
      За один вызов могут регистрироваться любые события, в т.ч. разного типа/с разными объектами/из разных кампаний.
      Каждое событие -- ссылка на хеш
        {
            object     => $event_object,
            event_type => $event_type,
            object_id  => $object_id,
            old_text   => $old_text,
            new_text   => $new_text,
            uid        => $uid,
        }

    Выгодно использовать mass_mail_notification вместо mail_notification, если много событий по одной и той же кампании:
    mail_notification делает все проверки прав каждый раз (на каждое событие),
    mass_mail_notification -- один раз на кампанию за один вызов

=cut

sub mass_mail_notification
{
    my ( $events ) = @_;

    # нет событий -- ничего не делаем
    return if ! @{$events||[]};

    my $log_syslog = Yandex::Log->new(
        use_syslog => 1,
        no_log => 1,
        log_file_name => 'servicing_events',
        syslog_prefix => 'EVENTS',
    );
    $log_syslog->out(@$events);
    my $bids2group_name = get_use_adgroups_by_bids([
        map {$_->{object_id}}
        grep {$_->{object} =~ /^(banner|phrase)$/}
        @$events
    ]);

    # распределяем все события по кампаниям
    my %events_by_cid;
    my %events_by_pid;
    my %events_by_bid;
    for my $e ( @$events ){
        if (!$e->{object_id}) {
             warn "cid for mail_notification not found";
             next;
        }
        if ($e->{object} eq 'camp') {
            push @{$events_by_cid{$e->{object_id}}}, $e;
        } elsif ($e->{object} eq 'banner' || $e->{object} eq 'phrase') {
            push @{$events_by_bid{$e->{object_id}}}, $e;
        } elsif ($e->{object} eq 'adgroup') {
            push @{$events_by_pid{$e->{object_id}}}, $e;
        } else {
            die("eventobject $e->{object} is not valid\n");
        }
    }
    my $pid2cid = %events_by_pid ? get_pid2cid(pid => [keys %events_by_pid]) : {};
    my $bid2cid = %events_by_bid ? get_bid2cid(bid => [keys %events_by_bid]) : {};
    for my $type (qw/bid pid/) {
        my $id2cid = $type eq 'bid' ? $bid2cid : $pid2cid;
        my $events_by_id = $type eq 'bid' ? \%events_by_bid : \%events_by_pid;
        for my $id (keys %$id2cid) {
            my $cid = $id2cid->{$id};
            if (!$cid) {
                warn "cid for mail_notification not found";
                next;
            }
            push @{$events_by_cid{$cid}}, @{$events_by_id->{$id}};
        }
    }

    # обрабатываем события по каждой кампании
    my @to_insert;
    for my $cid (grep {$_} keys %events_by_cid) {
        my $cid_events = $events_by_cid{$cid} || [];
        next unless @$cid_events;
        next unless rbac_is_scampaign(undef, $cid) || rbac_who_is(undef, $UID) eq 'manager';

        for my $e ( @$cid_events ){
            if ($e->{event_type} !~
                /^(c_start
                  |c_finish
                  |c_status
                  |c_pay_multicurrency
                  |c_autooptimisation
                  |c_strategy
                  |b_create
                  |b_text
                  |b_status
                  |b_cond
                  |b_word
                  |b_category
                  |b_geo
                  |ph_price
                  |ph_price_ctx
                  |ph_change
                  |c_day_budget
                  |c_day_budget_multicurrency
                  |b_retargeting
                  |ret_delete
                  |adgr_geo
                  |adgr_word
                  |adgr_retargeting
                  |adgr_mobile_content_device_type_targeting
                  |adgr_mobile_content_network_targeting
                  )$/x
               )
            {
                die("eventtype $e->{event_type} is not valid\n");
            }

            # определяем, что событие было по группе баннеров
            my ($group_name, $pid);
            if ($e->{object} eq 'adgroup') {
                $group_name = $e->{group_name};
                $pid = $e->{object_id};
            } elsif (($e->{object} eq 'banner' || $e->{object} eq 'phrase') && exists $bids2group_name->{$e->{object_id}}) {
                $pid = $bids2group_name->{$e->{object_id}}->{pid};
                $group_name = $bids2group_name->{$e->{object_id}}->{group_name};

                if ($e->{event_type} =~ /^(b_geo|b_word|b_retargeting)$/) {
                    $e->{object} = 'adgroup'; # считаем событием по группе
                    $e->{event_type} =~ s/^b_(geo|word|retargeting)$/adgr_$1/;
                }
            }

            my $data = {old_text => $e->{old_text}, new_text => $e->{new_text}};
            if ($pid) {
                $data->{pid} = $pid;
                $data->{group_name} = $group_name;
            }
            my $json_data = to_json($data);
            push @to_insert, [$e->{uid}, $e->{object}, $e->{object_id}, $e->{event_type}, $json_data, $UID, $cid];
        }
    }

    if (@to_insert) {
        foreach_shard uid => \@to_insert, by => sub {$_->[0]}, sub {
            my ($shard, $chunk) = @_;
            do_mass_insert_sql(PPC(shard => $shard), 'insert into events (objectuid, eventobject, objectid, eventtype, json_data, uid, cid) values %s', $chunk);
        };
    }
}

# --------------------------------------------------------------------

=head2 get_use_adgroups_by_bids

определяем использует ли клиент группы по bid

    my $use_adgroups = get_use_adgroups_by_bids([bid1, bid2, ...]);

возвращаем хеш
    {
        bid1 => {pid => 12132, group_name => 'group name'},
        bid2 => {pid => 12133, group_name => ''}, # группы использует, но имя не заданно
    }
=cut

sub get_use_adgroups_by_bids($) {
    my $bids = shift;

    return {} unless @$bids;

    my $result = get_hashes_hash_sql(PPC(bid => $bids), [
        "select b.bid
              , p.pid
              , IFNULL(p.group_name, '') as group_name
         from banners b
         join phrases p on p.pid = b.pid
         join campaigns c on c.cid = p.cid
         join users u on u.uid = c.uid
         left join clients_options cl_op on cl_op.ClientID = u.ClientID
        ", where => {
            'b.bid' => SHARD_IDS
        }
    ]);

    return $result;
}

1;

