package Manager;

=head1 NAME

    Manager

=head1 DESCRIPTION

    Код для работы с сущностью "менеджер"

=cut

use Direct::Modern;

use Carp qw/croak/;
use List::MoreUtils qw/uniq/;

use Yandex::DBTools;
use Yandex::DBShards;

use Settings;
use RBACElementary;
use RBACDirect;
use Client;
use Agency;
use Currencies;
use PrimitivesIds;

use base qw/Exporter/;

our @EXPORT = qw();

our @EXPORT_OK = qw(
    mass_get_manager_currencies
);

=head2 mass_get_manager_currencies

    Определяет "валюту менеджера" — ту валюту, в которой менеджер видит список клиентов и агентств.
    Возвращает самую частую валюту среди неархивных клиентов и неархивных агентств.

    $manageruid2currency = mass_get_manager_currencies($rbac, [$manager_uid1, $manager_uid2, ...], $type);

=cut

sub mass_get_manager_currencies {
    my ($rbac, $manager_uids, $type) = @_;

    # логика неархивности клиентов и агентств должна соответствовать DoCmd::cmd_showManagerMyClients
    # т.е. должны быть кампании в текущем сервисе (баян/директ) и неархинвый пользователь

    my %manageruid2currency;
    for my $manager_uid (@$manager_uids) {
        my @currencies;

        # агентства
        my $agency_uids = rbac_get_agencies_uids($rbac, $manager_uid);
        if ($agency_uids && @$agency_uids) {
            my $active_agency_uids = get_one_column_sql(PPC(uid => $agency_uids), [q/
                SELECT DISTINCT u.uid
                FROM users u
             /, WHERE => {
                    'u.uid' => SHARD_IDS,
                    'u.statusArch' => 'No',
            }]) || [];
            if (@$active_agency_uids) {
                my @agency_clientids = uniq values %{ rbac_get_agencies_clientids_by_uids( $active_agency_uids) };
                my $agency_currencies = mass_get_agency_currency($rbac, \@agency_clientids);
                push @currencies, values %$agency_currencies;
            }
        }

        # прямые клиенты
        my $client_ids = get_one_column_sql(PPC(shard => 'all'), [q/
            SELECT DISTINCT u.ClientID
            FROM campaigns c
            INNER JOIN users u ON c.uid = u.uid
         /, WHERE => {
                'c.ManagerUID' => $manager_uid,
                'c.type' => $type,
                'c.statusEmpty' => 'No',
                'u.statusArch' => 'No',
        }]) || [];
        if (@$client_ids) {
            my $client_currencies = mass_get_client_currencies($client_ids);
            push @currencies, map {$_->{work_currency}} values %$client_currencies;
        }

        my $currency;
        if (@currencies) {
            $currency = Currencies::get_dominant_currency(\@currencies);
        } else {
            my $manager_clientid = get_clientid(uid => $manager_uid);
            $currency = get_client_currencies($manager_clientid)->{work_currency};
        }
        $manageruid2currency{$manager_uid} = $currency;
    }

    return \%manageruid2currency;
}

1;
