package Mediaplan;

# $Id$

=head1 NAME

    Mediaplan

=head1 DESCRIPTION

    Tools for working with mediaplan
    Использует все точечные модули, не должно включать общие модули типа Common. 

=cut

use warnings;
use strict;

use Carp;

use BannersCommon;
use BannerTemplates qw/add_banner_template_fields validate_banner_template is_template_banner/;
use BannerImages;
use BS::TrafaretAuction;
use Campaign;
use CampaignQuery;
use CommonMaps;
use Currencies;
use Stat::OrderStatDay;
use Forecast qw/forecast_calc/;
use GeoTools;
use MediaplanOptions;
use MinusWords qw/polish_minus_words_array save_banner_minus_words/;
use MinusWordsTools;
use ModerateDiagnosis qw/get_diags/;
use Phrase;
use PhraseText;
use PhrasePrice;
use PhraseDoubles;
use Primitives;
use PrimitivesIds;
use RBACDirect;
use Settings;
use Sitelinks;
use TimeTarget;
use TextTools;
use HashingTools;
use ShardingTools;
use Tools;
use URLDomain;
use VCards;
use BS::History;
use Direct::ResponseHelper;
use Client;

use Lang::Unglue;

use Retargeting;
use Models::PhraseTools; 
use Models::DesktopBanner qw/create_desktop_banner update_desktop_banners/;
use Models::MobileBanner qw/create_mobile_banner update_mobile_banner/;
use Models::Phrase;
use Models::AdGroup;
use Models::CampaignOperations;
use Direct::Validation::Keywords qw/base_validate_keywords validate_add_keywords/;
use Direct::Validation::TurboLandings;
use Direct::Banners;
use Direct::Model::Creative;
use Direct::Model::Creative::Factory;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::IDN qw/is_valid_email/;
use Yandex::MyGoodWords;

use LogTools;
use Notification;

use Yandex::ListUtils;
use Yandex::Validate;
use Yandex::MirrorsTools::Hostings qw/strip_www/;
use Yandex::URL;

use OrgDetails;
use User;

use List::MoreUtils qw/any uniq/;
use List::Util qw/first/;
use JSON;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw/

    get_mediaplan_banner
    get_mediaplan_banner_form
    get_mediaplan_phrases_form
    get_user_mediaplan
    get_mediaplan_banners
    get_mediaplan_banners_count
    get_mediaplan_phrases
    accept_mediaplan
    delete_mediaplan
    validate_mediabanner
    validate_accept_mediaplan
    validate_mediaplan_mail
    validate_mediaplan
    update_mediaplan_stats
    get_mediaplan_stats
    get_mediaplans_stats
    get_fa_statuses
    get_common_geo_for_mediaplan
    add_optimizing_request
    accept_optimized_banners
    close_request_first_aid
    is_first_phrase_winner_by_ctr
    get_duplicate_phrases_ids
    add_mediaplan_banner
    is_allow_FA_lang


    update_mediaplan_phrases
    update_mediaplan_phrases_wrapper
    update_mediaplan_banner
    delete_mediaplan_banners
    delete_mediaplan_phrases
    end_mediaplan

    copy_banners_to_mediaplan
    copy_mediaplan_banners
    compare_banners
    mediaplan_banner_search_params
    get_optimization_request
    is_second_aid
    filter_fields_for_log
/;

# список полей в табл. mediaplan_banners
our @MEDIAPLAN_BANNER_FIELDS = qw/
    mbid cid type
    title title_extension
    body href
    geo
    domain
    statusShowsForecast
    timeShowsForecast
    source_bid
    source_pid
    vcard_id
    sitelinks_set_id
    mw_id
/;

our @MEDIAPLAN_BIDS_FIELDS = qw/
    id
    mbid
    phrase
    numword
    place
    statusPhrase
    cid
    showsForecast
    statusModerate
    phraseIdHistory
    lowCtr
    is_suspended 
/;
our @MEDIAPLAN_STATS_FIELDS = qw/
    cid
    mpid
    ManagerUID
    MediaUID
    accept_time
    accept_type
    requested
    request_type
    create_time
    mark
    mark_time
    comment
    AcceptUID
    end_comment
    optimize_type
    difficulty_type
    text_type
    is_lego_mediaplan
    reject_reason
/;

# список полей в таблицы optimizing_campaign_requests
our @OPTIMIZATION_FIELDS = qw/
    request_id
    cid
    MediaUID
    status
    create_time
    comment
    fio
    email
    budget
    ready_time
    accept_time
    start_time
    reject_comment
    added_banners_count
    improvements
    CreaterUID
    postpone_date
    request_comment
    req_type
    is_automatic
    is_support
/;

our @FIELDS_FOR_LOG = qw/
    mbid
    title
    title_extension
    body
    href
    banner_type
    geo
    vcard_id
    source_bid
    source_pid
    select_action/;

our %SUBFIELDS_FOR_LOG = (
    phrases =>      [qw/ph_id phrase phraseIdHistory statusModerate statusPhrase old_id previous_phrase/],
    retargetings => [qw/ret_id ret_cond_id statusPhrase/],
    extra_data   => [qw/banner_action phrases_action/]);
our $PREVIOUS_PREFIX = "previous_";

push @FIELDS_FOR_LOG, map {$PREVIOUS_PREFIX.$_} @FIELDS_FOR_LOG;

# Список языков, кому можно создавать заявку на ПП.
our @ALLOW_FA_LANG = qw/ru en/;

# Список доменных зон, на которых показывать тизер ПП в случае, если кампания подходит под все условия.
our @ALLOW_TEASER_DOMAIN = qw/ru by/;

# Список доменных зон, на которых не показывать тизер ПП при любом раскладе 
# (т.е. даже если стоит у пользователя галка что показывать тизер ПП).
our @DISALLOW_TEASER_DOMAIN = qw/tr/;

# Для выявления изменений, которые сделал медиапланнер с баннерами, мы сохраняем баннеры в отдельной таблице, и затем сравниваем.
# Но для сравнения нет необходимости иметь все поля, сравниваются только некоторые. Они перечислены ниже:
# Поля баннера, которые нужны для сравнения изменений
my @BANNER_COMPARE_FIELDS = qw/title title_extension body domain href geo banner_minus_words source_bid mbid/;

# Поля фраз, которые нужны для сравнения изменений
my @PHRASE_COMPARE_FIELDS = qw/phrase id md5 phraseIdHistory/;

=head2 get_mediaplan_banner_form

     Создаёт объект медиаплана из полей пришедшей формы.

=cut
sub get_mediaplan_banner_form
{
    my $form = shift;
    my $suffix = shift;
    my %OPT = @_;

    my $geosuffix = $suffix;
    $suffix =~ s/\s//g;
    $geosuffix = ($suffix) ? '_'.$suffix : '';
    $suffix = ($suffix) ? '-'.$suffix : '' ;
    my $mediaplan = {};
    
    my (@phrases, @rubrics);
    
    my @fields = qw/cid bid banner_type
                    title title_extension body href domain url_protocol geo
                    domain_sign city_code country_code geo_id
                    text_geo
                    json_banner_minus_words /;
    push @fields, @$VCARD_FIELDS_FORM;
    unless ($form->{"banner_with_phone".$suffix} || $form->{"with_ci".$suffix}) {
        delete $form->{$_.$suffix} for @$VCARD_FIELDS_FORM;
    }
    unless ($form->{"href".$suffix} && $form->{"href".$suffix} =~ /\S/
            && ($form->{"banner_with_href".$suffix} || $form->{"with_href".$suffix})
    ) {
        delete $form->{$_.$suffix} for qw/href domain url_protocol/;
    }
    
    if ($suffix) {
        my $cut_suffix = hash_cut $form, map { $_ . $suffix } @fields;
        my %cut_wo_suffix = map { $_ => $cut_suffix->{ $_ . $suffix }} @fields;
        hash_merge $mediaplan, \%cut_wo_suffix;
    } else {
        hash_merge $mediaplan, hash_cut $form, @fields;
    }

    $mediaplan->{banner_minus_words} = delete $mediaplan->{json_banner_minus_words};

    #Есть поле-исключениe: geo, которое имеет вид geo_[bid] (а не geo-[bid]), как все.
    my $geo = $mediaplan->{geo} || $form->{"geo".$geosuffix};
    $geo =~ s/\s+//g if defined $geo;
    $mediaplan->{geo} = $geo if defined $geo;

    $form->{sort} ||= 'phrase';
    my $phr_hash = get_mediaplan_phrases_form($form, $suffix, %OPT);
    $mediaplan->{error_text} = $phr_hash->{error_text};
    $mediaplan->{retargetings} = $phr_hash->{retargetings} if exists $phr_hash->{retargetings};
    my @all_phrases = @{$phr_hash->{phrases}};

    # Создаём хеш для вывода
    @phrases = sort {
                    my ($a1, $b1) = $form->{reverse} ? ($b, $a) : ($a,$b);
                    if ( !defined $form->{sort} || $form->{sort} eq 'phrase' ) {
                        return lc($a1->{phrase}||'') cmp lc($b1->{phrase}||'');
                    } else {
                        return $a1->{$form->{sort}} <=> $b1->{$form->{sort}};
                    }
                } @all_phrases;
    
    if (! @phrases && !@{$mediaplan->{retargetings} || []}) {
        # Если нет фраз, значит это не может быть, что данные из формы пришли, просто их вообще нет в форме, 
        # а это значит, что еще ничего в форму не вводили.
        # Исключение если мы пришли с нулевого шага создания медиаплана, то надо учитывать выбранный там регион.
        return hash_cut $mediaplan, [qw/geo/] if $mediaplan->{geo};
        return {};
    }

    $mediaplan->{worktimes} = get_worktimes_array( $mediaplan->{worktime} ) if $mediaplan->{worktime} && $form->{"banner_with_phone".$suffix};
    $mediaplan->{with_ci} = $mediaplan->{banner_with_phone} = ($form->{'banner_with_phone'.$suffix} || $form->{'with_ci'.$suffix}) ? 1 : 0;
    $mediaplan->{banner_with_href} = $mediaplan->{has_href} = ($mediaplan->{href} || !$mediaplan->{with_ci}) ? 1 : 0;
    $mediaplan->{phone} = compile_phone($mediaplan) if $mediaplan->{with_ci};
    $mediaplan->{href} = clear_banner_href($mediaplan->{href}, $mediaplan->{url_protocol});
    hash_merge $mediaplan, divide_href_protocol($mediaplan->{href});

    $mediaplan->{Phrases} = \@phrases;

    $mediaplan->{sitelinks} = Sitelinks::get_sitelinks_form($form, $suffix);

    $mediaplan->{banner_minus_words} = MinusWords::polish_minus_words_array($mediaplan->{banner_minus_words});

    # обрезаем начальные и конечные пробелы во всех полях
    for my $key (keys %$mediaplan) {
        $mediaplan->{$key} =~ s/^\s+|\s+$//g if defined $mediaplan->{$key};
    }

    $mediaplan->{is_template_banner} = is_template_banner($mediaplan);

    return $mediaplan;
}

=head2 get_mediaplan_phrases_form

     Создаёт список фраз медиаплана из полей пришедшей формы.

=cut
sub get_mediaplan_phrases_form
{
    my $form = shift;
    my $suffix = shift;
    my %OPT = @_;

    $suffix = '-'.$suffix if $suffix && !($suffix =~ /-\d+/);
    $suffix =~ s/\s//g;

    my %phrases_hash; # хэш для удаления дубликатов фраз
    my @phrases;
    my $phrases = $form->{"phrases".$suffix} || $form->{Phrases} || '';

    $phrases =~ s/\n/,/g;
    $phrases =~ s/\s*,\s*(,\s*)+/,/g;
    $phrases =~ s/-\s+/ -/g;
    $phrases =~ s/-+/-/g;
    $phrases =~ s/\s+/ /g;
    $phrases =~ s/^\s+|\s+$//g;
    $phrases = html2string( $phrases );

    my @parsed_phrases;
    my @result = ();
    foreach my $phrase (split(/,/, $phrases || '')) {
        my %hs;
        my @phrase_errors = ();

        if ( $phrase =~ /^::(\d+)+::(\d+)$/ ){
            # пропускаем пустые фразы
            next;
        } elsif ($phrase !~ /^([^\:]+)+::(\d+)+::(\d+)$/) {
            push @phrase_errors, iget('Некорректно задана фраза');
        } else {
            ($hs{phrase}, $hs{id0}, $hs{numword}) = split '::', $phrase;

            $hs{phrase} ||= '';
            $hs{id0} ||= 0;
            $hs{numword} ||= 0;

            if(! scalar(@phrase_errors)) {
                my $validation_result = base_validate_keywords([$hs{phrase}]);
                unless ($validation_result->is_valid) {
                    push @phrase_errors, @{$validation_result->one_error_description_by_objects};
                }
                if( !@phrase_errors ) {
    
                    my $phrase = $hs{phrase};
                    my @mass_not_valid = @{ validate_key_phrase( $phrase ) };
                    my $bad_word = join ',', @mass_not_valid;
                    if( defined $bad_word && $bad_word ne '' ) {
                        if (scalar @mass_not_valid > 1) {
                            push @phrase_errors, iget("Нельзя вычитать слова - (%s), содержащиеся в исходной ключевой фразе (%s)", $bad_word, string2html( $phrase ) );
                        } else {
                            push @phrase_errors, iget("Нельзя вычитать слово - (%s), содержащееся в исходной ключевой фразе (%s)", $bad_word, string2html( $phrase ) );
                        }
                    }
                }
                push @result, @phrase_errors;
            }
            push @parsed_phrases, \%hs;
        }
    }

    my $move = $form->{select_action} && $form->{select_action} eq "movePhrases";
    for my $line ( split "," , $form->{"ids_phrases".$suffix} || '' ) {
        my %hs;
        ( $hs{id0} , $hs{numword} ) = split ":" , $line ;
        $hs{phrase} = '';
        $hs{id0} ||= 0;
        $hs{numword} ||= 0;
        push @parsed_phrases, \%hs;
    }

    for (@parsed_phrases) {
        my %hs = %{$_};
        my $ph;
        $ph = get_mediaplan_phrase($form->{cid}, $hs{id0}) if ($hs{id0});
        if ($ph) {
            if ($hs{phrase} ne $ph->{phrase} && !$move) {
                my %old_phrase = (phrase => $hs{phrase}, phrase_old => $ph->{phrase}, numword => 1);
                my $new_word = phrase_should_null_phrase_id(\%old_phrase); 
                $hs{old_id} = $hs{id0};
                $hs{id} = ($new_word)? 0 : $hs{id0};
                $hs{id0} = $hs{id};
            }

            $hs{place} = $ph->{place};

            $hs{mbid} = $ph->{mbid};
            # Если копируем фразы из других баннеров, и не изменяли текст фразы, то флаги lowCtr и declined надо проставлять.
            # Иначе фразы считаются активными
            if (($move || !defined($form->{select_action})) && ($hs{phrase} eq $ph->{phrase} || $hs{phrase} eq '')) {
                $hs{declined} = $ph->{statusModerate} eq 'No';
                $hs{lowCtr} = $ph->{lowCtr};
            }
            $hs{phrase} = $ph->{phrase} unless ($hs{phrase});

            if( $hs{numword} ) {
                if(! exists $phrases_hash{$hs{phrase}}){
                    $phrases_hash{$hs{phrase}} = 1;
                    $hs{position} = $ph->{place};
                    $hs{statusPhrase} = $ph->{statusPhrase};
                }
            }
        }
        $hs{place} ||= PlacePrice::get_guarantee_entry_place();

        push @phrases, \%hs;
        
    }

    my $result = {phrases => \@phrases, errors => [map {html2string($_)} @result]};

    # ретаргетинг (retargeting_conditions_id - это ret_cond_id)
    if (exists $form->{"retargeting_conditions_id$suffix"} && exists $OPT{all_retargeting_conditions}) {
        my $retargeting_conditions_ids = $form->{"retargeting_conditions_id$suffix"};

        my @ids_retargetings = grep {$OPT{all_retargeting_conditions}->{$_}}
                               split /\s*,\s*/, ($retargeting_conditions_ids // '');
        $result->{retargetings} = [
            map {
                {
                    ret_cond_id => $_
                    , place => 0
                    , is_suspended => 0
                }
            }
            @ids_retargetings
        ];
    }

    # копирование/перенос условий ретаргетинга (ids_retargetings - это ret_id из mediaplan_bids_retargeting)
    if ($form->{"ids_retargetings$suffix"} && $form->{cid}) {
        my @ids_retargetings = split /\s*,\s*/, $form->{"ids_retargetings$suffix"};

        # проверяем, что все условия принадлежат нашей кампании
        my $ret_cond_id = get_one_column_sql(PPC(cid => $form->{cid}), ["select mbr.ret_cond_id
                                                    from mediaplan_bids_retargeting mbr
                                                      join mediaplan_banners mb using(mbid)
                                                   ", where => {'mbr.ret_id' => \@ids_retargetings, 'mb.cid' => $form->{cid}}
                                                  ]) || [];
        $result->{retargetings} = [
            map {
                {
                    ret_cond_id => $_
                    , place => 0
                    , is_suspended => 0
                }
            }
            @$ret_cond_id
        ] if @$ret_cond_id;
    }

    return $result;
}

sub calc_mediaplan_pages {
    
    
    my ($cid, $on_page, $options) = @_;

    my $banners = get_mediaplan_banners_count($cid, $options);
    my $page = ref $options->{page} ? ${$options->{page}} : undef;
    my $pages = int(1 + ($banners - 1) / $on_page);
    if ($page) {
        if ($page == -1) {
            $page = $pages
        } elsif ($page > $pages) {
            $page = 1
        }
    }
    
    return {
        total_banners => $banners,
        pages => $pages,
        ($page ? (
            current_page => $page,
            limit => $on_page, 
            offset => ($page - 1) * $on_page
        ) : ())
    }
}

=head2 get_user_mediaplan

    $options = {
        pages
        no_forecast
        get_changes
        bids
        search_banner
        client_nds => # значение НДС в процентах для клиента-владельца кампании
                     если указано, все денежные значения будут скорректированы к без-НДСному варианту
        client_discount # текущая скидка клиента в процентах
                          если указан, total и sum будут с учётом скидочного бонуса (сумма которого вернётся в поле bonus)
        ClientID # Клиент для определения транслокального дерева для интерфейса
        is_api # 0|1 вызов из апи, для использования дерева регионов для api
    };
    $mediaplan = get_user_mediaplan($uid, $cid, $options);

    TODO: на добрую половину get_user_mediaplan дублирует Common::get_user_camp. Надо заменить дублирующийся код на соответствующий вызовы.
    TODO: но предварительно придётся развязать зависимости между Common и Mediaplan (в Common подключается Mediaplan, т.е. из Mediaplan нельзя вызывать функции Common).
    TODO: проще всего отселить get_user_mediaplan и accept_optimized_banners в отдельный высокоуровневый модуль сродни EditCamp.

=cut

sub get_user_mediaplan
{
    my ($uid, $cid, $options) = @_;

    $options ||= {};
    return undef unless $cid =~ m/^\d+$/;

    my $vars = get_camp_info($cid, $uid, client_nds => $options->{client_nds}, client_discount => $options->{client_discount});
    return undef unless $vars && $vars->{cid};

    for my $f (qw/currency/){
        # TODO правильнее делать через  delete, но пока так сломается шапка в просмотре медиаплана
        $vars->{campaign}->{$f} = $vars->{$f} if exists $vars->{$f}; 
    }
    $vars->{campaign}->{timetarget_coef} = TimeTarget::timetarget_current_coef($vars->{timeTarget}, $vars->{timezone_id});

    for my $field (qw/sum sum_spent total/) {
        $vars->{$field} = round2s($vars->{$field}) unless defined $vars->{$field};
    }

    $vars->{ctr} = $vars->{shows} ? sprintf("%.2f", $vars->{clicks} / $vars->{shows} * 100)  : 0;
    $vars->{av} = $vars->{clicks} ? sprintf("%.2f", $vars->{sum_spent} / $vars->{clicks}) : "0.00";

    Campaign::convert_dates_for_template($vars, keep_source_data => 1, with_old_start_date_format => 1);

    if ($vars->{day_budget} && $vars->{day_budget} > 0 && $vars->{OrderID} && $vars->{OrderID} > 0) {
        $vars->{spent_today} = Stat::OrderStatDay::get_order_spent_today($vars->{OrderID});
    }

    $vars->{status} = CalcCampStatus( $vars );
    $vars->{strategy} = detect_strategy($vars);

    $vars->{money_type} = check_block_money_camp($vars) ? 'blocked' : 'real';

    hash_merge $vars, count_campaign_items($cid);

    $vars->{optimal_banners_on_page} = $options->{optimal_banners_num} || $Settings::DEFAULT_BANNERS_ON_PAGE;
    $vars->{banners_on_page} = $options->{banners_num} || $vars->{banners_per_page} || $vars->{optimal_banners_on_page};

    my $page_lnk = ref $options->{page} ? ${$options->{page}} : undef;
    hash_copy \my %banners_options, $options, qw/no_forecast get_changes/;
    my $pager = calc_mediaplan_pages($cid, $vars->{banners_on_page}, $options);
    $vars->{pages_num} = $pager->{pages};
    $vars->{all_banners_num} = $vars->{tabclass_search_count} = $pager->{total_banners};
    my $banners = get_mediaplan_banners($cid, hash_merge $options, hash_cut $pager, qw/limit offset/);

    $vars->{arr} = $vars->{media} = $banners if @$banners;
    
    # определение ошибок в медиаплане        
    my ($error_code, $error_text) = check_mediaplan($cid);
    $vars->{accept} = { accept_ok => $error_code ? 0 : 1, error_code => $error_code, error_text => $error_text };
    
    $vars->{banners_have_source_bid} = get_one_field_sql(PPC(cid => $cid), "SELECT 1 FROM mediaplan_banners WHERE cid = ? and source_bid > 0 LIMIT 1", $cid );    
    
    # закончили проверять ошибки

    return $vars;
}

=head2 check_mediaplan($cid)

    Определение ошибок в медиаплане (не заполнены необходимые поля)
    
        $cid - номер кампании
    Результат
        ($error_code, $error_text)
        
        $error_code - код ошибки или undef если ошибок нет
        $error_text - текстовое описание ошибки

=cut

sub check_mediaplan {
    
    my $cid = shift;

    my $banners = get_all_sql(PPC(cid => $cid),
        "SELECT
            mb.mbid,
            IF(mb.title IS NULL OR mb.title = '' OR mb.body IS NULL OR mb.body = ''
                OR mb.geo IS NULL OR mb.geo = '', mb.mbid, NULL) AS error_1,
            IF((mb.href IS NULL OR mb.href = '') AND (vc.phone IS NULL OR vc.phone = ''), mb.mbid, NULL) AS error_3
        FROM mediaplan_banners mb
            LEFT JOIN vcards vc USING(vcard_id)
        WHERE mb.cid = ?", $cid);

    my %errors = (error_1 => 0, error_2 => 0, error_3 => 0);
    my $next_mbids = [];
    foreach my $banner (@$banners) {
        push @$next_mbids, $banner->{mbid} unless $banner->{error_1} || $banner->{error_3};
        $errors{error_1}++ if $banner->{error_1};
        $errors{error_3}++ if $banner->{error_3};
    }
    
    for my $table (qw/mediaplan_bids mediaplan_bids_retargeting/) {
        last unless @$next_mbids;
        my $mbids = get_one_column_sql(PPC(cid => $cid), [
            "SELECT mbid FROM $table",
            WHERE => {mbid => $next_mbids},
            "GROUP BY mbid" 
        ]) || [];
        $next_mbids = xminus($next_mbids, $mbids);
    }
    $errors{error_2} = scalar @$next_mbids;
    
    my $error_code = first { $errors{"error_${_}"} > 0 } (1, 3, 2);
    my $error_text;
    if ($error_code) {
        
        my $quantity = $errors{"error_${error_code}"};
        if ($error_code == 1) {
            $error_text = get_word_for_digit( $quantity,
                iget('Медиаплан содержит %s баннер с незаполненными полями', $quantity),
                iget('Медиаплан содержит %s баннера с незаполненными полями', $quantity),
                iget('Медиаплан содержит %s баннеров с незаполненными полями', $quantity),
            );
        } elsif ($error_code == 2) {
            $error_text = get_word_for_digit( $quantity,
                iget('Медиаплан содержит %s баннер без ключевых фраз', $quantity),
                iget('Медиаплан содержит %s баннера без ключевых фраз', $quantity),
                iget('Медиаплан содержит %s баннеров без ключевых фраз', $quantity),
            );
        } elsif ($error_code == 3) {
            $error_text = get_word_for_digit( $quantity,
                iget('Медиаплан содержит %s баннер без ссылки на сайт, либо телефона', $quantity),
                iget('Медиаплан содержит %s баннера без ссылки на сайт, либо телефона', $quantity),
                iget('Медиаплан содержит %s баннеров без ссылки на сайт, либо телефона', $quantity),
            );
        }
    }

    return ($error_code, $error_text);
}

=head2 get_mediaplan_banners_count

    Возвращает количество баннеров медиаплана на кампанию или из отобранных options->{bids}.
    Не объединено с get_mediaplan_banners, потому что иногда необходимо отдельно узнать количество.
    Важно, чтобы условия where в данной функции и get_mediaplan_banners совпадали.

=cut
sub get_mediaplan_banners_count($;$) {
    my ($cid, $options) = @_;

    return get_mediaplan_banners_count_mass([$cid], $options)->{$cid} || 0;
}


=head2 get_mediaplan_banners_count_mass

    Возвращает количество баннеров медиаплана на кампанию или из отобранных options->{bids}.
    Не объединено с get_mediaplan_banners, потому что иногда необходимо отдельно узнать количество.
    Важно, чтобы условия where в данной функции и get_mediaplan_banners совпадали.

=cut
sub get_mediaplan_banners_count_mass {
    my ($cids, $options) = @_;
    return {}  if !@$cids;

   $options = {} unless defined $options;

   my $where = {cid => $cids};
   hash_merge $where, $options->{search_banner} if $options->{search_banner};

   my $bids = [grep {is_valid_id($_)} @{$options->{bids} || []}];
   $where->{mbid} = $bids  if @$bids;

   my $count_by_cid = get_hash_sql(PPC(cid => $cids), [
           "SELECT cid, COUNT(*)
           FROM mediaplan_banners mb",
           where => $where,
           'GROUP BY cid',
       ]);
   return $count_by_cid;
}


=head2 get_mediaplan_banners

    Достает из БД все баннеры для медиаплана.
    Параметры:
        cid        номер кампании, из которой выбрать объявление
        options    опции:
                bids        - уточненные номера баннеров, которые надо выбрать. Если не указано, тогда выбираются все баннеры.
                no_forecast - не ходить в ADVQ за прогнозом
                get_changes - вычислять изменения, совершенные медиапланнером.
                limit       - ограничение по количеству выбираемых баннеров (аналог MySQL limit)
                offset      - с какого баннера по счету начинать выборку (аналог MySQL offset)
                search_banner-поиск баннера

    Важно, чтобы условия where в данной функции и get_mediaplan_banners_count совпадали.

    На выходе:
        указатель на список баннеров медиаплана

    Типичное использование: 
        $mediaplan_banners = get_mediaplan_banners($cid, $options);

    Параметры:
        no_phrases - получить только тексты
        phrases_db_only - фразы только из базы, никуда больше не ходить, и объект фразы ничем лишним не забивать
        no_forecast - очевидно, не получать прогноз, но что-то странное все же происходит
        ClientID - Клиент для определения транслокального дерева для интерфейса
        is_api - 0|1 вызов из апи, для использования дерева регионов для api
        divide_href_protocol - разделить ссылки на протокол и саму ссылку
        unpack_phone - распаковать телефон на отдельные поля

=cut 

sub get_mediaplan_banners($;$) {

        my ($cids, $options) = @_;

        $options = {} unless defined $options;
        my $bids = [grep {is_valid_int($_)} @{$options->{bids} || []}];
        
        if (!ref($cids)) {
            $cids = [$cids];
        }
        $cids = [grep {is_valid_id($_)} @{$cids || []}];

        return [] unless scalar @$cids || scalar @$bids;

        my ($where, $choose_shard);
        $where->{'c.statusEmpty'} = 'No';
        $where->{'c.type__in'} = ['text','geo'];

        $where->{'mb.cid'} = $choose_shard->{cid} = $cids if @$cids;

        hash_merge $where, $options->{search_banner} if $options->{search_banner};

        # Возможна ситуация, когда $options->{search_banner} тоже имеет поле mbid. 
        # В данном случае считаем вручную установленные options->{bids} важнее. Перезаписываем.
        $where->{mbid} = $choose_shard->{mediaplan_bid} = $bids if @$bids;

        my %shard = choose_shard_param($choose_shard, [qw/cid mediaplan_bid/]);

        my $reserve_field = ($options->{get_changes})?', mbr.data_json':'';
        my $reserve_from = ($options->{get_changes})?'left join mediaplan_banners_original mbr using(mbid)':'';

        my $banners_sql = "SELECT mb.mbid, mb.title, mb.title_extension, mb.body, mb.href, mb.domain, mb.geo, mb.cid,
                           mb.type AS banner_type,
                           vc.country, vc.city, vc.phone, vc.name, vc.contactperson, vc.worktime, vc.street, vc.house, vc.build, vc.apart, vc.metro, vc.geo_id,
                           vc.im_client, vc.im_login, vc.extra_message, vc.contact_email, vc.org_details_id, mb.vcard_id,
                           od.ogrn,
                           mb.statusShowsForecast, if(DATEDIFF(NOW(),timeShowsForecast)>6,1,0) as ifFiredShowsForecast,
                           mb.source_bid, mb.source_pid, mb.sitelinks_set_id
                           , p.group_name
                           , a.map_id, a.map_id_auto, a.precision, a.kind
                           , CONCAT_WS(',', maps.x, maps.y) as manual_point
                           , CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) as manual_bounds
                           , CONCAT_WS(',', maps_auto.x, maps_auto.y) as auto_point
                           , CONCAT_WS(',', maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2) as auto_bounds
                           , mw.mw_text as banner_minus_words
                           , IFNULL(c.currency, 'YND_FIXED') AS currency
                           , bif.image_hash as image, bimp.name as image_name
                           , bif.mds_group_id as mds_group_id
                           , IFNULL(fd.filter_domain, mb.domain) filter_domain
                           , (co.fairAuction = 'Yes') AS fairAuction
                           , co.minus_words AS campaign_minus_words
                           , c.OrderID
                           , FIND_IN_SET('no_extended_geotargeting', c.opts)>0 as no_extended_geotargeting
                           "
                           . ", ". Direct::Model::Creative->get_db_columns(perf_creatives => 'perfc', prefix => 'perfc_')                           
                           ."
                           $reserve_field
                       FROM mediaplan_banners mb 
                           INNER JOIN campaigns c ON c.cid = mb.cid
                           LEFT JOIN camp_options co ON c.cid = co.cid
                           left join users u ON c.uid = u.uid
                           left join vcards vc on vc.vcard_id = mb.vcard_id
                           LEFT JOIN org_details od ON vc.org_details_id = od.org_details_id
                           left join addresses a on vc.address_id = a.aid
                           left join maps on a.map_id = maps.mid
                           left join maps maps_auto on a.map_id_auto = maps_auto.mid
                           left join minus_words mw on mw.mw_id = mb.mw_id
                           left join banner_images bim on (bim.bid = mb.source_bid and bim.statusShow = 'Yes')
                           left join banner_images_pool bimp on bimp.image_hash = bim.image_hash AND bimp.ClientID = u.ClientID
                           left join banner_images_formats bif on bif.image_hash = bim.image_hash
                           left join phrases p ON mb.source_pid = p.pid
                           left join filter_domain fd on fd.domain = mb.domain
                           left join banners_performance bp on bp.bid = mb.source_bid
                           left join perf_creatives perfc on perfc.creative_id = bp.creative_id
                           $reserve_from";

        # limit условие
        my %overshard_params = (order => 'mbid:num');
        if ($options->{limit}) {
            $overshard_params{limit} = $options->{limit};
            $overshard_params{offset} = $options->{offset};
        }

        my $banners = get_all_sql(PPC(%shard), [$banners_sql, where => $where, "ORDER BY mbid"]) || [];
        $banners = overshard (%overshard_params, $banners);

        my $all_reserved_phrases = {};

        # для баннеров медиаплана одной кампании можно вернуть изменения
        # TODO: сделать удобоваримо
        if ($options->{get_changes}) {
            my $tmp_reserves = get_all_sql(PPC(%shard), ["SELECT cid, data_json from mediaplan_banners_original", where => {cid => $cids}]);

            my $reserves;
            foreach my $tmp (@$tmp_reserves) {
                push @{$reserves->{$tmp->{cid}}}, $tmp->{data_json};
            }

            foreach my $cid (keys %$reserves) {
                $all_reserved_phrases->{$cid} = {};
                hash_merge $all_reserved_phrases->{$cid}, {map {$_->{id} => $_->{phrase}} map {@{decode_json_and_uncompress($_)->{Phrases}}} @{$reserves->{$cid}}};
            }
        }

        # fetch banners
        my @media = ();
        my (@sl_sets_ids, @source_pids);
        foreach (@$banners) {
            push @sl_sets_ids, $_->{sitelinks_set_id} if $_->{sitelinks_set_id}; 
            push @source_pids, $_->{source_pid} if $_->{source_pid};
        }
        @sl_sets_ids = uniq @sl_sets_ids;
        @source_pids = uniq @source_pids;
        my $sitelinks_sets = Sitelinks::get_sitelinks_by_set_id_multi(\@sl_sets_ids);
        my $banners_qty = {};
        if (@source_pids) {
            $banners_qty = get_hash_sql(PPC(%shard), ["SELECT pid, count(*) FROM banners",
                                                    WHERE => {pid => \@source_pids},
                                                    "GROUP BY pid"]);
        }
 
        my $cache = {};
        foreach my $banner (@$banners) {
            $banner->{predefined_shows} = 1;
            html2string($banner->{$_}) for qw/title title_extension body/;

            $banner->{banners_quantity} = $banners_qty->{$$banner{source_pid}} || 1;
            $banner->{banner_with_phone} = $banner->{with_ci} = $banner->{phone} ? 1 : 0;
            $banner->{banner_with_href} = $banner->{has_href} = ($banner->{href} || !$banner->{with_ci}) ? 1 : 0;
            hash_merge $banner, divide_href_protocol($banner->{href}) if ($options->{divide_href_protocol});

            if (!defined $banner->{domain} || ( defined $banner->{domain} && $banner->{domain} eq "" ) ) {
                $banner->{domain} = get_host($banner->{href});
            }

            if ($banner->{sitelinks_set_id}) {
                $banner->{sitelinks} = $sitelinks_sets->{$banner->{sitelinks_set_id}};
            }

            $banner->{geo} = GeoTools::substitute_temporary_geo($banner->{geo});

            $banner->{is_mediaplan} = 1;
            $banner->{is_bs_rarely_loaded} = 0;

            foreach (qw/banner_minus_words campaign_minus_words/) {
                $banner->{$_} = MinusWordsTools::minus_words_str2array($banner->{$_});
            }

            #TODO: кажется, что имеет смысл не перемешивать в кучу поля из визитки и поля баннера. 
            #надо доделать клиентскую часть, чтобы она умела работать в полем vcard, а не с набором разбросанных полей на уровне баннера.
            #сейчас же это выделяется для последующего сравнения визиток на странице просмотре медиаплана.
            $banner->{vcard} = hash_cut $banner, @$VCards::VCARD_FIELDS_FORM if $banner->{vcard_id};
            if ($options->{unpack_phone} && $banner->{vcard}->{phone}) {
                hash_merge $banner->{vcard}, parse_phone($banner->{vcard}->{phone});
            }

            unless ($options->{no_phrases}) {

                my $phrases = get_mediaplan_phrases(mbid => $banner->{mbid});

                # TODO: избавиться от дублирования (оставить Phrases)
                $banner->{phrases} = $banner->{Phrases} = $phrases;
                $banner->{bid} = $banner->{mbid};

                my @banner_phrases = @{$banner->{Phrases}};

                $banner = hash_merge $banner, add_banner_template_fields($banner, \@banner_phrases);
                if ($options->{no_forecast}) {
                    # Берутся ставки из БК.
                    # Считаем цены в крутилке
                    foreach my $ph ( @{$banner->{Phrases}} ) {
                        $banner->{OrderID} ||= $ph->{orderid};
                        $ph->{price} = Currencies::get_currency_constant($banner->{currency}, 'MAX_PRICE');
                        $ph->{phr} = $ph->{phrase};
                    }
                }
            }

            if ($options->{get_changes}) {
                $banner->{changes} = compare_banners(clear_banner_for_compare($banner), decode_json_and_uncompress($banner->{data_json}), $all_reserved_phrases->{$banner->{cid}});
                # Удаляем поле, потому что оно в конце концов в TT идет и там только замусоривает все.
                delete $banner->{data_json};
            }

            if ($banner->{perfc_creative_id}) {
                my $creative = Direct::Model::Creative::Factory->create($banner, \$cache, prefix => 'perfc_');
                if ($creative->creative_type eq 'video_addition') {
                    $banner->{video_resources} = $creative->to_template_hash();
                }
            }
        }

        unless ($options->{no_phrases} || $options->{phrases_db_only}) {
            # достаем условия ретаргетинга
            Retargeting::get_retargetings_into_mediaplan_banners($banners);

            Models::AdGroup::update_phrases_shows_forecast_mediaplan($banners);

            if ($options->{no_forecast}) {
                trafaret_auction($banners, { calc_rotation_factor => 1 } );            
            } else {
                forecast_calc($banners);
            }
        }

        if (! $options->{is_api} && $options->{ClientID}) {
            # модифицируем geo в самом конце, т.к. от него зависят торги и прогноз
            my $translocal_params = { ClientID => $options->{ClientID} };
            for my $banner (@$banners) {
                $banner->{geo} = GeoTools::modify_translocal_region_before_show($banner->{geo}, $translocal_params);
                $banner->{geo_names} = get_geo_names( $banner->{geo} );
                $banner->{geo_ids} = $banner->{geo};
            }
        }

        return $banners;
}

=head2 get_mediaplan_phrase

    Возвращает объект фразы для медиаплана по переданному id фразы

=cut
sub get_mediaplan_phrase {
    my ($cid, $id) = @_;
    return unless $id && $cid;
    my $phrases = get_mediaplan_phrases(id => $id, cid => $cid);
    return $phrases->[0];
}

=head2 get_mediaplan_phrases

    Возвращает список объектов фраз(со всеми необходимыми сопутствующими значениями) для медиаплана по переданному условию where.
    Вцелом where это хеш, который должен увдолетворять аналогичному параметру во многих функциях DBTools.
    В контексте медиапланов это хеш с одним или несколькими ключами: id, mbid, cid и т.п. по которым будет происходить отбор.

=cut
sub get_mediaplan_phrases {
    my %where = @_; 

    my %shard = choose_shard_param({mediaplan_bid => $where{mbid}, cid => $where{cid}}, [qw/mediaplan_bid cid/]);
    my $phrases = get_all_sql(PPC(%shard), ["SELECT id, phrase, numword, place, statusPhrase, cid, mbid, showsForecast as shows, phraseIdHistory,
                             statusModerate, statusModerate = 'No' as declined, lowCtr = 'Yes' as lowCtr, is_suspended",
                            "FROM mediaplan_bids", where=>\%where]);
    $_->{place} = PlacePrice::set_new_place_style($_->{place}) foreach @{$phrases};

    return $phrases;
}


=head2

    Утверждение медиаплана: перенос баннеров медиаплана в список обычных баннеров.

=cut
sub accept_mediaplan
{
    my ($cid, $procent, $strategy, $accept_method, $mediaplan, $UID) = @_;

    $procent = 0 if !defined $procent || $procent !~ /^\d+$/;
    my $uid  = get_owner(cid => $cid);
    my $OrderID = get_orderid(cid => $cid);
    my $from_cid  = $mediaplan->{cid} || $cid;   
    my $client_id = get_clientid(uid => $uid);
    my (%replaced_banners_bids, %replaced_pids);

    #открываем лог-файл для записи изменения во фразах.
    my $log_phrases = LogTools::messages_logger("UpdateBannerPhrases");

    my (%exists_bids, %exists_pids); 
    if ($accept_method eq 'merge' || $accept_method eq 'replace') {
        my $banners = get_all_sql(PPC(cid => $cid), "SELECT b.bid, p.pid FROM banners b JOIN phrases p USING(pid) WHERE p.cid = ?", $cid);
        foreach (@$banners) {
            $exists_bids{$_->{bid}} = 1;
            $exists_pids{$_->{pid}} = 1;
        } 
    }
    
    my $camp_info = CampaignQuery->get_campaign_data(cid => $cid, [qw/strategy ContextPriceCoef currency opts/]);
    $camp_info->{currency} ||= 'YND_FIXED';
    my $context_coeff = $camp_info->{ContextPriceCoef};
    my $currency = $camp_info->{currency};
    my $ret_id_for_delete = {};
    my $camp_strategy = $camp_info->{strategy} = Campaign::campaign_strategy($cid);
    my $is_different_places =  $camp_strategy->{name} eq 'different_places';
    my $no_ext_geo = $camp_info->{opts} =~ /\bno_extended_geotargeting\b/;
    my @groups;

    foreach my $mb ( @{$mediaplan->{media}} ) {

        my $banner = {
            cid => $cid,
            uid => $uid,
            statusEmpty => "No",
            statusModerate => ($mediaplan->{save_to_draft} ? "New" : "Ready"),
            geo_id => $mb->{geo_id} || 0,   # это часть визитки
            currency => $currency,
            image => ( $mb->{source_bid} ? get_banner_image($mb->{source_bid}) : undef ),
            no_extended_geotargeting => $no_ext_geo,
            OrderID => $OrderID,
            is_bs_rarely_loaded => 0,
        };

        hash_copy $banner, $mb, qw/title title_extension body href domain filter_domain geo source_bid sitelinks/,
                                qw/manual_point manual_bounds OrderID/,
                                @$VCards::VCARD_FIELDS;
        hash_merge $banner, VCards::parse_phone($banner->{phone});

        $banner->{map} = CommonMaps::check_address_map(
            $banner, 
            { ClientID => $client_id }
        );

        my $old_phrases;
        my $source_pid = $mb->{source_pid} || get_pid(bid => $mb->{source_bid});
        if ($source_pid) {
            $old_phrases = { map { $_->{phraseIdHistory} => $_ } @{BS::History::get_keywords_with_history({ pid => $source_pid })} };
        }

        my ($new_bid, $new_pid) = (0, 0);
        
        my $is_create_new_banner =
                $accept_method eq 'copy' || $accept_method eq 'replace'
                    || !$mb->{source_bid}
                    || $replaced_banners_bids{$mb->{source_bid}} # Если баннер с таким bid уже был заменен баннером из медиплана
                    || !exists $exists_bids{$mb->{source_bid}} # если исходный баннер был удален из кампании после создания медиаплана
                    || ($mb->{source_pid} && $replaced_pids{$mb->{source_pid}});
        # Не совсем очевидная вешь: утвердить медиаплан можно и в кампанию к другому клиенту, но в этом случае 
        # $accept_method будет принудительно прописана как 'copy', что приведет is_create_new_banner к true.
        if ($is_create_new_banner) {
            
            if ($mb->{source_pid}
                && $exists_pids{$mb->{source_pid}}
                && !$replaced_pids{$mb->{source_pid}}
                && !$exists_bids{$mb->{source_bid}}) {
            
                $replaced_pids{$mb->{source_pid}} = 1;
                $new_pid = $banner->{pid} = $mb->{source_pid};
            } else {
                # группа создаётся в update_phrases
                $new_pid = $banner->{pid} = 0;
            }
            $new_bid = $mb->{banner_type} eq 'desktop' ? create_desktop_banner($banner, $uid)->{bid} : create_mobile_banner($banner, $uid)->{bid};                                    
        } else {
            $banner->{bid} = $new_bid = $mb->{source_bid};
            $new_pid = $banner->{pid} = $source_pid;
            $replaced_banners_bids{$new_bid} = 1;
            $replaced_pids{$mb->{source_pid}} = 1 if $mb->{source_pid} && $exists_pids{$mb->{source_pid}};
            if ($mb->{banner_type} eq 'desktop') {
                update_desktop_banners([$banner], $uid, {ignore_callouts => 1});
            } else {
                update_mobile_banner($banner, $uid);
            }

            update_geo($new_pid, $banner->{geo}, $uid);
            do_delete_from_table(PPC(cid => $from_cid), 'bids', where => {pid => $banner->{pid}});
        }

        # TODO: плохо прерывать процесс
        error( iget("Не могу добавить баннер!") ) if !$new_bid;
        my @phrases_for_add_update;
        foreach my $mp ( @{$mb->{Phrases}} ) {

            my $old_phrase = $mp->{phraseIdHistory} && $old_phrases->{$mp->{phraseIdHistory}};
            my $price = 0;
            my $price_context = 0;

            if ($strategy eq 'current' && $old_phrase) {
                $price = $old_phrase->{price};
                $price_context = $old_phrase->{price_context} if $is_different_places;;
            } else {
                if ($strategy eq 'media_phrases_prices') {
                    local $banner->{phrases} = [$mp];
                    trafaret_auction([$banner], {});
                    my $place_data = PlacePrice::get_data_by_place($mp, $mp->{place});
                    $price = $place_data->{bid_price} / 1e6;
                } elsif ( $strategy eq 'media_min_prices' ) {
                    $price = Currencies::get_currency_constant($currency, 'MIN_PRICE');
                } elsif ( $strategy eq 'media_all_block_price' ) {
                    $price = $mediaplan->{media_all_block_price_bid};
                } else {
                    $procent = 0 if $strategy ne 'media_banners_prices';
                    my $place_data = PlacePrice::get_data_by_place($mp, $mp->{place});
                    $price = ($place_data->{bid_price} + ($place_data->{bid_price}/100) * $procent) / 1e6;
                }
                $price = currency_price_rounding($price, $currency, up => 1);
                $price_context = phrase_price_context($price, $context_coeff, $currency) if $is_different_places;
            }

            #для случая когда история возникла не в результате прямого копирования баннера в медиаплан
            if ($mp->{phraseIdHistory}) {
                my %phraseIdHistory = BS::History::parse_bids_history($mp->{phraseIdHistory});
                my $old_bid = (keys(%{$phraseIdHistory{banners}}))[0];
                $phraseIdHistory{banners} = {$new_bid => $phraseIdHistory{banners}->{$old_bid}};
                $mp->{phraseIdHistory} = BS::History::serialize_bids_history(\%phraseIdHistory);
            }

            # Записываем изменение в лог.
            if ($old_phrase) {
                $log_phrases->out(sprintf("edit phrase (from mediaplan): UID: %s\tcid: %s\tbid: %s\tphraseIdHistory: %s\told_value: %s\tnew_value: %s", 
                                  $UID, $cid, $new_bid, $mp->{phraseIdHistory}, $old_phrase->{phrase}, $mp->{phrase})) 
                    if ($old_phrase->{phrase} ne $mp->{phrase});
            } else {
                $log_phrases->out(sprintf("new phrase (accept mediaplan): UID: %s\tcid: %s\tbid: %s\tnew_value: %s\tphraseIdHistory: %s",
                        $UID, $cid, $new_bid, $mp->{phrase}, $mp->{phraseIdHistory} || '-'));
            }
            push @phrases_for_add_update, {
                            bid => $new_bid, 
                            cid => $cid, 
                            uid => $uid, 
                            phrase => $mp->{phrase},
                            place => $mp->{place} || PlacePrice::get_premium_entry_place(),
                            price => $price,
                            ($is_different_places ? (price_context => $price_context) : ()),
                            autobudgetPriority => 3,
                            phraseIdHistory => $mp->{phraseIdHistory},
                            is_suspended => $mp->{is_suspended},
                        };
        }
        my $geo = refine_geoid($banner->{geo} || '0', undef, {ClientID => $client_id});
        (undef, $new_pid) = update_phrases($new_pid, $cid, $mediaplan->{save_to_draft} ? 'New' : 'Ready', 'No', $uid, $geo, undef, 'Yes');
        do_update_table(PPC(bid => $new_bid), 'banners', {pid => $new_pid}, where => {bid => $new_bid});
        push @groups, {pid => $new_pid, banners => [$banner]};
        if (@phrases_for_add_update) {
            $_->{pid} = $new_pid foreach @phrases_for_add_update;
            mass_add_update_phrase($camp_info, \@phrases_for_add_update);
        }

        log_cmd({
            cmd => '_new_banner',
            uid => $uid,
            UID => $UID,
            cid => $cid,
            pid => $new_pid,
            bid => $new_bid,
            banner_type => $mb->{banner_type},
            where_from => "mediaplan",
        });

        MinusWords::save_banner_minus_words($new_pid, $mb->{banner_minus_words});

        my %old_ret_cond_id2price_context;
        if ($source_pid) {
            %old_ret_cond_id2price_context =  map { $_->{ret_cond_id} => $_->{price_context} } @{ Retargeting::get_group_retargeting(pid => [$source_pid])->{$source_pid}}
        }
        # переносим условия ретаргетинга
        if ($accept_method eq 'merge') {
            do_delete_from_table(PPC(pid => $new_pid), "bids_retargeting", where => {pid => $new_pid});
        }
        if (ref($mb->{retargetings}) eq 'ARRAY' && @{$mb->{retargetings}}) {
            my $new_banner = {
                cid => $cid,
                # TODO adgroup: можно удалить bid => когда полностью перейдем на группы
                # (сейчас используется для логирования цен)
                bid => $new_bid,
                pid => $new_pid,
                geo => $banner->{geo} || '0',
                currency => $currency,
            };

            my $new_ret_cond_id = {};
            if ($from_cid != $cid) {
                # при принятии в другую кампанию нужно скопировать условия
                my $clients = get_cid2clientid(cid => [$from_cid, $cid]);
                $new_ret_cond_id = Retargeting::copy_retargetings_between_clients(
                    old_client_id => $clients->{$from_cid},
                    new_client_id => $clients->{$cid},
                    old_ret_cond_ids => [map {$_->{ret_cond_id}} @{$mb->{retargetings}}],
                );
            }

            for my $ret_row (@{$mb->{retargetings}}) {
                $ret_row->{ret_cond_id} = $new_ret_cond_id->{ $ret_row->{ret_cond_id} } || $ret_row->{ret_cond_id};

                my $price_context;
                if ($strategy eq 'current' && defined $old_ret_cond_id2price_context{$ret_row->{ret_cond_id}}) {
                    $price_context = $old_ret_cond_id2price_context{$ret_row->{ret_cond_id}};
                } else {
                    if ($strategy eq 'media_all_block_price' ) {
                        $price_context = $mediaplan->{media_all_block_price_bid};
                    } else {
                        # пока не используем показометр
                        # $price_context = Pokazometer::get_price_for_coverage_by_cost_context($Settings::DEFAULT_COVERAGE_FOR_RETARGETING, $banner->{geo} || '0');
                        $price_context = Currencies::get_currency_constant($currency, 'MIN_PRICE');
                    }

                    # Если посчитанная ставка становится больше максимально разрешенной ставки, то ставим максимально разрешенную.
                    my $max_price = Currencies::get_currency_constant($currency, 'MAX_PRICE');
                    $price_context = $max_price if $price_context > $max_price;
                }

                $ret_row->{price_context} = $price_context;

                push @{$new_banner->{retargetings}}, $ret_row;

                $ret_id_for_delete->{ $ret_row->{ret_id} } = 1;
                # так как в медиапланах используются свои ret_id (отличные от основных групп), то чтобы не путаться и не сохранять
                # с неправильными id удаляем. При сохранении будет им новый ret_id присвоен.
                delete $ret_row->{ret_id};

            }

            Retargeting::update_group_retargetings($new_banner, insert_only => 1, camp_strategy => $camp_strategy);
        }
    }

    if ($accept_method eq 'replace') {
        my @banners = map {+{bid => $_}} keys %exists_bids;
        Models::Banner::stop_banners(\@banners, {uid => $UID});
        BannersCommon::mass_archive_banners(bids => [keys %exists_bids]);
    }
    
    Models::AdGroup::save_group_params(\@groups);

    delete_mediaplan($from_cid);

    if (%$ret_id_for_delete) {
        do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids_retargeting', where => {ret_id => [keys %$ret_id_for_delete]});
    }
    
    # удаляем все заявки на "Первую помощь"
    close_request_first_aid($from_cid);
    
    return;    
}

=head2 delete_mediaplan (cid)

    Удаляет медиаплан для кампании с номером $cid.

=cut
sub delete_mediaplan
{
    my $cid = shift;
    my $mbids = get_one_column_sql(PPC(cid => $cid), ["SELECT mbid FROM mediaplan_banners", where => { cid => $cid }]) || [];
    # Визитки - кандидаты на потенциальное удаление
    my $vcards = get_one_column_sql(PPC(cid => $cid), ["SELECT DISTINCT vcard_id from mediaplan_banners", where => { cid => $cid }]);
    do_delete_from_table(PPC(cid => $cid), 'mediaplan_mod_reasons', where => { bid => $mbids});
    do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids', where => { cid => $cid});
    do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids_retargeting', where => {mbid => $mbids});
    do_delete_from_table(PPC(cid => $cid), 'mediaplan_banners', where => { cid => $cid});
    do_delete_from_table(PPC(cid => $cid), 'mediaplan_banners_original', where => { cid => $cid});
    # Баннеры из медиаплана уже удалены, если визитки ни к чему больше не привязаны, то их можно и удалить.
    delete_vcard_from_db($vcards) if $vcards;
    delete_shard(mediaplan_bid => $mbids) if @{$mbids || []};
}

=head2 validate_mediabanner

    Проверка медиапланного объявления. 
    Сильно похожа на validate_banner

=cut

sub validate_mediabanner
{
    ## no critic (Freenode::DollarAB)
    my ($b, %O) = @_;
    my @result;

    # strip bad symbols.
    smartstrip($b->{$_}, dont_replace_angle_quotes => 1) for qw/body title title_extension/;
    smartstrip($b->{href});
    # strip phrases

    # если в качестве заголовка или текста объявления нули
    push @result, iget('Недопустимый заголовок или текст объявления') if defined $b->{title} && !$b->{title}
                                                                            || defined $b->{title_extension} && !$b->{title_extension}
                                                                            || defined $b->{body} && !$b->{body};
                                                                            
    unless (defined $b->{banner_type} && $b->{banner_type} =~ /^(desktop|mobile)$/) {
        push @result, iget('Неверный тип баннера (разрешен мобильный или десктопный тип баннера)');
    }

    if (my $title_error = Models::Banner::validate_banner_title($b->{title}, can_be_empty => 1)) {
        push @result, $title_error;
    }

    if (my $title_extension_error = Models::Banner::validate_banner_title_extension($b->{title_extension})) {
        push @result, $title_extension_error;
    }

    if (my $body_error = Models::Banner::validate_banner_body($b->{body}, can_be_empty => 1)) {
        push @result, $body_error;
    }
    push @result, iget('Регионы указаны неверно') if ( ($b->{geo}) !~ m/[\d,]+/);

    if (ref $b->{sitelinks} eq 'ARRAY' && scalar @{$b->{sitelinks}}) {
        push @result, Sitelinks::validate_sitelinks_set($b->{sitelinks}, $b->{href}, ClientID => $O{ClientID});
    }

    if( defined $b->{href} ) {
        $b->{href} = clear_banner_href($b->{href}, $b->{url_protocol});
        push @result, validate_banner_href( $b->{href} );
        push @result, iget('Ссылка баннера не может указывать на Турбо-страницу')
            if Direct::Validation::TurboLandings::is_valid_turbolanding_href($b->{href});
    }
    my $validation_result = Direct::Validation::Keywords::validate_add_keywords_oldstyle($b->{Phrases} // []);
    unless ($validation_result->is_valid) {
        push @result, @{$validation_result->one_error_description_by_objects};
    }

    if (my $templ_res = validate_banner_template( $b )) {
        push @result, @{$templ_res};
    }

    if(    defined $b->{phone}&& $b->{phone} ne '' 
        || defined $b->{city} && $b->{city} ne '' 
        || defined $b->{country} && $b->{country} ne '' 
        || defined $b->{name} && $b->{name} ne ''
        || defined $b->{contactperson} && $b->{contactperson} ne '' 
        || defined $b->{country_code} && $b->{country_code} ne '' 
        || defined $b->{city_code} && $b->{city_code} ne '' 
        || defined $b->{ext} && $b->{ext} ne '' 
        || defined $b->{street} && $b->{street} ne ''
        || defined $b->{house} && $b->{house} ne ''
        || defined $b->{build} && $b->{build} ne '' 
        || defined $b->{apart} && $b->{apart} ne ''
        || defined $b->{metro} && $b->{metro} ne ''
#        || defined $b->{worktime} && $b->{worktime} ne '' 

        || defined $b->{im_client} && $b->{im_client} ne '' 
        || defined $b->{im_login} && $b->{im_login} ne '' 
        || defined $b->{extra_message} && $b->{extra_message} ne '' 
        || defined $b->{contact_email} && $b->{contact_email} ne '' 
        || defined $b->{ogrn} && $b->{ogrn} ne '' 

    ) {
        if (not $O{skip_contactinfo}) {
            my @result_ci = validate_contactinfo($b);
            push @result, @result_ci;
        }
    }
    # в медиапланах вполне допускается создание объявлений без КИ и ссылки
    # обязательность наличия этих данных проверяется при превращении медиаплана в кампанию

    return @result;
}


=head2 validate_mediaplan

    Проверка медиаплана на валидность.

=cut

sub validate_mediaplan
{
    my ($mediaplan, %O) = @_;
    
    return [] if !$mediaplan;
    
    my %errors;
    push @{$errors{'common'}}, iget('В медиаплане отсутствуют объявления') if !$mediaplan->{media} || !scalar @{$mediaplan->{media}};
        
    my $exists_banners_type = BannersCommon::get_banners_type(mediaplan => [grep { $_ } map { $_->{mbid} } @{$mediaplan->{media}}]);         

    my $new_pid_cnt = 0;
    my %seen_pid;
    foreach my $banner (@{$mediaplan->{media}}) {
        $new_pid_cnt++  if !$banner->{source_pid} || !$seen_pid{$banner->{source_pid}}++;
        if ($banner->{Phrases}) {
            $banner->{phrases} = join ',', map {$_->{phrase}} @{$banner->{Phrases}};
        }
        my $key = ($banner->{mbid})?$banner->{mbid}:'common';
        if ($O{use_multierrors_format}) {
            my ($banner_errors) = BannersCommon::validate_banner($banner, {
                use_multierrors_format => 1, ClientID => $mediaplan->{ClientID},
                is_mediaplan_banner => 1, exists_banners_type => $exists_banners_type
            });
            # так как позже планируется отказываться от плоских ошибок, здесь нужно будет просто удалить else ветку.
            $errors{$key} = $banner_errors->{$banner->{mbid} || 0};
        } else {
            my @banner_errors = BannersCommon::validate_banner($banner, {
                ClientID => $mediaplan->{ClientID}, is_mediaplan_banner => 1,
                exists_banners_type => $exists_banners_type
            });
            if (@banner_errors) {
                push @{$errors{$key}}, @banner_errors;
            }
        }
    }

    # для медиапланов с типом принятия "замена" проверяем, что не вылезем за лимит групп
    if (($O{accept_type} || 'merge') eq 'replace') {
        my $group_limit = Client::get_client_limits($mediaplan->{ClientID})->{banner_count_limit};
        my $current_pid_cnt = get_one_field_sql(PPC(cid => $mediaplan->{cid}), [
                'SELECT count(pid) FROM phrases',
                WHERE => {cid => $mediaplan->{cid}},
            ]);

        push @{$errors{'common'}}, iget('Достигнуто максимальное количество групп объявлений в кампании - %s', $group_limit)
            if $current_pid_cnt+$new_pid_cnt > $group_limit;
    }

    return \%errors;
}


sub validate_accept_mediaplan
{
    my ($rbac, $uid, $vars, $currency) = @_;

    my @result = ();

    if ($vars->{mark}) {
        if ($vars->{mark} !~ /^[2345]$/ ) {
            push @result, iget("Разрешённые оценки: 2, 3, 4, 5") ;
        }
    } else {
        push @result, iget("Не указана оценка за составленный медиаплан") ;
    }

    my $is_valid_to_cid = is_valid_id($vars->{to_cid});
    if( defined $vars->{to_camp} && $vars->{to_camp} eq 'other' ) {
        if ($vars->{client_login} && $vars->{to_cid}) {
            push @result, iget('Неверно введен номер заказа')  if !$is_valid_to_cid;
            
            my $client_uid = get_uid_by_login( $vars->{client_login} );
            push @result, iget('Это не ваш клиент')  if !$client_uid && $vars->{client_login}
                                                        || !rbac_is_owner( $rbac, $uid, $client_uid );
            
            my $is_right_user_camp = $is_valid_to_cid && (get_owner(cid => $vars->{to_cid}) == $client_uid);
            push @result, iget('Задано неверное соответствие клиента и кампании')  if !scalar @result &&
                                                                                      !$is_right_user_camp;
            
            if ($vars->{to_cid}) {
                if ( $is_right_user_camp && ! rbac_is_owner_of_camp($rbac, $uid, $vars->{to_cid})) {

                    push @result, iget('У вас нет прав на указанную кампанию') ;
                }
            } else {
                push @result, iget('Не введен номер кампании для утверждения медиаплана');
            }

        } else {
            push @result, iget('Не введен логин клиента')  if !$vars->{client_login};
            push @result, iget('Не введен номер заказа') if !$vars->{to_cid};
        }
    } elsif( !$vars->{to_camp} || $vars->{to_camp} ne 'current' ) {
        push @result, iget('Неверно указана цель') ;
    }

    if ($vars->{strategy}) {
        if ($vars->{strategy} =~ /^(media_all_block_price|current|media_phrases_prices|media_min_prices|media_banners_prices)$/) {
            if ($vars->{strategy} eq 'media_all_block_price') {
                my $price = sprintf '%.2f', $vars->{media_all_block_price_bid};
                my $error = validate_phrase_price($price, $currency, dont_support_comma => 1);
                push @result, $error if $error;
            }
        } else {
            push @result, iget('Стратегия задана неверно');
        }
    } else {
        push @result, iget('Выберите стратегию');
    }

    return \@result;
}

=head2 validate_mediaplan_mail

    Проверка на валидность заявки на медиаплан.

=cut
sub validate_mediaplan_mail
{
    my $req = shift;
    my @res = ();
    
    push @res, iget('Не введен номер кампании') if !defined $req->{cid};
    push @res, iget('Не введен email назначения') if !defined $req->{mailto};
    push @res, iget('Не введен тип заявки') if !defined $req->{typeMedia};
    push @res, iget('Указан не разрешённый тип заявки') if defined $req->{typeMedia} && !defined $MediaplanOptions::REQUEST_TYPE{$req->{typeMedia}};
    push @res, iget('Не указана рекомендация для подбора текста объявления') if defined($req->{typeMedia}) && $req->{typeMedia} eq 'text' && !defined $req->{textTypeMedia};
    push @res, iget('Не введен тип текстов') if !defined $req->{texttype};
    push @res, iget('Не введен приоритет') if !defined $req->{priority};
    push @res, iget('Не введен регион') if !defined $req->{geo} && $req->{geo_type} == 1;
    push @res, iget('Не введен бюджет') if !defined $req->{budget};
    push @res, iget('Не введена цель кампании') if !defined $req->{target_camp};
    push @res, iget('Не введен рассчитываемый объём трафика') if !defined $req->{position};
    push @res, iget('Не указана целевая аудитория') if !defined $req->{target};

    push @res, iget('Неверно введен email менеджера') if !is_valid_email( $req->{manager_email} );
    push @res, iget('Неверно введен email назначения') if !is_valid_email( $req->{mailto} );
    push @res, iget('Неверно введен email для копии письма') if defined($req->{copy_email}) && ! is_valid_email($req->{copy_email});
    push @res, iget('Адрес для отправки заявки должен быть в домене @yandex-team') if defined($req->{copy_email}) && is_valid_email($req->{copy_email}) && ! ($req->{copy_email} =~ /.*\@yandex\-team\.ru/i);

    return @res ? join( "\n", @res) : undef;
}

=head2 update_mediaplan_stats
   

=cut

sub update_mediaplan_stats
{
    my ($rbac, $cid, $UID, $login_rights, $has_mediaplan_banners) = @_;

    my $optimizing_status = get_one_field_sql(PPC(cid => $cid), ["SELECT COUNT(*) FROM optimizing_campaign_requests", where =>{cid => $cid, status=>['New', 'InProcess', 'Ready'] }]);
    process_FA_request_on_set_servicing($rbac, $cid) if $optimizing_status;

    if(! $optimizing_status){
    
        my $muids = get_one_line_sql(PPC(cid => $cid), ["SELECT ManagerUID, MediaUID FROM mediaplan_stats",
                                                        where => {cid => $cid, accepted => 'No'}, "LIMIT 1"]);
        
        my $mediaplan_status = get_one_field_sql(PPC(cid => $cid), "select mediaplan_status from camp_options where cid=?", $cid);
        my $new_status;
        if ($mediaplan_status) {
            if ($mediaplan_status eq 'None') {
                $new_status = 'InProcess';
            } elsif ($mediaplan_status eq 'Complete') {
                $new_status = 'None' unless get_mediaplan_banners_count($cid); 
            }
        } else {
           $new_status = 'InProcess';
        }
        
        if ($new_status) {
            do_update_table(PPC(cid => $cid), 'camp_options', {mediaplan_status => $new_status}, where => {cid => $cid});
        }        
        
        if ( !defined $muids ) {
            #   запись для Медиаплана отсутствует, в любом случае создаём
            my $ManagerUID = rbac_is_scampaign($rbac, $cid)||0;
       
            do_insert_into_table(PPC(cid => $cid), 'mediaplan_stats', { mpid => get_new_id('mediaplan_mpid'), 
                                                            MediaUID => ($login_rights->{role} ne "media") ? 0 : $UID, 
                                                            ManagerUID => ($login_rights->{role} eq "manager") ? $UID : $ManagerUID,
                                                            cid => $cid, 
                                                            accepted => 'No', 
                                                            requested => 'No'});
        
        } elsif ( ($muids->{MediaUID} == 0 or !$has_mediaplan_banners) and $login_rights->{role} eq "media") {
            do_update_table(PPC(cid => $cid), 'mediaplan_stats', {MediaUID => $UID}, where => {cid => $cid, accepted => 'No'});
        } elsif ( $muids->{ManagerUID} == 0 and $login_rights->{role} eq "manager" ) {
            do_update_table(PPC(cid => $cid), 'mediaplan_stats', {ManagerUID => $UID}, where => {cid => $cid, accepted => 'No'});
        }
    }
}

=head2 get_mediaplan_stats

    Возвращает данные по текущему медиаплану (если есть)
    Параметры:
         cid -  номера кампании
=cut

sub get_mediaplan_stats($) {
    my $cid = shift;

    return get_mediaplans_stats([$cid])->{$cid};
}

=head2 get_mediaplans_stats

    Возвращает данные по текущему медиаплану (если есть)
    Параметры:
         cids -  номера кампаний
         rbac - указатель на RBAC-объект
         OPTIONS:
            short - сокращенная версия, только записи из таблицы mediaplan_stats
=cut

sub get_mediaplans_stats
{
    my ($cids, $rbac) = @_;

    my $mpids = get_one_column_sql(PPC(cid => $cids), ["SELECT max(mpid) FROM mediaplan_stats",
                    where => { cid => SHARD_IDS, accepted => 'No'}, "group by cid"]) || [];
    my $mediaplan_stats_fields_str = join ', ', 'ms.cid', map {"ms.".$_} @MEDIAPLAN_STATS_FIELDS;
    my $mediaplan_data = get_hashes_hash_sql(PPC(cid => $cids), ["SELECT $mediaplan_stats_fields_str, co.mediaplan_status
            FROM mediaplan_stats ms 
            left join camp_options co using (cid)",
            where => { 'ms.mpid' => $mpids }]) || {};
    # Если не передан rbac, то возвращаем упрощенный вариант данных.
    return $mediaplan_data unless ($rbac);

    # Если не передан rbac, то возвращаем упрощенный вариант данных.
    return $mediaplan_data unless ($rbac);

    my $optimizing_statuses = get_fa_statuses($cids);

    my $result;

    my $manager_uids = {};
    if (my @cids_to_check = grep {!$mediaplan_data->{$_}} @$cids) {
        $manager_uids = rbac_mass_is_scampaign($rbac, \@cids_to_check);
    }

    my $manager_logins;
    if (%$manager_uids) {
        $manager_logins = get_uid2login(uid => [values %$manager_uids]);
    }
    my @uids;
    foreach my $cid (@$cids) {
        push @uids, $mediaplan_data->{$cid}{ManagerUID};
        push @uids, $mediaplan_data->{$cid}{MediaUID};
    }
    my $logins = get_uid2login(uid => \@uids);

    foreach my $cid (@$cids) {
        my $mediaplan = $mediaplan_data->{$cid};
        if (%$mediaplan) {
            $mediaplan->{manager_login} = $logins->{$mediaplan->{ManagerUID}}
                if $mediaplan->{ManagerUID};
            $mediaplan->{mediaplanner_login} = $logins->{$mediaplan->{MediaUID}}
                if $mediaplan->{MediaUID};
                
            $result->{$cid} = $mediaplan;
        } else {
            $result->{$cid} = {
                cid => $cid,
                ManagerUID => $manager_uids->{$cid},
                manager_login => $manager_logins->{$manager_uids->{$cid}},
                mediaplan_status => 'None',
            };
        }
        if ($optimizing_statuses->{$cid}) {
            $result->{$cid}->{mediaplan_status} = 'FA';
        }
    }
    return $result;
}

sub get_fa_statuses
{
    my $cids = shift;
    return get_hashes_hash_sql(PPC(cid => $cids), [
        "SELECT cid, count(*) FROM optimizing_campaign_requests", 
        where =>{cid => SHARD_IDS, status=>['New', 'InProcess', 'Ready'] },
        "group by cid"
        ]);
}

sub get_common_geo_for_mediaplan($)
{
    my $cid = shift || 0;
    
    my $geo_ids_media = get_one_column_sql(PPC(cid => $cid), 'SELECT DISTINCT geo FROM mediaplan_banners WHERE cid = ?', $cid) || [];
    
    return scalar @{ $geo_ids_media } == 1 ? @{ $geo_ids_media }[0] : undef;
}

=head2 add_optimizing_request

    Добавляет заявку на Оптимизацию в БД.

=cut
sub add_optimizing_request
{
    my ($cid, $options) = @_;
    #Создаём копию хеша, чтобы можно было со спокойной совестью изменять его значения.
    my $params = defined($options)?{%{$options}}:{};

    $params->{request_id}     = get_new_id('opt_camp_request_id');
    $params->{cid}            = $cid;
    $params->{status}       ||= 'New';
    $params->{is_support}   ||= 0;
    $params->{is_automatic} ||= 0;
    $params->{req_type}     ||= 'FirstAid';
    $params->{CreaterUID}   ||= 0;
    $params->{create_time__dont_quote} ||= 'NOW()';

    # Все еще заполняем поля, которые больше не используем на всякий случай.
    # TODO: после релиза через некоторое время необходимо удалить следующий код:
    # Одновременно с чисткой полей из таблицы optimizing_campaign_requests
    #------------------
    $params->{request_type} = ($params->{is_support}) ? 'Support' : 'Normal';    
    $params->{auto_request} = ($params->{req_type} eq 'Moderate') ? 'ModerateNo' : ($params->{is_automatic}) ? 'Automatic' : 'No';
    #------------------

    return do_insert_into_table(PPC(cid => $cid), 'optimizing_campaign_requests', $params);

}

=head2 accept_optimized_banners

    Согласие с Первой помощью: объявления из медиаплана переносим в обыкновенные

    Первоначально -- просто кусок из cmd_acceptOptimize_step2 
    + доделка для Ускоренных: прием медиаплана целиком без промежуточных шагов

    Параметры: 
        $uid, $cid, $camp_info, $mbanners

        $camp_info -- хеш с данными о кампании, полученный из Campaign::get_camp_info
        $mbanners -- хеш с информацией по принятым объявлениям/фразам. Не обязательный. 
            Если отсутствует, то принимаются медиаплан целиком, со ставками/приоритетами от медиапланера
            Если задан -- будут приняты только перечисленные объявления и фразы; ставки и приоритеты -- из него же

    Формат $mbanners примерный:
    {
      '96030' => {
                   # фраза
                   '1069037' => {
                                'autobudgetPriority' => undef,
                                'broker' => 'yes',
                                'rank' => '51612',
                                'showsForecast' => undef,
                                'val' => '6.33'
                              },
                   '1069043' => {
                                    ...
                                }
                 },
      '96031' => {
                    ...
                 }
    }

=cut
sub accept_optimized_banners {
    my ($UID, $uid, $cid, $m_bids, $m_phrases_ids, $m_retargetings) = @_;

    my $camp_info = get_camp_info($cid);
    die 'no currency given' unless $camp_info && $camp_info->{currency};

    # эти данные необходимы для того, чтобы при принятии ПП в кампанию с независимым управлением ставками не получались нулевые цены на тематических площадках
    my $context_coeff = $camp_info->{ContextPriceCoef};
    $camp_info->{strategy} = Campaign::campaign_strategy($cid);
    my $is_different_places = $camp_info->{strategy}->{name} eq 'different_places';

    my $currency = $camp_info->{currency};
    my $min_price_constant = get_currency_constant($currency, 'DEFAULT_PRICE');
    my $max_price_constant = get_currency_constant($currency, 'MAX_PRICE');

    $m_bids ||= get_one_column_sql(PPC(cid => $cid), ['select mbid from mediaplan_banners', where => {cid => $cid}]);
    my $mediaplan = get_user_mediaplan($uid, $cid, {bids => $m_bids, page => \1});

    ## no critic (Freenode::DollarAB)
    my $mbanners = {};
    my $mretargetings = {};
    for my $b (@{$mediaplan->{arr}}) {
        my $mb = {};
        $mbanners->{$b->{mbid}} = $mb;
        for my $ph (@{$b->{Phrases}}) {
            $mb->{$ph->{id}} = {
                autobudgetPriority => $ph->{autobudgetPriority} || 3,
                rank               => $ph->{rank},
                showsForecast      => $ph->{showsForecast},
                numword            => $ph->{numword} || 0,
            };
        }

        my $mr = {};
        $mretargetings->{$b->{mbid}} = $mr;
        for my $ret (@{$b->{retargetings}}) {
            $mr->{$ret->{ret_id}} = {
                ret_cond_id => $ret->{ret_cond_id},
                price_context => $ret->{price_context}
            };
        }
    }
    
    my $media_bids;
    while( my ( $mbid, $banner ) = each %$mbanners ) {
        my @phids = ();
        while( my ( $idx, $ph ) = each %$banner ) {
            if (%$m_phrases_ids) {
                push @phids, $idx if $m_phrases_ids->{$idx};
            } else {
                push @phids, $idx;
            }

            # если включен автобюджет - устанавливаем минимальную ставку
            if ($camp_info->{autobudget} && $camp_info->{autobudget} eq 'Yes') {
                $ph->{price} = $min_price_constant;
            }
            
            $media_bids->{$idx} = $ph;
        }
        $banner->{phids} = [@phids]; 
    }
    
    my @bids = uniq(keys %$mbanners, keys %$mretargetings);
    if (@bids) {
        my $mediaplan_banners = get_mediaplan_banners($cid, {bids => \@bids, no_phrases => 1, no_forecast => 1, unpack_phone => 1});

        my $client_id = $camp_info->{ClientID};
        foreach my $mb (@$mediaplan_banners) {

            $mb->{statusEmpty} = 'No';
            $mb->{statusModerate} = 'Ready';
            $mb->{geo_id} ||= 0;
            $mb->{bid} = $mb->{source_bid};

            my $adgroup = {
                geo => $mb->{geo},
                currency => $currency,
                pid => $mb->{source_pid},
                banners => [$mb]
            };
            $adgroup = Models::AdGroup::save_group(
                $camp_info, $adgroup,
                ClientID => $camp_info->{ClientID},
                UID => $UID,
                pass_phrases => 1, ignore_tags => 1,
                ignore_retargetings => 1, ignore_minus_words => 1,
                where_from => "mediaplan",
            ); 
            my $new_bid = $adgroup->{banners}->[0]->{bid};
            my $pid = $adgroup->{pid};
            update_phrases($pid, $cid, 'Ready', 'No', $uid, undef);
            do_update_table(PPC(cid => $cid), 'banners', {statusModerate => 'Ready', statusBsSynced => 'No', statusPostModerate =>'No'}, 
                            where => {bid => $new_bid});
            Direct::Banners::delete_minus_geo(bid => $new_bid);
            
            # обрабатываем фразы
            my %old_phrases = map {; $_->{phraseIdHistory} => $_->{price} } grep { $_->{phraseIdHistory} } @{BS::History::get_keywords_with_history({pid => $pid})};
            if (my $ids = [grep {is_valid_id($_)} @{$mbanners->{$mb->{mbid}}->{phids}}]) {
                my $mediaplan_bids_fields_str = join ', ', @MEDIAPLAN_BIDS_FIELDS;
                my $mediaplan_bids = get_all_sql(PPC(cid => $cid), ["SELECT $mediaplan_bids_fields_str FROM mediaplan_bids",
                                                 where => {mbid =>$mb->{mbid}, id => $ids}]);

                my @phrases_for_add_update;

                foreach my $media_bid (@$mediaplan_bids) {
                    $media_bid->{statusBsSynced}         = '';
                    $media_bid->{clicks}                 = 0;
                    $media_bid->{shows}                  = 0;
                    $media_bid->{phraseIdHistory}        //= '';
                    $media_bid->{norm_phrase}            = Yandex::MyGoodWords::norm_words($media_bid->{phrase});
                    $media_bid->{phr}                    = $media_bid->{phrase};
                    $media_bid->{md5}                    = md5_hex_utf8($media_bid->{norm_phrase});
                    
                    $media_bid->{autobudgetPriority} = $media_bids->{$media_bid->{id}}{autobudgetPriority};
                    $media_bid->{rank} = $media_bids->{$media_bid->{id}}{rank};

                    my $price;
                    if ($media_bid->{phraseIdHistory} && $old_phrases{$media_bid->{phraseIdHistory}}) {
                        $price = $old_phrases{$media_bid->{phraseIdHistory}};
                    } else {
                        local $mb->{pid} = $pid;
                        local $mb->{phrases} = [$media_bid];
                        trafaret_auction([$mb], {});
                        my $place_data = PlacePrice::get_data_by_place($media_bid, $media_bid->{place});
                        $price = $place_data->{bid_price} / 1e6;

                        if (!is_valid_float($price) || $price < $min_price_constant){
                            $price = $min_price_constant;
                        } elsif($media_bid->{price} > $max_price_constant) {
                            $price = $max_price_constant;
                        }
                    }
                    $media_bid->{price} = $price;

                    push @phrases_for_add_update, {
                                    bid => $new_bid, 
                                    cid => $cid, 
                                    uid => $uid,
                                    phrase => $media_bid->{phrase},
                                    place => $media_bid->{place} || PlacePrice::get_guarantee_entry_place(),
                                    price => $media_bid->{price},
                                    ($is_different_places
                                        ? (price_context => phrase_price_context($media_bid->{price}, $context_coeff, $currency))
                                        : ()),
                                    autobudgetPriority => $media_bid->{autobudgetPriority},
                                    showsForecast => $media_bid->{showsForecast},
                                    phraseIdHistory => $media_bid->{phraseIdHistory},
                                    is_suspended => $media_bid->{is_suspended},
                                };
                }

                if (@phrases_for_add_update) {
                    do_delete_from_table(PPC(cid => $cid), 'bids', where => {pid => $pid});
                    $_->{pid} = $pid foreach @phrases_for_add_update;
                    mass_add_update_phrase($camp_info, \@phrases_for_add_update);
                }
            }

            # обрабатываем условия ретаргетинга
            if (ref($mretargetings) eq 'HASH' && exists $mretargetings->{ $mb->{mbid} }) {
                my $retargetings = [];
                if (%$m_retargetings) {
                    while( my ( $ret_id, $retargeting ) = each %{$mretargetings->{$mb->{mbid}}} ) {
                        push @$retargetings, $retargeting if $m_retargetings->{$ret_id};
                    }
                } else {
                    $retargetings = [ values %{$mretargetings->{$mb->{mbid}}} ];
                }
                my $banner_with_retargetings = {
                    cid => $cid,
                    bid => $new_bid,
                    retargetings => $retargetings,
                    currency => $currency,
                };

                Retargeting::update_group_retargetings($banner_with_retargetings, UID => $UID);
            }
        }
        
        # удаляем медиаплан
        delete_mediaplan($cid);
    }


    return;
}

sub close_request_first_aid
{
    my ($cids, $req_id, $status) = @_;

    $cids = [$cids] if ref $cids ne 'ARRAY';

    # выставляем заявкам на медиаплан статус "Утвержден"
    return do_update_table(PPC(cid => $cids), 'optimizing_campaign_requests', {
        'status' => $status || 'Accepted',
        'accept_time__dont_quote' => 'now()',
    }, where => {
        'status' => ['New', 'InProcess'],
        'cid' => $cids,
        $req_id ? ('request_id' => $req_id) : (),
    });
}

=head2 is_first_phrase_winner_by_ctr

    Функция определяет имеет ли первая фраза лучший CTR. Используется при удалении дублей во фразах медиаплана.
    Сначала сравнивается ctr для спецразмещения, который показывается на странице (p_ctr), где больше - та фраза победитель.
    Если они равны, тогда сравнивается ctr для входа в гарантию, который показывается на странице (ctr), где больше - та фраза победитель.
    Если они равны, тогда сравнивается количество минус слов в фразах, где больше - та фраза победитель.
    Если они равны, тогда сравнивается эффективный ctr для спецразмещения (p_ectr), где больше - та фраза победитель.
    Если и они равны, тогда сравнивается эффективный ctr для входа в гарантию (ectr), где больше - та фраза победитель.

    Фраза-победитель сохраняется, проигравший - потом удаляется.


=cut
sub is_first_phrase_winner_by_ctr ($$)
{
    my ($phrase1, $phrase2) = @_;
    return 1 if defined $phrase1 && !defined $phrase2;
    return 0 if !defined $phrase1 && defined $phrase2;

    my (undef, $phrase1_mw) = split /\s\-/, $phrase1->{phrase}, 2;
    my @phrase1_mw = split " ", MinusWords::polish_minus_words($phrase1_mw || "");
    my (undef, $phrase2_mw) = split /\s\-/, $phrase2->{phrase}, 2;
    my @phrase2_mw = split " ", MinusWords::polish_minus_words($phrase2_mw || "");

    my $compare = round2s($phrase1->{p_ctr}) <=> round2s($phrase2->{p_ctr}) ||
                  round2s($phrase1->{ctr}) <=> round2s($phrase2->{ctr}) ||
                  scalar(@phrase1_mw) <=> scalar(@phrase2_mw) ||
                  round2s($phrase1->{pectr}) <=> round2s($phrase2->{pectr}) ||
                  round2s($phrase1->{ectr}) <=> round2s($phrase2->{ectr});
    return 1 if $compare > 0;

    return 0;

}

=head2 get_duplicate_phrases_ids

    Принимает список кампаний, в которых надо проверить баннеры на дубликаты. 
    Возвращает массив id фраз подлежащих удалению.

=cut

sub get_duplicate_phrases_ids {
    my $cids = shift;
    my $banners = get_mediaplan_banners($cids);
    my %mbids_domain_geo = map {$_->{mbid} => (strip_www($_->{domain} || get_host($_->{href})) || "")."_".(join "_", sort {$a<=>$b} split /\s*,\s*/, $_->{geo})} @$banners;
    my $all_phrases = [];
    push @$all_phrases, @{$_->{Phrases}} foreach (@$banners);

    my $doubles = get_all_doubles($all_phrases, bids_domain_geo=>\%mbids_domain_geo);

    my $bids = [];
    my @ph_ids_for_delete;

    foreach my $phrases_doubles (values %$doubles) {
        foreach my $phrase_doubles (values %$phrases_doubles) {
            my $winner;
            foreach my $for_check (@$phrase_doubles) {
                if (is_first_phrase_winner_by_ctr($for_check, $winner)) {
                    # Если у новой фразы p_ctr/ctr выше, значит оставляем ее, а id старой фразы добавляем в список на удаление.
                    push @ph_ids_for_delete, $winner->{id} if defined($winner);
                    $winner = $for_check;
                } else {
                    # Новая фраза не выделилась ни ctrом, ни длиной минус-слов, поэтому ее в утиль.
                    push @ph_ids_for_delete, $for_check->{id};
                }
            }
        }

    }
    return \@ph_ids_for_delete;    
}


sub is_allow_FA_lang {
    my $langs = shift;
     return (keys (%$langs) == keys (%{hash_cut $langs, @ALLOW_FA_LANG})) ? 1 : 0 ;
}

=head2 add_mediaplan_banner

    Добавляет в БД новую запись о баннере медиаплана, а также сохраняет все фразы.

=cut
sub add_mediaplan_banner {
    my ($banner, %O) = @_;
    # Сохраняем баннер медиаплана

    my $mbid = $banner->{mbid} = get_new_id('mediaplan_bid', cid => $banner->{cid});
    my $new_banner = hash_cut $banner, @MEDIAPLAN_BANNER_FIELDS;
    $new_banner->{type} = $banner->{banner_type};
    my $translocal_options = $O{is_api} ? {tree => 'api'} : {ClientID => $O{ClientID}};
    $new_banner->{geo} = refine_geoid($new_banner->{geo}, undef, $translocal_options);
    do_insert_into_table(PPC(cid => $banner->{cid}), 'mediaplan_banners', $new_banner);
    if (!$O{save_only_banner}) {
        # Сохраняем фразы баннера медиаплана
        foreach (@{$banner->{phrases}}) { $_->{mbid} = $mbid; }
        my @fields = qw/id cid phrase numword mbid showsForecast place phraseIdHistory/;
        my $new_phids = get_new_id_multi('mediaplan_phid', scalar(@{$banner->{phrases}}));
        foreach my $ph (@{$banner->{phrases}}) {
            $ph->{showsForecast} = $ph->{shows};             
            $ph->{place} = 1;
            $ph->{id} = shift @$new_phids;
        }
        $new_banner->{phrases} = $banner->{phrases};
        my $values = [ map { my $ph = $_; ## no critic (ProhibitComplexMappings)
                             [map {$ph->{$_}} @fields ] } @{$banner->{phrases}} ];
        do_mass_insert_sql(PPC(cid => $banner->{cid}), 'INSERT INTO mediaplan_bids ('.join(',', @fields).') VALUES %s ', $values);
        # копируем условия ретаргетинга
        if(@{$banner->{retargetings}||[]}){
            my $bid_retargetings = $banner->{retargetings};

            my @bid_ret_fields = qw/ret_id ret_cond_id is_suspended mbid/;
            my $bid_ret_fields_str = join ', ', @bid_ret_fields;

            my $new_ret_ids = get_new_id_multi('mediaplan_ret_id', scalar(@$bid_retargetings));
            foreach my $ret (@$bid_retargetings) {
                $ret->{ret_id} = shift @$new_ret_ids;
                $ret->{mbid} = $mbid;
            }
            $banner->{retargeting} = $bid_retargetings;
            my @data_to_insert = map { my $ret = $_; 
                                    [ map { $ret->{$_} } @bid_ret_fields ] } @$bid_retargetings;
            do_mass_insert_sql(PPC(cid => $banner->{cid}), "INSERT INTO mediaplan_bids_retargeting  ($bid_ret_fields_str) VALUES %s",
                                    \@data_to_insert);
        }
    }
    return $mbid;
}

=head2 update_mediaplan_phrases

    Обновляет фразы медиаплана
    Возвращает флаг, нужно ли обновлять статистику, 
    если запрошен массив, дополнительно возвращает массив обновленных фраз (нас интересуют изменения id)

=cut

sub update_mediaplan_phrases {
    my ($mass_phrases, $mbid, %O) = @_;

    # Хэш на случай если фразу стерли и потом снова записали, тогда мы проверим по нормальной форме фразы. 
    my %MD5;

    # old_phrases - хэш (ключ: id фразы) фраз до сохранения, используется для логирования изменений фраз.
    # MD5 - хэш (ключ: md5 от нормальной формы фразы) фраз тоже до сохранения, используется для поиска старого id фразы до изменения.
    # TODO: а может быть и извне получать это...
    my $old_phrases = {map {$_->{id} => $_} @{get_mediaplan_phrases(mbid => $mbid) || []}};

    foreach my $old_phrase (values %$old_phrases) {
        $old_phrase->{norm_phrase} = Yandex::MyGoodWords::norm_words($old_phrase->{phrase});
        $MD5{md5_hex_utf8($old_phrase->{norm_phrase})} = $old_phrase;
    }

    my @mass_phrases = @$mass_phrases;
    my $vcard_values = $O{vcard_values};

    #открываем лог-файл для записи изменения во фразах.
    my $log_phrases = LogTools::messages_logger("UpdateBannerPhrases");

    my @ids = ();
    # начальный mbid.
    my $bid = $mbid;
    my $original_bid = $mbid;
    my @added_phrases = ();
    my $needs_update_stat_flag = 0;
    my $keyword_limit = get_client_limits($O{client_client_id})->{keyword_count_limit};

    # дальше "склеиваем" фразы по запятой и смотрим на привышение лимита, -1 - это "лишняя" запятая для первой фразы
    my @resulting_phrases;

    # идентификаторы фраз для удаления (собираются, если обновить фразу нельзя)
    my @ids_to_delete;
    foreach my $ph (@mass_phrases) {
        $needs_update_stat_flag = 1;
        my %b_p = %$ph;
        my $phrase = $b_p{phrase};

        if( $b_p{phrase} ne '' && !$b_p{active} ) {
            $b_p{mbid} = $bid;

            # при превышении допустимого количества фраз - разбиваем объявление на части
            if (@added_phrases >= $keyword_limit) {
                $bid = 0;
                @added_phrases = ();
            }

            push @added_phrases, $phrase;

            if (! $bid) {
                $bid = update_mediaplan_banner(undef, 
                                               $O{banner_form},
                                               $vcard_values,
                                               $O{sitelinks_set_id},
                                               $O{client_client_id},
                                              );
                $original_bid = $original_bid || $bid;
            }

            my $ph_id = $b_p{ph_id} || $b_p{old_id};

            unless ($ph_id) {
                my $ph_md5 = md5_hex_utf8(Yandex::MyGoodWords::norm_words($b_p{phrase}));
                $ph_id = $MD5{$ph_md5}->{id} if $MD5{$ph_md5};
            }
            if( !$b_p{active} && 
                $b_p{phrase} ne "" &&
               (!$ph_id || 
                $b_p{phrase} ne "" && $O{select_action} && $O{select_action} eq 'copyPhrases' ||
                $ph_id && $old_phrases->{$ph_id} && phrase_should_null_phrase_id({phrase     => $b_p{phrase},
                                                                                  phrase_old => $old_phrases->{$ph_id}->{phrase},
                                                                                  numword    => $b_p{numword}})
               )) {
                push @ids_to_delete, $b_p{old_id};
                # Если происходит копирование фразы или добавляется новая фраза, или стараю фраза сильно изменилась, то вставляем её в БД.
                $ph_id = get_new_id('mediaplan_phid');
                do_insert_into_table(PPC(mediaplan_bid => $mbid), "mediaplan_bids", 
                                                        {id           => $ph_id,
                                                         phrase       => $b_p{phrase},
                                                         statusPhrase => 'new',
                                                         place        => $b_p{place},
                                                         mbid         => $bid,
                                                         cid          => $O{cid},
                                                         numword      => $b_p{numword},
                                                         showsForecast=> $b_p{shows} || 0,
                                                         phraseIdHistory  => $b_p{phraseIdHistory} || undef,
                                                         is_suspended => 0,
                                                         } );
                $log_phrases->out(sprintf("new phrase (mediaplan): UID: %s\tcid: %s\tmbid: %s\tid: %s\tnew_value: %s\tphraseIdHistory: %s", 
                    $O{UID}, $O{cid}, $bid, $ph_id, $b_p{phrase}, $b_p{phraseIdHistory} || '-'));


            } elsif ( !$b_p{active} && $ph_id ) {
                # Сохраняем отредактированную ключевую фразу.
                # После редактирования баннера медиаплана, флаги statusModerate и lowCtr сбрасываются. 
                # Фраза больше не является отклонённой на модерации или отключенной за низкий CTR
                my $phrase_for_update = { phrase         => $b_p{phrase},
                                          statusPhrase   => 'new',
                                          place          => $b_p{place},
                                          mbid           => $bid,
                                          cid            => $O{cid},
                                          numword        => $b_p{numword},
                                          showsForecast  => $b_p{shows} || 0,
                                          statusModerate => 'Yes',
                                          lowCtr         => 'No' ,
                                        };
                # В случае, когда фраза в интерфейсе переносится из другого блока, то она не находится в $old_phrases, так как bid уже сменился,
                # и не имеет $b_p{phraseIdHistory}, так как значения берутся из FORM и там нет этого значения. Однако, это не означает, что phraseIdHistory
                # надо удалять. Поэтому, в случае, если оно нигде не нашлось, просто не надо обновлять это поле.
               my $phraseIdHistory = $b_p{phraseIdHistory} || $old_phrases->{$b_p{old_id}}{phraseIdHistory};
               $phrase_for_update->{phraseIdHistory} = $phraseIdHistory if defined($phraseIdHistory);

                do_update_table(PPC(mediaplan_bid => $mbid), "mediaplan_bids", $phrase_for_update, where => {id => $ph_id} );

                if ($b_p{phrase} ne ($old_phrases->{$ph_id} || {})->{phrase}) {
                    $log_phrases->out(sprintf("edit phrase (mediaplan): UID: %s\tcid: %s\tmbid: %s\tid: %s\tnew_value: %s\told_value: %s\tphraseIdHistory: %s",
                                          $O{UID},
                                          $O{cid},
                                          $bid,
                                          $ph_id,
                                          $b_p{phrase}, 
                                          ($old_phrases->{$ph_id} || {})->{phrase},
                                          $phraseIdHistory || '-'));
                }
            }

            $b_p{ph_id} = $ph_id;
            $b_p{mbid} = $bid;
            push @ids, $b_p{ph_id} if $b_p{ph_id};
            push @resulting_phrases, \%b_p;
        }
    }
    if (scalar @ids_to_delete) {
        $needs_update_stat_flag = 1;
        do_delete_from_table(PPC(mediaplan_bid => $mbid), 'mediaplan_bids',
                             where => {id => \@ids_to_delete, 
                                       mbid => $original_bid, 
                                       cid => $O{cid}});
    }
    # В API может одновременно несколько запросов выполняться, и фразы, созданные в одном,
    # могут потереться в другом. Так как из API фразы удаляются отдельным вызовом, передаем этот флаг:
    # do_not_delete_absent_phrases
    if ( !$O{do_not_delete_absent_phrases} && $bid > 0 && $O{cid} > 0 ) {
        $needs_update_stat_flag = 1;
        
        do_delete_from_table(PPC(mediaplan_bid => $mbid), 'mediaplan_bids', 
                               where => {id__not_in => \@ids, 
                                         mbid => $original_bid, 
                                         cid => $O{cid}});
    }

    return wantarray ? ($needs_update_stat_flag, \@resulting_phrases) : $needs_update_stat_flag;
}

=head2 update_mediaplan_phrases_wrapper

    Получает на вход хэш:
    add с фразами в виде 
        {
            mbid => идентификатор баннера медиаплана
            phrase => фраза
            place => номер позиции (1,2,3)
        }
    update с фразами в виде
        {
            old_id => идентификатор обновляемой фразы
            phrase => фраза (необязательно)
            place => номер позиции (1,2,3) (необязательно)
        }

    делает все нужные приготовления и обновления, работает в API
    TODO: чтобы использовать в интерфейсе нужно протестировать одновременные update и add

=cut
sub update_mediaplan_phrases_wrapper
{
    my ($rbac, $UID, $login_rights, $operator_chief_uid, %O) = @_;

    my $to_add = $O{add} || [];
    my $to_update = $O{update} || [];

    my $ret;
    # хэш по баннерам фраз, которые хотим добавить
    my %banners_keywords_add;

    foreach my $p (@$to_add) {
        hash_merge $p, PhraseText::get_phrase_props($p->{phrase});
        # записываем в хэш новый объект
        push @{$banners_keywords_add{$p->{mbid}}}, {%$p};
    }

    my @source_keyword_ids = map {$_->{source_id}} @$to_add;

    my $old_keywords;

    if (scalar @source_keyword_ids) {
        $old_keywords = {};
        for my $ph (@{BS::History::get_keywords_with_history({ph_ids => \@source_keyword_ids})}) {
            $old_keywords->{$ph->{id}} = $ph;
        }
    }

    # соответствие идентификатора фразы и объявления для обновления
    my $keywordid2mbid;

    if (scalar @$to_update) {
        $keywordid2mbid = get_hashes_hash_sql(PPC(shard => 'all'), ['select id, mbid from mediaplan_bids', 
            where => {id => [ map {$_->{old_id}} @$to_update]}]);
    }

    # хэш по баннерам фраз, которые хотим обновить
    my %banners_keywords_update;
    foreach my $p (@$to_update) {
        $p->{mbid} = $keywordid2mbid->{$p->{old_id}}{mbid};
        hash_merge $p, PhraseText::get_phrase_props($p->{phrase});
        # записываем в хэш новый объект
        push @{$banners_keywords_update{$p->{mbid}}}, {%$p};
    }

    # TODO: получать короткую версию баннера
    # TODO: рассклейка пересекающихся фраз минус-словами
    # TODO: ворнинги при изменении существующих фраз, при наличии фраз, одинаковых по нормальной форме
    my @banners_ids = uniq keys %banners_keywords_add, keys %banners_keywords_update;
    my $mediaplan_banners = get_mediaplan_banners( [], {bids => \@banners_ids, phrases_db_only => 1} );

    # Определяем ClientID (массово) для сайт-линков, минус-слов по сid из баннеров
    my $mediaplan_banners_cids = [ grep { is_valid_id($_) } map { $_->{cid} } @$mediaplan_banners ];
    my $cids2clientids = get_cid2clientid(cid => $mediaplan_banners_cids);
    # и uid для орг. деталей
    my $cids2uids = get_cid2uid(cid => $mediaplan_banners_cids);

    foreach my $banner (@$mediaplan_banners) {

        # собираем хэш по нормальной форме, содержащий новые фразы и количество их, совпадающее по нормальной форме,
        # а так же флаг updated - была ли фраза смерджена с существующей
        my $new_phrases_md5_hash;
        foreach my $p (@{$banners_keywords_add{$banner->{mbid}}}) {
            unless (exists $new_phrases_md5_hash->{$p->{md5}}) {

                $new_phrases_md5_hash->{$p->{md5}} = $p;
                $new_phrases_md5_hash->{$p->{md5}}{is_new} = 1;

                # Если для данной фразы был передан source_id (id фразы, для которой проводится оптимизация)
                # пытаемся скопировать ее phraseIdHistory, если фраза не слишком изменилась

                if ($p->{source_id} &&
                    !phrase_should_null_phrase_id({
                        phrase => $p->{phrase}, 
                        phrase_old => $old_keywords->{$p->{source_id}}{phrase}, 
                        numword => 1}
                    )) {
                    $new_phrases_md5_hash->{$p->{md5}}{phraseIdHistory} = $old_keywords->{$p->{source_id}}{phraseIdHistory};
                }
            }
            $new_phrases_md5_hash->{$p->{md5}}{cnt}++;
        }

        # собираем хэш для апдейта фраз на основе старого id фразы
        my $update_phrases_id_hash = {map {$_->{old_id} => $_} @{$banners_keywords_update{$banner->{mbid}}}};

        push @{$banner->{Phrases}}, @{$banners_keywords_add{$banner->{mbid}}};

        $banner->{Phrases} = process_phrases($banner->{Phrases});

        foreach my $p (@{$banner->{Phrases}}) {

            # по умолчанию новый и старый id всегда равен старому
            $p->{new_id} = $p->{id};
            $p->{old_id} = $p->{id};

            # в случае, если новые фразы совпадают по нормальной форме со старыми
            if (!$p->{is_new} && exists $new_phrases_md5_hash->{$p->{md5}}) {
                $p->{phrase} = $new_phrases_md5_hash->{$p->{md5}}{phrase};
                $p->{place} = $new_phrases_md5_hash->{$p->{md5}}{place};
                $new_phrases_md5_hash->{$p->{md5}}{updated} = 1;
            }

            if (exists $new_phrases_md5_hash->{$p->{md5}}) {
                $p->{phraseIdHistory} = $new_phrases_md5_hash->{$p->{md5}}{phraseIdHistory};
            }

            # обновление фраз
            if ($update_phrases_id_hash->{$p->{id}}) {

                if ($update_phrases_id_hash->{$p->{id}}{phrase} && 
                    phrase_should_null_phrase_id({
                        phrase => $update_phrases_id_hash->{$p->{id}}{phrase}, 
                        phrase_old => $p->{phrase}, 
                        numword => 1}
                    )) {
                    # если добавляем новую
                    $p->{new_id} = 0;
                }

                # для каждого потенциально измененного поля, обновляем объект фразы
                for my $field (qw/phrase place numword md5/) {
                    if ($update_phrases_id_hash->{$p->{id}}{$field}) {
                        $p->{$field} = $update_phrases_id_hash->{$p->{id}}{$field};
                    }
                }
            }
        }

        # если передан соответствующие флаг, расклеиваем минус-слова
        my $unglued;
        if ($O{unglue_phrases}) {
            $unglued = unglue_phrases([$banner], -1);
        }
        my @mass_phrases;
        foreach my $p (@{$banner->{Phrases}}) {
            if ($O{unglue_phrases} && $unglued && $p->{phrase_unglued_suffix}) {
                ($p->{phrase}, $p->{_phrase}) = ($p->{phrase}.$p->{phrase_unglued_suffix}, $p->{phrase});
            }
            # собираем массив фраз для обновления/добавления
            push @mass_phrases, {
                phrase => $p->{phrase},
                place => $p->{place},
                ph_id => $p->{new_id} || 0,
                old_id => $p->{old_id} || 0,
                shows => $p->{shows} || 0,
                numword => $p->{numword},
                md5 => $p->{md5},
                phraseIdHistory => $p->{phraseIdHistory},
            };
        }

        my $vcard_values = hash_cut($banner, @$VCARD_FIELDS, qw/cid geo_id/);

        $vcard_values->{org_details_id} = add_org_details({
            ogrn => $banner->{ogrn},
            uid => $cids2uids->{ $banner->{cid} }
        });

        my ($needs_update_stat_flag, $updated) = update_mediaplan_phrases(\@mass_phrases, $banner->{mbid},
            cid => $banner->{cid},
            client_client_id => $cids2clientids->{ $banner->{cid} },
            UID => $UID,
            sitelinks_set_id => $banner->{sitelinks_set_id},
            vcard_values => $vcard_values,
            do_not_delete_absent_phrases => 1,
            banner_form => hash_cut($banner, qw/title title_extension body href domain geo cid banner_type/),
        );

        if ($needs_update_stat_flag) {
            update_mediaplan_stats($rbac, $banner->{cid}, $UID, $login_rights, 1);
        }
        
        my $mbids = [keys %banners_keywords_add];
        # после обновления, обнуляем statusShowsForecast для того, чтобы при следующем обращении к баннеру, прогноз пересчитался
        # TODO: скрипт для посчету прогноза в фоне
        do_update_table(PPC(mediaplan_bid => $mbids), 'mediaplan_banners', {statusShowsForecast => 'New'}, where => {mbid => SHARD_IDS});

        foreach my $p (@$updated) {
            if ($new_phrases_md5_hash->{$p->{md5}}) {
                $new_phrases_md5_hash->{$p->{md5}}{id} = $p->{ph_id};
                $new_phrases_md5_hash->{$p->{md5}}{mbid} = $p->{mbid};
            }
            if ($update_phrases_id_hash->{$p->{old_id}}) {
                $update_phrases_id_hash->{$p->{old_id}}{id} = $p->{ph_id};
                $update_phrases_id_hash->{$p->{old_id}}{mbid} = $p->{mbid};
            }
        }

        $ret->{$banner->{mbid}}{add} = $new_phrases_md5_hash;
        $ret->{$banner->{mbid}}{update} = $update_phrases_id_hash;
    }

    return $ret;
}

=head2 COMMENT

    client_id должен быть владельца кампании (с медиапланом которой идет работа)

    $options:
        is_api -- вызов из API, не нужно модифицировать geo (по транслокальному дереву клиента)

=cut

sub update_mediaplan_banner {
    my ($bid, $banner_form, $vcard_values, $sitelinks_set_id, $client_id, $source_bid, $options) = @_;
    my $cid         = $banner_form->{cid};
    # -- BUGFIX DIRECT-29677
    my $client_uid  = get_owner(cid => $cid);
    $options //= {};
    
    $vcard_values->{uid} = $client_uid;
    $vcard_values->{cid} = $cid;

    if ($vcard_values->{phone}){
        $banner_form->{vcard_id} = create_vcards($client_uid, [$vcard_values])->[0]{vcard_id};
    } else {
        $banner_form->{vcard_id} = undef;
    }

    my $old_vcard_id = get_one_field_sql(PPC(cid => $cid), ["select vcard_id from mediaplan_banners", where => { mbid => $bid}]);

    if ($old_vcard_id and ($banner_form->{vcard_id} // -1) != $old_vcard_id) {
        dissociate_vcards($old_vcard_id);
    }

    if (! $options->{is_api}) {
        $banner_form->{geo} = GeoTools::modify_translocal_region_before_save($banner_form->{geo}, {ClientID => $client_id});
    }

    if ($bid) {
        my $translocal_options = $options->{is_api} ? {tree => 'api'} : {ClientID => $client_id};
        do_update_table(PPC(cid => $cid), 'mediaplan_banners', {
            cid => $cid,
            title => $banner_form->{title},
            title_extension => $banner_form->{title_extension},
            body => $banner_form->{body},
            href => $banner_form->{href},
            domain => $banner_form->{domain},
            geo => refine_geoid($banner_form->{geo}, undef, $translocal_options),
            statusShowsForecast => 'New',
            timeShowsForecast__dont_quote => 'NOW()',
            vcard_id => $banner_form->{vcard_id},
            sitelinks_set_id => $sitelinks_set_id
        }, where => {mbid => $bid});

    } else {
        my $new_banner = hash_merge {}, $banner_form;
        if ($source_bid) {
            $new_banner->{source_bid} = $source_bid;
        }
        $new_banner->{sitelinks_set_id} = $sitelinks_set_id;
        $bid = add_mediaplan_banner($new_banner, save_only_banner => 1, ClientID => $client_id);
    }
    MinusWords::save_mediaplan_banner_minus_words($bid, $banner_form->{banner_minus_words}, $client_id);

    return $bid;
}

=head2 delete_mediaplan_banners($UID, cid, mbids)

    Удаление баннеров из медиаплана вместе с фразами и визитками(по запросу)
    Параметры
        UID - UID
        cid - номер кампании
        mbids - [] массив id баннеров медиаплана

=cut

sub delete_mediaplan_banners {
    my ($UID, $cid, $mbids, %O) = @_;

    die unless is_valid_id($cid);
    for my $mbid (@$mbids) {
        delete_mediaplan_phrases($UID, $cid, mbids => [$mbid]);
    }

    if (scalar @$mbids) {
        do_delete_from_table(PPC(cid => $cid), 'mediaplan_banners', where => {mbid=>$mbids, cid=>$cid});
        do_delete_from_table(PPC(cid => $cid), 'mediaplan_banners_original', where => {mbid=>$mbids, cid=>$cid});
        delete_shard(mediaplan_bid => $mbids);
    }

    return 1;
}

=head2 delete_mediaplan_phrases

    Удаление фраз медиаплана
    параметры позиционные:
    UID - UID,
    cid - номер кампании
    параметры именованные:
        ids - список фраз
        mbids - список объявлений

    должен быть указан один из ids, mbids, у ids приоритет

=cut

sub delete_mediaplan_phrases {
    my ($UID, $cid, %O) = @_;

    die unless is_valid_id($cid);

    my $del_ph_id = [];
    my $ret_ids = [];

    if ($O{mbids} && scalar @{$O{mbids}}) {
        $del_ph_id = get_one_column_sql(PPC(cid => $cid), ["SELECT id FROM mediaplan_bids", where => { mbid => $O{mbids}, cid => $cid}]);
        do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids', where => {mbid => $O{mbids}, cid => $cid});

        $ret_ids = get_one_column_sql(PPC(cid => $cid), [
                "SELECT ret_id FROM mediaplan_bids_retargeting
                JOIN mediaplan_banners mb USING(mbid)",
                where => { mbid => $O{mbids}, cid => $cid },
            ]);
        do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids_retargeting', where => {mbid => $O{mbids}, ret_id => $ret_ids});
    }
    else {
        if ($O{ids} && scalar @{$O{ids}}) {
            my $bids = get_one_column_sql(PPC(cid => $cid), ["SELECT distinct mbid FROM mediaplan_bids", WHERE => {id => $O{ids}}]) || [];
            do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids', where => {id => $O{ids}, cid => $cid});

            $del_ph_id = $O{ids};
        }
        
        if (defined $O{ids_retargetings}) {
            $ret_ids = [grep {/^\d+$/} split /\s*,\s*/, $O{ids_retargetings}];
            die 'delMediaplanPhrases: not found ids_retargetings' unless @$ret_ids;

            # проверяем, что все условия принадлежат нашей кампании
            $ret_ids = get_one_column_sql(PPC(cid => $cid), ["select mbr.ret_id
                                             from mediaplan_bids_retargeting mbr
                                               join mediaplan_banners mb using(mbid)
                                            ", where => {'mbr.ret_id' => $ret_ids, 'mb.cid' => $cid}
                                           ]) || [];

            do_delete_from_table(PPC(cid => $cid), 'mediaplan_bids_retargeting', where => {ret_id => $ret_ids}) if @$ret_ids;
        }
    }

    #открываем лог-файл для записи изменения во фразах.
    my $log_phrases;
    if (@$del_ph_id || @$ret_ids){
        $log_phrases = LogTools::messages_logger("UpdateBannerPhrases");
    }
    if (@$del_ph_id) {
        #Логируем удаление фраз (только id)
        $log_phrases->out(sprintf("delete phrases (mediaplan): UID: %s\tcid: %s\tids: %s", $UID, $cid, join (',', @$del_ph_id)));
    }

    if(@$ret_ids){
        $log_phrases->out(sprintf("delete retargetings (mediaplan): UID: %s\tcid: %s\tids: %s", $UID, $cid, join (',', @$ret_ids)));
    }
    return 1;
}

=head2 end_mediaplan

Закончить медиаплан

Опции:

    uid
    accept_type

=cut

sub end_mediaplan {
    my ($rbac, $cid, $comment, %O) = @_;

    my $manager_data = get_mediaplan_stats($cid);
    hash_merge $manager_data, get_user_info($manager_data->{ManagerUID});

    do_update_table(PPC(cid => $cid), 'camp_options', { mediaplan_status => 'Complete' }, where => { cid => $cid } );

    my $accept_type = $O{accept_type};
    if (!$accept_type || !$MediaplanOptions::ACCEPT_TYPE{$accept_type}) {
        $accept_type = 'merge';
    }

    do_update_table(PPC(cid => $cid), 'mediaplan_stats', {
            end_comment => $comment,
            accept_type => $accept_type,
            is_lego_mediaplan => 0 + !!$O{is_lego_mediaplan},
        },
        where => {cid => $cid, accepted =>'No'},
    );

    my $mailvars = get_one_line_sql(PPC(cid => $cid), "select c.cid
                                                , c.cid as campaign_id
                                                , c.name as camp_name 
                                                , ifnull(co.email, u.email) as email
                                                , u.FIO as fio
                                                , u.uid as client_uid
                                                , u.login as ulogin
                                                , u.ClientID as client_id
                                                , c.type as campaign_type
                                                , c.AgencyID
                                           from campaigns c
                                             join users u on c.uid = u.uid
                                             left join camp_options co on co.cid = c.cid
                                           where c.cid = ?", $cid);
    $mailvars->{comment} = $comment;
    $mailvars->{manager_fio} = $manager_data->{fio};
    $mailvars->{manager_email} = $manager_data->{email};
    $mailvars->{manager_uid} = $manager_data->{ManagerUID};
    $mailvars->{accept_type} = $accept_type;

    $mailvars->{applied_by_uid} = $O{uid};

    add_notification($rbac, 'mediaplan_ready', $mailvars);

    return 1;
}

=head2 copy_banners_to_mediaplan (cid, bids)

    Копирует баннеры из bids в медиаплан.    

=cut

sub copy_banners_to_mediaplan {
    my ($cid, $bids, $client_chief_uid, $options) = @_;
    
    my $vcard_ids = get_hashes_hash_sql(PPC(cid => $cid), ["SELECT bid, vcard_id, pid FROM banners", where => {bid => $bids}]);
    my $full_groups = Models::AdGroup::is_completed_groups([map {$_->{pid}} values %$vcard_ids]);
    #открываем лог-файл для записи изменения во фразах.
    my $log_phrases = LogTools::messages_logger("UpdateBannerPhrases");
    my (@mediaplan_mod_reasons, @mbids);
    foreach my $bid (@$bids) {
        
        next unless $full_groups->{ $vcard_ids->{$bid}->{pid} };
        
        my $vcard_id = $vcard_ids->{$bid}->{vcard_id};
        my $mbanner = get_one_line_sql(PPC(uid => $client_chief_uid), 
                                    'select p.cid, b.title, b.title_extension, b.body, b.href, p.geo, b.domain, sitelinks_set_id, b.type,
                                            p.forecastDate as timeShowsForecast, p.pid as source_pid, p.mw_id
                                            from banners b join phrases p using(pid) where b.bid=? AND b.banner_type="text"', $bid);
        next unless $mbanner;
        my $mbid = get_new_id('mediaplan_bid', uid => $client_chief_uid);
        $mbanner->{mbid} = $mbid;
        $mbanner->{vcard_id} = $vcard_id;
        $mbanner->{source_bid} = $bid;
        # принудительно пересчитываем прогноз показов, чтоб значение было свежим и совпадало с редактированием
        $mbanner->{statusShowsForecast} = 'New';

        do_insert_into_table(PPC(uid => $client_chief_uid), 'mediaplan_banners', $mbanner);

        push @mbids, $mbid if $mbid;
        #логируем баннеры с какими фразами были скопированы в медиаплан.
        my $before_banners_phrases = get_hashes_hash_sql(PPC(uid => $client_chief_uid), "SELECT id, phrase, phraseIdHistory FROM mediaplan_bids WHERE mbid = ?", $mbid);

        my $bid_phrases = BS::History::get_keywords_with_history({bid => $bid});
        foreach my $ph (@$bid_phrases) {
            $ph->{mbid} = $mbid;
            $ph->{numword} = 1;
        }

        my @bid_fields = qw/ id cid phrase numword place showsForecast
                             phraseIdHistory
                             mbid statusModerate is_suspended /;
        my $new_phids = get_new_id_multi('mediaplan_phid', scalar(@$bid_phrases));
        foreach (@$bid_phrases) {
            $_->{id} = shift @$new_phids;
            $_->{place} = PlacePrice::get_guarantee_entry_place() unless $_->{place};
        }
        $mbanner->{phrases} = $bid_phrases;

        my @data_to_insert = map { my $ph = $_; 
                                   [ map { $ph->{$_} } @bid_fields ] } @$bid_phrases;
        do_mass_insert_sql(PPC(uid => $client_chief_uid), "INSERT INTO mediaplan_bids  (" . join(', ', @bid_fields) . ") VALUES %s",
                                \@data_to_insert);

                                                                                                                                                                                          
        my $after_banners_phrases = get_hashes_hash_sql(PPC(uid => $client_chief_uid), "SELECT id, phrase, phraseIdHistory FROM mediaplan_bids WHERE mbid = ?", $mbid);                    
        my $added_banners_phrases = hash_kgrep {!defined($before_banners_phrases->{$_})} $after_banners_phrases;                                                                           
        for my $ph (values %{$added_banners_phrases}) {                                                                                                                                    
            $log_phrases->out(sprintf("copy banner (to mediaplan): client_chief_uid: %s\tcid: %s\tmbid: %s\tid: %s\tphraseIdHistory: %s\tnew_value: %s", $client_chief_uid, $cid, $mbid, $ph->{id}, $ph->{phraseIdHistory}||"-", $ph->{phrase}));
        }                                                                                                                                                                              

        # Запоминаем ответы модерации.
        my $mod_reasons = get_diags($bid, 'banner'
                                                , bad_phrases => 1
                                                , bad_sitelinks => 1
                                                , bad_images => 1 );
        push @mediaplan_mod_reasons, [$mbid, YAML::Dump( $mod_reasons )] if(scalar keys %{$mod_reasons->{banner_diags}} > 0);

        # копируем условия ретаргетинга
        my @bid_ret_fields = qw/ret_id ret_cond_id is_suspended mbid/;
        my $bid_ret_fields_str = join ', ', @bid_ret_fields;

        my $bid_retargetings = get_all_sql(PPC(uid => $client_chief_uid), 
                                                 ["select ret_cond_id, is_suspended
                                                   from bids_retargeting
                                                  ", where => {pid => $vcard_ids->{$bid}->{pid}}]);

        my $new_ret_ids = get_new_id_multi('mediaplan_ret_id', scalar(@$bid_retargetings));
        foreach (@$bid_retargetings) {
            $_->{ret_id} = shift @$new_ret_ids;
            $_->{mbid} = $mbid;
        }
        $mbanner->{retargetings} = $bid_retargetings;

        @data_to_insert = map { my $ret = $_; 
                                [ map { $ret->{$_} } @bid_ret_fields ] } @$bid_retargetings;
        do_mass_insert_sql(PPC(uid => $client_chief_uid), "INSERT INTO mediaplan_bids_retargeting  ($bid_ret_fields_str) VALUES %s",
                                \@data_to_insert);
        $mbanner->{banner_type} = delete $mbanner->{type};                                
        log_mediaplan("copy_banners_to_mediaplan", $cid, {mbanner=>$mbanner});
    }

    # Сохраняем ответы модерации.
    do_mass_insert_sql(PPC(uid => $client_chief_uid), 'insert into mediaplan_mod_reasons (bid, data) values %s', \@mediaplan_mod_reasons);

    # Сохраняем резервную копию в формате json. Используется для сравнения изменений.
    if (! $options->{no_original} && @mbids) {
        my $mbanners = get_mediaplan_banners($cid, {bids => \@mbids});
        my $values = [];
        foreach my $mbanner (@$mbanners) {
            push @$values, [$mbanner->{mbid}, $cid, encode_json_and_compress(clear_banner_for_compare($mbanner))];
        }
        do_mass_insert_sql(PPC(uid => $client_chief_uid), 'insert into mediaplan_banners_original (mbid, cid, data_json) values %s', $values);
    }
}

=head2 copy_mediaplan_banners

    Копирование баннера внутри медиаплана.

=cut
sub copy_mediaplan_banners {
    my ($cid, $bids) = @_;

    #открываем лог-файл для записи изменения во фразах.
    my $log_phrases = LogTools::messages_logger("UpdateBannerPhrases");

    my @copied_bids;
    for my $bid (@$bids) {
        my $vcard_id = get_one_field_sql(PPC(cid => $cid), "select vcard_id from mediaplan_banners where mbid = ?", $bid );

        my $copy_bid = get_new_id('mediaplan_bid', cid => $cid);

        my ($fields_str) = make_copy_sql_strings(\@Mediaplan::MEDIAPLAN_BANNER_FIELDS);

        my $mbanner = get_one_line_sql(PPC(cid => $cid), ["SELECT $fields_str FROM mediaplan_banners", WHERE => {mbid => $bid}, limit => 1]);
        # Сохраняем предыдущие значения id полей, которые будут изменены при копировании. Требуется для логирования.
        copy_previous_fields_value($mbanner, qw/mbid vcard_id source_bid source_pid/);

        $mbanner->{vcard_id}   = $vcard_id;
        $mbanner->{mbid}       = $copy_bid;
        $mbanner->{source_bid} = $mbanner->{source_pid} = '';

        do_insert_into_table(PPC(cid => $cid), 'mediaplan_banners', hash_cut $mbanner, @Mediaplan::MEDIAPLAN_BANNER_FIELDS);

        if( $copy_bid ) {

            my $copied_phrases = get_all_sql(PPC(cid => $cid), "SELECT id, phrase, phraseIdHistory FROM mediaplan_bids WHERE mbid = ?", $bid);

            my %bids_default_override = (mbid => $copy_bid,
                                         is_suspended => 0);
            my ($bids_fields_str) = make_copy_sql_strings(\@Mediaplan::MEDIAPLAN_BIDS_FIELDS);
            my $bids = get_all_sql(PPC(cid => $cid), ["SELECT $bids_fields_str FROM mediaplan_bids", where => {mbid => $bid}, "ORDER BY id"]);
            my $new_phids = get_new_id_multi('mediaplan_phid', scalar(@$bids));
            my @bids_values = ();
            $mbanner->{phrases} = [];
            foreach my $ph (@$bids) {
                copy_previous_fields_value($ph, qw/id/);
                $ph->{id} = shift @$new_phids;
                $ph = hash_merge $ph, \%bids_default_override;
                push @bids_values, [map {$ph->{$_}} @Mediaplan::MEDIAPLAN_BIDS_FIELDS];
                push @{$mbanner->{phrases}}, $ph;
            }
            do_mass_insert_sql(PPC(cid => $cid), "INSERT INTO mediaplan_bids ($bids_fields_str) VALUES %s", \@bids_values);
            push @copied_bids, $copy_bid;

            my $copied_history = get_all_sql(PPC(cid => $cid), "SELECT id, phraseIdHistory FROM mediaplan_bids WHERE mbid = ? AND IFNULL(phraseIdHistory, '') <> ''", $copy_bid);
            foreach my $history (@$copied_history) {
                my %parsed_history = BS::History::parse_bids_history($history->{phraseIdHistory});
                if (keys %{$parsed_history{banners}}) {
                    $parsed_history{banners} = { 0 => $parsed_history{banners}->{(sort keys %{$parsed_history{banners}})[0]} };
                }
                do_update_table(PPC(cid => $cid), 'mediaplan_bids', {phraseIdHistory => BS::History::serialize_bids_history(\%parsed_history) || ''},
                                                  where => { id => $history->{id} } );
            }

            #Логируем копирование фраз                                                                                                                                                 
            for my $ph (@{$copied_phrases}) {                                                                                                                                          
                $log_phrases->out(sprintf("copy phrases (copy mediaplan): cid: %s\tbid: %s\tid: %s\tphraseIdHistory: %s\tnew_value: %s", $cid, $copy_bid, $ph->{id}, $ph->{phraseIdHistory}||'', $ph->{phrase}));
            }

            my @bids_ret_fields_to_copy = qw/
                                        ret_id
                                        ret_cond_id
                                        mbid
                                        is_suspended
                                        statusPhrase
                                    /;
            my %bids_ret_default_override = (mbid => $copy_bid);
            my ($bids_ret_fields_str) = make_copy_sql_strings(\@bids_ret_fields_to_copy);
            my $bids_ret = get_all_sql(PPC(cid => $cid), ["SELECT $bids_ret_fields_str FROM mediaplan_bids_retargeting", where => {mbid => $bid}, "ORDER BY ret_id"]);
            my $new_ret_ids = get_new_id_multi('mediaplan_ret_id', scalar(@$bids_ret));
            my @bids_ret_values = ();
            $mbanner->{retargetings} = [];
            foreach my $ret (@$bids_ret) {
                copy_previous_fields_value($ret, qw/ret_id/);
                $ret->{ret_id} = shift @$new_ret_ids;
                $ret = hash_merge $ret, \%bids_ret_default_override;
                push @bids_ret_values, [map {$ret->{$_}} @bids_ret_fields_to_copy];
                push @{$mbanner->{retargetings}}, $ret;
            }
            do_mass_insert_sql(PPC(cid => $cid), "INSERT INTO mediaplan_bids_retargeting ($bids_ret_fields_str) VALUES %s", \@bids_ret_values);
        } else {
            delete_shard(mediaplan_bid => $copy_bid);
        }
        $mbanner->{banner_type} = delete $mbanner->{type}; 
        log_mediaplan("copy_mediaplan_banners", $cid, filter_fields_for_log($mbanner));
    }
    return \@copied_bids;
}

=head2 copy_previos_fields_value

    Сохраняет копии полей, подставляя префикс, т.о. сохраняет предыдущие значения id полей, которые будут изменены при копировании.
    Изменяет переданные данные. Используется для логирования. 

=cut
sub copy_previous_fields_value {
    my ($object, @fields) = @_;
    foreach (@fields) {
        $object->{$PREVIOUS_PREFIX.$_} = $object->{$_};
    }
}

=head2 filter_fields_for_log

    Отсеивает нежуные поля из переданного объекта. Предполагается объект, который содержит поля FIELDS_FOR_LOG и SUBFIELDS_FOR_LOG.
    Для полей phrases производятся дополнительные манипуляции по определению характера действий, произведенными над ними.
    Используется перед записью действий, произведенными над медиапланом в логи.

    Пример использования:
    log_mediaplan("cmd_delMediaplanPhrases", $FORM{cid}, filter_fields_for_log($log_mbanner));

=cut
sub filter_fields_for_log($) {
    my $mbanner = shift;
    my $result = hash_grep {defined $_} hash_cut $mbanner, @FIELDS_FOR_LOG;
    if (defined($mbanner->{previous_phrases})) {
        my %previous_phrases = map {$_->{id} => $_->{phrase}} @{$mbanner->{previous_phrases}};
        my %new_phrases = map {$_->{ph_id} => $_->{phrase}} @{$mbanner->{phrases}};
        foreach (@{$mbanner->{phrases}}) {
            if (!$_->{ph_id} && $_->{old_id}) {
                $_->{previous_phrase} = delete $previous_phrases{$_->{old_id}};
            }
            if ($previous_phrases{$_->{old_id}} && $_->{phrase} ne $previous_phrases{$_->{old_id}}) {
                $_->{previous_phrase} = $previous_phrases{$_->{old_id}};
            }
        }
        my $deleted = xminus ([keys %previous_phrases], [keys %new_phrases]);
        foreach (@$deleted) {
            push @{$mbanner->{phrases}}, {phrase => $previous_phrases{$_}, ph_id => "DELETED", old_id => $_};
        }
        
    }
    foreach my $key (keys %SUBFIELDS_FOR_LOG) {
        if (ref $mbanner->{$key} eq 'ARRAY') {
            $result->{$key} = [map {hash_cut $_, @{$SUBFIELDS_FOR_LOG{$key}}} @{$mbanner->{$key}}];
        } else {
            $result->{$key} = hash_cut $mbanner->{$key}, @{$SUBFIELDS_FOR_LOG{$key}};
        }
        
    }
    foreach (grep {! m/$PREVIOUS_PREFIX/} @FIELDS_FOR_LOG) {
        delete $result->{$PREVIOUS_PREFIX.$_} if ($result->{$PREVIOUS_PREFIX.$_} || '') eq ($result->{$_} || '');
    }
    return $result;
}

=head2 clear_banner_for_compare

    При копировании баннера в медиаплан, сохнаряется резервная(эталонная) копия баннера в БД. Но чтобы не копировать все подряд, 
    а только действительно необходимые поля, то перед этим баннер очищается от ненужных полей.

=cut
sub clear_banner_for_compare {
    my $mbanner = shift;
    my $clean_mbanner = hash_cut $mbanner, @BANNER_COMPARE_FIELDS, qw/vcard sitelinks/;
    $clean_mbanner->{Phrases} = [ map {hash_cut $_, @PHRASE_COMPARE_FIELDS} (@{$mbanner->{Phrases}}) ];
    return $clean_mbanner;
}

=head2 compare_banners

    Сравнивает медиаплановые баннеры и возвращает поля и фразы, которые были изменены
    На входе два баннера: 
        mbanner - получившийся после работы медиапланера,
        reserve - который был сохранен как эталон при копировании баннера в медиаплан.
    На выходе хэш вида:
    {Title => 1, 
     Body  => 1,
     phrases => {1234 => 'added', 1235=>'added', 1236=>'moved', 1237=>'changed', ...}
     ...
    }

=cut
sub compare_banners {
    my ($mbanner, $reserve, $all_phrases) = @_;
    my %changes;
    foreach (@BANNER_COMPARE_FIELDS) {
        $changes{$_} = 1 if ($mbanner->{$_} && $reserve->{$_} && $mbanner->{$_} ne $reserve->{$_});
    }

    $changes{'vcard'} = 1 if (%{$reserve} && ($mbanner->{vcard} || $reserve->{vcard})) && VCards::compare_vcards($mbanner->{vcard}, $reserve->{vcard});
    $changes{'sitelinks'} = 1 if (%{$reserve} && ($mbanner->{sitelinks} || $reserve->{sitelinks})) && Sitelinks::compare_sitelinks($mbanner->{sitelinks}, $reserve->{sitelinks});

    my @mbanner_phrases = @{$mbanner->{Phrases}};
    my %reserve_phrases_ph_id = map {$_->{phrase} => $_->{id}} @{$reserve->{Phrases}};
    my %reserve_phrases_id_ph = map {$_->{id} => $_->{phrase}} @{$reserve->{Phrases}};
    my %mbanner_phrases_id_ph = map {$_->{id} => $_->{phrase}} @mbanner_phrases;

    my %added   = map {$_->{id} => 1} grep { ! defined $reserve_phrases_id_ph{$_->{id}} } @mbanner_phrases;
    my %changed = map {$_->{id} => 1} grep { ! $added{$_->{id}} && ! defined $reserve_phrases_ph_id{$_->{phrase}}} @mbanner_phrases;

    my %moved;

    # Фразы, которые были изменены так, что у них должен будет обнулиться CTR должны быть помечены как новые.
    # Проверяем все измененные фразы на этот случай и перекладываем из массива changed_phrases в added_phrases
    foreach my $mphrase (@mbanner_phrases) {
        my $id = $mphrase->{id};
        next unless ($changed{$id} || $added{$id});

        if ($changed{$id} && 
            phrase_should_null_phrase_id({phrase     => $mphrase->{phrase},
                                          phrase_old => $reserve_phrases_id_ph{$id},
                                          numword    => $mphrase->{numword}})) {
            $added{$id} = delete $changed{$id};
        }

        # Проверяем не случилось ли так, что фраза перенесена из другого баннера.
        if (defined $all_phrases->{$id} && $added{$id}) {
            if ($mbanner_phrases_id_ph{$id} eq $all_phrases->{$id}) {
                $moved{$id} = delete $added{$id};
            } else {
                $changed{$id} = delete $added{$id};
            }
        }

        # Если баннер в медиаплане был скопирован, то в копии фразы имеют такой же phraseIdHistory.
        # Такие фразы не считаем новыми.
        # phraseIdHistory: надо будет сконвертировать данные в mediaplan_banners_original -- где нет phraseIDHistory, добавить.
        # после этого можно здесь проверять наличие phraseIDHistory и удалить запись bsIdHistory в mediaplan_banners_original
        if ($mphrase->{phraseIdHistory} && $added{$id}) {
            delete $added{$id};
        }
    }
    # Складываем все в возвращаемый хэш
    my $res = {added => \%added, changed => \%changed, moved => \%moved};
    foreach my $k (keys %$res) {
        $changes{phrases} = hash_merge $changes{phrases}||{}, {map {$_ => $k} keys(%{$res->{$k}})};
    }
    delete $changes{phrases} unless %{$changes{phrases}};

    return {} if (! %{$reserve} && ! %{$changes{phrases}});
    return \%changes;
}

=head2 mediaplan_banner_search_params(\%FORM)

    Формирование хеша параметров поиска баннеров для get_mediaplan_banners
    и get_mediaplan_banners_count по данным формы
    Если параметров нет - возвращаем пустой хэш

=cut
sub mediaplan_banner_search_params {
    my ($form) = @_;
    my $search_banner;
    if ($form->{vcard_wo_href} && $form->{vcard_wo_href} ne 'false') {
        $search_banner->{'href__is_null'} = 1;
    } elsif ($form->{wo_phrases} && $form->{wo_phrases} ne 'false') {
        return {} unless defined($form->{cid}) && is_valid_id($form->{cid});
        my $bids = get_one_column_sql(PPC(cid => $form->{cid}), ["SELECT mbid FROM mediaplan_banners m LEFT JOIN mediaplan_bids mb USING (mbid)",
                                            where => {'m.cid'=>$form->{cid}}, "GROUP BY mbid HAVING COUNT(mb.id)=0 OR SUM(numword)=0"]) || [];
        $search_banner->{mbid} = $bids;
    }
    return $search_banner || {};
}


=head2 get_optimization_request

      Возвращает заявку на оптимизацию.
      Параметры:
      cid - номер кампании,
      request_id - номер заявки на ПП в БД
      options:
        status - в каких статусах должна быть заявка
        status__not_in - в каких статусах не может быть заявки
        media_user_data - добавить данные о медиапланере
        campaign_data - добавить данные о кампании
        hide_optimize_ready - вычислить флаг hide_optimize_ready

=cut
sub get_optimization_request {
    my ($cid, $request_id, %options) = @_;

    my $where = {cid => SHARD_IDS};
    $where->{request_id} = $request_id if $request_id;
    my @fields = @OPTIMIZATION_FIELDS;
    foreach (qw/status status__not_in/) {
        $where->{$_} = $options{$_} if (defined $options{$_});
    }
    if ($options{hide_optimize_ready}) {
        push @fields, 'IF(IFNULL(postpone_date, 0) < NOW(), 0, 1) AS hide_optimize_ready';
    }
    my $request = get_one_line_sql(PPC(cid => $cid), ["SELECT", join (",", @fields), "FROM optimizing_campaign_requests", where => $where, "ORDER BY request_id DESC LIMIT 1"]);
    return unless $request;
    if ($options{media_user_data} && $request->{MediaUID}) {
        $request->{MediaFIO} = get_one_user_field($request->{MediaUID}, 'fio');
    }
    if ($options{campaign_data}) {
        my $camp_data = CampaignQuery->get_campaign_data(cid => $cid, [qw/autobudget strategy_data/]);
        $request->{autobudget} = $camp_data->{autobudget};
        $request->{autobudget_sum} = from_json($camp_data->{strategy_data})->{sum};
    }
    $request->{is_second_aid} = is_second_aid($request->{req_type}) if $request;

    return $request;
}

sub is_second_aid {
    my $req_type = shift;
    return ($req_type || '') eq 'SecondAid' ? 1 : 0;    
}

1;
