package MetrikaCounters;

=head1 DESCRIPTION

Модуль для работы со счётчиками Метрики.

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq any/;

use MetrikaIntapi;
use Cache::SizeAwareMemoryCache;
use Yandex::HTTP;
use Settings;
use Direct::Model::MetrikaGoal;
use Yandex::DBTools;
use Yandex::I18n qw/iget/;
use JSON;
use LogTools qw/log_metrika_query/;
use SolomonTools;
use URI;
use Yandex::Trace;
use Yandex::Memcached qw//;
use Yandex::DateTime;
use Yandex::LiveFile;
use Yandex::ListUtils qw/chunks xminus/;

use Encode qw/decode_utf8/;

use RBACElementary;

use feature "state";

use base 'Exporter';
use Yandex::TVM2;
our @EXPORT = qw/
    get_uids_counters
    get_all_client_id_reps_counters
    get_all_uid_reps_counters
    is_counter_available_for
    filter_inaccessible_goal_ids
/;

#Максимальное количество счетчиков, обновляемых параллельными запросами
our $METRIKA_MAX_COUNTERS_PER_UPDATE_REQUEST = 50;

# Сколько счётчиков запрашивать за один запрос (ручка поддерживает до 200)
our $METRIKA_MAX_COUNTERS_PER_READ_REQUEST = 200;

# Фиксированный uid для внутренней рекламы для доступа к счетчикам сервисов Яндекса
# Константа INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS так-же есть в java в ru.yandex.direct.rbac.RbacService.java,
# и менять их надо одновременно
our $INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS = 1230527186;


my $CACHE = Cache::SizeAwareMemoryCache->new({namespace => 'MetrikaCounters', default_expires_in => 60});


=head2 is_counter_available_for

    my $is_available = is_counter_available_for($counter, $uid);

Проверка доступности счётчика для uid-а (или любого его представителя)
Параметры:
    $counter - номер счетчика
    $uid - uid
    skip_errors - не падать на ошибках из Метрики
        (если переданное значение - ссылка на скаляр - то при ошибке он будет разыменован и присвоен единице)


=cut

sub is_counter_available_for {
    my ($counter, $uid, %opts) = @_;

    my $counters = get_all_uid_reps_counters($uid, skip_errors => $opts{skip_errors});
    my $is_available = any {$_ == $counter} @$counters;
    return $is_available;
}


=head2 get_all_uid_reps_counters($uid)

Возвращает список всех счётчиков, доступных клиенту или любому его представителю
Параметры:
    $uid - uid
    skip_errors - не падать на ошибках из Метрики
        (если переданное значение - ссылка на скаляр - то при ошибке он будет разыменован и присвоен единице)

=cut

sub get_all_uid_reps_counters {
    my ($uid, %opts) = @_;

    my $result = $CACHE->get($uid);
    return $result  if $result;

    my $client_id = rbac_get_client_clientid_by_uid($uid);
    my $uids = defined($client_id) ? rbac_get_client_uids_by_clientid($client_id) : [$uid];

    my $counters_by_uid = get_uids_counters($uids, skip_errors => $opts{skip_errors});

    my @counters = uniq map {@$_} values %$counters_by_uid;
    if (!($opts{skip_errors} && ref $opts{skip_errors} eq 'SCALAR' && ${ $opts{skip_errors} })) {
    # кешируем только если не было ошибки
    $CACHE->set($_ => \@counters) for @$uids;
    }
    return \@counters;
}

=head2 get_all_client_id_reps_counters($client_id)

Возвращает список всех счётчиков, доступных клиенту или любому его представителю

=cut

sub get_all_client_id_reps_counters {
    my ($client_id, %opts) = @_;

    my $result = $CACHE->get($client_id);
    return $result  if $result;

    my $uids = $opts{is_internal_ad} ? [$INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS] : rbac_get_client_uids_by_clientid($client_id);

    my $counters_by_uid = get_uids_counters($uids, skip_errors => $opts{skip_errors});

    my @counters = uniq map {@$_} values %$counters_by_uid;
    if (!($opts{skip_errors} && ref $opts{skip_errors} eq 'SCALAR' && ${ $opts{skip_errors} })) {
    # кешируем только если не было ошибки
    $CACHE->set($_ => \@counters) for @$uids;
    }
    return \@counters;
}

=head2 get_client_id_reps_counters_by_ids($client_id,$counter_ids)

Возвращает список счётчиков, доступных клиенту или любому его представителю по идентификаторам счетчиков

=cut

sub get_client_id_reps_counters_by_ids {
    my ($client_id, $counter_ids, %opts) = @_;

    my $uids = $opts{is_internal_ad} ? [$INTERNAL_AD_UID_PRODUCTS_FOR_GET_METRIKA_COUNTERS] : rbac_get_client_uids_by_clientid($client_id);

    my $counters_by_uid = get_uids_counters_by_ids($uids, $counter_ids, skip_errors => $opts{skip_errors});

    my @counters = uniq map {@$_} values %$counters_by_uid;
    if (!($opts{skip_errors} && ref $opts{skip_errors} eq 'SCALAR' && ${ $opts{skip_errors} })) {
    }
    return \@counters;
}


=head2 get_uids_counters($uids)

Запрашивает в Метрике счётчики пользователя.
Параметры:
    $uids - arrayref uid'ов
    skip_errors - не падать на ошибках из Метрики
        (если переданное значение - ссылка на скаляр - то при ошибке он будет разыменован и присвоен единице)

Возвращает:

    { $uid => \@counters, ... }

=cut

sub get_uids_counters {
    my ($uids, %opts) = @_;

    return {} if ref $uids && !@$uids;

    my $answer = eval { MetrikaIntapi->new()->get_users_counters_num($uids) };
    if ($@) {
        if ($opts{skip_errors}) {
            ${ $opts{skip_errors} } = 1 if ref $opts{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "get_uids_counters");
            return {};
        } else {
            die $@;
        }
    }
    my %result = map {($_->{owner} => $_->{counter_ids})} @$answer;

    return \%result;
}

=head2 get_uids_counters_by_ids($uids)

Запрашивает в Метрике счётчики пользователя по идентификаторам счетчиков

=cut

sub get_uids_counters_by_ids {
    my ($uids, $counter_ids, %opts) = @_;

    return {} if ref $uids && !@$uids;
    return {} if ref $counter_ids && !@$counter_ids;

    my $answer = eval { MetrikaIntapi->new()->get_users_counters_num2($uids, $counter_ids) };
    if ($@) {
        if ($opts{skip_errors}) {
            ${ $opts{skip_errors} } = 1 if ref $opts{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "get_uids_counters");
            return {};
        } else {
            die $@;
        }
    }
    my %result = map {($_->{owner} => $_->{counter_ids})} @{$answer->{users}};

    return \%result;
}

=head2 get_client_counters_goals($client_id, $counters)

Получить активные и доступные клиенту цели по номерам счетчиков. Если цели счетчика есть в
Memcache, они берутся из него. 

=cut
sub get_client_counters_goals {
    my ($client_id, $counters, %opt) = @_;

    my $client_counters = get_all_client_id_reps_counters($client_id);
    
    my %client_counters_hash = map { $_ => 1 } @$client_counters;
    my @available_counters = grep { $client_counters_hash{$_} } @$counters;
    
    return get_counters_goals(\@available_counters, %opt);
}

=head2 get_counters_goals($counters)

Получить активные цели (без под целей) по номеру счетчика. Если цели счетчика есть в
Memcache, они берутся из него.

Параметры:
    $counters - [] массив, номера счетчиков
    skip_errors - не падать на ошибках из Метрики
        (если переданное значение - ссылка на скаляр - то при ошибке он будет разыменован и присвоен единице)
    get_steps - получать все подцели составных целей

Опции:
    no_cache

Результат:
    {
        counter_number_1 => [Direct::Model::MetrikaGoal, Direct::Model::MetrikaGoal, ....],
        counter_number_2 => undef 
    }
    ключ - номер счетчика, значение - массив [] целей, значение undef - ошибка получения целей

=cut

sub get_counters_goals {
    my ($counters, %opt) = @_;

    my $memd = Yandex::Memcached->new( servers =>
        $Settings::MEMCACHED_SERVERS );
    my $stored_goals = $opt{no_cache}
        ? {}
        : $memd->get_multi( map { sprintf "metrika_counter_goals_v2_%d", $_ } @$counters );

    my %counter_goals;
    my @online_counters;
    for my $counter (@$counters) {
        if (exists $stored_goals->{"metrika_counter_goals_v2_$counter"}) {
            push @{$counter_goals{$counter}}, map { 
                Direct::Model::MetrikaGoal->new({
                    goal_id => $_->{id},
                    goal_name => $_->{name},
                    goal_status => 'Active',
                    counter_status => 'Active',
                    goal_type => $_->{type},
                    goals_count => 0,
                    context_goals_count => 0,
                    goal_source => $_->{source},
                });
            } @{$stored_goals->{"metrika_counter_goals_v2_$counter"}};
        } else {
            push @online_counters, $counter;
        }
    }

    if (@online_counters) {
        my $downloaded_counters = download_counter_goals(\@online_counters,
            skip_errors    => $opt{skip_errors},
            get_steps      => $opt{get_steps},
        );

        my @memd_set_keys;
        for my $counter (keys %$downloaded_counters) {
            $counter_goals{$counter} = $downloaded_counters->{$counter};
            next if !defined $downloaded_counters->{$counter};

            my @memd_goals;
            for my $goal (@{$downloaded_counters->{$counter}}) {
                my $mgoal = {id => $goal->goal_id, name => $goal->goal_name, type => $goal->goal_type, source => $goal->goal_source};
                push @memd_goals, $mgoal;
            }
            push @memd_set_keys, ["metrika_counter_goals_v2_$counter", \@memd_goals, 10 * 60]; # 10 minutes
        }
        $memd->set_multi(@memd_set_keys) if @memd_set_keys;
    }

    return \%counter_goals;
}

=head2 download_counter_goals($counters)

Загрузить из Метрики активные цели по номеру счётчика.

Параметры:
    $counters – [] массив, номера счётчиков

Опции:
    skip_errors – не падать при всех ошибках

Результат:
    {
        counter_number_1 => [Direct::Model::MetrikaGoal, Direct::Model::MetrikaGoal, ....],
        counter_number_2 => undef
    }
    ключ – номер счётчика, значение – массив целей, значение [] – ошибка получения целей, их отсутствие или отсутствие счётчика

=cut

sub download_counter_goals {
    my ($counters, %opt) = @_;

    my $method = 'direct/counter_goals';
    my $URL = "${Settings::METRIKA_INT_API_READONLY_HOST}/$method";

    my $http_requests;
    my $chunk_no = 0;
    for my $chunk (chunks($counters, $METRIKA_MAX_COUNTERS_PER_READ_REQUEST)) {
        my $uri = URI->new($URL);
        $uri->query_form(counters => join(',', @$chunk));
        $http_requests->{$chunk_no} = {url => $uri->as_string};
        $chunk_no++;
    }

    my $ticket = eval {Yandex::TVM2::get_ticket($Settings::METRIKA_TVM2_ID)}
        or die "Cannot get ticket for $Settings::METRIKA_TVM2_ID: $@";

    my $profile = Yandex::Trace::new_profile('metrika:counter_goals', obj_num => scalar @$counters);
    my $http_result = Yandex::HTTP::http_parallel_request(
        GET => $http_requests,
        ipv6_prefer => 1,
        headers => {
            'X-Ya-Service-Ticket' => $ticket,
        }
    );
    log_metrika_query({
        action => 'metrika_counter_goals',
        requests => $http_requests,
        result => $http_result,
    });
    undef $profile;

    my %counter_goals = map {$_ => []} @$counters;

    my %stat;
    my $die_message;

    for (my $i = 0; $i < $chunk_no; $i++) {
        my $response = $http_result->{$i};

        if ($response->{is_success}) {
            my %counter_goals_chunk;
            my $arr = eval { from_json(decode_utf8($response->{content}))->{goals} };
            if ($@ || ref $arr ne 'ARRAY') {
                $die_message ||= $@ || 'expected ARRAY response';
                $stat{unparsable}++;
                next;
            }
            for my $counter_goals (@$arr) {
                # подцели составных целей Метрика присылает в цели в массиве steps
                for my $goal ($counter_goals->{goal}) {
                    next if _is_unknown_goal_type($goal);
                    push @{$counter_goals_chunk{$counter_goals->{counter_id}}}, getMetrikaGoalModel($goal);
                    if ($opt{get_steps} && defined $goal->{steps}) {
                        push @{$counter_goals_chunk{$counter_goals->{counter_id}}}, map { $_->{parent_goal_id} = $goal->{id}; getMetrikaGoalModel($_) } @{$goal->{steps}};
                    }
                }
            }
            for my $counter_id (keys %counter_goals_chunk) {
                $counter_goals{$counter_id} = $counter_goals_chunk{$counter_id};
            }
            $stat{success}++;
        } else {
            my $url = $http_requests->{$i}->{url};
            if ($opt{skip_errors}) {
                ${ $opt{skip_errors} } = 1 if ref $opt{skip_errors} eq 'SCALAR';
                LogTools::log_messages("skip_metrika_error", "download_counter_goals");
            } else {
                $die_message ||= "Metrika request failed for url $url: $response->{headers}->{Status} $response->{headers}->{Reason}";
            }
            if ($response->{headers}->{Status} =~ /^5/) {
                $stat{server_error}++;
            } elsif ($response->{headers}->{Status} =~ /^4/) {
                $stat{client_error}++;
            } else {
                $stat{unknown}++;
            }
        }
    }
    SolomonTools::send_requests_stat({external_system => "metrika", "sub_system" => "internal_api", method => $method}, \%stat);
    if ($die_message) {
        Carp::confess($die_message);
    }

    return \%counter_goals;
}

{
    my %known_goal_types = map {$_ => undef} @{Direct::Model::MetrikaGoal::GOAL_TYPE()};

    sub _is_unknown_goal_type {
        my $goal = shift;
        if (!exists($known_goal_types{$goal->{type}})) {
            LogTools::log_messages("metrika_goals_error", "unknown goal type: $goal->{type}");
            return 1;
        }
        return 0;
    }
}

=head2 getMetrikaGoalModel($goal)

Создать модель MetrikaGoal из хеша
Параметры:
  $goal - хеш с информацией о цели. Должен содержать обязательные поля id, name, type, goal_source.
          Если задан parent_goal_id заполяем соотв. поле в модели
=cut

sub getMetrikaGoalModel {
    my ($goal) = @_;

    my $model =  Direct::Model::MetrikaGoal->new({
        goal_id             => $goal->{id},
        goal_name           => $goal->{name},
        goal_status         => 'Active',
        counter_status      => 'Active',
        goal_type           => $goal->{type},
        goals_count         => 0,
        context_goals_count => 0,
        goal_source         => $goal->{goal_source}
    });

    if ($goal->{parent_goal_id}) {
        $model->parent_goal_id($goal->{parent_goal_id});
    }
    return $model;
}

=head2 check_counters_ecommerce($counters)

Проверить настройку ecommerce на счетчиках.
Проверка осуществляется путем запроса статистики по просмотрам товаров за последние 14 дней на счетчике.
Если просмотров больше нуля - значит ecommerce настроен верно.

Параметры:
    $counters -> [] массив, номера счетчиков

Результат:
    Хеш, вида {counter_num1 => true/false/undef, ...}
        true  -> ecommerce настроен верно
        false -> ecommerce не настроен
        undef -> не удалось получить данные (например, счетчик не существует)

=cut

sub check_counters_ecommerce {
    my ($counters) = @_;

    my (%http_requests, %result);

    for my $counter (@$counters) {
        my $uri = URI->new(sprintf("%s/stat/v1/data", $Settings::METRIKA_API_URL, $counter));
        $uri->query_form(
            ids => $counter,
            dimensions => undef,
            metrics => "ym:s:productImpressions",
            date1 => Yandex::DateTime->now()->subtract(days => 14)->strftime("%Y-%m-%d"),
            date2 => Yandex::DateTime->now()->strftime("%Y-%m-%d"),
        );

        $http_requests{$counter} = {url => $uri->as_string};
    }
    my $ticket = eval {Yandex::TVM2::get_ticket($Settings::METRIKA_API_TVM2_ID)}
        or die "Cannot get ticket for $Settings::METRIKA_API_TVM2_ID: $@";

    my $profile = Yandex::Trace::new_profile('metrika:check_counters_ecommerce', obj_num => scalar(keys %http_requests));
    my $http_response = Yandex::HTTP::http_parallel_request(
        GET => \%http_requests,
        ipv6_prefer => 1,
        headers => {
            'X-Ya-Service-Ticket' => $ticket,
        }
    );
    log_metrika_query({
        action => 'metrika_get_productImpressions_by_counters',
        requests => \%http_requests,
        result => $http_response,
    });
    undef $profile;

    for my $counter (keys %http_requests) {
        my $resp = $http_response->{$counter};

        if ($resp->{is_success}) {
            eval {
                my $num = decode_json($resp->{content})->{data}[0]{metrics}[0] || 0; # кол-во просмотров товаров
                $result{$counter} = $num > 0 ? 1 : 0;
                1;
            } and next;
        }

        $result{$counter} = undef;
    }

    return \%result;
}

=head2 update_counter_grants($grants_by_counter)

Установить для переданных счетчиков указанные списки доступа.

Параметры:
    $grants_by_counter - HashRef:
    {
        counter_id1 => [{user_login => login1, perm => perm1}, {user_login => login2, perm => perm2}]
        ...
    }

Результат:
    HashRef
    {
        counter_id1 => 0|1 - признак успешности операции,
        counter_id2 => ...
    }
=cut

# Метод удалим в DIRECT-105584
sub update_counter_grants {
    my ($grants_by_counter) = @_;

    my $result;    
    foreach my $counters (chunks([keys %$grants_by_counter], $METRIKA_MAX_COUNTERS_PER_UPDATE_REQUEST)){
        my $http_requests = {
            map {
                my $counter = $_;
                my ($uri) = URI->new(sprintf "%s/management/v1/counter/%d", $Settings::METRIKA_API_URL, $counter);
                $uri->query_form(
                    quota_ignore => 1,
                );
                $counter => {
                    url => $uri->as_string,
                    body => to_json({counter => {grants => $grants_by_counter->{$counter}}}),
                    headers => {'Content-Type' => 'application/json'},
                },
            } @$counters
        };
        my $ticket = eval {Yandex::TVM2::get_ticket($Settings::METRIKA_API_TVM2_ID)}
            or die "Cannot get ticket for $Settings::METRIKA_API_TVM2_ID: $@";
        my $profile = Yandex::Trace::new_profile('metrika:update_counter_grants', obj_num => scalar (keys %$grants_by_counter));
        my $http_result = Yandex::HTTP::http_parallel_request(
            PUT => $http_requests,
            ipv6_prefer => 1,
            headers => {
                'Content-Type' => 'application/json',
                'X-Ya-Service-Ticket' => $ticket,
            },
        );
        log_metrika_query({
            action => 'update_counter_grants',
            requests => $http_requests,
            result => $http_result,
        });
        undef $profile;
    
        for my $counter (keys %$grants_by_counter) {
            $result->{$counter} = $http_result->{$counter}->{is_success} ? 1 : 0;
        }
    }
    
    return $result;
}


sub _define_auth_token {
    my ($auth_token, $auth_token_file) = @_;

    state %token_live_file;

    unless (defined $auth_token) {
        $token_live_file{$auth_token_file} //= Yandex::LiveFile->new(filename => $auth_token_file);
        $auth_token = $token_live_file{$auth_token_file}->data;
        chomp $auth_token;
    }

    return $auth_token;
}

=head2 filter_inaccessible_goal_ids($client_id, $goal_ids, %options)

Отфильтровываем из списка цели, недоступные клиенту

=cut

sub filter_inaccessible_goal_ids {
    my ($client_id, $goal_ids, %options) = @_;

    my $client_goal_ids = { map {$_ => 1} @{get_client_goal_ids($client_id, goal_ids_for_check_ecom => $goal_ids,
        extra_goal_ids => $options{preserve_goal_ids}, extra_counter_ids => $options{extra_counter_ids},
        is_internal_ad => $options{is_internal_ad})} };
    my $filtered_goal_ids = [ grep { $client_goal_ids->{$_} } @$goal_ids ];

    return $filtered_goal_ids;
}

=head2 get_client_goal_ids($client_id, %options)

Возвращаем список целей, доступных клиенту

=cut

sub get_client_goal_ids {
    my ($client_id, %options) = @_;

    my $client_counters = get_all_client_id_reps_counters($client_id, is_internal_ad => $options{is_internal_ad});

    if ($options{extra_counter_ids} && @{$options{extra_counter_ids}}) {
        @$client_counters = uniq @$client_counters, @{$options{extra_counter_ids}};
    }

    my @client_goal_ids = map { $_->goal_id => 1 } grep { $_ } map { @$_ } grep { $_ }
            values get_counters_goals($client_counters, skip_errors => 1, get_steps => 1);

    if ($options{extra_goal_ids}) {
        @client_goal_ids = uniq @client_goal_ids, @{$options{extra_goal_ids}};
    }

    if ($options{goal_ids_for_check_ecom}) {
        my $goals = get_metrika_goals(where => { goal_id => $options{goal_ids_for_check_ecom} });
        my %counters = map {$_ => 1} @{$client_counters};
        # виртуальные цели ecommerce добавляем отдельно чтобы при фильтрации по счетчикам не потеряли из списка доступных целей
        for my $goal_id (keys %$goals) {
            if ($goals->{$goal_id}->{goal_type} eq 'ecommerce' && exists($counters{$goals->{$goal_id}->{ecommerce_counter_id}})) {
                push @client_goal_ids, $goal_id;
            }
        }
    }

    return \@client_goal_ids;
}

=head2 get_metrika_goals

    возвращаем по условию цели метрики
    $goals = get_metrika_goals(where => {goal_id => [1461, 1462]});
    $goals = get_metrika_goals(where => {goal_id => [1461, 1462]}, OR => {parent_goal_id => [129]});

    на входе условия:
        where => {goal_id => [...]}
        where => {parent_goal_id => [...]}

    на выходе:
    {
      '1461' => {
                  'counter_status' => 'Active',
                  'goal_id' => '1461',
                  'goal_status' => 'Active',
                  'goal_type' => 'number',
                  'name' => "Цель 1",
                  'parent_goal_id' => 2333,
                  'subgoal_index' => undef
                },
      '1462' => {
                  'counter_status' => 'Active',
                  'goal_id' => '1462',
                  'goal_status' => 'Deleted',
                  'goal_type' => 'number',
                  'name' => 'Goal name',
                  'parent_goal_id' => undef,
                  'subgoal_index' => undef
                },
    }

=cut

sub get_metrika_goals {
    my @where = @_;

    my $goals = get_hashes_hash_sql(PPCDICT, ["SELECT goal_id, name, goal_status, counter_status, goal_type, parent_goal_id, subgoal_index
                                               FROM metrika_goals", @where]);
    foreach my $goal_id (keys %$goals) {
        if ($goals->{$goal_id}->{goal_type} eq 'ecommerce') {
            $goals->{$goal_id}->{ecommerce_counter_id} = $goals->{$goal_id}->{name};
            # для виртуальных ecommerce целей заменяем название на переведенное на язык пользователя
            $goals->{$goal_id}->{name} = iget("eCommerce: Покупка (счетчик № %s)", $goals->{$goal_id}->{name});
        }
    }
    return $goals;
}

=head2 get_spav_counter_ids_from_db_by_cids

    возвращаем из БД идентификаторы технических счетчиков Метрики из организаций Справочника, для заданных кампаний

=cut


sub get_spav_counter_ids_from_db_by_cids {
    my ($cids) = @_;
    return get_one_column_sql(PPC(cid => $cids),
        ["SELECT DISTINCT metrika_counter FROM metrika_counters", WHERE => { cid => $cids, source => 'sprav', is_deleted => 0 } ]);
}

=head2 get_counter_ids_from_db_by_cids

    возвращаем из БД идентификаторы счетчиков Метрики для заданных кампаний

=cut

sub get_counter_ids_from_db_by_cids {
    my ($cids) = @_;
    return get_one_column_sql(PPC(cid => $cids),
        ["SELECT DISTINCT metrika_counter FROM metrika_counters", WHERE => { cid => $cids } ]);
}

=head2 get_grant_access_requests_status_by_ids($uids)

Запрашивает в Метрике информации о запрашиваемом ранее доступе к счетчикам

Параметры:
    $uid        - id пользователя
    skip_errors - не падать на ошибках из Метрики

=cut

sub get_grant_access_requests_status_by_ids {
    my ($uid, $counter_ids, %opts) = @_;

    return {} if ref $counter_ids && !@$counter_ids;

    my $answer = eval { MetrikaIntapi->new()->get_grant_access_requests_status($uid, $counter_ids) };
    if ($@) {
        if ($opts{skip_errors}) {
            ${ $opts{skip_errors} } = 1 if ref $opts{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "get_grant_access_requests_status_by_ids");
            return {};
        } else {
            die $@;
        }
    }

    return $answer;
}

=head2 add_or_update_goals_for_cid

    дописываем цели кампаний, которых нет в базе. возвращает цели
=cut


sub add_or_update_goals_for_cid {
    my ($cid, $goals) = @_;
    
    if (@$goals) {
        my %metrika_goals = map { $_->goal_id => $_ } @$goals;
        
        my $all_goals_ids = [keys %metrika_goals];
        my $known_goal_ids =  get_one_column_sql(PPCDICT(),[
            q/SELECT goal_id FROM metrika_goals/,
            WHERE => {goal_id => $all_goals_ids}]);

        my $missed_at_ppcdict_goal_ids = xminus($all_goals_ids, $known_goal_ids);

        if (@$missed_at_ppcdict_goal_ids){
            do_mass_insert_sql(PPCDICT(), 'INSERT IGNORE INTO metrika_goals (goal_id, name, goal_type) values %s',
                [map {[$_->goal_id, $_->goal_name, $_->goal_type]} @metrika_goals{@$missed_at_ppcdict_goal_ids}]
            );
        }
        if ($cid) {
            my $related_goals = get_one_column_sql(PPC(cid => $cid),[
                q/SELECT goal_id FROM camp_metrika_goals/,
                WHERE => {cid => $cid}],
            );

            my $unknown_goals = xminus([keys %metrika_goals], $related_goals);
            if (@$unknown_goals){
                do_mass_insert_sql(PPC(cid => $cid), 'INSERT IGNORE INTO camp_metrika_goals (cid, goal_id) values %s',
                    [map {[$cid, $_]} @$unknown_goals]
                );
            }
        }
        return [map { {
            id => $_->goal_id,
            name => $_->goal_name,
            goal_type => $_->goal_type,
            parent_goal_id => $_->has_parent_goal_id ? $_->parent_goal_id : undef } } @$goals];
    }
    
    return [];
} 

1;
