package MetrikaIntapi;

=head1 DESCRIPTION

Модуль для общения с интапи Метрики.

=head1 SINOPSYS

    my $metrika = MetrikaIntapi->new();
    my $response = $metrika->call($method => $data);

=cut

use Direct::Modern;
use Mouse;

use Yandex::ListUtils qw/xminus/;
use HTTP::Request;
use LWP::UserAgent;
use Yandex::HTTP;
use Yandex::TVM2;
use JSON;

use SolomonTools;

use Log::Any '$log', prefix => '[MetrikaIntapi] ';
use constant FETCH_TIMEOUT => 240;

=head2 $METRIKA_INTAPI_URL

Дефолтный урл интапи Метрики

=cut

our $METRIKA_INTAPI_URL //= 'http://internalapi.metrika.yandex.ru:8096';
our $METRIKA_TVM2_ID;

our $METRIKA_INTAPI_TIMEOUT //= 120;


has intapi_url => (is => 'ro', isa => 'Str', default => sub { $METRIKA_INTAPI_URL });
has timeout => (is => 'ro', isa => 'Int', default => sub { $METRIKA_INTAPI_TIMEOUT });
has ua => (is => 'ro', lazy => 1, builder => '_build_ua');


=head2 get_users_counters_num

Запрос счётчиков для uid-ов

=cut

sub get_users_counters_num {
    my $self = shift;
    my ($uids) = @_;

    my $uids_str = join q{,} => @{ ref $uids ? $uids : [$uids] };

    # URI также определён в $Settings::METRIKA_USERS_COUNTERS_NUM_URL, при изменении тут нужно поменять и эту переменную
    $log->trace("get tvm2 ticket");
    my $ticket = eval{Yandex::TVM2::get_ticket($METRIKA_TVM2_ID) } or do { $log->warn("Cannot get ticket for $METRIKA_TVM2_ID: $@"); die $@; };
    return $self->call('direct/user_counters_num' => { uids => $uids_str }, undef, tvm2_ticket => $ticket); #&external_class=0'
}

=head2 get_users_counters_num2

Запрос счётчиков для uid-ов по counterIds

=cut

sub get_users_counters_num2 {
    my $self = shift;
    my ($uids, $counter_ids) = @_;

    my $uids_str = join q{,} => @{ ref $uids ? $uids : [$uids] };
    my $counter_ids_str = join q{,} => @{ ref $counter_ids ? $counter_ids : [$counter_ids] };

    # URI также определён в $Settings::METRIKA_USERS_COUNTERS_NUM_URL, при изменении тут нужно поменять и эту переменную
    $log->trace("get tvm2 ticket");
    my $ticket = eval{Yandex::TVM2::get_ticket($METRIKA_TVM2_ID) } or do { $log->warn("Cannot get ticket for $METRIKA_TVM2_ID: $@"); die $@; };
    return $self->call('direct/user_counters_num2' => { uids => $uids_str, counterIds => $counter_ids_str }
        , undef, tvm2_ticket => $ticket); #&external_class=0'
}

=head2 get_existent_counters

Запрос существующих счётчиков и их source

=cut

sub get_existent_counters {
    my $self = shift;
    my ($counters) = @_;

    my %counter_id_list_body = (
        counter_id_list => $counters
    );

    $log->trace("get tvm2 ticket");
    my $ticket = eval{Yandex::TVM2::get_ticket($METRIKA_TVM2_ID) } or do { $log->warn("Cannot get ticket for $METRIKA_TVM2_ID: $@"); die $@; };
    return $self->call('internal/existent_counters', undef, \%counter_id_list_body, tvm2_ticket => $ticket);
}

=head2 get_grant_access_requests_status

Запрос на получение информации о запрашиваемом ранее доступе к счетчикам

=cut

sub get_grant_access_requests_status {
    my $self = shift;
    my ($uid, $counters) = @_;

    my %counter_id_list_body = (
        counter_id_list => $counters
    );

    $log->trace("get tvm2 ticket");
    my $ticket = eval{Yandex::TVM2::get_ticket($METRIKA_TVM2_ID) } or do { $log->warn("Cannot get ticket for $METRIKA_TVM2_ID: $@"); die $@; };
    my $response = $self->call('internal/user/'.$uid.'/grant_access_requests_status', undef, \%counter_id_list_body, tvm2_ticket => $ticket);
    
    my %result;
    foreach my $line (@{$response->{grant_request_status_list}}) {
        if (JSON::is_bool($line->{access_requested})) {
            $result{$line->{counter_id}} = ( $line->{access_requested} ? 1 : 0 );
        } else {
            $result{$line->{counter_id}} = 0;
        }
    }

    return \%result;
}

=head2 get_counters_extended_list

    Получение данных счетчика из Метрики, возвращает массив хешей.

=cut

sub get_counters_extended_list {
    my ($form_param) = @_;

    my $method = 'direct/user_counters_num_extended';
    my $URL = "${Settings::METRIKA_INT_API_READONLY_HOST}/$method";

    my @metrica_response_fields = qw/counters/;
    my $ticket = eval {Yandex::TVM2::get_ticket($Settings::METRIKA_TVM2_ID)}
        or die "Cannot get ticket for $Settings::METRIKA_TVM2_ID: $@";
    my $resp = Yandex::HTTP::submit_form('POST', $URL, $form_param, timeout => FETCH_TIMEOUT,
        headers => {'X-Ya-Service-Ticket' => $ticket});
    if ($resp->is_error) {
        my $status = SolomonTools::guess_status_by_code($resp);
        _send_metrics_to_solomon($method, {$status => 1});

        my $response_code = $resp->code();
        my $response_status_line = $resp->status_line();
        my $url = Yandex::HTTP::make_url($URL, $form_param);
        my $error = "Can't fetch statistic from $url. Got $response_code: $response_status_line";
        die $error;
    }
    my $content = $resp->decoded_content;
    if (!$content) {
        _send_metrics_to_solomon($method, {success => 1});
        return undef;
    }

    # разобрать ответ Метрики
    my $lines = eval { from_json($content) };
    if ($@ || ref $lines ne "ARRAY") {
        _send_metrics_to_solomon($method, {unparsable => 1});
        die ($@ || "expected ARRAY response");
    }
    foreach my $line (@{$lines}) {
        my @fields = keys %{$line};
        # если не получили всех нужных полей
        if (@{xminus(\@metrica_response_fields, \@fields)} > 0) {
            my $error = "Error in metrica response! Array must contain fields: ".join(", ",  \@metrica_response_fields).". Metrica response: ".join(", ",  @fields);
            _send_metrics_to_solomon($method, {unparsable => 1});
            die($error);
        }
    }
    _send_metrics_to_solomon($method, {success => 1});
    return $lines;
}

=head2 call

Универсальный запрос

=cut

sub call {
    my $self = shift;
    my ($method, $form, $body, %opt) = @_;

    my $url = $self->intapi_url . "/$method";
    my ($payload, $content_type);
    if ($body) {
        $payload = encode_json($body);
        $content_type = 'application/json';
    } else {
        $payload = Yandex::HTTP::get_param_string($form);
        $content_type = 'application/x-www-form-urlencoded';
    }

    $log->info("Call $url");
    $log->tracef(" > post data: %s", $payload) if $log->is_trace;
    my $resp;
    if (exists $opt{tvm2_ticket}) {
        my $headers = HTTP::Headers->new('X-Ya-Service-Ticket' => $opt{tvm2_ticket}, 'Content-Type' => $content_type);
        my $req = HTTP::Request->new(POST => $url, $headers, $payload);
        $resp = $self->ua->request($req);
    } else {
        $resp = $self->ua->post($url, Content => $payload);
    }

    if ($resp->is_error) {
        my $error = $resp->status_line;
        my $status = SolomonTools::guess_status_by_code($error);
        _send_metrics_to_solomon($method, {$status => 1});

        $log->info("Got error: $error");
        croak "Metrika call error: $error";
    }

    my $content = $resp->decoded_content;
    $log->infof('Got response %d characters', length($content));
    $log->tracef(" > response data: $content")  if $log->is_trace;

    if (!$content) {
        _send_metrics_to_solomon($method, {success => 1});
        return undef;
    }
    
    my $decoded_response = eval { from_json($content) };
    if (!$decoded_response) {
        _send_metrics_to_solomon($method, {unparsable => 1});
        croak "Invalid response format"
    }
    
    _send_metrics_to_solomon($method, {success => 1});
    return $decoded_response;
}

sub _send_metrics_to_solomon {
    my ($method, $stat) = @_;

    SolomonTools::send_requests_stat({external_system => "metrika", "sub_system" => "internal_api", method => $method}, $stat);
}

sub _build_ua {
    my $self = shift;

    return LWP::UserAgent->new(timeout => $self->timeout);
}


__PACKAGE__->meta->make_immutable();
1;

