package Metro;

=head1 DESCRIPTION

    $Id:$
    
    Модуль для работы с метро (геокодирование и геобаза)

=cut

use Direct::Modern;

use Settings;

use Yandex::HashUtils;
use Yandex::URL qw/get_num_level_domain/;
use List::Util qw/first/;
use List::MoreUtils qw/uniq/;
use Yandex::I18n;
use GeoTools;
use geo_regions;

use Text::Diff;
use Math::Trig qw/asin deg2rad/;
use Time::HiRes;

use JavaIntapi::GeosearchSearchObject;

use base qw/Exporter/;
our @EXPORT = qw/
search_metro
validate_metro
list_metro
format_metro
compare_metro
/;

=head2 search_metro($x, $y, $city, $host)

Найти (с помощью геокодера) станцию метро по координатам
Возвращает объект вида
{
    region_id => 20390,
    street => "Калининская линия",
    text => "Россия, Москва, Калининская линия, метро Новогиреево",
    name => "Новогиреево",
}

=cut

sub search_metro
{
    my ($x, $y, $city, $host) = @_;
    return undef unless $city;
    $host = $host ? get_num_level_domain($host, 3) : 'direct.yandex.ru';
    my ($metro, $error) = JavaIntapi::GeosearchSearchObject->_search_metro($x,$y);
    if ($metro) {
        my ($mx, $my) = @{$metro->{point}}{qw/x y/};
        $metro = hash_cut $metro, qw/text street geo_id/; # название станции и линия
        $metro->{name} = format_metro($metro);
        delete $metro->{geo_id};
        $metro->{region_id} = metro_region_id($metro->{name}, get_geo_numbers($city));
        $metro->{distance} = _calc_distance($y, $x, $my, $mx); # расстояние от точки до метро
        $metro = undef unless $metro->{region_id}; # не найден region_id - значит нет такой станции в городе
    }
    return $metro;
}

=head2 validate_metro($metro, $city)

Функция проверят, что указанная станция существует

=cut

sub validate_metro
{
    my ($metro, $city) = @_;
    $city =~ s/^\s+//;
    $city =~ s/\s+$//;
    return 1 if ($metro||'') eq ''; # станция не указана (или $metro == 0 - выбран пункт "--не выбрана--") - ок
    return 0 if !$city;
    return 0 unless $metro =~ /^\d+$/;
    return (grep { $_->{region_id} == $metro } @{(list_metro($city))}) ? 1 : 0;
}

=head2 list_metro

Все станции метро в указанном городе. Город - строка или geo_regions.region_id
[
  {
    name => "Автово",
    region_id => 20302,
  },
  ...
]

=cut

sub list_metro
{
    my ($city) = @_;

    if ($city && $city !~ /^\d+$/) {
        my $current_lang = Yandex::I18n::current_lang();
        # Перебираем названия на всех языках, в качестве маленькой оптимизации начинаем с текущего
        # (get_geo_numbers при первом вызове кэширует данные для языка)
        my @languages = ($current_lang, grep { $_ ne $current_lang } keys %Yandex::I18n::LOCALES);
        for my $lang (@languages) {
            my $city_geoid = get_geo_numbers($city, lang => $lang);
            if ($city_geoid) {
                $city = $city_geoid;
                last;
            }
        }
    }
    return [] unless $city;

    # если указана москва - добавляем московскую область:
    # ostroumov@ У нас есть станции  только для городов, а Мякинино в Подмосковье.  Подумаю, что делать в этом случае
    $city = $geo_regions::MSK_REGION if $city == $geo_regions::MOSCOW;
    return [] unless exists $geo_regions::METRO{$city};  # METRO is a locked hash, we'll die on next line without this check
    my $metro = $geo_regions::METRO{$city};
    return $metro;
}

=head2 format_metro

Функция преобразует название станции метро в вид, совпадающий с geo_regions и пригодный для вывода пользователю
На вход подается объект из геокодера
Чтобы искать расхождения между данными из геокодера и геобазы см. compare_metro, ppcMetroCheck.pl

=cut

sub format_metro
{
    my $data = shift;

    # если geo_id известен - берём имя по нему
    if (my $metro_id = $data->{geo_id}) {
        state $metro_name_by_id = {map {map {($_->{region_id} => $_->{name})} @$_} values %geo_regions::METRO};
        my $metro_name = $metro_name_by_id->{$metro_id};
        return $metro_name  if $metro_name;
    }

    my ($text, $street) = @$data{qw/text street/};
    $street ||= '';
    
    $text =~ s/^.*метро\s+//;
    $street =~ s/\s+линия$//;
    $text =~ s/ё/е/g;
    
    # moscow quirks
    if (lc $text eq 'арбатская' or lc $text eq 'смоленская') {
        $text .= " ($street)";
    }
    if (lc $text eq 'тимирязевская' and $street =~ /монорельс/i) {
        $text .= " (монорельс)";
    }
    $text =~ s/ имени / им. /gi;

    # spb quirks
    $text =~ s/(невского)\s+[12]$/$1/gi; # площадь александра невского 1/2

    # kiev quirks
    $text =~ s/\"//g;
    $text =~ s/Майдан/Площадь/g;

    return $text;
}

=head2 compare_metro($city_id, $xy, $verbose)

Служебная функция для поиска расхождений между данными в ppcdict.geo_regions и данными из геокодера
Использование: perl -ME -MMetro -e 'compare_metro(2,"30.257953,59.971618"',$verbose)
Второй параметр - координаты точки где-нибудь внутри города
С ключом $verbose=1 - показывает два списка (геокодера и базы)

37.625707,55.749908 - Москва
30.257953,59.971618 - Питер
30.52623,50.447896  - Киев

=cut

sub compare_metro
{
    my ($city, $xy, $verbose, %opt) = @_;
    my $geobase_metro = list_metro($city);
    my @geocoder_metro;
    $opt{drift_count} = 10 unless defined $opt{drift_count}; # сколько промежуточных точек посетить
    $opt{drift_distance} = 1 unless defined $opt{drift_distance}; # на какой диаметр от начальной точки расходимся, в градусах

    my ($x, $y) = split /,/, $xy;
    # если искать вокруг одной точки - геокодер находит не более 100 станций
    # поэтому, расходимся в разных направлениях
    my @drift;
    if ($opt{drift_count} and $opt{drift_distance}) {
        @drift = map {[
            sprintf "%.6f", $x+sin($_)/($_*$opt{drift_distance}),
            sprintf "%.6f", $y+cos($_)/($_*$opt{drift_distance}) 
        ]} (1 .. $opt{drift_count});
    }
    else {
        @drift = ($xy);
    }

    for my $XY (@drift) {
        my ($found, $error) = JavaIntapi::GeosearchSearchObject->_search_metro(@$XY);
        push @geocoder_metro, grep { $_ !~ /московская область/ }  map { lc format_metro($_) } @$found;
        Time::HiRes::sleep 0.1;
    }
    @geocoder_metro = uniq @geocoder_metro;
    my $h_geobase_metro = { map { lc $_->{name} => $_->{region_id} } @$geobase_metro };
    my $ok = 1;
    my $msg = '';
    # проверяем, что все станции, найденные геокодером, есть в нашей базе
    for (sort @geocoder_metro) {
        my $id = $h_geobase_metro->{$_};
        unless ($id) {
            $ok = 0;
            $msg .= "missing in geobase: $_\n";
        }
    }
    $msg .= "\n" if !$ok;
    # проверяем, что все станции, которые есть в нашей базе, находит геокодер
    my $h_geocoder_metro = { map { lc $_ => 1 } @geocoder_metro };
    for (sort keys %$h_geobase_metro) {
        unless ($h_geocoder_metro->{$_}) {
            $ok = 0;
            $msg .= "missing in geocoder: $_\n";
        }
    }
    if ($verbose and not $ok) {
        my $geocoder_txt = join "\n", uniq sort @geocoder_metro;
        my $geobase_txt = join "\n", uniq sort keys %$h_geobase_metro;
        
        $msg .= "\n===diff geocoder/geobase:\n";
        $msg .= diff \$geocoder_txt, \$geobase_txt;
        $msg .= "\n\n=== geocoder ===\n";
        $msg .= (join "\n", sort @geocoder_metro);
        $msg .= "\n\n=== geobase ===\n";
        $msg .= (join "\n", sort keys %$h_geobase_metro);
    }
    return $msg;
}

=head2 metro_region_id

По имени станции метро и id города получить region_id станции

%geobase::Metro = (
    $city => [
        {
            name => ...,
            region_id => ...,
        },
    ],
);

=cut

sub metro_region_id
{
    my ($name, $city_id) = @_;
    $city_id = $geo_regions::MSK_REGION if ($city_id||0) == $geo_regions::MOSCOW;
    my $metro = first { lc $_->{name} eq lc $name } @{ 
        exists $geo_regions::METRO{$city_id} ? $geo_regions::METRO{$city_id} : [] 
    };
    return $metro ? $metro->{region_id} : undef;
}

=head1 _calc_distance

    Функция для вычисления расстояния между двумя координатами

=cut

sub _calc_distance
{
    my ($lat1, $lon1, $lat2, $lon2) = map { deg2rad($_) } @_;
    return 2*asin(sqrt(sin(($lat2-$lat1)/2)**2 + cos($lat1)*cos($lat2)*sin(($lon2-$lon1)/2)**2)) * 6_378.137;
}


1;
