=head1 NAME

MinusWords - Functions for a work with minus words

=head1 DESCRIPTION

Common functions for a work with minus words

=cut
package MinusWords;

use strict;
use warnings;
use utf8;

use base qw/Exporter/;

our @EXPORT = qw(
    check_minus_words
    polish_minus_words
    polish_minus_words_array
    key_words_with_minus_words_intersection
    explain_intersection_result
    merge_private_and_library_minus_words

    get_campaign_minus_words
    save_campaign_minus_words
    save_group_minus_words
    save_banner_minus_words
    save_mediaplan_banner_minus_words
    save_library_minus_words
);

our @EXPORT_OK = qw/
    mass_save_minus_words
/;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Trace;
use MinusWordsTools;
use Yandex::MyGoodWords;
use Yandex::ListUtils qw/xsort xuniq xdiff xminus/;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::I18n;
use Yandex::Validate;
use Direct::Validation::MinusWords;
use PrimitivesIds;
use Primitives qw/schedule_forecast/;

use List::Util qw/sum/;
use List::MoreUtils qw/any all/;

=head2 check_minus_words ($minus_words, %options)

    Проверяет входную строку/массив с минус-словами:
        1) Проверка на отсутствие спецсимволов (соответствие формату [\+\!]?[$ALLOW_MINUS_WORD_LETTERS]+)
        2) Проверка длины без учета спецсимволов
        3) Отсутствие словосочетаний (для массива)

    Входные параметры:
        minus_words  => минус-слова (строка или массив)
        type         => тип: campaign|group|phrase

    Возвращяемые значения:
        Ссылка на массив с найденными ошибками (если ничего не найдено - ссылка на пустой массив)

=cut

sub check_minus_words {
    
    my $minus_words = shift;
    my %options = @_;
    my $type = $options{type} || '';

    return [] unless defined $minus_words;

    my $validation_result;
    if ($type eq 'campaign') {
        $validation_result = validate_campaign_minus_words($minus_words, polish_minus_words_array($minus_words))
    } elsif ($type eq 'phrase') {
        $validation_result = validate_keyword_minus_words($minus_words)
    } else {
        $validation_result = validate_group_minus_words($minus_words, polish_minus_words_array($minus_words), %{hash_cut(\%options, qw/max_overall_length/)})
    }
    
    return $validation_result->get_error_descriptions;
}

=head2 polish_minus_words ($minus_words, %options)

    Преобразовывает входную строку минус-слов в новую строку минус-слов в соответствии с заданными параметрами.
    Входная строка минус-слов может содержать слова, разделеные пробелами.

    Входные параметры:
        minus_words     => строка с минус-словами
        add_minus       => в начале каждого слова добавляется символ "минус".
                           Этот параметр нужен для создания строки для вывода строки в пользовательский интерфейс.
        return_if_undef => возврящает undef если строка минус-слов не определена (пустая строка считается определеной (defined))
        sort            => пересортировать слова в алфавитном порядке
        uniq            => оставить только уникальные слова (по полной форме, с учетом окончаний, т.е. "-слово -слова" это два уникальных слова)
        uniq_norm       => оставить только уникальные слова (по нормальной форме, без учета окончаний)

        По умолчанию активны следующие параметры: sort => 1, uniq_norm => 1

    Возвращяемое значение: новая строка минус-слов

=cut

sub polish_minus_words {
    my $minus_words_str = shift;
    my %options = @_;

    $options{sort} = 1 unless exists $options{sort};
    $options{uniq_norm} = 1 if !exists $options{uniq_norm} && !exists $options{uniq};

    return undef if $options{return_if_undef} && !defined $minus_words_str;
    $minus_words_str = '' unless defined $minus_words_str;

    # $ALLOW_MINUS_WORD_LETTERS включают в себя ".", "'", но в минус словах они не нужны, кроме ситуаций число.число.число...
    $minus_words_str =~ s/[^\!\+\Q$Settings::ALLOW_MINUS_WORD_LETTERS\E]/ /g;
    $minus_words_str =~ s/\'/ /g;

    # Разделим минус-слова вида: !слово1+слово2
    $minus_words_str =~ s/([\!\+][\Q$Settings::ALLOW_MINUS_WORD_LETTERS\E]+)/ $1/g;
    $minus_words_str =~ s/[\!\+](?:[^\Q$Settings::ALLOW_MINUS_WORD_LETTERS\E]+|$)/ /g;

    # "," уже были преобразованы в пробелы, поэтому разделяем только по пробелам
    my @minus_words = split ' ', $minus_words_str;

    # Остались слова, содержащие "." внутри себя.
    # Поэтому пробегаем каждое слово, и если в нем есть буквы, то заменяем точки на пробелы и разбиваем слова.
    # Надо, чтобы оставить только ситуации число.число[.число[ ]]....
    @minus_words = map { /[^0-9\.\!\+]/ && s/\./ /g; split ' '; } @minus_words;

    # Удалим обрамляющие '.' и заменим многоточия на точку
    @minus_words = map { /\./ ? do { s/\.{2,}/./g; s/^\.?(.*?)\.?$/$1/; length $_ ? $_ : () } : $_ } @minus_words;

    # Если необходмо, оставим только уникальные минус-слова
    @minus_words = xuniq { lc } @minus_words if $options{uniq} || $options{uniq_norm};
    if ($options{uniq_norm}) {
        # По нормальной форме фильтруем слова, НЕ начинающиеся с [!+]
        my %words_norm;
        @minus_words = grep {
            my $result = 1;
            if (!/^[\!\+]/ && defined(my $n = Yandex::MyGoodWords::norm_words($_))) {
                $result = !exists $words_norm{$n};
                $words_norm{$n} = 1;
            }
            $result;
        } @minus_words;
    }

    # Если необходимо, отсортируем
    @minus_words = xsort { lc } @minus_words if $options{sort};

    # Добавляем ! перед стоп-словами
    @minus_words = map { Yandex::MyGoodWords::is_stopword($_) ? "!$_" : $_ } @minus_words;

    # Если необходимо, добавляем "минус" перед словами
    if ($options{add_minus}) {
        s/^([\!\+\Q$Settings::ALLOW_MINUS_WORD_LETTERS\E]+)$/\-$1/ for @minus_words;
    }

    return join ' ', @minus_words;
}

=head2 polish_minus_words_array ($minus_words)

    Тоже самое, что polish_minus_words, только на входе фразы с массиве.

    Чистит и причесывает входной массив минус-фраз.

    На входе:
        minus_words - массив с минус-фразами

    На выходе:
        новый массив минус-фраз.

=cut

sub polish_minus_words_array {
    my $input_minus_words = shift;

    my @minus_words = @{$input_minus_words || []};

    # $ALLOW_MINUS_WORD_LETTERS включают в себя ".", "'", но в минус словах они не нужны, кроме ситуаций число.число.число...
    foreach my $minus_word (@minus_words) {
        $minus_word =~ s/^-//g;
        $minus_word =~ s/[^\!\+\"\Q$Settings::ALLOW_MINUS_PHRASE_LETTERS\E]/ /g;
        $minus_word =~ s/\'/ /g;
        # Разделим минус-слова вида: !слово1+слово2
        $minus_word =~ s/([\!\+][\Q$Settings::ALLOW_MINUS_PHRASE_LETTERS\E]+)/$1/g;
        $minus_word =~ s/[\!\+](?:[^\Q$Settings::ALLOW_MINUS_PHRASE_LETTERS\E]+|$)/ /g;
    }
    # Остались слова, содержащие "." внутри себя.
    # Поэтому пробегаем каждое слово, и если в нем есть буквы, то заменяем точки на пробелы и разбиваем слова.
    # Надо, чтобы оставить только ситуации число.число[.число[ ]]....
    foreach my $minus_phrase (@minus_words) {
        my @minus_phrase_words = split ' ', $minus_phrase;
        @minus_phrase_words = map { /[^0-9\.\!\+]/ && s/\./ /g; $_ } @minus_phrase_words;
        $minus_phrase = join ' ', @minus_phrase_words;
    }
    @minus_words = map { s/\s+/ /g; s/(?<=(\[|\"))\s+|\s+(?=(\]|"))//g; $_ } @minus_words;

    # Удалим обрамляющие '.' и заменим многоточия на точку
    @minus_words = map { /\./ ? do { s/\.{2,}/./g; s/^\.?(.*?)\.?$/$1/; length $_ ? $_ : () } : $_ } @minus_words;

    # Убираем краевые пробелы
    @minus_words = map { s/^\s*|\s*$//g; $_ } @minus_words;

    # Отсеиваем пустые строки
    @minus_words = grep { length($_) } @minus_words;

    # Если необходмо, оставим только уникальные минус-слова
    @minus_words = xuniq { lc } @minus_words;
    # По нормальной форме фильтруем слова, НЕ начинающиеся с [!+]
    my %words_norm;

    # Добавляем ! перед стоп-словами, однако не проставляем в квадратных скобках и кавычках
    @minus_words = map { join ' ', grep { length($_) } map { $_ =~ m/[\[\]\"]/ ? ($_) : map {Yandex::MyGoodWords::is_stopword($_) ? "!$_" : $_} split /\s+/, $_}
                         ($_ =~ '\"') ? ($_) : split /(\[.+?\])/, $_
                   } @minus_words;

    @minus_words = grep {
        my $result = 1;
        if (defined(my $n = Yandex::MyGoodWords::norm_words($_))) {
            $result = !exists $words_norm{$n};
            $words_norm{$n} = 1;
        }
        $result;
    } @minus_words;

    # Если необходимо, отсортируем
    @minus_words = xsort { lc } @minus_words;

    return \@minus_words;
}

=head2 key_words_with_minus_words_intersection (%args)

    Находит пересечения ключевых слов (фраз) с минус-словами на кампанию и/или группу

    Входные параметры:
        campaign_minus_words => минус-слова на кампанию
        minus_words          => минус-слова на баннер или группу
        key_words            => ключевые слова для баннера или группы, с которыми будем искать пересечения (строка)
        is_mediaplan         => режим работы с медиапланом

        Если какого-то вышеперечисленного параметра не задано, то соответствующие параметры будут загружены
        из БД по идентификаторам, в следующем порядке:
        campaign_minus_words => cid
        minus_words          => bid | pid | mbid (is_mediaplan)
        key_words            => bid | pid | cid | mbid (is_mediaplan)

    Дополнительно для медиапланов (is_mediaplan):
        Минус-слова и ключевые слова, в случае отсутствия, будут загружены по параметру mbid.
        Единые минус-слова на кампанию не загружаются.

    Возвращяемое значение (ссылка на хеш):
        errors               => ссылка на массив с ошибками
        key_words            => ссылка на массив с фразами, пересекающимися с минус-словами на баннер или группу
        campaign_key_words   => ссылка на массив с фразами, пересекающимися с минус-словами на кампанию
        minus_words          => ссылка на массив с минус-словами на баннер или группу, пересекающимися с фразами
        campaign_minus_words => ссылка на массив с минус-словами на кампанию, пересекающимися с фразами

=cut

sub _norm_exclamation_word {
    my $norm_excl_word = shift;
    return $norm_excl_word unless $norm_excl_word =~ m/!/;
    my $word = ($norm_excl_word =~ s/!//r);
    return Yandex::MyGoodWords::norm_words($word) // '';
}

sub _normalize_minus_phrases {
    my $phrases = shift;
    my @words;
    for my $phrase (@$phrases) {
        my $has_quotes = $phrase =~ m/"/ ? 1 : 0;
        my $minus_words = _get_norm_words_array($phrase, $has_quotes);

        my $minus_words_hash = { map { $_ => undef } @$minus_words };
        my $wo_squares = { map { $_ => undef } map { split /\s+/, scalar s/\[|\]//gr } @$minus_words };
        push @words, [$phrase, $minus_words, $has_quotes, $minus_words_hash, $wo_squares];
    }
    return \@words;
}

sub key_words_with_minus_words_intersection {
    my %args = @_;

    my @errors;
    my ($campaign_minus_words, $minus_words, $key_words);
    if (@{$args{campaign_minus_words} || []}) {
        push @errors, @{check_minus_words($args{campaign_minus_words}, type => 'campaign')} unless $args{skip_errors};
        $args{campaign_minus_words} = polish_minus_words_array($args{campaign_minus_words}) unless @errors;
    }

    if (@{$args{minus_words} || []}) {
        push @errors, @{check_minus_words($args{minus_words}, type => 'group', %{hash_cut(\%args, qw/max_overall_length/)})} unless $args{skip_errors};
        $args{minus_words} = polish_minus_words_array($args{minus_words}) unless @errors;
    }
    # Если есть ошибки в минус-словах, дальше не проверяем
    return {errors => \@errors} if @errors;

    my %intersection;
    if (!$args{is_mediaplan}) {

        $campaign_minus_words = get_words(
            ($args{cid} ? (cid => $args{cid}) : ()),
            priority_words => $args{campaign_minus_words},
            type => 'minus',
        );
        $minus_words = get_words(
            (
                $args{bid} ? (bid => $args{bid}) : (
                $args{pid} ? (pid => $args{pid}) : ())
            ),
            priority_words => $args{minus_words},
            type => 'minus',
        );
        $key_words = get_words(
            (
                $args{bid} ? (bid => $args{bid}) : (
                $args{pid} ? (pid => $args{pid}) : (
                $args{cid} ? (cid => $args{cid}) : ()))
            ),
            priority_words => $args{key_words},
            type => 'key',
        );

    } else {
        $campaign_minus_words = [];
        $minus_words = get_words(
            cid => $args{cid},
            mbid => $args{mbid},
            priority_words => $args{minus_words},
            type => 'minus',
            is_mediaplan => 1,
        );
        $key_words = get_words(
            cid => $args{cid},
            mbid => $args{mbid},
            priority_words => $args{key_words},
            type => 'key',
            is_mediaplan => 1,
        );

    }

    my $norm_camp_minus_words = _normalize_minus_phrases($campaign_minus_words);
    my $norm_minus_words = _normalize_minus_phrases($minus_words);

    foreach my $plus_phrase (@$key_words) {
        my $has_quotes = $plus_phrase =~ m/"/ ? 1 : 0;
        my @norm_plus_phrase;
        for my $norm_plus (@{ _get_norm_words_array([split /\s+-/, $plus_phrase, 2]->[0], $has_quotes) }) {
            my @square_words;
            if ($norm_plus =~ m/\[/) {
                my $plus_norm_word_wo_square = ($norm_plus =~ s/\[|\]//gr);
                push @square_words, map {
                    [scalar(m/!/) ? 1 : 0, $_, _norm_exclamation_word($_)]
                } split /\s+/, $plus_norm_word_wo_square;
            }
            push @norm_plus_phrase, [scalar($norm_plus =~ m/!/) ? 1 : 0, $norm_plus, _norm_exclamation_word($norm_plus), @square_words ? \@square_words : ()];
        }

        foreach my $minus_phrase (@$norm_camp_minus_words) {
            if (_check_need_warn(\@norm_plus_phrase, $has_quotes, @{$minus_phrase}[1,2,3,4])) {
                push @{$intersection{campaign_key_words}}, $plus_phrase;
                push @{$intersection{campaign_minus_words}}, $minus_phrase->[0];
            }
        }
        foreach my $minus_phrase (@$norm_minus_words) {
            if (_check_need_warn(\@norm_plus_phrase, $has_quotes, @{$minus_phrase}[1,2,3,4])) {
                push @{$intersection{key_words}}, $plus_phrase;
                push @{$intersection{minus_words}}, $minus_phrase->[0];
            }
        }
    }
    $intersection{$_} ||= [] for qw(campaign_key_words campaign_minus_words key_words minus_words);
    @{$intersection{$_}} = xuniq { lc } @{$intersection{$_}} for keys %intersection;

    return {
        errors => [],
        %intersection,
    };
}


sub _check_need_warn {
    my ($norm_plus_phrase, $has_plus_quotes, $norm_minus_phrase, $has_minus_quotes, $mw_hash, $mw_wo_squares) = @_;

    return 0 if $has_minus_quotes && !$has_plus_quotes;
    return 0 if !@$norm_minus_phrase; # skip if collapsed minus-words

    my $reduce_expand = $has_plus_quotes && $has_minus_quotes ? 0 : 1;
    my ($plus_norm_array, $result_for_reduce) = expand_plus_minus_phrases($norm_plus_phrase, $norm_minus_phrase, $mw_hash, $mw_wo_squares, $reduce_expand);
    if (!$reduce_expand) {
        return join(' ', sort map { $$_ } @$plus_norm_array) eq join(' ', sort @$norm_minus_phrase) ? 1 : 0;
    }
    return $result_for_reduce ? 1 : 0;
}

sub _get_norm_words_array {
    my ($phrase, $has_quotes) = @_;

    my @strs = grep {length($_) > 0} map {s/^\s|\s$|\"//; $_} split /(\[.+?\])/, $phrase;
    my @words;
    foreach my $str (@strs) {
        my @str_array;
        foreach my $w (map {$_ =~ s/\[|\]//g; $_} split /\s+/, $str) {

            (my $ww = $w) =~ s/[!+]//;
            if ($w =~ m/[!+]/ && Yandex::MyGoodWords::is_stopword($ww)) {
                push @str_array, "+$ww";
            } else {
                push @str_array, $w;                
            }
        }
        if ($str =~ /\[/) {

            # В скобках стоп-слова тоже фиксируются
            @str_array = map {$_ !~ m/^\+/ && Yandex::MyGoodWords::is_stopword($_) ? "+$_" : $_} @str_array; 
            # У нормализатора есть бага: norm_words("[+из москвы +в париж]") возвращается "[из москва +в париж]"
            # то есть плюс сразу после скобки стирается, а в если находится где-то в другом месте - не стирается.
            # Поэтому нормализуем пословно в нашем порядке и потом только соединяем обратно в строку.
            @str_array = map {Yandex::MyGoodWords::norm_words($_)} @str_array;
            $str = join ' ', @str_array;
            $str = sprintf("[%s]", $str) if (scalar(@str_array) > 1);
            push @words, $str;
        } else {

            my @to_words = map {$has_quotes && $_ !~ m/^\+/ && Yandex::MyGoodWords::is_stopword($_) ? "+$_" : $_} @str_array;            

            push @words, grep { defined $_ } map {Yandex::MyGoodWords::norm_words($_)} @to_words;
        }
    }

    @words = grep { defined $_ } @words;

    return \@words;
}

=head2 expand_plus_minus_phrases (plus_phrase, minus_phrase)

    Раскладывает плюс-фразы и минус-фразы по полочкам для дальнейшего пересечения.
    Для "раскладывания по полочкам" все фразы разбираются в массив, нормализуется, 
    В случае наличия квадратных скобок в ключевых словах смотрится нет ли таких же слов в минус-словах.
    Не надо ли убрать скобки у единичного слова
    Если в ключевых словах есть воскл. знак, а в минус-словах есть такое же слово без воскл. знака, то в ключевике
    воскл знак убирается.

    Возвращает хеш с ключами:
        plus_norm_array - нормальные формы всех ключевых слов, которые лежат в массиве в правильном порядке
        minus_norm_array - нормальные формы всех минус слов, которые лежат в массиве в правильном порядке
        has_plus_quotes - флаг имеет ли ключевая фраза кавычки
        has_minus_quotes - флаг имеет ли минус фраза кавычки

=cut

sub expand_plus_minus_phrases {
    my ($plus_norm_words, $minus_norm_words, $minus_norm_words_hash, $minus_norm_words_wo_squares, $reduce_expand) = @_;

    my %mw_hash = $reduce_expand ? %$minus_norm_words_hash : ();
    my $empty_mw = 0;
    my ($idx_from, $idx_to) = (0, -1);
    my $counter = 0;
    my @result_plus_words;
    foreach my $word (@$plus_norm_words) {
        $counter++;
        if ($word->[3]) {
            my @plus_norm_word_wo_square = map {
                $_->[0] && exists $minus_norm_words_wo_squares->{$_->[2]} ? $_->[2] : $_->[1]
            } @{$word->[3]};
            my $is_sub_set = scalar @{xminus($minus_norm_words, \@plus_norm_word_wo_square)};
            if ($is_sub_set) {
                push @result_plus_words, \ sprintf("[%s]", join " ", @plus_norm_word_wo_square);
                $idx_to++;
            } else {
                push @result_plus_words, map { \$_ } @plus_norm_word_wo_square;
                $idx_to += @plus_norm_word_wo_square;
            }
        } else {
            push @result_plus_words, $word->[0] && exists $minus_norm_words_wo_squares->{$word->[2]} ? \$word->[2] : \$word->[1];
            $idx_to++;
        }

        if ($reduce_expand && $counter % 7 == 0) {
            delete @mw_hash{map { $$_ } @result_plus_words[$idx_from..$idx_to]};
            if (0 == keys %mw_hash) {
                $empty_mw = 1;
                last;
            }
            $idx_from = $idx_to + 1;
        }
    }

    if ($reduce_expand && !$empty_mw) {
        delete @mw_hash{map { $$_ } @result_plus_words[$idx_from..$idx_to]};
        if (0 == keys %mw_hash) {
            $empty_mw = 1;
        }
    }

    return (\@result_plus_words, $empty_mw);
}

=head2 explain_intersection_result($intersection, %options)

    Обрабатывает результат работы функции key_words_with_minus_words_intersection
    и выводит массив сообщений об ошибках и предупреждениях в результате возможного пересечения

    Входные параметры:
        intersection => результат работы key_words_with_minus_words_intersection
        by           => minus_words (default) | key_words
                        от чего отталкиватся при генерации предупреждений: от пересечения минус-слов с ключевыми словами или наоборот

=cut

sub explain_intersection_result {
    my $intersection = shift;
    my %options = @_;

    $options{by} ||= 'minus_words';

    my @messages;
    if (@{$intersection->{errors}}) {
        @messages = @{$intersection->{errors}};
        return \@messages;
    }

    if ($options{by} eq 'minus_words') {
        if (@{$intersection->{campaign_minus_words}} || @{$intersection->{minus_words}}) {
            push @messages, iget("Следующие единые минус-фразы пересекаются с ключевыми фразами: %s.", join(', ', @{$intersection->{campaign_minus_words}})) if @{$intersection->{campaign_minus_words}};
            push @messages, iget("Следующие минус-фразы на объявление пересекаются с ключевыми фразами: %s.", join(', ', @{$intersection->{minus_words}})) if @{$intersection->{minus_words}};
            push @messages, iget("Данные минус-фразы не будут учитываться при показах по соответствующим ключевым фразам.");
        }
    } else {
        if (@{$intersection->{campaign_key_words}} || @{$intersection->{key_words}}) {
            push @messages, iget("Ключевые фразы (%s) пересекаются с едиными минус-фразами на кампанию.", join(', ', @{$intersection->{campaign_key_words}})) if @{$intersection->{campaign_key_words}};
            push @messages, iget("Ключевые фразы (%s) пересекаются с минус-фразами на объявление.", join(', ', @{$intersection->{key_words}})) if @{$intersection->{key_words}};
            push @messages, iget("Соответствующие минус-фразы не будут учитываться при показах по перечисленным ключевым фразам.");
        }
    }

    return \@messages;
}

=head2 merge_private_and_library_minus_words ($private_words, $library_words_list)

    Объединяет приватный список минус-фраз со списком библиотечных минус-фраз.

    Входные параметры:
        private_words      => массив с минус-фразами
        library_words_list => список массивов с минус-фразами

    Возвращяемое значение: При наличии библиотечных элементов - новый список минус-фраз или undef, если список получился пустым
                           При отсутствии библиотечных элементов $private_words

=cut

sub merge_private_and_library_minus_words {
    my ($private_words, $library_words_list) = @_;

    #если нет библиотечных элементов, то ничего не делаем
    return $private_words unless defined($library_words_list) && @$library_words_list;

    my @all_minus_words = @{$private_words || []};
    for my $lib_word (@{$library_words_list}) {
        push @all_minus_words, @$lib_word;
    }
    my $result = polish_minus_words_array(\@all_minus_words);

    return scalar(@$result) == 0 ? undef : $result;
}

=head2 get_words (%options)

    Возвращает хэш, где в качестве ключей выступают слова в нормализованном виде,
    а в качестве значений - список слов в тех формах, как они используются в тексте.

    Входные параметры:
        сid            => номер кампании
        pid            => номер группы
        bid            => номер баннера
        priority_words => если они указаны, то не надо доставать слова из БД, а необходимо использовать указанные
        type           => minus | key, тип используемых слов: минус-слова или ключевые слова
        is_mediaplan   => режим работы с медиапланом

    Дополнительно для медиапланов (is_mediaplan):
        mbid           => номер баннера в медиаплане (при этом должен быть указан номер кампании)

    Возвращяемое значение:
        Ссылка на хеш

=cut

sub get_words {
    my %options = @_;
    my $profile = Yandex::Trace::new_profile('minus_words:get_words');

    # В случае минус слов - приходит массив слов, в случае в ключевиками - приходит строка.
    my $words = $options{priority_words};

    if (!defined $options{priority_words}) {
        if ($options{type} eq 'minus') {
            if (!$options{is_mediaplan}) {
                if ($options{cid}) {
                    # Берем минус-слова на кампанию
                    $words = get_campaign_minus_words($options{cid});
                } elsif ($options{pid}) {
                    # Берем минус-слова на группу
                    $words = MinusWordsTools::minus_words_str2array(get_one_field_sql(PPC(pid => $options{pid}), q{SELECT mw_text FROM minus_words JOIN phrases USING (mw_id) WHERE pid = ?}, $options{pid}));
                } elsif ($options{bid}) {
                    # Берем минус-слова на группу по баннеру
                    $words = MinusWordsTools::minus_words_str2array(get_one_field_sql(PPC(bid => $options{bid}), q{
                        SELECT mw_text
                        FROM minus_words mw
                        JOIN phrases p ON (p.mw_id = mw.mw_id)
                        JOIN banners ba ON (ba.pid = p.pid)
                        WHERE ba.bid = ?
                    }, $options{bid}));
                }
            } else {
                # Режим работы с медиапланами
                if ($options{mbid} && $options{cid}) {
                    # Берем минус-слова на баннер в медиаплане
                    $words = MinusWordsTools::minus_words_str2array(get_one_field_sql(PPC(cid => $options{cid}), q{SELECT mw_text FROM minus_words JOIN mediaplan_banners USING (mw_id) WHERE mbid = ?}, $options{mbid}));
                }
            }
        } else {
            if (!$options{is_mediaplan}) {
                if ($options{cid}) {
                    # Берем ключевые слова всех баннеров кампании.
                    $words = get_one_column_sql(PPC(cid => $options{cid}), q{
                        SELECT bi.phrase
                        FROM (
                            SELECT p.pid
                            FROM phrases p
                            JOIN banners ba ON (ba.pid = p.pid)
                            WHERE p.cid = ? AND ba.statusArch = 'No'
                            GROUP BY p.pid
                        ) t
                        JOIN bids bi ON (bi.pid = t.pid)
                    }, $options{cid}) || [];
                } elsif ($options{pid}) {
                    $words = get_one_column_sql(PPC(pid => $options{pid}), q{SELECT phrase FROM bids bi WHERE bi.pid = ?}, $options{pid}) || [];
                } elsif ($options{bid}) {
                    $words = get_one_column_sql(PPC(bid => $options{bid}), q{
                        SELECT phrase
                        FROM bids bi
                        JOIN banners ba ON (ba.pid = bi.pid)
                        WHERE ba.bid = ?
                    }, $options{bid}) || [];
                }
            } else {
                # Режим работы с медиапланами
                if ($options{mbid} && $options{cid}) {
                    $words = get_one_column_sql(PPC(cid => $options{cid}), q{SELECT phrase FROM mediaplan_bids WHERE mbid = ?}, $options{mbid}) || [];
                }
            }
        }
    }

    return $words || [];
}


=head2 get_campaign_minus_words($cid)

    Возвращяет единые минус-слова на кампанию

=cut 

sub get_campaign_minus_words {
    my $cid = shift;
    return unless $cid && is_valid_id($cid);

    return MinusWordsTools::minus_words_str2array(get_one_field_sql(PPC(cid => $cid), 'SELECT minus_words FROM camp_options WHERE cid = ?', $cid));
}

=head2 save_minus_words ($minus_words, $client_id)

    Сохраняет минус-слова в таблице minus_words

=cut

sub save_minus_words {
    my ($minus_words, $client_id) = @_;

    return undef unless defined($minus_words) && @$minus_words;

    return mass_save_minus_words([$minus_words], $client_id)->{MinusWordsTools::minus_words_utf8_hashcode($minus_words)};

}

=head2 save_group_minus_words ($pid, $minus_words)

    Сохраняет минус-слова на группу

=cut

sub save_group_minus_words {
    my ($pid, $minus_words) = @_;
    return unless $pid;
    
    my $ClientID = get_clientid(pid => $pid);
    my $mw_id = save_minus_words($minus_words, $ClientID);

    $mw_id //= 'NULL';
    return do_sql(PPC(ClientID => $ClientID), ["UPDATE phrases
                SET LastChange = IF( mw_id <=> $mw_id, LastChange, NOW() ),
                    statusBsSynced = IF( mw_id <=> $mw_id, statusBsSynced, 'No' ),
                    mw_id = $mw_id",
                where => {pid => $pid}]
    );

}


=head2 save_banner_minus_words

    Синоним save_group_minus_words.
    Оставлен для старого кода.

=cut 

sub save_banner_minus_words { save_group_minus_words(@_) }

=head2 save_mediaplan_banner_minus_words ($mbid, $minus_words, $client_id)

    Сохраняет минус-слова на баннер медиаплана
    ClientID нужен, чтобы понять в какой шард вести запись (ClientID владельца кампании с медиапланом)

=cut

sub save_mediaplan_banner_minus_words {
    my ($mbid, $minus_words, $client_id) = @_;

    return unless $mbid && $client_id;

    my $mw_id = save_minus_words($minus_words, $client_id);

    return do_update_table(PPC(ClientID => $client_id), 'mediaplan_banners', {mw_id => $mw_id}, where => {mbid => $mbid});
}

=head2 save_campaign_minus_words ($cid, $minus_words)

    Сохраняет минус-слова на кампанию

=cut

sub save_campaign_minus_words {
    my ($cid, $minus_words) = @_;

    return unless $cid;

    my $campaign = get_one_line_sql(PPC(cid => $cid), "
        SELECT c.type, co.minus_words
        FROM campaigns c JOIN camp_options co ON (co.cid = c.cid)
        WHERE c.cid = ?", $cid
    );

    if (!MinusWordsTools::are_minus_words_equal($minus_words, (MinusWordsTools::minus_words_str2array($campaign->{minus_words} // '')))) {
        my $ClientID = get_clientid(cid => $cid);
        do_in_transaction {
                my $mw_id = save_minus_words($minus_words, $ClientID);

                do_sql(PPC(cid => $cid), q{
                    UPDATE campaigns ca JOIN camp_options co USING(cid)
                    SET co.minus_words = ?, co.mw_id = ?, ca.statusBsSynced = 'No',
                    LastChange = NOW()
                    WHERE ca.cid = ?
                }, MinusWordsTools::minus_words_array2str($minus_words), $mw_id, $cid);
            };

        do_update_table(PPC(cid => $cid), 'banners', {
            LastChange__dont_quote => 'NOW()',
            # Для динамической кампании сбрасываем также statusBsSynced на всех её баннерах
            (($campaign->{type} // '') eq 'dynamic' ? (statusBsSynced => 'No') : ()),
        }, where => {cid => $cid});

        # Пересчитываем прогноз
        do_update_table(PPC(cid => $cid), 'phrases', {
            statusShowsForecast => 'New',
            LastChange__dont_quote => 'LastChange',
        }, where => {cid => $cid});
        schedule_forecast($cid);
    }

    return;
}

=head2 mass_save_minus_words($mw_texts, $client_id)

=cut

sub mass_save_minus_words {
    my ($mw_texts, $client_id) = @_;

    my %mw_hash2text = map { MinusWordsTools::minus_words_utf8_hashcode($_) => MinusWordsTools::minus_words_array2str($_)} @$mw_texts;
    # Выберем существующие минус-слова
    my $mw_hash2id = get_hash_sql(PPC(ClientID => $client_id), ["SELECT mw_hash, mw_id FROM minus_words", where => {
        ClientID => $client_id,
        mw_hash => [keys %mw_hash2text],
    }, "FOR UPDATE"]);

    # Определим новые минус-слова (кандидаты на добавление)
    my @new_mw_hashes = grep { !$mw_hash2id->{$_} } keys %mw_hash2text;
    my $new_mw_ids = get_new_id_multi('mw_id', scalar @new_mw_hashes);

    # Добавим их в базу
    do_mass_insert_sql(PPC(ClientID => $client_id),
        "INSERT IGNORE INTO minus_words (mw_id, mw_hash, mw_text, ClientID) VALUES %s",
        [map { [shift @$new_mw_ids, $_, $mw_hash2text{$_}, $client_id] } @new_mw_hashes],
    );

    # Хоть таблица minus_words сейчас не имеет уникального ключа по связке ClientID + mw_hash,
    # но как задел на будущее -- выберем вновь вставленные данные (на случай, если кто-то уже успел их вставить раньше с другим id)
    $mw_hash2id = hash_merge $mw_hash2id, get_hash_sql(PPC(ClientID => $client_id), ["SELECT mw_hash, mw_id FROM minus_words", where => {
        ClientID => $client_id,
        mw_hash => \@new_mw_hashes,
    }]);

    return $mw_hash2id;
}


=head2 save_library_minus_words ($$mw_data, $client_id)

    Сохраняет библиотечные минус-слова в таблице minus_words

=cut

sub save_library_minus_words {
    my ($minus_words_data, $client_id) = @_;

    return undef unless defined($minus_words_data);

    my $minus_words_hash =  MinusWordsTools::minus_words_utf8_hashcode($minus_words_data->{text});
    # Выберем существующие минус-слова
    my $mw_hash2id = get_hash_sql(PPC(ClientID => $client_id), ["SELECT mw_hash, mw_id FROM minus_words", where => {
        ClientID => $client_id,
            mw_hash => $minus_words_hash,
            mw_name => $minus_words_data->{name},
    }, "FOR UPDATE"]);

    # Определим новые минус-слова (кандидаты на добавление)
    if ($mw_hash2id->{$minus_words_hash}) {
        return $mw_hash2id->{$minus_words_hash};
    }

    my $new_mw_id = get_new_id('mw_id');

    # Добавим в базу
    do_insert_into_table(PPC(ClientID => $client_id), 'minus_words', {
        mw_id      => $new_mw_id,
        mw_hash    => $minus_words_hash,
        mw_text    => MinusWordsTools::minus_words_array2str($minus_words_data->{text}),
        mw_name    => $minus_words_data->{name},
        ClientID   => $client_id,
        is_library => 1
    });

    return $new_mw_id;

}

1;
