=head1 NAME

MinusWordsTools - Functions for a work with minus words without dependences

=head1 DESCRIPTION

Functions for a work with minus words

Так как вариантов видов строк сейчас множество, приведу некоторый дайджест по функциям и строкам.
1. Старый тип строки, все слова являются отдельными минус словами, разделены они пробелами:
    "цветной попугай сидит на ветке" (функция: minus_words_array2interface_show_format(\@minus_words_array))
2. Старый тип строки, все слова являются отдельными минус словами, разделены они пробелами, перед каждым стоит минус:
    "-цветной попугай -сидит -на -ветке" (функция: minus_words_array2interface_show_format(minus_words_add_minus(\@minus_words_array)))
3. Строки для транспортов (ADVQ, БК), слова разделены через пробел, минус фразы выделены в круглые скобки.
    "(цветной попугай) сидит (на ветке)" (функция: minus_words_array2str_with_brackets(\@minus_words_array))
4. Строки для транспортов (ADVQ, БК), слова разделены через пробел, минус фразы выделены в круглые скобки, перед каждой фразой стоит минус.
    "-(цветной попугай) -сидит -(на ветке)" (функция: minus_words_array2str_with_brackets_and_minus(\@minus_words_array))
5. Строки для хранения в БД, слова представляют собой массив в json формате, всегда без минусов.
    '["цветной попугай", "сидит", "на ветке"]' (функции: minus_words_array2str(\@minus_words_array))


=cut

package MinusWordsTools;

use strict;
use warnings;
use HashingTools;
use Settings;

use Yandex::ListUtils qw/xdiff/;
use List::MoreUtils qw/any/;

use JSON;


=head2 minus_words_str2array

    Переводит строку со словами, разделенными пробелом или json-строку в массив минус-слов.
    Если это json-cтрока - преобразовывает в масси, если это строка с пробелами - разбивает по пробелам.
    На входе:
        строка.
    На выходе:
        ARRAYREF минус-слов

    Примечание: 
        После выхода задачи DIRECT-55547 в данной функции надо будет остаить только from_json, остальные проверки
        будут излишними, так как к этому моменту будет предполагаться, что в БД только валидный текст лежит.
=cut

sub minus_words_str2array {
    my $minus_words_str = shift;
    return [] unless defined $minus_words_str;

    my $minus_words;
    eval {
        if (any {$minus_words_str eq $_} qw/true false/) {
            # Если пришла просто строка <true> или <false>, то она распознается как валидный json и начинаются проблемы.
            # поэтому такие строки персонально переводим.
            $minus_words = [$minus_words_str];
        } else {
            $minus_words = from_json($minus_words_str);
        }
    };
    if ($@) {
        $minus_words = [split ' ', $minus_words_str];
    }
    return $minus_words;
}

=head2 minus_words_array2str

    Переводит массив минус-слов в строку со словами, разделенными пробелом
    На входе:
        ARRAYREF минус-слов
    На выходе:
        строка.
    
=cut

sub minus_words_array2str {
    my $minus_words_array = shift;

    return undef if !defined $minus_words_array || !@$minus_words_array;

    return to_json([sort @$minus_words_array]);
}

=head2 minus_words_array2interface_show_format

    Переводит массив минус-слов в строку со словами, разделенными пробелом для показа в альтернативных интерфейсах (XLS),
    совпадает со старым форматом представления минус слов в случае минус-фраз, состоящих из одного слова.
    Используется в медиапланах, в экспорте в БК, XLS.
    
    Примеры: "цветной попугай сидит на ветке", 

    На входе:
        ARRAYREF минус-слов
    На выходе:
        строка.
    
=cut
sub minus_words_array2interface_show_format {
    my $minus_words_array = shift;

    return '' unless defined $minus_words_array;

    return join ' ', @{minus_words_add_minus($minus_words_array)};
}

=head2 minus_words_interface_show_format2array

    Иногда к нам могут приходить минус-фразы, записанные строкой, например из XLS.
    Функция переводит строку со словами, разделенными пробелом в массив минус-слов.

    На входе:
        строка.
    На выходе:
        ARRAYREF минус-слов

=cut

sub minus_words_interface_show_format2array {
    my $minus_words_str = shift;
    return [] unless defined $minus_words_str;
    # минус-фразы задаются через пробел и дефис, также допускается перечисление через запятую,
    # а также перенос строки и минусы в конце слов как разделители фраз.
    my $word_delimeter = qr/^\-|,|\-\s|\s\-/;
    my @minus_words_array = split /$word_delimeter/, $minus_words_str;
    # разделили, теперь если остались другие символы-разделители, их уберем.
    foreach (@minus_words_array) {
        $_ =~ s/([!+]+)\s+/ $1/g; # ! кий =>  !кий
        $_ =~ s/^[\s,-]+|[\s,-]+$//g; # обрезка концевых пробелов.

    }
    # Выкинем пустые слова (например между несколькими подряд разделителями которые оказались)
    @minus_words_array = grep {length($_) > 0} @minus_words_array;

    return \@minus_words_array;
}

=head2 are_minus_words_equal

    Сравнивает два набора минус-слов.
    На входе:
        два ARRAYREF минус-слов
    На выходе:
        1 - минус-слова совпадают;
        0 - минус-слова не совпадают.

=cut

sub are_minus_words_equal {
    my ($new_minus_words, $old_minus_words) = @_;

    return @{xdiff($new_minus_words, $old_minus_words)} ? 0 : 1;
}

=head2 minus_words_add_minus

    Добавляет минусы к минус-словам.
    На входе:
        ARRAYREF минус-слов
    На выходе:
        ARRAYREF минус-слов с минусами

=cut

sub minus_words_add_minus {
    my $minus_words = shift;
    return [map {"-$_"} @{$minus_words || []}];
}

=head2 minus_words_array2str_with_brackets

    Переводит массив минус-слов в строку со словами, разделенными пробелом, при этом минус фразы (где слов больше одного)
    оборачивает в круглые скобки.
    
    Пример: "(цветной попугай) сидит (на ветке)"

    На входе:
        ARRAYREF минус-слов и фраз
    На выходе:
        строка.

=cut

sub minus_words_array2str_with_brackets {
    return _array2str_with_brackets(@_);
}

=head2 minus_words_array2str_with_brackets_and_minus

    Переводит массив минус-слов в строку со словами, разделенными пробелом. Добавляет минусы впереди слова или фразы (где слов больше одного),
    при этом минус фразы оборачивает в круглые скобки. Используется для экспорта в ADVQ.

    Пример: "-(цветной попугай) -сидит -(на ветке)"

    На входе:
        ARRAYREF минус-слов и фраз
    На выходе:
        строка.

=cut

sub minus_words_array2str_with_brackets_and_minus {
    return _array2str_with_brackets(@_, add_minus => 1);
}

sub _array2str_with_brackets {
    my ($minus_words_array, %options) = @_;
    my @minus_words_array_with_brackets = map {_add_brackets_if_nessesary($_)} @$minus_words_array;
    if ($options{add_minus}) {
        @minus_words_array_with_brackets = @{minus_words_add_minus(\@minus_words_array_with_brackets)};
    }
    return join ' ', @minus_words_array_with_brackets;
}

sub _add_brackets_if_nessesary {
    my ($minus_phrase) = @_;

    if ($minus_phrase =~ m/\s+/ || $minus_phrase =~ m/^".+"$/ &&  $minus_phrase !~ m/\s+/) {
        return "($minus_phrase)";
    } else {
        return $minus_phrase;
    }
}

=head2 minus_words_utf8_hashcode

    Возвращает хеш-код для списка минус слов. Используется при работе с минус-словами на группу.
    На входе:
        ARRAYREF минус-слов и фраз
    На выходе:
        строка.

=cut
sub minus_words_utf8_hashcode {
    my $minus_words = shift;
    return url_hash_utf8(minus_words_array2str([sort @$minus_words]));
}

=head2 minus_words_length_for_limit

    Возвращает общую длину минус-слов рассчитываемую как число "значимых" символов во всех минус-словах из списка.
    На входе:
        ARRAYREF минус-слов и фраз
    На выходе:
        длина (число)

=cut

sub minus_words_length_for_limit {
    my ($minus_phrases, $polished_minus_phrases) = @_;
    return length(join('', @{$polished_minus_phrases || $minus_phrases}) =~ s/[^\Q$Settings::ALLOW_MINUS_PHRASE_LETTERS\E]|[\Q!+-\[\]\"\E]//gr); 
}

1;
