
##################################################
#
#  Direct.Yandex.ru
#
#  MirrorsTools
#  tools for mirrors
#
#  $Id$
#
##################################################

=head1 NAME

MirrorsTools

=head1 DESCRIPTION

  all about mirrors

=cut

package MirrorsTools;

use warnings;
use strict;

use Carp qw/croak/;

use Settings;
use Yandex::DBTools;
use Yandex::Shell;
use Yandex::IDN qw(is_valid_domain);
use Yandex::MirrorsTools::Hostings qw/strip_www get_hostings/;
use Yandex::Retry qw/relaxed_guard/;
use Yandex::Trace;

use utf8;

our $VERSION = '0.01';

$Settings::MIRRORS_FILE_BDB ||= '';

# Сивол, который по ascii больше чем любой символ, разрешенный в доменных именах.
our $HEAVY_LETTER = '~';

# Не вызывать препроцессор (только для дебага и тестов)
our $SKIP_PREPROCESSOR_RUN //= 0;
our $MIRRORS_PREPROCESSOR //= '/usr/local/bin/direct-mirrors-preprocessor';

=head1 FUNCTIONS

=cut

#======================================================================

=head2 new

 my $mirror = MirrorsTools->new(option => 'value', ....);
 options:
   mirror_file    => 'mirror_file_name.res' - название файла, откуда брать информацию о зеркалах от поиска.
   dont_load_file => 1 or 0 - не загружать файл для выборки зеркал
   use_db         => 1 or 0 - для поиска зеркал использовать БД (таблица PPCDICT::mirrors)
   update_db      => 1 or 0 - записывать данные в БД, используется для обновления базы зеркал
   log            => $yandex_log - объект для логирования процесса reinitialize

=cut

sub new
{
    my $this = shift;
    my $class = ref($this) || $this;

    my $self = {@_};

    bless $self, $class;

    if (!$self->{dont_load_file} && !defined $self->{mirror_file}) {
        $self->{mirror_file} = $Settings::MIRRORS_FILE;
    }

    if (!$self->{use_db} && !$self->{dont_load_file} && (! defined $self->{mirror_file} || ! -r $self->{mirror_file})) {
        die "mirror file $self->{mirror_file} not found\n";
    }

    $self->{mirror_file_mtime} = 0;
    $self->{mirrors} = {};

    unless ($self->{log} && ref $self->{log} && UNIVERSAL::can($self->{log}, 'out')) {
        delete $self->{log};
    }

    $self->reinitialize();

    return $self;
}

#======================================================================

=head2 domain_filter

 get domain filter
 $domain = $mirror->domain_filter($domain);

=cut

sub domain_filter
{
    my $self = shift;
    my $orig_domain = shift;

    my $domain = $self->url2host($orig_domain);
    return undef if !is_valid_domain($domain);
    my $sdomain = $self->strip_domain($domain);

    while($domain ne $sdomain && $domain =~ /\./) {
        my $main_mirror = $self->get_main_mirror($domain);
        if ($main_mirror) {
            return $self->strip_domain($main_mirror);
        }
        $domain =~ s/^[^\.]+\.//;
    }

    return $self->strip_domain($self->get_main_mirror($domain) || $domain);
}

#======================================================================

=head2 url2host

 url2host
 $domain = $mirror->url2host($url);

=cut

sub url2host
{
    my $self = shift;
    my $url = shift;

    return undef unless defined $url;

    for ($url) {
        s[^[^/:]+://][]; # strip "http://"
        s[[/?#].*$][]; # strip path
        s[:.*$][]; # strip port
        $_ = lc($_);
    }

    return $url;
}

#======================================================================

=head2 get_main_mirror

 get main mirror
 $domain = $mirror->get_main_mirror($domain);

=cut

sub get_main_mirror
{
    my $self = shift;
    my $domain = shift;

    if ($self->{use_db}) {
        my $domain_lc = lc($domain);
        return get_one_field_sql(PPCDICT, ['SELECT redirect_domain FROM mirrors_correction', WHERE => {domain => $domain_lc}])
            || get_one_field_sql(PPCDICT, ['SELECT mirror FROM mirrors', WHERE => {domain => $domain_lc}]);
    } else {
        return $self->{mirrors}{lc($domain)};
    }
}

#======================================================================

=head2 strip_domain

 domain filter on domain

=cut

sub strip_domain
{
    my $self = shift;
    my $href = shift;

    return Yandex::MirrorsTools::Hostings::strip_domain($href);
}

#======================================================================

=head2 reinitialize

 reinitialize mirror data
 $mirror->reinitialize();

=cut

sub reinitialize
{
    my $self = shift;

    return if $self->{dont_load_file};

    my $mirror_file = $self->{mirror_file};

    die "$mirror_file not found\n" unless -r $mirror_file;
    return if $self->{mirrors} && $self->{mirror_file_mtime} == (stat($mirror_file))[9];

    my %MIRRORS;
    my $db_filename = $self->{db_filename} = $Settings::MIRRORS_FILE_BDB || "$mirror_file.btree";
    if (-f $db_filename) {
        unlink($db_filename) || die "Can't unlink $db_filename: $!";
    }
    # грузим BerkeleyDB только по требованию
    require BerkeleyDB;
    tie(%MIRRORS, "BerkeleyDB::Btree", -Filename => $db_filename, -Flags => BerkeleyDB::DB_CREATE()) or die "Can't open $db_filename: $!";

    my $mirrors = $self->{mirrors} = \%MIRRORS;

    my $preprocessed_file = "$mirror_file.preprocessed.gz";
    if (!$SKIP_PREPROCESSOR_RUN) {
        # вызываем java-препроцессор
        $self->log("run java-preprocessor");
        if (-f $preprocessed_file) {
            unlink($preprocessed_file) || croak "Can't unlink $preprocessed_file: $!";
        }

        my $java_profile = Yandex::Trace::new_profile('MirrorsTools:java_preprocess');
        yash_system($MIRRORS_PREPROCESSOR,
                $mirror_file,
                $preprocessed_file,
        );
        undef $java_profile;
    }
    if (-s $preprocessed_file < (-s $mirror_file) / 4) {
        croak "Size of preprocessed file is too small: ".(-s $preprocessed_file).", original file size: ".(-s $mirror_file);
    }

    $self->log("load preprocessed file");
    my $iter_profile = Yandex::Trace::new_profile('MirrorsTools:iter_preprocessed_file');
    open(my $fd, '-|', "zcat $preprocessed_file") || croak "Can't open file $preprocessed_file: $!";
    while(my $line = <$fd>) {
        chomp $line;
        my @hosts = split / /, $line;
        my $main_host = shift @hosts;
        for my $host (@hosts) {
            $mirrors->{$host} = $main_host if $host ne $main_host;
            # если есть главный домен для www.xxx -> прописываем его и к xxx,
            # но с меньшим приоритетом (если для xxx указан домен - он и остаётся)
            my $host_wo_www = strip_www($host);
            if ($host_wo_www ne $host && $host_wo_www ne $main_host) {
                $mirrors->{$host_wo_www} ||= $main_host;
            }
        }
    }
    close($fd) || die "Can't close zcat: $!";
    undef $iter_profile;

    if ($self->{update_db}) {
        $self->log("load mirrors correction from db");
        # достаем из БД наши правки по зеркалам
        my $mirrors_correction = get_all_sql(PPCDICT, "SELECT LOWER(domain) AS domain, LOWER(redirect_domain) AS redirect_domain, correction_type FROM mirrors_correction ORDER BY domain");

        for my $d (@{$mirrors_correction}) {
            $mirrors->{$d->{domain}} = $d->{redirect_domain};
        }

        # Ручные привязки прогоняем через BerkeleyDB::Btree, чтобы можно было воспользоваться функцией remove_cycles,
        # которая расчитывает на то, что домены будут отсортированы.
        my %manual_mirrors;
        tie (%manual_mirrors, "BerkeleyDB::Btree", -Flags => BerkeleyDB::DB_CREATE()) or die "Cannot create database: $!\n" ;

        $self->log("fill manual mirrors hash");
        %manual_mirrors = map {$_->{domain} => $_->{redirect_domain}} grep {$_->{correction_type} eq 'Manual'} @{$mirrors_correction};
        # Удаляем циклы из наших привязок.
        $self->log("remove cycles from manual mirrors corrections");
        my $rc_profile = Yandex::Trace::new_profile('MirrorsTools:remove_cycles', tags => "manual_mirrors");
        remove_cycles(\%manual_mirrors);
        undef $rc_profile;

        # для работы с хешами, которые привязаны через tie, рекомендуется итеррировать при помощи each
        # домены из ручных привязок записываем в общий список доменов. Чтобы они были приоритетными,
        # добавляем в начале домена строку, которая заведомо поставит домен в конец списка.
        $self->log("process manual mirrors hash");
        while(my ($d, $m) = each %manual_mirrors) {
            $mirrors->{$HEAVY_LETTER.$d} = $m;
            delete $mirrors->{$d} if defined ($mirrors->{$d});
        }

        # вновь убираем циклы из общего списка с учетом ручных привязок
        $self->log("remove cycles from mirrors");
        $rc_profile = Yandex::Trace::new_profile('MirrorsTools:remove_cycles', tags => "mirrors");
        remove_cycles($mirrors);
        undef $rc_profile;

        # возвращаем обратно домены, отнимая дописанный префикс (который переносил их в конец списка)
        $self->log("process manual mirrors hash - restore domains");
        while(my ($d) = each %manual_mirrors) {
            # Deleting from a tied hash or array may not necessarily return anything; it depends on the implementation of the tied package's
            # DELETE method, which may do whatever it pleases.
            # Разделено на две операции, так как у нас tied хеш.
            if ($mirrors->{$HEAVY_LETTER.$d}) {
                $mirrors->{$d} = $mirrors->{$HEAVY_LETTER.$d} ;
                delete $mirrors->{$HEAVY_LETTER.$d};
            }
        }

        $self->{mirror_file_mtime} = (stat($mirror_file))[9];

        $self->log("truncate mirrors_new table");
        do_sql(PPCDICT, "truncate table mirrors_new");

        # Сохраняем изменения.
        my @data_for_update_db;
        my $sleep_profile;
        $self->log("fill mirrors_new table");
        while(my ($d, $m) = each %$mirrors) {
            push @data_for_update_db, [ $d, $m ];
            if (@data_for_update_db >= 1_000) {
                my $guard = relaxed_guard times => 1;
                do_mass_insert_sql(PPCDICT, "insert into mirrors_new (domain, mirror)
                                              values %s
                                              on duplicate key update
                                              mirror = values(mirror)", \@data_for_update_db);
                @data_for_update_db = ();
                $sleep_profile = Yandex::Trace::new_profile('MirrorsTools:sleep');
            }
            undef $sleep_profile;
        }
        do_mass_insert_sql(PPCDICT, "insert into mirrors_new (domain, mirror)
                                              values %s
                                              on duplicate key update
                                              mirror = values(mirror)", \@data_for_update_db);

        $self->log("swap mirrors_new and mirrors tables");
        do_sql(PPCDICT, "rename table
                                    mirrors_new to mirrors_tmp,
                                    mirrors     to mirrors_new,
                                    mirrors_tmp to mirrors");

        # не очищаем таблицу, чтобы не нагружать БД лишний раз (ценой свободного места). очистим перед следующим обновлением
        # do_sql(PPCDICT, "truncate table mirrors_new");
        $self->log("done");
    } else {
        my $rc_profile = Yandex::Trace::new_profile('MirrorsTools:remove_cycles', tags => "without_db_update");
        remove_cycles($mirrors);
        undef $rc_profile;
        $self->{mirror_file_mtime} = (stat($mirror_file))[9];
    }

    $self->{db_filesize} = (stat($db_filename))[7];
    undef $mirrors;
    unlink($db_filename) || die "Can't unlink $db_filename: $!";
    # но $self->{mirrors} остаётся

    return;
}

=head2 remove_cycles (mirrors)

    Удаляет циклы и транзитивные замыкания в зеркалах доменов.
    Параметр:
        mirrors - хеш вида <домен> => <его главное зеркало>
                  Чтобы работало корректно (и стабильно!), важно, чтобы ключи были отсортированы в алфавитном порядке.
                  Это можно добиться прогоном хеша через BerkeleyDB::Btree

=cut
sub remove_cycles {
    my ($mirrors) = shift;

    while(my ($d, $m) = each %$mirrors) {
        my %seen;
        my $dt = $d;
        while((exists $mirrors->{$m} || exists $mirrors->{$HEAVY_LETTER.$m}) && !$seen{$dt}++) {
            $dt = $m;
            my $value = $mirrors->{$HEAVY_LETTER.$m} || $mirrors->{$m};
            $m = $mirrors->{$d} = $value;
            delete $mirrors->{$d} if $d eq $m;
        }
    }
}


=head2 log

    Залогировать сообщения в $self->{log} через метод out, если объект был сконструирован с переданным $log - объектом

=cut
sub log {
    my ($self) = shift;
    if ($self->{log}) {
        $self->{log}->out(@_);
    }
}

1;

