package MobileContent;

=pod
=head1 NAME

    MobileContent - модуль для получения данных о мобильном контенте из сторов

=head1 DESCRIPTION

    Модуль для получения данных о мобильном контенте из сторов.

=head1 METHODS

=cut

use Direct::Modern;

use JSON;
use List::MoreUtils qw/any/;
use Readonly;
use Scalar::Util qw/looks_like_number/;

use Settings;

use Yandex::CheckMobileRedirect qw/parse_store_url/;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::I18n;
use JavaIntapi::GetAppMobileContent;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    %OS_VERSIONS
    ajax_mobile_content_info
    icon_hash2url
/;

=head2 $USED_VERSION_PARTS

    Количество используемыех чисел в строках с версиями.
    Например, при $USED_VERSION_PARTS = 2 для версии "123.456.678" будет использовано 123.456

    Ниоткуда изменяться не должно, используется в юнит-тесте.

=cut
our $USED_VERSION_PARTS = 2;

=head2 %OS_VERSIONS

    Хеш мажорных версий поддерживаемых ОС для мобильного контента
    Ключи -- тип ОС как в mobile_content.os_type
    Значения -- ссылка на массив с версиями, отсортированными по возрастанию

    NB! 1) на то, что массивы отсортированы - полагается код экспорта в БК.
        2) Аналогичные хэши есть в Java
           (ru.yandex.direct.core.entity.adgroup.service.validation.types.MobileContentAdGroupValidation). 
           Поправил в Perl - поправь и в Java!
        3) После изменения версий следует перегенерить константы в js (i-const.js.ru)

=cut
Readonly::Hash our %OS_VERSIONS => (
    # https://en.wikipedia.org/wiki/Android_version_history
    'Android' => [ sort { _cmp_os_versions($a, $b) } qw(
        1.0
        1.1
        1.5
        1.6
        2.0
        2.1
        2.2
        2.3
        3.0
        3.1
        3.2
        4.0
        4.1
        4.2
        4.3
        4.4
        5.0
        5.1
        6.0
        7.0
        7.1
        8.0
        8.1
        9.0
        10.0
        11.0
    )],
    # https://en.wikipedia.org/wiki/History_of_iOS
    'iOS' => [ sort { _cmp_os_versions($a, $b) } qw(
        1.0
        2.0
        3.0
        3.1
        4.0
        4.1
        4.2
        5.0
        5.1
        6.0
        6.1
        7.0
        7.1
        8.0
        8.1
        8.2
        8.3
        8.4
        9.0
        9.1
        9.2
        9.3
        10.0
        10.1
        10.2
        10.3
        11.0
        11.1
        11.2
        11.3
        11.4
        12.0
        12.1
        12.2
        12.3
        12.4
        13.0
        13.1
        13.2
        13.3
        13.4
        13.5
        13.6
        13.7
        14.0
        14.1
    )],
);

# пока не поддерживаем настоящие размеры картинок, используем справочные
Readonly::Hash our %AVATARS_SIZES => (
    # alias => { Height => ..., Width => ... }
    'icon-s'            => { Height =>  16, Width =>  16 },
    'icon-s-retina'     => { Height =>  32, Width =>  32 },
    'icon'              => { Height =>  65, Width =>  65 },
    'icon-l'            => { Height =>  72, Width =>  72 },
    'icon-ld'           => { Height =>  80, Width =>  80 },
    'icon-ldd'          => { Height =>  90, Width =>  90 },
    'icon-xl'           => { Height => 144, Width => 144 },
    'icon-xld'          => { Height => 150, Width => 150 },
    'icon-ld-retina'    => { Height => 160, Width => 160 },
    'icon-ldd-retina'   => { Height => 180, Width => 180 },
    'icon-xld-retina'   => { Height => 300, Width => 300 },
    'orig'              => undef,
);

# ВАЖНО: хеш продублирован в Java-директе в ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService
# при внесении изменения здесь, нужно поправить значение и там
my $os_type2avatars_instance = {
    'iOS' => 'itunes-icon',
    'Android' => 'google-play-app-icon',
};

# ВАЖНО: хеш продублирован в Java-директе в ru.yandex.direct.core.entity.mobilecontent.service.MobileContentService
# при внесении изменения здесь, нужно поправить значение и там
my %os_type_2_store_name = (
    'Android' => 'Google Play',
    'iOS' => 'App Store',
);


=head2 ajax_mobile_content_info

    Принимает на вход ссылку на стор, возвращает хеш с даннымии
    про мобильный контент или ошибку, если не смогли понять ссылку
    или контент недоступен (mobile_content.is_available = 0)
    или нет данных про этот контент в SaaS-ручке

=cut

sub ajax_mobile_content_info
{
    my ($url, $client_id, %params) = @_;

    my $store_data = parse_store_url($url);

    my $result;
    my $error_msg = '';

    unless (keys %$store_data) {
        $error_msg = iget('Неправильный формат ссылки');
    } else {
        $store_data->{ClientID} = $client_id;
        $result = get_one_line_sql(PPC(ClientID => $store_data->{ClientID}),
                                    ["SELECT
                                        mobile_content_id,
                                        ClientID,
                                        name,
                                        store_country,
                                        prices_json,
                                        rating,
                                        rating_votes,
                                        icon_hash,
                                        os_type,
                                        min_os_version,
                                        available_actions,
                                        is_available,
                                        content_type,
                                        age_label
                                    FROM mobile_content",
                                    where => {
                                        ClientID         => $store_data->{ClientID},
                                        content_type     => $store_data->{content_type},
                                        os_type          => $store_data->{os_type},
                                        store_content_id => $store_data->{store_content_id},
                                        store_country    => $store_data->{store_country},
                                    }]);

        if (keys %$result && $result->{is_available}) {
            # если нашли контент в базе и он доступен
            my $icon_url = icon_hash2url($result->{os_type}, $result->{icon_hash}, 'icon');
            $result->{icon_url} = $icon_url;
            $result->{is_show_icon} = ( $result->{icon_url}? 1 : 0 );
            $result->{is_default_country} = $store_data->{is_default_country};
            my $prices;
            if ($result->{prices_json}) {
                my $all_prices = from_json($result->{prices_json});
                $prices = $all_prices->{$result->{store_country}};
            }
            $result->{prices} = $prices || {};
            $result->{available_actions} = [ $result->{available_actions} ];
        } else {
            # если приложение недоступно, то идем в SaaS ручку за данным
                # сохраняем контент в базу
                my $mobile_content = JavaIntapi::GetAppMobileContent->new( client_id => $client_id, url => $url )->call;

                $result = hash_merge($store_data, $mobile_content);
                $result->{is_available} = $mobile_content->{is_available} ? 1 : 0;

            if (keys %$result && $result->{is_available}) {
                # и контент доступен
                my $icon_url = icon_hash2url($result->{os_type}, $result->{icon_hash}, 'icon');
                $result->{icon_url} = $icon_url;
                $result->{is_show_icon} = ( $result->{icon_url}? 1 : 0 );
                my $prices;
                if ($result->{prices_json}) {
                    my $all_prices = from_json($result->{prices_json});
                    $prices = $all_prices->{$result->{store_country}};
                }
                $result->{prices} = $prices || {};
            } else {
                $error_msg = iget('Не удалось получить данные из магазина приложений');
            }
        }

        # оставляем в ответе только указанные поля
        my @filter_keys = qw/
            mobile_content_id
            name
            store_country
            prices
            rating
            rating_votes
            icon_url
            os_type
            min_os_version
            available_actions
            is_available
            content_type
            age_label
            is_show_icon
            is_default_country
        /;
        $result = hash_cut($result, @filter_keys);
    }

    my $resp = {};
    if ($error_msg) {
        $resp = { error => { message => $error_msg }, response => undef };
    } else {
        $resp = { error => undef, response => $result };
    }

    return $resp;
}

=head2 icon_hash2url

    Принимает на вход тип операционной системы:
        iOS,
        Android,
    хеш иконки в аватарнице вида:
    '21534/com.starshipstudio.thesurvivor__83c6c809263bf4b5c381b9a1e078e89e'
    и (опционально) алиас, определяющий размер. список доступных алиасов в %AVATARS_SIZES

    Возвращает ссылку на иконку (по умолчанию 65x65):
    '//avatars.mds.yandex.net/get-google-play-app-icon/21534/com.starshipstudio.thesurvivor__83c6c809263bf4b5c381b9a1e078e89e/icon'

=cut

sub icon_hash2url($$;$)
{
    my ($os_type, $icon_hash, $alias) = @_;

    return undef unless ($icon_hash && $os_type && exists $os_type2avatars_instance->{$os_type});

    if (!$alias || !exists $AVATARS_SIZES{$alias}) {
        # по умолчанию возвращаем ссылку на иконку 65x65
        $alias = 'icon';
    }

    my $icon_url = "//$Settings::AVATARS_MDS_HOST/get-$os_type2avatars_instance->{$os_type}/$icon_hash/$alias";
    return $icon_url;
}

=head2 get_version_parts($version)

    Получает из произвольной строки, похожей на версию - массив значений (мажорная, минорная части версии).
    Отрицательные или значения, не являющиеся числами, заменяются при этом на нули.
    Количество значений ограничено $USED_VERSION_PARTS.

    Параметры:
        $version - строка с версией (например "1..2.4")
    Результат:
        $parts  - ссылка на массив значений (для указанного примера - [1, 0])

=cut
sub get_version_parts {
    my @parts = split(qr/(?:\.\s*|\s+)/, (shift // ''), $USED_VERSION_PARTS + 1);
    return [ map { looks_like_number($_) && $_ > 0 ? $_ : 0 } @parts[0 .. $USED_VERSION_PARTS - 1] ];
}

=head2 is_version_non_zero_by_parts($version_parts)

    Проверить версию (из частей $version_parts) на то, что она не состоит из одних нулей

    Параметры:
        $version_parts  - ссылка на массив с частями версии (как результат get_version_parts)
    Результат:
        $is_non_zero    - булевый результат проверки

=cut
sub is_version_non_zero_by_parts {
    return any { $_ } @{shift()};
}

=head2 get_major_os_version($value)

    Получить мажорную версию (из первых $USED_VERSION_PARTS чисел) из строки.
    Правила преобразования описаны в get_version_parts.
    Используется только в тесте на словарь %OS_VERSIONS
    
    Параметры:
        $value          - строка с полной версией
    Результат:
        $major_version  - строка с "мажорной" версией или undef (для нулевой версии)

=cut
sub get_major_os_version {
    my $value = shift;
    my $version_parts = get_version_parts($value);
    my $os_version;

    if (is_version_non_zero_by_parts($version_parts)) {
        $os_version = join('.', @$version_parts);
    }

    return $os_version;
}

=head2 cmp_os_versions_by_parts($version1_parts, $version2_parts)

    Числовое сравнение версий по частям, разделенным точками.

    Параметры:
        $version1_parts - ссылка на массив с частями версии 1 (как результат get_version_parts)
        $version2_parts - ссылка на массив с частями версии 2 (аналогично)
    Результат:
        $cmp_res    - результат числового сравнения по всем частям версий (-1/0/1)

=cut
sub cmp_os_versions_by_parts {
    my ($version1_parts, $version2_parts) = @_;

    for (my $i = 0; $i < $USED_VERSION_PARTS; $i++) {
        if (my $cmp_part = ($version1_parts->[$i] <=> $version2_parts->[$i])) {
            return $cmp_part;
        }
    }
    return 0;
}

=head3 _cmp_os_versions($version1, $version2)

    Числовое сравнение версий по частям, разделенным точками.

    Параметры:
        $version1   - строка с версией 1
        $version2   - строка с версией 2
    Результат:
        $cmp_res    - результат числового сравнений по всем частям версии (-1/0/1)

=cut
sub _cmp_os_versions {
    return cmp_os_versions_by_parts(get_version_parts(shift), get_version_parts(shift));
}

=head2 get_store_name_from_mobile_content($mobile_content)

    my $store_name = get_store_name_from_mobile_content($mobile_content);

    По данным о мобильном контенте получить название "магазина" (store_name).
    При невозможности определить название - умирает.

    Параметры:
        $mobile_content - hashref, данные о мобильном контенте
    Результат:
        $store_name     - строка с название магазина

=cut
sub get_store_name_from_mobile_content {
    my $mobile_content = shift;
    my $store_name;

    unless ($mobile_content && ref $mobile_content eq 'HASH') {
        die '$mobile_content shoud be a hashref';
    }

    if (defined $mobile_content->{os_type}
        && exists $os_type_2_store_name{ $mobile_content->{os_type} }
    ) {
        $store_name = $os_type_2_store_name{ $mobile_content->{os_type} };
    } else {
        die q!Can't get store name from $mobile_content data!;
    }

    return $store_name;
}

# --------------------------------------------------------------------

=head2 get_apps_count_by_cid

Получаем количество различных приложений на кампанию, нужно для предупреждений по CPI-стратегии
my $count = get_apps_count_by_cid(123);

=cut

sub get_apps_count_by_cid($) {
    my $cid = shift;

    my $apps_count = get_one_field_sql(PPC(cid => $cid), [
        "select count(distinct amc.mobile_content_id) fr
         from phrases p
           join adgroups_mobile_content amc on amc.pid = p.pid
        ", where => {
            'p.cid' => SHARD_IDS
        }
    ]);

    return $apps_count;
}

1;
