package Models::AdGroup;

use Direct::Modern;

use Settings;

use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Validate qw/is_valid_int/;
use Yandex::IDN qw//;

use ADVQ6;
use GeoTools;
use TextTools;
use ShardingTools;
use MailNotification;
use Tag;
use Primitives;
use PrimitivesIds;
use BS::TrafaretAuction;
use BSAuction;
use ModerateChecks;
use MinusWords;
use MinusWordsTools;
use BannerFlags;
use Retargeting;
use URLDomain;
use Pokazometer qw/safe_pokazometer/;
use CommonMaps qw//;
use TimeTarget;
use Client;
use User qw/get_user_data/;
use PhrasePrice qw/phrase_price_context validate_autobudget_priority validate_phrase_price/;
use Tools;
use HierarchicalMultipliers qw/mass_get_hierarchical_multipliers save_hierarchical_multipliers delete_camp_group_hierarchical_multipliers/;
use Direct::Validation::HierarchicalMultipliers qw/validate_hierarchical_multipliers/;
use Campaign::Types qw/get_camp_type/;
use BannerTemplates qw/is_template_banner/;

use Models::Banner;
use Models::DesktopBanner qw/create_desktop_banners update_desktop_banners/;
use Models::MobileBanner qw/create_mobile_banners update_mobile_banners/;
use Models::Phrase;
use Models::AdGroupFilters;
use Models::CampaignOperations;

use Direct::DynamicConditions;
use Direct::AdGroups2 ();
use Direct::AdGroups2::MobileContent qw//;
use Direct::Validation::AdGroupsMobileContent qw//;
use Direct::Model::AdGroupMobileContent;
use Direct::PerformanceFilters;
use Direct::RetargetingConditions;
use Direct::Market;
use Direct::ResponseHelper qw(error);
use Direct::Banners;
use Direct::Creatives::Tools;
use Direct::Model::Creative::Constants;
use Direct::Bids::BidRelevanceMatch qw//;
use Direct::BannersPermalinks;

use List::Util qw/first/;
use Yandex::Clone qw/yclone/;
use List::MoreUtils qw/uniq any firstidx each_array none part all/;
use Yandex::ListUtils qw/xflatten xuniq xisect chunks xminus/;
use Carp qw/confess/;
use List::UtilsBy qw/partition_by/;
use HashingTools qw/md5_hex_utf8/;
use JSON qw/from_json/;

use base qw(Exporter);
our @EXPORT = qw/
    get_groups
    stop_groups
    resume_groups
    filter_arch_groups
    validate_group
    get_auto_adgroup_name
    has_camp_groups_with_disabled_geo
    get_first_excess_phrase_idx
/;

=head2 stop_groups($pids, $options)

    Останавливает показы групп
    
    $pids - список phrases.pid
    $options
         uid - пользователь выполнивший операцию(будет отправлено уведомление)
         bid - [bid, bid, bid ...] - остановить только указанные баннеры
    Возвращает информацию по остановленным и баннерам:
    {
        pid => [ bid1, ..., bidN ] - хеш, где ключ - идентификатор группы(pid), в которой остановлен хотя бы один баннер
                                            и значение - список остановленных баннеров(bid) в этой группе
    }

=cut

sub stop_groups {
    
    my ($ids, $options) = @_;
    my $pids = ref $ids eq 'ARRAY' ? $ids : [$ids];

    my $update_before = Models::Banner::get_update_before();

    my $filters = hash_cut $options, qw/bid/;
    #для dna дополнительно отфильтровываем баннеры-черновики и архивные кампании DIRECT-82673, DIRECT-84518
    $filters->{_TEXT} = "b.statusShow = 'Yes' and b.statusModerate <> 'New' and c.archived = 'No' and (ap.priority is null or ap.priority != $Settings::DEFAULT_CPM_PRICE_ADGROUP_PRIORITY)";

    my $result = {};
    my $banners = Models::Banner::get_pure_creatives([map {{pid => $_} } @$pids], $filters, {only_creatives => 1});

    if (@$banners) {
        my $banner_ids = [map { $_->{bid} } @$banners];
        my %banner_ids_with_disabled_geo = map { $_->{bid} => 1 } @{get_banners_with_disabled_geo(bid => $banner_ids)};
        my %groups_which_reset_statusBsSynced = map { $_->{pid} => 'No' } grep { $banner_ids_with_disabled_geo{$_->{bid}} } @$banners;

        push @{$result->{$_->{pid}}}, $_->{bid} for @$banners;
        my $pids_to_stop = [ keys $result ];

        Models::Banner::stop_banners($banners, hash_cut $options, qw/uid/);
        do_update_table(PPC(pid => $pids_to_stop), 'phrases',
            {statusAutobudgetShow => 'Yes', LastChange__dont_quote => 'LastChange',
                statusBsSynced__dont_quote => sql_case('pid', \%groups_which_reset_statusBsSynced, default__dont_quote => 'statusBsSynced')
            },
            where => {pid => SHARD_IDS}
        );

        # DIRECT-78307: В вызовах из нового интерфейса могут попадаться группы в которых ничего делать не надо
        # выбираем кампании только из тех групп которые действительно обновились
        my $cids = get_cids(pid => $pids_to_stop);
        schedule_forecast_multi($cids);
        
        Models::Banner::update_banner_statuses_is_obsolete($banner_ids, $update_before);
        Models::Banner::update_adgroup_statuses_is_obsolete($pids_to_stop, $update_before);
    }
    return $result;
}

=head2 resume_groups($pids, $options)

    Включает остановленные группы.
    
    $pids - список phrases.pid
    $options
         uid - пользователь выполнивший операцию(будет отправлено уведомление)
         bid - [bid, bid, bid ...] - остановить только указанные баннеры

    Возвращает информацию по включенным группа и баннерам:
    {
        pid => [ bid1, ..., bidN ] -  хеш, где ключ - идентификатор группы(pid), в которой запущен хотя бы один баннер
                                            и значение - список запущенных баннеров(bid) в этой группе
    }

=cut

sub resume_groups {
    
    my ($ids, $options) = @_;
    my $pids = ref $ids eq 'ARRAY' ? $ids : [$ids];

    my $update_before = Models::Banner::get_update_before();

    my $filters = hash_cut $options, qw/bid/;
    $filters->{_TEXT} = "b.statusShow = 'No' and (ap.priority is null or ap.priority != $Settings::DEFAULT_CPM_PRICE_ADGROUP_PRIORITY)";
    $filters->{arch_banners} = 0;

    my $result = {};
    my $banners = Models::Banner::get_pure_creatives([map {{pid => $_} } @$pids], $filters, {only_creatives => 1});

    # Временный костыль для миграции на новую схему смарт-баннеров.
    # В смарт-группы будут добавляться баннеры нового типа, performance_main.
    # Старые баннеры при этом скрываются и останавливаются.
    # Здесь мы предотвращаем включение старых баннеров после остановки/запуска группы для мигрированных групп.
    # Также предотвращаем включение медийных баннеров для клиентов с включенной фичей is_cpm_banner_campaign_disabled
    my %pids_with_performance_main_banners;
    my %performance_banners_by_client_id;
    my %cpm_banners_by_client_id;
    foreach my $banner (@$banners) {
        if ($banner->{real_banner_type} eq 'performance_main') {
            $pids_with_performance_main_banners{$banner->{pid}} = ();
        } elsif ($banner->{real_banner_type} eq 'performance') {
            $performance_banners_by_client_id{$banner->{ClientID}} //= [];
            push @{ $performance_banners_by_client_id{$banner->{ClientID}} }, $banner;
        } elsif (Models::Banner::is_cpm_banner($banner->{real_banner_type})) {
            $cpm_banners_by_client_id{$banner->{ClientID}} //= [];
            push @{ $cpm_banners_by_client_id{$banner->{ClientID}} }, $banner;
        }
    }
    my @banners_to_ignore;
    foreach my $client_id (keys %performance_banners_by_client_id) {
        if (Client::ClientFeatures::creative_free_interface_enabled($client_id)) {
            foreach my $banner (@{ $performance_banners_by_client_id{$client_id} }) {
                push @banners_to_ignore, $banner if exists $pids_with_performance_main_banners{$banner->{pid}};
            }
        }
    }
    foreach my $client_id (keys %cpm_banners_by_client_id) {
        if (Client::ClientFeatures::is_feature_cpm_banner_campaign_disabled_enabled($client_id)) {
            foreach my $banner (@{ $cpm_banners_by_client_id{$client_id} }) {
                push @banners_to_ignore, $banner
            }
        }
    }
    $banners = xminus($banners, \@banners_to_ignore);

    if (@$banners) {
        my $banner_ids = [map { $_->{bid} } @$banners];
        my %banner_ids_with_disabled_geo = map { $_->{bid} => 1 } @{get_banners_with_disabled_geo(bid => $banner_ids)};
        my %groups_which_reset_statusBsSynced = map { $_->{pid} => 'No' } grep { $banner_ids_with_disabled_geo{$_->{bid}} } @$banners;

        push @{$result->{$_->{pid}}}, $_->{bid} for @$banners;
        my $pids_to_resume = [ keys $result ];

        Models::Banner::resume_banners($banners, hash_cut $options, qw/uid/);
        do_update_table(PPC(pid => $pids_to_resume), 'phrases',
            {statusAutobudgetShow => 'Yes', LastChange__dont_quote => 'LastChange',
                statusBsSynced__dont_quote => sql_case('pid', \%groups_which_reset_statusBsSynced, default__dont_quote => 'statusBsSynced')
            },
            where => {pid => SHARD_IDS}
        );
        # DIRECT-78307: В вызовах из нового интерфейса могут попадаться группы в которых ничего делать не надо
        # выбираем кампании только из тех групп которые действительно обновились
        my $cids = get_cids(pid => $pids_to_resume );
        schedule_forecast_multi($cids);
        
        Models::Banner::update_banner_statuses_is_obsolete($banner_ids, $update_before);
        Models::Banner::update_adgroup_statuses_is_obsolete($pids_to_resume, $update_before);
    }
    return $result;
}

=head2 get_main_banner($group, $main_bid)

    Получить "главный" баннер из группы который будет показан в интерфейсе
    Если передан идентификатор баннера - отдаем этот баннер, при условии, что он есть в группе.
    Если баннера в группе нет - определяем "главный" сами.
=cut

sub get_main_banner {
    my ($group, $main_bid) = @_;

    my @banners = sort {$a->{bid} <=> $b->{bid}} @{$group->{banners}};
    my @filtered_banners;
    @filtered_banners = grep { $_->{bid} == $main_bid } @banners if $main_bid;
    @banners = @filtered_banners if @filtered_banners;

    return undef unless @banners;
    return $banners[0] if scalar @banners == 1;
    
    my $sort_key = exists $banners[0]->{ad_type} ? 'ad_type' : 'real_banner_type';

    my ($imagead_banners, $text_banners) = part {
        my $type = $_->{$sort_key} // '';
        return ($type eq 'image_ad' || $type eq 'cpc_video' || $type eq 'mcbanner') ? 0 : 1;
    } @banners;

    $imagead_banners //= [];
    $text_banners //= [];

    # active banner
    my $banner = first {
        ($_->{statusShow} // '') eq 'Yes'
            && (
                ($_->{statusActive} // '') eq 'Yes'
                || (
                    ($_->{statusPostModerate} // '') eq 'Yes'
                    && $group->{statusPostModerate} eq 'Yes'
                    && ( $_->{href} || ($_->{phoneflag} // '') eq 'Yes' )
                )
            )
    } @$text_banners, @$imagead_banners;
    return $banner // $text_banners->[0] // $imagead_banners->[0];
}

sub _separate_by {
    
    my ($by, $list) = @_;
    
    my %objects;
    foreach (@$list) {
        $objects{$$_{$by}} = [] unless exists $objects{$$_{$by}};
        push @{$objects{$$_{$by}}}, $_
    }
    
    return \%objects;    
}

=head2 send_groups_to_bs(%conditions)

    Отправка adgroup в БК

    %conditions (необходимо задать одно из условий)
        by_pid - номер группы
        by_bid - по id одного баннера из группы отправить в БК
                    все баннеры из группы

        only - (опциональный параметр) проставлять statusBsSynced только у указанных таблиц

=cut
sub send_groups_to_bs {
    my %conditions = @_;

    my @tables = qw/banners phrases bids bids_retargeting bids_dynamic/;
    if (my $only = delete $conditions{only}) {
        my %only_map = map { $_ => 1 } @$only;
        @tables = grep { $only_map{$_} } @tables;
    }

    my $pids;
    if ($conditions{by_pid}) {
        $pids = $conditions{by_pid};
    } elsif ($conditions{by_bid}) {
        $pids = get_pids(bid => $conditions{by_bid});
    }
    return undef unless $pids;

    foreach my $t (@tables) {
        do_update_table(PPC(pid => $pids), $t, {
            statusBsSynced => 'No',
            $t ne 'bids_dynamic' ? (
            $t =~ 'bids'
                ? ('modtime__dont_quote' => 'modtime')
                : ('LastChange__dont_quote' => 'LastChange')
            ) : (),
       }, where => {pid => SHARD_IDS})
    }

    return $pids;
}

=head2 get_groups($filters, $options)

    Получить группы

    $filters
        bid - выборка только по заданным номерам баннеров
        cid - выборка по кампании
        pid - выборка по идентификатору группы
        limit/offset - 
        tag_id - выборка группы по id тегов( == -1 выборка объявлений без тегов)
        tab - выборка групп по статусам баннеров
            возможные значения: active, off, arch, draft, decline, wait, running_unmoderated
        context - выборка по тексту баннера(заголовок или тело или url)  
        phrase - выбока по тексту фраз
        filter - выбока по фильтрам
        group_name - выборка по названию группы
        arch_banners => 1 | 0 | undef - выборка только архивных (1), только не архивных (0), или всех (undef) баннеров
        adgroup_types - массив типов групп, которые необходимо выбирать (["base", "dynamic", "mobile_content", "performance", "mcbanner"]),
                        по умолчанию выбираются "base"
        disabled_geo_only => 1 - выбрать только те группы, где есть баннеры с minus_geo

    $options
        get_tags - получить все теги у группы
        pure_groups - выборка только групп(без баннеров и фраз)
        only_pid - получить только номера групп(pid)
        pass_empty_groups - 1|0 0 - отфильтровывать пустые группы 1 - не отфильтровывать пустые группы 
        only_creatives - выборка баннеров без доп информации(без визитки, адреса, картинки)
        banner_limit/banner_offset - постраничная выборка баннеров из группы
                        (имеет смысл только при выборке одной группы)
        banner_order_by_status - сортировка баннеров по статусам (активные, остановленные, отклоненные)
        skip_banner_template - пропускать обработку шаблона баннера
        get_multiplier_stats => 1 | 0 - выбирать доп. статистику для различных корректировок. В группах появится
                        дополнительное поле, формат которого описан в HierarchicalMultipliers::adjustment_bounds()
        get_lang => 1 | 0 -- определить язык баннеров
        search_phrase_in_archive_too - при выборке по тексту фраз - искать еще и в bids_arc

    Результат
        [{banners => [], phrases => [], ....}, ....]
        общее число групп попадающих под условие $filters

=cut

# TODO: pass_empty_groups - сделать умолчальным поведением
# TODO adgroup: переделать вызов на ($camp, $filter, $options), join campaings/camp_options из _get_pure_groups убрать
# сейчас сделано для поддержки формата BannersCommon::get_banners
# TODO: дать описание параметру $options->{phrase_filters}
sub get_groups {

    my ($filters, $options) = @_;
    my ($groups, $total) = get_pure_groups($filters, $options);

    unless ($options->{pure_groups}) {
        my $phrases = _separate_by(pid => Models::Phrase::get_phrases($groups, filters => $options->{phrase_filters}));
        my $relevance_matches = Direct::Bids::BidRelevanceMatch->get_by(
            adgroup_id => [map {$_->{pid}} @$groups]
        )->items_by('adgroup_id');
        my $banner_options = hash_cut $options, qw/only_creatives get_lang/;
        if (exists $options->{banner_limit}) {
            $banner_options->{limit} = $options->{banner_limit}; 
            $banner_options->{offset} = $options->{banner_offset};
        }
        $banner_options->{adgroup_types} = $filters->{adgroup_types};
        $banner_options->{order_by_status} = $options->{banner_order_by_status} if exists $options->{banner_order_by_status};

        my ($creatives, $total_creatives) = @$groups ? Models::Banner::get_pure_creatives($groups, $filters, $banner_options) : ([], 0);

        my $banners = _separate_by(pid => $creatives);
        my $completed_groups = is_completed_groups([map {$_->{pid}} @$groups]);

        # отфильтровываем группы, которые содержат только (не)архивные баннеры
        if (defined $filters->{arch_banners} && !($options->{pass_empty_groups} // 1)) {
            $groups = [ grep { exists $banners->{$_->{pid}} } @$groups ];
        }

        foreach my $group (@$groups) {
            $group->{phrases} = $phrases->{$$group{pid}};
            $group->{relevance_match} = [ map { $_->to_template_hash } @{ $relevance_matches->{$group->{pid}} // [] } ];
            # число будет правильное при запросе одной группы
            $group->{total_banners} = $total_creatives;
    
            hash_merge $group, _compile_group_params($group);
            $group->{banners} = [map {
                hash_merge $_, compile_banner_params($group, $_, %{hash_cut $options, qw/easy_user skip_banner_template/});
                $_;
            } @{$banners->{$$group{pid}}}];
            $group->{is_completed_group} = $completed_groups->{$group->{pid}} || 0;
        }
        Retargeting::get_retargetings_into_groups($groups);
        Retargeting::get_target_interests_into_groups($groups);
        Retargeting::get_search_retargeting_into_groups_from_multipliers($groups);

        if ($filters->{adgroup_types} && any {$_ eq 'mobile_content'} @{$filters->{adgroup_types}}) {
            my $mobile_content_adgroups = [grep { $_->{adgroup_type} eq 'mobile_content' } @$groups];
            my $mobile_content_by_gid = Direct::AdGroups2::MobileContent->get_mobile_content_by(adgroup_id => [map { $_->{pid} } @$mobile_content_adgroups]);
            for my $group (@$mobile_content_adgroups) {
                $group->{mobile_content} = $mobile_content_by_gid->{$group->{pid}}->to_template_hash;
                for (@{$group->{banners}}) {
                    $_->{filter_domain} = $group->{mobile_content}->{publisher_domain} || Yandex::IDN::idn_to_ascii($group->{mobile_content}->{store_app_id});
                }
            }
        }

        if ($filters->{adgroup_types} && any {$_ eq 'dynamic'} @{$filters->{adgroup_types}}) {
            my $dyn_adgroups = [grep { $_->{adgroup_type} eq 'dynamic' } @$groups];
            my $dyn_conds_by_gid = Direct::DynamicConditions->get_by(adgroup_id => [map { $_->{pid} } @$dyn_adgroups], order_by => 'dyn_id', with_additional => 1)->items_by('gid');
            my $market_ratings = Direct::Market::get_domains_rating([map {$_->{main_domain}} @$dyn_adgroups]);

            for my $dyn_adgroup (@$dyn_adgroups) {
                $dyn_adgroup->{dynamic_conditions} = [map { $_->to_template_hash } @{$dyn_conds_by_gid->{$dyn_adgroup->{pid}}}];
                $dyn_adgroup->{market_rating} = $market_ratings->{$dyn_adgroup->{main_domain} // '_EMPTY_DOMAIN'};
            }
        }

        if ($filters->{adgroup_types} && any {$_ eq 'performance'} @{$filters->{adgroup_types}}) {
            my $performance_adgroups = [grep { $_->{adgroup_type} eq 'performance' } @$groups];
            my $performance_filters = Direct::PerformanceFilters->get_by(adgroup_id => [map {$_->{pid}} @$performance_adgroups], with_additional => 1)->items_by('adgroup_id');
            my @ret_cond_ids = uniq(grep { $_ } map { $_->ret_cond_id } map { @$_ } values %$performance_filters);
            my $ret_cond_by = Direct::RetargetingConditions->get_by(ret_cond_id => \@ret_cond_ids)->items_by;
            for my $performance_adgroup (@$performance_adgroups) {
                $performance_adgroup->{performance_filters} = [
                    map { hash_merge $_->to_template_hash, {retargeting => ($_->ret_cond_id ? $ret_cond_by->{ $_->ret_cond_id }->to_template_hash : undef)} }
                    @{$performance_filters->{$performance_adgroup->{pid}}}
                ];
            }
        }

        calc_banners_quantity($groups);
    }

    # geo-legal DIRECT-62480
    if (@$groups) {
        my $client_id = get_clientid(pid => $groups->[0]->{pid});
        my $b_minus_geo = get_all_sql(PPC(pid => [map { $_->{pid} } @$groups]), [
            "select b.pid, b.bid, bmg.minus_geo
            from banners b join banners_minus_geo bmg on bmg.bid = b.bid",
            where => {
                'b.pid' => SHARD_IDS,
                'b.statusShow' => 'Yes',
                'b.statusPostModerate__ne' => 'Rejected',
                'b.statusArch' => 'No',
                'bmg.type' => 'current',
            }
        ]);
        my %minus_geo_by_pid = partition_by { $_->{pid} } @$b_minus_geo;
        for my $g (@$groups) {
            next unless $g->{pid} && $minus_geo_by_pid{$g->{pid}};
            my @b_minus_geo = uniq map { @{get_num_array_by_str($_->{minus_geo})} } @{$minus_geo_by_pid{$g->{pid}}};
            my $geo = $g->{geo};
            my ($effective_geo, $g_minus_geo);
            if (@b_minus_geo) {
                ($effective_geo, $g_minus_geo) = GeoTools::exclude_region($geo, \@b_minus_geo, { ClientID => $client_id });
            }
            $g->{minus_geo} = \@b_minus_geo;
            $g->{effective_geo} = $effective_geo;
            $g->{disabled_geo} = join ",", @{$g_minus_geo//[]};
        }
    }

    if ($options->{count_banners_types}) {
        my $counts = get_all_sql(PPC(pid => [map { $_->{pid} } @$groups ]), [
            "select pid, banner_type, count(*) as `count` from banners",
            where => {
                pid => SHARD_IDS,
            },
            "GROUP BY pid, banner_type"
        ]);
        my %count_by_pid;
        for my $c (@$counts) {
            my %c = %$c;
            delete $c{pid};
            push @{$count_by_pid{$c->{pid}}}, \%c;
        }
        for my $g (@$groups) {
            $g->{group_banners_types} = $count_by_pid{$g->{pid}};
        }
    }
    
    if ($options->{get_phrases_statuses}) {
        bs_get_phrases_statuses($groups, {
            update_phrases => 1,
            update_retargetings => scalar(any {exists $_->{retargetings} || exists $_->{target_interests}} @$groups),
            update_dynamic_conditions => scalar(any { exists $_->{dynamic_conditions} } @$groups),
            update_performance => scalar(any { exists $_->{performance_filters} } @$groups),
        });
    }

    return wantarray
        ? ($groups, $total)
        : $groups;
}


=head2 has_camp_groups_with_disabled_geo

    Определяет, есть ли в заданных кампаниях|группах, группы с ограничением таргетинга.
    Принимает на вход фильтр:
       {
         cid | pid  => [кампании, группы]  #в случае, если будут указанны оба поля, то фильтровать будем по pid
         status => "active"  - учитывать только активные баннеры в группах
       }

    Возвращает:
     1 - в случае, если нашлась группа с ограничением таргетинга
     0 - если не нашлось групп с геоограничениями.
=cut
sub has_camp_groups_with_disabled_geo {
    my (%opts) = @_;

    my $profile = Yandex::Trace::new_profile('has_camp_groups_with_disabled_geo.with_cache');
    my $banners_minus_geo = get_banners_with_minus_geo(%opts);

    my $camp_has_group_with_disabled_geo = 0;
    my %banners_minus_geo_by_pid = partition_by { $_->{pid} } @$banners_minus_geo;
    my $disabled_geo_cache = {};
    foreach my $pid (keys %banners_minus_geo_by_pid) {
        my @group_minus_geo = uniq map { @{get_num_array_by_str($_->{minus_geo})} } @{$banners_minus_geo_by_pid{$pid}};
        #считаем, что geo и ClientID не могут быть разными в рамках одного pid. Берем из первого элемента
        my $geo = $banners_minus_geo_by_pid{$pid}->[0]->{geo};
        my $client_id = $banners_minus_geo_by_pid{$pid}->[0]->{ClientID};

        my $minus_geo_str = join ',', @group_minus_geo;
        my $digest = md5_hex_utf8(join '~', ($geo, $minus_geo_str));
        my $disabled_geo = $disabled_geo_cache->{$digest};

        if (!$disabled_geo) {
            $disabled_geo = GeoTools::get_disabled_geo($geo, \@group_minus_geo, $client_id);
            $disabled_geo_cache->{$digest} = $disabled_geo;
        }

        if (@$disabled_geo) {
            $camp_has_group_with_disabled_geo = 1;
            last;
        }
    }
    undef $profile;
    return $camp_has_group_with_disabled_geo;
}


=head2 get_banners_with_disabled_geo

    Находит баннеры ограничивающие таргетинг группы из-за минус-регионов
    Принимает на вход фильтр:
       {
         cid | pid | bid => [кампании, группы или баннеры]
         status => "active"
       }
    Возвращает баннеры, в виде следующей структуры
     [
        {
         bid => $bid
         pid => $pid
         disabled_geo => [минус-регионы, которые ограничивают таргетинг группы]
         flags => [список всех флагов баннера]
        }
     ]

=cut
sub get_banners_with_disabled_geo {
    my (%opts) = @_;

    my $profile = Yandex::Trace::new_profile('get_banners_with_disabled_geo.with_cache');
    my $banners_minus_geo = get_banners_with_minus_geo(%opts);

    my @bids_with_disabled_geo = ();
    my $disabled_geo_cache = {};
    for my $row (@$banners_minus_geo) {
        my $digest = md5_hex_utf8(join '~', ($row->{geo}, $row->{minus_geo}));
        my $disabled_geo = $disabled_geo_cache->{$digest};
        if (!$disabled_geo) {
            my @minus_geo = @{get_num_array_by_str($row->{minus_geo})};
            $disabled_geo = GeoTools::get_disabled_geo($row->{geo}, \@minus_geo, $row->{ClientID});
            $disabled_geo_cache->{$digest} = $disabled_geo;
        }
        if (@$disabled_geo) {
            push @bids_with_disabled_geo, {
                    bid => $row->{bid},
                    pid => $row->{pid},
                    disabled_geo => $disabled_geo,
                    flags => [ uniq split ',', $row->{flags} ]
                };
        }
    }
    return \@bids_with_disabled_geo;
}


=head2 get_banners_with_minus_geo

    Выбирает из БД баннеры содержащие минус-регионы (type='current')
    Принимает на вход фильтр:
       {
         cid | pid | bid => [кампании, группы или баннеры]  #если указанно несколько полей, то приоритет будет у менее крупного объекта
         status => "active" - выбирать только активные баннеры
       }
    Находит для заданного множества каманий/групп/баннеров все баннеры содержащие минус-регионы в таблице banners_minus_geo
    Возвращает баннеры, в виде следующей структуры
     [
        {
         bid => $bid
         pid => $pid
         ClientID => id клиента
         geo => строка с таргетингом группы (phrases.geo)
         flags => строка с флагами баннера (banners.flags)
         minus_geo => минус-регионы баннера (banners_minus_geo.minus_geo)
        }
     ]
=cut
sub get_banners_with_minus_geo {
    my (%opts) = @_;
    my ($key, $values);

    if ($opts{bid}) {
        $key = 'bid';
    } elsif($opts{pid}) {
        $key = 'pid';
    } elsif ($opts{cid}) {
        $key = 'cid';
    } else {
        croak "Missing option cid, pid or bid";
    }
    $values = ref($opts{$key}) eq 'ARRAY' ? $opts{$key} : [$opts{$key}];

    my $where = {
        "b.$key" => SHARD_IDS,
    };

    if (exists $opts{status} && $opts{status} eq 'active') {
        hash_merge $where, {
                'b.statusShow' => 'Yes',
                'b.statusPostModerate__ne' => 'Rejected',
                'b.statusArch' => 'No',
                'g.statusModerate' => 'Yes',
            };
    };

    my $banners_minus_geo = get_all_sql(PPC($key => $values), [
            "select g.pid, g.geo, bmg.minus_geo as minus_geo, b.bid, b.flags, u.ClientID
            from banners b
            join banners_minus_geo bmg on bmg.bid = b.bid and bmg.type = 'current'
            join phrases g on g.pid = b.pid
            join campaigns c on c.cid = g.cid
            join users u on u.uid = c.uid",
            where => $where,
        ]);

    return $banners_minus_geo;
}

=head get_geolegal_flags_for_banners

    Из указанного списка pid находит активные баннеры ограничиающие таргетиг группы
       и выбирает все флаги, по которым может потребоваться предоставление документов в модерацию.

    принимает на вход:
      pids - идентификаторы групп, для которых нужно выбрать флаги баннеров

    Возвращает список геолицензируемых флагов @flags_list, для отображения в нижней плашке уведомлений.

=cut
sub get_geolegal_flags_for_banners {
    my $pids = shift;

    my $filtered_banners = get_banners_with_disabled_geo('pid'=>$pids, 'status'=> 'active');

    my %geo_legal_flags_hash = map { $_ => 1 } @BannerFlags::GEO_LEGAL_FLAGS;
    my @flag_list = uniq grep {exists $geo_legal_flags_hash{$_} } map {s/:.+$//g; $_} map { @{$_->{flags}} } @$filtered_banners;
    return \@flag_list;
}

=head2 calc_banners_quantity
    
    Для каждой переданой группы подсчитывает параметры: banners_quanity, banners_arch_quantity, banners_canarch_quantity
    и записывает в группы, т.о. меняет входные данные.

=cut
sub calc_banners_quantity {
    my ($groups, $banner_status) = @_;

    my $banners_qty = {};
    if (my @pids = map {$_->{pid}} grep {$_->{pid}} @$groups) { 
        $banners_qty = {map { $_ => {banners_quantity => 0, 
                                    banners_arch_quantity => 0,
                                    banners_canarch_quantity => 0}} @pids};

        my $banners_qty_db = get_hashes_hash_sql(PPC(pid => \@pids), [
                        qq{
                            SELECT
                                pid,
                                COUNT(bid) banners_quantity,
                                COUNT(IF(statusArch = 'Yes', 1, NULL)) banners_arch_quantity,
                                COUNT(IF((statusArch = 'No' AND statusShow = 'No'), 1, NULL)) banners_canarch_quantity
                            FROM banners
                        },
                        WHERE => {pid => SHARD_IDS},
                        "GROUP BY pid"
                    ]);
        # Все что в БД нашли, записываем поверх.
        hash_merge $banners_qty, $banners_qty_db;

        if (defined ($banner_status) &&  Models::AdGroupFilters::is_valid_status_filter($banner_status)) {
            my $filter_banners = Models::AdGroupFilters::filter_banners_preview({tab=>$banner_status, pid=>\@pids});
            foreach (keys %$banners_qty) {
                # не заполняем данными группы, если они не должны показываться на текущей вкладке.
                next unless defined $filter_banners->{$_};
                $banners_qty->{$_}->{edit_banners_quantity} = scalar(@{$filter_banners->{$_}});
            }
        }
    }

    foreach my $group (@$groups) {
        # Если были переданы еще не сохраненные новые баннеры, то их тоже считаем.
        my $new_banners_count = scalar(grep {!$_->{bid}} @{$group->{banners}});
        $banners_qty->{$group->{pid}}->{banners_quantity} += $new_banners_count if $new_banners_count && exists $banners_qty->{$group->{pid}};

        hash_merge $group, $banners_qty->{$group->{pid}} if (exists $banners_qty->{$group->{pid}});
    }
    # Новые, еще не созданные группы имеют те количества баннеров, которые сейчас содержатся в каждом из groups
    foreach my $new_group (grep {!$_->{pid}} @$groups) {
        hash_merge $new_group, {banners_quantity => scalar(@{$new_group->{banners}}),
                                banners_arch_quantity => 0,
                                banners_canarch_quantity => 0,
                                edit_banners_quantity => scalar(@{$new_group->{banners}}),
                               };
    }

}
sub _compile_group_params {

    my ($group) = @_;

    my @phrases = @{$group->{phrases} || []};

    return {
        phrases => \@phrases,
        geo_names => get_geo_names($group->{geo}, ', '), # TODO!!! убрать, должно заполняться в modify_banners_geo_for_translocal_before_show
    };
}

=head2 get_pure_groups($filters, $options)

    Получить только группы(без баннеров и фраз)

    $filters
        bid - выборка только по заданным номерам баннеров
        pid - выборка по номерам групп
        cid - выборка по кампании
        limit/offset - 
        tag_id - выборка группы по id тегов( == -1 выборка объявлений без тегов)
        tab - выборка групп по статусам баннеров
            возможные значения: active, off, arch, draft, decline, wait, running_unmoderated
        context - выборка по тексту баннера(заголовок или тело или url)  
        phrase - выбока по тексту фраз
        adgroup_types - массив типов групп, которые необходимо выбирать (["base", "dynamic", "mobile_content", "performance", "mcbanner"]),
                        по умолчанию выбираются "base"
        disabled_geo_only => 1 - выбрать только те группы, где есть баннеры с minus_geo

    $options
        get_tags - получить все теги у группы
        only_pid - получить только номера групп(pid)
        camp_bs_queue_status - получить информацию о наличии кампании в очереди транспорта БК
        get_multiplier_stats => 1 | 0 - выбирать доп. статистику для различных корректировок. В группах появится
                        дополнительное поле, формат которого описан в HierarchicalMultipliers::adjustment_bounds()
        search_phrase_in_archive_too - при выборке по фразам - искать еще и в bids_arc
        
    Результат
        @$groups
        общее число групп попадающих под условие $filters

=cut

sub get_pure_groups {
    
    my ($conditions, $options) = @_;
    $conditions ||= {};
    my $filters = hash_kgrep sub {$_ !~ /^bid/ && $_ ne 'arch_banners'}, $conditions;
    my $tab_filters;
    if ($conditions->{bid}) {
        my @pids = @{get_pids(bid => $conditions->{bid})};
        if ($conditions->{pid}) {
            push @pids, ref $conditions->{pid} eq 'ARRAY' ? @{$conditions->{pid}} : $conditions->{pid};
        }
        $filters->{'g.pid'} = \@pids;
        $tab_filters = {pid => \@pids};
        delete $filters->{pid};
    }
    if ($filters->{cid}) {
        $tab_filters = {cid => $filters->{cid}};
        $filters->{'g.cid'} = delete $filters->{cid};
    }
    if ($filters->{pid}) {
        my $pids = delete $filters->{pid};
        $pids = [$pids] unless ref $pids eq 'ARRAY';
        $tab_filters = {pid => $pids};
        if ($filters->{'g.pid'}) {
            # если был указан и bid и pid - берем пересечение по pid
            $filters->{'g.pid'} = xisect($filters->{'g.pid'}, $pids);
        } else {
            $filters->{'g.pid'} = $pids;
        }
    }
    my $sql_t = q[
        SELECT %s
            %s
        FROM
            phrases g
            %s
    ];
    
    my @fields = ('g.cid', 'g.pid', 'g.pid AS adgroup_id', 'g.group_name', 'g.bid AS first_available_bid', 'g.geo', 'g.adgroup_type');
    my (@tables, @group_by, @binds);
    my %where;
    my $limit = '';

    my $with_dynamic = $filters->{adgroup_types} && any { $_ eq 'dynamic' } @{$filters->{adgroup_types}};

    if ($filters->{limit}) {
        push @binds, delete $filters->{limit};
        if (defined $filters->{offset}) {
            $limit = 'LIMIT ? OFFSET ?';
            push @binds, delete $filters->{offset};
        } else {
            $limit = 'LIMIT ?';
        }
    }
    
    if ($filters->{tag_id}) {
        push @tables, 'LEFT JOIN tag_group tg ON g.pid = tg.pid';
        push @group_by, 'g.pid'; 
        my $tag = delete $filters->{tag_id};
        if ($tag > 0) {
            $filters->{'tg.tag_id'} = $tag;
        } else {
            $filters->{'tg.pid__is_null'} = 1;
        }
    }
    
    if ($filters->{context}) {
        
        my @context = grep {$_ ne ''} map {s/^\s+|\s+$//g; $_} split /,/, $filters->{context};
        if (@context) {
            $where{_TEXT} = join " OR ", map {
                sprintf q[IF(b.banner_type = 'dynamic', 0, b.title LIKE %s) OR IF(b.banner_type != 'text', 0, b.title_extension LIKE %s) OR b.body LIKE %s OR b.href LIKE %s],
                    (sql_quote("\%$_\%")) x 4;
            } @context;
            
            push @fields, 'b.bid';
            push @tables, 'JOIN banners b ON b.pid = g.pid';
            push @group_by, 'g.pid';
        }
        delete @{$filters}{qw/context tab/};
    }

    my @filter_phrases;
    if (exists $filters->{phrase}) {
        my @keyword_tables = qw/bids/;
        if ($options->{search_phrase_in_archive_too}) {
            push @keyword_tables, 'bids_arc';
        }
        
        @filter_phrases = grep {$_ ne ''} map {s/^\s+|\s+$//g; $_} split /,/, $filters->{phrase};
        if (@filter_phrases) {
            my @conditions;
            for my $kw_table (@keyword_tables) {
                my $where_part = sql_condition([
                    "$kw_table.cid__dont_quote" => "g.cid",
                    "$kw_table.pid__dont_quote" => "g.pid",
                    _OR => [
                        map {(
                            _TEXT => "SUBSTRING_INDEX($kw_table.phrase, \" -\", 1) LIKE " . sql_quote('%' . sql_quote_like_pattern($_) . '%')
                        )} @filter_phrases
                    ],
                ]);
                push @conditions, sprintf("EXISTS (SELECT 1 FROM $kw_table WHERE %s)", $where_part);
            }

            if ($with_dynamic) {
                push @conditions, sprintf(
                    "EXISTS (
                        SELECT 1
                        FROM bids_dynamic bd_search
                        JOIN dynamic_conditions dc_search ON (bd_search.dyn_cond_id = dc_search.dyn_cond_id)
                        WHERE bd_search.pid = g.pid AND (%s)
                    )",
                    join(" OR ", map { 'dc_search.condition_name LIKE '.sql_quote("\%$_\%") } @filter_phrases)
                );
            }

            $where{_TEXT} = sql_condition({_OR => [ map {( _TEXT => $_ )} @conditions ]});
        }
        delete @{$filters}{qw/phrase tab/};

        for my $kw_table (@keyword_tables) {
            push @tables, "LEFT JOIN $kw_table ON $kw_table.cid = g.cid AND $kw_table.pid = g.pid";
        }
        my $kw_phrase;
        if (@keyword_tables > 1) {
            $kw_phrase = sprintf('GROUP_CONCAT(COALESCE(%s))', join(',', map { "$_.phrase" } @keyword_tables));
        } else {
            # keyword_tables[0] = 'bids'
            $kw_phrase = "GROUP_CONCAT(bids.phrase)"
        }

        if ($with_dynamic) {
            push @tables, "LEFT JOIN bids_dynamic bd ON (bd.pid = g.pid) LEFT JOIN dynamic_conditions dc ON (dc.dyn_cond_id = bd.dyn_cond_id)";
            push @fields, "
                IF(g.adgroup_type IN ('base', 'mcbanner'), $kw_phrase,
                IF(g.adgroup_type = 'dynamic', group_concat(dc.condition_name),'')) AS phrase
            ";
        } else {
            push @fields, "$kw_phrase AS phrase";
        }

        push @group_by, 'g.pid';
    }

    if (exists $filters->{group_name}) {
        if (my @filter_groups = (grep { $_ ne '' } map { s/^\s+|\s+$//; lc } split /,/, $filters->{group_name})) {
            $where{_TEXT} = join ' OR ', (map { 'g.group_name LIKE '.sql_quote("\%$_\%") } @filter_groups);
        }
        delete $filters->{group_name};
    }

    if (exists $filters->{tab}) {
        croak 'Inconsistent state, filter tab can only be used with cid, pid, bid filter' if !defined $tab_filters;
        my %condition = Models::AdGroupFilters::get_status_condition(delete $filters->{tab}, filter => $tab_filters);

        hash_merge $filters, $condition{where} if $condition{where};
        push @tables, xflatten $condition{tables} if $condition{tables};
        push @group_by, xflatten $condition{group_by} if $condition{group_by};
        push @fields,  xflatten $condition{fields} if $condition{fields};
    }
    
    unless ($options->{only_pid}) {
            push @fields,
                "IF(c.archived = 'Yes' OR g.adgroup_type IN ('dynamic', 'performance'), 0, g.is_bs_rarely_loaded) AS is_bs_rarely_loaded",
                "mw.mw_text minus_words",
                (map {"g.$_"} qw/statusBsSynced geo statusAutobudgetShow PriorityID/,
                                qw/statusShowsForecast forecastDate/,
                                qw/statusPostModerate statusModerate/),
                "gp.href_params";

            push @tables,
                "JOIN campaigns c ON g.cid = c.cid",
                "LEFT JOIN minus_words mw ON g.mw_id = mw.mw_id",
                "LEFT JOIN group_params gp ON (gp.pid = g.pid)";
    }

    # по умолчанию работаем только с текстовыми группами, что бы не менять все вызовы get_groups
    my $adgroup_types = delete($filters->{adgroup_types}) || ['base']; 
    if (any {$_ eq 'dynamic'} @$adgroup_types) {
        push @fields, "gd.main_domain_id, d.domain AS main_domain, gd.feed_id, gd.statusBlGenerated";
        push @tables,
            "LEFT JOIN adgroups_dynamic gd ON (gd.pid = g.pid)",
            "LEFT JOIN domains d ON (d.domain_id = gd.main_domain_id)";
    }

    if (any {$_ eq 'mobile_content'} @$adgroup_types) {
        push @fields, qw/
            amc.store_content_href
            amc.mobile_content_id
            amc.device_type_targeting
            amc.network_targeting
            amc.min_os_version
        /;

        # hack: избегаем повторного джойна, если он уже есть в условии табов
        if (none {/join \s+ adgroups_mobile_content\b/ixms} @tables) {
            push @tables, 'LEFT JOIN adgroups_mobile_content amc ON g.pid = amc.pid';
        }
    }

    if (any {$_ eq 'content_promotion' || $_ eq 'content_promotion_video'} @$adgroup_types) {
        push @fields, "if(g.adgroup_type IN ('content_promotion', 'content_promotion_video'),
                           ifnull(acp.content_promotion_type,'video'), null) AS content_promotion_content_type";
        push @tables, "LEFT JOIN adgroups_content_promotion acp ON acp.pid = g.pid";
    }

    if (any {$_ eq 'performance'} @$adgroup_types) {
        push @fields, (
            'ad_perf.statusBlGenerated',
            'feeds.feed_id as feed_id',
            'feeds.source as feed_source',
            'feeds.name as feed_name',
            'feeds.url as feed_url',
            'feeds.filename as feed_filename',
        );

        push @tables, 'LEFT JOIN adgroups_performance ad_perf ON g.pid = ad_perf.pid'
                    , 'LEFT JOIN feeds ON ad_perf.feed_id = feeds.feed_id';
    }

    if (any {$_ =~ /^(cpm_outdoor|cpm_indoor)$/} @$adgroup_types) {
        push @fields, (
                'ad_p_targ.page_blocks',
            );

        push @tables, 'LEFT JOIN adgroup_page_targets ad_p_targ ON g.pid = ad_p_targ.pid';
    }

    if (any {$_ eq 'internal'} @$adgroup_types) {
        push @fields, (
                'ad_int.level',
            );

        push @tables, 'LEFT JOIN adgroups_internal ad_int ON g.pid = ad_int.pid';
    }

    $where{'g.adgroup_type'} = $adgroup_types;

    my $disabled_geo_only = delete $filters->{disabled_geo_only};
    
    delete @{$filters}{qw/tab limit offset/};
    my $sql = sprintf $sql_t, 'SQL_CALC_FOUND_ROWS', join(',', @fields), join("\n", @tables);
    my %field_to_table_alias = ( uid => 'c', phrase => 'bi', );
    foreach (keys %$filters) {
        my $name = /\./ || /^_(TEXT|OR|AND)$/
            ? $_
            : ( ($field_to_table_alias{$_}||'g').".".$_);
        $where{$name} = $filters->{$_};
    }
    my %shard = choose_shard_param(\%where, [qw/uid pid bid cid tag_id/], set_shard_ids => 1);
    if ($disabled_geo_only) {
        my $sql = sprintf $sql_t, '', 'g.pid', join "\n", @tables;
        my $pids = get_one_column_sql(PPC(%shard), [ $sql, WHERE => \%where, ]);
        my @pids_with_disabled_geo;
        my $banners_minus_geo = get_all_sql(PPC(%shard), [
            "select g.pid, g.geo, group_concat(bmg.minus_geo) as minus_geo,
            u.ClientID
            from banners b
            join phrases g on g.pid = b.pid
            join banners_minus_geo bmg on bmg.bid = b.bid
            join campaigns c on c.cid = g.cid
            join users u on u.uid = c.uid",
            where => { 'g.pid' => $pids },
            'group by g.pid'
        ]);
        for my $row (@$banners_minus_geo) {
            my @minus_geo = @{get_num_array_by_str($row->{minus_geo})};
            my ($effective_geo, $disabled_geo) = GeoTools::exclude_region($row->{geo}, \@minus_geo, { ClientID => $row->{ClientID} });
            if (@$disabled_geo) {
                push @pids_with_disabled_geo, $row->{pid};
            }
        }
        if (@pids_with_disabled_geo == 0) {
            return wantarray ? ([], 0) : [];
        }
        %where = ( _AND => [ 'g.pid' => \@pids_with_disabled_geo, %where ] );
    }
    my $groups = get_all_sql(PPC(%shard), [$sql,
        WHERE => \%where, 
        (@group_by ? 'GROUP BY ' . join(',', uniq @group_by) : ()),
        ($limit ? ('ORDER BY g.pid', $limit) : 'ORDER BY NULL')
    ], @binds);
    my $total = select_found_rows(PPC(%shard));


    my $camps = {};
    unless ($options->{only_pid}) {
        my @unique_cids = keys %{+{map { $_->{cid} => undef } @$groups}};

        my $iter = each_array(@$groups, @{mass_get_hierarchical_multipliers([map { hash_cut $_, qw/cid pid/ } @$groups])});
        my $camp_multipliers;
        if ($options->{get_multiplier_stats}) {
            # Грузим коэффициенты для кампаний, они нам тоже нужня для расчётов
            my $camp_iter = each_array(@unique_cids, @{mass_get_hierarchical_multipliers([map { +{cid => $_} } @unique_cids])});
            while (my($cid, $multiplier_set) = $camp_iter->()) {
                $camp_multipliers->{$cid} = $multiplier_set;
            }
        }
        my %clients_by_cid;
        while (my ($group, $multiplier_set) = $iter->()) {
            $clients_by_cid{$group->{cid}} //= get_clientid(cid => $group->{cid});
            $group->{hierarchical_multipliers} = $multiplier_set;
            if ($options->{get_multiplier_stats}) {
                $group->{multiplier_stats} = HierarchicalMultipliers::adjustment_bounds(
                                                $multiplier_set,
                                                $camp_multipliers->{$group->{cid}},
                                                {group => $group, ClientID => $clients_by_cid{$group->{cid}}}
                                              );
            }
        }

        my @camp_fields = ( 
                "c.cid",
                (map {"c.$_"} qw/OrderID timeTarget timezone_id/,
                              qw/autobudget cid uid platform/,
                              qw/statusEmpty sum_to_pay autobudget_date/,
                              qw/autobudgetForecast autobudgetForecastDate statusOpenStat day_budget day_budget_show_mode/),
                (map {"co.$_"} qw/strategy/,
                               qw/day_budget_daily_change_count day_budget_stop_time/,
                               qw/device_targeting/),
                "if(json_type(c.strategy_data->>'\$.bid')='NULL', NULL, c.strategy_data->>'\$.bid') as autobudget_bid",
                "IF(c.strategy_name='no_premium', c.strategy_data->>'\$.place', NULL) as strategy_no_premium",
                "co.minus_words AS campaign_minus_words",
                "c.sum + IF(c.wallet_cid, wc.sum, 0) as sum",
                "c.sum_spent + IF(c.wallet_cid, wc.sum_spent, 0) as sum_spent",
                "c.ContextPriceCoef cContextPriceCoef",
                "co.fairAuction = 'Yes' fairAuction",
                "c.type camp_type",
                "c.archived = 'Yes' camp_is_arch",
                "IFNULL(c.currency, 'YND_FIXED') currency",
                "c.statusShow cstatusShow",
                "c.statusModerate cstatusModerate",
                "co.statusContextStop cstatusContextStop",
                "c.autoOptimization = 'Yes' auto_optimization",
                "wc.sum as wallet_sum",
                "wc.sum_spent as wallet_sum_spent", "wc.day_budget as wallet_day_budget", "wc.day_budget_show_mode as wallet_day_budget_show_mode",
        );
        my @camp_tables;
        if ($options->{camp_bs_queue_status}) {
            push @camp_fields, "IF(beq.cid OR bec.cid, 1, 0) camp_in_bs_queue";
            push @camp_tables, "LEFT JOIN bs_export_queue beq on beq.cid = c.cid",
                               "LEFT JOIN bs_export_candidates bec on bec.cid = c.cid";
        }

        $camps = get_hashes_hash_sql(PPC(cid => \@unique_cids), [
            sprintf("SELECT %s
                     FROM
                        campaigns c
                        LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid AND wc.uid = c.uid
                        JOIN camp_options co ON c.cid = co.cid
                        %s", join(',', @camp_fields), join("\n", @camp_tables)),
            WHERE => {
                "c.cid" => SHARD_IDS,
            }]);

    }

    unless ($options->{only_pid}) {
        my $pids = [map {$_->{pid}} @$groups];
        my $pid_with_lib_mw_id = get_all_sql(PPC(pid => $pids), ["
                SELECT pid, mw_id
                  FROM adgroups_minus_words",
            WHERE => { 'pid' => $pids }
        ]);
        my @lib_mw_id_to_fetch = uniq map {$_->{mw_id}} @$pid_with_lib_mw_id;
        my %lib_minus_words_by_pid = ();

        if (@lib_mw_id_to_fetch) {
            my $lib_minus_word_strs = get_all_sql(PPC(pid => $pids), [ "
                SELECT mw_id, mw_name, mw_text
                  FROM minus_words",
                WHERE => { 'mw_id' => \@lib_mw_id_to_fetch }
            ]);

            my %lib_minus_words = map {$_->{mw_id} => { name => $_->{mw_name}, words => MinusWordsTools::minus_words_str2array($_->{mw_text}) }} @$lib_minus_word_strs;

            for my $pid_to_mw_id (@$pid_with_lib_mw_id) {
                push @{$lib_minus_words_by_pid{$pid_to_mw_id->{pid}} ||= []}, $lib_minus_words{$pid_to_mw_id->{mw_id}};
            }
        }

        #если нет библиотечных элементов, то возвращаем пустой список
        for my $group (@$groups) {
            $group->{library_minus_words} = $lib_minus_words_by_pid{$group->{pid}} || [];
        }
    }

    if ($options->{get_tags}) {
        my $tags = Tag::get_groups_tags(pid => [map {$_->{pid}} @$groups]);
        foreach (@$groups) {
            next unless exists $tags->{$$_{pid}};
            $_->{tags} = {map {$_ => 1} @{$tags->{$$_{pid}}}};
        }
    }

    # если не было отсортировано в sql-е - сортируем в перле
    $groups = [sort {$a->{pid} <=> $b->{pid}} @$groups] unless $limit;
    
    foreach my $group (@$groups) {
        if ($group->{adgroup_type} eq 'mobile_content') {
            $group->{$_} = [split /,/, $group->{$_}] for qw/device_type_targeting network_targeting/;
        }
        if ($group->{adgroup_type} =~ /^(cpm_outdoor|cpm_indoor)$/) {
            $group->{page_blocks} = [map {{page_id => $_->{pageId}, imp_id => $_->{impId}}} @{from_json($group->{page_blocks} // [])}];
        }
        hash_merge $group, $camps->{$group->{cid}} if $camps->{$group->{cid}};

        $group->{campaign_minus_words} = MinusWordsTools::minus_words_str2array($group->{campaign_minus_words});
        $group->{minus_words} = MinusWordsTools::minus_words_str2array($group->{minus_words});
        $group->{cpm_banners_type} = $group->{adgroup_type};

        $group->{allowed_canvas_tags} = Direct::Model::Creative::Constants::ALLOWED_CANVAS_TAGS_BY_ADGROUP_TYPE()->{$group->{adgroup_type}} // {};

        $group->{adgroup_type} = 'cpm_banner' if $group->{adgroup_type} =~ /^(cpm_video|cpm_outdoor|cpm_indoor|cpm_audio|cpm_geoproduct|cpm_geo_pin)$/;
    }

    return wantarray
        ? ($groups, $total)
        : $groups
}

=head2 get_groups_gr ($filters, $options) 

    Возвращает все группы по filters в "современной" структуре. 
    Вычисляет статусы и ставки (при установке флагов). Результат готов для отправки в шаблон на страницу.
    _gr означает, что это в будущем будет переименована в get_groups, после полного перехода на группы.

    Параметры:
    $filters
        bid - выборка только по заданным номерам баннеров
        cid - выборка по кампании
        pid - выборка по идентификатору группы
        limit/offset - 
        tag_id - выборка группы по id тегов( == -1 выборка объявлений без тегов)
        tab - выборка групп по статусам баннеров
            возможные значения: active, off, arch, draft, decline, wait, running_unmoderated
        context - выборка по тексту баннера(заголовок или тело или url)  
        phrase - выбока по тексту фраз
        group_name - выборка по названию группы
        arch_banners => 1 | 0 | undef - выборка только архивных (1), только не архивных (0), или всех (undef) баннеров
        adgroup_types - массив типов групп, которые необходимо выбирать (["base", "dynamic", "mobile_content", "performance", "mcbanner"]),
                        по умолчанию выбираются "base"
        
    $options
        get_tags - получить все теги у группы
        pure_groups - выборка только групп(без баннеров и фраз)
        only_pid - получить только номера групп(pid)
        pass_empty_groups - 1|0 0 - отфильтровывать пустые группы 1 - не отфильтровывать пустые группы 
        only_creatives - выборка баннеров без доп информации(без визитки, адреса, картинки)
        banner_limit/banner_offset - постраничная выборка баннеров из группы
        total_banner_limit - максимальное число баннеров, которые могут содержаться во всех группах.
                       используется для макмисально обрезания баннеров для редактирования.
                       если группы содержать баннеров больше ограничения, то обрезаются как баннеры в последней
                       группе, проходящей по лимиту, так и последующие группы.
        banner_order_by_status - сортировка баннеров по статусам (активные, остановленные, отклоненные)
        skip_banner_template - пропускать обработку шаблона баннера

    на выходе:
        groups - список выбранных групп
        result_options - дополнительные значения, сформированные по мере выполнения функции:
            total - сколько всего групп удовлетворяет фильтру (не всегда совпадает с количеством элементов в groups)
            were_groups_cut_by_banner_limit - флаг были ли обрезаны группы из-за ограничений на общее количество баннеров

=cut
sub get_groups_gr {
    my ($filters, $options) = @_;

    # проверяем на наличие хотя бы одного условия для поиска
    if ( ! scalar keys %{hash_cut $filters, qw/uid cid pid/} ) {
        die "get_groups_gr: Not all required params specified";
    }
    $options->{skip_banner_template} = 1 if $options->{get_auction};

    my %cid_cache;
    my $result_options = {};
    my $groups;
    ($groups, $result_options->{total}) = get_groups($filters, $options);
    if ($options->{total_banner_limit}) {

        my $total_banners = scalar(map {@{$_->{banners}}} @$groups);
        if ($total_banners > $options->{total_banner_limit}) {
            my $group_count = 0; my $banners_count = 0;
            foreach my $group (@{$groups}) {
                $banners_count += scalar(@{$group->{banners}});
                last if ($banners_count > $options->{total_banner_limit});
                $group_count++;
            }
            $groups =  [splice(@$groups, 0, $group_count)];
            $result_options->{were_groups_cut_by_banner_limit} = 1;
        }
    }

    postprocess_groups_gr($groups, $options);
    
    return ($groups, $result_options);

}

=head2 postprocess_groups_gr
    
    Построцессинг групп полученных методом Models::AdGroup::get_groups_gr
    В первую очередь - доабвление данных с торгов БК + некоторые мелочи
    
    входные параметры:
        $groups - список групп [{},...,{}]
        $options - набор опций, аналогичен опциям для get_groups_gr, дополнительно:
            update_shows_forecast - 0|1, если 1 - для всех фраз идем в ADVQ за прогнозом показов (полезно если в группу добавили новые фразы)
            get_auction_for_new_phases - запрашивать торги не только по старым фразам, но и по новым
    
    на выходе:
        $groups - тот же что на входе копию групп не делаем

=cut

sub postprocess_groups_gr {
    my ($groups, $options) = @_;
    
    my $need_phrases_clicks = 0;

    # Составляем список групп, в которых есть все, что надо для показа, который крутится.
    my @full_groups;
    my $has_only_image_ad;

    # При получении статистики используется значение BannerID из корня, поэтому подставляем BannerID от первого баннера.
    foreach my $grp ( @$groups ) {

        $grp->{adgroup_id} = $grp->{pid};

        if ( $grp->{is_completed_group} ) {
            push @full_groups, $grp;

            my $banner = get_main_banner($grp);
            $grp->{BannerID} = $banner->{BannerID};
            foreach my $phr ( @{ $grp->{phrases} } ) {
                next if not defined $phr->{PhraseID};
                $phr->{BannerID} = $banner->{BannerID};
            }
            $has_only_image_ad ||= all { $_->{real_banner_type} eq 'image_ad' || $_->{real_banner_type} eq 'mcbanner'} @{$grp->{banners}};
        }
    }

    if ($options->{update_shows_forecast}) {
        # обновляем прогноз показов не смотря на статус, и не обновляя прогноз в БД
        advq_get_phrases_shows_multi(\@full_groups, timeout => 5);
    }

    # забираем данные из крутилки (торги)
    if ($options->{get_auction}) {
        $need_phrases_clicks = get_groups_auction(\@full_groups, hash_cut $options, qw/get_add_camp_options
                                                                                        camp_spent_today
                                                                                        camp_bs_queue_status
                                                                                        get_auction_for_search_stop_too
                                                                                        get_all_phrases
                                                                                        get_auction_for_new_phases/);
    }

    if ($need_phrases_clicks || $options->{ctx_from_bs}
        || $has_only_image_ad
        || $options->{get_phrases_statuses}
        || any {exists $_->{retargetings} || exists $_->{target_interests} || exists $_->{relevance_match}} @full_groups) {

        my $bs_get_phrases_statuses_options = {
            update_phrases => $options->{ctx_from_bs} || $options->{get_phrases_statuses} || $has_only_image_ad,
            update_retargetings => scalar(any {exists $_->{retargetings} || exists $_->{target_interests}} @full_groups),
            update_dynamic_conditions => scalar(any { exists $_->{dynamic_conditions} } @full_groups),
            update_performance => scalar(any { exists $_->{performance_filters} } @full_groups),
            update_relevance_matches => scalar(any { exists $_->{relevance_match} } @full_groups),
        };
        # Функция хоть и рассчитана на баннеры, но совместимо с группами, т.е. используются phrases и retargetings
        bs_get_phrases_statuses(\@full_groups, $bs_get_phrases_statuses_options);
    }

    return $groups;
}

=head2 get_groups_auction

    Отправляет все баннеры группы в аукцион БК. В данный момент БК ещё принимает данные в старом формате (без групп),
    поэтому пересобирает данные для БК и отправляет.
    Параметры:
        groups - массив стандартных хешей группы
        options:
            camp_spent_today
            get_add_camp_options
            camp_bs_queue_status
            get_auction_for_search_stop_too
            get_all_phrases
            no_pokazometer_data - не высчитывать данные показометра
            get_auction_for_new_phases - запрашивать торги не только по старым фразам, но и по новым

=cut

sub get_groups_auction {
    my ($groups, $options) = @_;

    my %cid_cache;
    my %filter_domain_cache;
    my $need_phrases_clicks = 0;
    my $banners = []; # для создания структуры для отправки в БК за ставками

    my $group_fields = [qw/
        pid adgroup_type geo banners_quantity is_bs_rarely_loaded device_type_targeting mobile_content
    /];

    my $camp_fields = [qw/
        spent_today currency OrderID strategy_no_premium
        fairAuction strategy platform day_budget camp_is_arch autobudget
        autobudget_bid cid device_targeting
    /];

    foreach my $group (@$groups) {
        my $banner = yclone(get_main_banner($group));
        next unless $banner;

        # NB: копируем только нужные поля, иначе некоторые запроcы в API
        # падают по лимиту памяти (см. DIRECT-64891)
        hash_merge $banner,
            Models::Banner::_copy_to_banner($group, {fields => $group_fields}),
            Models::Banner::_copy_to_banner($group, {fields => $camp_fields});

        # Мы всего баннера сделали клон, потом заполнили разными данными, чтобы отправить в БК,
        # но phrases передаем оригинальный, чтобы все данные по фразам записались в этот объект.
        # пропускаем новые фразы, они еще не сохранены, никакой статистики все равно нет.
        $banner->{phrases} = [grep { $_->{id} || $options->{get_auction_for_new_phases} } @{$group->{phrases}}];
        $_->{BannerID} = $banner->{BannerID} foreach @{$banner->{phrases}}; 
        
        # TODO удалить когда перейдем к структуре campaign - group - banners для отправки в торги
        # т.к. значения многих полей дублируются
        $banner->{banner_minus_words} = $group->{minus_words};

        # NB: есть вероятность что у групп из одной кампании домен будет одинаковым, 
        #     вычисление домена делается для каждой группы и занимает время - попытаемся
        #     сэкономить время с помощью кеширования
        if (! defined($banner->{filter_domain}) && defined($banner->{domain})) {
            my $filter_domain = $filter_domain_cache{$banner->{domain}};
            if (! defined $filter_domain) {
                $filter_domain = get_filter_domain($banner->{domain});
                $filter_domain_cache{$banner->{domain}} = $filter_domain;
            }
            $banner->{filter_domain} = $filter_domain;
        }

        if (!$cid_cache{$group->{cid}}) {
            my $vals = $cid_cache{$group->{cid}} = {};
            $vals->{timetarget_coef} = TimeTarget::timetarget_current_coef($group->{timeTarget}, $group->{timezone_id});
            $vals->{camp_rest} = round2s( $group->{sum} - $group->{sum_spent} );
            if ($options->{get_add_camp_options}) {
                if (exists $options->{camp_spent_today}) {
                    $vals->{spent_today} = $options->{camp_spent_today};
                } elsif($group->{day_budget} && $group->{day_budget} > 0 && $group->{OrderID} && $group->{OrderID} > 0) {
                    error("Переданы неправильные данные");
                }
            }
            if ($options->{camp_bs_queue_status}) {
                $vals->{camp_in_bs_queue} = $group->{camp_in_bs_queue};
            }
        }
        hash_merge $banner, $cid_cache{$group->{cid}};
            
        push @$banners, $banner; 
    }

    # для кампаний с отключенным поиском не ходим в "полные" торги, но ходим в bs_get_phrases_statuses за количеством кликов за 28 дней
    my $banners_with_search_on;
    if ($options->{get_auction_for_search_stop_too}) {
        $banners_with_search_on = $banners;
    } else {
        $banners_with_search_on = [grep { ($_->{strategy} || '') ne 'different_places' || $_->{platform} ne 'context' } @$banners];
    }

    if (@{$banners_with_search_on || []}) {
        trafaret_auction($banners_with_search_on);
    }
    
    if (scalar(@$banners) != scalar(@$banners_with_search_on)) {
        # если хоть по одному баннеру не сходили, то заберём клики из bs_get_phrases_statuses
        $need_phrases_clicks = 1;
    }
    
    foreach my $group (@$groups) {
        # для кампаний с независимым управлением(different_places) цены в РСЯ устанавливаются вручную
        Models::AdGroup::calc_context_price($group, currency => $group->{currency}, ContextPriceCoef => $group->{cContextPriceCoef}) 
            if $group->{strategy} && $group->{strategy} ne 'different_places';
    }

    unless ($options->{no_pokazometer_data}) {
        my @take_pokazometer = grep { !$_->{is_bs_rarely_loaded} && !$_->{camp_is_arch} } @$groups;
        if (@take_pokazometer) {
            safe_pokazometer(\@take_pokazometer, net => 'context', map {$_ => $options->{$_}} qw/get_all_phrases/);
        }
    }
        
    return $need_phrases_clicks;
}

=head2 save_group($campaign, $group, %options)

    Создание/обновление группы в кампании (+ баннеров, фраз/категорий, тегов)

    $campaign {
        uid  # используется для отправки уведомлений об изменениях в группе
        cid
        statusEmpty
        statusModerate
        strategy # хеш стратегии, используется для вычисления is_different_places для фраз
        ContextPriceCoef
    }

    $group {
        pid =>
        bid => BannerID # необязательный параметр, если отсутствует будет взят из баннеров (если онные отсутствуют, то до полного перехода на группы передавать обязательно)
        group_name => # для новой группы может отсутствовать, тогда по умолчанию пример значение AdGroup# <ID Группы>
        currency => # валюта
        geo => 
        banners => [] # могут отсутствовать
        phrases => []
        tags => [] # если теги отсутствуют или пустые, то будут удалены
    }

    %options
        UID - $self->{uid} из контроллера (пользователь от которого делается операция)
        i_know_href_params - обновлять параметры фраз
        ignore_suspended - см. Models::Phrase::save_phrases
        ignore_tags - пропускаем_update_group_tags($group);
        ignore_minus_words - не вызывать MinusWords::save_banner_minus_words($pid, $minus_words);
        ClientID - клиент для выбора транслокального гео-дерева
        pass_phrases => пропустить обновление фраз
        ignore_retargetings => пропустить обновление условий ретаргетинга
        ignore_image => пропустить обновление картинки у баннеров группы
        ignore_hierarchical_multipliers => пропустить обновление корректировок
        is_api => сохранение из АПИ (используется для выбора дерева регионов)
        geo_camp_vcard => баннеры сохраняются для кампании с типом geo
        where_from => откуда создали группу, нужно для логирования ("web", "api", "xls", "mediaplan")
        ignore_callouts => 1|0 - не менять уточнения на баннере

    Возвращает созданную/обновленную группу

=cut
sub save_group {
    my ($campaign, $group, %options) = @_;

    croak "Cannot save non-text adgroup"
        if ($group->{adgroup_type} // 'base') ne 'base' || defined $group->{main_domain} || $group->{main_domain_id};

    # Текстовые группы можно создавать только в текстовой/гео? кампании
    my $campaign_type = get_camp_type(cid => $campaign->{cid});
    croak "Cannot save adgroup in non-text campaign" unless $campaign_type eq 'text' || $campaign_type eq 'geo';

    my $banners = $group->{banners};

    my $uid = $campaign->{uid};
    my $is_different_places = $campaign->{strategy}->{name} eq 'different_places';

    my %params = (%options, uid => $uid);
    my ($geoflag, $old_group);
    my $ClientID = $options{ClientID} || get_clientid(uid => $uid);

    my $geo = $group->{geo};
    if (!$group->{geo_is_modified_before_save} && !$options{is_api}) {
        $geo = GeoTools::modify_translocal_region_before_save($group->{geo}, {ClientID => $ClientID});
    }
    my $translocal_opt = $options{is_api} ? {tree => 'api'} : {ClientID => $ClientID};
    $geo = refine_geoid($geo, \$geoflag, $translocal_opt);

    $group->{geo} = $geo;

    unless ($group->{pid}) {
        # create new ad-group
        $group->{pid} = get_new_id('pid', cid => $campaign->{cid});
        $group->{group_name} = get_auto_adgroup_name($group, uid => $uid) unless $group->{group_name};

        # берем bid из баннеров если не задан если в баннерах его тоже нет
        # (баннеры еще не созданы), то мы вычислим его позже
        my $banner = first {$_->{bid}} @$banners;
        $group->{bid} ||= $banner->{bid};

        do_insert_into_table(PPC(pid => $group->{pid}), 'phrases', {
            pid => $group->{pid},
            cid => $campaign->{cid},
            bid => $group->{bid},
            group_name => $group->{group_name},
            geo => $geo,
            statusModerate => 'New',
            statusPostModerate => 'No'
        });
        $group->{is_new} = 1;
        $group->{statusModerate} = 'New';
        $old_group = {
            phrases => [],
        };
    } else {
        my $groups = get_groups({pid => $group->{pid}}, {only_creatives => 1});
        $old_group = $groups->[0];
        
        # для поддержки обратной совместимости со старыми функциями
        $old_group->{pstatusModerate} = $old_group->{statusModerate};
        $old_group->{pstatusPostModerate} = $old_group->{statusPostModerate};
        $group->{statusModerate} = $old_group->{statusModerate};
        
        # first available bid (for old relationships)
        $group->{bid} ||= $old_group->{first_available_bid};

        $group->{group_name} = get_auto_adgroup_name($group, uid => $uid) unless $group->{group_name};

        # TODO adgroup: удалить после полного перехода к группам
        do_update_table(PPC(pid => $group->{pid}), 'phrases',
            { group_name => $group->{group_name} },
            where => {pid => $group->{pid}}
        );
    }

    $group->{cid} = $campaign->{cid};

    if ($banners && scalar @$banners) {
        
        my (@create_desktop_banners, @update_desktop_banners,
            @create_mobile_banners, @update_mobile_banners,
            @create_imagead_banners, @update_imagead_banners);
        foreach my $banner (@$banners) {
            my $vcard = $banner->{vcard} || $banner;

            $banner->{statusEmpty} = $campaign->{statusEmpty};
            $banner->{old_geo} = $old_group->{geo};
            $banner->{new_geo} = $group->{geo};
            # Существующие черновики сохраним черновиками
            $banner->{statusModerate} = (defined $banner->{statusModerate} && $banner->{statusModerate} ne 'New') ? $campaign->{statusModerate} : 'New';
            $banner->{map} = $vcard->{map} = CommonMaps::check_address_map($vcard, { 
                ClientID => $ClientID, 
                ignore_manual_point => ($options{where_from} // '') eq 'web' ? 0 : 1,
            });
            $banner->{pid} = $group->{pid};
            $banner->{cid} = $group->{cid};
            
            if ($banner->{bid}) {
                if ($banner->{banner_type} eq 'desktop') {
                    push @update_desktop_banners, $banner;
                } elsif ($banner->{banner_type} eq 'mobile') {
                    push @update_mobile_banners, $banner;
                } elsif ($banner->{banner_type} eq 'image_ad') {
                    push @update_imagead_banners, $banner;
                } else {
                    confess('unknown banner type ' . ($banner->{banner_type} || '""')); 
                }
            } else {
                # for set geoflag
                $banner->{geo} = $geo;
                if ($banner->{banner_type} eq 'desktop') {
                    push @create_desktop_banners, $banner;    
                } elsif ($banner->{banner_type} eq 'mobile') {
                    push @create_mobile_banners, $banner;
                } elsif ($banner->{banner_type} eq 'image_ad') {
                    push @create_imagead_banners, $banner;
                } else {
                    confess('unknown banner type ' . ($banner->{banner_type} || '""')); 
                }
            }
        }
        
        $group->{banners} = [];
        my %update_create_banners = (
            create => [
                { banners => \@create_desktop_banners, create_sub => \&create_desktop_banners, opt => hash_cut \%params, qw/ignore_callouts/ },
                { banners => \@create_mobile_banners,  create_sub => \&create_mobile_banners,  opt => {} },
                { banners => \@create_imagead_banners, create_sub => \&create_imagead_banners, opt => {} },
            ],
            update => [
                { banners => \@update_desktop_banners, update_sub => \&update_desktop_banners },
                { banners => \@update_mobile_banners,  update_sub => \&update_mobile_banners },
                { banners => \@update_imagead_banners, update_sub => \&update_imagead_banners },
            ],
        );
        for my $create (@{$update_create_banners{create}}) {
            next unless @{$create->{banners}};
            push @{$group->{banners}}, map { $_->{is_new} = 1; $_ } @{$create->{create_sub}->($create->{banners}, $uid, $create->{opt})};
        }
        
        my $update_options = hash_cut \%params, qw/ignore_vcard ignore_sitelinks ignore_image geo_camp_vcard moderate_declined ignore_callouts/;
        for my $update (@{$update_create_banners{update}}) {
            next unless @{$update->{banners}};
            push @{$group->{banners}}, @{$update->{update_sub}->($update->{banners}, $uid, $update_options)};
        }

        $group->{statusModerate} = 'Ready' if $group->{statusModerate} eq 'New' && any {$_->{statusModerate} eq 'Ready'} @{$group->{banners}};
        
        # логируем тип созданных баннеров
        for my $row (grep {$_->{is_new}} @{$group->{banners}}) {
            log_cmd({
                cmd => '_new_banner',
                uid => $campaign->{uid},
                UID => $options{UID},
                cid => $campaign->{cid},
                pid => $group->{pid},
                bid => $row->{bid},
                banner_type => $row->{type},
                where_from => $options{where_from},
            });
        }
        
        # adgroup: до момента полного перехода на группы
        unless($group->{bid}) {
            my $banner = first {$_->{bid}} @{ $group->{banners} };
            $group->{bid} ||= $banner->{bid};
            do_update_table(PPC(pid => $group->{pid}), 'phrases',
                { bid => $group->{bid}, LastChange__dont_quote => 'LastChange' },
                where => {pid => $group->{pid}}
            );
        }
    }
    
    # adgroup: до момента полного перехода на группы
    $group->{bid} or warn "No bid found";

    if (!$options{ignore_retargetings} && defined $group->{retargetings}) {
        my $new_banner = {
            cid => $campaign->{cid},
            pid => $group->{pid},
            bid => $group->{bid},
            currency => $group->{currency}
        };
        my @retargeting_condition_fields = qw(ret_id bid ret_cond_id price_context autobudgetPriority is_suspended pid);
        @{$group->{retargetings}} = map { hash_cut $_, @retargeting_condition_fields } @{$group->{retargetings}};
        $new_banner->{retargetings} = $group->{retargetings};
        Retargeting::update_group_retargetings($new_banner);
    }

    _update_group_tags($group) unless $options{ignore_tags};

    save_group_params($group);

    if (!$options{ignore_hierarchical_multipliers} && save_hierarchical_multipliers($group->{cid}, $group->{pid}, $group->{hierarchical_multipliers})) {
        send_groups_to_bs(by_pid => $group->{pid}, only => ['phrases']);
    }

    unless ($options{pass_phrases}) {
        my $indicators;
        my ($need_moderate_phrases, $changes) = check_moderate_phrases($group, $old_group);

        fill_fields_if_copy_group_before_save($group);
        ($group->{norm_phrases}, $indicators) = Models::Phrase::save_phrases($campaign, $group, $old_group, %params, copy_ctr => $group->{is_new}, is_different_places => $is_different_places);

        $need_moderate_phrases = 1 if $indicators->{added_new_phrases};

        if ($options{moderate_declined} && $old_group->{statusModerate} eq 'No') {
            $need_moderate_phrases = 1;
        }

        if (!$group->{is_new}) {
            if ($need_moderate_phrases && $need_moderate_phrases == 1) {
                my $group_status_moderate = ($campaign->{statusEmpty}||'No') ne 'Yes' && $group->{statusModerate} ne "New" ? "Ready" : "New";
                do_update_table(PPC(pid => $group->{pid}), 'phrases', {
                        statusModerate => $group_status_moderate,
                        statusBsSynced => 'No',
                        statusPostModerate__dont_quote => "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')",
                        statusShowsForecast => 'New',
                    }, where => { pid => $group->{pid} });
                
                clear_banners_moderate_flags(get_bids(pid => $group->{pid}));
                my $template_bids = [];
                if ($group_status_moderate eq 'Ready') {
                    $template_bids = adgroups_template_banners([$group->{pid}], non_draft => 1, non_archived => 1)->{$group->{pid}};
                }
                $template_bids = [map { sql_quote($_) } @$template_bids];
                # параметр активно используется в API для определения наличия изменений
                do_update_table(PPC(pid => $group->{pid}), 'banners', {
                    (@$template_bids ? (
                        statusModerate__dont_quote => sprintf("IF(bid IN (%s), 'Ready', statusModerate)", join ",", @$template_bids),  
                        statusBsSynced__dont_quote => sprintf("IF(bid IN (%s), 'No', statusBsSynced)", join ",", @$template_bids),
                        statusPostModerate__dont_quote => "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')",
                    ) : ()),
                    LastChange__dont_quote => 'NOW()'
                }, where => {pid => $group->{pid}});
                Direct::Banners::delete_minus_geo(pid => $group->{pid}, check_status => 1);
            } elsif (
                $indicators->{delete_params} ||
                $indicators->{changed_suspend_state} ||
                $indicators->{params_updated} ||
                $need_moderate_phrases # need_moderate_phrases тут должен быть равен "2"
            ) {
                do_update_table(PPC(pid => $group->{pid}), 'phrases', { 
                        statusBsSynced => 'No',
                        statusShowsForecast => 'New',
                        LastChange__dont_quote => 'LastChange',
                    }, where => { pid => $group->{pid} }
                );
            }
        }
    }

    if (! $options{ignore_minus_words}) {
        MinusWords::save_group_minus_words($group->{pid}, $group->{minus_words});
    }
    
    # defined $old_group->{geo} - значит обновляем группу иначе гео уже в БД
    if (defined $old_group->{geo} && $old_group->{geo} ne $group->{geo}) {
        _update_group_geo($group, $old_group, $uid, $translocal_opt);
        schedule_forecast($campaign->{cid});
    }


    return $group;
}

sub _update_group_tags {
    my $group = shift;
    
    if ($group->{tags} && @{$group->{tags}}) {
        my %tag_name2tag_id;
        my %tag_id2exists;
        my $all_campaign_tags = get_all_campaign_tags($group->{cid});
        for my $row (@$all_campaign_tags) {
            $tag_name2tag_id{ Tag::normalize_name($row->{value}) } = $row->{tag_id};
            $tag_id2exists{$row->{tag_id}} = 1;
        }
        my (@tags_to_add, @tag_ids);
        for my $tag (@{$group->{tags}}) {
            if (defined ($tag->{tag_name})) {
                if (exists $tag_name2tag_id{ Tag::normalize_name($tag->{tag_name}) }) {
                    push @tag_ids, $tag_name2tag_id{ Tag::normalize_name($tag->{tag_name}) };
                } else {
                    push @tags_to_add, $tag->{tag_name};
                }
            } elsif (defined $tag->{tag_id} && $tag_id2exists{$tag->{tag_id}}) {
                push @tag_ids, $tag->{tag_id};
            }
        }
        if (@tags_to_add) {
            @tags_to_add = xuniq { Tag::normalize_name($_) } @tags_to_add;
            my $added_tags = add_campaign_tags($group->{cid}, \@tags_to_add, return_inserted_tag_ids => 1);
            push @tag_ids, map { $_->{tag_id} } @$added_tags;
        }
        my $pid = $group->{pid};
        save_tags_groups([$pid], {map {$_ => [$pid]} @tag_ids}) if @tag_ids;
    } else {
        delete_group_tags($group->{pid});
    }
}

=head2 _update_group_geo

Вызывается только если значение в БД надо действительно
обновить. Поэтому phrases.LastChange не трогаем, оно само обновится.

=cut
sub _update_group_geo($$$$) {
    my ($new_group, $old_group, $uid, $translocal_opt) = @_;

    my $geo;
    $geo = refine_geoid($new_group->{geo}, \my $geoflag, $translocal_opt);
    do_update_table(PPC(pid => $new_group->{pid}), 'phrases',
        {geo => $geo, statusBsSynced => 'No', statusShowsForecast => 'New'},
        where => {pid => $new_group->{pid}});
    do_update_table(PPC(pid => $new_group->{pid}), 'banners',
        {geoflag => $geoflag, opts__smod => {geoflag => $geoflag}, statusBsSynced => 'No', LastChange__dont_quote => 'LastChange'},
        where => {pid => $new_group->{pid}});        

    # TODO adgroup: переделать нотификацию на группу к полной поддержке в интерфейсе
    my $bid = get_one_field_sql(PPC(pid => $new_group->{pid}), 'SELECT bid FROM phrases WHERE pid = ?', $new_group->{pid});  
    mail_notification('banner', 'b_geo', $bid, $old_group->{geo}, $new_group->{geo}, $uid);    
}

=head2 validate_group($group, $options)

    Валидация группы объявлений

    Параметры:    
        $group {
            currency => валюта кампании,
            strategy => стратегия на кампанию(хеш см. Campaign::campaign_strategy)
            adgroup_type => '', тип баннера (поддерживаются текстовые и группы рекламы мобильных приложений)
            banners => [],
            phrases => [],
            retargetings => []             
        }
        
        $options {
            exists_banners_type => обязательный параметр при обновлении баннеров (описанием см. validate_banner)
            skip_changing_banner_type => пропустить проверку на невозможность смены типа баннера, но непосредственно тип баннера проверяться будет
            exists_ret_conds => обязательный параметр - существующие условия ретаргетинга у пользователя {ret_cond_id1 => 1, ret_cond_id2 => 1} 
            exists_group_retargetings => обязательный параметр - существующие ретаргетинги у пользователя {ret_id1 => 1, ret_id2 => 1}
            ClientID => 
            login_rights => {}
            for_xls => 1|0  1 - проверка используется в xls            
            skip_price_check - не проверять цены на условия показа (например для первого шага редактирвания)
            check_href_available => 1|0 - проверять доступность ссылки баннеров (выполняет http запрос),
                по умолчанию check_href_available == 0, при check_href_available == 1 должен быть передан login_rights
            no_phrase_brackets => 1|0 1 - не проверять ошибки использования скобок () во фразах, скобки считаем недопустимыми
            link_errors_to_phrases => 0|1 1 - не добавлять ошибки, привязанные к фразе, в общий список, добавлять ошибки в массив errors внутри объекта с конкретной фразой, 
                                              к которой относится ошибка
        }
    
    Результат:
        Хеш с результатом валидации, ключи хеша - имена полей группы объявлений
        {
            common => [] - массив текстовых ошибок
            banners => 
            phrases =>
            retargetings =>
            group_name =>  
            hierarchical_multipliers
        }

=cut

sub validate_group {
    
    my ($group, $options) = @_;

    croak "Cannot validate dynamic adgroup"
        if ($group->{adgroup_type} // '') eq 'dynamic' || defined $group->{main_domain} || $group->{main_domain_id};

    my $group_errors = {};
    my $banners_have_errors = 0;

    my $translocal_opt = $options->{is_api} ? {tree => 'api'} : {ClientID => $options->{ClientID}};

    # Отдельно массово получаем доступные организации Справочника
    my $permalinks = [uniq map { $_->{permalink} } grep { $_->{permalink} } @{$group->{banners}}];
    my $valid_permalinks = {};
    if (@$permalinks && $options->{is_tycoon_organizations_feature_enabled}) {
        my $chief_uid = Rbac::get_chief(ClientID => $options->{ClientID});
        my $access_by_uid = Direct::BannersPermalinks::check_organizations_access([ $chief_uid ], $permalinks) // {};
        $valid_permalinks = { map { $_ => 1 } @{$access_by_uid->{$chief_uid} // []} };
    }

    foreach my $banner (@{$group->{banners}}) {

        my $banner_errors = Models::Banner::validate_banner($banner, $group,
            { i_know_href_params                        => 1,
                use_banner_with_flags                   => 1,
                ClientID                                => $options->{ClientID},
                login_rights                            => $options->{login_rights},
                check_href_available                    => $options->{check_href_available},
                skip_contactinfo                        => $options->{for_xls} || $options->{is_light}, # для xls и редактирования текстов не проверяем КИ в баннерах
                exists_banners_type                     => $options->{exists_banners_type},
                skip_imagead                            => $options->{for_xls},
                skip_changing_banner_type               => $options->{skip_changing_banner_type},
                valid_permalinks                        => $valid_permalinks,
            });
        # делаем еще один уровень вложенности ошибок, так как баннер - это сложная структура, там ошибки по полям,
        # а также могут быть ошибки в еще несозданных баннерах, и их не понятно как идентифицировать.
        if (%$banner_errors) {
            $banner->{errors} = $banner_errors;
            $banners_have_errors = 1;
        }
    }

    my $geo_error;
    $geo_error = iget('Не заданы регионы показа') unless defined $group->{geo} && $group->{geo} gt '';
    if (!$geo_error && !$banners_have_errors) {
        $geo_error = Models::Banner::check_geo_restrictions(
            $group->{banners}, undef, %$translocal_opt,
            geo => $group->{geo},
            pid => $group->{pid},
            cid => $group->{cid},
            get_just_error => 1,
        );
        $banners_have_errors = 1  if $geo_error; 
    }

    my $phrases_errors = [];
    my $relevance_matches_errors = [];
    my $retargetings_errors=[];
    if (! $options->{is_light}) {
        # По умолчанию валидация для старых фраз не проверяет длину. Но в некоторых случаях это требуется.
        # Поэтому в этих некоторых случаях проверяем длину отдельно и тут учитываем это.
        push @$phrases_errors, iget('Превышено максимальное количество ключевых фраз')
            if !$options->{no_group_oversized_error} && ($group->{is_group_oversized} || is_group_oversized($group, client_id => $options->{ClientID}));
        push @$phrases_errors, @{Models::Phrase::validate_phrases(
                                    $group->{phrases},
                                    {for_xls => $options->{for_xls}, no_phrase_brackets => $options->{no_phrase_brackets},
                                     link_errors_to_phrases => $options->{link_errors_to_phrases}}
                                 ) || []
                                };
 
        foreach my $phrase ($options->{skip_price_check} ? () : @{$group->{phrases}}) {
            if ($group->{strategy}->{is_autobudget}) {
                my $priority_err = validate_autobudget_priority($phrase->{autobudgetPriority});
                push @{$options->{link_errors_to_phrases} ? $phrase->{errors} : $phrases_errors}, $priority_err if $priority_err;
            } else {
                unless ($group->{strategy}->{is_search_stop}) {
                    my $price_err = validate_phrase_price($phrase->{price}, $group->{currency});            
                    push @{$options->{link_errors_to_phrases} ? $phrase->{errors} : $phrases_errors}, $price_err if $price_err;   
                }
                if ($group->{strategy}->{name} eq 'different_places' && !$group->{strategy}->{is_net_stop}) {
                    my $price_err = validate_phrase_price($phrase->{price_context}, $group->{currency});            
                    push @{$options->{link_errors_to_phrases} ? $phrase->{errors} : $phrases_errors}, $price_err if $price_err;
                }
            }
        }

        foreach my $relevance_match ($options->{skip_price_check} ? () : @{$group->{relevance_match}}) {
            if ($group->{strategy}->{is_autobudget}) {
                my $priority_err = validate_autobudget_priority($relevance_match->{autobudgetPriority});
                push @$relevance_matches_errors, $priority_err if $priority_err;
            } else {
                unless ($group->{strategy}->{is_search_stop}) {
                    my $price_err = validate_phrase_price($relevance_match->{price}, $group->{currency});
                    push @$relevance_matches_errors, $price_err if $price_err;
                }
            }
        }
        if (scalar(@{$group->{relevance_match} // []}) > 1) {
            push @$relevance_matches_errors, iget('Группа объявлений может содержать не более одного автотаргетинга');
        }

        my ($ret_conds, $exists_retargetings) = ($options->{exists_ret_conds}, $options->{exists_group_retargetings});
        foreach my $retargeting (@{$group->{retargetings}||[]}) {
            my $ret_cond = $ret_conds->{$retargeting->{ret_cond_id} // 0};
            if (!is_valid_int($retargeting->{ret_cond_id}) || !$ret_cond) {
                push @$retargetings_errors, iget('Несуществующее условие подбора аудитории');
            } elsif ($ret_cond->{properties} =~ /\bnegative\b/) {
                push @$retargetings_errors, iget("Условие подбора аудитории \"%s\" можно использовать только для корректировки ставок", $ret_cond->{condition_name});
            }
            if ($retargeting->{ret_id} && !(is_valid_int($retargeting->{ret_id}) && $exists_retargetings->{$retargeting->{ret_id}})) {
                push @$retargetings_errors, iget('Несуществующий id условия подбора аудитории');
            }
            
            next if $options->{skip_price_check};
            
            if ($group->{strategy}->{is_autobudget}) {
                my $priority_err = validate_autobudget_priority($retargeting->{autobudgetPriority});
                push @$retargetings_errors, $priority_err if $priority_err;
            } else {
                my $price_err = validate_phrase_price($retargeting->{price_context}, $group->{currency});            
                push @$retargetings_errors, $price_err if $price_err;
            }                        
        }

    }
    
    (my $name = $group->{group_name} || '') =~ s/(^\s+)|(\s+$)//g;
    my $group_name_error;
    $group_name_error = iget('Необходимо задать название группы') unless $name; 

    my $common_errors = [];
    push @$common_errors, iget('К группе необходимо добавить баннеры') unless @{$group->{banners}};
    
    if ($group->{adgroup_type} eq 'mobile_content') {
        if (!$group->{device_type_targeting} || any { !/^phone|tablet$/ } @{$group->{device_type_targeting}}) {
            $group_errors->{device_type_targeting} = [iget('Тип устройства для таргетинга задан некорректно')];
        }
        if (!$group->{network_targeting} || any { !/^wifi|cell$/ } @{$group->{network_targeting}}) {
            $group_errors->{network_targeting} = [iget('Тип связи для таргетинга задан некорректно')];
        } 
        my $group_vr = Direct::Validation::AdGroupsMobileContent::validate_mobile_adgroups([
            Direct::Model::AdGroupMobileContent->new(
                adgroup_name => $group->{group_name},
                geo => $group->{geo},
                client_id => $options->{ClientID},
                # баннеры проверяются выше
                banners => [],
                campaign_id => $group->{cid},
                map { $_ => $group->{$_} } grep { defined $group->{$_} } qw/store_content_href device_type_targeting network_targeting/
            )
        ])->get_objects_results->[0];
        unless ($group_vr->is_valid) {
            for my $field (qw/store_content_href device_type_targeting network_targeting/) {
                next if !$group_vr->get_field_result($field) || $group_errors->{$field}; 
                $group_errors->{$field} = $group_vr->get_field_result($field)->get_error_descriptions;
            }
        }


    }

    my $banners_limit_error = check_add_client_creatives_limits({
                                   pid => $group->{pid},
                                   new_creatives => scalar(grep {!$_->{bid}} @{$group->{banners}}) 
                              });

    push @$common_errors, $banners_limit_error if $banners_limit_error;

    my $camp_type = get_camp_type(cid => $group->{cid});
    my $multipliers_vr = validate_hierarchical_multipliers(
        $camp_type,
        $options->{ClientID},
        $group->{hierarchical_multipliers},
        adgroup_type => $group->{adgroup_type},
    );
    if (!$multipliers_vr->is_valid) {
        $group_errors->{hierarchical_multipliers} = [map { $_->text  } @{$multipliers_vr->get_errors}];
    }

    if ($options->{for_xls} && (my @video_creative_ids = map { $_->{video_resources}->{id} || () } @{$group->{banners}})) {
        # для xls номера видео-дополнений должны существовать
        # для cmd - не всегда
        my $creative_exists = Direct::Creatives::Tools->get_existing($options->{ClientID}, \@video_creative_ids, 'video_addition');
        for my $banner (@{$group->{banners}}) {
            if ($banner->{video_resources}->{id} && !$creative_exists->{$banner->{video_resources}->{id}}) {
                my %error = ( video_addition => [ iget("Видеодополнение №-%s не найдено", $banner->{video_resources}->{id}) ] );
                $banner->{errors} //= {};
                hash_merge $banner->{errors}, \%error;
                $banners_have_errors = 1;
            }
        }
    }

    $group_errors->{group_name} = [$group_name_error] if $group_name_error;
    $group_errors->{banners} = $banners_have_errors if $banners_have_errors;
    $group_errors->{phrases} = $phrases_errors if @$phrases_errors;
    $group_errors->{relevance_match} = $relevance_matches_errors if @$relevance_matches_errors;
    $group_errors->{retargetings} = $retargetings_errors if @$retargetings_errors;
    $group_errors->{common} = $common_errors if @$common_errors;
    $group_errors->{geo} = [$geo_error] if $geo_error;

    return $group_errors;
}

=head2 get_urls($group)

    Собирает из баннеров ссылки (href) и параметры (Param1, Param2) первой фразы баннера
    
    Возвращает
    {
        href => {param1 => , param2 => }
    }

=cut

    sub get_urls {
        
        my ($groups, %opt) = @_;
        
        my $extra_fields = $opt{with_extra_fields} // [];
        my %urls;
        foreach my $group (@$groups) {
            my $params = {};
            if ($group->{phrases} && @{$group->{phrases}} > 0) {
                my $phrase =  $group->{phrases}->[0];    
                $params = hash_cut $phrase, map {lc} @Models::Phrase::BIDS_HREF_PARAMS;
            }
            foreach my $banner (@{$group->{banners}}) {
                next unless $banner->{href};
                $urls{$banner->{href}} = {%$params};
                $urls{$banner->{href}}->{$_} = $banner->{$_} foreach @$extra_fields;
            }
    }
    
    return \%urls;
}


=head2 is_group_empty

    Пустая группа, это группа без баннеров.
    Используется для определения надо ли удалять группу, в случае, если в ней не осталось баннеров.
    Возвращает
        true - если в группе есть баннеры
        false - если их нет.

=cut
sub is_group_empty {
    
    my $pid = shift;

    return not get_one_field_sql(PPC(pid => $pid), 'SELECT pid FROM banners WHERE pid = ? LIMIT 1', $pid); 
}

=head2 is_group_cpm_price_default

    Используется для определения надо ли удалять группу.
    Дефолтную cpm_price группу нельзя удалить
    Возвращает
        true - если группа дефольтная
        false - если нет

=cut
sub is_group_cpm_price_default {

    my $pid = shift;
    my $res = get_one_field_sql(PPC(pid => $pid), 'SELECT pid FROM phrases
    join adgroup_priority using(pid)
    WHERE pid = ? and adgroup_priority.priority = ?', $pid, $Settings::DEFAULT_CPM_PRICE_ADGROUP_PRIORITY);
    return defined $res;
}

=head2 delete_groups ($groups, %options)

    Удалить группы из БД.
    На входе:
        $groups — ARRAYREF групп
    На выходе:
        error в случае, если была обнаружена ошибка, и undef в противоположном случае.

=cut

sub delete_groups {
    my ($groups) = @_;
    my $error;
    my (%pids_to_delete_by_cid, %pid_cid);
    $groups = ref $groups eq 'ARRAY' ? $groups : [$groups];
    foreach my $group (@$groups) {

        my $banners = Models::Banner::get_banners_for_delete({pid => $group->{pid}});
        if (@{$group->{banners} || []}) {
            my %group_banners = map {$_->{bid} => 1} @{$group->{banners}};
            $banners = [grep {$group_banners{$_->{bid}}} @$banners];
        }
        my ($success_count, $group_error) = Models::Banner::delete_banners($group->{cid}, $banners);

        # обновляем статус у кампании, если было что удалить
        Models::CampaignOperations::db_update_campaign_statusModerate($group->{cid}) if $success_count;

        $error ||= $group_error;
        # Удалять группа только если нет ошибок и если группа пустая.
        if (!$group_error && is_group_empty($group->{pid}) && !is_group_cpm_price_default($group->{pid})) {
            push @{$pids_to_delete_by_cid{$group->{cid}}}, $group->{pid};
            $pid_cid{$group->{pid}} = $group->{cid};
        }
    }

    my %existing_grouped_pids;
    foreach_shard pid => [map { @$_ } values %pids_to_delete_by_cid], sub {
        my ($shard, $pids_chunk) = @_;
        foreach my $pid (@$pids_chunk) {
            my $cid = $pid_cid{$pid};
            push @{$existing_grouped_pids{$cid}}, $pid;
        }

        Tag::delete_group_tags($pids_chunk);

        my $ids_cids = get_all_sql(PPC(shard => $shard), [
            "SELECT id, cid FROM bids", where => {pid => $pids_chunk},
            "UNION SELECT id, cid FROM bids_arc", where => {cid => [map { $pid_cid{$_} } @$pids_chunk], pid => $pids_chunk}]);

        foreach my $chunk (chunks($ids_cids, 1000)) {
            my $cids_chunk = [map { $_->{"cid"} } @$chunk];
            my $ids_chunk = [map { $_->{"id"} } @$chunk];
            do_delete_from_table(PPC(shard => $shard), 'bids_href_params', where => {cid => $cids_chunk, id => $ids_chunk});
        }

        do_delete_from_table(PPC(shard => $shard), 'bids_arc', where => {cid => [map { $pid_cid{$_} } @$pids_chunk], pid => $pids_chunk});
        foreach my $table (qw/bids bids_base bids_retargeting bids_dynamic dynamic_conditions bids_performance adgroup_bs_tags
                            group_params adgroups_mobile_content adgroups_cpm_banner adgroups_cpm_video adgroup_project_params adgroup_page_targets adgroups_minus_words adgroup_priority
                            video_segment_goals adgroups_dynamic adgroups_performance adgroups_text adgroups_internal adgroup_additional_targetings phrases/) {
            # !! НЕ ЗАБУДЬ ДОБАВИТЬ ТАБЛИЦУ В JAVA
            # https://a.yandex-team.ru/arc_vcs/direct/core/src/main/java/ru/yandex/direct/core/entity/adgroup/repository/AdGroupRepository.java?#L2341
            do_delete_from_table(PPC(shard => $shard), $table, where => {pid => $pids_chunk});
        }
        do_delete_from_table(PPC(shard => $shard), 'mod_object_version', where => {obj_id => $pids_chunk, obj_type => 'phrases'});
        do_delete_from_table(PPC(shard => $shard), 'mod_reasons', where => {id => $pids_chunk, type => 'phrases'});
    };

    foreach my $cid (keys %existing_grouped_pids) {
        delete_camp_group_hierarchical_multipliers($cid, $existing_grouped_pids{$cid});
    }

    return $error ? $error : undef;

}

=head2 get_auto_adgroup_name(group, %lang_option)

    По переданному объекту группы - генерит для нее название. По переданным дополнительным опциям может
    выбирать язык для этого названия.
    
    На вход:
        group - группа
        options:
            lang
            uid


    my $nm = get_auto_adgroup_name({..., pid => XXX}, lang => 'ru'); # Явно указанный язык
    my $nm = get_auto_adgroup_name({..., pid => XXX}, uid => 'ru');  # Загружает (и кэширует) язык уведомлений пользователя из БД
    my $nm = get_auto_adgroup_name({..., pid => XXX}, lang => 'ru'); # На языке по умолчанию - английском

=cut
{

    my %uid2lang_cache;

sub get_auto_adgroup_name {
    my ($group, %opts) = @_;

    my $lang = "unknown";
    if ($opts{lang}) {
        $lang = $opts{lang}
    } elsif ($opts{uid}) {
        unless (exists $uid2lang_cache{$opts{uid}}) {
            $uid2lang_cache{$opts{uid}} = User::get_user_data($opts{uid}, ['lang'])->{lang};
        }
        $lang = $uid2lang_cache{$opts{uid}};
    }
    my $pid = $group->{pid};
    if ($pid) {
        # Данный фрагмент работает для
        return $lang eq 'ru' ? "Группа №$pid" : "AdGroup №$pid";
    } else {
        return return $lang eq 'ru' ? "Новая группа объявлений" : "New AdGroup";
    }
}

}
=head2 save_flags($groups, $flag, $value)

    Установить и сохранить новое значение флага на все баннеры группы
    У баннеров из группы установит новое значение флагов (banner.hash_flags)

        $flag - имя флага
        $value - значение флага (== -1 - удалить флаг)
    
    Результат:

        хеш с баннерами, у которых поменялся набор флагов
        {
            bid => {cid => 123, flags => {flag1 => 1, flag2 => 18}}
        }

=cut
sub save_flags {
    
    my ($groups, $flag, $value) = @_;
    
    my %banners;
    for my $group (@$groups) {
        for my $banner (@{$group->{banners}}) {
            my ($new_flags, $is_changed) = BannerFlags::set_flag($banner->{hash_flags}, $flag, $value);
            if ($is_changed) {
                $banners{$banner->{bid}} = {cid => $group->{cid}, flags => $new_flags, pid => $group->{pid}};
                $banner->{hash_flags} = $new_flags;
            }
        }
    }
    
    if (keys %banners) {
        foreach_shard bid => [keys %banners], sub {
            my ($shard, $bids_chunk) = @_;
            my $case = sql_case(bid => {map {($_ => BannerFlags::serialize_banner_flags_hash($banners{$_}->{flags}))} @$bids_chunk});
            my @values_to_insert = map {[$_, $banners{$_}->{pid}, $banners{$_}->{cid}]} @$bids_chunk;

            do_in_transaction {
                do_sql(PPC(shard => $shard), ["UPDATE banners SET flags = $case, statusBsSynced = 'No'", WHERE => {bid => $bids_chunk}]);

                # Создаем/обновляем запись в специальной таблице, чтоб новый транспорт модерации подхватил изменение флагов пользователем.
                my $query = "INSERT INTO banner_user_flags_updates (bid, pid, cid)
                                VALUES %s ON DUPLICATE KEY UPDATE update_time = NOW()";
                do_mass_insert_sql(PPC(shard => $shard), $query, \@values_to_insert);
            }
        };
    }
    return \%banners;
}

=head2 get_groups_params

Выбирает group_params для указаной группы или групп.

    my $group_params_hashed_by_pid = get_groups_params($group_pid || [$group_pid1, $group_pid2, ..]);

=cut
sub get_groups_params {
    my ($groups) = @_;
    $groups = [$groups] unless ref $groups;
    return get_hashes_hash_sql(
        PPC(pid => $groups),
        [
            "select pid, has_phraseid_href, href_params from group_params",
            WHERE => {pid => SHARD_IDS},
        ],
    );
}

=head2 save_group_params($group || [$group,$group,$group,...])

    Расчитать и сохранить дополнительные параметры на группу|группы
    Параметры следующие:
        - has_phraseid_href - группа имеет баннер(ы) с параметром phraseid|phrase_id в урле

=cut
sub save_group_params {
    my $groups = shift;

    my @pids;
    foreach my $group (ref $groups eq 'ARRAY' ? @$groups : $groups) {
        if (any {$_->{href} && Models::Banner::has_phraseid_param($_->{href})} @{$group->{banners}}) {
            push @pids, $group->{pid};
        }
    }

    # цикл не выполнится, если массив @pids - пустой
    foreach_shard pid => \@pids, sub {
        my ($shard, $pids_chunk) = @_;

        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT IGNORE INTO group_params(pid, has_phraseid_href) VALUES %s',
            [map {[$_, 1]} @$pids_chunk],
            {
                dont_quote => 1,
                sleep => 1,
                max_row_for_insert => 5000,
            }
        );
    };
}


=head2 is_completed_groups($pid || $pids)

    Возвращает хеш с группами($pid), которые готовы к показам (группа полная)
    (есть хотя бы один баннер и условие показа (фраза и/или ретаргетинг))
    Фразы в bids у архивных кампаний это штатная ситуация:
    * если кампания не архивна, то её фразы надо искать только в bids;
    * если кампания архивна, то её фразы надо искать как в bids_arc, так и в bids.

    Возвращает
        {
            pid => 1,
            pid => 1,
        }

=cut

sub is_completed_groups {

    my $pids = shift;

    $pids = [$pids] unless ref $pids eq 'ARRAY';
    my $full_groups = get_one_column_sql(PPC(pid => $pids), [
        "SELECT p.pid
        FROM phrases p
             JOIN campaigns c on c.cid = p.cid",
        WHERE => {'p.pid' => SHARD_IDS},
        "AND EXISTS (SELECT 1 FROM banners b WHERE b.pid = p.pid)
        AND (
            p.adgroup_type IN ('base', 'mobile_content', 'mcbanner','cpm_banner', 'content_promotion_video', 'content_promotion') AND (
               c.archived = 'No' AND EXISTS (SELECT 1 FROM bids bi WHERE bi.pid = p.pid AND bi.is_suspended=0)
               OR c.archived = 'Yes' AND (
                      EXISTS (SELECT 1 FROM bids_arc bi WHERE bi.pid = p.pid AND bi.cid = p.cid AND bi.is_suspended=0)
                   OR EXISTS (SELECT 1 FROM bids bi WHERE bi.pid = p.pid AND bi.cid = p.cid AND bi.is_suspended=0)
               )
            )
            OR
            p.adgroup_type IN ('base', 'mobile_content', 'cpm_banner', 'cpm_video', 'cpm_outdoor', 'content_promotion_video', 'cpm_indoor', 'cpm_audio', 'cpm_geoproduct', 'cpm_geo_pin', 'content_promotion', 'cpm_geo_pin') AND (
               EXISTS (SELECT 1 FROM bids_base b WHERE b.pid = p.pid AND FIND_IN_SET('suspended', b.opts)=0 AND FIND_IN_SET('deleted', b.opts)=0 AND b.bid_type <> 'keyword')
               OR EXISTS (SELECT 1 FROM bids_retargeting b WHERE b.pid = p.pid AND b.is_suspended=0)
            ) OR
            p.adgroup_type = 'dynamic' AND EXISTS (SELECT 1 FROM bids_dynamic bd WHERE bd.pid = p.pid AND NOT FIND_IN_SET('suspended', bd.opts))
            OR
            p.adgroup_type = 'performance' AND EXISTS (SELECT 1 FROM bids_performance bp WHERE bp.pid = p.pid AND bp.is_suspended=0 AND bp.is_deleted=0)
            OR
            p.adgroup_type = 'cpm_yndx_frontpage' AND EXISTS (SELECT 1 FROM bids_retargeting b WHERE b.pid = p.pid AND b.is_suspended=0)
            OR
            p.adgroup_type = 'internal'
       )"]);
    
    return {map {$_ => 1} @$full_groups};
}


=head2 is_completed_group($pid)

    Готова ли группа к показам?

=cut

sub is_completed_group {
    my $pid = shift;
    return is_completed_groups($pid)->{$pid};
}


=head2 has_only_relevance_match

    Условием показа для группы объявлений являются только
    автотаргетинг (нет фраз и нет условий ретаргетинга)

=cut
sub has_only_relevance_match {
    my $group = shift;
    return $group->{relevance_match} && @{$group->{relevance_match}}
                && !@{$group->{phrases}} && !($group->{retargetings} && @{$group->{retargetings}});
}

=head2 update_phrases_shows_forecast($groups, %opts)

    update_phrases_shows_forecast($groups, %opts)

    Обновляет прогнозы показов (showsForecast) для фраз. Прогнозы запрашиваются через advq_.
    Не обновляются прогнозы для груп со статусом 'Archive' и 'Processed'.
    

    $groups - массив групп в том виде, который возвращает Models::AdGroup::get_groups:
    %opts   - параметры для advq_get_phrases_shows_multi, если вдруг потребуются


    Результат - актуализация значения showsForecast в $groups и в базе.

=cut

sub update_phrases_shows_forecast {
    my ($groups, %opts) = @_;

    my @update_groups = grep { $_->{statusShowsForecast} ne 'Processed' &&
                               $_->{statusShowsForecast} ne 'Archive'
                        } @$groups;

    return unless @update_groups;

    foreach_shard pid => \@update_groups, by => sub { $_->{pid} // $_->{adgroup_id} }, with_undef_shard => 1, sub {
        my ($shard, $groups) = (shift, shift);

        # без шарда могут приходить новые объекты [pid=0, pid=undef]
        if ($shard) {
            my @pids = map {$_->{pid}} @$groups;

            do_sql( PPC(shard => $shard), [
                q[ UPDATE phrases SET statusShowsForecast = 'Sending',
                                      LastChange = LastChange
                ],
                WHERE => {
                    pid => \@pids,
                }
            ]);
        }

        my $advq_res = advq_get_phrases_shows_multi($groups, %opts);
        if (($advq_res // '') eq 'real') {
            for my $group (@$groups) {
                $group->{statusShowsForecast} = 'Processed';
            }
        }

        if ($shard) {
            foreach my $ph_chunk ( chunks [ map { @{$_->{phrases}} } @$groups ], 100 ) {
                my (%id_shows, %id_phrases);
                foreach my $ph ( @$ph_chunk ) {
                    $id_shows  {$ph->{id}} = $ph->{showsForecast};
                    $id_phrases{$ph->{id}} = $ph->{phrase};
                }

                my $case_show   = sql_case(id => \%id_shows);
                my $case_phrase = sql_case(id => \%id_phrases);

                do_in_transaction {
                    my $bs_status_by_id = get_hash_sql(PPC(shard => $shard), [
                            "SELECT id, IF(showsForecast = $case_show, b.statusBsSynced, 'No')
                            FROM bids b JOIN phrases p ON b.pid = p.pid",
                            WHERE => {
                                id => [ keys %id_shows ],
                                'p.statusShowsForecast' => 'Sending',
                                'b.phrase__dont_quote' => $case_phrase,
                            },
                            'FOR UPDATE',
                        ]);
                    my @ids_to_update = keys %$bs_status_by_id;
                    return if !@ids_to_update;

                    do_update_table(PPC(shard => $shard),
                        bids => {
                            showsForecast__dont_quote  => $case_show,
                            statusBsSynced__dont_quote => sql_case(id => $bs_status_by_id),
                            modtime__dont_quote => 'modtime',
                        },
                        where => { id => \@ids_to_update },
                    );
                };

                my @ppids = uniq map {$_->{pid}} @$ph_chunk;

                do_sql( PPC(shard => $shard), [
                    q[ UPDATE phrases SET forecastDate = now(),
                                          statusShowsForecast = 'Processed',
                                          LastChange = LastChange ],
                    WHERE => {
                        pid => \@ppids,
                        statusShowsForecast => 'Sending',
                    },
                ]);
            }
        }
    };
}

=head2 update_phrases_shows_forecast_mediaplan

     Версия для медиапланов

=cut

sub update_phrases_shows_forecast_mediaplan {
    my ($banners, %opts) = @_;

    my @update_banners = grep {$_->{statusShowsForecast} ne 'Processed'} @$banners
        or return;

    foreach_shard mediaplan_bid => \@update_banners, by => 'mbid', sub {
        my ($shard, $banners) = (shift, shift);

        my @mbids = map {$_->{mbid}} @$banners;
        do_sql( PPC(shard => $shard), [
            q[ UPDATE mediaplan_banners 
                  SET statusShowsForecast = 'Sending'
                    , timeShowsForecast = NOW()
            ],
            WHERE => {
                mbid => \@mbids,
                statusShowsForecast__ne => 'Sending',
            }
        ]);


        advq_get_phrases_shows_multi(\@update_banners, %opts);


        foreach my $ph_chunk ( chunks [ map { @{$_->{phrases}} } @$banners ], 100 ) {

            my (%id_shows, %id_phrases);
            foreach ( @$ph_chunk ) {
                $id_shows  {$_->{id}} = $_->{showsForecast};
                $id_phrases{$_->{id}} = $_->{phrase};
                $_->{shows} = $_->{showsForecast};
            }

            my $case_show   = sql_case(id => \%id_shows);
            my $case_phrase = sql_case(id => \%id_phrases);

            do_sql( PPC(shard => $shard), 
                [
                    qq[ 
                        UPDATE mediaplan_bids b JOIN mediaplan_banners p ON b.mbid = p.mbid
                           SET b.showsForecast = $case_show
                    ],
                    WHERE => {
                        id => [ keys %id_shows ],
                        'p.statusShowsForecast' => 'Sending',
                        'b.phrase__dont_quote' => $case_phrase,
                    },
                ],
            );

            do_sql( PPC(shard => $shard), [
                q[ UPDATE mediaplan_banners
                      SET timeShowsForecast = NOW()
                        , statusShowsForecast = 'Processed'
                ],
                WHERE => {
                    mbid => \@mbids,
                    statusShowsForecast => 'Sending',
                },
            ]);
        }
    };
}

=head2 
=cut

sub filter_arch_groups {
    my $ids = $_[0];
    my $groups = get_one_column_sql(PPC(pid => $ids), ["SELECT pid FROM banners", 
        WHERE => {'pid' => SHARD_IDS, 'statusArch' => 'No'}, 
        'GROUP BY pid']
    );
    return $groups;
}

=head2 fill_absent_adgroup_fields

    Приходит группа из FORM, она не совсем полна, заполняем ее некоторыми полями из аналогичной группы из БД.
    Полями, которые требуются для отображения на странице.

    $to_adgroup -- данные из FORM
    $from_adgroups -- данные из базы

=cut
sub fill_absent_adgroup_fields {
    my ($to_adgroups, $from_adgroups, $mobile_contents) = @_;
    my $from_adgroup_hash = {map {$_->{pid} => $_} @{$from_adgroups}};
    my @necessary_adgroup_fields = qw/statusModerate adgroup_type group_banners_types is_bs_rarely_loaded/;
    my @necessary_banner_fields = qw/BannerID image_BannerID statusModerate statusActive statusShow statusAutobudgetShow day_budget day_budget_show_mode status ad_type/;
    foreach my $to_adgroup (@$to_adgroups) {

        if (ref $to_adgroup->{mobile_content} eq 'HASH' && $to_adgroup->{mobile_content}->{mobile_content_id}) {
            my $mobile_content_id = $to_adgroup->{mobile_content}->{mobile_content_id};
            if ($mobile_contents->{$mobile_content_id}) {
                $to_adgroup->{mobile_content} = $mobile_contents->{$mobile_content_id}->to_template_hash;
            } else {
                $to_adgroup->{mobile_content} = {}
            }
        }

        unless ($to_adgroup->{pid}) {
            my %types_count;
            for my $banner (@{$to_adgroup->{banners}}) {
                $types_count{$banner->{ad_type}}++;
            }
            $to_adgroup->{group_banners_types} = [ map { { banner_type => $_, count => $types_count{$_} } } keys %types_count ];
            $to_adgroup->{is_bs_rarely_loaded} = 0;
            
            # Ничего не копируем, если это новая группа.
            next;
        }

        my $from_adgroup = $from_adgroup_hash->{$to_adgroup->{pid}};
        foreach (@necessary_adgroup_fields) {
            $to_adgroup->{$_} = $from_adgroup->{$_} if defined $from_adgroup->{$_};
        }
        
        
        # объявления
        my $from_adgroup_banners_hash = {map {$_->{bid} => $_} @{$from_adgroup->{banners}}};
        foreach my $banner (@{$to_adgroup->{banners}}) {
            $banner->{adgroup_type} = $to_adgroup->{adgroup_type};
            next unless $banner->{bid};
            foreach (@necessary_banner_fields) {
                my $old_value = $from_adgroup_banners_hash->{$banner->{bid}}->{$_};
                $banner->{$_} = $old_value if defined $old_value;
            }
            $banner->{pstatusModerate} = $to_adgroup->{statusModerate};
        }
    }

}

=head2 calc_context_price($group, %options)

   Проставляем цену для контекста во фразы группы

   входные данные - ссылка на хеш = структура группы

   group:{
       phrases => [
           {
               rank    => 1,
               price   => 12.00,
               price_context   => 0.00,
               id      => 123,
           },
           ...
       ]
   }
   options: ContextPriceCoef => 10,    # коэффициент для цены в РСЯ,
            currency => 'YND_FIXED'|'RUB'|'USD'|...,    # валюта 


=cut

sub calc_context_price {
    my ($group, %options) = @_;

    my $currency = $options{currency} || $group->{currency};
    die 'no currency given' unless $currency;
    for my $phrase (@{$group->{phrases}}){
        next if $phrase->{price_context};
        $phrase->{price_context} = phrase_price_context($phrase->{price}, $options{ContextPriceCoef}, $currency);
    }
}

=head2 get_first_excess_phrase_idx

    Возвращает индекс первой фразы баннера, не помещающейся в ограничение на количество фраз клиента
    Или -1, если все фразы помещаются. Уже сохранённые фразы

    Опции:

    limit
    client_id

    # формат остался с тех пор, когда проверялась суммарная длина фраз

    Принимает ссылку на хеш с данными о баннере и его фразах:
    Если передано что-то, не похожее на баннер, возвращает undef или пустой список в скалярном и списочном контекстах соответственно.

    $banner = {
        [...]
        phrases => [
            [...]
            {
                [...]
                phr => 'phrase text 1',
                [...]
            }
            [...]
        ],
        [...]
    };
    $first_excess_phrase_idx = get_first_excess_phrase_idx($banner, client_id => $client_id);

=cut

sub get_first_excess_phrase_idx {
    my ($group, %O) = @_;

    return unless $group && ref($group) eq 'HASH' && $group->{phrases} && ref $group->{phrases} eq 'ARRAY';

    my $limit = $O{limit};
    $limit ||= get_client_limits($O{client_id})->{keyword_count_limit}  if $O{client_id};
    # ставим дефолтное значение, если передали client_id=0
    $limit ||= $Settings::DEFAULT_KEYWORD_COUNT_LIMIT  if defined $O{client_id};
    croak "Limit is not defined"  if !$limit;

    if (@{$group->{phrases}} > $limit){
        #если просили отдать группу с превышением фораз - отдаем через значение опции
        if ($O{get_affected_group_as} && ref $O{get_affected_group_as}){
            ${$O{get_affected_group_as}} = $group; 
        }
        return $limit;
    }
    return -1;
}

=head2 is_group_oversized

    Возвращает 1, если количество фраз в группе превышает допустимое и 0 в противном случае.
    Принимает ссылку на хеш с данными о группе и его фразах:
    Если передано что-то, не похожее на группу, возвращает undef или пустой список в скалярном и списочном контекстах соответственно.

    Опции (пробрасываются в get_first_excess_phrase_idx):

    limit
    client_id

    $group = {
        [...]
        phrases => [
            [...]
            {
                [...]
                phr => 'phrase text 1',
                [...]
            }
            [...]
        ],
        [...]
    };
    $group_oversized = is_group_oversized($group); # $group_oversized => 1|0|undef

=cut

sub is_group_oversized {
    my ($group, %O) = @_;

    my $excess_idx = get_first_excess_phrase_idx($group, %O);
    if (defined $excess_idx) {
        return ( $excess_idx != -1 ) ? 1 : 0;
    } else {
        return undef;
    }
}

=head2 on_copy_adgroup

    Убирает все идентификационные поля из группы, чтобы ее можно было сохранить как новую. Используется при копировании групп.

=cut
sub on_copy_adgroup {
    my $group = shift;

    delete $group->{$_} foreach (qw/pid status old deleted_bids/);
    if ($group->{is_bs_rarely_loaded}) {
        $_->{nobsdata} = 0 for @{$group->{phrases}};
        $group->{is_bs_rarely_loaded} = 0;
    }
    foreach my $banner (@{$group->{banners}}) {
        delete $banner->{$_} foreach (qw/pid bid BannerID adgroup_id/);
    }
    if (exists $group->{relevance_match}) {
        foreach(@{$group->{relevance_match}}) {
            $_->{bid_id} = undef; # вставляем условия как новые в новую группу
        }
    }
    if (exists $group->{retargetings}) {
        foreach(@{$group->{retargetings}}) {
            $_->{ret_id} = undef; # вставляем условия как новые в новую группу
        }
    }
    if (exists $group->{target_interests}) {
        foreach(@{$group->{target_interests}}) {
            $_->{ret_id} = undef; # вставляем условия как новые в новую группу
        }
    }
    if (exists $group->{dynamic_conditions}) {
        for my $cond (@{$group->{dynamic_conditions}}) {
            $cond->{filter_id} = $cond->{dyn_id};
            delete $cond->{$_} for qw/dyn_id adgroup_id/;
        }
    }
    if (exists $group->{performance_filters}) {
        for my $filter (@{$group->{performance_filters}}) {
            delete $filter->{$_} for qw/perf_filter_id adgroup_id/;
        }
    }
}

=head2 adgroups_template_banners($pids, %filters)

Получить группы и все её шаблонные баннеры
Параметры: 
    $pids - [] номера групп объявлений
    %filters - условия выборки шаблонных баннеров
        non_draft => 1|0 - не выбирать черновики
        non_archived => 1|0 - не выбирать архивные
Результат:
    $adgroups - {pid => [bid1, bid2, bid3]}
    В результате присутсвуют все запрошенные pid, значение - массив [] номеров шаблонных баннеров
    (пустой массив если таких баннеров нет)

=cut

sub adgroups_template_banners {
    
    my ($pids, %filters) = @_;
    
    my %where = (pid => SHARD_IDS);
    $where{statusArch} = $filters{non_archived} ? 'No' : 'Yes';
    $where{statusModerate__ne} = 'New' if $filters{non_draft};
    my $banners = get_all_sql(PPC(pid => $pids), ["SELECT bid, pid, title, title_extension, body, href FROM banners", WHERE => \%where]);
    my %adgroups;
    for my $banner (@$banners) {
       push @{$adgroups{$banner->{pid}}}, $banner->{bid} if is_template_banner($banner);  
    }
    for (@$pids) {
        $adgroups{$_} = [] unless exists $adgroups{$_}; 
    }
    return \%adgroups; 
}

=head2 fill_fields_if_copy_group_before_save

    Если группа новая, но у фраз есть phraseIdHistory, значит это копирование групп
    и надо обновить значения phraseIdHistory

    ! Изменяет входные данные !

    На входе:
        adgroup - хеш-группа с возможно полями is_new, pid, banners, phrases

=cut
sub fill_fields_if_copy_group_before_save {
    my $adgroup = shift;
    # Если группа новая, но у фраз есть phraseIdHistory, значит это копирование групп и надо обновить значения phraseIdHistory
    if ($adgroup->{is_new}) {

        my $new_bid = (defined $adgroup->{bid}) ? $adgroup->{bid} :
                            (defined $adgroup->{banners}) ? Models::AdGroup::get_main_banner($adgroup)->{bid} :
                                    Primitives::get_main_banner_ids_by_pids($adgroup->{pid})->{bid};
        foreach my $ph (@{$adgroup->{phrases} || []}) {
            if ($ph->{phraseIdHistory}) {
                my %phraseIdHistory = BS::History::parse_bids_history($ph->{phraseIdHistory});
                my $old_bid = (keys(%{$phraseIdHistory{banners}}))[0];
                $phraseIdHistory{banners} = {$new_bid => $phraseIdHistory{banners}->{$old_bid}};
                $ph->{phraseIdHistory} = BS::History::serialize_bids_history(\%phraseIdHistory);
            }
        }
    }
}

=head2 fill_video_segment_goals

    Заполняет поле video_goals для каждой группы.

=cut
sub fill_video_segment_goals {
    my $adgroups = shift;

    my @pids = map {$_->{pid}} @$adgroups;
    my $audience_types = get_all_sql(PPC(pid => \@pids), 
                            ["SELECT pid, audience_type FROM video_segment_goals",
                            where=> {pid => SHARD_IDS, is_disabled => 0}]);
    my $audience_types_by_pid = {};
    foreach my $type (@$audience_types) {
        push @{$audience_types_by_pid->{$type->{pid}}}, $type->{audience_type};
    }
    foreach my $adgroup (@$adgroups) {
        $adgroup->{video_goals} = [map {{"type" => $_}} @{$audience_types_by_pid->{$adgroup->{pid}}}];
    }
}

1;
