package Models::AdGroupFilters;

=head1 NAME

    Models::AdGroupFilters

=head1 DESCRIPTION

    Модуль содержит функции для формирования фильтров, понятных функции Models::AdGroup::get_groups.
    Если нужна выборка групп по большому количеству вычисляемых параметров, нужно описать функцию здесь и свести значения фильтров
    до понятных Models::AdGroup::get_groups.

=cut

use Carp;
use List::MoreUtils qw/uniq any/;
use Direct::Modern;
use Settings;
use Yandex::ListUtils qw/xisect/;
use Yandex::DBTools;
use Yandex::DBShards;

use Direct::Model::Creative::Constants;

my %conditions = (
    # active groups
    active => {
        _OR => {
            _AND => [
                'g.statusPostModerate' => 'Yes',
                'b.statusPostModerate' => 'Yes',
                _TEXT => 'IF(b.banner_type IN ("image_ad", "mcbanner", "cpm_banner", "cpc_video", "cpm_geoproduct"), im.statusModerate = "Yes" OR perfb.statusModerate = "Yes", 1)',
                _OR => {
                    _TEXT => q[IFNULL(b.href,'') != ''],
                    'b.phoneflag' => 'Yes',
                    'b.banner_type' => ['dynamic'],
                    'mc.is_available' => 1,
                    'perfc.statusModerate' => 'Yes',
                    _AND => {
                        'perfc.creative_type' => 'performance',
                        _OR => {
                            'perfc.layout_id' => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS(),
                            'perfc.layout_id__is_null' => 1,
                        },
                    },
                    'btl.tl_id__gt' => 0,
                },
                _TEXT => 'IF(b.banner_type IN ("cpm_outdoor"), b.statusModerate = "Yes" AND (mbp.pageStatusModerate = "Yes" or bmg.minus_geo IS NOT NULL), 1)',
                _TEXT => 'IF(g.adgroup_type IN ("cpm_geoproduct") AND btl.tl_id IS NOT NULL, btl.statusModerate = "Yes", 1)'
            ],
            #outdoor баннеры отправляются в БК, когда нет данных по операторам и от БК приходит активный статус. Не смотрим на него для outdoor
            _TEXT => 'IF(b.banner_type IN ("cpm_outdoor"), 0, b.statusActive = "Yes")'
        },
        'b.statusShow' => 'Yes',
    },
    off => {'b.statusShow' => 'No', 'b.statusArch' => 'No'},
    arch => {'b.statusArch' => 'Yes'},
    draft => {
        'b.statusArch' => 'No',
        _OR => {
            _AND => {
                'b.statusModerate' => 'New',
                'b.phoneflag' => 'New', 'b.statusArch' => 'No'
            },
            'b.bid__is_null' => 1,
            'g.statusModerate' => 'New',
        },
    },
    decline => {
        _OR => [
            # для картиночного (imagead) баннера проверяем, что картинка тоже промодерирована
            _TEXT => q/IF(b.banner_type IN ('image_ad', 'mcbanner', 'cpm_banner', 'cpc_video', 'cpm_geoproduct'),
                           IF(ISNULL(perfb.statusModerate),
                               IF(b.statusModerate = 'No', 'No', im.statusModerate),
                               IF(b.statusModerate = 'No', 'No', perfb.statusModerate)
                           ),
                           b.statusModerate
                       ) = 'No'/,
            'g.statusModerate' => 'No',
            'b.phoneflag' =>  'No', 'b.statusSitelinksModerate' => 'No',
            _AND => {
                'perfc.statusModerate' => 'No',
                _OR => {
                    'perfc.creative_type__ne' => 'performance',
                    _AND => {
                        'perfc.layout_id__not_in' => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS(),
                        'perfc.layout_id__is_not_null' => 1,
                    },
                },
            },
            _AND => {
                'bim.statusModerate' => 'No',
                'bim.statusShow' => 'Yes',
            },
            _AND => {
                'b.banner_type' => ['cpm_outdoor'],
                'b.statusModerate' => 'Yes',
                'mbp.pageStatusModerate' => 'No',
            },
            _AND => {
                'g.adgroup_type' => ['cpm_geoproduct'],
                'btl.tl_id__gt' => 0,
                'btl.statusModerate' => 'No'
            }
        ],
        'b.statusArch' => 'No',
        'g.statusModerate__not_in' => ['Ready', 'Sent', 'Sending'],
    },
    wait => {
        _OR => [
            'b.statusModerate' => ['Ready', 'Sent', 'Sending'],
            'g.statusModerate' => ['Ready', 'Sent', 'Sending'],
            _AND => {
                'bim.statusModerate' => ['Ready', 'Sent', 'Sending'],
                'bim.statusShow' => 'Yes',
            },
            _AND => {
                'b.banner_type' => ['cpm_outdoor'],
                'b.statusModerate' => 'Yes',
                'mbp.pageStatusModerate' => ['ready', 'sent'],
            },
            _AND => {
                'perfc.statusModerate' => ['Ready', 'Sent', 'Sending'],
                _OR => {
                    'perfc.creative_type__ne' => 'performance',
                    _AND => {
                        'perfc.layout_id__not_in' => Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS(),
                        'perfc.layout_id__is_not_null' => 1,
                    },
                },
                'b.statusModerate' => 'Yes',
            },
            _AND => {
                _OR => {
                    'b.phoneflag' => ['Ready', 'Sent', 'Sending'],
                    'b.statusSitelinksModerate' => ['Ready', 'Sent', 'Sending'],
                    'im.statusModerate' => ['Ready', 'Sent', 'Sending'],
                    'perfb.statusModerate' => ['Ready', 'Sent', 'Sending'],
                },
                'b.statusModerate__ne' => 'No',
                'g.statusModerate__ne' => 'No'
            },
            _AND => {
                'g.adgroup_type' => ['cpm_geoproduct'],
                'btl.tl_id__gt' => 0,
                'btl.statusModerate' => ['Ready', 'Sent', 'Sending'],
                'b.statusModerate__not_in' => ['New']
            }
        ],
        'b.statusArch' => 'No'
    },
    # для перфоманс баннеров нет постмодерации (соответсвенно в этой выборки их тоже быть не может)
    running_unmoderated => {
        'b.statusActive' => 'Yes',
        'b.statusShow' => 'Yes',
        _TEXT => q[NOT(b.statusPostModerate = 'Yes' AND g.statusPostModerate = 'Yes'
                    AND (IFNULL(b.href,'') != '' OR b.phoneflag = 'Yes' OR b.banner_type IN ('dynamic')))],
        _OR => {
            'b.statusModerate__ne' => 'Yes',
            'g.statusModerate__ne' => 'Yes',
            _AND => {
                _TEXT => q[(IFNULL(b.href,'') = '')],
                'b.phoneflag__ne' => 'Yes',
                'b.banner_type' => ['text'],
            }
        }
    },
    all=>{}, # используется в is_valid_status_filter для проверки валидности фильтра
);
my %join_tables = (
    active => [
        'LEFT JOIN adgroups_mobile_content amc ON amc.pid=g.pid',
        'LEFT JOIN mobile_content mc ON mc.mobile_content_id = amc.mobile_content_id',
        'LEFT JOIN banners_performance perfb ON b.bid = perfb.bid',
        'LEFT JOIN perf_creatives perfc ON perfc.creative_id = perfb.creative_id',
        'LEFT JOIN images im on im.bid = b.bid',
        #данный вложенный запрос делается join-ом т.к. в select-те запрос группировки будет выполнятся каждный раз для одного баннера
        #при переписывании на java лучше заполнять данные по пейджам после основого запроса
        'LEFT JOIN (select i_mbp.bid, IF(SUM(i_mbp.statusModerate = "Yes") > 0, "Yes", IF(SUM(i_mbp.statusModerate = "Ready" or i_mbp.statusModerate = "Sent" or i_mbp.statusModerate = "Maybe") > 0, "Ready", "No")) as pageStatusModerate
         from moderate_banner_pages i_mbp join banners b on i_mbp.bid = b.bid _PAGE_MOD_FILTER_ group by b.bid) mbp on mbp.bid = b.bid',
        'LEFT JOIN banner_turbolandings btl on btl.bid = b.bid',
        'LEFT JOIN banners_minus_geo bmg ON b.bid = bmg.bid and bmg.type = "current"',
    ],
    wait => [
        'LEFT JOIN banners_performance perfb ON b.bid = perfb.bid',
        'LEFT JOIN perf_creatives perfc ON perfc.creative_id = perfb.creative_id',
        'LEFT JOIN images im on im.bid = b.bid',
        'LEFT JOIN banner_images bim ON b.bid = bim.bid',
        'LEFT JOIN banner_turbolandings btl on btl.bid = b.bid',
        'LEFT JOIN (select i_mbp.bid, IF(SUM(i_mbp.statusModerate = "Yes") > 0, "Yes", IF(SUM(i_mbp.statusModerate = "Ready" or i_mbp.statusModerate = "Sent" or i_mbp.statusModerate = "Maybe") > 0, "Ready", "No")) as pageStatusModerate
         from moderate_banner_pages i_mbp join banners b on i_mbp.bid = b.bid _PAGE_MOD_FILTER_ group by b.bid) mbp on mbp.bid = b.bid',
    ],
    decline => [
        'LEFT JOIN banner_images bim ON b.bid = bim.bid',
        'LEFT JOIN banners_performance perfb ON b.bid = perfb.bid',
        'LEFT JOIN perf_creatives perfc ON perfc.creative_id = perfb.creative_id',
        'LEFT JOIN images im on im.bid = b.bid',
        'LEFT JOIN banner_turbolandings btl on btl.bid = b.bid',
        'LEFT JOIN (select i_mbp.bid, IF(SUM(i_mbp.statusModerate = "Yes") > 0, "Yes", IF(SUM(i_mbp.statusModerate = "Ready" or i_mbp.statusModerate = "Sent" or i_mbp.statusModerate = "Maybe") > 0, "Ready", "No")) as pageStatusModerate
         from moderate_banner_pages i_mbp join banners b on i_mbp.bid = b.bid _PAGE_MOD_FILTER_ group by b.bid) mbp on mbp.bid = b.bid',
    ],
);


=head2 get_status_condition (tab, filter)

    Возвращает собранные во едино значения для последующей выборки из БД групп с заданными условиями.
    Условия задаются названиями вкладок на странице: active, off, arch, draft, decline, wait, running_unmoderated
    На входе:
        tabs - название вкладки или массив вкладок
        opts{filter} - фильтр является обязательным параметром. Поддерживается список pid/cid. Пример вызова get_status_condition($conditions->{tab}, filter => {pid => $pid}))
                       Используется для выборки данных во вложенном запросе к таблице moderate_banner_pages
    На выходе:
        хеш с полями where, group_by, fields, tables, которые подходят в дальшенем для подставки в SQL запрос через DBTools

=cut

# TODO hrustyashko@ принимать adgroup_type и не join-ить лишних таблицы

sub get_status_condition {
    my ($tabs, %opts) = @_;
    $tabs = 'active' if !$tabs;
    $tabs = [$tabs] unless ref $tabs;
    return () if any {$_ eq 'all'} @$tabs;

    my %filter = (defined $opts{filter})? %{$opts{filter}} : ();

    my $page_mod_filter = "where i_mbp.is_removed = 0 ";
    if ($filter{pid}) {
        my $pids = ref $filter{pid} ne 'ARRAY' ? [$filter{pid}] : $filter{pid};
        $page_mod_filter = $page_mod_filter . "and b.pid in (".join(',', map {sql_quote($_)} @$pids).")";
    } elsif ($filter{cid}) {
        my $cids = ref $filter{cid} ne 'ARRAY' ? [$filter{cid}] : $filter{cid};
        $page_mod_filter = $page_mod_filter . "and b.cid in (".join(',', map {sql_quote($_)} @$cids).")";
    } else {
        croak 'Does not exist filter. cid or pid needed';
    }

    my $known_tabs = xisect [keys %conditions], $tabs;
    croak 'Does not exist condition for tabs: ' . (join ', ', @$tabs) unless @$known_tabs;

    my ($where, $tables) = ([], []);
    foreach my $tab (@$known_tabs) {
        push @$where, $conditions{$tab};
        if ($join_tables{$tab}) {
            for my $table (@{$join_tables{$tab}}) {
                my $replaced_table = $table;
                if ($table =~ /_PAGE_MOD_FILTER_/) {
                   $replaced_table = $table =~ s/_PAGE_MOD_FILTER_/$page_mod_filter/r;
                }
                push @$tables, $replaced_table;
            }
        }
    }

    my $join_banners = 'JOIN banners b ON b.pid = g.pid';
    #Если целевой статус может быть и у пустых групп - используем LEFT JOIN banners
    $join_banners = "LEFT $join_banners" if @{xisect([qw/draft decline wait/], $tabs)};
    return (
        where => (@$where > 1 ? {_OR => [map { ('_AND' => $_) } @$where]} : $where->[0]),
        group_by => 'g.pid',
        fields => 'b.bid',
        tables => join(" \n", ($join_banners, uniq @$tables)),
    );
}

=head2 is_valid_status_filter(tab)

    Функция проверяется, существует ли фильтр условий на предложенную вкладку или нет.
    На входе: название вкладки
    На выходе: 0 - не существует, 1 - существует.

=cut
sub is_valid_status_filter {
    my $tab = shift;
    return (defined $tab && exists($conditions{$tab})) ? 1 : 0;
}

=head2 filter_banners_preview(%conditions)

    Формирование фильтра к функции Models::AdGroup::get_groups() для отбора баннеров по виду показа.

    Параметры:
        $conditions - условия показа баннеров
            pid => id группы объявлений
            tab => выборка по статусам баннеров
                active - активные
                off - остановленные
                arch - заархивированные
                draft - черновики
                decline - отклонённые
                wait - на модерации
                running_unmoderated - работают без модерации
        %options - опции
            do_not_devide_pid_bid - возвращать результат как {pid => [], bid => []}.
                Если не указан флаг, то возвращать с разбиением какие bid к какому pid относятся {123 => [111,222], 234=>[333]}.
    Результат:
        $filter - значение фильтра в формате, понятном функции get_groups()

=cut

sub filter_banners_preview {
    my $conditions = shift;
    my %options = @_;

    my $pid = $conditions->{pid};
    my $sql;
    my %filter;
    if  (defined $conditions->{tab}) {
        my %status_cond = get_status_condition($conditions->{tab}, filter => {pid => $pid});
        my $sql = sprintf ("SELECT b.bid, g.pid FROM phrases g %s", $status_cond{tables} || 'LEFT JOIN banners b USING(pid)');
        my $result = get_all_sql(PPC(pid => $pid), [$sql, WHERE => { 'g.pid' => SHARD_IDS, %{$status_cond{where} || {}} }]) || [];
        if ($options{do_not_devide_pid_bid}) {
            return { pid => [uniq map { $_->{pid} } @$result],
                     bid => [uniq map { $_->{bid} } @$result],
                   };
        } else {
            my $pid_bids = {};
            push @{$pid_bids->{$_->{pid}}}, $_->{bid} foreach (@$result);
            return $pid_bids;
        }
    }
    return {};
}

1;
