package Models::Banner;

use Direct::Modern;

use BannerTemplates;
use Campaign::Types;

use List::MoreUtils qw/uniq any all none/;
use List::Util qw/sum/;
use Yandex::Clone qw/yclone/;

use Settings;
use MailNotification;

use Primitives;
use PrimitivesIds;
use Property;
use VCards;
use TextTools;
use OrgDetails;
use GeoTools;
use URLDomain;
use AggregatorDomains qw//;
use BannerImages;
use LogTools;
use MTools;
use ModerateChecks;
use RedirectCheckQueue;
use Sitelinks;
use BannerFlags;
use MinusWords;
use HashingTools qw/md5_hex_utf8/;
use Carp qw/croak/;
use ShardingTools;
use Direct::AdGroups2::MobileContent;
use Direct::Validation::BannersMobileContent qw//;
use Direct::Model::BannerMobileContent;
use Direct::Model::BannerCreative;
use Direct::Model::CanvasCreative;
use Direct::Model::Creative;
use Direct::Model::Creative::Constants;
use Direct::Model::AdGroupMobileContent;
use Direct::Model::Banner;
use Direct::Model::Image;
use Direct::Model::ImageFormat;
use Direct::Model::ImagePool;
use Direct::Model::BannerImageAd;
use Direct::Model::BannerCpcVideo qw//;
use Direct::Model::BannerCpmOutdoor qw//;
use Direct::Model::BannerCpmIndoor qw//;
use Direct::Model::BannerCpmAudio qw//;
use Direct::Model::BannerCpmGeoPin qw//;
use Direct::Model::VideoAddition;
use Direct::Banners::Measurers qw//;
use Direct::BannersAdditions;
use Direct::BannersPermalinks;
use Direct::BannersPixels;
use Direct::BannersPrices;
use Direct::BannersPlacementPages;
use Direct::BannersResources;
use Direct::Model::AdditionsItemCallout;
use Direct::Model::TurboLanding::Banner;
use Direct::TurboLandings;
use Direct::Validation::BannersAdditions;
use Direct::Validation::Banners qw/validate_banner_display_href/;
use Direct::ValidationResult;
use Direct::Model::Banner::Constants;

use Models::AdGroupFilters qw/get_status_condition/;

use Yandex::IDN;
use Yandex::HashUtils;
use Yandex::ScalarUtils;
use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::URL;
use Yandex::Validate qw/is_valid_id/;
use Yandex::ListUtils qw/chunks/;
use Lang::Guess qw/analyze_text_lang analyze_text_lang_with_context/;
use Direct::Model::Banner::LanguageUtils;

use base qw/Exporter/;
our @EXPORT = qw/
    compile_banner_params
    
    has_camp_active_banners
    mass_has_camps_active_banners

    delete_entry_edited_by_moderator
    check_banner_ignore_moderate

    add_banner_error_text
    convert_banner_error_text_to_list
    validate_banner_title
    validate_banner_title_extension
    validate_banner_body

    check_geo_restrictions
    get_geo_restrictions

    has_delete_banner_problem
    fill_in_can_delete_banner
    delete_banners

/;
# символы, которые считаем пробелами
my $SPACES = " \x{00a0}"; # \x{00a0} -- неразрывный пробел

# количество кампаний в которых ищем активные баннеры
my $CIDS_CHUNK_SIZE = 500;

my $creatives_cache=\{};

# проперти переключения на новый запрос получения кампаний с активными баннерами
state $ENABLE_NEW_CAMPS_HAS_ACTIVE_BANNERS_REQUEST_PROP = Property->new('enable_new_camps_has_active_banners_request');

=head2 

    Поля, разложенные по структурам, для заполнения баннера

=cut
our %banner_fields = (
    group => {
        fields => [qw/pid adgroup_id adgroup_type group_name geo geo_names statusAutobudgetShow PriorityID
                      statusShowsForecast phrases retargetings search_retargetings is_completed_group banners_quantity banners_arch_quantity banners_canarch_quantity
                      minus_words group_banners_types
                      is_bs_rarely_loaded disabled_geo effective_geo cpm_banners_type
                  /],
        prefix => {p => [qw/statusModerate statusBsSynced statusPostModerate/]}
    },
    adgroup_base => {
        fields => [qw/relevance_match/],
    },
    adgroup_dynamic => {
        fields => [qw/feed_id main_domain main_domain_id dynamic_conditions statusBlGenerated/],
    },
    adgroup_mobile_content => {
        fields => [qw/mobile_content store_content_href device_type_targeting network_targeting target_interests relevance_match/],
    },
    adgroup_performance => {
        fields => [qw/feed_id feed_source feed_name feed_url feed_filename performance_filters statusBlGenerated/],
    },
    adgroup_content_promotion_video => {
        fields => [qw/relevance_match/],
    },
    adgroup_content_promotion => {
        fields => [qw/relevance_match/],
    },
    camp => {
        fields => [qw/spent_today currency OrderID strategy_no_premium
            sum_to_pay cstatusShow autobudget_date autobudgetForecast
            autobudgetForecastDate statusOpenStat cstatusModerate fairAuction
            cstatusContextStop auto_optimization strategy platform
            day_budget_daily_change_count day_budget_stop_time
            day_budget day_budget_show_mode cContextPriceCoef camp_is_arch
            autobudget autobudget_bid timeTarget timezone_id sum sum_spent
            statusEmpty cid camp_type
            campaign_minus_words
            wallet_day_budget wallet_day_budget_show_mode
            device_targeting
        /]
    },
);

=head2 compile_banner_params

  Собирает параметры для баннера на основе данных собственно баннера и кампании

=cut 

#TODO заменить на вызов compile_banner_params($camp, $group, $banner, %options)
sub compile_banner_params {

    my ($group, $banner, %options) = @_;

    my @phrases = @{$group->{phrases}};

    my $count_declined_phrases = scalar grep {$_->{statusModerate} eq 'No'} @phrases;
    my $banner_status_options = {
        banner => {
            real_banner_type => $banner->{real_banner_type},
            banner_statusModerate => $banner->{banner_statusModerate},
            statusModerate => $banner->{statusModerate},
            statusPostModerate => $banner->{statusPostModerate},
            statusBsSynced => $banner->{statusBsSynced},
            statusArch => $banner->{statusArch},
            BannerID => $banner->{BannerID},
            bid => $banner->{bid},
            statusShow => $banner->{statusShow},
            statusActive => $banner->{statusActive},
            phoneflag => $banner->{phoneflag},
            statusSitelinksModerate => $banner->{statusSitelinksModerate},
            display_href_statusModerate => $banner->{display_href_statusModerate},
            image_statusModerate => $banner->{image_statusModerate},
            turbolanding_statusModerate => (exists $banner->{turbolanding} ? $banner->{turbolanding}->{status_moderate} : $banner->{turbolanding_statusModerate}),
            href => $banner->{href},
            countDeclinedPhrases => $count_declined_phrases,
            creative_layout_id => $banner->{creative_layout_id},
            creative_statusModerate => $banner->{creative_statusModerate}, # TODO!!! уже есть в creative, возможно тут можно убрать?
            imagead_statusModerate => $banner->{im_statusModerate},
            b_perf_statusModerate => $banner->{b_perf_statusModerate},
            video_addition_statusModerate => $banner->{video_resources}->{status_moderate},
            minus_geo => $banner->{minus_geo},
        },
        phrases => hash_cut($group, qw/statusModerate statusPostModerate statusBsSynced statusBlGenerated/),
        camp => {
            sum_total => ($group->{sum}||0) - ($group->{sum_spent}||0),
            statusModerate => $group->{cstatusModerate},
            statusShow => $group->{cstatusShow},
            sum_to_pay => $group->{sum_to_pay},
        },
        banner_additions => {
            callout_statusModerate => ref($banner->{callouts}) eq 'ARRAY' ? [map {$_->{status_moderate}} @{$banner->{callouts}}] : [],
        },
        banner_placement_pages => {
            pages_moderation => ref($banner->{placement_pages}) eq 'ARRAY' ? [ map {{status_moderate => $_->{status_moderate}, status_moderate_operator => $_->{status_moderate_operator}}} @{$banner->{placement_pages}} ] : [],
        }
    };
    my $msg = calculateBannerStatus($banner_status_options, %options);

    # Вычисляем общую цену, если такая есть
    my @prices = uniq map {$_->{price}} grep {$_->{rank} && $_->{rank} > 2} @phrases;
    my $common_price = !@prices ? undef : @prices==1 ? $prices[0] : 0;
    my $performance_creative = $group->{adgroup_type} eq 'performance' && $banner->{real_banner_type} eq 'performance' ?
        Direct::Model::Creative->from_db_hash(
            {
                creative_creative_id => delete $banner->{creative_id},
                map { $_ => delete $banner->{$_} } qw/
                        creative_alt_text creative_height creative_href creative_name creative_template_id
                        creative_preview_url creative_statusModerate creative_sum_geo creative_width
                /,
            },
            $creatives_cache,
            prefix => 'creative_',
        )->to_hash()
        :
        undef;

    return hash_merge ($options{skip_banner_template} ? {} : add_banner_template_fields($banner, $group->{phrases})), {
                real_banner_type => $banner->{real_banner_type},
                hash_flags      => BannerFlags::get_banner_flags_as_hash($banner->{flags}),
                archive         => $banner->{statusArch},
                status          => _get_status_message($msg),  # Компиляцию строк лучше делать в шаблоне. Это задача уровня представления.
                                                               # html на уровне модели врядли удастся использовать более чем на одной-двух страничках
                status_warning  => $msg->{warning},
                enable          => $msg->{enable_edit},
                broker_price    => $common_price,
                is_moderate_decline => $msg->{decline_show},
                partly_declined_phrases => $count_declined_phrases && $group->{statusModerate} eq 'Yes' ? 1 : 0,
                domain_ascii    => Yandex::IDN::idn_to_ascii($banner->{domain}),
                declined_show => $msg->{declined_show},
                statusOpenStat  => $group->{statusOpenStat},
                is_bl_nothing_generated => $msg->{is_bl_nothing_generated},
                is_bl_processing => $msg->{is_bl_processing},
                $group->{adgroup_type} eq 'performance' && $banner->{real_banner_type} eq 'performance' ?
                    (creative => $performance_creative) : (),
    };
}

sub _get_status_message
{
    my $msg = shift;
    
    my $txt = join(" ", @{$msg->{text} || []});
    
    # удаляем пробелы и точки в конце
    $txt =~ s/[\s\.]+$//;
    
    return $txt;
}

=head3 calculateBannerStatus(opt, %options)

opt:
    banner. real_banner_type
            statusModerate
            BannerID
            statusArch
            bid
            statusShow
            statusActive
            statusBsSynced
            phoneflag
            href
            statusPostModerate
            statusSitelinksModerate
            image_statusModerate
            
    phrases.statusModerate
            statusBsSynced
            statusPostModerate
            countDeclinedPhrases
    
    camp.   sum_total
            statusShow
            statusModerate
options:
    easy_user

return:
    text         => array(text)
    enable_edit  => bool
    show_reasons => bool
    warning      => undef | text

=cut

sub calculateBannerStatus
{
    my $opt = shift;
    my %options = @_;
    
    my ($camp, $banner, $phrases, $banner_additions, $banner_placement_pages) = ($opt->{camp}, $opt->{banner}, $opt->{phrases}, $opt->{banner_additions}, $opt->{banner_placement_pages});
    
    my ($message, @mess) = ({});

    my $status_info = get_banner_status_info($opt, %options);

    $message->{enable_edit} = 1;
    
    if ($status_info->{archived}) {
        # объявление заархивировано
        $message->{enable_edit} = 0;
    }
    if ($status_info->{show_accepted} && ($status_info->{moderate_wait} || $options{easy_user}) && !$status_info->{moderate_contact_declined} && !$status_info->{moderate_declined} && !$status_info->{moderate_placement_pages_declined}) {
        push @mess, iget('Принято к показам.');
    } 
    if ($options{easy_user}) {
        if ($status_info->{stopped}) {
            push @mess, iget("Остановлено.");
        }
        if ($status_info->{show_active}) {
            push @mess, iget('Идут показы.');
        }
    }
    if ($status_info->{show_active} && $banner->{statusModerate} eq 'Sent' && $banner->{statusPostModerate} eq 'Yes' && !$options{easy_user}) {
        push @mess, iget('Идут показы.');
    }
    if (!$status_info->{stopped} && $status_info->{moderate_banner_accepted}) {
        push @mess, iget("Объявление принято.");
    }
    if ($status_info->{moderate_contact_wait}) {
        push @mess, iget("Контактная информация ожидает модерации.");
    }
    if ($status_info->{moderate_sitelinks_wait}) {
        push @mess, iget("Быстрые ссылки ожидают модерации.");
    }
    if ($status_info->{moderate_image_wait}) {
        push @mess, iget("Изображение ожидает модерации.");
    }
    if ($status_info->{moderate_display_href_wait} && !$status_info->{draft}) {
        push @mess, iget("Отображаемая ссылка ожидает модерации.");
    }
    if ($status_info->{moderate_video_addition_wait}) {
        push @mess, iget("Видеодополнение ожидает модерации.");
    }  
    if ($status_info->{moderate_turbolanding_wait}) {
        push @mess, iget("Турбо-страница ожидает модерации.");
    }
    if ($status_info->{moderate_placement_pages_wait}) {
        push @mess, iget("Ожидает модерации операторов.");
    }

    if ($status_info->{moderate_wait} && (!$status_info->{show_accepted} || $options{easy_user})) {
        push @mess, iget('Ожидает модерации.');
    }
    if ($status_info->{draft}) {
        $message->{declined_show} = 0;
        push @mess, iget("Черновик.");
    }
    if ($status_info->{moderate_placement_pages_absent}) {
        push @mess, iget("Размеры или длительность креатива не соответствуют выбранным поверхностям.");
    }
    if ($status_info->{moderate_placement_pages_activization}) {
        push @mess, iget("Идет активизация операторами.");
    }

    my @declined_flags = qw/moderate_contact_declined moderate_phrases_declined moderate_sitelinks_declined
        moderate_image_declined moderate_callout_declined moderate_display_href_declined moderate_video_addition_declined moderate_turbolanding_declined moderate_placement_pages_declined/;
    my %flag2msg = (
        moderate_contact_declined         => iget_noop('контактная информация'),
        moderate_phrases_declined         => iget_noop('часть фраз'),
        moderate_sitelinks_declined       => iget_noop('быстрые ссылки'),
        moderate_image_declined           => iget_noop('изображение'),
        moderate_callout_declined         => iget_noop('уточнения'),
        moderate_display_href_declined    => iget_noop('отображаемая ссылка'),
        moderate_video_addition_declined  => iget_noop('видеодополнение'),
        moderate_turbolanding_declined    => iget_noop('Турбо-страница'),
        moderate_placement_pages_declined => iget_noop('Операторы'),
    );
    if (any { $status_info->{$_} } ('moderate_declined', @declined_flags) ) {
        $message->{declined_show} = 1;
    }
    if ($status_info->{moderate_declined} && !$status_info->{moderate_wait}) {
        push @mess, iget('Отклонено модератором.');
    }
    $status_info->{$_} ||= 0 for @declined_flags;
    if (sum(map { $status_info->{$_} } @declined_flags) > 1) {
        push @mess, iget('Отклонены: ') . join(', ', map { iget($flag2msg{$_}) } grep {$status_info->{$_}} @declined_flags) . '.';
    } elsif ($status_info->{moderate_contact_declined}) {
        push @mess, iget('Контактная информация отклонена.');
    } elsif ($status_info->{moderate_phrases_declined}) {
        push @mess, iget("Часть фраз отклонена.");
    } elsif ($status_info->{moderate_sitelinks_declined}) {
        push @mess, iget('Быстрые ссылки отклонены.');
    } elsif ($status_info->{moderate_image_declined}) {
        push @mess, iget('Изображение отклонено.');
    } elsif ($status_info->{moderate_callout_declined}) {
        push @mess, iget('Уточнения отклонены.');
    } elsif ($status_info->{moderate_display_href_declined}) {
        push @mess, iget('Отображаемая ссылка отклонена.');
    } elsif ($status_info->{moderate_video_addition_declined}) {
        push @mess, iget('Видеодополнение отклонено.');
    } elsif ($status_info->{moderate_turbolanding_declined}) {
        push @mess, iget('Турбо-страница отклонена.')
    } elsif ($status_info->{moderate_placement_pages_declined}) {
        push @mess, iget('Отклонено у некоторых операторов.')
    }
    if ($status_info->{activation} && !$status_info->{moderate_wait}) {
        push @mess, iget("Идет активизация.");
    }
    if ($status_info->{bl_nothing_generated}) {
        push @mess, iget("Объявления не созданы.");
        $message->{is_bl_nothing_generated} = 1;
    } elsif ($status_info->{bl_processing}) {
        push @mess, iget("Идет обработка.");
        $message->{is_bl_processing} = 1;
    }

    $message->{text} = \@mess;

    # WARNINGS для активного баннера
    if ($banner->{statusActive} eq 'Yes') {
        if (($status_info->{moderate_wait} || $status_info->{moderate_declined}) && ($banner->{real_banner_type} // '') !~ /^(cpm_outdoor)$/) {
            $message->{warning} = iget("Идут показы предыдущей версии объявления!");
        }
        if ($status_info->{moderate_contact_declined} && !$banner->{countDeclinedPhrases}) {
            $message->{warning} = iget('Идут показы без ссылки «Адрес и телефон»!');
        }
        if ($status_info->{moderate_contact_declined} && $banner->{countDeclinedPhrases}) {
            $message->{warning} = iget('Идут показы по принятым фразам без ссылки «Адрес и телефон»!');
        }
        if (!$status_info->{moderate_contact_declined} && $status_info->{moderate_phrases_declined}) {
            $message->{warning} = iget("Идут показы по принятым фразам!");
        }
    }

    return $message;
}

=head2 get_banner_status_info

    Возвращаем хэш со статусной информацией.
    {
        archived                    => undef | 1,
        show_active                 => undef | 1,
        show_accepted               => undef | 1,
        moderate_wait               => undef | 1,
        moderate_banner_accepted    => undef | 1,
        moderate_contact_wait       => undef | 1,
        moderate_sitelinks_wait     => undef | 1,
        moderate_declined           => undef | 1,
        moderate_contact_declined   => undef | 1,
        moderate_sitelinks_declined => undef | 1,
        moderate_phrases_declined   => undef | 1,
        moderate_image_declined     => undef | 1,
        moderate_image_wait         => undef | 1,
        moderate_display_href_declined => undef | 1,
        moderate_display_href_wait  => undef | 1,
        moderate_turbolanding_declined => undef | 1,
        moderate_turbolanding_wait  => undef | 1,
        moderate_banner_pages_declined => undef | 1,
        moderate_banner_pages_wait  => undef | 1,
        draft                       => undef | 1,
        activation                  => undef | 1,
        stopped                     => undef | 1,
    }

=cut

sub get_banner_status_info {
    my $opt = shift;
    my %options = @_;
    
    my ($camp, $banner, $phrases, $banner_additions, $banner_placement_pages) = ($opt->{camp}, $opt->{banner}, $opt->{phrases}, $opt->{banner_additions}, $opt->{banner_placement_pages});
    
    my $status_info = {
        archived                    => undef,
        show_active                 => undef,
        show_accepted               => undef,
        moderate_wait               => undef,
        moderate_banner_accepted    => undef,
        moderate_contact_wait       => undef,
        moderate_sitelinks_wait     => undef,
        moderate_video_addition_wait => undef,
        moderate_declined           => undef,
        moderate_contact_declined   => undef,
        moderate_sitelinks_declined => undef,
        moderate_phrases_declined   => undef,
        moderate_image_declined     => undef,
        moderate_image_wait         => undef,
        moderate_display_href_declined => undef,
        moderate_display_href_wait  => undef,
        moderate_callout_declined   => undef,
        moderate_video_addition_declined => undef,
        moderate_turbolanding_declined => undef,
        moderate_turbolanding_wait  => undef,
        moderate_placement_pages_declined => undef,
        moderate_placement_pages_wait  => undef,
        moderate_placement_pages_absent  => undef,
        moderate_placement_pages_activization  => undef,
        draft                       => undef,
        activation                  => undef,
        stopped                     => undef,
    };
    
    if ($banner->{statusArch} && $banner->{statusArch} eq 'Yes') {
        $status_info->{archived} = 1;
        return $status_info;
    }

    if ($options{easy_user}) {
        if ($camp->{statusModerate} eq 'No') {
            $status_info->{moderate_declined} = 1;
        }
        elsif ($camp->{statusModerate} eq 'Sent' || $camp->{statusModerate} eq 'Ready') {
            $status_info->{moderate_wait} = 1;
        }
    }

    my $RE_ON_MODERATION = qr/^(Ready|Sent|Sending)$/;
    
    no warnings 'uninitialized';
    my $phone_on_moderate     = $banner->{phoneflag} && $banner->{phoneflag} =~ /^(Ready|Sent|Sending)$/i;
    my $sitelinks_on_moderate = $banner->{statusSitelinksModerate} && $banner->{statusSitelinksModerate} =~ /^(Ready|Sent|Sending)$/i;
    my $image_on_moderate     = $banner->{image_statusModerate}    && $banner->{image_statusModerate}    =~ /^(Ready|Sent|Sending)$/i;
    my $phone_accepted        = $banner->{phoneflag} && $banner->{phoneflag} eq 'Yes';
    
    my $display_href_on_moderate = ($banner->{display_href_statusModerate} || '') =~ /^(Ready|Sent|Sending)$/i;
    my $turbolanding_on_moderate = ($banner->{turbolanding_statusModerate} || '') =~ /^(Ready|Sent|Sending)$/i;
    my $video_add_on_moderate = $banner->{video_addition_statusModerate} =~ $RE_ON_MODERATION;
    
    # объявление принято (визитка не считается при наличии ссылки)
    my $banner_draft         = $banner->{statusModerate} eq 'New' || $phrases->{statusModerate} eq 'New' || $camp->{statusModerate} eq 'New' || $banner->{creative_statusModerate} eq 'New';
    my $banner_on_moderate   = $banner->{statusModerate} =~ /^(Ready|Sent|Sending)$/;
    my $imagead_on_moderate  = $banner->{imagead_statusModerate} =~ /^(Ready|Sent|Sending)$/;
    my $b_perf_on_moderate  = $banner->{b_perf_statusModerate} =~ /^(Ready|Sent|Sending)$/;
    
    # для графических объявлений баннер считаем на модерации если картинка еще модерируется
    if ($banner->{imagead_statusModerate} =~ /^(Ready|Sent|Sending)$/) {
        $banner_on_moderate = 1;
    }
    # для креативов считаем, что баннер находится на модрации, только если баннер не черновик
    if ($banner->{statusModerate} ne 'New' && $banner->{creative_statusModerate} =~ /^(Ready|Sent|Sending)$/) {
        $banner_on_moderate = 1;
    }
    # для динамических считаем что баннер находится на модерации если условие тоже на модерации
    if ($banner->{real_banner_type} eq 'dynamic') {
        $banner_on_moderate ||= $phrases->{statusModerate} =~ /^(Ready|Sent|Sending)$/;
    }

    my $placement_pages_declined = $banner_placement_pages && @{$banner_placement_pages->{pages_moderation}} && any {$_->{status_moderate} eq 'No'} @{$banner_placement_pages->{pages_moderation}};
    my $placement_pages_accepted = $banner_placement_pages && @{$banner_placement_pages->{pages_moderation}} && any {$_->{status_moderate} eq 'Yes'} @{$banner_placement_pages->{pages_moderation}};
    my $placement_pages_on_moderate = !$placement_pages_accepted && any {$_->{status_moderate} =~ qr/^(Ready|Sent|Sending|Maybe)$/} @{$banner_placement_pages->{pages_moderation}};
    my $placement_pages_absent = $banner_placement_pages && ! @{$banner_placement_pages->{pages_moderation}};
    my $placement_pages_activization = $banner_placement_pages && @{$banner_placement_pages->{pages_moderation}} && any {$_->{status_moderate} eq 'Yes' && $_->{status_moderate_operator} ne 'Yes'} @{$banner_placement_pages->{pages_moderation}};

    my $is_creative_obsolete = scalar grep { $_ == $banner->{creative_layout_id} } @{ Direct::Model::Creative::Constants::OBSOLETE_LAYOUT_IDS() };
    my $banner_text_accepted = $banner->{statusModerate} eq 'Yes' && $phrases->{statusModerate} eq 'Yes' || $banner->{creative_statusModerate} eq 'Yes' || $is_creative_obsolete;
    my $banner_text_declined = $banner->{statusModerate} eq 'No'  || $phrases->{statusModerate} eq 'No' || $banner->{creative_statusModerate} eq 'No' && !$is_creative_obsolete;
    my $banner_post_accepted = $banner->{statusPostModerate} eq 'Yes' && $phrases->{statusPostModerate} eq 'Yes' && ($banner->{href} || $phone_accepted);
    my $camp_can_shows       = (!$camp->{statusShow} || $camp->{statusShow} eq 'Yes') && $camp->{sum_total} > 0.001 && (!$camp->{sum_to_pay} || $camp->{sum_to_pay} < 0.01);
    
    if ($banner_on_moderate) {
        if ($banner_post_accepted && $banner->{statusShow} eq 'Yes' && $camp_can_shows && !$imagead_on_moderate && !$b_perf_on_moderate
            && ($placement_pages_accepted || $banner->{real_banner_type} !~ /^(cpm_outdoor)$/)) {
            $status_info->{show_active} = 1;
        } else {
            $status_info->{moderate_wait} = 1;
        }
    }

    if ($banner_post_accepted && $banner->{statusShow} eq 'Yes') {
        if ($banner_on_moderate && !$camp_can_shows || $camp->{statusModerate} eq 'Yes' && ($camp->{sum_to_pay} && $camp->{sum_to_pay} > 0.001 || $camp->{sum_total} < 0.01)) {
            $status_info->{show_accepted} = 1;
        }
    }

    # если текст объявления(без ссылки) принят, телефон или сайтлинки еще на модерации    
    if ($banner_text_accepted && ($phone_on_moderate || $sitelinks_on_moderate)) {
        $status_info->{moderate_banner_accepted} = 1;
    }
    if ($phone_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_contact_wait} = 1;
    }
    if ($sitelinks_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_sitelinks_wait} = 1;
    }
    if ($image_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_image_wait} = 1;
    }
    if ($display_href_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_display_href_wait} = 1;
    }
    if ($video_add_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_video_addition_wait} = 1;
    }
    if ($turbolanding_on_moderate && !$banner_on_moderate && !$banner_text_declined) {
        $status_info->{moderate_turbolanding_wait} = 1;
    }
    if ($banner_text_declined && !$banner_on_moderate) {
        $status_info->{moderate_declined} = 1;
    }

    if ($banner_text_accepted && $banner->{phoneflag} && $banner->{phoneflag} eq 'No') {
        $status_info->{moderate_contact_declined} = 1;
    }

    if ($banner_text_accepted && $banner->{countDeclinedPhrases}) {
        $status_info->{moderate_phrases_declined} = 1;
    }

    if ($banner_text_accepted && $banner->{statusSitelinksModerate} eq 'No') {
        $status_info->{moderate_sitelinks_declined} = 1;
    }
    if ($banner_text_accepted && defined $banner->{image_statusModerate} && $banner->{image_statusModerate} eq 'No') {
        $status_info->{moderate_image_declined} = 1;
    }
    if ($banner_text_accepted && ($banner->{display_href_statusModerate} || '') eq 'No') {
        $status_info->{moderate_display_href_declined} = 1;
    }
    if ($banner_text_accepted && $banner->{video_addition_statusModerate} eq 'No') {
        $status_info->{moderate_video_addition_declined} = 1;
    }
    if ($banner_text_accepted && $banner->{turbolanding_statusModerate} eq 'No') {
        $status_info->{moderate_turbolanding_declined} = 1;
    }

    if ($banner->{statusModerate} eq 'Yes' && $placement_pages_on_moderate) {
        $status_info->{moderate_placement_pages_wait} = 1;
    }
    if ($banner->{statusModerate} eq 'Yes' && $placement_pages_declined) {
        $status_info->{moderate_placement_pages_declined} = 1;
    }
    #для outdoor минус гео всегда вычитает гео группы. Не показывает плашку про отсутствие блоков при наличии минус гео
    if ($banner->{real_banner_type} eq 'cpm_outdoor' && $banner->{banner_statusModerate} eq 'Yes' && $placement_pages_absent && !defined $banner->{minus_geo}) {
        $status_info->{moderate_placement_pages_absent} = 1;
    }

    if ($banner->{statusModerate} eq 'Yes' && $placement_pages_activization) {
        $status_info->{moderate_placement_pages_activization} = 1;
    }

    if ($banner_draft) {
        $status_info->{draft} = 1;
    }

    if ($banner->{statusModerate} ne 'New'                                                              # баннер не черновик
        && $camp->{sum_total} > 0.001                                                                   # на кампании есть деньги
        && ( $banner->{statusBsSynced} ne 'Yes' 
                    || ($phrases->{statusBsSynced} ne 'Yes' && $banner->{statusShow} && $banner->{statusShow} eq 'Yes') )    # banners.statusBsSynced не Yes или phrases.statusBsSynced не Yes
        && ( !$camp->{statusShow} || $camp->{statusShow} eq 'Yes' )                                     # кампания не остановлена
        && ( $banner->{BannerID} || $banner->{statusShow} && $banner->{statusShow} eq 'Yes' )           # баннер не остановлен или был в БК
        && ( # у баннера есть либо ссылка, либо телефон, либо это мобильная/динамическая/перфоманс кампания
            $banner->{href}
            || $banner->{phoneflag} eq 'Yes'
            || $banner->{real_banner_type} =~ /^(?:dynamic|mobile_content|performance|image_ad|cpc_video)$/,
        )
        && ( !$sitelinks_on_moderate )
        && ( $banner->{BannerID} && $banner->{statusShow} eq 'No'
                || $banner->{statusPostModerate} =~ /^(Yes|Rejected)$/ && $phrases->{statusPostModerate} =~ /^(Yes|Rejected)$/
                    && ($banner->{real_banner_type} !~ /^(image_ad|cpc_video|cpm_outdoor|cpm_indoor|cpm_audio)$/ || $banner->{statusModerate} eq 'Yes')
           )
        && (  (!$placement_pages_on_moderate && !$placement_pages_absent)
              || $banner->{real_banner_type} !~ /^(cpm_outdoor)$/
           )
    ) {
        $status_info->{activation} = 1;
    }

    if ($banner->{statusShow} eq 'No' && !$status_info->{draft} && !$status_info->{moderate_declined} && !$status_info->{moderate_phrases_declined} && !$status_info->{moderate_wait}) {
        $status_info->{stopped} = 1;
    }

    if (
        $banner->{real_banner_type} =~ /^(?:dynamic|performance)$/ &&   # Динамический или смарт баннер
        ($banner->{BannerID} && $banner->{statusShow} eq 'Yes') &&      # Был в БК, не остановлен
        (none { defined $_ } values %$status_info) &&                   # Других статусов нет
        $phrases->{statusBlGenerated} ne 'Yes'                          # Статус генерации фраз не Yes
    ) {
        if ($phrases->{statusBlGenerated} eq 'No') {
            $status_info->{bl_nothing_generated} = 1;
        } elsif ($phrases->{statusBlGenerated} eq 'Processing') {
            $status_info->{bl_processing} = 1;
        }
    }

    # Статус show_active выставляем в случае отсутствия других статусов
    $status_info->{show_active} = 1 if $camp_can_shows && (none { defined $_ } values %$status_info);

    $status_info->{moderate_callout_declined} = 1 if any {$_ eq 'No'} @{$banner_additions->{callout_statusModerate}};

    return $status_info;
}


=head2 stop_banners($banners, $options)

    Останавливает показы объявлений
    
    $banners - список баннеров [{bid => }, {bid => }, {bid => }]
    $options
        uid - пользователь выполнивший операцию(будет отправлено уведомление)

=cut

sub stop_banners {
    
    my ($banners, $options) = @_;
    
    my $quantity = do_update_table(PPC(bid => [map {$_->{bid}} @$banners]), 'banners',
                                        {statusShow => 'No',
                                         statusBsSynced => 'No',
                                         LastChange__dont_quote => 'NOW()',
                                        },
                                        where => {'bid' => SHARD_IDS, 'statusShow' => 'Yes'});

    if ($options->{uid}) {
        mass_mail_notification([map {
            {
                object => 'banner', event_type => 'b_status',
                old_text => 'start', new_text => 'stop',
                object_id => $_->{bid}, uid => $options->{uid}
            }
        } @$banners]);
    }
                                
    return $quantity;
}

=head2 resume_banners($banners, $options)

    Включает остановленные группы
    
    $banners - список баннеров [{bid => }, {bid => }, {bid => }]
    $options

         uid - пользователь выполнивший операцию(будет отправлено уведомление)
         
    Не обновляет статус для включенных и архивных объявлений. Ошибок наружу не отдает.

=cut

sub resume_banners {
    
    my ($banners, $options) = @_;

    my $quantity = 0;
    for my $chunk (sharded_chunks bid => [map { $_->{bid} } @$banners]) {
        $quantity += do_sql(PPC(shard => $chunk->{shard}), ["
                                  UPDATE banners b
                                    SET
                                        b.statusShow = 'Yes'
                                       , b.statusBsSynced = 'No'
                                       , b.LastChange = NOW()",
                                    WHERE => {
                                        'b.bid' => $chunk->{bid},
                                        'b.statusArch' => 'No',
                                        'b.statusShow' => 'No',
                                    }]);
                                    
        my @mail_notifications;
        foreach my $bid (@{$chunk->{bid}}) {
                
            push @mail_notifications, { object => 'banner', event_type => 'b_status', object_id => $bid,
                                            old_text => 'stop', new_text => 'start', uid => $options->{uid} };
        }

        mass_mail_notification(\@mail_notifications);
    }

    return $quantity;
}

=head2 update_banner_statuses_is_obsolete

    инвалидировать агрегированные статусы для объявлений

=cut

sub update_banner_statuses_is_obsolete {
    my ($banner_ids, $update_before) = @_;
    die 'Missing required parameter: update_before' unless $update_before;

    do_update_table(PPC(bid => $banner_ids), 'aggr_statuses_banners',
        { is_obsolete => '1', updated__dont_quote => 'NOW()' },
        where => {'bid' => SHARD_IDS, 'updated__lt' => $update_before}
    );
}

=head2 update_adgroup_statuses_is_obsolete

    инвалидировать агрегированные статусы для групп 

=cut

sub update_adgroup_statuses_is_obsolete {
    my ($adgroup_ids, $update_before) = @_;
    die 'Missing required parameter: update_before' unless $update_before;

    do_update_table(PPC(pid => $adgroup_ids), 'aggr_statuses_adgroups',
        { is_obsolete => '1', updated__dont_quote => 'NOW()' },
        where => {'pid' => SHARD_IDS, 'updated__lt' => $update_before}
    );
}

=head2 get_update_before

    получение update_before для update_banner_statuses_is_obsolete
    чтобы если процессор обновит статус, то не инвалидировать его

=cut

sub get_update_before {
    my $update_before = get_one_field_sql(PPCDICT, "SELECT NOW()");
    return $update_before;
}

=head2 create_base_banners($banners, $uid, $options)

    Создание баннеров (вместе с визиткой, картинкой, сайтлинками) для одного пользователя.
    При наличии визитки у баннеров она будет также создана (или используется уже существующая в этой кампании у этого пользователя).
    Баннеры для пустой кампании (statusEmpty == 'Yes') или баннеры с statusModerate == 'New' сохраняются как черновики.
    
    Параметры:
        $banners - массив баннеров [{}] которые необходимо создать.
            Визитка передаётся вместе с баннером.
            Каждый баннер - это хеш {} со набором полей:
            {
                cid => номер кампании
                pid => номер группы объявлений
                banner_type => desktop|mobile - тип сохраняемого баннер
                title => 
                title_extension => 
                body => 
                href => 
                domain => 
                flags => строка - серриализованное значение флагов для баннера
                geo => геотаргетинг на группу объявлений  
                statusEmpty => Yes|No - Yes - кампания в которую добавляется баннер пустая (без баннеров) 
                statusModerate => New|Any - New - сохранить баннер как черновик, другие значения здесь не имеют значений 
                
                # vcard (может отсутствовать)
                geo_id => - при отсутствии будет вычислен по city
                address_id =>
                org_details_id =>  
                ogrn =>
                
                phone 
                country city street house build apart metro
                name contactperson contact_email
                worktime 
                extra_message 
                im_client im_login
                
                sitelinks => массив [] сайтлинков, может отсутствовать
                image => '' - hash загружаемой картинки, может отсутствовать
            }
            В качестве ОГРН в визитке можно передавать org_details_id и/или непосредственно ogrn 
            
        $uid - id владельца
        $options - хеш {} с параметрами функции (может отсутствовать):
            ignore_callouts => 1|0 - не менять уточнения на баннере
            
    Результат:
        $banners - исходный массив [] баннеров с выставленным значение bid

=cut

sub create_base_banners {
    
    my ($banners, $uid, $options) = @_;
    $options ||= {};
    
    croak 'unknown banner type' unless all { defined $_->{banner_type} && $_->{banner_type} =~ /^(desktop|mobile|image_ad)$/ } @$banners;

    # Проверим, что баннер создается в текстовой группе
    my $has_non_text_adgroup = get_one_field_sql(PPC(pid => [map { $_->{pid} } @$banners]), [
        q{SELECT 1 FROM phrases}, where => {pid => SHARD_IDS, adgroup_type__ne => 'base'}, "LIMIT 1"
    ]);
    croak "Cannot create banners in non-text adgroup using `create_base_banners`" if $has_non_text_adgroup;

    my (@vcards, @vcard_to_banner);
    foreach my $banner (@$banners) {
        my $vcard = $banner->{vcard} || $banner;
        # у баннера нет визитки
        next unless $vcard->{phone};
        
        $vcard->{geo_id} ||= 0;
        $vcard->{address_id} = $banner->{map}->{aid} if ref $banner->{map} eq 'HASH';
        $vcard->{uid} = $uid;
        $vcard->{phone} = compile_phone(hash_cut $vcard, @VCards::PHONE_FIELDS);
        $vcard->{org_details_id} = add_org_details(make_org_details_from_vcard($vcard, {uid => $uid}));
        $vcard->{cid} = $banner->{cid};
        $vcard = hash_cut $vcard, @$VCARD_FIELDS_DB;

        push @vcards, $vcard;
        push @vcard_to_banner, $banner;
    }

    if (@vcards) {
        VCards::create_vcards($uid, \@vcards); 
        for my $idx (0..$#vcards) {
            $vcard_to_banner[$idx]{vcard_id} = $vcards[$idx]{vcard_id};
        }
    }
   
    my $ClientID = get_clientid(uid => $uid);
    
    my @fields = (qw/bid cid pid vcard_id banner_type title title_extension body href/,
                  qw/domain reverse_domain flags statusModerate statusPostModerate/, 
                  qw/sitelinks_set_id statusSitelinksModerate phoneflag/,
                  qw/geoflag opts/);
    my $bids = get_new_id_multi(bid => scalar(@$banners), uid => $uid);
    my (@banners_to_insert, @banners_to_redirect, %groups_to_moderate);
    foreach my $source_banner (@$banners) {
     
        $source_banner->{bid} = shift @$bids;
        
        my $banner = hash_cut $source_banner, @fields;
        $banner->{flags} = BannerFlags::serialize_banner_flags_hash($banner->{flags}) if $banner->{flags}; 
        
        $banner->{href} = clear_banner_href($banner->{href}) if defined $banner->{href};
        $banner->{domain} = $banner->{domain_redir} = '' if !defined $banner->{href} || $banner->{href} eq '';
        
        unless ($banner->{domain}) {
            my ($href_domain, $domain_need_check) = RedirectCheckQueue::domain_need_check_redirect(
                { href => $banner->{href} }        
            );
            $banner->{domain} = $href_domain;  
            push @banners_to_redirect, {bid => $source_banner->{bid}} if $domain_need_check;
        }
        $banner->{reverse_domain} = reverse_domain( $banner->{domain} ) if $banner->{domain};

        $banner->{$_} = html2string($banner->{$_}) foreach qw/title title_extension body/;
        foreach (qw/href domain reverse_domain/) {
            $banner->{$_} = undef if defined $banner->{$_} && $banner->{$_} =~ m/^\s*$/s;
        };
        
        $banner->{sitelinks_set_id} = undef;
        $banner->{statusSitelinksModerate} = 'New';
        
        $source_banner->{statusModerate} = $banner->{statusModerate} = $source_banner->{statusEmpty} ne "Yes" && $source_banner->{statusModerate} ne "New" ? 'Ready' : 'New';
        $banner->{statusPostModerate} = 'No'; 
        $banner->{phoneflag} = $banner->{vcard_id} ? $banner->{statusModerate} : 'New';
        # && $banner->{pid} - баннер может быть создан до создания группы объявлений (это неправильное поведение, должно быть наоборот)
        $groups_to_moderate{$banner->{pid}} = 1 if $banner->{statusModerate} eq 'Ready' && $banner->{pid};
    
        my $geoflag;
        my $geo = GeoTools::modify_translocal_region_before_save($source_banner->{geo}, {ClientID => $ClientID});
        refine_geoid($geo, \$geoflag, {ClientID => $ClientID});
        
        $banner->{geoflag} = $geoflag;
        $banner->{opts} = Yandex::DBTools::_sql_set_set(undef, {geoflag => $geoflag});
        
        push @banners_to_insert, [map {$banner->{$_}} @fields];
    }
    
    do_mass_insert_sql(PPC(uid => $uid),
        sprintf('INSERT INTO banners(%s) VALUES %%s', join ',', map {$_ eq 'banner_type' ? sql_quote_identifier('type') : sql_quote_identifier($_)} @fields),
        \@banners_to_insert);
    RedirectCheckQueue::push_banners(\@banners_to_redirect) if @banners_to_redirect;  
    
    # группы черновики отправляем на модерацию, т.к. появился как минимум один баннер не черновик
    if (my @pids = keys %groups_to_moderate) {
         do_update_table(PPC(uid => $uid), 'phrases',
            {statusModerate => 'Ready'},
            where => {statusModerate => 'New', pid => \@pids});
    }
          
    my (%camps_active_banners, %cids_statusModerate);
    my $cid2clientid = get_key2clientid(cid => [uniq map { $_->{cid} } @$banners]);
    foreach my $banner (@$banners) {
        update_filter_domain($banner->{bid}, $banner->{domain});
        if ($banner->{statusModerate} ne 'New') {
            unless (exists $camps_active_banners{$banner->{cid}}) {
                $camps_active_banners{$banner->{cid}} = has_camp_active_banners($banner->{cid});
            }
            $cids_statusModerate{$banner->{cid}} = 1 if !$camps_active_banners{$banner->{cid}};
        }
    }
    AggregatorDomains::update_aggregator_domains({map { $_->{bid} => $_->{href} } grep { $_->{href} } @$banners});
    
    if (my @cids = keys %cids_statusModerate) { 
        do_update_table(PPC(uid => $uid), 'campaigns',
            {statusModerate => 'Ready'},
            where => {statusModerate => 'No', cid => \@cids});
    }
    
    _create_base_banner_addons($banners, $uid, $options);
    _on_base_banners_created($banners, $uid);
    
    return $banners;
}

sub _on_base_banners_created {
    
    my ($banners, $uid) = @_;
    
    my @notifications;
    foreach my $banner (@$banners) {
    
        my $contacts = get_contacts_string($banner, "\n", {dont_include_ids => 1});
        my $new_banner_text = join "\n", (map { $banner->{$_} // '' } qw/title body href domain/), $contacts;
        $new_banner_text = html2string($new_banner_text);
        
        push @notifications, {
            object     => 'banner',
            event_type => 'b_create', 
            object_id  => $banner->{bid}, 
            old_text   => '', 
            new_text   => $new_banner_text, 
            uid        => $uid,
        };
    }
    
    mass_mail_notification(\@notifications);
}

# создание у новых баннеров дополнений (сайтлинки, картинка, дополнения)
sub _create_base_banner_addons {
    
    my ($banners, $uid, $options) = @_;
    
    my $ClientID = get_clientid(uid => $uid);
    my %banner_values;
    my @images;
    my @display_hrefs_to_insert;
    foreach my $banner (@$banners) {        
        if (Sitelinks::need_save_sitelinks_set($banner->{sitelinks})) {
            my $sitelinks_set_id = Sitelinks::save_sitelinks_set($banner->{sitelinks}, $ClientID);
            $banner_values{$banner->{bid}} = {
                sitelinks_set_id => $sitelinks_set_id,
                statusSitelinksModerate => $banner->{statusModerate}
            };
        }
        
        if ($banner->{image}) {
            push @images, {
                bid => $banner->{bid},
                cid => $banner->{cid},
                ClientID => $ClientID,
                image_hash => $banner->{image},
                statusModerate => $banner->{statusModerate},
            };
        }

        if ($banner->{display_href}) {
            push @display_hrefs_to_insert, [
                $banner->{bid},
                $banner->{display_href},
            ];
        }
    }

    # создаем дополнения к баннерам
    if (! $options->{ignore_callouts}) {
        _mass_update_banners_additions([grep {ref($_->{callouts}) eq 'ARRAY'} @$banners], {}, $ClientID);
    }

    # отображаемые ссылки
    if (@display_hrefs_to_insert) {
        do_mass_insert_sql(PPC(uid => $uid),
            'INSERT INTO banner_display_hrefs(bid, display_href) VALUES %s',
            \@display_hrefs_to_insert,
        );
    }

    mass_banner_assign_image(@images);
    _update_banner_values(\%banner_values, $uid);
    
    return $banners;
}

sub _update_banner_values {
    
    my ($banner_values, $uid) = @_;

    my @bids = keys %$banner_values;
    return unless @bids; 
    
    my %case_values = (sitelinks_set_id => {}, statusSitelinksModerate => {});
    foreach (@bids) {
        $case_values{sitelinks_set_id}->{$_} = $banner_values->{$_}->{sitelinks_set_id};
        if (exists $banner_values->{$_}->{statusSitelinksModerate}) {
            $case_values{statusSitelinksModerate}->{$_} = $banner_values->{$_}->{statusSitelinksModerate};
        }
    }
    
    do_update_table(PPC(uid => $uid), 'banners',
        {
            statusBsSynced => 'No',
            map {
                $_ => sql_case(bid => $case_values{$_}, default__dont_quote => $_) 
            } qw/sitelinks_set_id statusSitelinksModerate/
        },
        where => {bid => \@bids}, dont_quote => [qw/sitelinks_set_id statusSitelinksModerate/]
    );
}

=head2 update_base_banners($banners, $uid, $options)

    Сохранение изменений в баннерах (вместе с визиткой, картинкой, сайтлинками) для одного пользователя.
    При наличии визитки у баннеров она будет также создана (или используется уже существующая в этой кампании у этого пользователя).
    Баннеры для пустой кампании (statusEmpty == 'Yes') или баннеры с statusModerate == 'New' сохраняются как черновики.
    
    Параметры:
        $banners - массив баннеров [{}] которые необходимо обновить.
            Визитка передаётся вместе с баннером.
            Каждый баннер - это хеш {} со набором полей:
            {
                bid => номер баннера
                cid => номер кампании
                pid => номер группы объявлений
                banner_type => desktop|mobile - тип сохраняемого баннер
                title =>
                title_extension=>
                body => 
                href => 
                domain => 
                statusEmpty => Yes|No - Yes - кампания в которую добавляется баннер пустая (без баннеров)
                statusModerate => New|Any - New - сохранить баннер как черновик, другие значения здесь не имеют значений
                
                # vcard (может отсутствовать)
                geo_id => - при отсутствии будет вычислен по city
                address_id =>
                org_details_id =>  
                ogrn =>
                
                phone 
                country city street house build apart metro
                name contactperson contact_email
                worktime 
                extra_message 
                im_client im_login
                
                sitelinks => массив [] сайтлинков, может отсутствовать
                image => '' - hash загружаемой картинки, может отсутствовать
            }
            В качестве ОГРН в визитке можно передавать org_details_id и/или непосредственно ogrn
            Отсутствующая визитка означет отвязать визитку (если раньше она имелась) у сохраняемого баннера. 
            
        $uid - id владельца
        $options - хеш {} с параметрами функции (может отсутствовать)
            ignore_vcard => 1|0 - 1 не обрабатывать визитку в баннере (не удалять, не обновлять, не создавать)
            ignore_sitelinks => 1|0 - 1 не обрабатывать сайтлинки в баннере (не удалять, не обновлять, не создавать)
            ignore_image => 1|0 - 1 не обрабатывать картинку в баннере (не удалять, не обновлять, не создавать) 
            geo_camp_vcard => 1|0 - 1 если баннеры сохраняются для кампании с типом geo
            moderate_declined => 1|0 - 1 повторная отправка на модерацию отклоненного баннера и/или визитки, даже если они не изменились
            ignore_callouts => 1|0 - не менять уточнения на баннере
            
    Результат:
        $banners - исходный массив [] баннеров

=cut

sub update_base_banners {
    
    my ($banners, $uid, $options) = @_;
    $options ||= {};

    # Проверим, что баннер обновляется в текстовой группе
    my $has_non_text_adgroup = get_one_field_sql(PPC(pid => [map { $_->{pid} } @$banners]), [
        q{SELECT 1 FROM phrases}, where => {pid => SHARD_IDS, adgroup_type__ne => 'base'}, "LIMIT 1"
    ]);
    croak "Cannot update banners in non-text adgroup using `update_base_banners`" if $has_non_text_adgroup;

    my $sql = qq[SELECT b.bid, b.type AS banner_type, b.title, b.title_extension, b.body, b.href, 
                        ifnull(b.domain,'') domain,
                        ifnull(vc.phone,'') phone, ifnull(vc.contactperson,'') contactperson,
                        ifnull(vc.worktime,'') worktime, ifnull(vc.street,'') street, ifnull(vc.house,'') house,
                        ifnull(vc.build,'') build, ifnull(vc.apart,'') apart, ifnull(vc.metro,'') metro,
                        ifnull(vc.city,'') city, ifnull(vc.country,'') country,
                        ifnull(vc.name,'') name, ifnull(vc.geo_id, 0) geo_id,
                        vc.im_client, vc.im_login, vc.extra_message, vc.contact_email, vc.org_details_id,
                        b.statusModerate, b.phoneflag, b.statusPostModerate, 
                        vc.address_id,
                        b.sitelinks_set_id, b.statusSitelinksModerate,
                        bdh.display_href, bdh.statusModerate as display_href_statusModerate,
                        bim.image_hash As image, bim.statusShow AS image_statusShow,
                        b.flags,
                        b.language
                   FROM banners b
                        LEFT JOIN vcards vc ON vc.vcard_id = b.vcard_id
                        LEFT JOIN banner_images bim USING(bid)
                        LEFT JOIN banner_display_hrefs bdh USING(bid)
    ];
    my @bids = map {$_->{bid}} @$banners;
    my $old_banners = get_hashes_hash_sql(PPC(uid => $uid), [$sql, WHERE => {'b.bid' => \@bids}]);
    
    croak "can't change banner type" if any { !$_->{banner_type} || $_->{banner_type} ne $old_banners->{$_->{bid}}->{banner_type} } @$banners;

    my $client_id = get_clientid(uid => $uid);
    my (@update_banners, @banners_to_redirect);
    my %update_groups;
    my %changed_hrefs_by_bids;

    # для старых баннеров выбираем дополнения (коллауты)
    my $old_bids = [keys %$old_banners];
    my $old_callouts = Direct::BannersAdditions->get_by(additions_type => "callout", get_banner_id => 1, banner_id => $old_bids)->items_callouts_by("banner_id");
    for my $bid (keys %$old_banners) {
        $old_banners->{$bid}->{callouts} = $old_callouts->{$bid} if exists $old_callouts->{$bid};
    }

    my $banners_to_fill_language = [];

    foreach my $banner (@$banners) {
        my $vcard = $banner->{vcard} || $banner;
        
        if (defined $banner->{metro} and ref $banner->{metro}) {
            $banner->{metro} = $banner->{metro}{region_id};
        }
        $vcard->{phone} = compile_phone(hash_cut $vcard, @VCards::PHONE_FIELDS);

        $banner->{href} = clear_banner_href($banner->{href}) if defined $banner->{href};
        $banner->{domain} = $banner->{domain_redir} = '' if !defined $banner->{href} || $banner->{href} eq '';
        $banner->{$_} = html2string($banner->{$_}) foreach qw/title title_extension body/;
        
        my $old_banner = $old_banners->{$banner->{bid}};
        my $old_vcard = $old_banners->{$banner->{bid}}->{vcard} || $old_banners->{$banner->{bid}};

        # переотправить фразы в БК после появления параметра «PhraseID»
        if ($banner->{href} && has_phraseid_param($banner->{href})
            && !has_phraseid_param($old_banner->{href} || '')) {

            $update_groups{$banner->{pid}}->{statusBsSynced} = 'No';
        }

        unless ($banner->{domain}) {
            my ($href_domain, $domain_need_check) = RedirectCheckQueue::domain_need_check_redirect(
                { href => $banner->{href} },
                hash_cut $old_banner, qw/domain href/
            );
            $banner->{domain} = $href_domain;
            push @banners_to_redirect, $banner if $domain_need_check;
        }
        
        my %changes = map { $_ => 0 } qw/banner vcard/;
        foreach (qw/title title_extension body href domain/) {
            if (($old_banner->{$_} || '') ne ($banner->{$_} || '')) {
                $changes{banner} = 1;
                last;
            }
        }

        $banner->{language} = $old_banner->{language};
        foreach (qw/title title_extension body/) {
            if (($old_banner->{$_} // '') ne ($banner->{$_} // '')) {
                $banner->{language} = $Direct::Model::Banner::Constants::BANNER_NO_LANGUAGE;
                push(@$banners_to_fill_language, $banner->{bid});
                last;
            }
        }

        if (($old_banner->{href} || '') ne ($banner->{href} || '')) {
            $changed_hrefs_by_bids{ $banner->{bid} } = $banner->{href};
        }
        $changes{vcard} = compare_vcards($vcard, $old_vcard) unless $options->{ignore_vcard};
        my $need_moderate_banner = check_moderate_banner($banner, $old_banner) || 
                                   check_moderate_banner_for_regions(BannerFlags::get_banner_flags_as_hash($old_banner->{flags}), $banner->{new_geo}, $banner->{old_geo}, {ClientID => $client_id});
        my $need_moderate_contactinfo = check_moderate_contactinfo($vcard, $old_vcard, geo_camp_vcard => $options->{geo_camp_vcard});
        
        if ($banner->{statusModerate} eq 'Ready' && $vcard->{phoneflag} eq 'New') {
            $need_moderate_contactinfo = 1;
        }
        
        if ($options->{moderate_declined}) {
            $need_moderate_banner = 1 if $old_banner->{statusModerate} eq 'No';
            $need_moderate_contactinfo = 1 if $old_banner->{phoneflag} eq 'No';
        }
        
        $banner->{reverse_domain} = reverse_domain( $banner->{domain} ) if $banner->{domain} ne '';
        foreach (qw/href domain reverse_domain/) {
            $banner->{$_} = undef if !defined $banner->{$_} || $banner->{$_} =~ m/^\s*$/s;
        };
        if (
            $need_moderate_banner
            || $need_moderate_contactinfo
            || (!$banner->{display_href} && $old_banner->{display_href}) 
            || any {$_ == 1} values %changes
        ) {
        
            my @columns;
            my $ignore_moderate = check_banner_ignore_moderate($banner);
            
            if ($changes{banner}) {
                push @columns, qw/title title_extension body href domain reverse_domain language/;
            }

            # отправляем баннер на модерацию, если 
            # 1) в нем были большие изменения
            # 2) изменилась визитка, а сам баннер отклонен ранее (для учета отклонений по ОГРН)
            if ($need_moderate_banner) {
                $banner->{statusModerate} = $ignore_moderate ? 'New' : 'Ready';
                
                if ($banner->{statusModerate} eq 'Ready') {
                    $update_groups{$banner->{pid}}->{statusModerate} = 'Ready';
                }
                $banner->{statusPostModerate} = 'No';
                $banner->{display_href_statusModerate} = 'Ready';
                push @columns, 'statusPostModerate', 'statusModerate';
            } else {
                $banner->{statusModerate} = $old_banner->{statusModerate};
                push @columns, 'statusModerate'; 
            }
            
            if (!$options->{ignore_vcard} && ($changes{vcard} || $need_moderate_banner || $need_moderate_contactinfo)) {

                $vcard->{org_details_id} = add_org_details(make_org_details_from_vcard($vcard, {uid => $uid}));
                $banner->{vcard_id} = $vcard->{vcard_id} = add_vcard_to_banner($banner->{bid}, $vcard, {dont_update_banners => 1})->{vcard_id};
                
                push @columns, 'vcard_id';
                if (!$banner->{vcard_id}) {
                    $banner->{phoneflag} = 'New';
                    push @columns, 'phoneflag'; 
                } elsif ($need_moderate_contactinfo || $need_moderate_banner) {
                    $banner->{phoneflag} = $ignore_moderate ? 'New' : 'Ready';
                    push @columns, 'phoneflag';
                } 
            }
            
            $banner->{statusBsSynced} = 'No';

            push @update_banners, {
                bid => $banner->{bid},
                cid => $banner->{cid},
                fields_digest => md5_hex_utf8(join '~', @columns),
                banner => hash_cut $banner, @columns, 'bid'
            };
        }
    }
    
    foreach_shard bid => \@update_banners, chunk_size => 800, sub {
        my ($shard, $banners) = @_;

        my %banners_by_digest;
        push @{$banners_by_digest{$_->{fields_digest}}}, $_->{banner} foreach @$banners;
        
        foreach my $banners (values %banners_by_digest) {

            my @fields = grep {$_ ne 'bid'} keys %{$banners->[0]};
            my %case_args;
            my @bids;
            foreach my $banner (@$banners) {
                foreach my $field (@fields) {
                    if ($field eq 'statusPostModerate') {
                        $case_args{$field}->{$banner->{bid}}
                            = sprintf "IF(statusPostModerate = 'Rejected', 'Rejected', %s)", $banner->{$field} ? sql_quote($banner->{$field}) : 'statusPostModerate';
                    } else {
                        $case_args{$field}->{$banner->{bid}} = $banner->{$field}
                    }
                }
                push @bids, $banner->{bid};
            }

            do_update_table(PPC(shard => $shard), 'banners',
                {
                    statusBsSynced => 'No',
                    map {
                        my $dont_quote = $_ =~ /^(statusPostModerate)$/;
                        # default__dont_quote => $_
                        $_ => sql_case(bid => $case_args{$_}, dont_quote_value => $dont_quote)
                    } keys %case_args
                },
                where => {bid => \@bids}, dont_quote => \@fields
            );
        }

        foreach my $banner (@$banners) {
            update_filter_domain($banner->{bid}, $banner->{banner}->{domain});
        }

        # удаляем версии объектов модерации для визиток, если они в статусе "черновик", или вообще отвязаны от баннеров
        my @bids_with_unlinked_vcard = map { $_->{bid} } grep { ($_->{banner}->{phoneflag} // '') eq 'New' } @$banners;
        VCards::delete_vcards_mod_versions(\@bids_with_unlinked_vcard);

    };
    _update_base_banner_addons($banners, $old_banners, $uid, hash_cut $options, qw/moderate_declined ignore_sitelinks ignore_image ignore_callouts/);

    my @update_bids;
    my (%camps_active_banners, %cids_statusModerate);
    foreach my $banner (@update_banners) {
        push @update_bids, $banner->{bid};
        if ($banner->{banner}->{statusModerate} ne 'New') {
            my $cid = $banner->{cid}; 
            unless (exists $camps_active_banners{$cid}) {
                $camps_active_banners{$cid} = has_camp_active_banners($cid);
            }
            $cids_statusModerate{$cid} = 1 if !$camps_active_banners{$cid};
        }
    }
    
    if (my @cids = keys %cids_statusModerate) { 
        do_update_table(PPC(uid => $uid), 'campaigns',
            {statusModerate => 'Ready'},
            where => {statusModerate => 'No', cid => \@cids});
    }
    
    clear_banners_moderate_flags(\@update_bids);
    # удаляем запись из mod_edit, чтобы в будущем не смущать пользователя тем, что его объявление редактирвоалось.
    delete_entry_edited_by_moderator('banner', \@update_bids);
    RedirectCheckQueue::push_banners(\@banners_to_redirect) if @banners_to_redirect;
    
    AggregatorDomains::update_aggregator_domains(\%changed_hrefs_by_bids);

    _update_group_values(\%update_groups, $uid);
    _on_base_banners_updated($banners, $old_banners, $uid);

    Direct::Model::Banner::LanguageUtils::add_banners_to_fill_language_queue($banners_to_fill_language);

    return $banners;
}

sub _on_base_banners_updated {
    
    my ($banners, $old_banners, $uid) = @_;
    
    my @notifications;
    foreach my $banner (@$banners) {
    
        my $old_banner = $old_banners->{$banner->{bid}};
    
        my $old_contacts_string = get_contacts_string($old_banner, "\n", {dont_include_ids => 1});
        my $new_contacts_string = get_contacts_string($banner, "\n", {dont_include_ids => 1});
        
        my $old_banner_only_text = join "\n", map { str $old_banner->{$_} } qw/title title_extension body href domain/;
        my $new_banner_only_text = join "\n", map { str $banner->{$_} } qw/title title_extension body href domain/;
    
        my $old_banner_text = join "\n", $old_banner_only_text, $old_contacts_string, Sitelinks::get_diff_string($old_banner->{sitelinks});
        my $new_banner_text = join "\n", $new_banner_only_text, $new_contacts_string, Sitelinks::get_diff_string($banner->{sitelinks});

        next if $old_banner_text eq $new_banner_text;

        $old_banner_text = html2string($old_banner_text);
        $new_banner_text = html2string($new_banner_text);

        push @notifications, {
            object     => 'banner',
            event_type => 'b_text', 
            object_id  => $banner->{bid}, 
            old_text   => $old_banner_text, 
            new_text   => $new_banner_text, 
            uid        => $uid,
        };
    }
    
    mass_mail_notification(\@notifications);
}

# сохранение у существующих баннеров дополнений (сайтлинки, картинка, дополнения)
sub _update_base_banner_addons {
    
    my ($banners, $old_banners, $uid, $options) = @_;
    $options ||= {};
    
    my $sitelinks = Sitelinks::get_sitelinks_by_set_id_multi([map { $_->{sitelinks_set_id} ? $_->{sitelinks_set_id} : () } values %$old_banners]);

    my $ClientID = get_clientid(uid => $uid);
    my (@assign_images, @remove_image_bids);
    my (@update_display_hrefs, %remove_display_hrefs, @remoderate_display_hrefs);
    my (%banner_values, %remove_sitelinks);
    my @bids_to_set_changed;
    foreach my $banner (@$banners) {

        my $old_banner = $old_banners->{$banner->{bid}};
        my $old_sitelinks = $old_banner->{sitelinks_set_id} ? $sitelinks->{$old_banner->{sitelinks_set_id}} : [];
        $old_banner->{sitelinks} = $old_sitelinks;
        
        my $need_moderate_sitelinks = check_moderate_sitelinks($banner, $old_banner);
        if ($options->{moderate_declined} && $old_banner->{statusSitelinksModerate} eq 'No') {
            $need_moderate_sitelinks = 1;
        }
        my $change_sitelinks = 0;
        unless ($options->{ignore_sitelinks} || !Sitelinks::compare_sitelinks($old_sitelinks, $banner->{sitelinks} || [])) {
            $change_sitelinks = 1;    
        }

        my $need_moderate_banner = $banner->{statusModerate} eq 'Ready' ? 1 : 0;

        if (!$options->{ignore_sitelinks} && ($change_sitelinks || $need_moderate_sitelinks || $need_moderate_banner)) {

            if ($banner->{href} && Sitelinks::need_save_sitelinks_set($banner->{sitelinks})) {
            
                $banner_values{$banner->{bid}} = {};    
                $banner->{sitelinks_set_id} = Sitelinks::save_sitelinks_set($banner->{sitelinks}, $ClientID);
                $banner_values{$banner->{bid}}->{sitelinks_set_id} = $banner->{sitelinks_set_id};
                
                if ($need_moderate_sitelinks || !scalar @$old_sitelinks || $need_moderate_banner) {
                    $banner->{statusSitelinksModerate} = $banner->{statusModerate} eq 'New' ? 'New' : 'Ready';
                    $banner_values{$banner->{bid}}->{statusSitelinksModerate} = $banner->{statusSitelinksModerate};
                }
            } elsif (!$banner->{href} || all { !(defined $_->{href} && $_->{href} ne '') && !(defined $_->{title} && $_->{title} ne '') } @{$banner->{sitelinks}}) {

                # пользователь удалил все сайтлинки (сознательно) или не вводил их до этого и не ввел сейчас или вообще убрал основную ссылку
                $banner->{sitelinks_set_id} = undef;
                $banner->{statusSitelinksModerate} = 'New';
                
                $banner_values{$banner->{bid}} = {
                    sitelinks_set_id => undef,
                    statusSitelinksModerate => 'New' 
                };
                
                push @{$remove_sitelinks{$banner->{cid}}}, $banner->{bid} if $old_banner->{sitelinks_set_id};
            }
        }

        my $change_image = 0;
        $change_image = ($old_banner->{image} || '') eq ($banner->{image} || '') ? 0 : 1 unless $options->{ignore_image};
        # если старая картинка была помечена как удаленная и добавляем новую
        $change_image = 1 if $banner->{image} && $old_banner->{image} && $old_banner->{image_statusShow} eq 'No';
        
        if (!$options->{ignore_image}) {
            if ($banner->{image}) {
                push @assign_images, {
                    bid => $banner->{bid},
                    cid => $banner->{cid},
                    ClientID => $ClientID,
                    image_hash => $banner->{image},
                    statusModerate => $banner->{statusModerate} eq 'New' ? 'New' : 'Ready',
                } if $change_image || $need_moderate_banner;
            } elsif ($change_image) {
                push @remove_image_bids, $banner->{bid};
            }
        }

        # отображаемый урл
        if (exists $banner->{display_href} && ($banner->{display_href} || '') ne ($old_banner->{display_href} || '')) {
            push @bids_to_set_changed, $banner->{bid};
            if ($banner->{display_href}) {
                push @update_display_hrefs, [
                    $banner->{bid},
                    $banner->{display_href},
                    $banner->{display_href_statusModerate} || 'Ready',
                ];
            }
            elsif ($old_banner->{display_href}) {
                push @{$remove_display_hrefs{$banner->{cid}}}, $banner->{bid};
            }
        }
        elsif ($need_moderate_banner) {
            push @remoderate_display_hrefs, $banner->{bid};
        }
    }

    mass_banner_assign_image(@assign_images) if @assign_images; 
    mass_banner_remove_image(\@remove_image_bids) if @remove_image_bids;

    if (@update_display_hrefs) {
        do_mass_insert_sql(PPC(uid => $uid),
            'INSERT INTO banner_display_hrefs(bid, display_href, statusModerate) VALUES %s
            ON DUPLICATE KEY UPDATE
            display_href=VALUES(display_href),
            statusModerate=VALUES(statusModerate)',
            \@update_display_hrefs,
        );
    }
    if (%remove_display_hrefs) {
        my @bids = map {@$_} values %remove_display_hrefs;
        do_delete_from_table(PPC(uid => $uid), 'banner_display_hrefs', where => { bid => \@bids });
        do_delete_from_table(PPC(uid => $uid), 'mod_object_version', where => {obj_type => 'display_href', obj_id => \@bids});
    }
    if (@remoderate_display_hrefs) {
        do_update_table(PPC(uid => $uid), 'banner_display_hrefs',
            { statusModerate => 'Ready' },
            where => { bid => \@remoderate_display_hrefs },
        );
    }

    if (@bids_to_set_changed) {
        do_update_table(PPC(uid => $uid),
            banners => {LastChange__dont_quote => 'NOW()'},
            where => {bid => \@bids_to_set_changed},
        );
    }

    # обновляем дополнения к баннерам
    if (! $options->{ignore_callouts}) {
        _mass_update_banners_additions($banners, $old_banners, $ClientID);
    }

    _update_banner_values(\%banner_values, $uid);
    while (my ($cid, $bids) = each %remove_sitelinks) {
        Sitelinks::delete_sitelinks_mod_versions($bids);
    }
}

=head2 _mass_update_banners_additions

Обновление/создание дополнений к баннерам
$banners -- список баннеров с полем callouts (список строк уточнений):
    [
        {bid => NNN,
         callouts => ["CCC1", "CCC2", ...],
         ...
        },
        ...
    ]

$old_banners -- хеш со старыми баннерами

perl -MModels::Banner -MDDP -e 'Models::Banner::_mass_update_banners_additions([{bid => 1824065, callouts => ["callout 1", "callout 2", "callout 4"]}], $old_banners, 385179)'

=cut

sub _mass_update_banners_additions($$$) {
    my ($banners, $old_banners, $ClientID) = @_;

    my $items_callouts = [];
    my $banners_without_callouts = {};

    for my $banner (@$banners) {

        my $old_callouts_by_text = {};
        if (exists $old_banners->{$banner->{bid}} && exists $old_banners->{$banner->{bid}}->{callouts}) {
            for my $old_item (@{ $old_banners->{$banner->{bid}}->{callouts} }) {
                $old_callouts_by_text->{$old_item->callout_text} = $old_item;
            }
        }
        my $old_callouts_all_texts = join("\n", sort keys %$old_callouts_by_text);
        my $new_callouts_all_texts =
            ref($banner->{callouts}) eq 'ARRAY' && @{$banner->{callouts}}
            ? join("\n", sort map {$_->{callout_text}} @{$banner->{callouts}})
            : "";

        if ($old_callouts_all_texts ne $new_callouts_all_texts && length($new_callouts_all_texts) > 0) {
            # уточнения поменялись
            for my $callout_text (map {$_->{callout_text}} @{$banner->{callouts}}) {

                my $one_callout;
                if (exists $old_callouts_by_text->{$callout_text}) {
                    $one_callout = $old_callouts_by_text->{$callout_text};
                } else {
                    $one_callout = Direct::Model::AdditionsItemCallout->new();
                    $one_callout->banner_id($banner->{bid});
                    $one_callout->client_id($ClientID);
                    $one_callout->callout_text($callout_text);
                }

                push @$items_callouts, $one_callout;
            }
        } elsif (length($old_callouts_all_texts) > 0 && length($new_callouts_all_texts) == 0) {
            # раньше были уточнения, а теперь убрали
            $banners_without_callouts->{$banner->{bid}} = [];
        }
    }

    if (@$items_callouts) {
        my $banner_additions = Direct::BannersAdditions->new(items_callouts => $items_callouts);
        $banner_additions->save();
    }

    # отвязываем уточнения если их все удалили из баннера
    if (%$banners_without_callouts) {
        Direct::BannersAdditions::link_to_banners($banners_without_callouts, "callout")
    }
}

# _update_group_values({pid => {statusBsSynced => 'No', statusModerate => 'Ready'}}, $uid)
sub _update_group_values {
    
    my ($group_values, $uid) = @_;

    my @pids = keys %$group_values;
    return unless @pids; 
    
    my %case_values;
    foreach my $pid (@pids) {
        while (my ($field, $value) = each %{$group_values->{$pid}}) {
            if ($field eq 'statusModerate') {
                $case_values{statusModerate}->{$pid}
                    = sprintf "IF(statusModerate = 'New', %s, statusModerate)", sql_quote($value); 
            } else {
                $case_values{$field}->{$pid} = $value;
            }
        } 
    }
    
    do_update_table(PPC(uid => $uid), 'phrases',
        {
            LastChange__dont_quote => 'LastChange',
            map {
                $_ => sql_case(pid => $case_values{$_}, default__dont_quote => $_, dont_quote_value => $_ eq 'statusModerate') 
            } keys %case_values
        },
        where => {pid => \@pids}, dont_quote => [keys %case_values]
    );
}


=head2 delete_entry_edited_by_moderator(bid)

    Удаление данных из mod_edit о том, что баннер редактирвоался модератором

=cut

sub delete_entry_edited_by_moderator {
    my ($type, $id) = @_;
    my %shard_keys = (banner => 'bid');

    die "Can not determine shard for $type / $id" unless $shard_keys{$type};

    return do_sql(PPC($shard_keys{$type} => $id), ["delete from mod_edit", 'where'=>{type => $type, id => SHARD_IDS}]);
}

=head2 has_camp_active_banners

    Определяет, есть ли у кампании активные баннеры.
    На входе cid и mediaType, на выходе true | false

=cut

sub has_camp_active_banners {
    my ($cid, $mediaType) = @_;

    return mass_has_camps_active_banners([$cid], {$cid => $mediaType})->{$cid};
}

=head2 mass_has_camps_active_banners

    Определяет по списку id кампаний, есть ли у каждой из кампаний активные баннеры.
    На входе arrayref cids и hashref cid => mediaType, на выходе структура вида { сid => true|false, }

=cut

sub mass_has_camps_active_banners {
    my ($cids, $type_by_cid) = @_;

    return {} unless scalar @$cids;

    my @text_campaign_cids;
    my @media_campaign_cids;
    while ( my ($cid, $type) = each %$type_by_cid ) {
        if (camp_kind_in(type => $type, 'base')) {
            push @text_campaign_cids, $cid;
        } elsif (is_media_camp(type => $type)) {
            push @media_campaign_cids, $cid;
        }
    }

    my $text_campaigns = {};
    if (@text_campaign_cids) {
        my $is_enabled = ($ENABLE_NEW_CAMPS_HAS_ACTIVE_BANNERS_REQUEST_PROP->get(60) // 0);
        if ($is_enabled) {
            foreach my $cids_chunk (chunks(\@text_campaign_cids, $CIDS_CHUNK_SIZE)) {
                LogTools::log_messages("has_active_banners_query_new", "cids=".join(",", @$cids_chunk));
                my %condition = Models::AdGroupFilters::get_status_condition('active', filter => {cid => $cids_chunk});
                my $text_campaigns_chunk = get_hash_sql(PPC(cid => $cids_chunk), ["
                                                            SELECT g.cid
                                                            FROM phrases g",
                                                            $condition{tables},
                                                            WHERE => {'g.cid' => SHARD_IDS,
                                                                      _TEXT => sql_condition($condition{where}),
                                                            } ]);
                hash_merge($text_campaigns, $text_campaigns_chunk);
            }
        } else {
            LogTools::log_messages("has_active_banners_query", "cids=".join(",", @text_campaign_cids));
            $text_campaigns = get_hash_sql(
                PPC(cid => \@text_campaign_cids),
                [
                    'SELECT c.cid FROM campaigns c',
                    WHERE => {  
                        'c.cid' => SHARD_IDS,                  
                        _TEXT   => "EXISTS (SELECT 1 FROM banners b JOIN phrases ph USING(pid) WHERE b.cid = c.cid AND $MTools::IS_ACTIVE_CLAUSE)",
                    }
                ]
            );
        }
    }

    my $media_campaigns = {};
    if (@media_campaign_cids) {
        $media_campaigns = get_hash_sql(
            PPC(cid => \@media_campaign_cids),
            [
                'SELECT g.cid FROM media_banners b JOIN media_groups g USING(mgid)',
                WHERE => {
                    _TEXT   => $MTools::IS_ACTIVE_MEDIA_CLAUSE,
                    'g.cid' => SHARD_IDS,
                }
            ]
        );
    }

    return { map { $_ => 1 } ( keys %$text_campaigns, keys %$media_campaigns ) };
}

=head2 check_banner_ignore_moderate

    Получить флаг, игнорировать ли модерацию для этого баннера, на вход должны поступить:
    StatusEmpty из кампании
    StatusModerate из баннера
    
=cut

sub check_banner_ignore_moderate {
    my $banner = shift;
    return ($banner->{statusEmpty}||'') eq "Yes" || ($banner->{statusModerate}||'') eq "New";
}

=head2 has_phraseid_param(href)

    Определяет, используется ли в href параметр «PhraseID».

=cut

sub has_phraseid_param($) {
    my ($href) = @_;

    return !!($href =~ /\{phrase\_?id|param127|retargeting_id\}/i); 
}

=head2 has_delete_banner_problem($banner)

    Определяет может ли быть удален баннер, если не может, то возвращает текст причины.
    
    Параметры:
        (перичесленные поля обязательны)
        $group - группа объявлений 
            { sum => , statusModerate => , statusPostModerate => }
            sum - деньги на кампанию (всего)
        $banner - проверяемый баннер, имеющий поля: 
                  sum, camp_in_bs_queue, banner_statusPostModerate, banner_statusModerate,
                  group_statusPostModerate, banner_statusPostModerate, BannerID
                  Извлекаются из БД при помощи функции get_banners_for_delete()
        
    Результат:
        0 - баннер может быть удалён
        текст ошибки - не может быть удалён

=cut

    # тоже самое условие есть в data/t/campaigninfo.html и data/block/i-direct-easy/i-direct-easy.tt2
    # и XLSCampaign::apply_camp_group_snapshot()

sub has_delete_banner_problem {
    
    my $banner = shift;
    return iget('Ошибка! Неправильный номер баннера.') if !is_valid_id($banner->{bid});
    if (
        $banner->{BannerID} != 0
        || ($banner->{camp_in_bs_queue} && !($banner->{banner_statusPostModerate} eq 'No' && $banner->{banner_statusModerate} eq 'New'))
        || ($banner->{banner_statusModerate} eq 'Yes' && $banner->{group_statusModerate} eq 'Yes'
                || $banner->{banner_statusPostModerate} =~ /^(Yes|Rejected)$/
                # если баннер добавлен в существующую группу
                || $banner->{group_statusPostModerate} =~ /^(Yes|Rejected)$/ && $banner->{banner_statusPostModerate} ne "No"
            ) && $banner->{sum} > 0
        || (defined $banner->{group_priority} && $banner->{group_priority} == $Settings::DEFAULT_CPM_PRICE_ADGROUP_PRIORITY)
    ) {
        return iget("Удаление баннера %s невозможно", $banner->{bid});
    }
    return 0;
}


=head2 fill_in_can_delete_banner($banners)

    Прописывает в баннеры признак can_delete_banner

    Параметры:
        $banners - [ { bid => N, ... }, ... ]

=cut

sub fill_in_can_delete_banner {
    my ($banners) = @_;

    my $banners_for_del = get_banners_for_delete({bid => [ map {$_->{bid}} @$banners ]});
    my %can_del = map {($_->{bid} => !has_delete_banner_problem($_))} @$banners_for_del;

    for my $banner (@$banners) {
        $banner->{can_delete_banner} = $can_del{$banner->{bid}};
    }

    return;
}

=head2 does_client_have_image($client_id, $image_hash)
    
Определяет, что картинка есть у клиента

Возвращает image_type

=cut

sub does_client_have_image {
    my ($client_id, $image_hash) = @_;

    my $image_type = get_one_field_sql( PPC(ClientID => $client_id), [
            'SELECT image_type
            FROM banner_images_pool bip
            JOIN banner_images_formats bif USING(image_hash)',
            WHERE => {
                ClientID => $client_id,
                image_hash => $image_hash,
            },
            LIMIT => 1,
        ]);
    return $image_type;
}


=head2 add_banner_error_text

    Используется для сбора ошибок с привязками к названию полей в баннерах
    Добавляет тексты ошибок в хеш вида:
    { '1234' => { 'contactinfo' => ['Нет телефона', 'Нет имени представителя'],
                  'title'       => ['Неправильные символы'],
                },
      '5678' => {...}
    }

=cut

sub add_banner_error_text
{
    my ($errors, $bid, $field, $error_text, %options) = @_;
    my $error_texts = (ref ($error_text) eq 'ARRAY') ? $error_text : [$error_text];
    my @msgs = grep {$_} @$error_texts; 
    return unless scalar @msgs;
    $errors ||= {};
    $errors->{$bid} ||= {};
    $errors->{$bid}->{$field} ||= [];
    
    @msgs = map {$options{prefix} . ' ' . $_} @msgs if $options{prefix};
    push @{$errors->{$bid}->{$field}}, @msgs;
    return $errors;
}

=head2 convert_banner_error_text_to_list

    Старая система показа ошибок подразумевает складирование всех ошибок в массив, а затем их показ.
    При проверки баннера сейчас используется система хешей, где для каждого баннера для каждого поля прописан список ошибок.
    С переходом на новую систему всего кода планируется потом от этой функции избавиться.
    Данная функция конвертирует ошибки из системы хешей в список.

=cut
sub convert_banner_error_text_to_list
{
    my $errors = shift;

    my @result = ();
    for my $error (values (%{$errors})) {
        push @result, map {@{$_}} values(%{$error});
    }

    return @result;
}


=head2 validate_banner_title

    Проверяет заголовок баннера.
    За исключением текстов ошибок и констант брат-близнец функции validate_banner_body.

    $error = validate_banner_title($banner_title);
    $error = validate_banner_title($banner_title, can_be_empty=>1);
    $error => undef || $error_text

=cut

sub validate_banner_title {
    my ($title, %options) = @_;

    return iget('Поле обязательно для заполнения') unless $title || $options{can_be_empty}; 

    my $count_title = scalar(my @tmp1 = $title =~ /$TEMPLATE_METKA/g);
    return iget("В Заголовке 1 можно использовать шаблон только один раз") if $count_title > 1;
    my $MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1 = $MAX_TITLE_UNINTERRUPTED_LENGTH + 1;

    if (lc($title) =~ $DISALLOW_BANNER_LETTER_RE) {
        return iget('В Заголовке 1 можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры и знаки пунктуации');
    }

    my $count_of_narrow_symbols = (my $title_without_narrow_symbols = $title) =~ s/$NARROW_SYMBOLS_RE//g;
    my $length_without_narrow_symbols = length($title_without_narrow_symbols);

    state $increase_ad_text_limits_prop = Property->new('increase_ad_text_limits');
    my $use_new_ad_text_limits = $increase_ad_text_limits_prop->get(120);
    my $max_text_length = $use_new_ad_text_limits ? $NEW_MAX_TITLE_LENGTH : $MAX_TITLE_LENGTH;

    my $title_length = $use_new_ad_text_limits ? length($title) : $length_without_narrow_symbols;

    if ($title_length > $max_text_length + 2 * $count_title) {
        return iget('Максимальная длина Заголовка 1 превышена');
    } elsif (!$use_new_ad_text_limits && $count_of_narrow_symbols > $MAX_NUMBER_OF_NARROW_CHARACTERS) {
        my $msg = get_word_for_digit($MAX_NUMBER_OF_NARROW_CHARACTERS,
            iget_noop('В Заголовке 1 вы можете использовать не более %d точки, запятой, двоеточия, точки с запятой, кавычки и восклицательного знака'),
            iget_noop('В Заголовке 1 вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
            iget_noop('В Заголовке 1 вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
        );
        return iget($msg, $MAX_NUMBER_OF_NARROW_CHARACTERS);
    }elsif ($title =~ m/([^ \#\-]{$MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1,})/) {
        my $exceeding_word = $1;
        my $truncated_exceeding_word = TextTools::truncate_text($exceeding_word, $MAX_TITLE_UNINTERRUPTED_LENGTH+2, '...');
        my $msg = get_word_for_digit($MAX_TITLE_UNINTERRUPTED_LENGTH,
            iget_noop('В Заголовке 1 недопустимы слова длиной более %d символа. Слишком длинные слова: "%s".'),
            iget_noop('В Заголовке 1 недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
            iget_noop('В Заголовке 1 недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
        );
        return iget($msg, $MAX_TITLE_UNINTERRUPTED_LENGTH, $truncated_exceeding_word);
    } elsif ($title =~ m/[$SPACES],[^$SPACES]/) {
        return iget('Неправильное использование знаков препинания в Заголовке 1');
    } else {
        return undef;
    }
}

=head2 validate_banner_title_extension

    Проверяет заголовок баннера.
    За исключением текстов ошибок и констант брат-близнец функции validate_banner_body.

    $error = validate_banner_title_extension($banner_title_extension);
    $error => undef || $error_text

=cut

sub validate_banner_title_extension {
    my ($title_extension) = @_;

    return undef unless $title_extension;

    my $count_title_extension = scalar(my @tmp1 = $title_extension =~ /$TEMPLATE_METKA/g);
    return iget("В Заголовке 2 можно использовать шаблон только один раз") if $count_title_extension > 1;
    my $MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1 = $MAX_TITLE_UNINTERRUPTED_LENGTH + 1;

    if (lc($title_extension) =~ $DISALLOW_BANNER_LETTER_RE) {
        return iget('В Заголовке 2 можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры и знаки пунктуации');
    }

    my $count_of_narrow_symbols = (my $title_extension_without_narrow_symbols = $title_extension) =~ s/$NARROW_SYMBOLS_RE//g;
    my $length_without_narrow_symbols = length($title_extension_without_narrow_symbols);

    if ($length_without_narrow_symbols > $MAX_TITLE_EXTENSION_LENGTH + 2 * $count_title_extension) {
        return iget('Максимальная длина Заголовка 2 превышена');
    } elsif ($count_of_narrow_symbols > $MAX_NUMBER_OF_NARROW_CHARACTERS) {
        my $msg = get_word_for_digit($MAX_NUMBER_OF_NARROW_CHARACTERS,
            iget_noop('В Заголовке 2 вы можете использовать не более %d точки, запятой, двоеточия, точки с запятой, кавычки и восклицательного знака'),
            iget_noop('В Заголовке 2 вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
            iget_noop('В Заголовке 2 вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
        );
        return iget($msg, $MAX_NUMBER_OF_NARROW_CHARACTERS);
    }elsif ($title_extension =~ m/([^ \#\-]{$MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1,})/) {
        my $exceeding_word = $1;
        my $truncated_exceeding_word = TextTools::truncate_text($exceeding_word, $MAX_TITLE_UNINTERRUPTED_LENGTH+2, '...');
        my $msg = get_word_for_digit($MAX_TITLE_UNINTERRUPTED_LENGTH,
            iget_noop('В Заголовке 2 недопустимы слова длиной более %d символа. Слишком длинные слова: "%s".'),
            iget_noop('В Заголовке 2 недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
            iget_noop('В Заголовке 2 недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
        );
        return iget($msg, $MAX_TITLE_UNINTERRUPTED_LENGTH, $truncated_exceeding_word);
    } elsif ($title_extension =~ m/[$SPACES],[^$SPACES]/) {
        return iget('Неправильное использование знаков препинания в Заголовке 2');
    } else {
        return undef;
    }
}

=head2 validate_banner_body

    Проверяет текст баннера.
    За исключением текстов ошибок и констант брат-близнецvalidate_banner_body функции validate_banner_title.

    $error = validate_banner_body($banner_body);
    $error = validate_banner_body($banner_body, can_be_empty=>1);
    $error => undef || $error_text

=cut

sub validate_banner_body {
    my ($body, %options) = @_;

    return iget('Поле обязательно для заполнения') unless $body || $options{can_be_empty}; 

    my $count_body = scalar(my @tmp2 = $body =~ /$TEMPLATE_METKA/g);
    return iget("В тексте объявления можно использовать шаблон только один раз") if $count_body > 1;
    
    my $MAX_BODY_UNINTERRUPTED_LENGTH_PLUS_1 = $MAX_BODY_UNINTERRUPTED_LENGTH + 1;

    if (lc($body) =~ $DISALLOW_BANNER_LETTER_RE) {
        return iget('В тексте рекламного сообщения можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры и знаки пунктуации');
    }

    my $count_of_narrow_symbols = (my $body_without_narrow_symbols = $body) =~ s/$NARROW_SYMBOLS_RE//g;
    my $length_without_narrow_symbols = length($body_without_narrow_symbols);

    if ($length_without_narrow_symbols > $MAX_BODY_LENGTH + 2 * $count_body) {
        return iget('Максимальная длина рекламного текста превышена');
    } elsif ($count_of_narrow_symbols > $MAX_NUMBER_OF_NARROW_CHARACTERS) {
        my $msg = get_word_for_digit($MAX_NUMBER_OF_NARROW_CHARACTERS,
            iget_noop('В рекламном тексте вы можете использовать не более %d точки, запятой, двоеточия, точки с запятой, кавычки и восклицательного знака'),
            iget_noop('В рекламном тексте вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
            iget_noop('В рекламном тексте вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
        );
        return iget($msg, $MAX_NUMBER_OF_NARROW_CHARACTERS);
    } elsif ($body =~ m/([^ \#\-]{$MAX_BODY_UNINTERRUPTED_LENGTH_PLUS_1,})/) {
        my $exceeding_word = $1;
        my $truncated_exceeding_word = TextTools::truncate_text($exceeding_word, $MAX_BODY_UNINTERRUPTED_LENGTH+2, '...');
        my $msg = get_word_for_digit($MAX_BODY_UNINTERRUPTED_LENGTH,
            iget_noop('В рекламном тексте недопустимы слова длиной более %d символа. Слишком длинные слова: "%s".'),
            iget_noop('В рекламном тексте недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
            iget_noop('В рекламном тексте недопустимы слова длиной более %d символов. Слишком длинные слова: "%s".'),
        );
        return iget($msg, $MAX_BODY_UNINTERRUPTED_LENGTH, $truncated_exceeding_word);
    } elsif ($body =~ m/[$SPACES],[^$SPACES]/) {
        return iget('Неправильное использование знаков препинания в тексте баннера');
    } else {
        return undef;
    }
}

=head2 validate_banner_type($banner, $exists_banners_type, $skip_changing_banner_type, $is_mediaplan_banner)

    Проверка типа баннера
    
    Параметры:
        $banner - хеш {} c полями bid|mbid, banner_type
        $exists_banners_type - хеш {bid1 => 'desktop', bid2 => 'mobile'} содержащий текущие типы баннеров (ключ bid, значение - текущий тип баннера)
        $skip_changing_banner_type - пропустить проверку невозможности изменения типа баннера
        $is_mediaplan_banner - проверяемый баннер медиаплановый (будет использоваться mbid вместо bid)
    
    Результата:    
        $error_text || undef
        undef - проверка выполнена успешно

=cut

sub validate_banner_type {

    my ($banner, $exists_banners_type, $skip_changing_banner_type, $is_mediaplan_banner) = @_;
    
    my $bid = $is_mediaplan_banner ? $banner->{mbid} : $banner->{bid}; 	
    if (!$banner->{banner_type}) {
        return iget('Задайте тип баннера');
    } elsif ($banner->{banner_type} !~ /^(desktop|mobile)$/) {      
        return iget('Тип баннера указан неверно (разрешены мобильный или десктопный баннеры)');      
    } elsif ($bid && !$skip_changing_banner_type) {
        my $old_banner_type = $exists_banners_type && exists $exists_banners_type->{$bid} ? $exists_banners_type->{$bid} || '' : '';        
        if ($banner->{banner_type} ne $old_banner_type) {
           return $old_banner_type eq 'mobile'
                    ? iget('Невозможно изменить тип баннера с мобильного на десктопный')
                    : iget('Невозможно изменить тип баннера с десктопного на мобильный');
        }
    }

    return undef;
}

=head2 validate_banner($banner, $adgroup, $options)

    Параметры:
        $banner - {}

        $options
            exists_banners_type     -- в случае обновления баннера (if $banner->{bid}) этот параметр обязателен,
                                        хранит типы существующих баннеров {bid => 'desktop', bid2 => 'mobile' ...}
            skip_changing_banner_type -- пропустить проверку на невозможность смены типа баннера, но непосредственно тип баннера проверяться будет
            skip_contactinfo        -- если установлен, то КИ будет проигнорирована
            skip_max_phrases_length -- если установлен, то не будет проводиться проверка суммарной максимальной длины фраз в баннере
            use_banner_with_flags   -- если установлен, то ссылка и КИ будут проверены в соответствии с $banner->{banner_with_(href|phone)}
            ClientID                -- ClientID клиента для гео ф-ций для выбора транслокально дерева регионов
            is_api                  -- вызов из API для выбора API-транслокально дерева регионов
	
            если установлен флаг use_banner_with_flags, то контактная информация должна быть в $banner->{vcard}

    Результат:   
        {title => ['', ''], body => ['', '']} - хеш текстовых ошибок по полям        

=cut

sub validate_banner {
    my ($banner, $adgroup, $options) = @_;

    my %errors=();
    my $bid = $banner->{bid} || 0;
    # strip bad symbols.
    smartstrip($banner->{$_}, dont_replace_angle_quotes => 1) for qw/body title title_extension/;
    smartstrip($banner->{href});

    if (($banner->{ad_type} ne 'image_ad' && $banner->{ad_type} ne 'cpc_video') && ($adgroup->{adgroup_type} ne 'mobile_content')) {
        my $title_error = validate_banner_title($banner->{title});
        push @{$errors{title}}, $title_error if $title_error;

        my $title_extension_error = validate_banner_title_extension($banner->{title_extension});
        push @{$errors{title_extension}}, $title_extension_error if $title_extension_error;

        my $banner_body_error = validate_banner_body($banner->{body});
        push @{$errors{body}}, $banner_body_error if $banner_body_error;
    }
    elsif ($banner->{ad_type} eq 'image_ad') {
        my $imagead_error = validate_banner_imagead($banner, $adgroup, $options);
        if ($imagead_error) {
            push @{$errors{image_ad}}, $imagead_error;
        }
    }
    
    # работаем с автоматически определяемым доменом
    if ($options->{check_href_available} && $banner->{href}) {
        my $domain_change_error = url_domain_getsign($banner, $options->{login_rights});
        push @{$errors{href}}, $domain_change_error if $domain_change_error; 
    }

    # для баннеров РМП валидация шаблонов осуществляется внутри Direct::Validation::BannersMobileContent::validate_mobile_banners
    if (!(($banner->{adgroup_type} // '') eq 'mobile_content' && $banner->{ad_type} ne 'image_ad')) {
        my $banners_template_errors = validate_banner_template($banner, $adgroup->{phrases});
        push @{$errors{body}}, @$banners_template_errors if @$banners_template_errors;
    }

    my $type_error = validate_banner_type($banner, $options->{exists_banners_type}, $options->{skip_changing_banner_type});
    push @{$errors{banner_type}}, $type_error if $type_error;

    # Проверяем ссылку...
    if( defined $banner->{href} && (!$options->{use_banner_with_flags} || $banner->{has_href}) ) {
        $banner->{href} = clear_banner_href($banner->{href}, $banner->{url_protocol});
        my @banner_href_errors = validate_banner_href($banner->{href});
        push @{$errors{href}}, @banner_href_errors if @banner_href_errors;
    } 

    # ... отображаемую ссылку...
    if ($banner->{display_href}) {
        push @{$errors{href}}, iget("Задана отображаемая ссылка, но не указана основная ссылка в объявлении")  if !$banner->{href};
        if (my $dh_error = Direct::Validation::Banners::validate_banner_display_href($banner->{display_href} || undef)) {
            # создаём ValidationResult для обработки шаблонов
            my $vr = Direct::ValidationResult->new();
            $vr->add_generic($dh_error);
            $vr->process_descriptions(__generic => {field => iget("'Отображаемая ссылка'")});
            push @{$errors{display_href}}, $vr->get_first_error_description;
        }
    }

    # ... и пермалинки организаций Справочника
    if ($banner->{permalink} && !$options->{valid_permalinks}{$banner->{permalink}}) {
        push @{$errors{permalink}}, iget('Организация не найдена');
    }

    # ... и сайтлинки
    if (!$options->{ignore_sitelinks} && ref $banner->{sitelinks} eq 'ARRAY' && scalar @{$banner->{sitelinks}}) {
        foreach (@{$banner->{sitelinks}}) {
            $_->{href} = clear_banner_href($_->{href}, $_->{url_protocol});
        }
        my @sitelinks_errors = Sitelinks::validate_sitelinks_set(
            $banner->{sitelinks},
            $banner->{href},
            banner_turbolanding => $banner->{turbolanding},
            ClientID => $options->{ClientID},
        );
        if ($options->{check_href_available}) {
            # Проверяем на доступность быстрые ссылки
            for my $sitelink (@{$banner->{sitelinks}}) {
                my $result_check = get_url_domain($sitelink->{href}, {check_for_http_status => 1} );
                if (!$result_check->{res}) {
                    push @sitelinks_errors, $result_check->{msg};
                    last;
                }
            }
        }
        push @{$errors{sitelinks}}, @sitelinks_errors if @sitelinks_errors;
    }

    # проверяем уточнения
    if (ref($banner->{callouts}) eq 'ARRAY') {
        my $additions_callouts = [
            map {
                Direct::Model::AdditionsItemCallout->new(
                    client_id => $options->{ClientID},
                    callout_text => $_->{callout_text},
                )
            }
            @{ $banner->{callouts} }
        ];
        my $banner_model = Direct::Model::Banner->new(additions_callouts => $additions_callouts);
        my $callouts_errors = Direct::Validation::BannersAdditions::validate_banner_link_callout($banner_model);
        push @{$errors{callouts}}, uniq @{$callouts_errors->get_error_descriptions()} if ! $callouts_errors->is_valid;
    }

    # Проверяем минус-слова
    if (defined $banner->{banner_minus_words}) {
        if (my @x = @{MinusWords::check_minus_words($banner->{banner_minus_words}, type => 'banner')}) {
            push @{$errors{banner_minus_words}}, @x;
        }
    }

    # Нельзя указываать сайтлинки, если отсутствует основная ссылка
    if( (!$options->{use_banner_with_flags} || $banner->{has_href}) &&
        (!defined $banner->{href} || $banner->{href} eq '') &&
        (!$options->{ignore_sitelinks} && Sitelinks::need_save_sitelinks_set($banner->{sitelinks}))
    ) {
        push @{$errors{href}}, iget("Отсутствует основная ссылка при наличии быстрых");
    }
    
    if (($banner->{adgroup_type} // '') eq 'mobile_content' && !$errors{href}) {
        my $banner_model = new Direct::Model::BannerMobileContent(
            title => $banner->{title}, body => $banner->{body},
            href => defined $banner->{href} && length($banner->{href}) > 0 ? $banner->{href} : undef,  
            sitelinks_set_id => undef,
            client_id => $options->{ClientID},
            language => $banner->{language} || $Direct::Model::Banner::Constants::BANNER_NO_LANGUAGE,
        );

        my ($cid, $content_lang) = (defined $adgroup->{campaign}) ?
                                ($adgroup->{campaign}->{cid}, $adgroup->{campaign}->{content_lang}) :
                                ($adgroup->{cid}, CampaignQuery->get_campaign_data(cid => $adgroup->{cid}, [qw/content_lang/], get_also_empty_campaigns => 1)->{content_lang});

        my $campaign_model = new Direct::Model::Campaign(id => $cid, content_lang=>$content_lang);
        my $group_model = new Direct::Model::AdGroupMobileContent(geo => 0, campaign => $campaign_model);

        my $vr = Direct::Validation::BannersMobileContent::validate_mobile_banners([$banner_model], $group_model);
        my $href_vr = $vr->get_objects_results->[0]->get_field_result('href');
        push @{$errors{href}}, @{$href_vr->get_error_texts} if $href_vr && !$href_vr->is_valid;
        if ($banner->{ad_type} ne 'image_ad' && $banner->{ad_type} ne 'cpc_video') {
            $vr->process_objects_descriptions(Direct::Banners->WEB_MOBILE_CONTENT_FIELD_NAMES);
            my $body_vr = $vr->get_objects_results->[0]->get_field_result('body');
            push @{$errors{body}}, @{$body_vr->get_error_descriptions} if $body_vr && !$body_vr->is_valid;
            my $title_vr = $vr->get_objects_results->[0]->get_field_result('title');
            push @{$errors{title}}, @{$title_vr->get_error_descriptions} if $title_vr && !$title_vr->is_valid;
        }
    }

    # ... и контактную информацию 
    my $vcard = $banner->{vcard} || $banner;
    if ($options->{use_banner_with_flags}) {
        # со временем из всего if'а должна остаться только эта ветка, а $options->{use_banner_with_flags} станет не нужна 
        if ( ($banner->{has_vcard} || $banner->{banner_with_phone}) &&  !$options->{skip_contactinfo}) {
            my @contactinfo_errors;
            if ($banner->{vcard} && $banner->{vcard}->{_vcard_validation_result}) {
                @contactinfo_errors = @{ $banner->{vcard}->{_vcard_validation_result} }; 
            } else {
                @contactinfo_errors = validate_contactinfo($vcard);
            }
            push @{$errors{contactinfo}}, @contactinfo_errors if @contactinfo_errors;
        }

        # для рекламы мобильных можно не указывать href
        if (!$banner->{adgroup_type} || $banner->{adgroup_type} ne 'mobile_content') { 
            push @{$errors{href}}, iget('Не введена ссылка на сайт')
                if ( $banner->{has_href} &&
                    !($banner->{turbolanding} || $banner->{turbolanding_id}) &&
                    ( !defined $banner->{href} || $banner->{href} !~ /\S/ ) &&
                    ( !$banner->{permalink} )
                );
            push @{$errors{href}}, iget('Не введена ссылка на сайт')
                if (
                    !$banner->{has_href}
                        && !($banner->{turbolanding} || $banner->{turbolanding_id})
                        && ($banner->{ad_type} eq 'image_ad' || !($banner->{banner_with_phone} || $banner->{has_vcard}))
                        && ( !$banner->{permalink} )
                );
        }
    } elsif (any {defined $vcard->{$_} && $vcard->{$_} ne ''} @VCards::VCARD_FIELDS_FORM) {
        # TODO избавиться от этой ветки (для этого отовсюду должен приходить корректный banner_with_(href|phone))

        if (not $options->{skip_contactinfo}) {
            my @contactinfo_errors = validate_contactinfo($vcard);
            push @{$errors{contactinfo}}, @contactinfo_errors if @contactinfo_errors;
        }
    } else {
       push @{$errors{href}}, iget('Не введена ссылка на сайт')
           if (
               (!defined $banner->{href} || $banner->{href} !~ /\S/)
               && !($banner->{turbolanding} || $banner->{turbolanding_id})
               && ( !$banner->{permalink} )
           ) ;
    }

    if (!$options->{ignore_image}) {
        if ($banner->{image_url}) {
            unless ($banner->{image_url} =~ m!^https?://! && 
                                   (Yandex::IDN::is_valid_domain(get_host($banner->{image_url})) || Yandex::Validate::is_valid_ip(get_host($banner->{image_url}))) ) {
                push @{$errors{image}}, iget('Некорректная ссылка на изображение');
            }
            if ($banner->{ad_type} ne 'image_ad' && $banner->{image_url} =~ /^$Settings::CANVAS_URL/){
                push @{$errors{image}}, iget('Невозможно использовать изображения Конструктора для текстово-графических объявлений');
            }
        } else {
            my $image_hash = $banner->{image};
            if ($image_hash) {
                my $image_type = get_one_field_sql(PPC(ClientID => $options->{ClientID}), [q{SELECT image_type FROM banner_images_formats}, where => {image_hash => $image_hash}]);
                if (!$image_type) {
                    push @{$errors{image}}, iget('Указанного изображения не существует');
                } elsif ($image_type eq 'small') {
                    if (!$banner->{bid} || !get_one_field_sql(PPC(bid => $banner->{bid}), [q{SELECT 1 FROM banner_images}, where => {bid => $banner->{bid}, image_hash => $image_hash}])) {
                        push @{$errors{image}}, iget('Размер изображения меньше допустимого');
                    }
                }
            }
        }
    }
    return \%errors;
}


=head2 check_geo_restrictions($banners, $errors, %params)
    
    Аналогичный код есть в Direct::Validation::Banners::validate_banner_geo_targeting. 
    Текущий вариант считается устаревшим и использования следует избегать.
    В данный момент все изменения следует дублировать там.

    Проверка корректности геотаргетинга у баннера по тексту баннера(заголовок, тело, сайтлинки)
    Именованные параметры (%params)
        geo, geo_ids -- проверяемый регион
        ClientID     -- клиент для выбора транслокального дерева регионов
        tree         -- предопределенное дерево регионов, используется для API

=cut

{
my $RUSSIA_GEO_WARNING = iget_noop("Если для показов объявления выбран регион Россия, то текст объявления должен быть на русском языке или продублирован на русском языке.");

my %region_errors = (
    uk => {
        geo => [$geo_regions::UKR, $geo_regions::KRIM], # Крым нужен для клиентов со страной Россия, им тоже позволенно создявать укр. объявления на Крым
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Украину или измените текст объявления.'),
            for_group => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Украину, поскольку часть текстов в группе написаны на украинском языке (%s).'),
        }
    },
    kk => {
        geo => [$geo_regions::KAZ],
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Казахстан или измените текст объявления.'),
            for_group => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Казахстан, поскольку часть текстов в группе написаны на казахском языке (%s).'),
        }
    },
    tr => {
        geo => [$geo_regions::TR],
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Турцию или измените текст объявления.'),
            for_group => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Турцию, поскольку часть текстов в группе написаны на турецком языке (%s).'),
        }
    },
    uz => {
        geo   => [ $geo_regions::UZB ],
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Узбекистан или измените текст объявления.'),
            for_group  => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Узбекистан, поскольку часть текстов в группе написаны на узбекском языке (%s).'),
        }
    },
    be => {
        geo => [$geo_regions::BY],
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Беларусь или измените текст объявления.'),
            for_group => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Беларусь, поскольку часть текстов в группе написаны на белорусском языке (%s).'),
        }
    },
    vie => {
        geo => [$geo_regions::ASIA],
        error => {
            for_banner => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Азию или измените текст объявления.'),
            for_group => iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга на Азию, поскольку часть текстов в группе написаны на вьетнамском языке (%s).'),
        }
    },
    en => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
    de => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
    es => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
    pt => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
    cs => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
    pl => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => $RUSSIA_GEO_WARNING,
    },
);

sub check_geo_restrictions {
    
    my ($banners, $errors, %params) = @_;

    my @banners = ref $banners eq 'ARRAY' ? @$banners : ($banners);
    return 0 unless @banners;

    my $geo = $params{geo} || $banners[0]->{geo_ids} || $banners[0]->{geo};

    # restrictions for creatives in group
    my $pids = [uniq grep {defined $_} map {$_->{pid}} @banners];
    my $pid = $params{pid} || @{$pids || []}[0];
    my $bids = [uniq grep {defined $_} map {$_->{bid}} @banners];

    my $cid = $params{cid} || get_cid(pid => $pid) || [grep {defined $_} map {$_->{cid}} @banners]->[0];
    my $camp_content_lang;
    if (defined $cid && camp_kind_in(cid=>$cid, 'camp_lang')) {
        $camp_content_lang = CampaignQuery->get_campaign_data(cid => $cid, [qw/content_lang/], get_also_empty_campaigns => 1)->{content_lang};
    }
    
    if ($pid) {
        my $creatives = get_pure_creatives([
            {pid => $pid}
        ], {
            (@$bids ? ('bid__not_in' => $bids) : ())
        }, {only_creatives => 1});
        push @banners, @$creatives;
    }
    # если geo по прежнему не задано, берем его из первого баннера
    if ( $pid and not defined $geo ) {
        # Так как нельзя делать круговую зависимость модулей (нельзя вызвать get_groups),
        # то извлекаем geo группы самостоятельно.
        $geo = get_one_field_sql(PPC(pid => $pid), "select geo from phrases where pid = ?", $pid);
    }
    
    my @error_banners;
    my @error_langs;
    my $new_banner_counter = 0;
    foreach my $banner (@banners) {
        next if $banner->{ad_type} // '' eq 'image_ad' && !$banner->{creative};
        my $text = $banner->{ad_type} // '' eq 'image_ad' && $banner->{creative} ? $banner->{creative}->{creative_text} : join ' ', (@$banner{qw/body title/}, $banner->{title_extension} // ());
        my $translocal_opt = $params{tree} ? {tree => $params{tree}} : {ClientID => $params{ClientID}};
        my $geo_restrictions = get_geo_restrictions(
            $text, $geo, %$translocal_opt, camp_content_lang => $camp_content_lang);
        if (exists $geo_restrictions->{error}) {
            push @error_banners, $banner->{bid};
            $banner->{errors} ||= ();
            push @{$banner->{errors}{title}}, $geo_restrictions->{error}->{for_banner} unless defined $camp_content_lang;
            push @error_langs, $geo_restrictions->{lang};
        }
    }

    if (@error_banners) {

        @error_banners = map {($_) ? sprintf(iget_noop("№ %s"), $_) 
                                   : sprintf(iget_noop("Новое объявление %s"), ++$new_banner_counter)
                             } @error_banners;

        my $error;
        if (scalar(uniq @error_langs) > 1) {
            # Ошибки больше чем с одним языком, показывать общую ошибку.
            $error = (@error_banners == 1) ?
                iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга или скорректируйте тексты для объявления: %s.')
                :
                iget_noop('Язык объявлений должен соответствовать региону их показа. Измените настройки геотаргетинга или скорректируйте тексты для объявлений: %s.')
        } else {
            $error = $region_errors{$error_langs[0]}->{error}->{for_group};
            $error = sprintf($error, (@error_banners == 1) ? iget_noop("объявление: %s") : iget_noop("объявления: %s"));

        }

        $error = sprintf($error, join (", ", @error_banners));

        if ($params{get_just_error}) {
            return iget($error);
        } else {
            add_banner_error_text(
                $errors, $banners[0]->{bid} || 0, 'geo', iget($error), %params
            );
            return 1;
        }
    }
    return 0;
}

=head2 get_geo_restrictions

    Ограничения на регион в зависимости от текста баннера

    На входе:
        text - текст, который надо проверить (обычно это уже заранее склеенные заголовок, описание объявления и заголовки его быстрых ссылок)
        geo - регион
        params
            tree - 
            ClientID - ID клиента
            camp_content_lang - язык кампании (если есть)
            cid - номер кампании (при необходимости задать либо camp_content_lang, либо cid)
            for_banner_only - только тексты ошибок для баннеров
=cut
sub get_geo_restrictions($$;%) {
    my ($text, $geo, %params) = @_;

    my $lang = $params{camp_content_lang} || analyze_text_lang_with_context($params{ClientID_for_analyze_text_language}, $text) || '';
    my $region = $region_errors{$lang};
    return {} unless $lang && $region;

    my $result = {};

    my $translocal_opt = $params{tree} ? {tree => $params{tree}} : {ClientID => $params{ClientID}};
    unless (is_targeting_in_region($geo, join(",", @{$region->{geo}}), $translocal_opt)) {
        if (exists $region->{error}) {
            $result->{error} = ($params{for_banner_only}) ? $region->{error}->{for_banner} : $region->{error};
            $result->{lang} = $lang;
        }

        return $result if %$result;
    }

    return $result;
}

=head2 get_geo_warnings($text, $geo, %params)


    На входе:
        text - текст, который надо проверить (обычно это уже заранее склеенные заголовок, описание объявления и заголовки его быстрых ссылок)
        geo - регион
        params
            tree - 
            ClientID - ID клиента
            camp_content_lang - язык кампании
            cid - номер кампании (при необходимости задать либо camp_lang, либо cid)
=cut
sub get_geo_warnings($$;%) {
    my ($text, $geo, %params) = @_;

    my $text_lang = $params{camp_content_lang} || analyze_text_lang_with_context($params{ClientID_for_analyze_text_language}, $text);
    return {} unless $text_lang;

    my $translocal_opt = $params{tree} ? {tree => $params{tree}} : {ClientID => $params{ClientID}};

    for my $lang (keys %region_errors) {
        my $warning = $region_errors{$lang};

        if (exists $warning->{warning}
            && is_targeting_in_region($geo, join(",", @{$warning->{geo}}), $translocal_opt)
            && $lang eq $text_lang
           )
        {
            return {warning => iget($warning->{warning})};
        }
    }

    return {};
}
}

=head2 get_geo_by_banner_text($banner)

    Если это возможно - определяет регион показа баннера по-умолчанию

=cut

{
my %lang_to_geo  = ( uk => $geo_regions::UKR,
                     kk => $geo_regions::KAZ,
                     tr => $geo_regions::TR,
                     be => $geo_regions::BY,
                   );

sub get_geo_by_banner_text($) {
    my ($banner) = @_;

    my $text = join ' ', @$banner{qw/body title title_extension/},
                ref $banner->{sitelinks} eq 'ARRAY'
                    ? map {$_->{title} || ''} @{$banner->{sitelinks}}
                    : ();
    my $lang = analyze_text_lang($text);

    if ($lang && exists $lang_to_geo{$lang}) {
        return {lang => $lang,
                geo => $lang_to_geo{$lang}};
    }
    return undef;
}
}

=head2 get_pure_creatives($groups, $filters, $options)

    Получить баннеры(креативы) на группу
    
    $groups - список группы в формате [{pid => }]
                достаточно указать только pid
    $filters - дополнительное условие выборки баннеров по значениям полей из banners (в том числе поле bid)
    $options - параметры запроса
        only_creatives => выборка баннеров без доп информации(без визитки, адреса, картинки)
        order_by_status => сортировка баннеров по статусам в порядке: активные, остановленные, отклоненные
        adgroup_types => массив типов групп, для которых выбираются баннеры
                        (используется в качестве подсказки для выборки полного набора атрибутов,
                            в фильтрации не участвует)

        get_all_images => выбирает BannerID всех картинок когда-либо существующих в баннере, в том числе удаленные 
        get_lang => 1 | 0 -- определить язык баннеров

    Результат
        @$banners

=cut
# Возвращяет индекс баннера для сортировки с опцией order_by_status
sub _get_banner_sort_idx {
    my $banner = shift;
    return $banner->{statusShow} eq 'Yes' ? 0 : $banner->{statusModerate} eq 'No' ? 2 : 1;
}

sub get_pure_creatives {
    
    my ($groups, $creatives_filters, $options) = @_;
    unless (@$groups) {
        return wantarray ? ([], 0) : [];
    }
    $options ||= {};
    # Из фильтра берем только поля, начинующеся с bid, признак архивности и выражения типа _OR, _TEXT
    # Причем ниже обрабатываются и pid и поля с точкой, но приводить в порядок это надо аккуратно, вдумчиво.
    my $filters = hash_kgrep sub {$_ =~ /^bid/ || $_ eq 'arch_banners' || $_ =~ /^_/}, $creatives_filters;

    my (@tables);
    my $group_by = '';
    my @fields = (qw/b.bid b.title b.title_extension b.body b.href b.domain b.flags/,
                    qw/b.pid b.statusSitelinksModerate b.sitelinks_set_id b.vcard_id/,
                    qw/b.phoneflag b.statusModerate b.BannerID b.statusShow b.statusBsSynced/,
                    qw/b.statusArch b.statusActive/,
                    'b.type AS banner_type, (b.type = "mobile") AS is_mobile, b.banner_type AS real_banner_type, b.language AS lang',
                    'if(b.banner_type in ("image_ad", "mcbanner", "cpm_banner", "cpc_video", "cpm_outdoor", "cpm_yndx_frontpage", "cpm_indoor", "cpm_geo_pin", "cpm_audio", "content_promotion"), b.banner_type, "text") as ad_type',
                    'md.domain AS adgroup_main_domain',
                    'p.cid',
                    'b.statusPostModerate statusPostModerate',
                    'p.adgroup_type',
                    'FIND_IN_SET("no_extended_geotargeting", c.opts)>0 as no_extended_geotargeting',
                    'u.ClientID',
                    'aid.disclaimer_text AS disclaimer',
                    'aie.experiment_json AS experiment',
                    'bmg.minus_geo',
                    'cbr.categories_bs',
                    'ap.priority as group_priority',
                 );

    my $sql = q{
        SELECT %s %s FROM banners b
        JOIN phrases p ON (p.pid = b.pid)
        LEFT JOIN adgroups_dynamic gd ON (p.adgroup_type = "dynamic" AND gd.pid = b.pid)
        LEFT JOIN domains md ON (md.domain_id = gd.main_domain_id)
        LEFT JOIN campaigns c on c.cid = b.cid
        LEFT JOIN users u on u.uid = c.uid
        LEFT JOIN banners_additions bad ON (bad.bid=b.bid AND bad.additions_type="disclaimer")
        LEFT JOIN additions_item_disclaimers aid ON (aid.additions_item_id = bad.additions_item_id)
        LEFT JOIN additions_item_experiments aie ON (aie.additions_item_id = bad.additions_item_id)
        LEFT JOIN banners_minus_geo bmg on bmg.bid = b.bid and bmg.type = 'current'
        LEFT JOIN catalogia_banners_rubrics cbr on cbr.bid = b.bid
        LEFT JOIN adgroup_priority ap on (c.type = "cpm_price" and ap.pid = b.pid)
        %s
    };

    # $options->{adgroup_types} - это подсказка для функции get_pure_creatives, больше этот параметр использовать не нужно
    if ($options->{adgroup_types} && any {$_ eq 'mobile_content'} @{$options->{adgroup_types}}) {
        push @fields, qw/
            bmc.reflected_attrs
            bmc.primary_action
        /;
        push @tables, "LEFT JOIN banners_mobile_content bmc ON bmc.bid = b.bid";
    }

    push @fields,
        Direct::Model::BannerCreative->get_db_columns(banners_performance => 'bp_va', prefix => 'bp_va_'),
        Direct::Model::VideoAddition->get_db_columns(perf_creatives => 'pc_va', prefix => 'pc_va_');
    push @tables,
        'LEFT JOIN banners_performance bp_va on bp_va.bid = b.bid',
        'LEFT JOIN perf_creatives pc_va on pc_va.creative_id = bp_va.creative_id and b.banner_type IN ("text", "mobile_content")';

    push @fields, 
        Direct::Model::Image->get_db_columns(images => 'im', prefix => 'im_'),
        Direct::Model::ImageFormat->get_db_columns(banner_images_formats => 'imf', prefix => 'imf_'),
        Direct::Model::ImagePool->get_db_columns(banner_images_pool => 'imp', prefix => 'imp_');

    push @tables,
        q[LEFT JOIN images im on im.bid = b.bid],
        q[LEFT JOIN banner_images_formats imf on imf.image_hash = im.image_hash],
        q[LEFT JOIN banner_images_pool imp on imp.ClientID = u.ClientID and imp.image_hash = im.image_hash];
            

    push @fields, (
        'perfc.creative_id',
        'perfc.creative_type as creative_type',
        'perfc.name as creative_name',
        'perfc.width as creative_width',
        'perfc.height as creative_height',
        'perfc.live_preview_url as creative_live_preview_url',
        'perfc.alt_text as creative_alt_text',
        'perfc.href as creative_href',
        'perfc.preview_url as creative_preview_url',
        'perfc.sum_geo as creative_sum_geo',
        'perfc.statusModerate as creative_statusModerate',
        'perfc.template_id as creative_template_id',
        'perfc.moderate_info as creative_moderate_info',
        'perfc.source_media_type as creative_source_media_type',
        'perfc.layout_id as creative_layout_id',
        'perfc.additional_data as creative_additional_data',
        'perfc.duration as creative_duration',
        'perfc.has_packshot as creative_has_packshot',
        'perfc.is_adaptive as creative_is_adaptive',
    );
    push @fields, Direct::Model::BannerCreative->get_db_columns(banners_performance => 'b_perf', prefix => 'b_perf_');
    push @tables, "LEFT JOIN banners_performance b_perf ON b_perf.bid = b.bid"
                    # тут нужно нормальное условие на соответствие типов креативов и кампаний
                , "LEFT JOIN perf_creatives perfc on perfc.creative_id = b_perf.creative_id
                            and b.banner_type IN ('image_ad', 'mcbanner', 'cpm_banner', 'cpc_video', 'cpm_outdoor', 'performance', 'cpm_yndx_frontpage', 'cpm_indoor', 'cpm_audio', 'cpm_geo_pin')";
    
    push @fields, (
        'ifnull(cpromo.id,cpv.content_promotion_video_id) as content_promotion_content_id',
        'ifnull(cpromo.url,cpv.video_href) as content_promotion_url',
        'ifnull(cpromo.preview_url,cpv.video_preview_url) as content_promotion_preview_url',
        'ifnull(bcpromo.visit_url,bcpv.packshot_href) as content_promotion_visit_url',
        'ifnull(cpromo.type, if(cpv.content_promotion_video_id is not null,"video",null)) as content_promotion_content_type'
    );

    push @tables, "LEFT JOIN banners_content_promotion_video bcpv ON bcpv.bid = b.bid",
                  "LEFT JOIN content_promotion_video cpv ON cpv.content_promotion_video_id = bcpv.content_promotion_video_id",
                  "LEFT JOIN banners_content_promotion bcpromo ON bcpromo.bid = b.bid",
                  "LEFT JOIN content_promotion cpromo ON cpromo.id = bcpromo.content_promotion_id";

    unless ($options->{only_creatives}) {

        push @fields, 'IFNULL(fd.filter_domain, b.domain) filter_domain',
                        qw/vc.phone vc.name vc.street vc.house vc.build vc.apart vc.metro vc.contactperson/,
                        qw/vc.worktime vc.city vc.country vc.geo_id vc.im_client vc.im_login vc.extra_message/,
                        qw/vc.contact_email vc.org_details_id vc.address_id/,
                        'addresses.precision as auto_precision',
                        q[CONCAT_WS(',', maps.x, maps.y) as manual_point],
                        q[CONCAT_WS(',', maps.x1, maps.y1, maps.x2, maps.y2) as manual_bounds],
                        q[CONCAT_WS(',', maps_auto.x, maps_auto.y) as auto_point],
                        q[CONCAT_WS(',', maps_auto.x1, maps_auto.y1, maps_auto.x2, maps_auto.y2) as auto_bounds],
                        'bim.image_hash as image',
                        'bimf.image_type', 'bimf.width AS image_width', 'bimf.height AS image_height',
                        'bimf.image_hash', 'bimf.namespace', 'bimf.mds_group_id', 'bimf.avatars_host',
                        'bim.statusModerate as image_statusModerate',
                        'bimp.name as image_name', 'bim.PriorityID as image_PriorityID',
                        'bim.BannerID AS image_BannerID, bim.image_id',
                        'bdh.display_href', 'bdh.statusModerate as display_href_statusModerate',
                        'tl.tl_id as tl_tl_id, tl.name as tl_name, tl.href as tl_href',
                        'btl.statusModerate as tl_statusModerate', 'btlp.href_params as turbolanding_href_params',
                        'btns.tns_id'
                        ;
                        
        push @tables,
            q[left join vcards vc on vc.vcard_id = b.vcard_id],
            q[left join filter_domain fd on fd.domain = b.domain],
            q[left join addresses on vc.address_id = addresses.aid],
            q[left join maps on addresses.map_id = maps.mid],
            q[left join maps maps_auto on addresses.map_id_auto = maps_auto.mid],
            q[left join banner_images bim on bim.bid = b.bid and bim.statusShow = 'Yes'],
            q[left join banner_images_pool bimp on bimp.ClientID = u.ClientID AND bimp.image_hash = bim.image_hash],
            q[left join banner_images_formats bimf on bimf.image_hash = bim.image_hash],
            q[LEFT JOIN banner_display_hrefs bdh on bdh.bid=b.bid],
            q[LEFT JOIN banner_turbolandings btl on btl.bid=b.bid],
            q[LEFT JOIN turbolandings tl on btl.tl_id=tl.tl_id],
            q[LEFT JOIN banner_turbolanding_params btlp on btlp.bid=b.bid],
            'LEFT JOIN banners_tns btns ON btns.bid = b.bid',
            ;

        if ($options->{get_all_images}) {
            do_sql(PPC(shard => 'all'), "SET SESSION group_concat_max_len = 100000");
            push @fields, 'GROUP_CONCAT(bima.BannerID) AS all_images_BannerID';
            push @tables, q[left join banner_images bima on bima.bid = b.bid and bima.BannerID > 0];
            $group_by = 'group by b.bid';
        }
    }
    
    
    my %where;
    if (defined (my $arch_banners = delete $filters->{arch_banners})) {
        $where{'b.statusArch'} = $arch_banners ? 'Yes' : 'No';
    }

    if (! $filters->{pid}) {
        my @pids = grep {$_} map {$_->{pid}} @$groups;
        $filters->{pid} = \@pids if @pids;
    }
    if (! $filters->{bid}) {
        my @bids = grep {$_} map {$_->{bid}} @$groups;
        $filters->{bid} = \@bids if @bids;
    }
    return wantarray ? ([], 0) : [] unless @{$filters->{pid}} || @{$filters->{bid}};

    for my $field (keys %$filters) {
        my $name = $field =~ /^_|\./ ? $field : "b.$field";
        $where{$name} = $filters->{$field};
    }

    my @pids = grep {$_} map {$_->{pid}} @$groups;
    return wantarray ? ([], 0) : [] unless @pids;
    
    $where{'b.pid'} = \@pids if @pids;

    $sql = sprintf $sql, ($options->{limit} ? 'SQL_CALC_FOUND_ROWS' : ''), join(',', @fields), join("\n", @tables);

    my %shard = choose_shard_param(\%where, [qw/pid bid/], set_shard_ids => 1);    
    # не делаем OFFSET/LIMIT в sql, а только на перловой стороне, иначе получается offset/limit 2 раза и записи пропускаются
    my $banners_pre = get_all_sql(PPC(%shard), [$sql, WHERE => \%where, $group_by]);
    # Дополнительно сортируем по статусам, если указана опция order_by_status
    $banners_pre = [sort { _get_banner_sort_idx($a) <=> _get_banner_sort_idx($b) || $a->{bid} <=> $b->{bid} } @$banners_pre] if $options->{order_by_status};
    my %overshard_opt;
    unless ($options->{order_by_status}) {
        $overshard_opt{order} = ['-ad_type', 'bid:num'];
    }
    if ($options->{limit}) {
        $overshard_opt{limit} = $options->{limit};
        if ($options->{offset}) {
            $overshard_opt{offset} = $options->{offset};
        }
    }
    my $banners = (overshard %overshard_opt, $banners_pre) || [];

    my $total = $options->{limit}
        ? select_found_rows(PPC(%shard))
        : @$banners;
    
    my @sl_sets_ids = uniq grep {$_} map {$_->{sitelinks_set_id}} @$banners;
    my $sitelinks_sets = Sitelinks::get_sitelinks_by_set_id_multi(\@sl_sets_ids);

    my $mobile_content_banners = [grep {$_->{adgroup_type} eq 'mobile_content'} @$banners];
    my $mobile_content = @$mobile_content_banners
                         ? Direct::AdGroups2::MobileContent->get_mobile_content_by(adgroup_id => [map {$_->{pid}} @$mobile_content_banners])
                         : {};

    my @bids = map { $_->{bid} } @$banners;
    my $banner_measurers = Direct::Banners::Measurers->get_by(banner_id => \@bids)->items_by('banner_id');

    my $cache = {};
    foreach my $banner (@$banners) {
        $banner->{reflected_attrs} = [split(/,/, $banner->{reflected_attrs})]  if defined $banner->{reflected_attrs};
        $banner->{sitelinks} = yclone($sitelinks_sets->{$banner->{sitelinks_set_id}}) if $banner->{sitelinks_set_id};
        $banner->{href} = clear_banner_href($banner->{href});
        $banner->{has_href} = ($banner->{href}) ? 1 : 0;
        $banner->{has_vcard} = ($banner->{phone}) ? 1 : 0;
        $banner->{measurers} =  [map { $_->to_template_hash } @{$banner_measurers->{$banner->{bid}} || []}];
        if ($options->{get_lang}) {
            if (Direct::Model::Banner::LanguageUtils::is_empty_language($banner->{lang})) {
                $banner->{lang} = analyze_text_lang($banner->{title}, $banner->{title_extension}, $banner->{body});
            }
        }
        if ($banner->{im_bid} && ($banner->{ad_type} eq 'image_ad' || $banner->{ad_type} eq 'mcbanner')
            || $banner->{b_perf_bid} && ($banner->{ad_type} eq 'image_ad' || $banner->{ad_type} eq 'cpm_banner')
            || $banner->{ad_type} eq 'cpc_video' || $banner->{ad_type} eq 'cpm_outdoor' || $banner->{ad_type} eq 'cpm_indoor'
            || $banner->{ad_type} eq 'cpm_audio' || $banner->{ad_type} eq 'cpm_geo_pin'
        ) {
            my $banner_image_ad;
            if ($banner->{ad_type} eq 'image_ad') {
                $banner_image_ad = Direct::Model::BannerImageAd->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'mcbanner') {
                $banner_image_ad = Direct::Model::BannerMcbanner->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpm_banner') {
                $banner_image_ad = Direct::Model::BannerCpmBanner->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpc_video') {
                $banner_image_ad = Direct::Model::BannerCpcVideo->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpm_outdoor') {
                $banner_image_ad = Direct::Model::BannerCpmOutdoor->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpm_indoor') {
                $banner_image_ad = Direct::Model::BannerCpmIndoor->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpm_audio') {
                $banner_image_ad = Direct::Model::BannerCpmAudio->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            } elsif ($banner->{ad_type} eq 'cpm_geo_pin') {
                $banner_image_ad = Direct::Model::BannerCpmGeoPin->from_db_hash(hash_merge({}, $banner, { banner_type => $banner->{real_banner_type} }), \$cache);
            }
            $banner->{ad_type} = 'cpm_banner' if ($banner->{ad_type} eq 'cpm_outdoor' || $banner->{ad_type} eq 'cpm_indoor' || $banner->{ad_type} eq 'cpm_audio' || $banner->{ad_type} eq 'cpm_geo_pin');

            if ($banner->{im_bid}) {
                # imagead/picture
                my $image = Direct::Model::Image->from_db_hash($banner, \$cache, prefix => 'im_');
                $image->format(Direct::Model::ImageFormat->from_db_hash($banner, \$cache, prefix => 'imf_'));
                $image->pool(Direct::Model::ImagePool->from_db_hash($banner, \$cache, prefix => 'imp_'));
                $banner_image_ad->image_ad($image);
            } else { # $banner->{b_perf_bid}
                my $banner_creative = Direct::Model::BannerCreative->from_db_hash($banner, \$cache, prefix => 'b_perf_');
                my $creative_data = hash_merge($banner, { creative_creative_id => $banner->{creative_id} });
                my $creative_type = delete $banner->{creative_type};
                my $creative;
                # TODO possible check if creative type available for banner
                if ($creative_type eq 'canvas') {
                    $creative = Direct::Model::CanvasCreative->from_db_hash($creative_data, \$cache, prefix => 'creative_');
                } elsif ($creative_type eq 'html5_creative') {
                    $creative = Direct::Model::CanvasHtml5Creative->from_db_hash($creative_data, \$cache, prefix => 'creative_');
                } elsif ($creative_type eq 'video_addition' || $creative_type eq 'bannerstorage') {
                    $creative = Direct::Model::VideoAddition->from_db_hash($creative_data, \$cache, prefix => 'creative_');
                } else {
                    die "unknown creative type $creative_type";
                }
                $banner_creative->creative($creative);
                $banner_image_ad->creative($banner_creative);
            }
            hash_merge $banner, $banner_image_ad->to_template_hash;
            $banner->{banner_statusModerate} = $banner->{statusModerate};
            $banner->{statusModerate} = get_aggregate_status_moderate($banner, $banner->{im_statusModerate} ? $banner->{im_statusModerate} : $banner->{b_perf_statusModerate});
        }
        if ($banner->{real_banner_type} =~ /text|mobile_content/
             && $banner->{pc_va_creative_id}) {

            $banner->{video_resources} = Direct::Model::VideoAddition->from_db_hash($banner, \$cache, prefix => 'pc_va_')->to_template_hash;
            $banner->{video_resources}{status_moderate} = $banner->{bp_va_statusModerate};
        }
        
        my %landing = map {$_ => delete $banner->{$_}} qw/tl_tl_id tl_name tl_href tl_statusModerate/;
        if ($landing{tl_tl_id}) {
            $landing{tl_bid} = $banner->{bid};
            $banner->{turbolanding} = Direct::Model::TurboLanding::Banner->from_db_hash(\%landing, \$cache, prefix => 'tl_')->to_template_hash();
        }

        $banner->{disable_videomotion} = is_videomotion_disabled($banner) ? 1 : 0;
    }

    # заполняем уточнения
    Direct::BannersAdditions::add_callouts_to_banners($banners);

    # заполняем видео дополнения
    Direct::BannersResources::add_video_resources_to_banners([ grep { $_->{real_banner_type} eq 'text' } @$banners ]);

    # добавляем пикселы
    Direct::BannersPixels::add_pixels_to_banners($banners);

    # добавляем цены на товары на баннерах
    Direct::BannersPrices::add_banner_prices_to_banners($banners);

    # добавляем результаты модерации pageId баннеров
    Direct::BannersPlacementPages::add_banner_placement_pages_to_banners($banners);

    # добавляем пермалинки организаций Справочника
    Direct::BannersPermalinks::add_banner_permalinks_to_banners($banners);

    # Удалим adgroup_main_domain для не-динамических баннеров
    delete $_->{adgroup_main_domain} for grep { $_->{real_banner_type} ne 'dynamic' } @$banners;

    return wantarray
        ? ($banners, $total)
        : $banners;
}

=head2 _copy_to_banner

    Внутренняя вспомогательная функция
    Копирование полей в баннер

    Пока еще используется в BannersCommon в старой версии get_banners

=cut
sub _copy_to_banner {

    my ($object, $description) = @_;
    my $to_banner = {};
    my $fields = $description->{fields};
    for my $field (@$fields){
        next unless exists $object->{$field};
        if (ref $object->{$field}) {
            $to_banner->{$field} = yclone($object->{$field});
        } else {
            $to_banner->{$field} = $object->{$field};
        }
    }
    while (my ($prefix, $fields) = each %{$description->{prefix} || {}}) {
        foreach (@$fields) {
            $to_banner->{"$prefix$_"} = $object->{$_} if exists $object->{$_}; 
        } 
    }
    return $to_banner;
}

=head2 get_banners_for_delete

    Возвращает баннеры со всеми необходимыми полями для проверки на удаление и удаления баннеров.
    На входе:
        - HASHREF с полями pid или bid

    На выходе:
        баннеры с полями bid, BannerID, group_statusPostModerate, group_statusModerate,
        banner_statusPostModerate, banner_statusModerate sum camp_in_bs_queue

=cut
sub get_banners_for_delete {
    my $search_options = shift;

    if  (none {defined($search_options->{$_})} qw/bid pid/) {
        return;
    }

    my $fields = "b.bid,
                  bp.creative_id,
                  c.uid,
                  b.BannerID,
                  IF(beq.cid OR bec.cid, 1, 0) AS camp_in_bs_queue,
                  g.statusPostModerate AS group_statusPostModerate,
                  g.statusModerate AS group_statusModerate,
                  b.statusPostModerate AS banner_statusPostModerate,
                  b.statusModerate AS banner_statusModerate,
                  c.sum + IF(c.wallet_cid, wc.sum, 0) AS sum,
                  ap.priority as group_priority";
    my $from = "banners b
                JOIN phrases g ON b.pid = g.pid
                JOIN campaigns c ON g.cid = c.cid
                LEFT JOIN bs_export_queue beq ON beq.cid = c.cid
                LEFT JOIN bs_export_candidates bec ON bec.cid = c.cid
                LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid AND wc.uid = c.uid
                LEFT JOIN banners_performance bp ON bp.bid = b.bid
                LEFT JOIN adgroup_priority ap on ap.pid = b.pid";

    my $where = {map {'b.'.$_ => $search_options->{$_}} keys %$search_options};
    my %shard = choose_shard_param($where, [qw/pid bid/], set_shard_ids => 1);

    my $banners = get_all_sql(PPC(%shard), [sprintf ("SELECT %s FROM %s", $fields, $from), where => $where]);

    return $banners;

}

=head2 delete_banners ($cid, \@banners, \%opts)

    удаляем список баннеров
    delete_banners($cid, \@banners, \%opts);

    Не смотря на то, что ошибок удаления баннеров может быть много, те баннеры, что можно удалить - удаляем,
    из тех только про первый запрещенный выводим ошибку.
    
    Параметры
        $cid - номер кампании
        \@banners - массив баннеров
        \%opts - опции:
            skip_checks - флаг для пропуска проверки о возможности удаления баннера,
                            используется в скрипте protected/one-shot/delete_banner.pl,
                            использовать в других местах НЕ РЕКОМЕНДУЕТСЯ

=cut

sub delete_banners {

    my ($cid, $banners, $opts) = @_;

    my $error;
    my @delete_banners;
    if (exists $opts->{skip_checks}) {
        @delete_banners = @$banners;
    } else {
        # Те баннеры, что удалять нельзя, пропускаем.
        @delete_banners = grep {my $res = has_delete_banner_problem($_); $error||=$res; !$res} @$banners;
    }

    my $bids = [ grep{is_valid_id($_)} map {$_->{bid}} @delete_banners];
    my $pids = [ grep{is_valid_id($_)} map {$_->{pid}} @delete_banners];

    do_in_transaction {
        #Перед удалением записей из banners и banner_turbolandings нужно уменьшить счетчики в camp_metrika_goals
        _decrease_camp_metrika_goal_counters($cid, $bids);
        # удаляем объявления из базы
        my $canvas_or_video_ids = get_one_column_sql(PPC(cid => $cid), ["select banner_creative_id from banners_performance", where => {bid => $bids}]);
        del_banner_from_db($bids, $pids);
    
        # обновляем sum_geo у креативов, связанных с удаленными баннерами
        update_creatives_geo(\@delete_banners);
    
        # удаляем объявления из очереди на проcтукивание редиректа
        do_delete_from_table(PPC(cid => $cid), 'redirect_check_queue', where => {object_id => $bids, object_type => 'banner'});
    
        # удалить картинки
        do_delete_from_table(PPC(cid => $cid), 'banner_images', where => { bid => $bids });

        # удалить графические объявления
        my $image_ids = get_one_column_sql(PPC(cid => $cid), ["select image_id from images", where => {bid => $bids}]);
        do_delete_from_table(PPC(cid => $cid), 'images', where => { bid => $bids });

        # удалить причины отклонения
        do_delete_from_table(PPC(cid => $cid), 'mod_reasons', where => { id => $bids, type => [qw/banner contactinfo sitelinks_set image display_href/]});
        do_delete_from_table(PPC(cid => $cid), 'mod_reasons', where => {id => $image_ids, type => 'image_ad'}) if @$image_ids;
        do_delete_from_table(PPC(cid => $cid), 'mod_reasons', where => {id => $canvas_or_video_ids, type => ['canvas', 'video_addition']}) if @$canvas_or_video_ids;

        # удалить версии объектов для модерации
        do_delete_from_table(PPC(cid => $cid), 'mod_object_version', where => { obj_id => $bids, obj_type => [qw/banner contactinfo sitelinks_set image display_href image_ad canvas video_addition/]});
    
        # удалить уточнения
        do_delete_from_table(PPC(cid => $cid), 'banners_additions', where => {bid => $bids});
    
        # удалить отображаемые урлы
        do_delete_from_table(PPC(cid => $cid), 'banner_display_hrefs', where => {bid => $bids});
        # удалить кастомные тексты отображаемых урлов
        do_delete_from_table(PPC(cid => $cid), 'banner_display_href_texts', where => {bid => $bids});

        # удалить атрибуты лидформ баннеров
        do_delete_from_table(PPC(cid => $cid), 'banner_leadform_attributes', where => {bid => $bids});
    
        # удалить дополнительные ссылки на креатив
        do_delete_from_table(PPC(cid => $cid), 'banner_additional_hrefs', where => {bid => $bids});

        # удалить названия (объект рекламирования)
        do_delete_from_table(PPC(cid => $cid), 'banner_names', where => {bid => $bids});

        # удалить логотипы
        do_delete_from_table(PPC(cid => $cid), 'banner_logos', where => {bid => $bids});
        # удалить кнопки
        do_delete_from_table(PPC(cid => $cid), 'banner_buttons', where => {bid => $bids});
        # удалить карточки мультибаннера
        do_delete_from_table(PPC(cid => $cid), 'banner_multicard_sets', where => {bid => $bids});
        do_delete_from_table(PPC(cid => $cid), 'banner_multicards', where => {bid => $bids});
        # удалить данные об изменении пользовательских флагов
        do_delete_from_table(PPC(cid => $cid), 'banner_user_flags_updates', where => {bid => $bids});
        # удалить данные о паблишерах дзена
        do_delete_from_table(PPC(cid => $cid), 'banner_publisher', where => {bid => $bids});

        #удалить турболендинги баннеров
        do_delete_from_table(PPC(cid => $cid), 'banner_turbolandings', where => {bid => $bids});
        #удалить дополнительные параметры ссылок турболендингов баннеров
        do_delete_from_table(PPC(cid => $cid), 'banner_turbolanding_params', where => {bid => $bids});
        # удалить счетчики метрики турболендингов баннеров
        do_delete_from_table(PPC(cid => $cid), 'camp_turbolanding_metrika_counters', where => {bid => $bids});

        # удалить связки баннеров с организациями справочника
        do_delete_from_table(PPC(cid => $cid), 'banner_permalinks', where => {bid => $bids});

        # удалить связки баннеров с телефонами
        do_delete_from_table(PPC(cid => $cid), 'banner_phones', where => {bid => $bids});
    };

    return (scalar(@delete_banners), $error);
}

sub _decrease_camp_metrika_goal_counters {
    my ($cid, $bids) = @_;
    
    return unless @$bids;

    my $decreased_camp_turbolandings;
    
    my $banner_turbolandings = get_all_sql( PPC( cid => $cid), [
        q/SELECT bid, tl_id FROM banner_turbolandings/,
        WHERE => {cid => $cid, bid => $bids}
    ]);
    
    my $sitelink_turbolandings = get_all_sql( PPC( cid => $cid), [
        q/SELECT distinct bid, tl_id
            FROM banners b
            JOIN sitelinks_set_to_link s2l ON (b.sitelinks_set_id = s2l.sitelinks_set_id)
            JOIN sitelinks_links sl ON (sl.sl_id = s2l.sl_id)/,
        WHERE => {cid => $cid, bid => $bids, tl_id__gt => 0}
    ]);
    

    my $tl_data = Direct::TurboLandings::get_metrika_counters_and_goals_by_tl_id(
            PPC(cid => $cid),
            [uniq map {$_->{tl_id}} ((@$banner_turbolandings, @$sitelink_turbolandings))]
        );
    
    my $processed;
    foreach my $row (@$banner_turbolandings, @$sitelink_turbolandings){
        my ($bid, $tl_id ) = @$row{qw/bid tl_id/};
        next if $processed->{"$bid.$tl_id"}++;
        next unless @{$tl_data->{$tl_id}->{goals} // []};
        
        foreach my $goal_id (@{$tl_data->{$tl_id}->{goals}}){
            $decreased_camp_turbolandings->{$goal_id}--;
        }
        
    }

    if (keys %$decreased_camp_turbolandings) {
         do_mass_update_sql(PPC(cid => $cid), 'camp_metrika_goals',
                            'goal_id',
                            {
                                map { $_ => {
                                        links_count => 'links_count + '.$decreased_camp_turbolandings->{$_}
                                    }
                                } keys %$decreased_camp_turbolandings
                            },
                            where => {cid => $cid},
                            byfield_options => {
                                links_count => {dont_quote_value => 1}
                            }
        );
    }
    
    return;
}

=head2 update_creatives_geo

    Пересчитывает и перезаписывает гео у креативов-черновиков

=cut

sub update_creatives_geo {
    my ($banners) = @_;

    my $banners_performance = [grep {$_->{creative_id}} @$banners];
    return unless @$banners_performance;

    my $uid2client_id = get_key2clientid(uid => [map {$_->{uid}} @$banners_performance]);
    my $translocal_options_by_creative_id = {map {$_->{creative_id} => {ClientID => $uid2client_id->{$_->{uid}}} } @$banners_performance};

    my $case_values = Direct::Model::BannerPerformance::Manager->get_joined_geo($translocal_options_by_creative_id);

    my @creative_ids = keys %$case_values;

    do_update_table(PPC(creative_id => \@creative_ids), 'perf_creatives', {
        sum_geo__dont_quote => sql_case(creative_id => $case_values, default__dont_quote => 'sum_geo'),
    }, where => {
        creative_id => \@creative_ids,
        statusModerate => ['New', 'Error'],
    });

    return
}

=head2 del_banner_from_db

    Удаляет баннер из БД, аккуратно обрабатывая все зависимости.

=cut
sub del_banner_from_db($$)
{
    my ($bid, $pids) = @_;

    my @bids = ref($bid) eq 'ARRAY' ? @$bid : [$bid];

    return unless grep {/^\d+$/} @bids;

    # Сохраняем cid-ы для ускорения удаления медиабаннеров
    my $cids = get_cids(bid => \@bids);

    for my $table (qw/banner_display_hrefs banners_mobile_content banners_performance banners banners_minus_geo
                      aggregator_domains moderate_banner_pages banner_moderation_versions banner_measurers banners_tns
                      banners_performance_main/) {
        do_delete_from_table(PPC(bid => \@bids), $table, where => {bid => SHARD_IDS});
    }

    foreach_shard bid => \@bids, sub {
        my ($shard, $bids_chunk) = @_;
        my @data = map { [$_, 'NOW()'] } @$bids_chunk;
        do_mass_insert_sql(PPC(shard => $shard), "INSERT IGNORE INTO deleted_banners (bid, deleteTime) VALUES %s"
            , \@data
            , { sleep => 1,
                max_row_for_insert => 1000,
                dont_quote => 1}
        );
    };

    do_update_table(PPC(cid => $cids), "mediaplan_banners"
        , { source_bid => 0 }
        , where => { source_bid => \@bids, cid => SHARD_IDS }
    );

    delete_shard(bid => \@bids);
}

=head2 validate_banner_imagead

=cut

sub validate_banner_imagead
{
    my ($banner, $adgroup, $options) = @_;

    # временная очистка пользовательских данных в случае пустых значений в объектах image_ad и creative DIRECT-58615
    if ($banner->{creative}) {
        delete $banner->{creative} unless $banner->{creative}->{creative_id};
    }
    if ($banner->{image_ad}) {
        delete $banner->{image_ad} unless $banner->{image_ad}->{hash};
    }

    if ($banner->{image_ad} && $banner->{creative}) {
        return iget('Для графического баннера должен быть определен только креатив или только изображение');
    }
    unless ($banner->{image_url} || $banner->{image_ad} || $banner->{creative}) {
        return iget('Поле "Изображение" обязательно для Графических объявлений');
        # после публичного запуска DIRECT-56873 изменить текста на: return iget('Для графического баннера должен быть определен креатив или изображение');
    }
    unless ($banner->{creative}) {
        return if !$banner->{image_ad};

        return unless $adgroup->{cid};

        my $bid = $banner->{bid} || 0;

        my $new_image_format = get_one_line_sql(PPC(cid => $adgroup->{cid}), [
                "select width, height from banner_images_formats",
                where => { image_hash => $banner->{image_ad}->{hash} }
            ]);
        unless ($new_image_format) {
            return if $options->{skip_imagead}; # для xls - попробуем скачать картинку потом
            return iget("Неверно указано поле %s", 'imagead_hash');
        }
        if ($bid) {
            my $prev_image_format = get_one_line_sql(PPC(bid => $bid), [
                    "select width, height from banner_images_formats imf join images im using(image_hash)",
                    where => { bid => $bid }
                ]);
            if ($prev_image_format->{width} != $new_image_format->{width} || $prev_image_format->{height} != $new_image_format->{height}) {
                return iget("Можно заменить только изображение такого же размера");
            }
        }
    } else {
        my $creative_client_id = get_one_field_sql(
            PPC(ClientID => $options->{ClientID}), [
                q/select ClientID from perf_creatives/,
                where => [
                    creative_id__int => $banner->{creative}->{creative_id},
                    creative_type => 'canvas'
                ]
            ]
        );

        return unless $creative_client_id;
        
        return iget('Canvas-креатив %s вам не доступен', $banner->{creative}->{creative_id})
            unless ($creative_client_id == $options->{ClientID});
    
        return;
    }
}

=head2 get_aggregate_status_moderate($banner, $child_status_moderate)

   Вычисление агрегационного статуса модерации баннера.
   Для баннеров составные части которого модерируются отдельно.
   Например для imagead - banners + banners_performance. 

   Параметры:
     $banner - {}
     $child_status_moderate - statusModerate от дочерней составляющией баннера

=cut

sub get_aggregate_status_moderate {
    my ($banner, $child_status_moderate) = @_;

    my $banner_status = $banner->{statusModerate};
    my $banner_status_post = $banner->{statusPostModerate};

    my $mod_status_re = qr/(Ready|Sending|Sent)/;

    my $status;
    if ($banner_status eq 'No' || $child_status_moderate eq 'No') {
        $status = 'No';
    }
    elsif ($child_status_moderate =~ /$mod_status_re/) {
        $status = $1;
    }
    elsif ($banner_status =~ /$mod_status_re/) {
        $status = $banner_status_post eq 'Yes' ? $banner_status_post : $1;
    }
    elsif ($banner_status eq 'New' && ($child_status_moderate eq 'New' || $child_status_moderate eq 'Yes')) {
        $status = 'New';
    } elsif (($banner_status eq 'Yes' || ($banner_status ne 'New' && $banner_status_post eq 'Yes')) && $child_status_moderate eq 'Yes') {
        $status = 'Yes';
    } elsif ($banner_status_post =~ /No|Rejected/) {
        $status = 'No';
    } else {
        croak "Cant guess banner moderation status. Banner statusModerate: $banner_status, Child statusModerate: $child_status_moderate";
    }

    return $status;
}

=head2 is_videomotion_disabled

Признак того, что для данного баннера запрещен показ videomotion

=cut

sub is_videomotion_disabled
{
    my ($banner) = @_;
    my $flags = BannerFlags::get_banner_flags_as_hash($banner->{flags}, all_flags => 1);
    return $flags->{media_disclaimer} || $banner->{real_banner_type} eq 'mobile_content';
}

=head2 is_cpm_banner

Признак медийности баннера

=cut

sub is_cpm_banner
{
    my ($banner_type) = @_;
    return $banner_type eq 'cpm_banner' || $banner_type eq 'cpm_outdoor' || $banner_type eq 'cpm_indoor' ||
        $banner_type eq 'cpm_audio' || $banner_type eq 'cpm_geo_pin';
}

1;
