package Models::Campaign;

use strict;
use warnings;
use utf8;

use Settings;

use Carp;
use List::Util qw/minstr max/;
use List::MoreUtils qw/notall uniq/;
use List::MoreUtils qw/uniq part any/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Validate;
use Yandex::HashUtils;

use PrimitivesIds qw/get_clientid/;
use MTools;
use Primitives;
use PlacePrice;
use Client;
use Campaign qw/get_camp_info/;
use CampaignTools;
use Campaign::Types;
use Mediaplan qw/get_mediaplan_banners_count/;
use BannersCommon;
use BannerImages;
use RBACDirect;
use Tools qw/log_cmd/;
use Stat::OrderStatDay;

use Models::AdGroup;
use Models::AdGroupFilters;
use Models::CampaignOperations;
use Direct::Validation::BannersPerformance;

use Direct::ImageFormats;
use Direct::Validation::Campaigns qw//;
use MetrikaCounters qw//;

use base qw(Exporter);
our @EXPORT = qw/
    filter_full_campaigns
/;

sub validate_campaign {

    my ($campaign, $options) = @_;
    $options ||= {};
    my @errors;

    my $new_groups = 0;
    foreach my $group (@{$campaign->{groups}}) {
        $new_groups++ unless $group->{pid};

        if (! $options->{no_check_limits}) {
            my $check_add = check_add_client_creatives_limits({
                pid           => $group->{pid},
                new_creatives => scalar(grep {!$_->{bid}} @{$group->{banners}})
            });
            push @errors, $check_add if $check_add;
        }
    }

    if (! $options->{no_check_limits}) {
        # check if the final group doesn't exceed the MAX_BANNERS_COUNT
        my $check_add = check_add_client_groups_limits({
            uid => $campaign->{uid},
            new_groups => $new_groups,
            cid => $campaign->{cid}
        });
        push @errors, $check_add if $check_add;
    }

    if ($campaign->{camp_name}) {
        push @errors, validate_campaign_name($campaign->{camp_name});
    }

    return @errors;
}

=head2 validate_campaign_name($campaign_name)

    Валидация поля "Название кампании"
=cut

sub validate_campaign_name {
    my ($campaign_name) = @_;

    my @errors;
    if (defined($campaign_name) && length($campaign_name) > 0) {
        push @errors, iget('Название кампании пусто (одни пробелы)') if $campaign_name =~ m/^\s*$/;
        push @errors, iget('Название кампании содержит спецсимволы') if ($campaign_name =~ /[<>]/ || $campaign_name =~ /(\P{print})/);
        push @errors, iget('Название кампании не должно превышать 255 символов') if length($campaign_name) > $Direct::Validation::Campaigns::MAX_TITLE_LENGTH;
    } else {
        push @errors, iget('Вы должны указать название кампании');
    }
    return @errors;
}

=head2 validate_campaign_metrika_counters

    Валидация поля "Счётчики Метрики"
=cut

sub validate_campaign_metrika_counters {
    my ($metrika_counters, $mediaType, $uid, $operator_uid, %options) = @_;

    my @errors;
    my @metrika_counters;
    my $operator_client_id = $operator_uid ? get_clientid(uid => $operator_uid) : 0; 
    if (defined $metrika_counters) {
        @metrika_counters = ref $metrika_counters eq 'ARRAY' ? @$metrika_counters
            : (split /[\s,]+/, ($metrika_counters =~ s/(^[\s,]+|[\s,]+$)//gr));
        if (grep { /[^\d]/ || $_ > $Direct::Validation::Campaigns::MAX_METRIKA_COUNTER_ID || $_ <= 0 } @metrika_counters) {
            push @errors, iget('Дополнительные счетчики Метрики указаны неверно');
        }
    }
    my $counter_limit = $Direct::Validation::Campaigns::MAX_METRIKA_COUNTERS_COUNT_WITH_FEATURE;
    if (($mediaType // '') eq 'performance') {
        if (!@metrika_counters) {
            push @errors, iget('Не указан счетчик Метрики');
        } elsif (@metrika_counters > 1) {
            push @errors, iget('Можно указать только один счетчик Метрики');
        } elsif (@metrika_counters) {
            if (!($operator_client_id && $Settings::DO_NOT_VALIDATE_ACCESS_TO_METRIKA_GOALS{$operator_client_id})
                && !MetrikaCounters::is_counter_available_for($metrika_counters[0], $uid)) 
            {
                push @errors, iget('Счетчик недоступен')
            }
        }
    } elsif (scalar(@metrika_counters) > $counter_limit) {
        push @errors, iget('Можно указать не более %d дополнительных счетчиков Метрики', $counter_limit);
    }
    return @errors;
}

=head2 filter_full_campaigns(@campaigns)

    Отфильтровать кампании готовые к показам.
    
    Параметры
        @campaigns - массив кампаний (должен быть определен ключ cid)
            ({cid => } {cid => })

    Результат
        @full_campaigns - массив исходных кампаний (@campaigns) которые готовы к показам            

=cut

sub filter_full_campaigns {

    my @campaigns = map {ref $_ eq 'ARRAY' ? @$_ : $_} @_;
    my $full_campaigns = is_completed_campaigns([map {$_->{cid}} @campaigns]);
    return grep {$full_campaigns->{$_->{cid}}} @campaigns;
}

=head2 is_completed_campaigns($cid || $cids)

    Возвращает хеш с кампаниями($cid), которые готовы к показам
    (в кампании есть хотя бы одна группа с баннером и условием показа (ИЛИ фраза/ретаргетинг/условия_нацеливания))
    
    Возвращает
        {
            cid => 1, 
            cid => 1 
        } 

=cut

sub is_completed_campaigns {

    my $cids = shift;

    $cids = [$cids] unless ref $cids eq 'ARRAY';
    my $full_campaigns = get_hash_sql(PPC(cid => $cids), [
        "SELECT c.cid, 1
        FROM campaigns c",
        WHERE => {'c.cid' => SHARD_IDS},
        "AND exists (SELECT 1
                      FROM phrases p
                     WHERE p.cid = c.cid
                       AND EXISTS (SELECT 1 FROM banners b WHERE b.pid = p.pid)
                       AND (
                           c.archived = 'No' AND EXISTS (SELECT 1 FROM bids bi WHERE bi.pid = p.pid)
                           OR c.archived = 'Yes' AND EXISTS (SELECT 1 FROM bids_arc bi WHERE bi.pid = p.pid AND bi.cid = p.cid)
                           OR EXISTS (SELECT 1 FROM bids_retargeting b WHERE b.pid = p.pid)
                           OR EXISTS (SELECT 1 FROM bids_dynamic bd WHERE bd.pid = p.pid)
                       )
     )"]);

    return $full_campaigns;
}


=head2 get_user_camp_gr ($uid, $cid, $search_options, $options)

    Извлекает из базы кампанию вместе с объявлениями
    _gr означает, что метод будет в скором времени переименован в Models::Campaign::get_user_camp взамен Common::get_user_camp

    Входные параметры в точности совпадают с Common::get_user_camp

=cut
sub get_user_camp_gr {
    my ( $uid, $cid, $search_options, $options ) = @_;

    return get_user_camp_gr_mass($uid, [$cid], $search_options, $options)->{$cid};
}


=head2 get_user_camp_gr_mass ($uid, $cids, $search_options, $options)

    Извлекает из базы кампании вместе с объявлениями

=cut
sub get_user_camp_gr_mass {
    my ($uid, $cids, $search_options, $options) = @_;

    return {}  if !@$cids;

    $search_options ||= {};
    $options ||= {};

    croak "Invalid campaign id" if notall {is_valid_id($_)} @$cids;

    my $camps = get_camp_info($cids, $uid, %{hash_cut $options, qw/client_nds client_discount without_multipliers/});
    return {}  if !@$camps;

    my $all_campaigns_goals = CampaignTools::get_campaigns_goals($cids);

    my @order_ids =
        map {$_->{OrderID}}
        grep {$_->{day_budget} && $_->{day_budget} > 0 && $_->{OrderID} && $_->{OrderID} > 0}
        @$camps;
    my $spent_today = Stat::OrderStatDay::get_order_spent_today_multi(\@order_ids);

    my @cids = map {$_->{cid}} @$camps;
    my $campaign_tags = Tag::mass_get_all_campaign_tags(\@cids);
    my $untagged_groups_num = Tag::mass_get_untagged_groups_num(\@cids);

    my $check_block_money_camps = mass_check_block_money_camps($camps);

    my $mediaplan_banners_count = Mediaplan::get_mediaplan_banners_count_mass(\@cids);

    my $client_id = get_clientid(uid => $uid);
    my $check_add_groups = Client::check_add_client_groups_limits_mass($client_id => \@cids, {new_groups => 1});

    my $available_images = BannerImages::get_campaign_available_images_mass(\@cids);
    my $available_pictures = Direct::ImageFormats->get_by(client_id => $client_id, filter => {image_type => 'image_ad'})->to_template_array();

    my $client_limits = get_client_limits($client_id);

    my $camp_status_by_cid = Campaign::CalcCampStatus_mass($camps, %{hash_cut($options, qw/easy_user/)});

    my $last_month_shows = {};
    if (my @mcb_cids = map {$_->{cid}} grep {$_->{type} eq 'mcb'} @$camps) {
        $last_month_shows = get_hash_sql(PPC(cid => \@mcb_cids), [
                "SELECT cid, sum(last_month_shows)
                FROM media_groups",
                where => { cid => \@mcb_cids },
                'GROUP BY cid',
            ]);
    }

    my $groups_on_page = $options->{groups_num} || ($camps->[0] && $camps->[0]->{banners_per_page}) || $options->{optimal_groups_num} || $Settings::DEFAULT_GROUPS_ON_PAGE;

    # query groups
    my $all_groups = [];
    my $groups_result_options = {};

    if (!$search_options->{no_groups}) {
        my $adgroup_types =
            $search_options->{adgroup_types} ||
            [ uniq map {@$_} map {get_camp_supported_adgroup_types(type => $_->{type})} @$camps ];

        my $groups_search_options = hash_merge {
            uid => $uid,
            adgroup_types => $adgroup_types,
        },
        hash_cut($search_options, qw/bid context phrase pid group_name tab/);

        if ( $search_options->{page} && is_valid_id($search_options->{page})) {
            croak 'Paging is not supported for multi-campaign query'  if @$camps > 1;
            $groups_search_options->{limit} = $groups_on_page;
            $groups_search_options->{offset} = ($groups_on_page * ($search_options->{page} - 1));
        }

        my $max_spent_today = max(values %$spent_today);
        my $groups_options = hash_merge $options, {
            get_auction => (defined $options->{get_auction} ? $options->{get_auction} : 1),
            get_auction_for_search_stop_too => $options->{get_auction_for_search_stop_too},
            get_add_camp_options => 1,
            camp_spent_today => $max_spent_today,
            pass_empty_groups => $options->{pass_empty_groups} // 1,
            get_tags => (defined $options->{get_tags} ? $options->{get_tags} : 1),
            # получать данные по цене, показам, кликам из БК(для стратегии "отдельное размещение")
        };

        # если указаны метки - выбирать объявления с метками, добавляем условие в поиск баннеров
        if ($search_options->{tag}) {
            $groups_search_options->{tag_id} = $search_options->{tag};
            $_->{is_search_tags} = 1  for @$camps;
        }

        if (defined $search_options->{tab}) {
            my $tab_status_bids_filter = Models::AdGroupFilters::filter_banners_preview($search_options, do_not_devide_pid_bid => 1);
            hash_merge $groups_search_options, $tab_status_bids_filter;
        }

        # группы для different_places-кампаний получаем с отдельными параметрами
        if (my @cids_w_dp = map {$_->{cid}} grep {($_->{strategy} || '') eq 'different_places'} @$camps) {
            my ($gr, $gro) = Models::AdGroup::get_groups_gr(
                hash_merge({cid => \@cids_w_dp}, $groups_search_options),
                hash_merge({ctx_from_bs => 1, get_all_phrases => 1}, $groups_options),
            );
            push @$all_groups, @$gr;
            $groups_result_options->{total} += $gro->{total};
            $groups_result_options->{were_groups_cut_by_banner_limit} ||= $gro->{were_groups_cut_by_banner_limit};
        }

        if (my @cids_wo_dp = map {$_->{cid}} grep {($_->{strategy} || '') ne 'different_places'} @$camps) {
            my ($gr, $gro) = Models::AdGroup::get_groups_gr(
                hash_merge({cid => \@cids_wo_dp}, $groups_search_options),
                hash_merge({ctx_from_bs => 0}, $groups_options),
            );
            push @$all_groups, @$gr;
            $groups_result_options->{total} += $gro->{total};
            $groups_result_options->{were_groups_cut_by_banner_limit} ||= $gro->{were_groups_cut_by_banner_limit};
        }
    }

    # Все баннеры всех групп. Ищем для них данные о модерации
    my @bids = map { map {$_->{bid}} @{$_->{banners}} } @$all_groups;
    my $mod_edited_banners = is_banner_edited_by_moderator_multi(\@bids);

    # distribute and fill data
    for my $camp (@$camps) {
        my $cid = $camp->{cid};

        if ($camp->{till_date} && is_valid_date($camp->{autobudget_date})) {
            # задана желаемая дата окончания показов
            $camp->{till_date} = minstr($camp->{till_date},$camp->{autobudget_date});
        }

        $camp->{is_camp_locked} = LockObject->new({object_type=>'campaign', object_id => $cid})->load() ? 1 : 0;

        if ($camp->{autobudget} eq 'Yes') {
            $camp->{manual_autobudget_sum} ||= $camp->{strategy_decoded}->{sum};
        }

        $camp->{campaign_goals} = $all_campaigns_goals->{campaigns_goals}->{$cid} || {};
        # goal counts?

        # Чтобы не вытаскивать все группы, считаем сколько их надо. camp->{banners_per_page} - поле в БД, не переименовано.
        $camp->{groups_on_page} = $groups_on_page;
        $camp->{optimal_groups_on_page} = $options->{optimal_groups_num} || $Settings::DEFAULT_GROUPS_ON_PAGE;

        #есть функция для подсчета этих значений, ее и оставить, а это не надо
        $camp->{wallet} = {map {$_ => delete $camp->{"wallet_$_"} || 0} qw/is_enabled cid sum_spent sum total sum_to_pay sum_last/};

        $camp->{total} = $camp->{sum} - $camp->{sum_spent};
        $camp->{total} += $camp->{wallet}->{total} if $camp->{wallet}->{total};
        $camp->{sum} += $camp->{wallet}->{sum} if $camp->{wallet}->{sum};
        # специальный флажок, указывающий, что к суммам самой кампании уже добавлены суммы кошелька.
        # нужен для правильной работы get_camp_status_info
        $camp->{sum_counted_with_wallet} = 1 if $camp->{wallet}->{sum};

        Campaign::convert_dates_for_template($camp, keep_source_data => 1, with_old_start_date_format => 1);
        $camp->{spent_today} = $spent_today->{$camp->{OrderID}}  if $spent_today->{$camp->{OrderID}};

        $camp->{strategy} = Campaign::campaign_strategy($camp);

        $camp->{tags} = $campaign_tags->{$cid} || [];
        $camp->{untagged_banners_num} = $untagged_groups_num->{$cid} || 0;

        $camp->{money_type} = $check_block_money_camps->{$cid} ? 'blocked' : 'real';
        $camp->{status} = $camp_status_by_cid->{$cid};
        $camp->{timetarget_coef} = TimeTarget::timetarget_current_coef($camp->{timeTarget}, $camp->{timezone_id});

        my %WARNPLACES;
        if ( !$search_options->{no_groups} ) {
            $camp->{were_groups_cut_by_banner_limit} =  $groups_result_options->{were_groups_cut_by_banner_limit};
            $camp->{tabclass_search_count} = $camp->{all_banners_num} =  $groups_result_options->{total};
            $camp->{pages_num} = int(1 + ($groups_result_options->{total} - 1) / $groups_on_page);

            my $groups = [grep {$_->{cid} == $cid} @$all_groups];

            for my $group (@{$groups}) {
                BannerFlags::get_retargeting_warnings_flags($group, detailed => $options->{detailed_retargeting_warnings});

                # проверяем, был ли баннер отредактирован модератором
                foreach my $banner (@{$group->{banners}}) {
                    $banner->{is_edited_by_moderator} = $banner->{bid} && $mod_edited_banners->{$banner->{bid}};
                }

                for ( $group->{pid} ? @{$group->{phrases}} : () ) {
                    if ( $_->{place} && $_->{place} > 0 && $_->{place} > calcPlace($_->{price}, $_->{guarantee}, $_->{premium}) ) {
                        $WARNPLACES{"_".$group->{pid}."_".$_->{id}} = $_->{place};
                    }
                }

                if ($options->{add_group_oversized_warnings}) {
                    $group->{is_oversized} = Models::AdGroup::is_group_oversized($group, client_id => $camp->{ClientID});
                }
            }

            $camp->{groups} = $groups;
            # устанавливаем has_banners только если выбрали хотя бы один баннер. если мы не выбрали баннеров, то это ещё не значит, что их нет.
            $camp->{has_groups} = 1 if @$groups;
            $camp->{warnplaces} = keys %WARNPLACES;
            $camp->{warnplaces_str} = "( {".join(",", map {"\"$_\" : ".$WARNPLACES{$_}} (keys %WARNPLACES))."} )";
        }

        # !!! get_media_groups in not mass - skipped due to obsoleteness and complexity
        if ($camp->{type} eq 'mcb') {
            $camp->{last_month_shows} = $last_month_shows->{$cid} || 0;
            $camp->{lowMonthShowsNoPay} = ($camp->{last_month_shows} and $camp->{last_month_shows} < $Settings::MIN_MONTH_MEDIA_SHOWS) ? 1 : 0 ;

            if (!$search_options->{no_groups}) {
                my $mg_filter = {
                    cid         => $camp->{cid},
                    no_banners  => 0,
                    mediaType   => $camp->{type},
                    tab         => $search_options->{tab},
                };
                $camp->{media_groups} = [ get_media_groups($mg_filter) ];
            }
        }

        $camp->{mediaplan}->{count} = $mediaplan_banners_count->{$cid} || 0;
        $camp->{exceed_limit_banners} = $check_add_groups->{$cid};

        $camp->{images} = $available_images->{$cid} || [];
        $camp->{pictures} = $available_pictures;

        # !!! rbac calls is not mass - used in single-cid calls only
        if (defined $options->{rbac}) {
            $camp->{agency_uid} = rbac_is_agencycampaign($options->{rbac}, $cid);
            $camp->{manager_uid} = rbac_is_scampaign($options->{rbac}, $cid) if ! $camp->{agency_uid};
            if ($camp->{agency_uid}) {
                $camp->{agency_info} = get_user_info($camp->{agency_uid});
            } elsif ($camp->{manager_uid}) {
                $camp->{manager_info} = get_user_info($camp->{manager_uid});
            }
        }

        $camp->{$_} = ($camp->{opts}->{$_})? 1 : 0  for qw/no_title_substitute enable_cpc_hold no_extended_geotargeting/;

        # set constants
        $camp->{MIN_PHRASE_RANK_WARNING} = $Settings::MIN_PHRASE_RANK_WARNING;
        $camp->{MAX_PHRASE_RANK_WARNING} = $Settings::MAX_PHRASE_RANK_WARNING;
        $camp->{MAX_BANNER_LIMIT} = $client_limits->{banner_count_limit};
        $camp->{MAX_KEYWORD_LIMIT} = $client_limits->{keyword_count_limit};
    }

    my %data_by_cid = map {($_->{cid} => $_)} @$camps;
    return \%data_by_cid;
}

=head2 get_camp_domains_phones

    Функция берет из базы все возможные домены и номера телефонов для кампании.
    Используется для отправки лицензий пользователей.
    На входе:
        cid - номер кампании
    На выходе:
        ссылка на хеш с полями domains и phones

=cut
sub get_camp_domains_phones {
    my $cid = shift;
    my @domains = grep {defined} @{get_one_column_sql(PPC(cid => $cid), "SELECT DISTINCT domain FROM banners WHERE cid = ?", $cid)};
    my @phones = grep {defined} @{get_one_column_sql(PPC(cid => $cid), "SELECT DISTINCT vc.phone FROM campaigns c
                                                                    INNER JOIN banners b on c.cid=b.cid
                                                                    INNER JOIN vcards vc ON b.vcard_id=vc.vcard_id
                                                                    WHERE vc.cid = ?", $cid)};
    return {domains => \@domains,
            phones => \@phones};
}


1;
