package Models::CampaignOperations;

use Direct::Modern;

use Settings;

use List::MoreUtils qw/any/;
use Campaign::Types;
use Yandex::DBTools;
use Yandex::DBShards;
use MTools;
use BalanceQueue;
use LogTools;
use Property;

use base qw/Exporter/;
our @EXPORT = qw/
    db_update_campaign_statusModerate
    check_block_money_camp
    mass_check_block_money_camps
/;

=head2 db_update_campaign_statusModerate(cid, statusModerate)

    Обновляет статус модерации у кампании. 
    Если задано statusModerate - то устанавливает его, 
        если нет - то пересчитывает на основе статусов модерации у всех баннеров в кампании.
                   если баннеров нет -- делает кампанию черновиком
    Если указан флаг $dont_update_db = 1, то вместо применения изменений в базе функция вернёт посчитанные статусы

=cut

sub db_update_campaign_statusModerate
{
    my $cid = shift || return;
    my $Status = shift;
    my $dont_update_db = shift;

    my $current_camp;
    my $new_status_post_moderate;
    
    unless ($Status) {
        state $property = Property->new('AGGREGATED_CAMP_STATUS_MODERATE');
        my $threshold = $property->get(60) || 0;
        if ($dont_update_db) {
            # в этом режиме нужно всегда получать информацию о кампаниях
            $threshold = 0;
        }

        my $type = get_camp_type(cid => $cid) // 'type_undefined';
        if ( camp_kind_in(type => $type, 'base')  or any {$type eq $_} qw/cpm_banner cpm_yndx_frontpage content_promotion/ ) {
            $current_camp = get_one_line_sql( PPC(cid => $cid), qq| SELECT /* sendppc */
                                                        c.statusModerate
                                                        , co.statusPostModerate
                                                        , count(*) cnt
                                                        , sum( IF( b.statusModerate = 'New' or p.statusModerate = 'New', 1, 0 ) ) cnt_new
                                                        , sum( IF( b.statusModerate in ('Yes', 'No') or p.statusModerate in ('Yes', 'No') or b.phoneflag in ('Yes', 'No'), 1, 0 ) ) cnt_mod
                                                        , sum( IF( b.statusModerate = 'Yes'
                                                                   and ( IFNULL(b.href, '') != ''
                                                                      or b.phoneflag = 'Yes'
                                                                      or t.statusModerate = 'Yes'
                                                                      or b.banner_type IN ('dynamic', 'mobile_content')
                                                                      or (b.banner_type IN ('image_ad', 'cpc_video') and c.type = 'mobile_content')
                                                                      or bp.permalink > 0 )
                                                                   and p.statusModerate = 'Yes'
                                                                   and COALESCE(i.statusModerate, b_perf.statusModerate, 'Yes') = 'Yes', 1, 0
                                                                 )
                                                             ) cnt_accept
                                                        , sum( IF( b.statusModerate = 'Yes' and ( IFNULL(b.href, '') = '' and b.phoneflag = 'Wait' ) and p.statusModerate = 'Yes', 1, 0 ) ) cnt_wait
                                                        , sum( IF( b.statusModerate = 'No'
                                                                   or p.statusModerate = 'No'
                                                                   or COALESCE(i.statusModerate, b_perf.statusModerate, 'Yes') = 'No'
                                                                   or (IFNULL(b.href, '') = '' and b.phoneflag = 'No')
                                                                   or (IFNULL(b.href, '') = '' and t.statusModerate = 'No'), 1, 0
                                                                 )
                                                             ) cnt_decline
                                                        , sum( IF((b.statusModerate = 'Yes' or b.statusPostModerate = 'Yes')
                                                                   and COALESCE(i.statusModerate, b_perf.statusModerate, 'Yes') = 'Yes'
                                                                   and (p.statusModerate = 'Yes' or p.statusPostModerate = 'Yes')
                                                                   and (b.href is not null or b.phoneflag = 'Yes'
                                                                       or t.statusModerate = 'Yes'
                                                                       or b.banner_type IN ('dynamic', 'mobile_content') ), 1, 0
                                                                 )
                                                             ) cnt_preliminarily
                                                        , sum( IF(p.statusModerate = 'Yes', 1, 0) ) AS group_cnt_accepted
                                                        , c.AgencyUID, c.ManagerUID, c.OrderID, c.sum, c.sum_to_pay, c.uid
                                                        , c.wallet_cid
                                                        , c.type, c.cid
                                                    FROM campaigns c 
                                                        left join camp_options co using(cid)
                                                        JOIN phrases p ON p.cid = c.cid
                                                        JOIN banners b ON p.pid = b.pid
                                                        LEFT JOIN images i ON b.bid = i.bid
                                                        LEFT JOIN banners_performance b_perf on (b.banner_type = 'image_ad' or b.banner_type = 'cpc_video') and b_perf.bid = b.bid
                                                        LEFT JOIN banner_turbolandings t ON (b.cid = t.cid AND b.bid = t.bid)
                                                        LEFT JOIN banner_permalinks bp ON bp.bid = b.bid AND bp.permalink_assign_type = 'manual'
                                                    WHERE
                                                        c.cid = ?
                                                        AND c.ClientID mod 100 >= | 
                                                    .$threshold.' '.
                                                qq| GROUP BY 
                                                        c.statusModerate |, $cid );
        } elsif ( is_media_camp(type => $type) ) {
            LogTools::log_stacktrace("DIRECT-72895" => "called on media camp");
            # TODO оторвать эту ветку кода
            $current_camp = get_one_line_sql( PPC(cid => $cid), qq| SELECT /* sendppc */
                                                        c.statusModerate
                                                        , co.statusPostModerate
                                                        , count(*) cnt
                                                        , sum( IF( g.statusModerate = 'New' or b.statusModerate = 'New', 1, 0 ) ) cnt_new
                                                        , sum( IF( g.statusModerate in ('Yes', 'No') or b.statusModerate in ('Yes', 'No'), 1, 0 ) ) cnt_mod
                                                        , 0 cnt_accept
                                                        , 0 cnt_wait
                                                        , 0 cnt_preliminarily
                                                        , sum( IF( b.statusModerate = 'No' or g.statusModerate = 'No', 1, 0 ) ) cnt_decline
                                                        , c.AgencyUID, c.ManagerUID, c.OrderID, c.sum, c.sum_to_pay, c.uid
                                                        , c.wallet_cid
                                                        , c.type, c.cid
                                                   FROM campaigns c
                                                   left join camp_options co using(cid)
                                              LEFT JOIN media_groups g USING(cid)
                                              LEFT JOIN media_banners b USING(mgid)
                                                  WHERE c.cid = ?
                                               GROUP BY c.statusModerate |, $cid );

            
            # Базовые баннеры, прошедшие модерацию
            my $moderated_md5s = get_one_column_sql(PPC(cid => $cid),"
                                SELECT distinct(b.md5_picture)
                                  FROM campaigns c 
                                  JOIN media_groups g USING(cid)
                                  JOIN media_banners b USING(mgid)
                                 WHERE g.statusModerate = 'Yes'
                                   AND b.statusModerate = 'Yes'
                                   AND c.cid = ?",
                                                    $cid) || [];
            $current_camp->{cnt_preliminarily} = $current_camp->{cnt_accept} = get_one_field_sql(PPCDICT, ["
                SELECT COUNT(*)
                  FROM media_files",
                 WHERE => {
                     format_id => base_media_format(cid => $cid),
                     md5 => $moderated_md5s,
                    }]);
        } else {
            warn "unknown campaign type for cid $cid";
        }

        $Status = calc_campaign_status_moderate($current_camp);
        unless ($Status) {
            if ($dont_update_db) {
                return {
                    statusModerate     => undef,
                    statusPostModerate => undef,
                };
            } else {
                return;
            }
        }

    } else {
        # получаем статусы модерации для кампании 
        $current_camp = get_one_line_sql(PPC(cid => $cid), "select
                                                    c.OrderID, c.ManagerUID, c.AgencyUID,
                                                    c.statusModerate, co.statusPostModerate,
                                                    c.sum, c.sum_to_pay, c.type,
                                                    c.wallet_cid,
                                                    c.uid, c.cid
                                                 from campaigns c 
                                                         left join camp_options co using(cid) 
                                                where c.cid = ?", $cid);
    }

    $new_status_post_moderate = calc_campaign_status_post_moderate($current_camp, StatusModerate => $Status);
    
    my $exist_blocked_money;

    if ($current_camp
            && ref $current_camp eq 'HASH' 
            && $current_camp->{statusPostModerate} && $current_camp->{statusPostModerate} ne 'Accepted'
            && $new_status_post_moderate && $new_status_post_moderate eq 'Accepted') {

        # sending notification to balance only if exists check for pay or money on campaign
        # check_block_money_camp для кампаний на общем счёте смотрит на statusPostModerate из базы
        # поэтому проверяем _до_ того, как обновить что-то в базе
        $exist_blocked_money = check_block_money_camp($current_camp);
    }

    # если поменялся statusPostModerate - переотправляем кампанию в БК
    # undef означает, что статус обновлять не надо (решаем таким образом (undef вместо предыдущего значения) в том числе проблему гонок)

    my $new_statusBsSynced;
    if (! $current_camp->{statusPostModerate}
        || (
            $new_status_post_moderate
            && $current_camp->{statusPostModerate} eq $new_status_post_moderate
            # закрываем случай гонок с экспортом в БК, когда Yes на кампании ставится с большим опозданием от баннеров
            && ($current_camp->{statusModerate} eq 'Yes' || $Status ne 'Yes' || $current_camp->{OrderID} )
            )
    ) {
        # empty
    } else {
        # переотправляем в БК, если были изменения в статусе
        $new_statusBsSynced = 'No';
    }

    if ($dont_update_db) {
        return {
            statusModerate     => $Status,
            statusPostModerate => $new_status_post_moderate,
        };
    }

    do_sql(PPC(cid => $cid), "UPDATE campaigns c
                         left join camp_options co using(cid)
                     SET c.statusModerate = ?
                       , start_time = start_time
                       , co.statusPostModerate = IF(co.statusPostModerate = 'Accepted', 'Accepted', IFNULL(?, co.statusPostModerate))
                       , c.statusBsSynced = IFNULL(?, c.statusBsSynced)
                   WHERE cid = ?
             ", $Status, $new_status_post_moderate, $new_statusBsSynced, $cid);

    if ($exist_blocked_money) {
        my @cids_to_send_to_balance = ($cid);
        if ($current_camp && $current_camp->{wallet_cid}) {
            my $cids_under_wallet = get_one_column_sql(PPC(cid => $cid), ['
                SELECT cid
                FROM campaigns
            ',  WHERE => {
                    'uid' => $current_camp->{uid},
                    'wallet_cid' => $current_camp->{wallet_cid},
                    'type' => get_camp_kind_types('under_wallet'),
                },
            ]);
            push @cids_to_send_to_balance, @$cids_under_wallet;
        }
        BalanceQueue::add_to_balance_info_queue($current_camp->{uid},
            cid => \@cids_to_send_to_balance,
            BalanceQueue::PRIORITY_CAMP_WITH_BLOCKED_MONEY,
        );
    }

    return $Status;
}

=head2 calc_campaign_status_moderate

    Вычисляет правильное значение для campaigns.statusModerate

    Вход:
        {
            statusModerate =>
            , cnt =>
            , cnt_new =>
            , cnt_accept =>
            , cnt_declined =>
            , cnt_preliminarily =>
            , cnt_mod =>
            , cnt_wait =>
        }

=cut

sub calc_campaign_status_moderate($)
{
    my $current_camp = shift;

    if (camp_kind_in(type => $current_camp->{type}, 'internal')) {
        # для кампаний внутренней рекламы statusModerate всегда Yes: https://st.yandex-team.ru/DIRECT-150642#611b56247750a94c3aec22d9
        return 'Yes';
    }

    my $new_status_moderate;

    if (
       !exists $current_camp->{statusModerate}                     # почему то не получили текущий статус кампании
        || ( $current_camp->{statusModerate} eq 'New' && !$current_camp->{cnt_mod} )    # кампания черновик и нет отмодерированных баннеров
        || $current_camp->{statusModerate} eq 'Ready'              # кампания ожидает отправки на модерацию
        || $current_camp->{cnt_new} == $current_camp->{cnt}        # все баннеры черновики
        || $current_camp->{cnt_wait} == $current_camp->{cnt}       # все баннеры с контакной информацией, которая ожидает модерации
        || ( $current_camp->{cnt_mod}
             && !$current_camp->{cnt_accept}
             && !$current_camp->{cnt_preliminarily}
             && !$current_camp->{cnt_decline})
        || !($current_camp->{cnt_mod} || $current_camp->{cnt_preliminarily}) # нет полностью/частично/предварительно промодерированных баннеро
     ) {
            return undef; # current status
       } else {
            my $has_accepted = $current_camp->{cnt_accept} > 0
                                || $current_camp->{cnt_preliminarily} > 0;
            $new_status_moderate = $has_accepted ? 'Yes' : 'No';
       }

    return $new_status_moderate;
}

=head2 calc_campaign_status_post_moderate

    Вычисляет правильное значение для camp_options.statusPostModerate

=cut

sub calc_campaign_status_post_moderate($;%)
{
    my ($current_camp, %O) = @_;

    # если кампания когда-либо была принята, либо принята сейчас -- то Accepted,
    #   иначе -- при наличии денег или счетов - не меняем,
    #   иначе -- запрещаем оплату

    my $status_post_moderate = $current_camp->{statusPostModerate} eq 'Accepted' || $O{StatusModerate} eq 'Yes' ?
                                'Accepted'
                                : (check_block_money_camp($current_camp) ? undef : 'No');

    return $status_post_moderate;
}

=head2 check_block_money_camp($campaign, %O)

    Определяет какие на кампании есть либо будут деньги - заблокированные, либо нет.
    Условие: не была в БК, не сервисируемая, не агентсткая, и (предварительно принята, либо отклонена на модерации)

    Такие условия выставлены из-за того, что нет возможности получить от Биллинга тип денег на кампании

    Опции:
        is_enable_wallet -- перенос для включения общего счета (используется для определения промодерированности кампании общего счета)
        type_only -- not delete condition for sum_to_pay (needed for moderateClientNew.pl)

    для общего счета нужно в $camp_row передать поля wallet_cid, uid
    https://jira.yandex-team.ru/browse/DIRECT-22284

=cut

sub check_block_money_camp {
    my ($camp) = shift;
    return mass_check_block_money_camps([$camp], @_)->{$camp->{cid}};
}

=head2 mass_check_block_money_camps($campaigns, %O)

    Определяет какие на кампаниях есть либо будут деньги - заблокированные, либо нет.
    Условие: не была в БК, не сервисируемая, не агентсткая, и (предварительно принята, либо отклонена на модерации)
    Формат кампании:
    {
        'AgencyUID' => undef,
        'ManagerUID' => undef,
        'OrderID' => '0',
        'statusModerate' => 'Sent',
        'statusPostModerate' => 'New',
        'sum' => '0.000000',
        'sum_to_pay' => '0.000000',
        'type' => 'text',
        'uid' => '278863838',
        'cid' => 1234,
        'wallet_cid' => '0'
    }

    Такие условия выставлены из-за того, что нет возможности получить от Биллинга тип денег на кампании

    Опции:
        is_enable_wallet -- перенос для включения общего счета (используется для определения промодерированности кампании общего счета)
        type_only -- not delete condition for sum_to_pay (needed for moderateClientNew.pl)

    для общего счета нужно в элементах $campaigns передать поля wallet_cid, uid
    https://jira.yandex-team.ru/browse/DIRECT-22284

=cut

sub mass_check_block_money_camps {
    my ( $camps, %opt ) = @_;

    my %res;
    my %cid_to_wallet;
    my %uids;
    my %wallet_ids;
    foreach my $camp ( @$camps ) {

        my $type = $camp->{type} || $camp->{mediaType};
        die 'campaign type not defined' unless $type;

        if ( ( $camp->{wallet_cid} || $type eq 'wallet' ) && $camp->{uid} && ! $opt{is_enable_wallet} ) {
            # есть общий счет на кампании или запросили сам общий счет, считаем с учетом всех кампаний в группе с одним общим счетом
            # если вычисляем статус кампании при включении счета ($opt{is_enable_wallet}), то не нужно смотреть на кампании под счетом
            my $wallet_cid = $type eq 'wallet' ? $camp->{cid} : $camp->{wallet_cid};

            $cid_to_wallet{ $camp->{cid} } = $wallet_cid;

            $uids{ $camp->{uid} }      = undef;
            $wallet_ids{ $wallet_cid } = undef;
        }
        
        $res{ $camp->{cid} } = [ $camp ]; # если под кошельком ничего нет, смотрим на сам кошелёк
    }

    my %camps_by_wallet;
    if ( keys %uids and keys %wallet_ids ) {

        my $camps_wallets = get_all_sql(
            PPC(uid => [ keys %uids ]),
            [
                'select c.wallet_cid
                        , c.ManagerUID
                        , c.AgencyUID
                        , (c.sum + IF(wc.cid, wc.sum, 0)) AS sum
                        , (c.sum_to_pay + IF(wc.cid, wc.sum_to_pay, 0)) AS sum_to_pay
                        , c.statusModerate
                        , co.statusPostModerate
                   from campaigns c
                   left join camp_options co using(cid)
                   left join campaigns wc on wc.cid = c.wallet_cid
                ',
                WHERE => {
                    'c.uid'         => SHARD_IDS,
                    'c.wallet_cid'  => [ keys %wallet_ids ],
                    'c.type'        => get_camp_kind_types('under_wallet'),
                    'c.statusEmpty' => 'No',
                }
            ]
        );

        foreach my $row ( @$camps_wallets ) {
            my $wallet_cid = $row->{'wallet_cid'};
            $camps_by_wallet{ $wallet_cid } ||= [];
            push @{ $camps_by_wallet{ $wallet_cid } }, $row;
        }
    }

    my %blocked_money;
    foreach my $camp ( @$camps ) {

        my $wallet_cid = $cid_to_wallet{ $camp->{cid} };
        my $campaigns = ( defined( $wallet_cid ) && exists( $camps_by_wallet{ $wallet_cid } ) )
                            ? $camps_by_wallet{ $wallet_cid }
                            : $res{ $camp->{cid} };

        my $blocked_money = 1; # если хотя бы на одной кампании не блокируем деньги, то не блокируем всю группу на общем счете
        my $sum = 0;
        my $sum_to_pay = 0;
        for my $camp_info ( @$campaigns ) {
            $blocked_money = $blocked_money
                             && ! $camp_info->{ManagerUID}
                             && ! $camp_info->{AgencyUID}
                             && (($camp_info->{statusPostModerate} || '' ) ne 'Accepted')
                             && ( $camp_info->{statusPostModerate} && $camp_info->{statusPostModerate} eq "Yes" 
                                         || $camp_info->{statusModerate} && $camp_info->{statusModerate} =~ /^(No|Ready|Sent|New)$/) 
                                 ? 1 : 0;

            $sum += $camp_info->{sum} // 0;
            $sum_to_pay += $camp_info->{sum_to_pay} // 0;
        }
        if (! $opt{type_only}) {
            # not delete condition for sum_to_pay !!!
            #   needed for moderateClientNew.pl
            $blocked_money = $blocked_money && ($sum || $sum_to_pay) ? 1 : 0;
        }

        $blocked_money{ $camp->{cid} } = $blocked_money;
    }

    return \%blocked_money;
}
1;
