use strict;
use warnings;
use utf8;

package Models::Phrase;

use Yandex::DBTools;
use Yandex::DBShards;
use List::Util qw/first max/;
use List::MoreUtils qw/any uniq/;
use Yandex::HashUtils;
use Yandex::I18n;

use Settings;
use PhraseText;
use PlacePrice qw/calcPlace/;
use Currencies;
use CampAutoPrice::Common;
use Primitives;
use PrimitivesIds;
use Tools;
use MailNotification;
use HashingTools;
use LogTools;
use Models::PhraseTools;
use Direct::Validation::Keywords;
use PhrasePrice qw//;

use base 'Exporter';
our @EXPORT = qw/
    mass_add_update_phrase
    mass_update_phrases_user_params
    validate_phrase_one_href_param
    validate_phrases
    get_phrases
/;

use Data::Dumper;

our @BIDS_HREF_PARAMS = qw/Param1 Param2/;


=head2 save_phrases($campaign, $new_group, $old_group, %options)

# %options 
#    uid
# $new_group должен содержать
#    bid - номер любого баннера из группы(используется для отправки уведомлений "измененния по баннеру ..." и 
#    для сохранения старых связей bids.bid)  

=cut
sub save_phrases {
    my ($campaign, $new_group, $old_group, %options) = @_;
    my $currency = $campaign->{currency};
    die 'no currency given' unless $currency;

    my $log = LogTools::messages_logger("UpdateBannerPhrases");
    my %params = (%options, 'log' => $log);

    my $min_price_constant = $options{is_cpm_campaign} ? 'MIN_CPM_PRICE' : 'MIN_PRICE';
    my $min_price = ($campaign->{autobudget} eq 'Yes' || $campaign->{strategy}->{is_search_stop}) ? 0 : get_currency_constant($currency, $min_price_constant);

    my %phrases = merge_with_old_phrases($new_group,
        $new_group->{Phrases} || $new_group->{phrases},
        (ref $old_group->{phrases} eq 'ARRAY' ? $old_group->{phrases} : $old_group->{phr}),
        %params, min_price => $min_price);

    my @notifications = @{$phrases{notifications}};
    my @new_phrases = @{$phrases{phrases_for_add_update}};

    my @up_phrases = @{$phrases{up_phrases}};
    my @up_params = @{$phrases{up_params}};

    my ($norm_phrases, $groups_to_moderate);
    if (@new_phrases) {
        my $res = mass_add_update_phrase($campaign, \@new_phrases, %options);
        ($norm_phrases, $groups_to_moderate) = @$res{qw/norm_phrases groups_to_moderate/};
    }

    _update_live_phrases($new_group, \@up_phrases, where_from => $options{where_from}) if @up_phrases;

    if (@up_params && $options{i_know_href_params}) {
        mass_update_phrases_user_params({map {
            ($_->{id} => hash_cut $_, @BIDS_HREF_PARAMS, 'to_delete', 'cid')
        } @up_params});
        if (my @phrases = grep {!$_->{to_delete}} @up_params) {
            foreach_shard cid => \@phrases, sub {
                my ($shard, $phrases_chunk) = @_;
                my @ids = map {$_->{id}} @$phrases_chunk;
                do_update_table(PPC(shard => $shard),
                    bids => {statusBsSynced => 'No'},
                    where => {id => \@ids},
                );
            };
        }
    }
    
    mass_mail_notification(\@notifications);
    
    my %changes = (
        added_new_phrases => $phrases{added_new_phrases},
        phrase_set_changed => scalar(@new_phrases), # Флажок, если: added_new_phrases или фразы с изменёнными текстами или удалённые фразы
        params_updated => scalar(@up_params),
        phrases_updated => scalar(@up_phrases), 
        changed_suspend_state => $phrases{changed_suspend_state},
        changed_phrase_data => $phrases{changed_phrase_data},
        delete_params => (any {$_->{to_delete}} @up_params),
        moderate_new => $groups_to_moderate && scalar @$groups_to_moderate ? 1 : 0,
    ); 
    return($norm_phrases || {}, \%changes);  
}

=head2 _update_live_phrases($group, $phrases, %options)

    options:
        where_from => откуда обновляют фразы, нужно для логирования ("web", "api", "xls", "mediaplan")

=cut

sub _update_live_phrases {
    
    my ($group, $phrases, %options) = @_;
    
    my ($cid, $bid) = @{$group}{qw/cid bid/};
    
    my @log_prices;
    foreach my $phrase (@$phrases) {
        my $row = hash_merge {warn => 'Yes', statusBsSynced => 'No'},
                    hash_cut $phrase, qw/place price price_context autobudgetPriority showsForecast is_suspended/;
        do_update_table(PPC(cid => $cid), "bids", $row, where => {id => $phrase->{id}});
        
        push @log_prices, {
            cid => $cid,
            pid => $group->{pid},
            bid => $bid,
            id => $phrase->{id},
            type => ($options{where_from} // '') eq 'api' ? 'update' : 'update1', # указываем специальный тип при обновлении из API
            price => $phrase->{price},
            price_ctx => $phrase->{price_context},
            currency    => $group->{currency},
        };
    }

    clear_auto_price_queue($cid) if @$phrases;

    LogTools::log_price(\@log_prices);
}

=head2 merge_with_old_phrases ($group, $phrases, $old_phrases, %options)

Соединить фразы из старых и новых групп

Возвращает список:
    (
        notifications => ...,
        phrases_for_add_update => added, deleted bids and bids with changed keyword
        up_params => bids with updated user params (Param1, Param2),
        up_phrases => phrases with modified params (prices etc.)
        changed_suspend_state => ...,
        added_new_phrases => ...,
        changed_phrase_data => ...
    )

=cut
sub merge_with_old_phrases {
    
    my ($group, $phrases, $old_phrases, %options) = @_;
    
    my (%MD5, %IDS);
    for my $ph (@$phrases) {
        ensure_phrase_have_props($ph);
        unless($ph->{md5}) { # пустая фраза
            warn "empty phrase!";
            next;
        }
        $MD5{ $ph->{md5} } = $ph;
        if (defined $ph->{id}) {
            $IDS{ $ph->{id} } = $ph->{md5};
        }
    }
    
    my $uid = $options{uid};
    my ($bid, $pid, $cid) = @{$group}{qw/bid pid cid/};
    # Метод вызывается только в save_phrases, который в свою очередь вызывает ещё mass_mail_notification.
    # mass_mail_notification ещё не переделан под группы и ему пока нужен bid для фразы. 
    # Поэтому, чтобы нотификации еще шли - все же заполняем значение поля bid.
    $bid = get_bids(pid => $pid)->[0] if !defined $bid && $pid;
    my %params = (%options, pid => $pid, cid => $cid, bid => $bid);
    my (@notifications, @up_phrases, @up_params, @phrases_for_add_update, $changed_suspend_state, $changed_phrase_data);
    foreach my $el (@$old_phrases) {
        # пересчитываем md5 на случай смены леммера
        my $md5 = PhraseText::get_phrase_props($el->{phrase})->{md5};
        # Фраза не изменилась
        if (exists $MD5{$md5}) {
            my %result = _bring_previous_phrase($MD5{$md5}, $el, %params);
            push @notifications, @{$result{changes}} if $result{changes};
            push @up_phrases, $result{up_phrase} if $result{up_phrase};
            push @up_params, $result{up_params} if $result{up_params}; 
            push @phrases_for_add_update, $result{add_phrase} if $result{add_phrase};

            $changed_suspend_state ||= $result{changed_suspend_state} if $result{changed_suspend_state};
            $changed_phrase_data ||= $result{changed_phrase_data} if $result{changed_phrase_data};

            delete $MD5{$md5};
            delete $MD5{$IDS{$el->{id}}} if exists $IDS{$el->{id}}; # На случай, если поменялся md5 (смена леммера)
        } elsif (exists $IDS{$el->{id}} && exists $MD5{$IDS{$el->{id}}}) {
            # Изменилась фраза, но не изменился id
            # new_phrase = $MD5{$IDS{$el->{id}}}

            my %result = _bring_changed_phrase($MD5{$IDS{$el->{id}}}, $el, %params);
            push @notifications, @{$result{changes}} if $result{changes};
            push @phrases_for_add_update, $result{add_phrase} if $result{add_phrase};

            $changed_suspend_state ||= $result{changed_suspend_state} if $result{changed_suspend_state};

            delete $MD5{$IDS{$el->{id}}};
        } else {
            # фраза удалена
            push @phrases_for_add_update, { bid => $bid, pid => $pid, cid => $cid, uid => $uid, phrase => '', id => $el->{id}, price => 0 };
            push @up_params, { id => $el->{id}, to_delete => 1, cid => $cid };
            
            # for mail notification (delete phrases)
            push @notifications, { object => 'phrase', event_type => 'ph_change', object_id => $bid, 
                                    old_text => $el->{phrase},  new_text => "", uid => $uid };
            $options{log}->out(sprintf "delete phrase: UID: %s\tcid: %s\tbid: %s\tpid: %s\told_value: %s",
                                $options{UID}, $cid, $bid, $el->{id}, $el->{phrase});                                    
        }
    }
    
    my $added_new_phrases;
    # Добавляем оставшиеся фразы
    for my $ph ( sort { $a->{phrase} cmp $b->{phrase} } values %MD5 ) {
        
        my $place = $options{is_cpm_campaign} ? undef : calcPlace( $ph->{price}, $ph->{guarantee}, $ph->{premium});

        push @phrases_for_add_update, {
            bid => $bid,
            pid => $pid, cid => $cid, uid => $uid, 
            phrase => $ph->{phrase},
            numword => $ph->{numword},
            norm_phrase => $ph->{norm_phrase},
            norm_hash => $ph->{norm_hash},
            md5 => $ph->{md5},
            place => $place,
            price => max($options{min_price}, $ph->{price}),
            price_context => $ph->{price_context},
            autobudgetPriority => $ph->{autobudgetPriority},
            showsForecast => $ph->{showsForecast},
            is_suspended => $ph->{is_suspended} || 0,
            HrefParams => hash_cut($ph, @BIDS_HREF_PARAMS),
            phraseIdHistory => $ph->{phraseIdHistory},
        };
        $added_new_phrases = 1;

        # Записываем изменение в лог.
        $options{log}->out(sprintf "new phrase: UID: %s\tcid: %s\tbid: %s\tnew_value: %s",
                            $options{UID}, $cid, $bid, $ph->{phrase});
    }
    
    return (
        notifications => \@notifications,
        phrases_for_add_update => \@phrases_for_add_update,
        up_params => \@up_params,
        up_phrases => \@up_phrases,
        changed_suspend_state => $changed_suspend_state,
        added_new_phrases => $added_new_phrases,
        changed_phrase_data => $changed_phrase_data,
    )
}


sub _bring_previous_phrase {
    # new phrase, old phrase
    my ($new, $old, %options) = @_;
    my ($bid, $pid, $cid, $uid, $is_different_places) = @options{qw/bid pid cid uid is_different_places/};
    my $is_cpm_campaign = $options{is_cpm_campaign};
    
    my ($up_phrase, $up_params, $add_phrase, $changed_suspend_state, $changed_phrase_data);
    my @changes;
    
    if ($options{ignore_suspended}) {
        $new->{is_suspended} = $old->{is_suspended};
    }

    foreach (qw/price price_context/) {
        $old->{$_} += 0;
        $new->{$_} += 0;
    }

    if ( $new->{phrase} eq $old->{phrase} ) {
        # Изменение цены фразы отслеживаем отдельно. Изменение остальных параметров отслеживаем здесь
        if (   $new->{autobudgetPriority} ne ($old->{autobudgetPriority}||3)
            || (defined $new->{showsForecast} && $new->{showsForecast} ne $old->{showsForecast})
            || $new->{is_suspended} != $old->{is_suspended}
        ) {
            # Выставим флажок для дальнейшей переотправки в БК всей группы
            $changed_phrase_data = 1;
        }

        if (   $new->{price} != $old->{price}
               # price_context учитываем только для стратегий "независимое управление"
            || ($is_different_places && ($new->{price_context} // 0) != ($old->{price_context} // 0))
            || $changed_phrase_data
        ) {

            $up_phrase = {
                id => $old->{id},
                price => $new->{price} || $old->{price},
                price_context => $new->{price_context} || $old->{price_context},
                showsForecast => $new->{showsForecast},
                place => $is_cpm_campaign ? undef : calcPlace($new->{price}, $new->{guarantee}, $new->{premium}),
                autobudgetPriority => validate_priority( $new->{autobudgetPriority} ),
                is_suspended => defined $new->{is_suspended} ? $new->{is_suspended} : $old->{is_suspended},
            };
            # если включается/выключается фраза - нужно переотправлять в БК всю группу, поэтому тащим за собой флажок
            $changed_suspend_state = $up_phrase->{is_suspended} != $old->{is_suspended};
        }

        # обновляем параметры фраз
        if ( _check_change_user_params($old, $new) ) {
            $up_params = hash_cut $new, @BIDS_HREF_PARAMS;
            $up_params->{id} = $old->{id};
            $up_params->{cid} = $cid;
        }
    } else {

        my $place = $is_cpm_campaign ? undef : calcPlace($new->{price}, $new->{guarantee}, $new->{premium});
        $add_phrase = {
            # TODO adgroup: удалить bid => после полного перехода к группам 
            bid => $bid, 
            pid => $pid, cid => $cid, uid => $uid, 
            phrase => $new->{phrase},
            numword => $new->{numword},
            norm_phrase => $new->{norm_phrase},
            norm_hash => $new->{norm_hash},
            md5 => $new->{md5},
            id => $old->{id},
            price => $new->{price},
            price_context => ($old->{price_context})? $new->{price_context} : $old->{price_context},
            place => $place,
            autobudgetPriority => $new->{autobudgetPriority}, 
            showsForecast => $new->{showsForecast},
            phraseIdHistory => $old->{phraseIdHistory},
            statusModerate=>  $old->{statusModerate},
            is_suspended => (defined($new->{is_suspended})) ? $new->{is_suspended} : $old->{is_suspended},
        };
        # т.к. фраза может изменить свой bids.id внутри функции mass_add_update_phrase - передаем и обновляем параметры постоянно =(
        # if (_check_change_user_params($old, $new)) {
            $add_phrase->{HrefParams} = {};
            hash_merge $add_phrase->{HrefParams}, hash_cut $new, @BIDS_HREF_PARAMS;
        # }
        
        # for mail notification (change phrases)
        push @changes, { object  => 'phrase', event_type => 'ph_change', object_id => $bid,
                            old_text => $old->{phrase}, new_text => $new->{phrase}, uid => $uid };
        $options{log}->out(sprintf "edit phrase: UID: %s\tcid: %s\tbid: %s\tpid: %s\told_value: %s\tnew_value: %s",
                            $options{UID}, $cid, $bid, $new->{id}, $old->{phrase}, $new->{phrase});
    }

    # for mail notification (prices)
    if (defined $new->{price} && defined $old->{price} && $new->{price} != $old->{price}) {
        push @changes, { object => 'phrase', event_type => 'ph_price', object_id => $bid,
                            old_text => $old->{price}, new_text => $new->{price}, uid => $uid };
    }

    return (
        up_phrase => $up_phrase,
        up_params => $up_params,
        add_phrase => $add_phrase,
        changed_suspend_state => $changed_suspend_state,
        changed_phrase_data => $changed_phrase_data,
        @changes ? (changes => \@changes) : ()
    );
}

=head2 _check_change_user_params

    проверяет есть ли измененные параметры для фраз

=cut
sub _check_change_user_params
{
    my ($old, $new) = @_;

    return scalar
                grep {
                    my $oldv = "".(defined $old->{lc($_)} ? $old->{lc($_)} : "");
                    my $newv = "".(defined $new->{$_} ? $new->{$_} : "");

                    $oldv ne $newv;
                } @BIDS_HREF_PARAMS;
}

sub _bring_changed_phrase {
    
    # new phrase, old phrase
    my ($new, $old, %options) = @_;
    my ($bid, $pid, $cid, $uid) = @options{qw/bid pid cid uid/};
    my @changes;
    
    if ($options{ignore_suspended}) {
        $new->{is_suspended} = $old->{is_suspended};
    }
    my $place = calcPlace( $new->{price}, $new->{guarantee}, $new->{premium} );
    my $add_phrase = {
        bid => $bid,
        pid => $pid, cid => $cid, uid => $uid, 
        phrase => $new->{phrase},
        numword => $new->{numword},
        norm_phrase => $new->{norm_phrase},
        norm_hash => $new->{norm_hash},
        md5 => $new->{md5},
        id => $old->{id},
        price => $new->{price},
        price_context => max($options{min_price}, $new->{price_context}),
        place => $place,
        autobudgetPriority => $new->{autobudgetPriority}, 
        showsForecast => $new->{showsForecast},
        statusModerate=>  $old->{statusModerate},
        phraseIdHistory => $old->{phraseIdHistory},
        is_suspended => (defined($new->{is_suspended})) ? $new->{is_suspended} : $old->{is_suspended},
    };
    # т.к. фраза может изменить свой bids.id внутри функции mass_add_update_phrase - передаем и обновляем параметры постоянно =(
    # if (_check_change_user_params($old, $new) ) {
        $add_phrase->{HrefParams} = {};
        hash_merge $add_phrase->{HrefParams}, hash_cut $new, @BIDS_HREF_PARAMS;
    # }

    # for mail notification (change phrases)
    push @changes, { object => 'phrase', event_type => 'ph_change', object_id => $bid,
                        old_text => $old->{phrase}, new_text => $new->{phrase}, uid => $uid };
    $options{log}->out(sprintf "edit phrase: UID: %s\tcid: %s\tbid: %s\tpid: %s\told_value: %s\tnew_value: %s",
                        $options{UID}, $cid, $bid, $new->{id}, $old->{phrase}, $new->{phrase});                        
    return (
        add_phrase => $add_phrase,
        changed_suspend_state => $add_phrase->{is_suspended} != $old->{is_suspended},
        @changes ? (changes => \@changes) : ()
    )            
}

=head2 mass_add_update_phrase

    Обновляет, удаляет и добавляет фразы для баннера. Массовая версия.

    Параметры позиционные:
        campaign - (strategy, ContextPriceCoef, currency) для расчета логгируемой ставки в сети, strategy => {name => , net => {}, search => {}}
        phrases - ссылка на массив

        OPTS 
            from_stat (0|1) - изменения приехали со страниц статистики (для логгирования)
            i_know_href_param - поддержка обратной совместимости для param1|param2

=cut
sub mass_add_update_phrase($$;%) {
    my ($campaign, $phrases, %OPT) = @_;

    local $campaign->{strategy} = $campaign->{strategy}->{name};

    # список полей для sql запроса
    my @fields = qw/
                    cid
                    pid
                    phrase 
                    place 
                    norm_phrase 
                    numword 
                    price 
                    autobudgetPriority 
                    showsForecast 
                    warn 
                    price_context
                    is_suspended
                    /;

    my @mail_notifications;
    my @log_prices;
    my $norm_to_id = {};
    my $groups_to_moderate = {}; # Группы (условия показа), которые нужно отправить на модерацию, т.к. были добавлены фразы с statusModerate = 'New'
    my $href_params_by_norm = {};
    my @up_href_params = ();

    die 'no cid or pid in phrase at mass_add_update_phrase ' . Dumper($phrases) if any {!$_->{pid} || !$_->{cid}} @$phrases;
    foreach_shard cid => $phrases, sub {
        my ($shard, $phrases_chunk) = @_;

        my %inserted_phrases; # после добавления фраз в бд - отсюда будем вытаскивать данные для log_price
        my @phrases_to_insert_update; # массив ссылок на массив для do_mass_insert_sql
        my @phrases_to_delete; # массив фраз(id+cid) для массового удаления
        my %phraseid_history; # хэш bid -> norm_phrase -> {cid => ..., phraseIdHistory => ...}

        # получаем старые версии фраз, для которых указан id
        my @old_phrases_ids = map {$_->{id}} grep {$_->{id}} @$phrases_chunk;
        
        my $old_phrases;
        if ( scalar @old_phrases_ids ) {
            $old_phrases = get_hashes_hash_sql(PPC(shard => $shard), [
                "SELECT id, phrase as phrase_old, statusModerate FROM bids", 
                where => { id => \@old_phrases_ids }]
            );
        }

        foreach my $phrase (@$phrases_chunk) {
            $phrase->{phrase} =~ s/[,]/ /g;
            $phrase->{phrase} =~ s/^\s*(.*?)\s*$/$1/;

            if ($phrase->{phrase} eq '') {
                if ($phrase->{id}) {
                    push @phrases_to_delete, hash_cut $phrase, qw/id cid/;
                    push @log_prices, {
                        cid => $phrase->{cid},
                        bid => $phrase->{bid},
                        pid => $phrase->{pid},
                        id => $phrase->{id},
                        type => 'delete2',
                        currency => $campaign->{currency},
                    };
                }
                next;
            }

            my $phrase_values = _prepare_phrase_for_db(%$phrase);
            if (defined $phrase->{HrefParams}) {
                $href_params_by_norm->{ $phrase_values->{norm_phrase} } = hash_merge(
                    { cid => $phrase->{cid} },
                    hash_cut($phrase->{HrefParams}, @BIDS_HREF_PARAMS)
                );
            }

            $phrase_values->{price_context} = defined $phrase->{price_context} ? $phrase->{price_context} : 0;

            my $should_null_phrase_id = 0;
            if ($phrase->{id}) {
                $old_phrases->{$phrase->{id}}{phrase} = $phrase_values->{phrase};
                $should_null_phrase_id = phrase_should_null_phrase_id($old_phrases->{$phrase->{id}});
            }

            # Если id не задан или надо удалить phrase_id и добавить фразу заново - попадаем сюда
            if ( !$phrase->{id} || $should_null_phrase_id ) {
                if ($phrase->{id}) {
                    # копируем параметры от старой фразы к новой, и затем удаляем от старой
                    push @up_href_params, {
                        to_copy => 1,
                        from_id => $phrase->{id},
                        to_phrase_norm => $phrase_values->{norm_phrase},
                        cid => $phrase->{cid},
                    };
                    push @up_href_params, {
                        to_delete => 1,
                        id => $phrase->{id},
                        cid => $phrase->{cid},
                    };
                    
                    push @phrases_to_delete, hash_cut $phrase, qw/id cid/;
                    push @log_prices, {
                        cid => $phrase->{cid},
                        bid => $phrase->{bid},
                        pid => $phrase->{pid},
                        id => $phrase->{id},
                        type => 'delete3',
                        currency => $campaign->{currency},
                    };
                    
                    if ($should_null_phrase_id) {
                        $phrase_values->{phraseIdHistory} = undef;
                    } 
                }
                $phrase_values->{statusModerate} = $phrase->{statusModerate} 
                                                && $phrase->{statusModerate} =~ /^(New|Yes|No)$/ 
                                                && !$should_null_phrase_id 
                                    ? $phrase->{statusModerate} : 'New';

                # Пометим группу (условие показа) на перемодерацию, если статус модерации "New"
                $groups_to_moderate->{$phrase_values->{pid}} = 1 if $phrase_values->{statusModerate} eq 'New';

                # performance: получаем новые id пачкой, а не по одному, для этого пока запоминаем 0 вместо id
                push @phrases_to_insert_update, [ 0, $phrase_values->{statusModerate}, @{$phrase_values}{@fields} ];
                if ($phrase_values->{phraseIdHistory}) {
                    $phraseid_history{$phrase_values->{pid}}->{$phrase_values->{norm_phrase}} = hash_cut $phrase_values, qw/cid phraseIdHistory/;
                }

                $inserted_phrases{$phrase_values->{norm_phrase}} = $phrase_values;

                push @mail_notifications, {
                                object     => 'banner',
                                event_type => 'b_word', 
                                object_id  => $phrase->{bid}, 
                                old_text   => '', 
                                new_text   => $phrase_values->{phrase}, 
                                uid        => $phrase->{uid},
                            };
            } else {
                $norm_to_id->{$phrase->{norm_phrase}} = $phrase->{id};
                my $statusModerate = $old_phrases->{$phrase->{id}}{statusModerate} || 'New';
                push @phrases_to_insert_update, [ $phrase->{id}, $statusModerate, @{$phrase_values}{@fields} ];
                # Пометим группу (условие показа) на перемодерацию, если статус модерации "New"
                $groups_to_moderate->{$phrase->{pid}} = 1 if $statusModerate eq 'New';
                push @log_prices, {
                    cid => $phrase->{cid},
                    bid => $phrase->{bid},
                    pid => $phrase->{pid},
                    id => $phrase->{id},
                    type => 'update3',
                    price => $phrase_values->{price},
                    price_ctx => PhrasePrice::phrase_camp_price_context($campaign, $phrase_values),
                    currency => $campaign->{currency},
                };

                if ($phrase->{id} && $OPT{i_know_href_params}) {
                    my $item = { id => $phrase->{id}, cid => $phrase->{cid} };
                    hash_merge $item, hash_cut $phrase->{HrefParams}, @BIDS_HREF_PARAMS;
                    push @up_href_params, $item;
                }
            }
        } # end of foreach my $phrase (@$phrases_chunk)


        if (@phrases_to_delete > 0) {
            my $phrases_ids_to_delete = [ map { $_->{id} } @phrases_to_delete ];
            do_delete_from_table(PPC(shard => $shard), 'bids', where => { id => $phrases_ids_to_delete });
        }

        # реогранизуем @phrases_to_delete в хэш cid->id->undef
        my %phrases_del_hash;
        for my $p (@phrases_to_delete) {
            $phrases_del_hash{$p->{cid}}->{$p->{id}} = undef;
        }
        # чистим bids_phraseid_history
        while(my ($cid, $ids_hash) = each %phrases_del_hash) {
            do_delete_from_table(PPC(shard => $shard), 'bids_href_params', where => { cid => $cid, id => [keys %$ids_hash] });
            do_delete_from_table(PPC(shard => $shard), 'bids_phraseid_history', where => {cid => $cid, id => [keys %$ids_hash]});
        }

        my $update_insert_sql = 'INSERT INTO bids (
                                            id,
                                            statusModerate, 
                                            '.join(', ', @fields).'
                                        ) 
                                 VALUES %s
                                 ON DUPLICATE KEY UPDATE 
                                            statusModerate = values(statusModerate), 
                                            '.join(', ', map {"$_ = VALUES($_)"} grep {$_ ne 'pid'} @fields);

        my $new_ids_count = scalar ( grep { !$_->[0] } @phrases_to_insert_update );
        my $new_ids = get_new_id_multi('phid', $new_ids_count);

        foreach my $row (@phrases_to_insert_update) {
            $row->[0] = shift @$new_ids  if !$row->[0];
        }
        
        do_mass_insert_sql(PPC(shard => $shard), $update_insert_sql, \@phrases_to_insert_update, {max_row_for_insert => 100});

        # извлекаем только что добавленные фразы в базу, Устанавливаем их соответствие по norm_phrase
        my $ids = get_all_sql(PPC(shard => $shard), [
            'SELECT id, norm_phrase, pid, cid FROM bids',
            where => {
                pid => [map {$_->{pid}} values %inserted_phrases],
                norm_phrase => [map {$_->{norm_phrase}} values %inserted_phrases]
            }
        ]);

        # собираем данные для лога для добавленных фраз
        my @phraseid_history_update;
        foreach my $ph_id (@$ids) {
            $norm_to_id->{$ph_id->{norm_phrase}} = $ph_id->{id};
            if (my $phraseid = $phraseid_history{$ph_id->{pid}}->{$ph_id->{norm_phrase}}) {
                push @phraseid_history_update, [$ph_id->{cid}, $ph_id->{id}, $phraseid->{phraseIdHistory}];
            }
            
            push @log_prices, {
                cid => $ph_id->{cid},
                pid => $ph_id->{pid},
                id => $ph_id->{id},
                type => ($OPT{from_stat} ? 'insert4' : 'insert1'),
                price => $inserted_phrases{$ph_id->{norm_phrase}}->{price},
                price_ctx => PhrasePrice::phrase_camp_price_context($campaign, $inserted_phrases{$ph_id->{norm_phrase}}),
                currency => $campaign->{currency},
            };
        }
        do_mass_insert_sql(PPC(shard => $shard),
                        "INSERT INTO bids_phraseid_history (cid, id, phraseIdHistory)
                              VALUES %s
             ON DUPLICATE KEY UPDATE phraseIdHistory = VALUES(phraseIdHistory), update_time = NOW()"
            , \@phraseid_history_update
        );

    }; # end of foreach_shard

    # выполняем блок вне зависимости от $OPT{i_know_href_params}
    if (@up_href_params) {
        foreach (grep {$_->{to_copy}} @up_href_params) {
            $_->{to_id} = $norm_to_id->{ $_->{to_phrase_norm} };
        }
        mass_update_phrases_user_params( { map { ($_->{to_id} || $_->{id}) => $_ } @up_href_params } );
    }

    if ($OPT{i_know_href_params}) {
        my $id_to_href_params = {
                                  map { $norm_to_id->{$_} => $href_params_by_norm->{$_} }
                                 grep { defined $href_params_by_norm->{$_} }
                                 keys %$norm_to_id
                                };
        mass_update_phrases_user_params( $id_to_href_params );
    }

    mass_mail_notification(\@mail_notifications);
    LogTools::log_price(\@log_prices);
    clear_auto_price_queue([uniq map {$_->{cid}} @log_prices]) if @log_prices;

    return {
        norm_phrases => $norm_to_id,
        groups_to_moderate => [keys %$groups_to_moderate],
    };
}

=head2 mass_update_phrases_user_params($data)

    Обновляет/устанавливает пользовательские параметры на фразы
    Каждый элемент $data обязан содержать cid

=cut
sub mass_update_phrases_user_params($) {
    my $data = shift;

    die 'no cid in data at mass_update_phrases_user_params ' . Dumper($data) if any {!$_->{cid}} values %$data;

    my @sharded_data;
    while (my ($id, $row) = each %$data)  {
        # в качестве key_id может быть как id, так и to_id
        push @sharded_data, hash_merge({ key_id => $id }, $row);
    }
    foreach_shard cid => \@sharded_data, sub {
        my ($shard, $data_chunk) = @_;

        my (@push_data, %delete_data, %copy_data);

        foreach my $row (@$data_chunk) {
            if ((defined $row->{Param1} || defined $row->{Param2})
                && !$row->{to_delete}
                && !$row->{to_copy}
            ) {
                push @push_data, [$row->{cid}, $row->{key_id}, $row->{Param1}, $row->{Param2}];
            }

            if (! (defined $row->{Param1} || defined $row->{Param2}) || $row->{to_delete}) {
                push @{$delete_data{$row->{cid}}}, $row->{key_id};
            }

            if ($row->{to_copy}) {
                # этих ключей не было, они все появились из-за шардинга
                delete $row->{key_id};
                push @{$copy_data{$row->{cid}}}, $row;
            }
        }

        # если требуется скопировать параметры от другой фразы
        for my $cid (keys %copy_data) {
            my %from_to = map { $_->{from_id} => $_->{to_id} } @{$copy_data{$cid}};
            my $cdata = get_all_sql(PPC(shard => $shard), [
                "SELECT cid, id, param1, param2 FROM bids_href_params",
                where => { cid => $cid, id => [keys %from_to] }
            ]);

            foreach my $d (@$cdata) {
                push @push_data, [ $d->{cid}, $from_to{ $d->{id} }, $d->{param1}, $d->{param2} ];
            }
        }

        # обвноляем параметры
        if (@push_data) {
            # обновляем данные данные
            my $sql = "INSERT INTO bids_href_params (cid, id, param1, param2)
                            VALUES %s
           ON DUPLICATE KEY UPDATE param1=VALUES(param1), param2=VALUES(param2)";

            do_mass_insert_sql(PPC(shard => $shard), $sql, \@push_data, {max_row_for_insert => 100});
        }

        # удаляем параметры для удаленных фраз
        while(my ($cid, $ids) = each %delete_data) {
            # удаляем опустевшие данные
            do_delete_from_table(PPC(shard => $shard), 'bids_href_params', where => {cid => $cid, id => $ids});
        }
    }; #end of foreach_shard

    return 1;
}

=head2 _prepare_phrase_for_db

    Подготавливает параметры фразы для вставки в БД

=cut

sub _prepare_phrase_for_db
{
    my %O = @_;

    my $phrase_values = hash_cut \%O, qw/bid cid pid place phrase numword norm_phrase norm_hash md5 is_suspended/;
    ensure_phrase_have_props($phrase_values);

    $phrase_values->{price} = $O{price} || 0;
    $phrase_values->{autobudgetPriority} = validate_priority( $O{autobudgetPriority} );
    $phrase_values->{showsForecast} = (!$O{showsForecast} || $O{showsForecast} !~ /^\d+$/) ? 0 : $O{showsForecast};
    $phrase_values->{warn} = 'Yes';
    $phrase_values->{phraseIdHistory} = $O{phraseIdHistory} if $O{phraseIdHistory};

    return $phrase_values;
}
=head2 validate_phrase_one_href_param

    Проверяет корректность значения параметра ссылки (это которые Param1, Param2 в API).
    Принимает значение параметра.
    Возвращает текст ошибки. Если всё хорошо, возвращяет undef и пустой список в скалярном и списочном контекстах соответственно.

    $error = validate_phrase_one_href_param($phrase->{Param1});
    push @errors, validate_phrase_one_href_param($phrase->{Param1});

=cut

sub validate_phrase_one_href_param {
    my ($param_value) = @_;

    if ($param_value) {
        return iget('Параметр должен быть строкой') if ref $param_value;
        return iget('Параметр содержит недопустимые символы') unless $param_value !~ $Settings::DISALLOW_BANNER_LETTER_RE && length($param_value) > 0;
        return iget('Превышена максимальная длина параметра в %d символов', $Settings::MAX_HREF_PARAM_LENGTH) if length($param_value) > $Settings::MAX_HREF_PARAM_LENGTH;
    }

    return;
}

=head2 validate_phrases

    Валидация фраз.
    
    Параметры:
        $phrases - массив [] фраз [{phrase => ''}, {phrase => []} ....]
        $options {
            keyword_field_name => 'Phrase' - имя поля с текстом ключевого слово, нужно для подстановки в детализацию ошибки валидации
            no_phrase_brackets => 1|0 1 - не проверять ошибки использования скобок () во фразах
            for_xls => 1|0  1 - проверка используется в xls (к сообщению об ошибке будет добавлен номер строки)
            link_errors_to_phrases => 0|1 1 - не добавлять ошибки в общий список, добавлять ошибки в массив errors внутри объекта с конкретной фразой, 
                                              к которой относится ошибка. Также, ошибки привязанные к фразам подставляются без текста фразы в ошибке.
        }
    
    Результаты:
        [] - массив текстовых ошибок (пустой массив при их отсутвии)

=cut

sub validate_phrases
{
    my ($phrases, $options) = @_;
    my @errors;

    my $keyword_field_name = $options->{keyword_field_name} || 'Phrase';
    my $only_phrases = [map {$_->{phrase}} @$phrases];
    my $brackets_errors;
    if ( ! $options->{no_phrase_brackets} ) {
        my $br_result = process_phrase_brackets($only_phrases);
        if ($br_result) {
            push @errors, $br_result;
            $brackets_errors = 1;
        }
    }

    my $not_process_phrase_text_placeholder_in_errors = $options->{link_errors_to_phrases} ? 1 : 0;
    my $phrase_vr = base_validate_keywords($only_phrases, {brackets_errors => $brackets_errors,
                                                           not_process_phrase_text_placeholder_in_errors => $not_process_phrase_text_placeholder_in_errors, });
    $phrase_vr->process_objects_descriptions( keyword => { field => $keyword_field_name,
                                                           $not_process_phrase_text_placeholder_in_errors 
                                                                ? (phrase => '', phrase_extended => '')
                                                                : () } );

    unless ($phrase_vr->is_valid) {
        foreach my $object (@{$phrase_vr->get_objects_results}) {
            next if $object->is_valid;

            my $phrase = $phrases->[$object->position];
            if ($options->{link_errors_to_phrases}) {
                push @{$phrase->{errors}}, $object->get_first_error_description;
            } else {
                push @errors, $options->{for_xls}
                    ? iget("Строка %s: ", $phrase->{line_number}) . $object->get_first_error_description
                    : $object->get_first_error_description;
            }
        }
    }
    
    # проверяем параметры ссылки (a.k.a. Param1 и Param2 в API)
    for my $phrase (@$phrases) {
        for my $param_name (@BIDS_HREF_PARAMS) {
            my $param_val = exists $phrase->{$param_name} ? $phrase->{$param_name} : $phrase->{lc($param_name)};
            my $err = validate_phrase_one_href_param( $param_val );
            if ($err) {
                $err = $param_name . ": " . $err;
                if ($options->{link_errors_to_phrases}) {
                    push @{$phrase->{errors}}, $err;
                } else {
                    push @errors, $options->{for_xls} ? iget("Строка %s: ", $phrase->{line_number}) . $err : $err;    
                }
            }
        }
    }

    return \@errors;
}

=head2 get_phrases($groups, %options)

    Массив фраз для групп объявлений

    %options:
        not_use_own_stat - не использовать параметры от баннера
        filters -- дополнительные условия отбора фраз в DBTools-совместимом формате

=cut

sub get_phrases {
    
    my ($groups, %options) = @_;
    my $BIDS_QUERY_LIMIT = 200;    
    
    # фразы заархивированных кампаний со временем перемещаются в bids_arc (но не сразу)
    # так что фразы могут быть и в той, и в другой таблице
    my $sql = q[
        SELECT
            b.phrase, b.price, b.price_context, b.id, b.PhraseID, b.pid
            , b.norm_phrase
            , b.optimizeTry, b.autobudgetPriority, bph.phraseIdHistory, b.showsForecast
            , b.place, b.statusModerate, b.is_suspended
            , bhp.param1, bhp.param2
        FROM %s b
            left join bids_phraseid_history bph using(cid, id)
            left join bids_href_params bhp using(cid, id)
    ];

    my %groups;
    my (@pids, @archive_pids, @archive_cids);
    foreach (@$groups) {
        push @pids, $_->{pid};
        if ($_->{camp_is_arch}) {
            push @archive_cids, $_->{cid};
            push @archive_pids, $_->{pid};
        }
        
        $groups{$_->{pid}} = $_;
    }

    my %queries = (
        bids => {pid => [sort {$a <=> $b} @pids]},
        @archive_pids ? (bids_arc => {
            pid => [sort {$a <=> $b} @archive_pids],
            'b.cid' => \@archive_cids
        }) : ()
    );

    my @phrases;
    $options{filters} ||= {};
    foreach my $t (keys %queries) {
        my $filter = hash_cut $queries{$t}, 'b.cid';
        my $pids = $queries{$t}->{pid};
        while (my @ids = splice @$pids, 0, $BIDS_QUERY_LIMIT) {
            my $p = get_all_sql(PPC(pid => \@ids), [sprintf($sql, $t), WHERE => {
                (%$filter, pid => SHARD_IDS),
                %{$options{filters}}
            }]);
            push @phrases, @$p
        }
    }
    
    foreach my $ph (@phrases) {
        
        # TODO: удалить (уже хранится уровнем выше, в группе) DIRECT-18531
        $ph->{phr} = $ph->{phrase};
        $ph->{PriorityID} = $groups{$ph->{pid}}->{PriorityID};
        # TODO adgroup: в задаче DIRECT-20684 должно быть исправлено (т.к. фраза напрямую не соотносится с баннером)
        # и сейчас значение $groups{$ph->{pid}}->{BannerID} нет
        $ph->{BannerID} = $groups{$ph->{pid}}->{BannerID};
        
        
        $ph->{md5} = md5_hex_utf8($ph->{norm_phrase} // '');
        $ph->{declined} = ($groups{$ph->{pid}}->{statusModerate}||'')  =~ /^(Yes|Ready|Sent|Sending)$/
                            && $ph->{statusModerate} eq 'No';
        
        if ($options{not_use_own_stat}) {
            $ph->{PriorityID} = $ph->{BannerID} = $ph->{PhraseID} = 0;
            $ph->{phraseIdHistory} = '';
        }
    }

    return \@phrases;
}

1;
