package Moderate::ResyncQueue;

=encoding utf8

=head1 NAME

    Moderate::ResyncQueue

=cut

use Direct::Modern;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Validate qw/is_valid_id is_valid_int/;
use Yandex::HashUtils;
use Direct::Banners;
use Moderate::ReModeration;
use Campaign::Types qw/get_camp_kind_types/;

use Settings;

use List::MoreUtils qw/any none/;

=head2
=item CRITICAL_PRIORITY
=cut

use constant CRITICAL_PRIORITY => 100;

# время кеширования параметров ленивой отправки, секунд
my $RESYNC_PROPERTY_CACHE_TIME = 60;

my $DEFAULT_CRIT_LIMIT        = 250;
my $CRIT_LIMIT_PROP_NAME      = 'mod_resync_crit_limit';
my $DEFAULT_NOCRIT_LIMIT      = 500;
my $NOCRIT_LIMIT_PROP_NAME    = 'mod_resync_nocrit_limit';
my $DEFAULT_MAX_TOTAL_SIZE    = 10000;
my $MAX_TOTAL_SIZE_PROP_NAME  = 'mod_resync_max_total_size';
my $DEFAULT_MAX_AGE_MINUTES   = 15;
my $MAX_AGE_MINUTES_PROP_NAME = 'mod_resync_max_age_minutes';

my %OBJECT_TABLE = (
    banner  => {
        table => 'banners',
        field => 'statusModerate',
        key   => 'bid',
    },
    phrases  => {
        table => 'phrases',
        field => 'statusModerate',
        key   => 'pid',
    },
    contactinfo => {
        table => 'banners',
        field => 'phoneFlag',
        key   => 'bid',
        condition  => { 'vcard_id__gt' => 0 }, # Сбрасываем статус модерации визитки, только если она есть
    },
    sitelinks_set => {
        table => 'banners',
        field => 'statusSitelinksModerate',
        key   => 'bid',
        condition  => { 'sitelinks_set_id__gt' => 0 }, # Сбрасываем статус модерации сайтлинков, только если они есть
    },
    image => {
        table => 'banner_images',
        field => 'statusModerate',
        key   => 'bid',
    },
    display_href => {
        table => 'banner_display_hrefs',
        field => 'statusModerate',
        key   => 'bid',
    },
    banner_logo => {
        table => 'banner_logos',
        field => 'statusModerate',
        key   => 'bid',
    },
    banner_button => {
        table => 'banner_buttons',
        field => 'statusModerate',
        key   => 'bid',
    },
    banner_multicard => {
        table => 'banner_multicard_sets',
        field => 'statusModerate',
        key   => 'bid',
    },
    mobile_content => {
        table => 'mobile_content',
        field => 'statusIconModerate',
        key => 'mobile_content_id',
        condition => {icon_hash__is_not_null => 1}
    },
    callout => {
        table => 'additions_item_callouts',
        field => 'statusModerate',
        key => 'additions_item_id',
        condition => {is_deleted => 0},
    },
    image_ad => {
        table => 'banners',
        field => 'statusModerate',
        key   => 'bid',
    },
    canvas => {
        table => 'banners_performance',
        field => 'statusModerate',
        key   => 'banner_creative_id',
    },
    html5_creative => {
        table => 'banners_performance',
        field => 'statusModerate',
        key   => 'banner_creative_id',
    },
    video_addition => {
        table => 'banners_performance',
        field => 'statusModerate',
        key   => 'banner_creative_id',
    },
    turbolanding => {
        table => 'banner_turbolandings',
        field => 'statusModerate',
        key   => 'bid',
    },
    cpm_video => {
        table => 'banners',
        field => 'statusModerate',
        key   => 'bid',
    },
    cpm_yndx_frontpage => {
        table => 'banners',
        field => 'statusModerate',
        key   => 'bid',
    },
    fixcpm_yndx_frontpage => {
        table => 'banners',
        field => 'statusModerate',
        key   => 'bid',
    },
    perf_creative => {
        table => 'perf_creatives',
        field => 'statusModerate',
        key   => 'creative_id',
    },
);

# key - ключ шардинга, по умолчанию равен ключу, field - в каком поле лежит значение
my %OBJECT_SHARD_KEYS = (
    bid => { field => 'id' },
    pid => { field => 'id' },
    mobile_content_id => { key => 'ClientID', field => 'client_id' },
    additions_item_id => { key => 'ClientID', field => 'client_id' },
    image_id => { key => 'cid', field => 'cid' },
    banner_creative_id => { key => 'cid', field => 'cid' },
);

=head2 mod_resync($objects, %params)

    Добавить объекты из $objects в очередь ленивой переотправки на модерацию.
    Если переданы некорректные или недостаточные данные - умирает.

    mod_resync($objects);
    mod_resync($objects2, log => Yandex::Log->new(...));

    Пример одного объекта - ссылка на хеш со следующими полями:
        id         => bid | pid | mobile_content_id | image_id | banner_creative_id,
        type       => banner | phrases | contactinfo | sitelinks_set | image | banner_logo | banner_button | mobile_content | image_ad | canvas | video_addition | html5_creative
        priority   => -128..127,
        remoderate => 0 | 1
        client_id  => ClientID, # нужен для записей mobile_content
        cid        => cid, # нужен для image_ad

    Параметры позиционные:
        $objects    - ссылка на массив объектов для переотправки. формат объекта описан выше
    Параметры именованные:
        log         - опционально, объект Yandex::Log для логирование объектов,
                      для которых не определился шард

=cut
sub mod_resync {
    my ($objects, %params) = @_;

    my %data_by_key;

    foreach my $object ( @{$objects || []} ) {
        $object->{remoderate} //= 0;
        $object->{priority} //= 0;
        die "Invalid object type '$object->{type}'" unless is_valid_type($object->{type});
        die "Invalid object id '$object->{id}'" unless is_valid_id($object->{id});
        die "Invalid object priority '$object->{priority}'" unless is_valid_priority($object->{priority});
        die "Invalid object remoderate '$object->{remoderate}'" unless is_valid_remoderate($object->{remoderate});

        my $obj_key = $OBJECT_TABLE{$object->{type}}->{key};
        unless (is_valid_extra_data($object)) {
            die "Need object field '$OBJECT_SHARD_KEYS{$obj_key}->{field}' for object type '$object->{type}'";
        }

        push @{ $data_by_key{$obj_key} }, $object;
    }

    foreach my $obj_key (keys %data_by_key) {
        my $shard_key = $OBJECT_SHARD_KEYS{$obj_key}->{key} || $obj_key;
        my $key_field = $OBJECT_SHARD_KEYS{$obj_key}->{field};
        foreach_shard $shard_key => $data_by_key{$obj_key}, by => $key_field, with_undef_shard => 1, sub {
            my ($shard, $data) = (shift, shift);

            if ($shard) {
                do_mass_insert_sql(
                    PPC(shard => $shard),
                    'INSERT INTO mod_resync_queue(object_id, object_type, priority, remoderate)
                     VALUES %s
                     ON DUPLICATE KEY UPDATE priority   = greatest(priority,   VALUES(priority)),
                                             remoderate = greatest(remoderate, VALUES(remoderate))
                    ',
                    [ map { [@$_{ qw/id type priority remoderate/ }] } @$data ],
                );
            } elsif ($params{log} && ref $params{log} && UNIVERSAL::can($params{log}, 'out')) {
                $params{log}->out({data_without_shard => $data, shard => $shard});
            }
        };
    }
}

=head2 get_all_objects_by($key => $ids, %params)

    Получить все дочерние объекты кампаний или баннеров из $ids для добавление
    в очередь ленивой переотправки на модерацию.

    get_all_objects_by($key => $ids, priority => 123, remoderate => 0);

    Где:
        $key - cid или bid
        $ids - ссылка на массив с cid'ами или bid'ами,
        priority - приоритет, значение -128..127,
        remoderate - 0 | 1

=cut
sub get_all_objects_by {
    my ($key, $ids, %params) = @_;

    if (none { $key eq $_ } qw/bid cid/) {
        die "Bad key $key. Key must be bid or cid!";
    }

    my $priority = $params{priority};
    my $remoderate = $params{remoderate};

    my @objects;
    # object_type: banner, sitelinks_set, image, contactinfo, display_href, turbolanding, banner_logo, banner_button, banner_multicard
    my $data = get_all_sql(PPC($key => $ids), [
        '
            SELECT
                b.bid,
                b.vcard_id,
                b.sitelinks_set_id,
                bim.image_id,
                IF (bdh.bid IS NULL, 0, 1) as has_display_href,
                IF (bl.bid IS NULL, 0, 1) as has_logo,
                IF (bb.bid IS NULL, 0, 1) as has_button,
                IF (bms.bid IS NULL, 0, 1) as has_multicard_set,
                IF (bt.bid IS NULL, 0, 1)  as has_turbolanding
            FROM
                banners AS b
            LEFT JOIN banner_images AS bim ON bim.bid = b.bid AND bim.statusShow = "Yes"
            LEFT JOIN banner_display_hrefs AS bdh ON bdh.bid = b.bid
            LEFT JOIN banner_logos AS bl ON bl.bid = b.bid
            LEFT JOIN banner_buttons AS bb ON bb.bid = b.bid
            LEFT JOIN banner_multicard_sets AS bms ON bms.bid = b.bid
            LEFT JOIN banner_turbolandings AS bt  ON bt.bid = b.bid 
        ',
        # performance и internal баннеры не отправляются в старую Модерацию
        WHERE => {"b.$key" => SHARD_IDS, 'b.banner_type__not_in' => ['performance', 'internal']}
    ]);

    for my $row (@$data) {
        push @objects, {
            id => $row->{bid},
            type => 'banner',
            priority => $priority,
            remoderate => $remoderate,
        };

        if ($row->{sitelinks_set_id}) {
            push @objects, {
                id => $row->{bid},
                type => 'sitelinks_set',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{image_id}) {
            push @objects, {
                id => $row->{bid},
                type => 'image',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{vcard_id}) {
            push @objects, {
                id => $row->{bid},
                type => 'contactinfo',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{has_display_href}) {
            push @objects, {
                id => $row->{bid},
                type => 'display_href',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{has_logo}) {
            push @objects, {
                id => $row->{bid},
                type => 'banner_logo',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{has_button}) {
            push @objects, {
                id => $row->{bid},
                type => 'banner_button',
                priority => $priority,
                remoderate => $remoderate,
            };
        }

        if ($row->{has_multicard_set}) {
            push @objects, {
                id => $row->{bid},
                type => 'banner_multicard',
                priority => $priority,
                remoderate => $remoderate,
            }
        }

        if ($row->{has_turbolanding}) {
            push @objects, {
                id => $row->{bid},
                type => 'turbolanding',
                priority => $priority,
                remoderate => $remoderate,
            };
        }
    }

    # object_type: phrases
    if ($key eq 'cid') {
        # для кампаний
        $data = get_all_sql(PPC(cid => $ids), [
            'SELECT pid FROM phrases p JOIN campaigns c ON p.cid = c.cid',
            WHERE => {
                'c.cid' => SHARD_IDS,
                'c.type__not_in' => get_camp_kind_types('mod_export_campaigns_only')  # performance и internal-группы не переотправляются в модерацию
            }
        ]);
    } elsif ($key eq 'bid') {
        # для баннеров
        $data = get_all_sql(PPC(bid => $ids), [
            'SELECT p.pid FROM phrases p JOIN banners b on b.pid = p.pid JOIN campaigns c ON c.cid = p.cid',
            WHERE => {
                'b.bid' => SHARD_IDS,
                'c.type__not_in' => get_camp_kind_types('mod_export_campaigns_only')  # performance и internal-группы не переотправляются в модерацию
            }
        ]);
    }
    for my $row (@$data) {
        push @objects, {
            id => $row->{pid},
            type => 'phrases',
            priority => $priority,
            remoderate => $remoderate,
        };
    }

    # object_type: mobile_content
    my $from_table;
    my $key_field;
    if ($key eq 'cid') {
        # для кампаний
        $from_table = 'FROM phrases p';
        $key_field = 'p.cid';
    } elsif ($key eq 'bid') {
        # для баннеров
        $from_table = 'FROM banners b JOIN phrases p ON p.pid = b.pid';
        $key_field = 'b.bid';
    }
    $data = get_all_sql(PPC($key => $ids), [
        'SELECT DISTINCT mobc.mobile_content_id, mobc.ClientID',
        $from_table,
        'JOIN adgroups_mobile_content ag ON ag.pid = p.pid
         JOIN mobile_content mobc ON mobc.mobile_content_id = ag.mobile_content_id',
        WHERE => {
            $key_field => SHARD_IDS,
            'mobc.icon_hash__is_not_null' => 1,
        },
    ]);
    for my $row (@$data) {
        push @objects, {
            id => $row->{mobile_content_id},
            type => 'mobile_content',
            priority => $priority,
            remoderate => $remoderate,
            client_id => $row->{ClientID},
        };
    }

    # object_type: callout
    if ($key eq 'cid') {
        # для кампаний
        $from_table = 'FROM banners b join campaigns c ON c.cid = b.cid';
        $key_field = 'b.cid';
    } elsif ($key eq 'bid') {
        # для баннеров
        $from_table = 'FROM banners b';
        $key_field = 'b.bid';
    }
    $data = get_all_sql(PPC($key => $ids), [
        'SELECT DISTINCT aic.additions_item_id, aic.ClientID',
        $from_table,
        'JOIN banners_additions ba ON ba.bid = b.bid
         JOIN additions_item_callouts aic ON aic.additions_item_id = ba.additions_item_id',
        WHERE => {
            $key_field => SHARD_IDS,
            'ba.additions_type' => 'callout',
            'aic.is_deleted' => 0,
        },
    ]);
    for my $row (@$data) {
        push @objects, {
            id => $row->{additions_item_id},
            type => 'callout',
            priority => $priority,
            remoderate => $remoderate,
            client_id => $row->{ClientID},
        };
    }

    # object_type: image_ad
    $data = get_all_sql(PPC($key => $ids), [
        'SELECT cid, image_id FROM images',
        WHERE => {$key => SHARD_IDS}
    ]);
    for my $row (@$data) {
        push @objects, {
            cid => $row->{cid},
            id => $row->{image_id},
            type => 'image_ad',
            priority => $priority,
            remoderate => $remoderate,
        };
    }

    # object_type: canvas, video_addition
    $data = get_all_sql(PPC($key => $ids), [
        'SELECT b_perf.cid, b_perf.banner_creative_id, perf_cr.creative_type
         FROM banners_performance b_perf
         LEFT JOIN perf_creatives perf_cr USING(creative_id)',
        WHERE => {
            "b_perf.$key" => SHARD_IDS,
            'perf_cr.creative_type' => [qw/canvas video_addition html5_creative/]
        }
    ]);
    for my $row (@$data) {
        push @objects, {
            cid => $row->{cid},
            id => $row->{banner_creative_id},
            type => $row->{creative_type},
            priority => $priority,
            remoderate => $remoderate,
        };
    }

    return \@objects;
}

=head2 process_resync_queue

    Выбирает из ленивой очереди объекты для переотправки 
    и меняет соответствующий статус на Ready.
    Сначала пытается выбрать объекты с приоритетом > 100;

    Параметры:

    $stat_cb - coderef для получения статистики очереди переотправки
    $shard   - шард, опционально

=cut

sub process_resync_queue {
    my ($stat_cb, $shard) = @_;

    my $crit_limit_value = get_crit_limit($RESYNC_PROPERTY_CACHE_TIME);
    my $nocrit_limit_value = get_nocrit_limit($RESYNC_PROPERTY_CACHE_TIME);
    my $max_total_size_value = get_max_total_size($RESYNC_PROPERTY_CACHE_TIME);
    my $max_age_minutes_value = get_max_age_minutes($RESYNC_PROPERTY_CACHE_TIME);

    my $objects = get_all_sql( PPC(shard => $shard),
        'SELECT object_id, object_type, remoderate FROM mod_resync_queue WHERE priority_inverse < -'
        . CRITICAL_PRIORITY 
        . ' ORDER BY priority_inverse, add_time'
        . ' LIMIT ' . $crit_limit_value
    );

    if (@$objects) {
        return _update_resync_queue($objects, $shard);
    }

    my $stat = $stat_cb->();

    return if $stat->{all}{total_size}      > $max_total_size_value
           || $stat->{all}{max_age_minutes} > $max_age_minutes_value;

    $objects = get_all_sql( PPC(shard => $shard), 
        'SELECT object_id, object_type, remoderate FROM mod_resync_queue WHERE priority_inverse >= -'
        . CRITICAL_PRIORITY
        . ' ORDER BY priority_inverse, add_time'
        . ' LIMIT ' . $nocrit_limit_value
    );

    if ( @$objects ) {
        _update_resync_queue($objects, $shard);
    }
}

=head2 is_valid_priority

    Значение priority должно быть в диапазоне -128..127,
    поскольку хранится в mysql как знаковый tinyint

=cut
sub is_valid_priority {
    my $val = shift;
    return is_valid_int($val, -128, 127);
}

=head2 is_valid_remoderate

    Значение remoderate быть 0 или 1

=cut
sub is_valid_remoderate {
    my $val = shift;
    return is_valid_int($val, 0, 1);
}

=head2 is_valid_type($type)

    Проверить допустимость типа объекта

    Параметры:
        $type - тип объекта для переотправки
    Результат:
        $is_valid_type - 0/1 допустим ли этот тип для переотправки

=cut
sub is_valid_type {
    my $type = shift;
    return $OBJECT_TABLE{$type} ? 1 : 0;
}

=head2 is_valid_extra_data($object)

    Проверить, нужны ли для указанного типа объекта дополнительные данные (обычно client_id)
    и присутствуют ли они

    Параметры:
        $object     - хеш с данными для переотправки
    Результат:
        $is_needed  - 0/1 не нужны или нужны

=cut
sub is_valid_extra_data {
    my $object = shift;

    my $obj_key = $OBJECT_TABLE{$object->{type}}->{key};

    return (!$OBJECT_SHARD_KEYS{$obj_key}->{key}
            || $OBJECT_SHARD_KEYS{$obj_key}->{key} eq $obj_key
            || $object->{ $OBJECT_SHARD_KEYS{$obj_key}->{field} }
           ) ? 1 : 0;
}

=head3 _update_resync_queue

    Внутренняя функция, для обновления очереди mod_resync_queue. 
    - Скидывает statusModerate на ready, 
    - Если требуется ручная перемодерация, добавляет bid в pre_moderate_banners
    - Удалет записи из mod_resync_queue

    Параметры:

    $objects - ссылка на массив хешей обрабатываемых объектов: [{'object_type' => 'banner','remoderate' => 0,'object_id' => 36132}, ... ];
    $shard   - шард, опционально

=cut

sub _update_resync_queue {
    my ($objects, $shard) = (shift, shift);

    my (@ids, %objects);

    foreach my $object ( @$objects ) {
        push @{$objects{$object->{object_type}} ||= []}, $object;
    }

    foreach my $type ( keys %objects ) {
        my ($table, $field, $key, $condition) = @{$OBJECT_TABLE{$type}}{ qw/table field key condition/ };

        my (@all_object_ids, @object_ids, @bids_pre_moderate);

        @all_object_ids = map { $_->{object_id} } @{$objects{$type}};
        if ($type eq 'phrases') {
            # оставляем только баннеры или группы, которые могут уйти в транспорте в Модерацию
            @object_ids = @{ get_one_column_sql(PPC(shard => $shard),
                ["select t.${key} FROM $table t JOIN campaigns c on c.cid = t.cid",
                    where => {
                        "t.${key}" => \@all_object_ids,
                        'c.type' => get_camp_kind_types('mod_export'),
                        'c.type__not_in' => get_camp_kind_types('mod_export_campaigns_only')
                    }
                ])};
        } elsif ($type eq 'banner') {
            # оставляем только баннеры или группы, которые могут уйти в транспорте в Модерацию
            # фильтруем продвижение Услуг и продвижение на Еде
            @object_ids = @{ get_one_column_sql(PPC(shard => $shard),
                ["SELECT t.${key}
                FROM $table t
                JOIN campaigns c on c.cid = t.cid
                LEFT JOIN adgroups_content_promotion acp on acp.pid = t.pid",
                    WHERE => {
                        "t.${key}" => \@all_object_ids,
                        'c.type' => get_camp_kind_types('mod_export'),
                        'c.type__not_in' => get_camp_kind_types('mod_export_campaigns_only'),
                        _OR => {
                            'acp.content_promotion_type__is_null' => 1,
                            'acp.content_promotion_type__not_in' => [qw/service eda/],
                        }
                    }
                ])};
        } else {
            @object_ids = @all_object_ids;
        }

        my %bids_to_export = map { $_ => 1} @object_ids;
        foreach ( @{$objects{$type}} ) {
            push @bids_pre_moderate, $_->{object_id} if ($key eq 'bid' && $_->{remoderate} == 1 && $bids_to_export{$_->{object_id}});
        }

        if (@bids_pre_moderate) {
            # баннеры требующие ручной перемодерации
            Moderate::ReModeration->set_pre_moderation_flag(\@bids_pre_moderate);
        }

        # Объекты в статусе Ready переведем в начале в Sent, чтоб ESS-транспорт смог увидеть последующий переход в Ready.
        my %where_for_ready = (
            $key => \@object_ids,
            "${field}" => 'Ready',
        );
        hash_merge \%where_for_ready, $condition;
        do_update_table(PPC(shard => $shard), $table, {$field => 'Sent'}, where => \%where_for_ready);

        my %where = (
            $key => \@object_ids,
            "${field}__ne" => 'New',
        );
        hash_merge \%where, $condition;
        do_update_table(PPC(shard => $shard), $table, {$field => 'Ready'}, where => \%where);
        if ($table eq 'banners') {
            Direct::Banners::delete_minus_geo($key => \@object_ids);
        }

        do_sql( PPC(shard => $shard), [
            qq[ DELETE FROM mod_resync_queue ],
                WHERE => {
                    object_type => $type,
                    object_id   => \@all_object_ids,
                },
            ]
        );
    }
}


{
     my $crit_limit_prop = Property->new($CRIT_LIMIT_PROP_NAME);

=head2 get_crit_limit

    Возращает заданный параметр crit_limit или значение по умолчанию

=cut
    sub get_crit_limit {
        my $cache_time = shift;
        return eval { $crit_limit_prop->get($cache_time); } || $DEFAULT_CRIT_LIMIT;
    }

=head2 set_crit_limit

    Устанавливает заданный параметр crit_limit

=cut
    sub set_crit_limit {
        $crit_limit_prop->set(shift);
    }
}

{
     my $nocrit_limit_prop = Property->new($DEFAULT_NOCRIT_LIMIT);

=head2 get_nocrit_limit

    Возращает заданный параметр nocrit_limit или значение по умолчанию

=cut
    sub get_nocrit_limit {
        my $cache_time = shift;
        return eval { $nocrit_limit_prop->get($cache_time); } || $DEFAULT_NOCRIT_LIMIT;
    }

=head2 set_nocrit_limit

    Устанавливает заданный параметр nocrit_limit

=cut
    sub set_nocrit_limit {
        $nocrit_limit_prop->set(shift);
    }
}

{
     my $max_total_size_prop = Property->new($MAX_TOTAL_SIZE_PROP_NAME);

=head2 get_max_total_size

    Возращает заданный параметр max_total_size или значение по умолчанию

=cut
    sub get_max_total_size {
        my $cache_time = shift;
        return eval { $max_total_size_prop->get($cache_time); } || $DEFAULT_MAX_TOTAL_SIZE;
    }

=head2 set_max_total_size

    Устанавливает заданный параметр max_total_size

=cut
    sub set_max_total_size {
        $max_total_size_prop->set(shift);
    }
}

{
     my $max_age_minutes_prop = Property->new($MAX_AGE_MINUTES_PROP_NAME);

=head2 get_max_age_minutes

    Возращает заданный параметр max_age_minutes или значение по умолчанию

=cut
    sub get_max_age_minutes {
        my $cache_time = shift;
        return eval { $max_age_minutes_prop->get($cache_time); } || $DEFAULT_MAX_AGE_MINUTES;
    }

=head2 set_max_age_minutes

    Устанавливает заданный параметр max_total_size

=cut
    sub set_max_age_minutes {
        $max_age_minutes_prop->set(shift);
    }
}

1;
