##################################################
#
#  Direct.Yandex.ru
#
#  ModerateChecks
#
#
#  $Id$
#
#
##################################################

=head1 NAME

ModerateChecks - всяческие проверки на необходимость модерации

=cut

package ModerateChecks;

use strict;
use warnings;
use utf8;

our @ISA = qw(Exporter);
our @EXPORT = qw(

            check_moderate_phrase
            check_moderate_phrases
            check_moderate_region
            check_moderate_banner
            check_moderate_contactinfo
            check_moderate_sitelinks
            check_moderate_banner_for_regions

);

use Sitelinks;
use BannerTemplates;
use GeoTools;
use Yandex::MyGoodWords;
use TextTools;
use geo_regions;
use Property;

use Yandex::HashUtils;

# Регионы при экслюзивном переходе на которые требуется перемодерация
our @DANGER_REGIONS = (
    # TODO: возможно тут не хватает Беларусии
    $geo_regions::KAZ
);

=head2 BANNER_FLAGS_REMODERATE

    Список в каком регионе требуется для какого флага особая лицензия, которая требует дополнительно
    проверять объявления в модерации. 

=cut

our %BANNER_FLAGS_REMODERATE = (
    $geo_regions::RUS => [
        'forex'
    ],
);

=head2 MODERATE_EVERY_KEYWORD_CHANGE

    Переотправлять на модерацию любые изменения в тексте фраз

=cut

use constant MODERATE_EVERY_KEYWORD_CHANGE => 'moderate_every_keyword_change';

=head2 $MODERATE_EVERY_KEYWORD_CHANGE_TIMEOUT

    Сколько секунд считаем валидным значение MODERATE_EVERY_KEYWORD_CHANGE

=cut

our $MODERATE_EVERY_KEYWORD_CHANGE_TIMEOUT = 3 * 60;

{
my $moderate_every_keyword_change_property = Property->new(MODERATE_EVERY_KEYWORD_CHANGE);


=head2 check_moderate_phrase( new_phrase, old_phrase )

    Проверяет необходимость модерации ФРАЗЫ после изменения

    Именение фразы требует перемодерации.

    Возвращает:
        0 - если модерации не требуется
        1 - если модерация обязательна

=cut

sub check_moderate_phrase
{
    my $new = shift;
    my $old = shift || return 1;

    # если фразу полностью удалили
    return 0 if $new eq '';

    # если Модерация просила слать все изменения
    return ($new eq $old) ? 0 : 1;
}


=head2 check_moderate_phrases({ phrases => ..., geo => ... })

    Проверяет необходимость модерации УСЛОВИЯ показа баннера(все фразы, гео таргентинг)

    Не отправлять на модерацию, если:
        1. Изменился гео таргентинг (в любую сторону)
        2. Ни одна из фраз не требует перемодерации
        # 3. Тематика выбранных рубрик каталога совпадает с тематикой сайта и фраз.
        4. Удалилась фраза, если условие было допущено предыдущий раз на модерации.

    Отправлять, если:
        1. Добавилось плюс слово и предыдущий результат модерации условия был - 'No' или 'New'(черновик)
        2. Удалилась фраза, и условие отклонено.

    Возвращает:
        result:
            0 - если модерации не требуется
            1 - если модерация обязательна
            2 - если необходимость модерации будет зависеть от результата модерации
        changes:
            deleted_phrases - если были только удалены фразы

=cut

sub check_moderate_phrases
{
    my $new_group = shift;
    my $old_group = shift || return 1;

    # гео-таргентинг не проверяем
    my $result = 0;
    my %changes;
    my (@new_phrases, @old_phrases);
    foreach my $p (qw/Phrases phr phrases/) {
        if (ref $new_group->{$p} eq 'ARRAY') {
            @new_phrases = @{$new_group->{$p}};
        }
        
        if (ref $old_group->{$p} eq 'ARRAY') {
            @old_phrases = @{$old_group->{$p}};
        }
    }
   
    my ($has_changed_phrases, $has_new_phrases) = (0, 0);
    # проверяем все фразы поотдельности на необходимость модерации
    foreach my $new_phrase ( @new_phrases ) {
        $new_phrase->{norm_phrase} = Yandex::MyGoodWords::norm_words($new_phrase->{phrase}) if !$new_phrase->{norm_phrase};
        my $old_phrase = (grep {$_->{id} == ($new_phrase->{id}||0) || $_->{norm_phrase} eq $new_phrase->{norm_phrase}} @old_phrases)[0];
        
        # если фраза новая - отправляем на модерацию
        if (!$old_phrase) {
            $has_new_phrases ||= 1;
            $result = 1, last;
        } else {
            if ($moderate_every_keyword_change_property->get($MODERATE_EVERY_KEYWORD_CHANGE_TIMEOUT)) {
                $has_changed_phrases ||= 1 if without_minuswords($old_phrase->{phrase}) ne without_minuswords($new_phrase->{phrase});
            } else {
                $has_changed_phrases ||= 1 if without_minuswords($old_phrase->{norm_phrase}) ne without_minuswords($new_phrase->{norm_phrase});
            }
            my $lresult = check_moderate_phrase($new_phrase->{phrase} || '', $old_phrase->{phrase} || '');

            # если уже было 1, или стало - то сохраняем 1
            # если было 2 - то сохраняем 2
            # иначе результат текущей проверки (по сути 0, либо 2)
            $result = ($result == 1 || $lresult == 1) ? 1 : ($result ? $result : $lresult);

        }
    }
    
    # если изменилось кол-во фраз
    my $changed_count_phrase = @new_phrases != @old_phrases ? 1 : 0;

    my $phrases_not_accepted;
    if (defined($old_group->{pstatusModerate})) { # для старых вызовов функций, где присутствуют поля pstatusModerate.
        $phrases_not_accepted = !$old_group->{pstatusModerate} || $old_group->{partly_declined_phrases} || $old_group->{pstatusModerate} ne 'Yes' ? 1 : 0;
    } else {
        $phrases_not_accepted = !$old_group->{statusModerate} || $old_group->{partly_declined_phrases} || $old_group->{statusModerate} ne 'Yes' ? 1 : 0;
    }
    if (check_moderate_region($new_group->{geo},$old_group->{geo})){
        $result = 1;
    }
    
    # если есть изменения допускающие показов баннера без модерации,
    # то смотрим на предыдущий результат модерации баннера
    if ($result == 2 || $changed_count_phrase && $result != 1) {
        $result = $phrases_not_accepted ? 1 : 2;
    }
   
    $changes{new_phrases} = $has_new_phrases;
    $changes{changed_phrases} = $has_changed_phrases;
    $changes{delete_only} = !($has_new_phrases ||$has_changed_phrases) && @old_phrases > @new_phrases;

    return $result, \%changes;
}

}

=head2 without_minuswords
    Получает на вход текст фразы.
    Возвращает копию полученного текста, исключив из нее минус-слова
=cut

sub without_minuswords {
    my ($phrase) = @_;

    $phrase =~ s/\s+-\w.+$//;

    return $phrase;
}

=head2 check_moderate_region( new_geo, old_geo, geo_tree )

    Проверяет необходимость модерации БАННЕРА после изменения информации о геотаргетинге

    Отправлять на модерацию, если новый регион - Казахстан, или его под-регион, эксклюзивно.
    Если казахстан плюс что-то еще вне Казахстана, то переотправлять не нужно.

    Беларусь возможно будет сделана позже

    get_tree - гео дерево по которому считаем необходимость модерации, пока передавать не нужно,
    для других регионов может понадобиться, по умолчанию { tree => 'ru' }

=cut

sub check_moderate_region {
    my $new_banner_geo = shift;
    my $old_banner_geo = shift;
    my $geo_tree = (shift || {tree => 'ru'});

    foreach my $danger_geo (@DANGER_REGIONS) {
        if ( # если новое гео эксклюзивно входит в региор требующий перемодерации
             # а старое нет (не входит, или не экслюзивно)
            is_targeting_in_region($new_banner_geo, $danger_geo, $geo_tree)
            && !is_targeting_in_region($old_banner_geo, $danger_geo, $geo_tree)
        ) {
            return 1;
        }
    }

    return; # не нужно модерировать
}

=head2 check_moderate_banner_for_regions(banner_flags_hash, new_geo, old_geo, geo_tree)

    DIRECT-50356: Переотправка на модерацию по флагу forex в случае смены региона на Россию
    Сделано с заделом на увеличение оганичений по флагу и регионам. Добавлять новые правила надо в BANNER_FLAGS_REMODERATE
    По смыслу было сделано, основываясь на модуле CheckAdv, который был почти забыт и устарел.

=cut

sub check_moderate_banner_for_regions {
    my ($banner_flags_hash, $new_banner_geo, $old_banner_geo, $geo_tree) = @_;
    $geo_tree ||= {tree => 'api'};

    #my $banner_flags_hash = BannerFlags::get_banner_flags_as_hash($banner_flags);
    foreach my $constraint_geo (keys %BANNER_FLAGS_REMODERATE) {
        foreach my $flag (@{$BANNER_FLAGS_REMODERATE{$constraint_geo}}) {
            if ($banner_flags_hash->{$flag} &&
                is_targeting_include_region($new_banner_geo, $constraint_geo, $geo_tree) &&
                !is_targeting_include_region($old_banner_geo, $constraint_geo, $geo_tree)) {
                return 1;
            }
        }
    }
    return 0;
}

=head2 check_moderate_banner( banner )

    Проверяет необходимость модерации БАННЕРА после изменений(без контактной информации)

    Не отправлять на модерацию, если:
        1. Удалена ссылка на сайт.
        2. Изменился домен.
        3. Незначительно изменился текст или заголовок баннера - например, добавился пробел между словом и знаком препинания.

=cut

sub check_moderate_banner
{
    my $new = shift;
    my $old = shift;

    my ($old_banner, $new_banner);
    for my $f (qw/title title_extension body/ ) {

        $new->{$f} ||= '';
        $old->{$f} ||= '';

        $new_banner .= space_cleaner($new->{$f})||'';
        $old_banner .= space_cleaner($old->{$f})||'';
    }

    # удаляем пробелы(п.3)
    $new_banner = space_cleaner($new_banner);
    $old_banner = space_cleaner($old_banner);

    my $old_href = $old->{href} || '';
    my $new_href = $new->{href} || '';

    my $is_domain_ok =
        ($old->{real_banner_type} || 'text') eq 'mobile_content'
        || ($new->{domain} || '') eq ($old->{domain} || '')
        || !$new->{domain} && $old->{domain} && $new->{phone};
    
    if ( $new_banner eq $old_banner

         # проверка на удаление ссылки взамен визитки
         && ( $new_href eq $old_href || $new_href eq '' && $old_href ne '' && $new->{phone} )
         && $is_domain_ok
         && (!$old->{statusModerate} || $old->{statusModerate} !~ /Sent|Sending|No/)
    ) {
        return 0;
    }

    return 1;

}

=head2 check_moderate_contactinfo( contactinfo )

    Проверяет необходимость модерации КОНТАКТНОЙ ИНФОРМАЦИИ.
    Не отсылаем, если:
        1. Изменился номер дома/корпуса/офиса.
        2. Изменился IM Client или IM Login.
        3. Изменилось время работы (кроме случая когда контактная информация сейчас отклонена)
        4. Изменился E-Mail (кроме случая когда контактная информация сейчас отклонена)
    Опции
        geo_camp_vcard -- особые правила для проверки КИ у гео-кампаний

=cut

# !!! subs_placing: оттащить в VCards
sub check_moderate_contactinfo
{
    my ($new, $old, %O) = @_;

    my $new_clone = hash_copy {}, $new, keys %{$new || {}};
    # если получаем телефон не в формате +d#ddd#ddddddd#, пытаемся привести к этому виду (грязный хак)
    if( ( ! defined $new->{phone} || $new->{phone} !~ m/\#/ ) && $old->{phone} && $old->{phone} =~ m/\#/){
      if ($new->{country_code} || $new->{city_code} || $new->{phone} || $new->{ext}) {
            $new_clone->{phone} = join "#", map {$_ || ''} ( $new->{country_code}, $new->{city_code}, $new->{phone}, $new->{ext} );
      }
    }

    my @test_fields;
    if ($O{geo_camp_vcard}) {
        @test_fields = qw/extra_message/;
    } else {
        @test_fields = qw/country city phone name contactperson street house build apart extra_message worktime org_details_id/;
        push @test_fields, ('contact_email', 'im_login') if $old->{phoneflag} && $old->{phoneflag} ne 'Yes';
    }

    my ($old_ci, $new_ci) = ('', '');
    for (@test_fields) {
        $new_ci .= $new_clone->{$_}||'';
        $old_ci .= $old->{$_}||'';

        $new_ci = space_cleaner($new_ci);
        $old_ci = space_cleaner($old_ci);
    }
    
    my $result = 0;
    $result = 1 if $new_ci ne $old_ci;
    return $result;
}

=head2 check_moderate_sitelinks($new_banner, $old_banner)

    Проверяет, надо ли посылать на модерацию сет сайтлинков

    Вход:
        $new_banner -- ссылка на хеш, описывающий баннер ПОСЛЕ изменений
        $old_banner -- ссылка на хеш, описывающий баннер ДО изменений
    Результат:
        логическое значение, отображающую необходимость отправки на модерацию сета сайтлинков

=cut

sub check_moderate_sitelinks
{

    my $new = shift;
    my $old = shift;

    return ( defined $new->{sitelinks_set_id} 
             && (!defined $old->{sitelinks_set_id} || $new->{sitelinks_set_id} != $old->{sitelinks_set_id}) )
           || defined $old->{statusSitelinksModerate} && $old->{statusSitelinksModerate} eq 'No'
           || ( defined( $old->{'sitelinks'} ) && defined( $new->{'sitelinks'} ) && Sitelinks::compare_sitelinks( $new->{'sitelinks'}, $old->{'sitelinks'} ) );
}

1;
