package ModerateDiagnosis;

# $Id$

=head1 NAME

    ModerateDiagnosis

=head1 DESCRIPTION

    Причины отклонения фраз, баннеров, кампаний, контактной информации, сайтлинков, картинок, дополнений.

=cut

use warnings;
use strict;

use Hash::Util qw/lock_hash/;
use List::MoreUtils qw/any/;

use Settings;
use HashingTools;
use PrimitivesIds;

use Yandex::DBTools;
use Yandex::I18n;
use Yandex::DBShards;

use YAML::Syck;
$YAML::Syck::ImplicitUnicode = 1;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw/
    get_diags
    bayan_get_diags
    mass_get_diags
    get_diags_hash
    get_diag_by_id
    mass_get_banner_additions_diags
/;

# Хеш со списком id причин, для которых в интерфейсе будем показывать ссылку "Более детальные комментарии описаны здесь"
my %details_link = (
    102 => 1,   # Авторское право
    2   => 1,   # Не соответствует действующему законодательству
    99  => 1,   # Сайт не соответствует требованиям
    1   => 1,   # Рекламная политика Яндекса не соблюдена
    69  => 1,   # Политическая реклама
    43  => 1,   # Требуется лицензия, сертификат или ГП
);


=head2 get_diags

Получить причину отклонения по id и типу

=cut

sub get_diags($$;%)
{
    my ($id, $record_type, %O) = @_;

    my $res = mass_get_diags([$id], $record_type, %O);

    return defined $res ? $res->{$id} : undef;
}

=head2 bayan_get_diags(id, record_type)

    Возвращает результат модерации по группе / баннеру в БаЯне.

    Логика немного отличается от аналога для тексстовых кампаний - тем что при указании группы -- аггрегируется информация по всем медийным баннерам из этой группы.

=cut

sub bayan_get_diags
{
    my ($id, $record_type, %O) = @_;

    my $result;
    my $all_diags = get_diags_hash();

    my $field = ($record_type =~ /group/) ? 'g.mgid' : 'b.mbid';
    my %shard = (($record_type =~ /group/ ? 'pid' : 'mbid') => $id);

    my $rows = get_all_sql(PPC(%shard), [qq{select r.id, r.type, r.reason
                                 from media_groups g
                                      join media_banners b on g.mgid=b.mgid
                                      join mod_reasons r on (
                                            r.id=b.mbid and r.type='media_banner'
                                            or r.id=b.mgid and r.type='media_group'
                                      )
                                }, where => { $field => $id }]);

    if ($rows) {
        foreach my $reason (@{$rows}) {
            my $yaml_reasons = eval { YAML::Syck::Load($reason->{reason}) };

            if ($@) {
                warn $@;
            } elsif ($yaml_reasons && ref $yaml_reasons eq 'ARRAY') {
                $result ||= {};
                foreach my $r (@{$yaml_reasons}) {
                    $result->{banner_diags}{$reason->{type}}{$reason->{id}}{$r->{id}} = get_diag_by_id($r->{id}, $all_diags);
                }
            }
        }
    }

    return $result;
}

=head2 mass_get_diags

Получить причины отклонения по списку id и типу

=cut

sub mass_get_diags($$;%)
{
    my ($ids, $record_type, %O) = @_;

    my $all_diags = get_diags_hash();

    my $result = {};

    die "media banners not support" if $record_type =~ /^media_(group|banner)$/;

    my $pids_by_bids = get_bid2pid(bid => $ids);

    foreach_shard bid => [keys %$pids_by_bids], sub {
        my ($shard, $bids_chunk) = @_;

        my $fields_str = "b.bid, b.pid, r.id, IF(r.type IN ('image_ad','canvas','html5_creative'), 'banner', r.type) as type, r.reason";
        my $rows = get_all_sql(PPC(shard => $shard), [
                                "SELECT $fields_str
                                FROM banners b
                                        left join banner_images bim on bim.bid = b.bid
                                        left join banner_display_hrefs bdh on bdh.bid = b.bid
                                        left join banner_turbolandings bt on bt.bid = b.bid
                                        left join banner_logos bl on bl.bid = b.bid
                                        left join banner_buttons bb on bb.bid = b.bid
                                        left join banner_multicard_sets bms on bms.bid = b.bid
                                        join mod_reasons r on r.id = b.bid and (
                                            r.type = 'banner' and b.statusModerate='No' or
                                            r.type = 'contactinfo' and b.phoneflag='No' or
                                            r.type = 'sitelinks_set' and b.statusSitelinksModerate='No' or
                                            r.type = 'image' and bim.statusModerate = 'No' or
                                            r.type = 'display_href' and bdh.statusModerate = 'No' or
                                            r.type = 'turbolanding' and bt.statusModerate = 'No' or
                                            r.type = 'banner_logo' and bl.statusModerate = 'No' or
                                            r.type = 'banner_button' and bb.statusModerate = 'No' or
                                            r.type = 'banner_multicard' and bms.statusModerate = 'No'
                                        )
                                    ", where => {
                                        "b.bid" => $bids_chunk
                                    },
                                    "union all",
                                "SELECT $fields_str
                                FROM banners b
                                        left join images i on i.bid = b.bid
                                        left join banners_performance bp ON bp.bid = b.bid
                                        left join moderate_banner_pages mbp on mbp.bid = b.bid and mbp.is_removed = 0
                                        join mod_reasons r on (
                                            r.id = b.pid and r.type = 'phrases' or
                                            r.id = i.image_id and r.type = 'image_ad' and i.statusModerate = 'No' or
                                            r.id = bp.banner_creative_id and r.type = 'canvas' and bp.statusModerate = 'No' or
                                            r.id = bp.banner_creative_id and r.type = 'html5_creative' and bp.statusModerate = 'No' or
                                            r.id = bp.banner_creative_id and r.type = 'video_addition' and bp.statusModerate = 'No' or
                                            r.id = mbp.moderate_banner_page_id and r.type = 'banner_page' and mbp.statusModerate = 'No'
                                        )
                                    ", where => {
                                        "b.bid" => $bids_chunk
                                    },
                                    "union all",
                                # возвращаем type=performance_common, если креатив был отмодерирован модерацией Директа
                                # (и, соответственно, номера причин отклонения будут указывать на записи moderate_diags с type=common)
                                # если же креатив был отмодерирован старой модерацией в BannerStorage, то возвращаем type=performance_legacy
                                # (в этом случае номера причин отклонения будут указывать на записи в moderate_diags с type=performance)
                                "SELECT
                                    b.bid,
                                    b.pid,
                                    r.id,
                                    IF(
                                        JSON_EXTRACT(pc.additional_data, '\$.moderated_by_direct') is not null and JSON_EXTRACT(pc.additional_data, '\$.moderated_by_direct') = true,
                                        'performance_common',
                                        'performance_legacy'
                                    ) as type,
                                    r.reason
                                FROM banners b
                                    join banners_performance bp ON bp.bid = b.bid
                                    join perf_creatives pc ON pc.creative_id = bp.creative_id and pc.creative_type = 'performance' and pc.statusModerate = 'No'
                                    join mod_reasons r on r.id = pc.creative_id and r.type = 'perf_creative'
                                ", where => {
                                    "b.bid" => $bids_chunk
                                },
        ]);

        # Подгружаем причины с type=performance из moderate_diags только если они действительно нужны
        my $performance_diags;
        if (any {$_->{type} eq 'performance_legacy'} (@{$rows || []}) ) {
            $performance_diags = get_diags_hash('performance');
        }

        foreach my $reason (@{$rows || []}) {
            $result->{$reason->{bid}} ||= {};

            my $ret = $result->{$reason->{bid}};

            my $yaml_reasons = eval { YAML::Syck::Load($reason->{reason}) };

            if ($@) {
                warn $@;

            } elsif ($yaml_reasons) {

                foreach my $r (@{$yaml_reasons}) {
                    my $result_reason_type_key;
                    my $diag;
                    if ($reason->{type} eq 'performance_common' || $reason->{type} eq 'performance_legacy') {
                        $diag = { %{get_diag_by_id($r->{id}, ($reason->{type} eq 'performance_common' ? $all_diags :  $performance_diags))} };
                        $result_reason_type_key = 'perf_creative';
                    } else {
                        $diag = { %{get_diag_by_id($r->{id}, $all_diags)} };
                        $result_reason_type_key = $reason->{type};
                    }

                    # добавляем кастомный комментарий и скриншоты, если они есть
                    $diag->{comment} = $r->{comment} if $r->{comment};
                    $diag->{screenshots} = $r->{screenshots} if $r->{screenshots};

                    push @{$ret->{banner_diags}{$result_reason_type_key}}, $diag;

                    if ($O{bad_phrases} && $r->{list}) {
                        my @id_list = map {$_->{id}} @{$r->{list}};
                        next unless @id_list;

                        my $phrases = [];
                        if (! $record_type || $record_type eq 'banner') {
                            $phrases = get_all_sql(PPC(shard => $shard), ["SELECT id, phrase FROM bids", where=>{id => \@id_list}]);
                        } elsif ($record_type eq 'media_group') {
                            $phrases = get_all_sql(PPC(shard => $shard), ["SELECT id, phrase FROM mcb_phrases", where=>{id => \@id_list}]);
                        }

                        my %real_phrases_hash = map {$_->{id} => $_} @$phrases;

                        foreach my $ph (@{$r->{list}}) {
                            # несуществующие сейчас в директе фразы - пропускаем
                            next unless $ph->{id} && $real_phrases_hash{$ph->{id}} && $real_phrases_hash{$ph->{id}}->{phrase};
                            $ret->{bad_phrases}{md5_hex_utf8($real_phrases_hash{$ph->{id}}->{phrase})} = $real_phrases_hash{$ph->{id}}->{phrase};
                        }
                    }

                    if ($O{bad_sitelinks} && $reason->{type} eq 'sitelinks_set' && $r->{id}) {
                        $ret->{bad_sitelinks} //= {reasons => []};
                        push @{$ret->{bad_sitelinks}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }

                    if ($O{bad_images} && $reason->{type} eq 'image' && $r->{id}) {
                        $ret->{bad_images}{$reason->{id}} ||= get_one_line_sql(PPC(shard => $shard), "select name, image_hash from banner_images where bid = ?", $reason->{id});
                        push @{$ret->{bad_images}{$reason->{id}}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }

                    if ($O{bad_display_hrefs} && $reason->{type} eq 'display_href' && $r->{id}) {
                        $ret->{bad_display_hrefs} //= {reasons => []};
                        push @{$ret->{bad_display_hrefs}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }

                    if ($O{bad_turbolandings} && $reason->{type} eq 'turbolanding' && $r->{id}) {
                        $ret->{bad_turbolandings} //= {reasons => []};
                        push @{$ret->{bad_turbolandings}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }

                    if ($O{bad_logos} && $reason->{type} eq 'banner_logo' && $r->{id}) {
                        $ret->{bad_logos} //= {reasons => []};
                        push @{$ret->{bad_logos}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }

                    if ($O{bad_buttons} && $reason->{type} eq 'banner_button' && $r->{id}) {
                        $ret->{bad_buttons} //= {reasons => []};
                        push @{$ret->{bad_buttons}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }
                
                    if ($O{bad_multicards} && $reason->{type} eq 'banner_multicard' && $r->{id}) {
                        $ret->{bad_multicards} //= {reasons => [], multicards_by_reason => {}};
                        push @{$ret->{bad_multicards}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                        if ($r->{list}) {
                            my @multicard_ids = map {$_->{id}} @{$r->{list}};
                            if (@multicard_ids) {
                                $ret->{bad_multicards}->{multicards_by_reason}{$r->{id}} = \@multicard_ids;
                            }
                        }
                    }

                    if ($O{bad_banner_pages} && $reason->{type} eq 'banner_page' && $r->{id}) {
                        my $data = get_one_line_sql(PPC(shard => $shard), "select pageId, comment as operator_comment from moderate_banner_pages where moderate_banner_page_id = ?", $reason->{id});
                        $ret->{bad_banner_pages}{$data->{pageId}} ||= get_one_line_sql(PPCDICT, "select oo.name as page_name from placements p join outdoor_operators oo on p.owner_login = oo.login where p.pageId = ?", $data->{pageId});
                        if ($O{login_rights} && $O{login_rights}->{role} =~ /^(manager|super|superreader|support)$/) {
                            $ret->{bad_banner_pages}{$data->{pageId}}->{operator_comment} ||= $data->{operator_comment};
                        }
                        push @{$ret->{bad_banner_pages}{$data->{pageId}}->{reasons}}, get_diag_by_id($r->{id}, $all_diags);
                    }
                }

            } else {
                ##для заглушки операторов, когда нет причин отклонения
                if ($O{bad_banner_pages} && $reason->{type} eq 'banner_page') {
                    my $data = get_one_line_sql(PPC(shard => $shard), "select pageId, comment as operator_comment from moderate_banner_pages where moderate_banner_page_id = ?", $reason->{id});
                    $ret->{bad_banner_pages}{$data->{pageId}} ||= get_one_line_sql(PPCDICT, "select oo.name as page_name from placements p join outdoor_operators oo on p.owner_login = oo.login where p.pageId = ?", $data->{pageId});
                    if ($O{login_rights} && $O{login_rights}->{role} =~ /^(manager|super|superreader|support)$/) {
                        $ret->{bad_banner_pages}{$data->{pageId}}->{operator_comment} ||= $data->{operator_comment};
                    }
                    push @{$ret->{bad_banner_pages}{$data->{pageId}}->{reasons}}, get_diag_by_id(-1, $all_diags);
                }
            }
        }
    };

    return $result;
}

=head2 mass_get_banner_additions_diags

    Массовое получение причин отклонения на модерации для расширений объявления

=cut

sub mass_get_banner_additions_diags {
    my ( $shard, $type, $ids ) = @_;

    die 'no shard given'         if ! defined $shard;
    die 'no addition type given' if ! defined $type;

    # иначе могут появиться нежелательные пересечения id => type
    die 'several addition types at once not supported' if ref( $type );

    my $all_diags = get_diags_hash();

    my $rows = get_all_sql($shard, [
        'SELECT id, reason FROM mod_reasons',
        WHERE => { id => $ids, type => $type, statusModerate => 'No', },
    ]);

    my %result;
    foreach my $item ( @{ $rows || [] } ) {
        next if ! $item->{reason}; # skip not rejected

        $result{ $item->{id} } ||= [];

        my $rst = $result{ $item->{id} };

        my $reasons = eval { YAML::Syck::Load( $item->{reason} ) };
        if ( $@ ) {
            warn $@;
        } elsif ( $reasons ) {
            foreach my $r ( @$reasons ) {
                push @$rst, get_diag_by_id( $r->{id}, $all_diags );
            }
        }
    }

    return \%result;
}



=head2 get_diags_hash

Получить все тексты отклонений (в текущем языке)

=cut

# Получить хэш со всеми причинами отклонений на модерации
{
    my %cache;
sub get_diags_hash {
    my ($type) = @_;
    $type ||= 'common';

    my $lang = Yandex::I18n::current_lang();

    if (!$cache{$type}->{$lang} || time - $cache{$type}->{$lang}->{time} >= 600) {
        my $sth = exec_sql(PPCDICT, "select md.diag_id, ifnull( md.textFull, md.text ) diag_text, md.text as short_text, allow_first_aid, bad_reason, unban_is_prohibited, token
                                from moderate_diags md where md.type = ?", $type);
        my %data;
        while (my $row = $sth->fetchrow_hashref) {
            $row->{diag_text} = iget($row->{diag_text});
            $row->{short_text} = iget($row->{short_text});
            # Будем ли показывать ссылку "Более детальные комментарии описаны здесь"
            $row->{show_details_url} = exists $details_link{$row->{diag_id}} ? 1 : 0;
            $row->{grab_tooltip} = 1 if !$row->{token};
            lock_hash(%$row);
            $data{$row->{diag_id}} = $row;
        }
        lock_hash(%data);
        $cache{$type}->{$lang} = {time => time(), data => \%data};
    }

    return $cache{$type}->{$lang}->{data};
}
}

=head2 get_diag_by_id

Получить по id текст отклонения

=cut

sub get_diag_by_id($$){
    my ($id, $hashref) = @_;
    if (exists $hashref->{$id}) {
        return $hashref->{$id};
    } else {
        return {
            'diag_id' => $id,
            'diag_text' => "Текст неизвестен",
            'grab_tooltip' => 1,
        };
    }
}

=head2 creative_get_diags($rejection_reason_ids, $type)

Получить описание для причин отлонения креатива

Параметры:
    $rejection_reason_ids - массив [] id приин отклонения
    $type - тип причины отклонения: 'common' или 'performance' (по умолчанию)

Результат:
     $reasons - {
         rejection_reason_id => {
             name => '.....',
             description => '.....'
         }
     }

=cut

sub creative_get_diags {
    my ($rejection_reason_ids, $type) = @_;
    $type ||= 'performance';

    my $diags = get_diags_hash($type);
    return { map {
        $_ => exists $diags->{$_} ? {description => $diags->{$_}->{diag_text}, name => $diags->{$_}->{short_text}} : {}
    } @$rejection_reason_ids };
}

1;
