package MoneyTransfer;

# $Id$

=head1 NAME

  MoneyTransfer

=head1 DESCRIPTION

    Модуль с функциями для перевода денег с кампаний на кампании,
    а так же для другими функциями, связанными с финансовыми операциями

=cut

use warnings;
use strict;
use List::Util qw/max min sum/;
use List::MoreUtils qw/any uniq/;
use Yandex::HashUtils;
use Yandex::Balance;
use Yandex::I18n;
use Yandex::ScalarUtils;
use Yandex::Validate qw/is_valid_float/;

use Settings;
use Currencies;
use Campaign;
use Campaign::Types;
use RBACElementary;
use RBACDirect;
use TextTools;
use Tools;
use Yandex::DBTools;
use Yandex::DBShards;
use YandexOffice;
use User;
use Client;
use Client::CustomOptions;
use Primitives;
use PrimitivesIds;
use Currency::Format;
use Common qw/get_user_camps_by_sql/;
use Models::CampaignOperations;
use TTTools;

use base qw/Exporter/;

our @EXPORT = qw/
                validate_transfer_money_rough
                prepare_transfer_money
                validate_transfer_money
                process_transfer_money

                pay_error_code_2_text
                prepare_and_validate_pay_camp
                /;

use utf8;

my @types_for_simultaneously_payment = qw/text cpm_banner cpm_deals/;

sub TRANSFER_ERRORS {
    my ($type, $error_type, $param, $extra_param) = @_;

    my $transfer_errors = {
        'ManyToManyError' => {
            'text'      => iget('Перенос средств с нескольких кампаний на несколько невозможен'),
            'wallet'    => iget('Перенос средств с нескольких счетов на несколько невозможен')
        },
        'DifferentCurrenciesError' => {
            'text'      => iget('Перенос средств между кампаниями в разных валютах невозможен'),
            'wallet'    => iget('Перенос средств между счетами в разных валютах невозможен')
        },
        'NotEnoughMoneyError' => {
            'text'      => iget('Недостаточно прав для переноса денег между кампаниями агентства. %s', $param),
            'wallet'    => iget('Недостаточно прав для переноса денег между счетами агентства. %s', $param),
        },
        'NotOwnerError' => {
            'text'      => iget("Кампания %s вам не принадлежит.", $param),
            'wallet'    => iget("Счет %s вам не принадлежит.", $param),
        },
        'PayAndTransferError' => {
            'text'      => iget("Оплата кампании %s и перенос средств на неё запрещены.", $param),
            'wallet'    => iget("Оплата общего счета %s и перенос средств на него запрещены.", $param),
        },
        'TransferToCampWithBlockedMoneyError' => {
            'text'      => iget("Перенос на кампанию с заблокированными деньгами невозможен"),
            'wallet'    => iget("Перенос на счет с заблокированными деньгами невозможен"),
        },
        'NotEnoughRightsForTransferError' => {
            'text'      => iget("Недостаточно прав для переноса денег между кампаниями агентства",$param),
            'wallet'    => iget("Недостаточно прав для переноса денег между счетами агентства",$param),
        },
        'CantTransferToAgencyAccountError' => {
            'text'      => iget("Перенос денег на кампанию %s, принадлежащую агенству, невозможен.",$param),
            'wallet'    => iget("Перенос денег на счет %s, принадлежащий агенству, невозможен.",$param),
        },
        'CantTransferFromAgencyAccountError' => {
            'text'      => iget("Перенос денег с кампании %s, принадлежащей агенству, невозможен.",$param),
            'wallet'    => iget("Перенос денег со счета %s, принадлежащего агенству, невозможен.",$param),
        },
        'TransferToAccountPosibleSummBonusError' => {
            'text'      => iget("Перенос средств на кампанию %s возможен на сумму не менее %s с учётом скидочного бонуса",$param,$extra_param),
            'wallet'    => iget("Перенос средств на счет %s возможен на сумму не менее %s с учётом скидочного бонуса",$param, $extra_param),
        },
        'TransferToAccountPosibleSummError' => {
            'text'      => iget("Перенос средств на кампанию %s возможен на сумму не менее %s",$param,$extra_param),
            'wallet'    => iget("Перенос средств на счет %s возможен на сумму не менее %s",$param, $extra_param),
        },
        'TransferFromAccountMinumalSummBonusError' => {
            'text'      => iget("Минимальная сумма для переноса с учетом скидочного бонуса %s",$param),
            'wallet'    => iget("Минимальная сумма для переноса с учетом скидочного бонуса %s",$param),
        },
        'TransferFromAccountPosibleSummError' => {
            'text'      => iget("Перенос средств возможен на сумму не менее %s",$param),
            'wallet'    => iget("Перенос средств возможен на сумму не менее %s",$param),
        },
        'TargetCampaignInSourcesError' => {
            'text'      => iget("Целевая кампания %s не должна присутствовать среди кампаний-источников",$param),
            'wallet'    => iget("Целевой счет %s не должен присутствовать среди счетов-источников",$param),
        },
        'LessThan0Error' =>{
            'text'      => iget("Остаток средств на кампании не может быть меньше 0."),
            'wallet'    => iget("Остаток средств на счету не может быть меньше 0."),
        },
        'TransferMoneyFromManagersToRegularError' =>{
            'text'      => iget("Перенос средств с кампании,  обслуживаемой персональным менеджером, на обычную кампанию невозможен."),
            'wallet'    => iget("Перенос средств со счета,  обслуживаемого персональным менеджером, на обычную кампанию невозможен."),
        },
        'FirstTransferNotLessThanError' =>{
            'text'      => iget("Первый перенос на кампанию, обслуживаемую персональным менеджером, возможен на сумму не менее %s", $param),
            'wallet'    => iget("Первый перенос на счет, обслуживаемый персональным менеджером, возможен на сумму не менее %s",$param),
        },
        'MoneyBlockedError' =>{
            'text'      => iget("Невозможно переносить заблокированные деньги."),
            'wallet'    => iget("Невозможно переносить заблокированные деньги."),
        },
        'BalanceCannotBeLessThanWithBonusOrHaveToBe0Error' =>{
            'text'      => iget("Остаток средств на кампании не может быть меньше %s с учётом скидочного бонуса, либо должен быть равен 0.",$param),
            'wallet'    => iget("Остаток средств на счету не может быть меньше %s с учётом скидочного бонуса, либо должен быть равен 0.",$param),
        },
        'BalanceCannotBeLessThanOrhaveToBe0Error' =>{
            'text'      => iget("Остаток средств на кампании не может быть меньше %s, либо должен быть равен 0.",$param),
            'wallet'    => iget("Остаток средств на счету не может быть меньше %s, либо должен быть равен 0.",$param),
        },
        'MinSummToTransferBonusError' =>{
            'text'      => iget("Минимальная сумма для переноса с учетом скидочного бонуса %s",$param),
            'wallet'    => iget("Минимальная сумма для переноса с учетом скидочного бонуса %s",$param),
        },
        'BalanceCannotBeLessThanOrhaveToBe0Error' =>{
            'text'      => iget("Остаток средств на кампании не может быть меньше %s, либо должен быть равен 0.",$param),
            'wallet'    => iget("Остаток средств на счету не может быть меньше %s, либо должен быть равен 0.",$param),
        },
        'BalanceCannotBeLessThanOrhaveToBe0Error' =>{
            'text'      => iget("Остаток средств на кампании не может быть меньше %s, либо должен быть равен 0.",$param),
            'wallet'    => iget("Остаток средств на счету не может быть меньше %s, либо должен быть равен 0.",$param),
        },
        'TransferPossibleForSummNotLessError' =>{
            'text'      => iget("Перенос средств возможен на сумму не менее %s",$param),
            'wallet'    => iget("Перенос средств возможен на сумму не менее %s",$param),
        },
        'BalanceCannotBeLessThanMinimumRateError' =>{
            'text'      => iget("Остаток средств на кампании №%s не может быть меньше чем максимальная ставка(%s).",$param,$extra_param),
            'wallet'    => iget("Остаток средств на счету №%s не может быть меньше чем максимальная ставка(%s).",$param,$extra_param),
        },
        'BalanceCannotBeLessThan30MinutsWorkError' =>{
            'text'      => iget("Остаток средств на кампании №%s не может быть меньше чем на 30 мин. работы.",$param),
            'wallet'    => iget("Остаток средств на счету №%s не может быть меньше чем на 30 мин. работы.",$param),
        },
        'TransferMoneyDeniedError' =>{
            'text'      => iget("Для данного клиента запрещен перенос средств"),
            'wallet'    => iget("Для данного клиента запрещен перенос средств"),
        },

    };

    return $transfer_errors->{$type}{$error_type} if exists $transfer_errors->{$type} && exists $transfer_errors->{$type}{$error_type};
    return undef;
}

=head2 validate_transfer_money_rough

    Грубая проверка валидности входных параметров

=cut

sub validate_transfer_money_rough {

    my $params = shift;

    foreach my $type (qw/to from/) {
        foreach my $cid (keys %{$params->{$type}}) {
            if ($params->{$type}{$cid} !~ /^\d+(\.\d*)?$/) {
                return iget('Неверно указана сумма переноса.');
            }
        }
    }

    return iget('Перенос невозможен: не выбраны кампании (кампания)') if !(keys %{$params->{from}}) || !(keys %{$params->{to}});
}

=head2 prepare_transfer_money

    Подготавливает необходимые данные для проведения транзакции:
        1) получает данные из базы по кампаниям и пользователю
        2) формирует массивы кампаний

    Перенос возможен только между кампаниями в одинаковой валюте.

    На вход хэш формата:
    {
        to => {cid => sum, ...}    # sum -- сумма для переноса без НДС и без скидочного бонуса
        from => {cid => sum, ...}    # sum -- сумма для переноса без НДС и без скидочного бонуса
    }

    На выходе получается ссылка не хеш: {
        campaigns_from => [{cid => 123, sum => ...}], # см. _get_from_camps_transfer(); суммы с НДС и без скидочного бонуса
        campaigns_to => [{cid => 456, moderated => ...}], # см. _get_to_camps_transfer()
        ClientID =>
        not_resident =>
        total_move =>
        total_get =>
    }

    На выходе для мультивалютных кампаний все суммы получаются без НДС и без скидочного бонуса.

=cut

sub prepare_transfer_money {

    my (undef, $uid, $params) = @_;

    $params->{campaigns_from} = _get_from_camps_transfer([keys %{$params->{from}}]);
    $params->{campaigns_to} = _get_to_camps_transfer([keys %{$params->{to}}]);

    $params = hash_merge $params, get_user_data($uid, [qw/ClientID not_resident/]);

    my @client_ids_without_nds = map {$_->{ClientID}} grep {!defined $_->{NDS} && $_->{currency} ne 'YND_FIXED'} (@{$params->{campaigns_to}}, @{$params->{campaigns_from}});
    my $nds_data = mass_get_client_NDS(\@client_ids_without_nds, fetch_missing_from_balance => 1);

    $params->{campaigns_from} = _modify_camp_sums_nds($params->{campaigns_from}, remove_nds => 1, nds_data => $nds_data);
    $params->{campaigns_to} = _modify_camp_sums_nds($params->{campaigns_to}, remove_nds => 1, nds_data => $nds_data);

    for my $camp_from (@{$params->{campaigns_from}}) {
        $camp_from->{sum_move} = $params->{from}{$camp_from->{cid}};

        # если разница между остатком и суммой переноса мала - округляем сумму переноса
        if ((abs($camp_from->{sum_move} - $camp_from->{sum_rest}) - 0.01) < -0.0000001) {
            $camp_from->{sum_move} = $camp_from->{sum_rest};
        }

        # Проверка для сервисируемых кампаний
        $camp_from->{serviced} = rbac_is_scampaign(undef, $camp_from->{cid});
        $params->{total_move} += $camp_from->{sum_move};
    }

    for my $camp_to (@{$params->{campaigns_to}}) {
        $camp_to->{sum_get} = $params->{to}{$camp_to->{cid}};

        # Проверка для сервисируемых кампаний
        $camp_to->{serviced} = rbac_is_scampaign(undef, $camp_to->{cid});
    }

    $params->{total_move} = round2s($params->{total_move});

    return $params;
}

=head2 validate_transfer_money

    Проверки возможности проведения транзакции.
    Возвращает пустую строку, если все хорошо, и текст ошибки, в противном случае.

    Выполняемые проверки:
        Кто и куда может переносить:
        1) агентство может переводить деньги между кампаниями всех своих клиентов
        2) простой клиент может переносить деньги только между своими кампаниями
        3) менеджер может переносить деньги только между сервисируемыми кампаниями своих сервисируемых клиентов, и не может между кампаниями своих агентств
        4) нельзя переносить деньги между не своими кампаниями для клиента, кампаниями не своего агентства для агентств

        Когда нельзя переносить:
        1) на непромодерированную кампанию перенести деньги нельзя
        2) Нельзя перенсти деньги с кампании A на нее же
        3) Остаток средств на кампании-источнике должен быть больше 0
        4) Если деньги на кампании-источнике или целевой кампании заблокированы - переносить нельзя
        5) если у целевой кампании есть статус statusNoPay - переносить нельзя
        6) нельзя переносить деньги с сервисируемой кампании на не сервесируемую
        7) между кампаниями в разных валютах
        8) на клиента можно поставить опцию "запретить перенос средств", тогда с клиента нельзя перенести деньги на другого клиента

        Другие ограничения и проверки:
        1) если кампания-источник остановлена, то можно перенести либо все деньги либо не менее MIN_TRANSFER_MONEY, причем чтобы на кампании осталось не менее MIN_TRANSFER_MONEY
        2) если кампания-источник активная, то перенести можно не менее MIN_TRANSFER_MONEY, причем должно остаться не меньше чем на 30 минут работы

    Для мультивалютных кампаний все суммы передаются без НДС и без скидочного бонуса.

    Принимает необязательные именованные параметры:
        client_discount – процент скидки клиента; используется для переносов внутри одного клиента, когда суммы вводятся с учётом скидочного бонуса
                          указание этого параметра приводит к тому, что в сообщениях об ошибках значения констант будут указаны также со скидочным бонусом

=cut

sub validate_transfer_money {

    my ($rbac, $UID, $params, %O) = @_;

    my $login_role = rbac_who_is_detailed($rbac, $UID)->{role};
    # -- search for wallet campaign
    my $has_wallet = grep{ $_->{type} eq 'wallet' }(@{$params->{campaigns_from}}, @{$params->{campaigns_to}});
    my $ERROR_TYPE = $has_wallet > 0 ? 'wallet':'text';


    if (scalar @{$params->{campaigns_to}}>1 && scalar @{$params->{campaigns_from}}>1) {
         return TRANSFER_ERRORS('ManyToManyError', $ERROR_TYPE );
    }

    my @currencies = uniq map {$_->{currency}} @{$params->{campaigns_to}}, @{$params->{campaigns_from}};
    my $currency;
    if (scalar @currencies == 1) {
        $currency = $currencies[0];
    } else {
        return TRANSFER_ERRORS('DifferentCurrenciesError', $ERROR_TYPE );
    }

    my $min_transfer_constant = $O{custom_min_transfer_money} ?
        calc_min_pay_sum(get_clientid(cid =>  $params->{campaigns_to}->[0]->{cid}), $currency)
        : get_currency_constant($currency, 'MIN_TRANSFER_MONEY');
    my $min_transfer_sum_str = format_sum_of_money($currency, $min_transfer_constant);

    foreach my $camp_to (@{$params->{campaigns_to}}) {

        my $CAMP_TO_ERROR_TYPE = $camp_to->{type} eq 'wallet' ? 'wallet':'text';

        if (!$camp_to || !$camp_to->{moderated}) {
            return iget("Перенос невозможен.");
        } elsif ($camp_to->{statusNoPay} eq 'Yes') {
            return TRANSFER_ERRORS('PayAndTransferError', $CAMP_TO_ERROR_TYPE,$camp_to->{cid} );
        }
        $camp_to->{money_type} = check_block_money_camp($camp_to) ? 'blocked' : 'real';
        if ($camp_to->{money_type} ne 'real') {
            return TRANSFER_ERRORS('TransferToCampWithBlockedMoneyError', $CAMP_TO_ERROR_TYPE );
        }

        if ( $camp_to->{AgencyUID} && $login_role =~ /client$/ ) {
            if (! rbac_get_allow_transfer_money($rbac, $UID, $camp_to->{cid}) ) {
                return TRANSFER_ERRORS('NotEnoughRightsForTransferError', $CAMP_TO_ERROR_TYPE, $camp_to->{cid} );
            }
        } else {
            if (!rbac_user_allow_edit_camp($rbac, $UID, $camp_to->{cid}) ) {
                return TRANSFER_ERRORS('NotOwnerError', $CAMP_TO_ERROR_TYPE, $camp_to->{cid} );
            }
        }

        if ($login_role eq 'manager' && $camp_to->{AgencyID}) {
            return TRANSFER_ERRORS('CantTransferToAgencyAccountError', $CAMP_TO_ERROR_TYPE, $camp_to->{cid} );
        }

        if (round2s($camp_to->{sum_get}) < $min_transfer_constant && round2s($camp_to->{sum_get}) < round2s($params->{total_move})) {
            if ($O{client_discount}) {
                return TRANSFER_ERRORS('TransferToAccountPosibleSummBonusError', $CAMP_TO_ERROR_TYPE, $camp_to->{cid},$min_transfer_sum_str );
            } else {
                return TRANSFER_ERRORS('TransferToAccountPosibleSummError', $CAMP_TO_ERROR_TYPE, $camp_to->{cid},$min_transfer_sum_str );
            }
        }

        my $cid_clid = get_hash_sql(PPC(cid => [ map { $_->{cid} } @{$params->{campaigns_from}}, @{$params->{campaigns_to}} ]),
            [ "select c.cid, u.ClientID from campaigns c join users u on u.uid = c.uid", where => { 'c.cid' => SHARD_IDS } ]);
        my $users_options = mass_get_user_custom_options([ values %$cid_clid ]);
        for my $camp_from (@{$params->{campaigns_from}}) {

            if ($camp_from->{cid} == $camp_to->{cid}) {
                return TRANSFER_ERRORS('TargetCampaignInSourcesError', ($camp_from->{type} eq 'wallet'? 'wallet':'text'), $camp_from->{cid} );
            }

            if ($login_role eq 'manager' && $camp_from->{AgencyID}) {
                return TRANSFER_ERRORS('CantTransferFromAgencyAccountError', ($camp_from->{type} eq 'wallet'? 'wallet':'text'), $camp_from->{cid} );
            }

            # перенос разрешен только внутри одного агентства
            if ($camp_from->{AgencyID} != $camp_to->{AgencyID}) {
                return TRANSFER_ERRORS('LessThan0Error', 'text');
            }

            # С сервисируемой на несервисируемую переносить нельзя
            if ( $camp_from->{serviced} && !$camp_to->{serviced} ) {
                return TRANSFER_ERRORS('TransferMoneyFromManagersToRegularError',$ERROR_TYPE);
            }

            # с клиента запрещен перенос денег
            my $clientid_from = $cid_clid->{$camp_from->{cid}};
            my $user_from_options = $users_options->{$clientid_from};
            if ($user_from_options->{disallow_money_transfer} && $cid_clid->{$camp_from->{cid}} != $cid_clid->{$camp_to->{cid}} ) {
                return TRANSFER_ERRORS('TransferMoneyDeniedError', 'text');
            }
        }
    }

    for my $camp_from (@{$params->{campaigns_from}}) {

        $camp_from->{money_type} = check_block_money_camp($camp_from) ? 'blocked' : 'real';

        my $CAMP_FROM_ERROR_TYPE = $camp_from->{type} eq 'wallet' ? 'wallet':'text';

        if ( $camp_from->{AgencyUID} && $login_role =~ /client$/ ) {
            if (! rbac_get_allow_transfer_money($rbac, $UID, $camp_from->{cid}) ) {
                return TRANSFER_ERRORS('NotEnoughMoneyError', $CAMP_FROM_ERROR_TYPE, $camp_from->{cid});
            }
        } else {
            if (!rbac_user_allow_edit_camp($rbac, $UID, $camp_from->{cid}) ) {
                return TRANSFER_ERRORS('NotOwnerError', $CAMP_FROM_ERROR_TYPE, $camp_from->{cid});
            }
        }

        if (round2s(round2s($camp_from->{sum_rest}) - $camp_from->{sum_move}) < 0) {
            return TRANSFER_ERRORS('LessThan0Error', $CAMP_FROM_ERROR_TYPE);
        }

        if ($camp_from->{money_type} ne 'real') {
            return TRANSFER_ERRORS('MoneyBlockedError', $CAMP_FROM_ERROR_TYPE);
        }

        # прогнозируемый остаток средств на кампании после переноса
        my $future_rest = round2s(round2s($camp_from->{sum_rest}) - round2s($camp_from->{sum_move}));

        if ($camp_from->{stopped}) {
            # для остановленной кампании можно переносить либо все средства, либо не менее MIN_TRANSFER_MONEY, причём, чтобы на кампании осталось тоже не менее MIN_TRANSFER_MONEY
            if ( $future_rest - 0.01 >= -0.0000001) {
                if ($camp_from->{sum_move} - $min_transfer_constant < -0.0000001) {
                    if ($camp_from->{currency} ne 'YND_FIXED' && $O{client_discount}) {
                        return TRANSFER_ERRORS('TransferFromAccountMinumalSummBonusError',$CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                    } else {
                        return TRANSFER_ERRORS('TransferFromAccountPosibleSummError', $CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                    }
                } elsif ($future_rest - $min_transfer_constant < -0.0000001) {
                    if ($camp_from->{currency} ne 'YND_FIXED' && $O{client_discount}) {
                        return TRANSFER_ERRORS('BalanceCannotBeLessThanWithBonusOrHaveToBe0Error', $CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                    } else {
                        return TRANSFER_ERRORS('BalanceCannotBeLessThanOrhaveToBe0Error', $CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                    }
                }
            }
        } else {
            my $wallet_without_active_camps = $camp_from->{type} eq 'wallet' &&
                                            get_user_camps_by_sql({'c.uid' => $camp_from->{uid},
                                                                   'c.wallet_cid' => $camp_from->{cid},
                                                                   'c.type' => get_camp_kind_types('under_wallet'),
                                                                   'c.statusShow' => 'Yes',
                                                                  },
                                                                  {shard => {cid => $camp_from->{cid}}, only_count => 1})->{count} == 0;

            # для запущенной кампании проверяем резерв
            if ($camp_from->{sum_move} < $min_transfer_constant &&
                # если денег меньше минимального перевода, но это общий счет и переводят весь остаток - то разрешаем.
                ! ($wallet_without_active_camps &&
                   sprintf("%.2f", $camp_from->{sum_move}) == sprintf("%.2f", $camp_from->{sum_rest}))
            ) {
                if ($camp_from->{currency} ne 'YND_FIXED' && $O{client_discount}) {
                    return TRANSFER_ERRORS('MinSummToTransferBonusError', $CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                } else {
                    return TRANSFER_ERRORS('TransferPossibleForSummNotLessError', $CAMP_FROM_ERROR_TYPE, $min_transfer_sum_str);
                }
            } elsif ($camp_from->{platform} eq 'context') {
                my $bid = Campaign::get_max_bid($camp_from->{cid}, $currency)->{price_context};
                if ($future_rest < $bid) {
                    return TRANSFER_ERRORS('BalanceCannotBeLessThanMinimumRateError', $CAMP_FROM_ERROR_TYPE, $camp_from->{cid}, $bid);
                }
            } else {
                my @cids_from;
                if ($camp_from->{type} eq 'wallet') {
                    @cids_from = @{get_one_column_sql(PPC(uid => $camp_from->{uid}), "
                        select c.cid
                        from campaigns c
                        where c.statusEmpty = 'No'
                          and c.archived = 'No'
                          and c.statusShow = 'Yes'
                          and c.statusModerate != 'New'
                          and c.type = 'text'
                          and c.uid = ?
                          and c.wallet_cid = ?
                    ", $camp_from->{uid}, $camp_from->{cid}) || []};
                } else {
                    @cids_from = ($camp_from->{cid});
                }

                my $camps_min_rest = @cids_from
                    ? max(sum(map {get_camp_min_rest($_, only_forecast => 1, dont_round => 1)} @cids_from), get_currency_constant($currency, 'MIN_TRANSFER_MONEY'))
                    : 0;
                if (sprintf("%.2f", $future_rest) < sprintf("%.2f", $camps_min_rest)) {
                    return TRANSFER_ERRORS('BalanceCannotBeLessThan30MinutsWorkError', $CAMP_FROM_ERROR_TYPE, $camp_from->{cid});
                }
            }
        }
    }

    return ''; #ok
}

=head2 process_transfer_money

    проведение транзакций, используется интерфейс Баланса
    для мультивалютных: суммы в переданных данных о кампаниях должны быть без НДС и без скидочного бонуса
    Возвращает:
        0 - все хорошо
        текст ошибки - что-то пошло не так
        умирает - если умирает вызов баланса

    именованные параметры:
        timeout - timeout запроса к Биллингу, в API используется 300 секунд,
            если не передать, то будет дефолт из Yandex::Balance, то есть 60 секунд
        detailed_response - отдавать ли развернутый ответ. если истина, то вернётся массив движений средств.
        move_all_qty - переносить все что есть на кампаниях
        is_enable_wallet -- перенос для включения общего счета (используется для определения промодерированности кампании общего счета)
        dont_create_campaigns -- не создавать кампании в Балансе, а считать их уже созданными и актуальными

=cut

sub process_transfer_money {

    my (undef, $uid, $UID, $params, %O) = @_;

    my @params_sources;
    my @params_dests;

    my $sum_move = 0;

    my @client_ids_without_nds = grep {$_ && $_ > 0} map {$_->{ClientID}} grep {!defined $_->{NDS} && $_->{currency} ne 'YND_FIXED'} @{$params->{campaigns_from}}, @{$params->{campaigns_to}};
    my $nds_data = {};
    if (@client_ids_without_nds) {
        $nds_data = mass_get_client_NDS(\@client_ids_without_nds, fetch_missing_from_balance => 1);
    }

    $params->{campaigns_from} = _modify_camp_sums_nds($params->{campaigns_from}, add_nds => 1, additional_fields => [qw/sum_move/], nds_data => $nds_data);
    for my $camp_from (@{$params->{campaigns_from}}) {
        my $digit_count = get_currency_constant($camp_from->{currency} // 'YND_FIXED', 'PRECISION_DIGIT_COUNT');
        my $one_param = {
            ServiceID       => $Settings::SERVICEID{direct},
            ServiceOrderID  => $camp_from->{cid},
# EXPIRES 20150815: пока не исправили все суммы на кампаниях в Балансе до соответствующих знаков, округляем всегда до 6 знаков
            QtyOld          => sprintf( "%.6f", $camp_from->{sum} ),
            QtyNew          => sprintf( "%.6f", $camp_from->{sum} - $camp_from->{sum_move} ),
#            QtyOld          => sprintf( "%.${digit_count}f", $camp_from->{sum} ),
#            QtyNew          => sprintf( "%.${digit_count}f", $camp_from->{sum} - $camp_from->{sum_move} ),
            Tolerance       => 10**(-$digit_count),
        };

        if ($O{move_all_qty}) {
            $one_param->{AllQty} = 1;
        }

        push @params_sources, $one_param;
    }

    $params->{campaigns_to} = _modify_camp_sums_nds($params->{campaigns_to}, add_nds => 1, additional_fields => [qw/sum_get/], nds_data => $nds_data);
    for my $camp_to (@{$params->{campaigns_to}}) {
        my $digit_count = get_currency_constant($camp_to->{currency} // 'YND_FIXED', 'PRECISION_DIGIT_COUNT');
        my $qty_delta = sprintf( "%.${digit_count}f", $camp_to->{sum_get} );
        my $tolerance = 10**(-$digit_count);

        if ($O{move_all_qty} && $qty_delta < $tolerance) {
            # биллинг не умеет распределять все средства (AllQty), если QtyDelta = 0.
            $qty_delta = $tolerance;
        }

        push @params_dests, {
            ServiceID       => $Settings::SERVICEID{direct},
            ServiceOrderID  => $camp_to->{cid},
            QtyDelta        => $qty_delta,
        };
        $sum_move += $qty_delta;
    }

    # параметры для CreateTransfer
    my @request_params;
    push @request_params, $UID, \@params_sources, \@params_dests;

    # create in balance for new camp
    if (!$O{dont_create_campaigns}) {
        my $result_create_campaigns_balance = create_campaigns_balance(undef, $UID, [map {$_->{cid}} @{$params->{campaigns_to}}], is_enable_wallet => $O{is_enable_wallet});
        if( !$result_create_campaigns_balance || defined $result_create_campaigns_balance->{error} || !$result_create_campaigns_balance->{balance_res} ) {
            return iget("Перенос невозможен");
        }
    }

    # transfer money

    my $res = balance_create_transfer(\@request_params, $O{timeout}, $O{detailed_response});

    if ($res eq 'BANNED_AGENCY_TRANSFER') {
        return (iget("возможность переноса средств между клиентами заблокирована."));
    } elsif ($res eq 'ORDERS_NOT_SYNCHRONIZED') {
        return (iget("Перенос невозможен, повторите операцию через 5 минут"));
    } elsif ($res eq 'TRANSFER_BETWEEN_LOYAL_GENERAL_CLIENTS') {
        return (iget("Перенос средств возможен только между заказами лояльных, либо нелояльных клиентов"));
    } elsif ($res eq 'NOT_ENOUGH_FUNDS_FOR_REVERSE') {
        return (iget("Перенос с данной кампании невозможен. Обратитесь в службу поддержки"));
    } elsif ($res eq 'NON_RESIDENT_TRANSFER') {
        return iget('Перенос невозможен. Переносы для субклиентов-нерезидентов запрещены.');
    } elsif (!ref($res) && $res) {
        return (iget("Перенос невозможен"));
    } else {
        log_cmd({cmd => '_transfer_money', UID => $UID, uid => $uid, cid_from =>join(',', map {$_->{cid}} @{$params->{campaigns_from}}), cid_to => join(',', map {$_->{cid}} @{$params->{campaigns_to}}), sum => $sum_move});
    }

    if ($O{detailed_response}) {
        return $res;
    } else {
        return 0;
    }
}

=head2 pay_error_code_2_text

    параметры именованные:

        error_min_shows
        error_code
        error_param
        error_cid
        error_region

        currency
        pseudo_currency_id
        product_id -- только для !is_direct

        is_direct
        easy_direct

    для ba.yandex.ru и Легкого интерфейса [в у.е.] - сообщение в рублях

=cut

sub pay_error_code_2_text {

    my %O = @_;

    my $campaign_type = $O{error_cid} ? get_camp_type(cid => $O{error_cid}) || 'text' : 'text';
    my $is_wallet = $campaign_type eq 'wallet' ? 1 : 0;

    my $has_nds = ($O{currency} eq 'YND_FIXED') ? 1 : 0;
    if ($O{error_code} == 1) {
        my $min_pay_sum = calc_min_pay_sum(get_clientid(cid =>  $O{error_param}), $O{currency});
        if ( $is_wallet ) {
            return iget("Сумма оплаты за один общий счет не должна быть меньше %s. Ошибочная сумма в общем счете №%s", _payerr_sum($min_pay_sum, %O, nds => $has_nds), $O{error_param});
        }
        return iget("Сумма оплаты за одну  кампанию не должна быть меньше %s. Ошибочная сумма в кампании №%s", _payerr_sum($min_pay_sum, %O, nds => $has_nds), $O{error_param});

    } elsif ($O{error_code} == 3) {
        return iget("Вы не указали ни одной суммы");
    } elsif ($O{error_code} == 4) {
        return iget("Извините, функция оплаты в данный момент недоступна. Попробуйте перезагрузить страницу. Если это не помогает зайдите позже.")
    } elsif ($O{error_code} == 5) {
        if ( $is_wallet ) {
            return iget("Неправильно указана оплачиваемая сумма в общем счете N%s", $O{error_param});
        }
        return iget("Неправильно указана оплачиваемая сумма в кампании N%s", $O{error_param});
    } elsif ($O{error_code} == 6) {
        if ( $is_wallet ) {
            return iget("Неправильно указана сумма в общем счете №%s (сумма должна быть больше 0)", $O{error_param});
        }
        return iget("Неправильно указана сумма в кампании N%s (сумма должна быть больше 0)", $O{error_param});
    } elsif ($O{error_code} == 9) {
        if ( $is_wallet ) {
            return iget("Оплата общего счета N%s запрещена", $O{error_param});
        }
        return iget("Оплата кампании N%s запрещена", $O{error_param});
    } elsif ($O{error_code} == 10) {
        if ( $is_wallet ) {
            return iget('Единовременный платеж за однин или несколько ваших общих счетов, обслуживаемых персональным менеджером, должен быть не меньше %s или не должен превышать размер возможного отсроченного платежа.', _payerr_sum($O{error_param}, %O));
        }
        return iget('Единовременный платеж за одну или несколько ваших кампаний, обслуживаемых персональным менеджером, должен быть не меньше %s или не должен превышать размер возможного отсроченного платежа.', _payerr_sum($O{error_param}, %O));
    } elsif ($O{error_code} == 11) {
        return iget('Количество тысяч показов должно быть целым числом.');
    } elsif ($O{error_code} == 13) {

        if ($O{product_id} && is_turkish_mcb(product_info(ProductID => $O{product_id})->{product_type})) {
            return iget('Для турецкой медийной кампании единовременно можно оплатить не меньше %s',
                    _payerr_sum($O{error_param}, %O));
        } elsif (!$O{error_region}) {
            return iget('Для региональной медийной кампании единовременно можно оплатить не меньше %s', _payerr_sum($O{error_param}, %O));
        } else {
            return iget('Для медийной кампании с геотаргетингом, включающим %s регион, единовременно можно оплатить не меньше %s',
                    $O{error_region} eq 'north-west' ? iget('северо-западный') : iget('московский'),
                    _payerr_sum($O{error_param}, %O));
        }

    } elsif ($O{error_code} == 14) {
        return iget('Оплата медийной кампании с недостаточным для сохранения условия трафиком запрещена, месячный прогноз показов должен быть не меньше %s тыс.', TTTools::format_units($O{error_param}, 'shows'));
    } elsif ($O{error_code} == 16) {
        return iget('Введён неверный промо-код');
    } elsif ($O{error_code} == 17) {
        return iget('Ошибка использования промо-кода');
    } elsif ($O{error_code} == 18) {
        return iget('Кампании, прошедшие предварительную проверку, можно оплачивать только отдельным счетом. Сформируйте счет отдельно для каждой подобной кампании.');
    } elsif ($O{error_code} == 19) {
        return iget('Одновременно можно оплатить только кампании одного типа обслуживания.');
    } elsif ($O{error_code} == 20) {
        return iget('Нельзя оплатить менее чем один пакет МКБ (кампания №%s).', $O{error_param});
    } elsif ($O{error_code} == 21) {
        return iget('Одновременно можно оплатить только кампании одного типа и валюты.');
    } elsif ($O{error_code} == 22) {
        return iget('Кампания %s подключена к общему счету, оплата невозможна.', $O{error_param});
    } elsif ($O{error_code} == 23) {
        return iget('Для оплаты медийных кампаний необходимо выставить отдельный счет. Оплата вместе с другими кампаниями недоступна.');
    } elsif ($O{error_code} == 24) {
        return iget('Для оплаты кампаний со сделками необходимо выставить отдельный счет. Оплата вместе с кампаниями без сделок недоступна.');
    } elsif ($O{error_code} == 25) {
        return iget('Для оплаты медийных кампаний на Главной необходимо выставить отдельный счет. Оплата вместе с другими кампаниями недоступна.');
    }

    return '';
}

=head2 prepare_and_validate_pay_camp

    Подготавливает структуру данных и валидирует ее для выставления счетов,

    перенесено из контроллера payforall

    Параметры позиционные:
        uid
        UID
        rbac
        login_rights

    Параметры именованные:
        sums - ссылка на хэш {cid => sum}; для мультивалютных sum -- сумма без скидочного бонуса, с НДС или без — зависит от параметра sums_with_nds
        sums_with_nds — суммы в sums с НДС (1) или без (0); если не указан — суммы без НДС
        multiuser
        is_easy_payment
        pseudo_currency
        agency_uid
        with_nds — ссылка на хеш кампания => 1/0 (есть ли НДС в сумме для этой кампании)

    Возвращает ссылку на модифицированный хэш входных параметров, добавляемые поля:
        error_code
        error_param
        error_cid
        error_region
        error_min_shows

        currency
        pseudo_currency_id
        product_id

        client_id
        not_resident
        overdraft_bill
        has_agency
        CreateRequest - запрос в баланс на создание счета
        pay_notification_data - данные для создания нотификаций об оплате

=cut

sub prepare_and_validate_pay_camp {

    my ($uid, $UID, $rbac, $login_rights, %O) = @_;

    my $params = \%O;

    my $cid_pay_method = {};
    my @pay_notification_data = ();

    if (!%{$O{sums}}) {
        $params->{error_code} = 3;
    }

    # проверяем, является ли пользователь не-резидентом
    if (!defined $params->{not_resident}) {
        $params->{not_resident} = get_one_user_field( $O{agency_uid} || $O{client_chief_uid}, 'not_resident') || 'No';
    }

    my (@cids_to_create, @CreateRequest, $pay_method, $serivicing_AgencyID, %product_info_cache, $first_product, %manager_offices, $is_media, $engineid);
    my ($product_id, $currency, $first_currency);

    my @cids = keys %{$O{sums}};
    my $all_camps_info = get_hashes_hash_sql(PPC(cid => \@cids), ["
                SELECT c.cid, c.uid, c.type
                     , c.statusNoPay
                     , c.AgencyUID
                     , IF (c.type = 'mcb', (SELECT sum(g.last_month_shows) FROM media_groups g where g.cid = c.cid), NULL) as last_month_shows
                     , IFNULL(c.currency, 'YND_FIXED') AS currency
                     , c.ProductID
                     , u.ClientID
                     , c.wallet_cid
                  FROM campaigns c
                  LEFT JOIN users u ON c.uid = u.uid",
                 WHERE => {cid => SHARD_IDS},
        ],
    );

    my $agencie_ids = get_uid2clientid(uid => [grep { defined } map {$_->{AgencyUID}} values %$all_camps_info]);
    my @client_ids = uniq map { $_->{ClientID} } grep { $_->{currency} ne 'YND_FIXED' } values %$all_camps_info;
    my $clients_nds_data = mass_get_client_NDS(\@client_ids, fetch_missing_from_balance => 1, rbac => $rbac);

    if (
        (any { $_->{type} eq "cpm_deals" } values %$all_camps_info) &&
        (any { $_->{type} ne "cpm_deals" } values %$all_camps_info)
    ) {
        $params->{error_code} = 24; # https://st.yandex-team.ru/DIRECT-80868
    } elsif (
        (any { $_->{type} eq "cpm_banner" } values %$all_camps_info) &&
        (any { $_->{type} ne "cpm_banner" } values %$all_camps_info)
    ) {
        $params->{error_code} = 23; # https://st.yandex-team.ru/DIRECT-80868
    } elsif (
        (any { $_->{type} eq "cpm_yndx_frontpage" } values %$all_camps_info) &&
        (any { $_->{type} ne "cpm_yndx_frontpage" } values %$all_camps_info)
    ) {
        $params->{error_code} = 25; # https://st.yandex-team.ru/DIRECT-80868
    }

    for my $cid ( sort {$b <=> $a} keys %{$O{sums}} ) {
        last if defined $params->{error_code};
        my $sum_from_sums = $O{sums}->{$cid};

        # проверяем способ оплаты
        my $current_pay_method = check_method_pay($cid);

        if (!defined $pay_method) {
            $pay_method = $current_pay_method;
        } elsif (defined $pay_method && ($pay_method ne $current_pay_method || $pay_method eq 'with_block')) {
            # тексты ошибок находятся в функции pay_error_code_2_text
            $params->{error_code} = 18;
            last;
        }

        # проверяем на то, что нам передали числа
        if ($sum_from_sums !~ /^\d+(?:\.\d+)?$/) {
            $params->{error_code} = 5;
            $params->{error_param} = $cid;
            $params->{error_cid} = $cid;
            last;
        }

        my $camp_info = $all_camps_info->{$cid};
        $currency = $camp_info->{currency};
        $first_currency = $currency unless defined $first_currency;
        $params->{currency} = $currency;
        $product_id = $camp_info->{ProductID};
        my $product = product_info(ProductID => $product_id);

        # от чайников в интерфейсе приходят рубли, пересчитываем в ye
        if ($O{is_easy_payment} && $currency eq 'YND_FIXED') {
            $sum_from_sums = sprintf("%.6f", sprintf("%d", $sum_from_sums) / $O{pseudo_currency}->{rate});
        } else {
            $sum_from_sums = sprintf("%.6f", $sum_from_sums);
            $sum_from_sums =~ s/\.00$//;
        }

        if (is_media_camp(type => $camp_info->{type})) {
            if (is_package_mcb($product->{product_type})) {
                $camp_info->{lowMonthShowsNoPay} = 0;
            } else {
                $camp_info->{lowMonthShowsNoPay} = ($camp_info->{last_month_shows} and $camp_info->{last_month_shows} < (is_turkish_mcb($product->{product_type}) ? $Settings::MIN_MONTH_MEDIA_SHOWS_FOR_TURKEY : $Settings::MIN_MONTH_MEDIA_SHOWS)) ? 1 : 0;
            }
        }

        my ($sum_with_nds, $sum);
        if ($currency eq 'YND_FIXED') {
            $sum_with_nds = $sum = $sum_from_sums;
        } else {
            my $client_id = $camp_info->{ClientID};
            my $nds_value = $clients_nds_data->{$client_id};
            die "No NDS known for ClientID $client_id" unless defined $nds_value;

            if ($O{sums_with_nds} || $O{with_nds}->{$cid}) {
                $sum = Currencies::remove_nds($sum_from_sums, $nds_value);
                $sum_with_nds = $sum_from_sums;
            } else {
                $sum = $sum_from_sums;
                $sum_with_nds = Currencies::add_nds($sum_from_sums, $nds_value);
            }
        }

        my $agency_id = ($camp_info->{AgencyUID}) ? $agencie_ids->{$camp_info->{AgencyUID}} : 0;
        # одновременно позволено платить либо за самоходные/сервисируемые кампании, либо за кампании одного агентства
        if (!defined $serivicing_AgencyID && $agency_id > 0) {
            $serivicing_AgencyID = $agency_id;
        } elsif (defined $serivicing_AgencyID && $serivicing_AgencyID != $agency_id) {
            $params->{error_code} = 19;
            last;
        }

        # не даем оплачивать кампании с подключенным общим счетом
        if ($camp_info->{wallet_cid}) {
            $params->{error_code} = 22;
            $params->{error_param} = $cid;
            $params->{error_cid} = $cid;
            last;
        }

        # не даём одновременно оплачивать кампании с разными продуктами
        if (!defined $first_product) {
            $first_product = $product;
        } elsif (!_can_pay_campaigns_simultaneously($first_product, $product, $first_currency, $currency)) {
            $params->{error_code} = 21;
        }

        $engineid = $product->{EngineID};
        $is_media = ($engineid == $Settings::SERVICEID{bayan}) ? 1 : 0;

        if ( $camp_info->{lowMonthShowsNoPay} && $is_media ) {
            # нельзя платить за МКБ кампанию если прогнозируемое количество показов в момент сохранения условия показа было меньше критического значения
            $params->{error_code} = 14;
            $params->{error_param} = is_turkish_mcb($product->{product_type}) ? $Settings::MIN_MONTH_MEDIA_SHOWS_FOR_TURKEY : $Settings::MIN_MONTH_MEDIA_SHOWS;
            $params->{error_cid} = $cid;
            last;
        } elsif ( $product->{UnitName} eq 'Shows' && $sum !~ /^\d+(?:\.0+)?$/ ) {
            $params->{error_code} = 11;
            $params->{error_param} = $cid;
            $params->{error_cid} = $cid;
            last;
        } elsif ( $sum == 0 ) {
            $params->{error_code} = 6;
            $params->{error_param} = $cid;
            $params->{error_cid} = $cid;
            last;
        }

        if ($camp_info->{statusNoPay} eq 'Yes' || camp_kind_in(type => $camp_info->{type}, 'no_send_to_billing')) {
            $params->{error_code} = 9;
            $params->{error_param} = $cid;
            $params->{error_cid} = $cid;
            last;
        }

        push( @cids_to_create, $cid );

        my $cur_uid = $uid;

        if ( $O{multiuser} ) {
            $cur_uid = $camp_info->{uid};
        }

        my $uid_for_validate_pay = (! $O{multiuser} && rbac_who_is($rbac, $cur_uid) =~ /client/) ? $O{client_chief_uid} : $cur_uid;
        if ($O{error} = validate_pay_camp($uid_for_validate_pay, $cid, undef, undef, $rbac)) {
            return \%O;
        }

        if ((any {$_ eq $product->{UnitName}} qw/Bucks QuasiCurrency/) && !is_package_mcb($product->{product_type})) {
            my $min_pay;
            if ($params->{custom_min_pay}) {
                $min_pay = calc_min_pay_sum($camp_info->{ClientID}, $currency);
            } else {
                $min_pay = get_currency_constant($currency, 'MIN_PAY');
            }
            my $difference = $sum - $min_pay;
            if ($difference < -0.0001) {
                $params->{error_code} = 1;
                $params->{error_param} = $cid;
                $params->{error_cid} = $cid;
                if ($currency eq 'YND_FIXED') {
                    $params->{pseudo_currency_id} = $O{pseudo_currency}->{id};
                }
                $params->{product_id} = $product_id;
                last;
            }
        }

        my $manager_uid = rbac_is_scampaign($rbac, $cid);
        if ($manager_uid && ! $O{agency_uid}) {
            $manager_offices{$manager_uid} ||= get_manager_office($manager_uid);
        }

        if (!defined $params->{error_code} && $is_media) {
            # баян
            my ($min_shows, $targ_region);
            if (is_package_mcb($product->{product_type})) {
                if ($sum < 1) {
                    $params->{error_code} = 20;
                    $params->{error_param} = $cid;
                    $params->{error_cid} = $cid;
                    last;
                }
                ($min_shows, $targ_region) = ($product->{packet_size}/1000, 225); # 225 - Россия
                $sum *= $product->{Price}; # в интерфейсе указывается количество оплачиваемых пакетов
            }
            else {
                ($min_shows, $targ_region) = get_mcb_camp_min_shows($cid);
            }
            my $min_pay = $min_shows * $product->{Price} / $product->{Rate};
            # для резидентов добавляем NDS
            if ($params->{not_resident} eq 'No' && $manager_offices{$manager_uid}->{use_nds}) {
                $min_pay *= 1 + $Settings::NDS_RU_DEPRECATED;
            }
            if (! $login_rights->{super_control} && $sum * $product->{Rate} < $min_shows) {
                $params->{error_code} = 13;
                $params->{error_param} = $min_pay;
                $params->{error_cid} = $cid;
                $params->{error_region} = $targ_region;
                $params->{error_min_shows} = $min_shows;
                if ($currency eq 'YND_FIXED') {
                    $params->{pseudo_currency_id} = $O{pseudo_currency}->{id};
                }
                $params->{product_id} = $product_id;
            }
        }

        my $qty;
        if ($product->{UnitName} ne 'Shows') {
            if (is_package_mcb($product->{product_type})) {
                # qty - количество единиц товара - количество показов для МКБ, в т.ч. пакетных
                # TODO: хорошо бы спрятать магическую тысячу в Rate из таблицы products --pankovpv
                $qty = $product->{packet_size}*($sum/$product->{Price});
            }
            else {
                $qty = $sum_with_nds;
            }
        } elsif ($product->{UnitName} eq 'Shows') {
            $qty = int($sum) * 1000;
        }

        # qty - кол-во показов или денег
        push @CreateRequest, {
            ServiceID => $product->{EngineID},
            ServiceOrderID => $cid,
            Qty => $qty,
        };
        # save notification data for pay on serviced campaigns
        push @pay_notification_data, [$cid, $sum, $O{client_chief_uid}];
    }

    # Регистрируем кампании в балансе
    if (!defined $params->{error_code} && @cids_to_create) {
        my $camp_balance_response = create_campaigns_balance( $rbac, $UID, \@cids_to_create );

        if( $camp_balance_response->{error} ) {
            $O{error} = $camp_balance_response->{error};
        } elsif (!$camp_balance_response->{balance_res}) {
            $O{error} = iget("Ошибка создания счёта, попробуйте повторить операцию.");
        } else {
            $params->{client_id} = $camp_balance_response->{client_id};
        }
        if ($O{error}) {
            return \%O;
        }
    }

    # после того как оторвали минимальный платеж для клиентов на обслуживании менеджером
    # платёж не может стать овертдравтным принудительно
    $params->{overdraft_bill} = 0;

    if ( !defined $params->{error_code} && @CreateRequest == 0 ) {
        $params->{error_code} = 3;
        if ($currency eq 'YND_FIXED') {
            $params->{pseudo_currency_id} = $O{pseudo_currency}->{id};
        }
        $params->{product_id} = $product_id;
    }

    $params->{has_agency} = (defined $serivicing_AgencyID) ? 1 : 0;
    $params->{CreateRequest} = \@CreateRequest;
    $params->{pay_notification_data} = \@pay_notification_data;
    $params->{currency} = $currency;
    # после перехода в реальную валюту в Балансе на кампании остаётся старый (уешный) продукт
    # и выставление счёта на N единиц товара "фишки директа" становится неправильным
    # изменять продукт они не могут, но сделали подпорку BALANCE-19969
    # раньше была возможности указать напрямую строку счёта (т.е. стоимость), а не количество товара
    # флажком QtyIsAmount

    return $params;
}

=head2 _payerr_sum

    Параметры:
        sum -- значение суммы или название мультивалютной константы
    Именованные параметры:
        is_direct
        easy_direct
        currency
        pseudo_currency_id
        error_min_shows -- только для !is_direct
        product_id -- только для !is_direct
        nds

=cut

sub _payerr_sum {
    my ($sum, %O) = @_;

    if ($O{is_direct}) {
        if (is_valid_float($sum)) {
            if (!$O{easy_direct}) {
                return format_sum_of_money($O{currency}, $sum, {nds => $O{nds}});
            } else {
                return format_sum_of_money_pseudo($O{pseudo_currency_id}, $sum);
            }
        } else {
            return format_const($O{currency}, $sum, ($O{easy_direct} ? $O{pseudo_currency_id} : undef), {nds => $O{nds}});
        }
    } else {
        my $product_info = product_info(ProductID => $O{product_id});
        if (!is_valid_float($sum)) {
            my $currency = $product_info->{currency};
            $sum = get_currency_constant($currency, $sum);
        }
        die '_payerr_sum can only handle multiples of 1k shows' if $product_info->{Rate} != 1000;
        my $sum_shows = $O{error_min_shows} / $product_info->{Rate};
        return iget('%s тыс. показов',
                    TTTools::format_int($sum_shows));
    }
}

=head2 _get_from_camps_transfer

    Получаем информацию по кампаниям источникам
    Для мультивалютных кампаний суммы возвращаются с НДС и без скидочного бонуса

=cut

sub _get_from_camps_transfer {
    my ($cids) = @_;

    # есть подозрение, что sum_spent не используется
    my $data = get_all_sql(PPC(cid => $cids), ["
                                SELECT c.cid
                                     , IF(wwc.is_sum_aggregated = 'Yes', c.sum_balance, c.sum) AS sum, c.sum_spent
                                     , IF(wwc.is_sum_aggregated = 'Yes', c.sum_balance, c.sum) - c.sum_spent AS sum_rest
                                     , c.statusModerate = 'New'
                                       or ( c.statusShow='No'
                                            && (co.stopTime is null or unix_timestamp(now()) - unix_timestamp(co.stopTime) > ?))
                                       or ( c.statusModerate = 'No' && c.statusActive = 'No' )
                                       or ( c.statusBsSynced = 'Yes' AND c.finish_time IS NOT NULL AND YEAR(c.finish_time) > 0 AND DATE(c.finish_time) < DATE(NOW()) )
                                            AS stopped
                                     , c.OrderID, c.ManagerUID, c.AgencyID, c.AgencyUID, statusPostModerate, statusModerate
                                     , IFNULL(c.currency, 'YND_FIXED') AS currency
                                     , u.ClientID
                                     , c.uid
                                     , c.type
                                     , c.wallet_cid
                                  FROM campaigns c
                                       LEFT JOIN wallet_campaigns wwc ON (wwc.wallet_cid = IF(c.type = 'wallet', c.cid, c.wallet_cid))
                                       LEFT JOIN users u ON u.uid = c.uid
                                       LEFT JOIN camp_options co ON co.cid = c.cid
                                ", where => {'c.cid' => SHARD_IDS}], $Settings::TRANSFER_DELAY_AFTER_STOP);

    # у кошельков sum_rest нужно считать со всеми кампаниями под ним, sum - имеет странный смысл
    # sum_spent не чиним, потому что кажется он не нужен
    my @wallet_cids = map { $_->{cid} } grep { $_->{type} eq 'wallet' } @$data;
    if (@wallet_cids) {
        my $wallets_rest = get_hash_sql(PPC(cid => \@wallet_cids), [
                                            'SELECT wc.cid
                                                  , wc.sum + SUM(c.sum - c.sum_spent) AS sum_rest
                                               FROM campaigns wc
                                               JOIN campaigns c ON c.uid = wc.uid AND c.wallet_cid = wc.cid',
                                              WHERE => {'wc.cid' => SHARD_IDS},
                                          'GROUP BY wc.cid']);
        for my $camp (@$data) {
            next if ! exists $wallets_rest->{ $camp->{cid} };
            $camp->{sum_rest} = $wallets_rest->{ $camp->{cid} };
        }
    }

    return $data;
}

=head2 _get_to_camps_transfer

    Получаем информацию по кампаниям приемникам

=cut

sub _get_to_camps_transfer {
    my ($cids) = @_;

    return get_all_sql(PPC(cid => $cids), ["
                            SELECT c.cid, statusNoPay
                                 , (statusModerate = 'Yes' and co.statusPostModerate != 'Yes' or ManagerUID > 0 or AgencyUID > 0)
                                        AS moderated
                                 , OrderID, ManagerUID, AgencyID, AgencyUID, statusPostModerate, statusModerate
                                 , IFNULL(c.currency, 'YND_FIXED') AS currency
                                 , u.ClientID
                                 , c.type
                                 , c.wallet_cid
                              FROM campaigns c
                                LEFT JOIN users u ON u.uid = c.uid
                                LEFT JOIN camp_options co on c.cid = co.cid
                            ", where => {'c.cid' => SHARD_IDS}]);
}

=head2 modify_camp_sums_nds

    Убирает/добавлет НДС с сумм на кампаниях. Работает с sum sum_spent sum_rest.
    НДС получает либо из ключа NDS в кампании, либо по ClientID из кампании.

=cut

sub _modify_camp_sums_nds {
    my ($camps_data, %O) = @_;

    my $nds_data = $O{nds_data};
    die 'no nds_data given' unless $nds_data;

    for my $camp (@$camps_data) {
        next if $camp->{currency} eq 'YND_FIXED';
        my $client_id = $camp->{ClientID};
        my $client_nds;
        if (defined $camp->{NDS}) {
            $client_nds = $camp->{NDS};
        } else {
            $client_nds = $nds_data->{$client_id} || 0;
        }
        $camp->{NDS} = $client_nds;
        my @fields = qw/sum sum_spent sum_rest/;
        push @fields, @{$O{additional_fields}} if $O{additional_fields};
        for my $field (@fields) {
            next unless defined $camp->{$field};
            if ($O{remove_nds}) {
                $camp->{$field} = Currencies::remove_nds($camp->{$field}, $client_nds);
            } elsif ($O{add_nds}) {
                $camp->{$field} = Currencies::add_nds($camp->{$field}, $client_nds);
            }
        }
    }

    return $camps_data;
}

=head2 _can_pay_campaigns_simultaneously

Функция определяет, можно ли оплачивать кампании в одном запросе на оплату

Параметры
    $first_product -- описание продукта первой кампании
    $second_product -- описание продукта второй кампании
    $first_currency -- валюта первой кампании
    $second_currency -- валюта второй кампании

Результат
    1 - можно, 0 - нельзя

=cut

sub _can_pay_campaigns_simultaneously {
    my ($first_product, $second_product, $first_currency, $second_currency) = @_;

    if ($first_product->{ProductID} == $second_product->{ProductID}) {
        return 1;
    }

    # директовские кампании после перехода в валюту без копирования сохраняют старый ProductID, но изменют свою валюту
    # при совпадении валюты даём одновременно оплачивать директовские кампании, которые сейчас в валюте, но имеют продукт в у.е.
    # также можно оплачивать в одном запросе кампании с product_type="text",product_type="cpm_banner" и product_type="cpm_deals"
    if ((any {$_ eq $first_product->{type}} @types_for_simultaneously_payment)
        && (any {$_ eq $second_product->{type}} @types_for_simultaneously_payment)
        && $second_product->{EngineID} == $first_product->{EngineID}
        && $first_currency eq $second_currency) {
        return 1;
    }

    return 0;
}

1;

