package Monitor;

=pod
    $Id$
    подобие RRD

=cut

use strict;
use warnings;

use base qw/Exporter/;

use Settings;
use Yandex::TimeCommon;
use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::HashUtils;

our $MAX_STAT_DAYS = 30;

our @EXPORT = qw//;

our @AGG = qw/sec hour day/;

# запись значения в таблицу
sub save_values {
    my ($vals, $time) = @_;
    while(my ($name, $val) = each %$vals) {
        save_target_value($name, $val, $time);
    }
}

=head2 accumulate_values

    добавление значений 

=cut
sub accumulate_values {
    my ($vals, $time) = @_;
    while(my ($name, $val) = each %$vals) {
        accumulate_target_value($name, $val, $time);
    }
}

=head2 save_target_value

    запись значения в таблицу

    Параметры 
        $name -- имя таргета
        $value_with_descr -- число/хеш с описанием (см. примеры)
        $time


    Примеры вызовов

    # Простой вариант. Таргет должен обязательно существовать к моменту сохранения значения, иначе -- die.
    Monitor::save_target_value('very_interesting_camps', $N);

    # Полный вариант. Таргет по необходимости создается/обновляется. 
    # возможные ключи для описания значения: 
    #   value       -- (обязательно) собственно значение
    #   description -- (не обязательно) человекопонятное описание таргета
    #   desc        -- (не обязательно) синоним description. Если указано и description, и desc -- будет использован desc. 
    #   units       -- (не обязательно) единицы измерения. 'num' (штуки, кампании, объявления, у.е. и т.п. числа) или 'proc' (проценты). По умолчанию -- 'num'.
    #   old_name    -- (не обязательно) предыдущее имя таргета (все значения сохранятся на старый id, имя таргета изменится при сохранении первого значения),
    #                  позволяет менять названия таргетов без миграций
    Monitor::save_target_value('very_interesting_banners', {value => $N, desc => 'Очень интересные объявления'});
    Monitor::save_target_value('disabled_behavior', {value => $N, desc => 'Кампаний с отключенным поведенческим таргетингом', units => 'proc'});
    Monitor::save_target_value('disabled_behavior', {value => $N, description => 'Кампаний с отключенным поведенческим таргетингом', units => 'proc'});

=cut 

sub save_target_value {
    my ($name, $value_with_descr, $time) = @_;
    $time ||= time();
    my $value = ref $value_with_descr eq 'HASH' ? $value_with_descr->{value} : $value_with_descr;
    my $target = get_or_create_target($name, $value_with_descr);
    my @agg = @AGG;
    # пишем само значение в минимальный уровень
    my $oldagg  = shift @agg;
    do_sql(MONITOR, "REPLACE INTO monitor_values_$oldagg (target_id, measure_time, value) 
                            VALUES (?, ?, ?)", 
                            $target->{target_id}, unix2mysql(ts_round($time, $oldagg)), $value);
    # пишем аггрегаты
    for my $newagg (@agg) {
        do_sql(MONITOR, "REPLACE INTO monitor_values_$newagg (target_id, measure_time, value)
                                SELECT target_id, ?, avg(value)
                                  FROM monitor_values_$oldagg
                                 WHERE target_id = ?
                                   AND measure_time >= ?
                                   AND measure_time < ?
                                 GROUP BY target_id",
                                   unix2mysql(ts_round($time, $newagg)),
                                   $target->{target_id},
                                   map {unix2mysql($_)} ts_get_borders($time, $newagg)
                                   );
        $oldagg = $newagg;
    };
}


=head2 accumulate_target_value

    Аналогично save_target_value, но значения суммируются вместо усреднения.
    Полезно для подсчетов "сколько всего мы приняли от Каталогии баннеров за день"

    Таргеты прозрачно создаются и обновляются, как и в save_...

    К таргетам типа "процентны" не применяется -- падает

    Код значительно повторяет первую часть save_target_value, но объединять и абстрагировать не представляется пока необходимым. 

=cut
sub accumulate_target_value {
    my ($name, $value_with_descr, $time) = @_;
    $time ||= time();
    my $value = ref $value_with_descr eq 'HASH' ? $value_with_descr->{value} : $value_with_descr;
    my $target = get_or_create_target($name, $value_with_descr);
    die "incompatible units for accumulate" if $target->{units} eq 'proc';
    for my $lvl (@AGG){
        do_sql(MONITOR, 
            "INSERT INTO monitor_values_$lvl (target_id, measure_time, value) 
            VALUES (?, ?, ?)
            ON DUPLICATE KEY UPDATE value = value + values(value)", 
            $target->{target_id}, unix2mysql(ts_round($time, $lvl)), $value
        );
    }
}


=head2 get_or_create_target

    Создает, обновляет, или просто читает из базы таргет с указанным именем

=cut 

sub get_or_create_target
{
    my ($name, $target_descr) = @_;

    my $target_values = ref $target_descr eq 'HASH' ? hash_cut($target_descr, qw/description desc units/) : {};
    $target_values->{name} = $name;
    if (exists $target_values->{desc}) {
        $target_values->{description} = delete $target_values->{desc};
    }

    my $target = get_one_line_sql(MONITOR, "SELECT target_id, name, description, units FROM monitor_targets WHERE name = ?", $name);
    # проверим, есть ли что-то по старому имени
    if (!$target && ref $target_descr eq 'HASH' && $target_descr->{old_name}) {
        $target = get_one_line_sql(MONITOR, "SELECT target_id, name, description, units FROM monitor_targets WHERE name = ?", $target_descr->{old_name});
    }

    if (!$target || grep {defined $target_values->{$_} && $target_values->{$_} ne $target->{$_}} keys %$target_values) {
        if ($target) {
            $target_values->{target_id} = $target->{target_id};
        } else {
            $target_values->{units} ||= 'num';
        }
        do_insert_into_table(MONITOR, "monitor_targets", $target_values, on_duplicate_key_update => 1, key => 'target_id');
        $target = get_one_line_sql(MONITOR, "SELECT target_id, name, description, units FROM monitor_targets WHERE name = ?", $name);
    }

    if ( !$target->{target_id} ) {
        die "No such target: '$name'";
    }

    return $target;
}

=head2 get_values_for_interval

    Получить сумму всех сохраненных значений за прошедший $interval времени по указанным таргетам

    Все параметры именованные.
        interval    - интервал времени, за который нужно извлечь значения в формате, который поддерживает Yandex::DateTime::duration. Обязательный

    Интересующие таргеты указываются одним из трех параметров:
        targets     - ссылка на массив имен таргетов
        target      - имя таргета
        targets_like - LIKE паттерн, по которому будут извлекаться таргеты. За экранирование % и _ ответственнен вызывающий

    Возвращает ссылку на хэш, где для каждого таргета указана сумма.

    perl -Mmy_inc='..' -MMonitor -MDDP -e 'my $v = Monitor::get_values_for_interval(targets_like => "moderate\\_results.%", interval => "5min"); p $v'

=cut
sub get_values_for_interval {
    my (%O) = @_;

    my $interval = $O{interval};
    unless ($interval) {
        die "need interval";
    }
    my $duration = duration($interval);
    my $max_duration = duration($MAX_STAT_DAYS.'d');
    if (DateTime::Duration->compare($duration, $max_duration) > 0) {
        die "interval is too big. Monitor doesn't store stats longer than $MAX_STAT_DAYS days";
    }

    if ((grep { defined $O{$_} } qw/target targets targets_like/) != 1) {
        die "must specify either of 'target', 'targets' and 'targets_like'";
    }
    my $now = now();
    my $then = $now - $duration;
    my $seconds = $now->subtract_datetime_absolute($then)->in_units('seconds');

    my %name_condition;
    if ($O{targets}) {
        $name_condition{'t.name'} = $O{targets};
    } elsif ($O{targets_like}) {
        $name_condition{'t.name__like'} = $O{targets_like};
    } elsif ($O{target}) {
        $name_condition{'t.name'} = $O{target};
    }

    my $values = get_hash_sql(MONITOR, ["
                                    SELECT name, sum(value) value
                                      FROM monitor_targets t
                                           JOIN monitor_values_sec s using (target_id)",
                                     WHERE => {
                                            %name_condition,
                                            measure_time__ge__dont_quote => "(NOW() - INTERVAL $seconds SECOND)",
                                        },
                                     "GROUP BY name"]
        );

    return $values;
}

=head2 get_last_target_time

    Получить время последней записи значений таргета $target_name

    Возвращает объект unix epoch с временем последней записи, или undef, если записей нет вообще.

    perl -Mmy_inc='..' -MMonitor -MYandex::TimeCommon -MDDP -e 'my $v = unix2human(Monitor::get_last_target_time("moderate_results.objects.processed")); p $v'

=cut
sub get_last_target_time {
    my ($target_name) = @_;

    my $target_id = get_one_field_sql(MONITOR, "SELECT target_id FROM monitor_targets WHERE name = ?", $target_name);
    if (!defined $target_id) {
        return undef;
    }

    my $last_time = get_one_field_sql(MONITOR, "
                                    SELECT UNIX_TIMESTAMP(MAX(measure_time))
                                      FROM monitor_values_sec
                                     WHERE target_id = ?",
                                     $target_id
        );

    return $last_time;
}

1;
