package Notification;

# $Id$

=head1 NAME

    Notification

=head1 DESCRIPTION

    Пакет для работы с уведомлениями

=cut

use Direct::Modern;

use Storable qw/dclone/;

use Settings;
use RBACDirect;
use RBACElementary;
use Rbac qw/:const/;
use PrimitivesIds;
use Primitives;
use EnvTools;
use Yandex::DBTools;
use Yandex::DBShards;
use NotificationTools;
use MailService;
use Yandex::HashUtils;
use Yandex::IDN qw/is_valid_email/;
use Yandex::I18n;
use Yandex::Trace;
use Yandex::Validate qw/is_valid_id/;
use EventLog;
use User;
use Client qw/get_client_data/;
use YandexOffice;
use AdGroupTools;
use CampaignTools qw/get_content_promotion_content_type/;
use Campaign::Types qw/camp_kind_in is_media_camp/;

use List::Util qw/max/;
use List::MoreUtils qw/uniq none any/;
use POSIX qw(strftime);

use Yandex::TimeCommon qw(human_date);

use base qw/Exporter/;
our @EXPORT = qw/
    add_notification
/;

my $default_notification_set = {
    active_orders_money_out             => {notify_proc => \&notification_active_orders_money_out, eventlog_proc => \&eventlog_active_orders_money_out}
    , active_orders_money_out_on_client => {notify_proc => \&notification_active_orders_money_out_on_client}
    , active_orders_money_warning       => {notify_proc => \&notification_active_orders_money_warning, eventlog_proc => \&eventlog_active_orders_money_warning}
    , auto_servicing                    => {notify_proc => \&notification_auto_servicing}
    , notify_order_money_in_wo_eventlog => {notify_proc => \&notification_notify_order_money_in}
    , notify_order_money_out_blocking   => {notify_proc => \&notification_notify_order_money_out_blocking}
    , mediaplan_client_accept           => {notify_proc => \&notification_mediaplan_client_accept}
    , mediaplan_ready                   => {notify_proc => \&notification_mediaplan_ready}
    , new_camp_info                     => {notify_proc => \&notification_new_camp_info}
    , reps_added_client_rep             => {notify_proc => \&notification_reps_added_client_rep}
    , reservicing_request               => {notify_proc => \&notification_reservicing_request}
    , servicing_request                 => {notify_proc => \&notification_servicing_request}
    , moderate_result                   => {notify_proc => \&notification_moderate_result, eventlog_proc => \&eventlog_moderate_result}
    , sms_moderate_result               => {notify_proc => \&notification_sms_moderate_result}
    , moderate_callout_result           => {notify_proc => \&notification_moderate_callout_result}
    , teamleader_change_client          => {notify_proc => \&notification_teamleader_change_client}
    , camp_finished_wo_eventlog         => {notify_proc => \&notification_camp_finished}
    , warn_place                        => {notify_proc => \&notification_warn_place, eventlog_proc => \&eventlog_warn_place}
    , moderate_send_warn                => {notify_proc => \&notification_moderate_send_warn}
    , offline_stat_ready                => {notify_proc => \&notification_offline_stat_ready}
    , autobudget_forecast_wrong         => {notify_proc => \&notification_autobudget_forecast_wrong}
    , optimization_request              => {notify_proc => \&notification_optimization_request}
    , optimization_result               => {notify_proc => \&notification_optimization_result}
    , mobile_content_monitoring         => {notify_proc => \&notification_mobile_content_monitoring}
    , make_supersubclient_to_agency     => {notify_proc => \&notification_make_supersubclient_to_agency}
    , pay_supersubclient_to_agency      => {notify_proc => \&notification_pay_supersubclient_to_agency}
    , paused_by_day_budget              => {notify_proc => \&notification_paused_by_day_budget, eventlog_proc => \&eventlog_paused_by_day_budget}
    , non_resident_pay_notice           => {notify_proc => \&notification_non_resident_pay_notice}
    , retargeting_goals_check           => {notify_proc => \&notification_retargeting_goals_check, eventlog_proc => \&eventlog_retargeting_goals_check}
    , currency_convert_finished         => {notify_proc => \&notification_currency_convert_finished, eventlog_proc => \&eventlog_currency_convert_finished}
    , client_potential_report_ready     => {notify_proc => \&notification_client_potential_report_ready}
    , stat_rollback                     => {notify_proc => \&notification_stat_rollback}
    , broad_match_uncheck               => {notify_proc => \&notification_broad_match_uncheck}
    , xls_image_upload_failure          => {notify_proc => \&notification_xls_image_upload_failure}
    , new_domains                       => {notify_proc => \&notification_about_new_domains}
    , prove_login                       => {notify_proc => \&prove_login}
    , force_currency_convert            => {notify_proc => \&notification_force_currency_convert}
    , camp_copy_report                  => {notify_proc => \&notification_camp_copy_report}
    , subclients_convert_finished       => {notify_proc => \&_notification_subclients_convert_finished}
    , bs_monitor_banner                 => {notify_proc => \&notification_bs_monitor_banner}
    , blocked_client                    => {notify_proc => \&notification_about_blocked_client}
    , auto_price_camp_queue_task_failed => {notify_proc => \&notification_auto_price_camp_queue_task_failed}
    , new_ya_agency_client              => {notify_proc => \&notification_new_ya_agency_client}
    , ya_agency_order_paid              => {notify_proc => \&notification_ya_agency_order_paid}
    , experiment_finished                => {notify_proc => \&notification_experiment_finished}


};

my $empty_notification_set = {};

my $wallet_notification_set = {
    active_orders_money_out_reminder  => {notify_proc => \&notification_active_orders_money_out_reminder}
    , paused_by_day_budget => {notify_proc => \&notification_paused_by_day_budget_wallet, eventlog_proc => \&eventlog_paused_by_day_budget}
    , autopay_error => {notify_proc => \&notification_autopay_error}
    , active_orders_money_out => {notify_proc => \&notification_active_orders_money_out, eventlog_proc => \&eventlog_wallet_money_out}
    , active_orders_money_warning => {notify_proc => \&notification_active_orders_money_warning, eventlog_proc => \&eventlog_wallet_money_warning}
    , notify_order_money_in_wo_eventlog => {notify_proc => \&notification_notify_order_money_in}
    , active_orders_money_out_with_auto_overdraft => {notify_proc => \&notification_active_orders_money_out_with_auto_overdraft, eventlog_proc => \&eventlog_wallet_money_out_with_auto_overdraft}
};

my $touch_notification_set = {
    moderate_result => { notify_proc => \&notification_moderate_result_for_touch, eventlog_proc => \&eventlog_moderate_result }
    , sms_moderate_result => { notify_proc => \&notification_sms_moderate_result_for_touch }
    , active_orders_money_out => { notify_proc => \&notification_active_orders_money_out_for_touch, eventlog_proc => \&eventlog_wallet_money_out }
    , active_orders_money_out_reminder => { notify_proc => \&notification_active_orders_money_out_for_touch }
    , active_orders_money_warning => { notify_proc => \&notification_active_orders_money_warning_for_touch, eventlog_proc => \&eventlog_wallet_money_warning }
};

my $DEFAULT_NOTIFICATION_SET_NAME = 'default';
my $WALLET_NOTIFICATION_SET_NAME = 'wallet';
my $SUBJECT_NOTIFICATION_SET_NAME = 'subject';
my $TOUCH_NOTIFICATION_SET_NAME = 'touch';
my $GEO_NOTIFICATION_SET_NAME = 'geo';
my $API_ONLY_CONTENT_PROMOTION_NOTIFICATION_SET_NAME = 'api_only_promotion';
my %API_ONLY_CONTENT_PROMOTION_TYPES = (
    services => 1,
    eda => 1
);

our %NOTIFICATION_SET = (
    # NB! При добавлении нового шаблона для sms надо не забыть его добавить в MailService::_tt_name_to_event_name
    $DEFAULT_NOTIFICATION_SET_NAME => $default_notification_set
    , $WALLET_NOTIFICATION_SET_NAME => $wallet_notification_set
    , $SUBJECT_NOTIFICATION_SET_NAME => $empty_notification_set
    , $TOUCH_NOTIFICATION_SET_NAME => $touch_notification_set
    , $GEO_NOTIFICATION_SET_NAME => $empty_notification_set
    , $API_ONLY_CONTENT_PROMOTION_NOTIFICATION_SET_NAME => $empty_notification_set
);


=head3 @get_manager_info_of_cid_fields_used_in_crm

    Пересечение списка полей, которые заполняет в $vars функция get_manager_info_of_cid,
    и %CRM::VALID_CRM_FIELDS (полей, которые могут быть отправлены в CRM)

=cut
my @get_manager_info_of_cid_fields_used_in_crm = qw/
    campaign_id
    campaign_name
    campaign_type
    client_id
    client_login
    manager_uid
/;

#-----------------------------------------------------------
=head2 add_notification

    Добавление новой нотификации. Для правильного заполениня футера письма в переданных полях должен присутствовать cid или ClientID клиента.
    Если в нотификации отсутсвуют эти поля, то в $vars должна быть подготовленная структура с данными про агентство/менеджера/страну клиента
    (см. описание get_agency_manager_info_of_clientid).

    NB! Если отправляем нотификацию агентству, то в $vars нужно устанавливать флаг is_agency_notification, иначе получим неправильный футер.

=cut
sub add_notification
{
    my (undef, $notification_name, $vars, $options) = @_;
    my $profile = Yandex::Trace::new_profile("add_notification", tags => $notification_name);

    $vars = dclone($vars);

    $options //= {};
    die "incorrect \$options parameter: $options" if ref($options) ne 'HASH';

    if ($vars->{cid}) {
        hash_merge $vars, _camp_add_data(undef, $vars->{cid});
    } elsif ($vars->{campaign}->{cid}) { # paused_by_day_budget и camp_finished
        hash_merge $vars, _camp_add_data(undef, $vars->{campaign}->{cid});
    } elsif ($vars->{ClientID}) {
        # добавляем данные про агенство/менеджера/страну клиента для заполнения футера нотификации
        hash_merge $vars, get_agency_manager_info_of_clientid(undef, $vars->{ClientID}, 'text', hash_cut($options, qw/init_manager_uid init_ag_manager_uid/));
    }
    $vars->{client_id} //= $vars->{ClientID};

    my $notification_set_name = get_notification_set_name($vars);
    my $notification_set = $NOTIFICATION_SET{$notification_set_name};

    if (!$notification_set) {
        die "$notification_set_name notification set name not found\n";
    }

    if (exists $notification_set->{$notification_name}) {

        my @proctypes = ();
        unless (is_sandbox()) {
            push @proctypes, 'notify_proc';
        }
        unless ($vars->{campaign_type} && $vars->{campaign_type} eq 'mcb') {
            push @proctypes, 'eventlog_proc';
        }

        for my $proc_type (@proctypes) {
            $notification_set->{$notification_name}->{$proc_type}(undef, $vars, $options) if $notification_set->{$notification_name}->{$proc_type};
        }

    }
}

=head2 get_notification_set_name

    По vars из add_notification получает название продукта, по которому будет отправляться нотификация.
    От продукта зависит возможность отправки и текст нотификаций.

=cut
sub get_notification_set_name {
    my ($vars) = @_;

    # subject campaign
    if ($vars->{master_cid}) {
        return $SUBJECT_NOTIFICATION_SET_NAME;
    }

    # geo campaign
    if ($vars->{campaign_type} && $vars->{campaign_type} eq 'geo') {
        return $GEO_NOTIFICATION_SET_NAME;
    }

    # content promotion service campaign
    if ($vars->{campaign_type} && $vars->{campaign_type} eq 'content_promotion' && $vars->{cid}
        && $API_ONLY_CONTENT_PROMOTION_TYPES{get_content_promotion_content_type($vars->{cid})}) {
        return $API_ONLY_CONTENT_PROMOTION_NOTIFICATION_SET_NAME;
    }

    # touch campaign or touch client's wallet
    if ($vars->{is_touch_camp}
        || ($vars->{campaign_type} && $vars->{campaign_type} eq 'wallet' && $vars->{is_touch_client})) {
        return $TOUCH_NOTIFICATION_SET_NAME;
    }

    # wallet
    if ($vars->{campaign_type} && $vars->{campaign_type} eq 'wallet') {
        return $WALLET_NOTIFICATION_SET_NAME;
    }

    # all the rest
    return $DEFAULT_NOTIFICATION_SET_NAME;
}

#-----------------------------------------------------------
# notifications
#-----------------------------------------------------------
sub notification_active_orders_money_out
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    if (camp_kind_in(type => $vars->{campaign_type}, 'money_transfer')) {

        my $mail_template = 'active_orders_money_out';
        my $sms_template = Client::ClientFeatures::has_feature($vars->{client_id}, 'use_dynamic_threshold_for_send_order_warnings') ?
            'active_orders_money_out_campaign_stopped_sms' : 'active_orders_money_out_sms';

        if($vars->{client_email}){
            send_prepared_mail( $mail_template, $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name});
        }

        send_prepared_sms($sms_template, $vars);

        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('active_orders_money_out_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
        }
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        send_prepared_mail('active_orders_money_out_to_manager', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyDirectLetters};

    } else {

        if($vars->{client_email}){
            send_prepared_mail(
                'active_orders_money_out_mcb',
                get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid}),
                $Settings::NOTIFICATION_EMAIL_FROM,
                $vars,
                'fio');
        }

        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('active_orders_money_out_to_manager_mcb', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
        }
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        send_prepared_mail('active_orders_money_out_to_manager_mcb', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyMcbLetters};
    }
}

=head2 notification_active_orders_money_out_for_touch

    Тачёвый Директ. Нотификация пользователю о том, что у него закончились деньги

=cut

sub notification_active_orders_money_out_for_touch {
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    if ($vars->{client_email}) {
        send_prepared_mail('active_orders_money_out_touch', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name});
    }

    my $sms_template = Client::ClientFeatures::has_feature($vars->{client_id}, 'use_dynamic_threshold_for_send_order_warnings') ?
        'active_orders_money_out_campaign_stopped_sms' : 'active_orders_money_out_touch_sms';

    send_prepared_sms($sms_template, $vars);
}

=head2 notification_active_orders_money_out_with_auto_overdraft

    Нотификация пользователя о том, что средства на кошельке исчерпаны полностью,
    даже с учётом подключенного порога (автоовердрафта), и показы остановлены.

=cut
sub notification_active_orders_money_out_with_auto_overdraft {
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    my $mail_template = 'active_orders_money_out_with_auto_overdraft';
    my $sms_template = 'active_orders_money_out_with_auto_overdraft_sms';

    if ($vars->{client_email}) {
        send_prepared_mail(
            $mail_template, $email_additional_options,
            $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}
        );
    }
    send_prepared_sms($sms_template, $vars);

    my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
    if (!$man_info->{manager_use_crm} && $man_info->{email}) {
        send_prepared_mail('active_orders_money_out_with_auto_overdraft_to_manager',
            $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }
}

=head2 notification_active_orders_money_out_reminder

Нотификация пользователю о том, что несколько дней назад у него закончились деньги на

=cut
sub notification_active_orders_money_out_reminder {
    my (undef, $vars, $options) = @_;
    my $email_additional_options = Notification::get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
    my $mail_template = 'active_orders_money_out_reminder';
    my $sms_template = 'active_orders_money_out_reminder_sms';
    send_prepared_mail($mail_template, $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name});
    send_prepared_sms($sms_template, $vars);
}

=head2 eventlog_active_orders_money_out

    Записывает в event-лог событие "На кампании закончились деньги"
    $vars = {
        uid           => $uid,
        client_uid    => $uid,
        fio           => $fio,
        cid           => $cid,
        campaign_id   => $cid,
        camp_name     => $name,
        campaign_name => $name,
        client_login  => $client_login,
        client_email  => $email,
        client_fio    => $fio,
        client_phone  => $client_phone,
        client_id     => $ClientID,
        campaign_type => ($mediaType || 'text'),
    }

=cut

sub eventlog_active_orders_money_out {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(slug => 'money_out', ClientID => $vars->{client_id}, cid => $vars->{campaign_id});
}

sub eventlog_wallet_money_out {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(slug => 'money_out_wallet', ClientID => $vars->{client_id}, cid => $vars->{campaign_id});
}

=head2 eventlog_wallet_money_out_with_auto_overdraft

    Записывает в event-лог событие "Достигнут порог отключения"

=cut
sub eventlog_wallet_money_out_with_auto_overdraft {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(slug => 'money_out_wallet_with_ao', ClientID => $vars->{client_id}, cid => $vars->{campaign_id});
}

#-----------------------------------------------------------
sub notification_active_orders_money_out_on_client
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info($vars->{manager_uid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('active_orders_money_out_on_client_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
sub notification_active_orders_money_warning
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    if (camp_kind_in(type => $vars->{campaign_type}, 'money_transfer')) {

        my $mail_template = 'active_orders_money_warning';
        my $sms_template = 'active_orders_money_warning_sms';

        send_prepared_mail($mail_template, $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{client_email};
        send_prepared_sms($sms_template, $vars);

        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('active_orders_money_warning_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
        }
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        send_prepared_mail('active_orders_money_warning_to_manager', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyDirectLetters};

    } else {

        send_prepared_mail('active_orders_money_warning_mcb', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 'fio') if $vars->{client_email};

        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('active_orders_money_warning_to_manager_mcb', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
        }
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        send_prepared_mail('active_orders_money_warning_to_manager_mcb', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyMcbLetters};
    }
}

=head2 notification_active_orders_money_warning_for_touch

    Тачёвый Директ. Нотификация пользователю о том, что у него скоро закончаться деньги

=cut
sub notification_active_orders_money_warning_for_touch {
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    send_prepared_mail('active_orders_money_warning_touch', $email_additional_options,
        $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{client_email};
    send_prepared_sms('active_orders_money_warning_touch_sms', $vars);
}

=head2 eventlog_active_orders_money_warning

    Записывает в event-лог событие "На кампании израсходовано 80% средств"
    $vars = {
        fio           => $fio,
        cid           => $cid,
        uid           => $uid,
        client_uid    => $uid,
        campaign_id   => $cid,
        procent       => $money_warning_value,
        camp_name     => $name,
        client_login  => $client_login,
        client_email  => $email,
        client_fio    => $fio,
        client_phone  => $client_phone,
        client_id     => $ClientID,
        campaign_type => ($mediaType || 'text'),
        rest          => sprintf("%1.02f", $rest),
        sum_last      => sprintf("%1.02f", $sum_last),
        currency      => $currency,
    }

=cut

sub eventlog_active_orders_money_warning {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(slug => 'money_warning', ClientID => $vars->{client_id}, cid => $vars->{campaign_id}, params => {sum_rest => $vars->{rest}, currency => $vars->{currency}});
}

sub eventlog_wallet_money_warning {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(slug => 'money_warning_wallet', ClientID => $vars->{client_id}, cid => $vars->{campaign_id}, params => {sum_rest => $vars->{rest}, currency => $vars->{currency}});
}

#-----------------------------------------------------------
sub notification_notify_order_money_in
{
    my (undef, $vars, $options) = @_;

    $vars->{start_time} = human_date($vars->{start_time_ts});
    $vars->{start_date} = strftime('%Y-%m-%dT%T', localtime($vars->{start_time_ts})); # ISO8601
    $vars->{campaign_name} = $vars->{camp_name};

    if (camp_kind_in(type => $vars->{campaign_type}, 'money_transfer')) {
        if (! $vars->{wallet_cid}) {
            # не шлем уведомление по кампании если на ней подключен общий счет
            my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
            send_prepared_mail('notify_order_money_in', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{client_email};
            send_prepared_sms('notify_order_money_in_sms', $vars);

            my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
            if (!$man_info->{manager_use_crm} && $man_info->{email}) {
                send_prepared_mail('notify_order_money_in_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
            }

            my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
            send_prepared_mail('notify_order_money_in_to_manager', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars)
                if $man_of_ag_info->{email}
                   && $man_of_ag_info->{sendAgencyDirectLetters};

            # to agency
            if ($vars->{agency_email}) {
                my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{agency_uid});
                send_prepared_mail('notify_order_money_in_to_agency', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});

                # посылаем копию письма представителю-шефу если у данного представителя стоит галка "отсылать копии шефу"
                my $agency_info = get_agency_info($vars->{agency_uid});
                if ($agency_info->{agency_email_to_chief}) {
                    my $agency_chief_info = get_agency_info(rbac_get_chief_rep_of_agency_rep($vars->{agency_uid}));
                    if ($agency_chief_info->{email} && lc($agency_chief_info->{email}) ne lc($vars->{agency_email})) {
                        my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $agency_chief_info->{uid});
                        send_prepared_mail('notify_order_money_in_to_agency', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
                    }
                }
            }
        }
    } else {

        my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
        send_prepared_mail('notify_order_money_in_mcb', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{client_email};

        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('notify_order_money_in_to_manager_mcb', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
        }
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        send_prepared_mail('notify_order_money_in_to_manager_mcb', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyMcbLetters};

        # to agency
        if ($vars->{agency_email}) {
            my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{agency_uid});
            send_prepared_mail('notify_order_money_in_to_agency_mcb', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1}) if $vars->{agency_email};

            # посылаем копию письма представителю-шефу если у данного представителя стоит галка "отсылать копии шефу"
            my $agency_info = get_agency_info($vars->{agency_uid});
            if ($agency_info->{agency_email_to_chief}) {
                my $agency_chief_info = get_agency_info(rbac_get_chief_rep_of_agency_rep($vars->{agency_uid}));
                if ($agency_chief_info->{email} && lc($agency_chief_info->{email}) ne lc($vars->{agency_email})) {
                    send_prepared_mail('notify_order_money_in_to_agency_mcb', $agency_chief_info->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
                }
            }
        }
    }
}

#-----------------------------------------------------------
sub notification_notify_order_money_out_blocking
{
    # используется из java NotifyOrder
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    if (camp_kind_in(type => $vars->{campaign_type}, 'under_wallet')) {
        send_prepared_mail('notify_order_money_out_blocking', $email_additional_options,
            $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{client_uid} || $vars->{client_email};
    }
}

#-----------------------------------------------------------
sub notification_new_camp_info
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info($vars->{manager_uid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('new_camp_info_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
# добавление нового представителя (письмо клиенту или менеджеру)
#
# в письме клиенту убрали ФИО из шаблона письма по причине абьюза спамерами
# https://st.yandex-team.ru/DIRECT-84359
# https://st.yandex-team.ru/DIRECT-86394
sub notification_reps_added_client_rep
{
    my (undef, $vars, $options) = @_;

    if($vars->{manager_uid}){
        my $man_info = get_manager_info($vars->{manager_uid}, $vars);
        if (!$man_info->{manager_use_crm}) {
            send_prepared_mail('reps_added_client_rep_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        }
    } else {

        my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
        send_prepared_mail('reps_added_client_rep', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }
}

#-----------------------------------------------------------
sub notification_servicing_request
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info($vars->{manager_uid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('servicing_request_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

sub notification_auto_servicing
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('servicing_notification_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
sub notification_reservicing_request
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info($vars->{manager_uid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('reservicing_request_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
sub notification_mediaplan_ready
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
    my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);

    if ($vars->{accept_type} eq 'replace' && !$man_info->{manager_uid} && !$vars->{AgencyID}) {
        send_prepared_mail('mediaplan_replace_ready_to_client', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $vars->{email};
    }
    else {
        send_prepared_mail('mediaplan_ready_to_client', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $vars->{email};
    }
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('mediaplan_ready_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
sub notification_mediaplan_client_accept
{
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('mediaplan_client_accept_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars) if $man_info->{email};
    }
}

#-----------------------------------------------------------
sub notification_moderate_result
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{uid});
    if (! $vars->{media}) {
        # баннеры успели удалить, ничего не отсылаем
        return if $vars->{banners_count} == 0;

        if ($vars->{stat}{postmoderated} && !($vars->{stat}{declined} || $vars->{stat}{declined_partly})) {
            # если есть объявления на ПОСТмодерации - отправляем отдельным письмом
            AdGroupTools::separate_adgroups_from_banners_struct(
                $vars
                , extract_bool_flags => [qw/is_accepted is_declined_partly is_declined/]
            );
            send_prepared_mail('avtomoderation_result', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{uid};

        } elsif ($vars->{stat}{accepted} || $vars->{stat}{declined} || $vars->{stat}{declined_partly} || $vars->{stat}{disabled_geo}) {
            # если email адрес провалидирован - то отправляем письмо клиенту
            my $vars_copy = {%$vars, moderate_declined => $vars->{stat}{declined_partly} ? 'частично' : 'полностью'};
            AdGroupTools::separate_adgroups_from_banners_struct(
                $vars
                , extract_bool_flags => [qw/is_accepted is_declined_partly is_declined/]
            );
            send_prepared_mail('moderate_result', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{uid};

            # отправляем письмо менеджеру
            my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
            if (!$man_info->{manager_use_crm} && $man_info->{email}) {
                my $vars_copy = {%$vars, is_for_manager => 1};
                AdGroupTools::separate_adgroups_from_banners_struct(
                    $vars_copy
                    , extract_bool_flags => [qw/is_accepted is_declined_partly is_declined/]
                );
                send_prepared_mail('moderate_result', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars_copy);
            }

            # отправляем письмо менеджеру агентства если нужно
            my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
            if ($man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyDirectLetters}) {
                my $vars_copy = {%$vars, is_for_manager => 1};
                send_prepared_mail('moderate_result', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars_copy);
            }
        }

    } else {
        send_prepared_mail('moderate_result_media', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{uid};

        # отправляем письмо менеджеру
        my $man_info = get_manager_info_of_cid($vars->{cid}, $vars);
        if (!$man_info->{manager_use_crm} && $man_info->{email}) {

            my $vars_copy = {%$vars, is_for_manager => 1};
            send_prepared_mail('moderate_result_media_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars_copy);

        }

        # отправляем письмо менеджеру агентства если нужно
        my $man_of_ag_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
        if ($man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyMcbLetters}) {
            my $vars_copy = {%$vars, is_for_manager => 1};
            send_prepared_mail('moderate_result_media_to_manager', $man_of_ag_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars_copy);
        }
    }
}

=head2 notification_moderate_result_for_touch

    Тачёвый Директ. Нотификация пользователю о результатах модерации

=cut
sub notification_moderate_result_for_touch {
    my (undef, $vars, $options) = @_;

    # баннеры успели удалить, ничего не отсылаем
    return if $vars->{banners_count} == 0;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{uid});

    my $vars_copy = { %$vars, moderate_declined => $vars->{stat}{declined_partly} ? 'частично' : 'полностью' };
    AdGroupTools::separate_adgroups_from_banners_struct(
        $vars
        , extract_bool_flags => [ qw/is_accepted is_declined_partly is_declined/ ]
    );
    send_prepared_mail('moderate_result_touch', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name}) if $vars->{uid};
}

=head2 notification_sms_moderate_result

Нотификация пользователю о результатах модерации по смс

=cut
sub notification_sms_moderate_result {
    my (undef, $vars, $options) = @_;
    send_prepared_sms('moderate_result_sms', $vars);
}

=head2 notification_sms_moderate_result_for_touch

    Тачёвый Директ. Нотификация пользователю о результатах модерации по смс

=cut
sub notification_sms_moderate_result_for_touch {
    my (undef, $vars, $options) = @_;
    send_prepared_sms('moderate_result_touch_sms', $vars);
}

=head2 eventlog_moderate_result

    Записывает в event-лог событие "Результат модерации"

=cut

sub eventlog_moderate_result {
    my (undef, $vars, $options) = @_;

    return unless $vars && !$vars->{media} && $vars->{banners_count} > 0;

    # в notification_moderation_result баннеры делятся на два вида: групповые ложатся в adgroups->banners, безгрупповые остаются в banners
    my $banners = [];
    if (exists $vars->{adgroups} && ref($vars->{adgroups}) eq 'ARRAY') {
        for my $adgroup (@{ $vars->{adgroups} }) {
            if ($adgroup && ref($adgroup) eq 'HASH' && exists $adgroup->{banners} && ref($adgroup->{banners}) eq 'ARRAY') {
                push @$banners, @{ $adgroup->{banners} };
            }
        }
    }
    if (exists $vars->{banners} && ref($vars->{banners}) eq 'ARRAY') {
        push @$banners, @{ $vars->{banners} };
    }

    if ($banners && ref($banners) eq 'ARRAY' && @$banners) {
        my @events;
        for my $banner (@$banners) {
            my %moderation_results;
            if ($banner->{is_declined}) {
                $moderation_results{global} = 'declined';
            } elsif ($banner->{is_declined_partly}) {
                $moderation_results{global} = 'declined_partly';
            } elsif ($banner->{is_accepted}) {
                $moderation_results{global} = 'accepted';
            }
            $moderation_results{text} = $banner->{statusModerate} if $banner->{statusModerate};
            for my $type(qw/phrases contactinfo sitelinks_set/) {
                $moderation_results{$type} = $banner->{$type}{statusModerate} if $banner->{$type} && ref($banner->{$type}) eq 'HASH' && $banner->{$type}{statusModerate};
            }
            $moderation_results{geo_exception} = [ keys %{$banner->{geo_exception}} ] if $banner->{geo_exception} && ref($banner->{geo_exception}) eq 'HASH';

            my $params = {
                results => \%moderation_results,
                is_edited_by_moderator => $banner->{is_edited_by_moderator} ? 1 : 0,
            };

            push @events, {slug => 'banner_moderated', ClientID => $vars->{ClientID}, cid => $banner->{cid}, bid => $banner->{bid}, params => $params };
        }
        EventLog::log_several_events(\@events) if @events;
    }
}

=head2 notification_moderate_callout_result

    уведомление клиенту о результатах модерации уточнений

=cut

sub notification_moderate_callout_result
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(uid => $vars->{uid});

    if ($vars->{uid}) {
        # отправляем письмо клиенту
        send_prepared_mail('moderate_callout_result', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name} || 'fio');

        # отправляем письма менеджерам
        my $managers_info = get_managers_info_of_client(undef, $vars->{uid});
        for my $man_info (@$managers_info) {
            if (!$man_info->{manager_use_crm} && $man_info->{email}) {
                send_prepared_mail('moderate_callout_result', $man_info->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name} || 'fio');
            }
        }
        # отправляем письма менеджерам агентств если нужно
        my $managers_of_ag_info = get_managers_info_of_agencies_of_client(undef, $vars->{uid});
        for my $man_of_ag_info (@$managers_of_ag_info) {
            next unless $man_of_ag_info->{email} && $man_of_ag_info->{sendAgencyDirectLetters};
            send_prepared_mail('moderate_callout_result', $man_of_ag_info->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1}, $options->{mail_fio_tt_name} || 'fio');
        }
    }
}

#-----------------------------------------------------------
# напоминание пользователю о принятии кампании на модерации
sub notification_moderate_send_warn
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{uid});

    send_prepared_mail('moderate_send_warn', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

#-----------------------------------------------------------
# напоминание пользователю о принятии кампании на модерации
sub notification_offline_stat_ready
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{uid});

    send_prepared_mail('offline_stat_ready', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

#-----------------------------------------------------------
# Письмо о несоблюдении автобюджета
sub notification_autobudget_forecast_wrong
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{AgencyUID});

    send_prepared_mail('autobudget_forecast_wrong', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

#-----------------------------------------------------------
# ответ клиенту на заказ услуги "Первой помощи"
sub notification_optimization_request
{
    my (undef, $vars, $options) = @_;

    # DIRECT-64918: письма first_aid не шлём
    return if !$vars->{is_second_aid};

    my $prefix =  ($vars->{is_second_aid}) ? 'second_aid_' : 'first_aid_';
    my $tt_name = (defined $vars->{comment}) ? 'request_declined_to_client' : 'request_to_client';

    send_prepared_mail($prefix.$tt_name, $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

#-----------------------------------------------------------
# [повторное] уведомление пользователю о готовности заказа "Первой помощи"
sub notification_optimization_result
{
    my (undef, $vars, $options) = @_;

    if (!$vars->{is_second_aid}) {
        my $tt_name = ($vars->{daystogo} =~ /^(3|14)$/) ? 'first_aid_result_to_client_repeat'
                                                        : 'first_aid_result_to_client';
        send_prepared_mail($tt_name, $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }
}

#-----------------------------------------------------------

=head2 notification_bs_monitor_banner

    Отправляет письмо и смс (если нужно) об остановке/возобновлении показов мониторингом доступности доменов.

    vars: {
        uid =>  ...,            # кому отправлять письмо/смс
        ClientID => ...,        # для футера
        state => "dead/alive",  # уведомление об остановке или возобновлении
        domain_list => [...],   # список доменов о которых уведомляем
        camp_list => [          # список кампаний, о которых уведомляем
          {
            cid => ...,
            name => ...,        # название кампании
          },
          ...
        ],
    }

=cut

sub notification_bs_monitor_banner
{
    my (undef, $vars, $options) = @_;

    send_prepared_mail("bs_monitor_banner", $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 1);
    if ($options->{need_sms}) {
        # отправляем круглосуточно
        my $sms_time = "00:00:00:00";
        send_prepared_sms('bs_monitor_banner_sms', {%$vars, sms_time => $sms_time}, dont_use_cid => 1);
    }
}


=head2 notification_mobile_content_monitoring

Уведомление про остановку/запуск показов мобильных объявлений
из-за недоступности приложений в сторах.

    $vars = {
        uid => ...,
        FIO => ...,
        state => ...,   # alive / dead
        changes => {
            cid => ...,
            store_content_href => ...,
        },
    };

=cut

sub notification_mobile_content_monitoring
{
    my (undef, $vars, $options) = @_;

    send_prepared_mail('mobile_content_monitoring', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

#-----------------------------------------------------------
#
sub notification_make_supersubclient_to_agency
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{agency_uid});
    send_prepared_mail('make_supersubclient_to_agency', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
}

#-----------------------------------------------------------
#
sub notification_pay_supersubclient_to_agency
{
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{agency_uid});
    send_prepared_mail('pay_supersubclient_to_agency', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
}

#-----------------------------------------------------------
# тимлидер:
#   добавил/удалил/сменил шефа у клиента либо у агентства
#   объединил клиентов
#   снял ограничение на первый платеж
sub notification_teamleader_change_client
{
    my (undef, $vars, $options) = @_;

    if ($vars->{teamleader_uid}) {

        my $team_info = get_manager_info($vars->{teamleader_uid}, $vars);

        if (! $team_info->{manager_use_crm}) {
            send_prepared_mail('teamleader_change_client', $vars->{teamleader_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);

            # если у тимлидера есть супер-тимлидер посылаем ему копию
            my $super_teamleader_uid = rbac_get_superteamleader_of_team(undef, $vars->{teamleader_uid});
            if ($super_teamleader_uid) {

                my $super_team_info = get_manager_info($super_teamleader_uid, $vars);

                if (! $super_team_info->{manager_use_crm}) {
                    $vars->{for_superteamleader} = 1;
                    $vars->{teamleader_login} = $team_info->{login};
                    send_prepared_mail('teamleader_change_client', $super_teamleader_uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
                }
            }
        }
    }
}


=head2 notification_camp_finished

    Отправляем почтой и SMSкой уведомление клиенту и менеджеру о кампании, остановившейся по достижению даты окончания.

    notification_camp_finished(undef, $vars, $options);
    $vars = {
        campaign => {
            cid => 123,                    # cid кампании
            finish_date => '2011-06-17',   # дата окончания кампании (в MySQL-совместимом формате)
            name => 'Новая кампания',      # название кампании
            ClientID => 123456,            # ClientID клиента, которому принадлежит кампания
        },
        user => {
            uid => 456,                    # uid главного представителя клиента
            email => 'devnull@yandex.ru',  # email, на который посылать уведомление
            FIO => 'Иванов Иван Иванович', # ФИО главного представителя клиента
            login => 'ivanovii',           # логин главного представителя клиента
            phone => '+79161234567',                # телефон главного представителя клиента
            lang => 'ru',                  # язык, на котором надо отправлять пользователю почту и SMS
        },
    };

=cut

sub notification_camp_finished
{
    my (undef, $vars, $options) = @_;

    # эти ключи используются в MailService::send_prepared_mail и MailService::send_prepared_sms
    hash_merge $vars, {
        cid => $vars->{campaign}->{cid},
        uid => $vars->{user}->{uid},
        fio => $vars->{user}->{FIO},
    };

    # уведомления клиенту
    my $user_email = $vars->{user}->{email};
    if ( $user_email && is_valid_email($user_email) ) {
        my $key_name_for_fio_in_var = 'fio';
        my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{user}->{uid});
        send_prepared_mail('camp_finished', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM,
            $vars, $key_name_for_fio_in_var);
    }
    send_prepared_sms('camp_finished_sms', $vars);

    # уведомление менеджеру (только если НЕ используется CRM)
    my $man_info = get_manager_info_of_cid($vars->{cid}, $vars );
    my $man_email = $man_info->{email};
    if ( !$man_info->{manager_use_crm} && $man_email && is_valid_email($man_email) ) {
        send_prepared_mail('camp_finished_to_manager', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }

    # уведомление менеджеру агентства (тот же текст, что и менеджеру)
    my $man_of_agency_info = get_manager_info_of_agency_of_cid(undef, $vars->{cid}, $vars);
    my $agency_email = $man_of_agency_info->{email};
    if ( $man_of_agency_info->{sendAgencyDirectLetters} && $agency_email && is_valid_email($agency_email) ) {
        send_prepared_mail('camp_finished_to_manager', $agency_email, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }

    # уведомление агентству (тот же текст, что и менеджеру)
    if ($vars->{campaign}{AgencyUID} && $vars->{campaign}{AgencyUID} > 0) {
        my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{campaign}{AgencyUID});
        send_prepared_mail('camp_finished_to_manager', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
    }
}

=head2 notification_warn_place

    Отправляет на email уведомление об изменении позиций фраз.
    $vars = {
        cid => $cid,
        client_uid => $CAMP{$cid}->{uid},
        client_fio => $CAMP{$cid}->{fio},
        fio => $CAMP{$cid}->{fio},
        camp_name => $CAMP{$cid}->{name},
        client_login => $CAMP{$cid}->{login},
        client_email => $CAMP{$cid}->{email},
        client_phone => $CAMP{$cid}->{phone},
        sendWarn => $CAMP{$cid}->{sendWarn},
        ClientID => $CAMP{$cid}->{ClientID},
        data => [
            {
                bid => $bid,
                banner_name => $banner_name,
                ph_gar => [ # фразы, вытесненные из гарантии
                    {
                        id => $id,
                        phrase => $phrase_text,
                    },
                    [...]
                ],
                ph_first => [{id => $id, phrase => $phrase_text}, [...]],
                ph_spec => [{id => $id, phrase => $phrase_text}, [...]],
                rub_spec => [{id => $id, phrase => $phrase_text}, [...]], # рубрики, вытесненные из спецразмещения
                rub_first => [{id => $id, phrase => $phrase_text}, [...]],
                rub_gar => [{id => $id, phrase => $phrase_text}, [...]],
            },
            [...]
        ],
    };

=cut

sub notification_warn_place {
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});

    send_prepared_mail('warn_place', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 'client_fio', undef, use_send_warn => 1);
}

=head2 eventlog_warn_place

    Записывает в event-лог событие "У фразы изменился объём трафика"
    $vars совпадает с таковым из notification_warn_place.

=cut

sub eventlog_warn_place {
    my (undef, $vars, $options) = @_;

    my @events;
    my @banners;
    push @banners, @{$vars->{banners}} if ref($vars->{banners}) eq 'ARRAY' && @{$vars->{banners}};
    push @banners, map {@{$_->{banners}}} @{$vars->{adgroups}} if ref($vars->{adgroups}) eq 'ARRAY' && @{$vars->{adgroups}};

    for my $banner (@banners) {
        for my $phrase_type(qw/ph rub/) {
            my %place_names = (first => 'FIRST', spec => 'PREMIUM', gar => 'GARANT');
            while(my($vars_name, $event_name) = each %place_names) {
                my $group_name = "${phrase_type}_$vars_name";
                if ($banner->{$group_name} && ref($banner->{$group_name}) eq 'ARRAY' && @{$banner->{$group_name}}) {
                    for my $phrase(@{$banner->{$group_name}}) {
                        push @events, {slug => 'warn_place', ClientID => $vars->{ClientID}, cid => $vars->{cid}, bid => $banner->{bid}, bids_id => $phrase->{id}, params => {old_place => $event_name}};
                    }
                }
            }
        }
    }
    EventLog::log_several_events(\@events) if @events;
}

=head2 notification_paused_by_day_budget

    Отправляем письмо об остановке кампаний по дневному бюджету.

    notification_paused_by_day_budget(undef, $vars, $options);
    $vars = {
        campaign => {
            cid => 123,                    # cid кампании
            name => 'Новая кампания',      # название кампании
            day_budget_stop_time => '201112061523',  # время остановки кампании по дневному бюджету (в MySQL-совместимом формате)
            ClientID => 232,               # id клиента
            login => 'login111',
        },
        chief_rep => {
            uid => 456,                    # UID главного представителя
        },
    };

=cut

sub notification_paused_by_day_budget {
    my (undef, $vars, $options) = @_;

    $vars->{cid} = $vars->{campaign}{cid}; # отправляем письмо также на email, указанный в кампании
    send_prepared_mail('paused_by_day_budget', $vars->{chief_rep}{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 1);
}

=head2 notification_paused_by_day_budget_wallet

    Отправляем письмо об остановке всех кампаний под кошельком по дневному бюджету.

    notification_paused_by_day_budget_wallet(undef, $vars, $options);
    $vars = {
        campaign => {
            cid => 123,                    # cid кампании
            day_budget_stop_time => '201112061523',  # время остановки кампании по дневному бюджету (в MySQL-совместимом формате)
            ClientID => 232,               # id клиента
            type => 'wallet',              # тип кампании
        },
        chief_rep => {
            uid => 456,                    # UID главного представителя
            login => 'login111',
        },
        sms_time => '09:00:21:00'
    };

=cut

sub notification_paused_by_day_budget_wallet {
    my (undef, $vars, $options) = @_;

    $vars->{login} //= $vars->{chief_rep}{login};
    $vars->{cid} = $vars->{campaign}{cid}; # отправляем письмо также на email, указанный в кошельке
    if ($options->{send_email}) {
        send_prepared_mail('paused_by_day_budget_wallet', $vars->{chief_rep}{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 1);
    }

    $vars->{uid} //= $vars->{chief_rep}{uid};
    send_prepared_sms('paused_by_day_budget_wallet_sms', $vars);
}

=head2 eventlog_paused_by_day_budget

    Записывает в event-лог событие "кампания остановилась по дневному бюджету"
    $vars совпадает с таковым из notification_paused_by_day_budget

=cut

sub eventlog_paused_by_day_budget {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(
        slug => (($vars->{campaign}->{type} // '') eq 'wallet' ? 'paused_by_day_budget_wallet' : 'paused_by_day_budget'),
        ClientID => $vars->{campaign}{ClientID},
        cid => $vars->{campaign}{cid},
        params => {bs_stop_time => $vars->{campaign}{day_budget_stop_time} },);
}

=head2 notification_non_resident_pay_notice

    Отправлям менеджеру письмо о запросе счёта нерезидентом

=cut

sub notification_non_resident_pay_notice {
    my (undef, $vars, $options) = @_;

    send_prepared_mail('non_resident_pay_notice', $vars->{manager}->{email}, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 1);
}

# --------------------------------------------------------------------

=head2 notification_retargeting_goals_check

Цель метрики которая используется в условии ретаргетинга, более не доступна клиенту

=cut

sub notification_retargeting_goals_check {
    my (undef, $vars, $options) = @_;

    $vars->{retargetings_count} = scalar(@{ $vars->{retargetings} });
    $vars->{retargetings_names} = join ", ", map {qq/"$_->{condition_name}"/} @{ $vars->{retargetings} };

    send_prepared_mail('retargeting_goals_check', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

# --------------------------------------------------------------------

=head2 eventlog_retargeting_goals_check

Цель метрики которая используется в условии ретаргетинга, более не доступна клиенту

=cut

sub eventlog_retargeting_goals_check {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(
        slug => 'retargeting_goals_check',
        ClientID => $vars->{ClientID},
        params => {retargetings => $vars->{retargetings}},
    );
}

=head2 notification_currency_convert_finished

    Отправляем письмо о завершении перехода в реальную валюту клиенту и его представителям, агентствам и менеджерам.

    Шлём на указанный на странице перевода адрес + на адрес главного представителя, если он отличается + представителям
    по общей логике (если включена галка получать уведомления по всем кампаниям).
    Если там не указали адрес, то на указанный в настройках пользователя для главного представителя + по общей логике для представителей.

    notification_currency_convert_finished(undef, $vars, $options);
    $vars = {
        email => 'text@example.org',    # email, введённый пользователем в форме заявки на переход
        ClientID => 123456,    # ClientID переходящего клиента
        uid => 987654,  # uid представителя, который оставлял заявку на переход и которому надо написать SMS, может отсутствовать
        client_chief_uid => 12345, # uid главного представителя
        convert_type => 'COPY'|'MODIFY',    # каким способом переводим клиента
        new_currency => 'RUB'|'USD'|'EUR'|...,    # в какую валюту производился переход
    };

=cut

sub notification_currency_convert_finished {
    my (undef, $vars, $options) = @_;

    my $client_id = $vars->{ClientID};
    my $client_chief_uid = $vars->{client_chief_uid};

    my $client_chief_data = get_user_info($client_chief_uid);
    $vars->{client_chief} = hash_cut $client_chief_data, qw/fio login email/;

    # письмо на email, указанный на странице перехода
    if ($vars->{email}) {
        my $client_chief_uid_data = get_user_info($client_chief_uid);
        if ($client_chief_uid_data && $client_chief_uid_data->{email} && $client_chief_uid_data->{email} ne $vars->{email}) {
            # email отличается, посылаем на него письмо
            # в письме указываем ФИО того, кто заказывал переход, т.к. email указывал именно он
            my $uid_data;
            if (!$vars->{uid} || $client_chief_uid == $vars->{uid}) {
                $uid_data = $client_chief_uid_data;
            } else {
                $uid_data = get_user_info($vars->{uid});
            }
            $vars->{fio} = $uid_data->{fio};
            send_prepared_mail('currency_convert_finished', $vars->{email}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        }
    }

    $vars->{fio} = $client_chief_data->{fio};

    # письма главному представителю + остальным представителям
    send_prepared_mail('currency_convert_finished', $client_chief_uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 1);

    # письмо всем представителям агентств, работающих с клиентом, у которых есть кампании для этого клиента
    my $agency_relation_data = get_agency_clients_relations_data($client_id);
    if ($agency_relation_data && @$agency_relation_data) {
        my @agency_clientids = map { $_->{agency_client_id} } grep { $_->{client_archived} && $_->{client_archived} eq 'No' } @$agency_relation_data;
        if (@agency_clientids) {
            for my $agency_client_id (@agency_clientids) {
                my $agency_uids = Rbac::get_reps(ClientID => $agency_client_id, role => $ROLE_AGENCY);
                if ($agency_uids && @$agency_uids) {
                    my $agency_uids_with_campaigns = get_one_column_sql(PPC(uid => $client_chief_uid), ['SELECT DISTINCT AgencyUID FROM campaigns', WHERE => {uid => $client_chief_uid, AgencyUID => $agency_uids}]) || [];
                    for my $agency_rep_uid (@$agency_uids_with_campaigns) {
                        my $email_additional_options = get_additional_email_params(uid => $agency_rep_uid);
                        send_prepared_mail('currency_convert_finished_manager_agency', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
                    }
                }
            }
        }
    }

    # письмо менеджерам клиента
    my $manager_uids = rbac_get_managers_of_client(undef, $vars->{uid});
    if ($manager_uids && @$manager_uids) {
        for my $manager_uid (@$manager_uids) {
            send_prepared_mail('currency_convert_finished_manager_agency', $manager_uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        }
    }
}

=head2 notification_client_potential_report_ready

    Отсылает письмо пользователю, заказавшему отчет по потенциалу клиента (обычно медиапланеру)

=cut

sub notification_client_potential_report_ready
{
    my (undef, $vars, $options) = @_;

    send_prepared_mail('client_potential_report_ready', $vars->{operator_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);

    # дополнительно отправляем уведомления по всем отчетам на rpk@yandex-team.ru
    local $Yandex::SendMail::ENVELOPE_SENDER = $Settings::NOTIFICATION_EMAIL_FROM;
    send_prepared_mail('client_potential_report_ready', 'rpk@yandex-team.ru', $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

=head2 notification_stat_rollback

    Отсылает письмо клиенту/менеджеру, об откате статистики / возврату средств

=cut
sub notification_stat_rollback {
    my (undef, $vars, $options) = @_;

    $vars->{cid} = $vars->{cids}->[0]->{cid};
    $vars->{cids_str} = join ', ', map { $_->{cid} } @{$vars->{cids}};
    $vars->{has_wallet} = any { $_->{wallet_cid} } @{$vars->{cids}};

    if ($vars->{type} eq 'mcb') {
        send_prepared_mail( 'stat_rollback_mcb', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    } else {
        if (!$vars->{for_manager_only}) {
            # отправляем письмо клиенту
            send_prepared_mail('stat_rollback', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        }

        # у профи-клиента могут быть менеджерские кампании (с разными менеджерами)
        my @managers = uniq( grep { $_ } map { $_->{ManagerUID} } @{ $vars->{cids} } );
        # поэтому отправляем копию уведомления каждому менеджеру
        foreach my $manager_uid (@managers) {
            my $man_info = get_manager_info($manager_uid, $vars);
            if (!$man_info->{manager_use_crm}) {
                send_prepared_mail('stat_rollback_to_manager', $manager_uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
            }
        }
    }
}

=head2 notification_xls_image_upload_failure

    Отсылает письмо оператору о том, что возникла ошибка при загрузке изображений через Excel.

=cut

sub notification_xls_image_upload_failure {
    my (undef, $vars) = @_;

    my $uid  = $vars->{uid};
    my $role = $vars->{role};

    my $email_additional_options = get_additional_email_params( uid => $vars->{uid}, cid => $vars->{cid} );

    if ( $role eq 'agency' ) {
        send_prepared_mail( 'xls_image_upload_failure_agency',
            $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 0 );
        return;
    }

    if ( $role eq 'client' ) {
        send_prepared_mail( 'xls_image_upload_failure_client',
            $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 0 );
        return;
    }

    if ( $role =~ /^(?:manager|media|support)$/ ) {
        send_prepared_mail( 'xls_image_upload_failure_manager',
            $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 0 );
        return;
    }

    if ( $role eq 'super' ) {
        # суперы писем не получают
        return;
    }

    # что с остальными ролями:
    # суперридеры - не могут загружать Excel-файлы
    # вешальщики - в ppcXLSImageUploadFailureNotification.pl есть логика, которая подставляет uid менеджера
    # если какая-то роль появится в будущем, её нужно будет учесть здесь
    die "Cannot email: unrecognised role $role for user $uid";
}

=head2 eventlog_currency_convert_finished

    Записывает в event-лог событие "завершён переход в реальную валюту"
    $vars совпадает с таковым из notification_currency_convert_finished

=cut

sub eventlog_currency_convert_finished {
    my (undef, $vars, $options) = @_;

    EventLog::log_event(
        slug => 'currency_convert_finished',
        ClientID => $vars->{ClientID},
        params => hash_cut $vars, qw/new_currency convert_type/,
    );
}

=head2 notification_broad_match_uncheck

    Отправляет письмо (или уведомление в CRM) менеджеру об отключении клиентом
        показов кампании по "дополнительным релевантным фразам"

=cut
sub notification_broad_match_uncheck {
    my (undef, $vars, $options) = @_;

    my $man_info = get_manager_info($vars->{ManagerUID}, $vars);
    if (!$man_info->{manager_use_crm}) {
        send_prepared_mail('broad_match_uncheck', $man_info->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    }
}

=head2 notification_about_new_domains

    Отправляет письмо представителю агентства о появлении новых доменов на аккаунтах субклиентов
    используется из java: JobSendWarnClientDomains

=cut
sub notification_about_new_domains {
    my (undef, $vars, $options) = @_;
    send_prepared_mail('new_domains_to_agency', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
}

=head2 prove_login

    Отправляет письмо клиенту о необходимости подтверждения логина при добавлении его в качестве субклиента
    или представителю агентства при подтверждении логина клиентом

=cut
sub prove_login {
    my (undef, $vars) = @_;

    hash_merge $vars, _get_agency_info_by_agency_client_id(undef, $vars->{agency_id}, 'text');
    if ( $vars->{client_email} ) {
        # если запрос от агенства, то отправляем письмо пользователю
        send_prepared_mail('prove_login_to_client', $vars->{client_email}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        return;
    }
    if ( $vars->{agency_uid} ) {
        # если запрос от клиента, то отправляем письмо представителю агентства
        send_prepared_mail('login_confirmed_to_agency', $vars->{agency_uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
        return;
    }

    die "Cannot email: no client_email or agency_uid";
}

=head2 notification_force_currency_convert

    Уведомление о дате принудительной конвертации в валюту

    $vars => {
        convert_date =>,
        currency =>,
        chief_login =>,
        chief_uid =>,
        ClientID =>,
    };

=cut

sub notification_force_currency_convert {
    my (undef, $vars, $options) = @_;

    my $email_additional_options = get_additional_email_params( uid => $vars->{chief_uid} );
    send_prepared_mail( 'force_currency_convert', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars );
}

=head2 notification_camp_copy_report

    Отправляет письмо c отчетом по скопированным кампаниям

    vars = {
        uid => <заказчик отчета>,
        client_id => <ClientID заказчика отчета>
        fio => <ФИО заказчика отчета>,
        new_login => <логин на который скопированны кампании>,
        old_login => <логин с которого скопированны кампании>,
        banners_report_name => <название отчета по баннерам>,
        phrases_report_name => <название отчета по фразам>,
        retargetings_report_name => <название отчета по ретаргетингам>,
        dynamic_conditions_report_name => <название отчета по динамическим условиям>,
        performance_filters_report_name => <название отчета по фильтрам смарт-баннерв>,
    };

=cut
sub notification_camp_copy_report {
    my (undef, $vars, $options) = @_;
    send_prepared_mail('camp_copy_report', $vars->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
}

=head2 _notification_subclients_convert_finished

    Отправляет письмо главному представителю агентства с результатом
    массовой конвертации субклиентов в валюту
    См. скрипт ppcSendSubclientsConvertedMails.pl

    $vars => {
        agency_chief => {
            uid => $agency_chief_uid,
            FIO => $agency_chief_fio,
        },
        converted_clients_cnt => scalar(@$converted_client_ids),
        unconverted_clients_cnt => scalar(@unconverted_clients),
        unconverted_clients => [
            {
                ClientID => $client_id1,
                chief_login => $chief_login1,
            },
            ...
        ],
    }

=cut

sub _notification_subclients_convert_finished {
    my (undef, $vars, $options) = @_;

    my $agency_chief_uid = $vars->{agency_chief}->{uid};

    my $mailvars = hash_cut $vars, qw(converted_clients_cnt unconverted_clients_cnt unconverted_clients);
    $mailvars->{FIO} = $vars->{agency_chief}->{FIO};
    hash_merge $mailvars, get_agency_manager_info_of_clientid(undef, $vars->{agency_id}, 'text'); # менеджера агентства всегда берем для Директа ('text')

    my $email_additional_options = get_additional_email_params(uid => $vars->{agency_chief}->{uid});
    send_prepared_mail('subclients_convert_finished', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, {%$mailvars, is_agency_notification => 1}, $options->{mail_fio_tt_name});
}

=head2 notification_autopay_error

    отправляет письмо клиенту об ошибке при пополнении кошелька в рамках автопополнения

=cut
sub notification_autopay_error {
    my (undef, $vars, $options) = @_;
    my $email_additional_options = Notification::get_additional_email_params(cid => $vars->{cid}, uid => $vars->{client_uid});
    my %template_by_letter_type = (
        'not_enough_funds_error' => 'autopay_error_not_enough_funds',
        'expired_card_error' => 'autopay_error_expired_card',
        'other_error' => 'autopay_error_other',
        );
    my $mail_template = $template_by_letter_type{$vars->{letter_type} // ''};
    if ($mail_template) {
        send_prepared_mail($mail_template, $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, $options->{mail_fio_tt_name});
        send_prepared_sms("${mail_template}_sms", $vars);
    }
}

=head2 notification_experiment_finished

письмо клиенту об окончании эксперимента

=cut

sub notification_experiment_finished {
    my (undef, $vars, $options) = @_;

    my $exp = $vars->{experiment};
    my $pcid = $exp->{primary_cid};
    my $scid = $exp->{secondary_cid};

    my $camps = get_hashes_hash_sql(PPC(ClientID => $exp->{ClientID}), [
            'SELECT cid, uid, c.name, u.FIO
            FROM campaigns c
            JOIN users u USING(uid)',
            WHERE => { cid => [$pcid, $scid] },
        ]);

    my $uid = $camps->{$pcid}->{uid};
    $vars->{fio} = $camps->{$pcid}->{FIO};
    $vars->{members} = {
        primary => $camps->{$pcid},
        secondary => $camps->{$scid},
    };

    send_prepared_mail('experiment_finished', $uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
    return;
}

=head2 notification_about_blocked_client

письмо агенству и/или менеджеру о блокировке пользователя

=cut

sub notification_about_blocked_client {
    my (undef, $vars, $options) = @_;
    return unless $vars->{is_agency_camp} || $vars->{is_manager_camp};

    hash_merge $vars, get_user_data(rbac_get_chief_rep_of_client($vars->{ClientID}), [qw/login fio/]);
    if ($vars->{is_agency_camp}) {
        send_prepared_mail('blocked_client_to_agency', $vars->{agency}->{chief_rep}->{uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_notification => 1});
        send_prepared_mail('blocked_client_to_manager', $vars->{ag_manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_camp => 0}) if $vars->{ag_manager_uid};
    } elsif ($vars->{is_manager_camp}) {
        send_prepared_mail('blocked_client_to_manager', $vars->{manager_uid}, $Settings::NOTIFICATION_EMAIL_FROM, {%$vars, is_agency_camp => 0}) if $vars->{manager_uid};
    }
}


=head2 notification_auto_price_camp_queue_task_failed

Нотификация пользователю о том, что не удалось сохранить ставки кампании, заданные с помощью мастера цен

=cut
sub notification_auto_price_camp_queue_task_failed {
    my (undef, $vars, $options) = @_;

    my $camp_info = $options->{camp_info};
    my $user_info = get_user_data($vars->{uid}, [qw/phone fio email/]);
    my $client_info = get_user_data($camp_info->{uid}, [qw/phone fio email/]);
    hash_merge $vars, $user_info, {
        campaign_name => $camp_info->{name},
        client_phone => $client_info->{phone},
        client_fio   => $client_info->{fio},
    };

    my $role = rbac_who_is(undef, $vars->{uid});
    $vars->{is_agency_notification} = 1 if $role eq 'agency';

    my $email_additional_options = Notification::get_additional_email_params(cid => $vars->{cid}, uid => $vars->{uid});
    send_prepared_mail('auto_price_camp_queue_task_failed', $email_additional_options, $Settings::NOTIFICATION_EMAIL_FROM, $vars, 0);
}

=head2 notification_new_ya_agency_client

Нотификация менеджерам о том, что еще один клиент заказал обслуживание Я.Агенством

=cut
sub notification_new_ya_agency_client {
    my (undef, $vars, $options) = @_;

    $vars->{product_name} //= $Settings::YaAgency_PRODUCTS->{$vars->{product_type}} // $vars->{product_type};
    $vars->{login} //= get_login(uid => rbac_get_chief_rep_of_client($vars->{client_id}));

    #Если задан email - отправляем на него уведомление клиенту
    if ($vars->{email}){
        send_prepared_mail(
            'new_ya_agency_client_client',
            {to => $vars->{email}},
            $Settings::NOTIFICATION_EMAIL_FROM,
            $vars,
            0
        );
    }

    #Отправляем уведомления менеджерам
    send_prepared_mail(
            'new_ya_agency_client',
            { to => $_, 'x-otrs-fromfeedback' => 'direct_setup', 'x-login' => $vars->{login} },
            ($vars->{email} || $Settings::NOTIFICATION_EMAIL_FROM), $vars, 0
        ) foreach (split /\s*,\s*/, $Settings::YA_AGENCY_NOTIFICATION_EMAIL_TO);
}

=head2 notification_ya_agency_order_paid

Нотификация менеджерам о том, что клиент оплатил обслуживание Я.Агенством

=cut
sub notification_ya_agency_order_paid {
    my (undef, $vars, $options) = @_;

    $vars->{product_name} //= $Settings::YaAgency_PRODUCTS->{$vars->{product_type}} // $vars->{product_type};

    my $chief_uid;
    if ($vars->{client_id}) {
        $chief_uid = rbac_get_chief_rep_of_client($vars->{client_id});
        $vars->{login} //= get_login(uid => $chief_uid);
        $vars->{email} = NotificationTools::get_emails_by_uids([$chief_uid])->[0];
    }

    #Уведомление клиенту
    send_prepared_mail(
        'ya_agency_order_paid_client',
        Notification::get_additional_email_params(uid => $chief_uid),
        $Settings::NOTIFICATION_EMAIL_FROM,
        $vars,
        0
    ) if $chief_uid;

    #Уведомления менеджерам
    send_prepared_mail(
            'ya_agency_order_paid',
            { to => $_, 'x-otrs-fromfeedback' => 'direct_setup', 'x-login' => $vars->{login} },
            ($vars->{email} || $Settings::NOTIFICATION_EMAIL_FROM), $vars, 0
        ) foreach (split /\s*,\s*/, $Settings::YA_AGENCY_NOTIFICATION_EMAIL_TO);
}

#-----------------------------------------------------------
# get info about managers/agencies
#-----------------------------------------------------------
sub get_manager_info_of_cid($$) {
    my ($cid, $vars) = @_;

    my $camp_info = get_one_line_sql(PPC(cid => $cid),
       "SELECT c.name AS campaign_name
             , c.type AS campaign_type
             , c.ManagerUID
             , u.login AS client_login
             , u.FIO AS client_FIO
             , u.uid AS client_uid
             , u.ClientID AS client_id
         FROM campaigns c
         JOIN users u ON c.uid = u.uid
        WHERE cid = ?", $cid
    ) || {};

    my $man_info;
    if ($camp_info->{ManagerUID}) {
        $man_info = get_one_line_sql(PPC(uid => $camp_info->{ManagerUID}),
           "SELECT mu.email
                 , mu.FIO
                 , mu.uid AS manager_uid
                 , mu.login AS manager_login
                 , mu.phone AS manager_phone
                 , IFNULL(muo.manager_use_crm, 'No') = 'Yes' AS manager_use_crm
              FROM users mu
                   LEFT JOIN users_options muo ON mu.uid = muo.uid
             where mu.uid = ?", $camp_info->{ManagerUID}
        ) || {};

        # этого поля не было
        delete $camp_info->{ManagerUID};

        hash_merge($man_info, $camp_info);
    } else {
        # в оригинале был один запрос с campaigns join users. Если кампания не менеджерская - то ничего не находилось
        return {};
    }

    if ($man_info->{manager_login}) {
        # NB! при изменении полей, добавляемых здесь в $vars - проверь и отредактируй
        # @get_manager_info_of_cid_fields_used_in_crm
        $vars->{campaign_id}   = $cid;
        $vars->{manager_login} = $man_info->{manager_login};
        $vars->{manager_uid}   = $man_info->{manager_uid};
        $vars->{manager_email} = $man_info->{email};
        $vars->{manager_FIO}   = $man_info->{FIO};
        $vars->{manager_phone} = $man_info->{manager_phone};
        $vars->{campaign_name} = $man_info->{campaign_name};
        $vars->{campaign_type} = $man_info->{campaign_type};
        $vars->{client_login}  = $man_info->{client_login};
        $vars->{client_FIO}    = $man_info->{client_FIO};
        $vars->{client_id}     = $man_info->{client_id};
    }

    return $man_info;
}

#-----------------------------------------------------------
sub get_manager_info($$) {
    my ($manager_uid, $vars) = @_;

    my $man_info = get_one_line_sql(PPC(uid => $manager_uid),
       "SELECT u.uid
             , u.login
             , u.email
             , u.FIO
             , u.uid as manager_uid
             , IFNULL(uo.manager_use_crm, 'No') = 'Yes' AS manager_use_crm
         FROM users u
              LEFT JOIN users_options uo ON u.uid = uo.uid
        WHERE u.uid = ?", $manager_uid
    ) || {};

    if ($man_info->{login}) {
        $vars->{manager_login} = $man_info->{login};
        $vars->{manager_uid} = $man_info->{manager_uid};
        $vars->{manager_email} = $man_info->{email};
        $vars->{manager_FIO}   = $man_info->{FIO};
    }

    return $man_info;
}

=head2 get_managers_info_of_client

    возвращает информацию обо всех менеджерах клиента

=cut

sub get_managers_info_of_client($$) {
    my (undef, $cl_uid) = @_;

    my $managers_uid = rbac_get_managers_of_client(undef, $cl_uid);
    return [] unless @$managers_uid;

    my $managers_info = get_all_sql(PPC(uid => $managers_uid), [
           "SELECT mu.email AS email
                 , mu.FIO AS FIO
                 , mu.uid AS uid
                 , mu.login AS login
                 , IFNULL(muo.manager_use_crm, 'No') = 'Yes' AS manager_use_crm
              FROM users mu
                   LEFT JOIN users_options muo ON mu.uid = muo.uid",
             where => { 'mu.uid' => SHARD_IDS } ]
        );

    return $managers_info;
}

#-----------------------------------------------------------
sub get_manager_info_of_agency_of_cid($$$) {
    my (undef, $cid, $vars) = @_;

    my ($camp_type, $agency_uid) = get_one_line_array_sql(PPC(cid => $cid), "select type, AgencyUID from campaigns where cid = ?", $cid);
    return {} unless $agency_uid;

    $camp_type ||= 'text';
    my $man_of_ag_uid = rbac_get_manager_of_agency(undef, $agency_uid, $camp_type);
    return {} unless $man_of_ag_uid;

    my $man_info = get_one_line_sql(PPC(uid => $man_of_ag_uid),
       "select u.login
             , u.email
             , u.FIO
             , IFNULL(uo.manager_use_crm, 'No') AS manager_use_crm
             , u.uid AS manager_uid
          FROM users u
               LEFT JOIN users_options uo USING(uid)
         WHERE u.uid = ?", $man_of_ag_uid
    ) || {};

    return {} if $man_info->{manager_use_crm} eq 'Yes'; # не посылать письма менеджерам которые используют CRM

    my $agency_info = get_one_line_sql(PPC(uid => $agency_uid), q/
        SELECT IFNULL(uo.sendAgencyDirectLetters, 'No') = 'Yes' as sendAgencyDirectLetters
             , IFNULL(uo.sendAgencyMcbLetters,    'No') = 'Yes' as sendAgencyMcbLetters
        FROM users_options uo
        WHERE uo.uid = ?/, $agency_uid);
    hash_merge $man_info, $agency_info;

    if ($man_info->{login}) {
        $vars->{manager_login} = $man_info->{login};
        $vars->{manager_email} = $man_info->{email};
        $vars->{manager_FIO}   = $man_info->{FIO};
        $vars->{manager_uid}   = $man_of_ag_uid;
    }

    return $man_info;
}

=head2 get_managers_info_of_agencies_of_client

    возвращает информацию обо всех менеджерах агентства клиента

=cut

sub get_managers_info_of_agencies_of_client($$) {
    my (undef, $cl_uid) = @_;

    my $agencies_uids = rbac_get_agencies_of_client(undef, $cl_uid);
    return [] unless @$agencies_uids;

    my $camp_type = 'text'; # на данный момент функция НЕ учитывает менеджеров агентств по медийным и гео-кампаниям
    my $man_of_agencies_uid = hash_grep { $_ } rbac_mass_get_manager_of_agencies(undef, $agencies_uids, $camp_type);
    return [] unless values %$man_of_agencies_uid;

    my $managers_info = get_hashes_hash_sql(PPC(uid => [values %$man_of_agencies_uid]), [
       "select u.uid
             , u.login
             , u.email
             , u.FIO
             , IFNULL(uo.manager_use_crm, 'No') AS manager_use_crm
          FROM users u
               LEFT JOIN users_options uo USING(uid)",
         WHERE => { 'u.uid' => SHARD_IDS} ]
    );

    # не посылать письма менеджерам которые используют CRM
    $man_of_agencies_uid = hash_grep { !$managers_info->{$_} || $managers_info->{$_}->{manager_use_crm} eq 'Yes' } $man_of_agencies_uid;
    return [] unless values %$man_of_agencies_uid;

    my $agencies_info = get_all_sql(PPC(uid => $agencies_uids), [q/
        SELECT uo.uid
             , IFNULL(uo.sendAgencyDirectLetters, 'No') = 'Yes' as sendAgencyDirectLetters
             , IFNULL(uo.sendAgencyMcbLetters,    'No') = 'Yes' as sendAgencyMcbLetters
        FROM users_options uo/,
        WHERE => { 'uo.uid' => SHARD_IDS }]);
    for my $ag_info (@$agencies_info) {
        my $man_of_ag_uid = $man_of_agencies_uid->{$ag_info->{uid}};
        next unless $man_of_ag_uid;

        my $man_info = $managers_info->{$man_of_ag_uid};
        next unless $man_info;

        for my $flag (qw/sendAgencyDirectLetters sendAgencyMcbLetters/) {
            $man_info->{$flag} ||= $ag_info->{$flag};
        }
    }

    return [values %$managers_info];
}

#-----------------------------------------------------------
sub get_agency_info($) {
    my $agency_uid = shift;

    my $agency_info = get_one_line_sql(PPC(uid => $agency_uid),
       "SELECT u.uid
             , u.login
             , u.email
             , u.FIO
             , IFNULL(uo.agency_email_to_chief, 'No') = 'Yes' AS agency_email_to_chief
          FROM users u
               LEFT JOIN users_options uo ON u.uid = uo.uid
         WHERE u.uid = ?", $agency_uid
    ) || {};

    return $agency_info;
}

#-----------------------------------------------------------
=head2 get_additional_email_params

receives:
    Hash, for example:
    (
        uid => '1234',
        to  => 'somebody@somewhere.ru',
        cid => '1234',
        bcc => 'somebody@somewhere.ru',
        cc  => 'somebody@somewhere.ru'
    )
    Note that 'uid' or 'to' field is mandatory.

returns:
    Hashref like above but without 'cid' field and with 'reply-to' field if campaign is controlled by agency or manager.
    If 'cid' wasn't in the method argumens or campaign isn't controlled by agency or manager then 'reply-to' will not generated.

=cut
sub get_additional_email_params
{
    my %params = @_;
    my $manager_of_agency;
    my $reply_to;

    delete $params{uid} unless defined $params{uid}; # stop concat warnings

    die "'uid' or 'to' is mandatory." if !(exists $params{uid} && defined $params{uid}) && !defined $params{to};

    if((exists $params{uid} && defined $params{uid}) && rbac_who_is(undef, $params{uid}) eq 'agency'){
        # is agency

        my $manager = rbac_get_manager_of_agency(undef, $params{uid}, 'text');
        $reply_to = get_one_user_field($manager, 'email') if $manager;

    } elsif(defined $params{cid}){
        # is client or subclient

        if ($params{cid} =~ /^\d+$/) {
            my $camp_info = get_one_line_sql(PPC(cid => $params{cid}),
                                             "SELECT COALESCE(c.AgencyUID, c.ManagerUID) as agency_or_manager_uid
                                                   , c.type
                                                   , c.uid
                                              FROM campaigns c
                                              WHERE c.cid = ?", $params{cid}) || {};

            if ($camp_info->{agency_or_manager_uid}) {

                # сразу известен менеджер или агентство на кампании, ставим reply-to на него
                $reply_to = get_one_user_field($camp_info->{agency_or_manager_uid}, 'email');

            } elsif ($camp_info->{type} eq 'wallet') {

                # тип кампании - общий счет, менеджер не известен
                my $manager_uids = get_all_sql(PPC(cid => $params{cid}), [
                    "select c.ManagerUID, count(*) camp_count
                     from campaigns c
                    ", where => {
                        "c.uid" => $camp_info->{uid}
                        , "c.wallet_cid" => $params{cid}
                        , "c.ManagerUID__gt" => 0
                        , "c.statusEmpty" => "No"
                    }, "group by" => "c.ManagerUID"
                    , "order by camp_count desc"
                ]) || [];

                # подставляем менеджера у которого больше всего кампаний под счетом
                if (@$manager_uids && $manager_uids->[0] && $manager_uids->[0]->{ManagerUID}) {
                    $reply_to = get_one_user_field($manager_uids->[0]->{ManagerUID}, 'email');
                }
            }
        }
    }

    $params{reply_to} = $reply_to if $reply_to;
    delete $params{cid} if exists $params{cid};

    return \%params;
}


=head2 _camp_add_data(cid)

    По номеру кампании получить ссылку на хэш с дополнительными данными для нотификаций

=cut
sub _camp_add_data($)
{
    my (undef, $cid) = @_;

    my $camp_data = get_one_line_sql(PPC(cid => $cid), "
                    SELECT sc.master_cid, c.uid, c.ManagerUID, c.AgencyID, cl.country_region_id, c.type, c.ClientID,
                           FIND_IN_SET('is_touch', c.opts) AS is_touch_camp,
                           FIND_IN_SET('is_touch', clo.client_flags) AS is_touch_client
                      FROM campaigns c
                      LEFT JOIN subcampaigns sc ON c.cid = sc.cid
                      JOIN clients cl ON c.ClientID = cl.ClientID
                      LEFT JOIN clients_options clo ON clo.ClientID = cl.ClientID
                     WHERE c.cid = ?
                     ", $cid);

    my %ret = (
        master_cid => $camp_data->{master_cid},
        client_uid => $camp_data->{uid},
        client_id => $camp_data->{ClientID},
        client_country_region_id => $camp_data->{country_region_id},
        campaign_type => $camp_data->{type},
        is_touch_camp => $camp_data->{is_touch_camp},
        is_touch_client => $camp_data->{is_touch_client},
    );

    if ( $camp_data->{ManagerUID} ) {
        $ret{is_manager_camp} = 1;
        get_manager_info_of_cid($cid, \%ret);
        $ret{manager_office} = YandexOffice::get_manager_office($camp_data->{ManagerUID});
    } elsif ( $camp_data->{AgencyID} ) {
        # если кампания Баяна, то берем менеджера агенства для Баяна, иначе менеджера агенства для Директа
        my $interface_type = is_media_camp(cid => $cid) ? 'mcb' : 'text';
        hash_merge \%ret, _get_agency_info_by_agency_client_id(undef, $camp_data->{AgencyID}, $interface_type);
    }

    return \%ret;
}

=head2 get_agency_manager_info_of_clientid(undef, $client_id, $interface_type, $options)

    Функция собирает все данные для футера нотификации по ClientID клиента.

    Входные параметры:
        $client_id
        $interface_type
        $options - именованные опции
            {
                init_manager_uid => (0|1) - автоматически определять и добавлять в $vars manager_uid если клиент сервисируемый
                init_ag_manager_uid => (0|1) - автоматически определять и добавлять в $vars ag_manager_uid если клиент агентский
            }

    В ответе хеш с полями в зависимости от клиента:
    * самостоятельный
        response = {
             client_country_region_id   225
        }
    * менеджерский
        response = {
            client_country_region_id   225,
            is_manager_camp            1,
            manager_email              "i.vinogradov@yandex-team.ru",
            manager_FIO                "Адель Дьяконова",
            manager_login              "yndx-ivvinogradov",
            manager_phone              2513,
            manager_office             {
                country_code        "RU",
                office_email        "adv@yandex-team.ru",
                office_id           1,
                office_name         "Москва",
                office_nick         "msk",
                office_phone        "+7 (495) 739-70-00",
                office_short_name   "Мск",
                sales_add_phone     2000,
                serviced_regions    [
                    [0] 0
                ],
                servicing_border    300.000000,
                support_phone       "+7 (495) 739-37-77",
                use_nds             1
            }
        }
    * агентский
        response = {
            agency                     {
                chief_rep   {
                    email   "sales@it-agency.ru",
                    fio     "Лира Ященко",
                    phone   "495 585-81-67"
                },
                name        "IT-Agency"
            },
            agency_country_region_id   225,
            client_country_region_id   225,
            is_agency_camp             1,
            manager_email              "n.burtseva@yandex-team.ru",
            manager_FIO                "Адольф Симкин",
            manager_login              "yndx-burtsevan",
            manager_phone              2785,
            manager_office             {
                country_code        "RU",
                office_email        "adv@yandex-team.ru",
                office_id           1,
                office_name         "Москва",
                office_nick         "msk",
                office_phone        "+7 (495) 739-70-00",
                office_short_name   "Мск",
                sales_add_phone     2000,
                serviced_regions    [
                    [0] 0
                ],
                servicing_border    300.000000,
                support_phone       "+7 (495) 739-37-77",
                use_nds             1
            }
        }

=cut

sub get_agency_manager_info_of_clientid
{
    my (undef, $client_id, $interface_type, $options) = @_;

    my $client_country_region_id;
    my $client_data = get_client_data($client_id, [qw/country_region_id/]);
    $client_country_region_id = $client_data->{country_region_id} if ($client_data);

    my %ret = (
        client_country_region_id => $client_country_region_id,
    );

    my $agency_client_id = Primitives::get_client_first_agency($client_id);

    if ($agency_client_id) {
        # агентский клиент
        hash_merge \%ret, _get_agency_info_by_agency_client_id(undef, $agency_client_id, $interface_type, $options);
    } else {
        my $manager_uid = rbac_get_primary_managers_of_clients_by_clientids([$client_id])->{$client_id};

        if ($manager_uid) {
            # менеджерский клиент
            my $manager = get_user_data($manager_uid, [qw/email fio login phone/]);

            $ret{is_manager_camp} = 1;
            $ret{manager_uid} = $manager_uid if $options->{init_manager_uid};
            $ret{manager_login} = $manager->{login};
            $ret{manager_email} = $manager->{email};
            $ret{manager_FIO} = $manager->{fio};
            $ret{manager_phone} = $manager->{phone};
            $ret{manager_office} = YandexOffice::get_manager_office($manager_uid);
        } else {
            # самостоятельный клиент
        }
    }

    return \%ret;
}

=head2 _get_agency_info_by_agency_client_id

    Получить данные агентства для футера по AgencyID

=cut
sub _get_agency_info_by_agency_client_id
{
    my (undef, $agency_client_id, $interface_type, $options) = @_;

    my %ret;
    if (is_valid_id($agency_client_id)) {
        my $ag_manager_uid = rbac_get_manager_of_agency_clientid(undef, $agency_client_id, $interface_type);
        my $ag_manager = get_user_data($ag_manager_uid, [qw/email fio login phone/]);
        my $ag_country_region_id;
        my $ag_data = get_client_data($agency_client_id, [qw/country_region_id/]);
        $ag_country_region_id = $ag_data->{country_region_id} if $ag_data;

        $ret{is_agency_camp} = 1;
        $ret{agency_country_region_id} = $ag_country_region_id;
        $ret{agency} = get_client_data($agency_client_id, [qw/name/]);
        $ret{agency}{chief_rep} = get_user_data(rbac_get_chief_rep_of_agency($agency_client_id), [qw/uid fio email phone/]);
        $ret{ag_manager_uid} = $ag_manager_uid if $options->{init_ag_manager_uid};
        $ret{manager_login} = $ag_manager->{login};
        $ret{manager_email} = $ag_manager->{email};
        $ret{manager_FIO} = $ag_manager->{fio};
        $ret{manager_phone} = $ag_manager->{phone};
        $ret{manager_office} = YandexOffice::get_manager_office($ag_manager_uid);
    }

    return \%ret;
}

1;
